//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ FDD(FD55GFR) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "filepath.h"
#include "fileio.h"
#include "rtc.h"
#include "iosc.h"
#include "config.h"
#include "fdc.h"
#include "fdi.h"
#include "fdd.h"

//===========================================================================
//
//	FDD
//
//===========================================================================
//#define FDD_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
FDD::FDD(VM *p) : Device(p)
{
	// foCXID
	dev.id = MAKEID('F', 'D', 'D', ' ');
	dev.desc = "Floppy Drive";

	// IuWFNg
	fdc = NULL;
	iosc = NULL;
	rtc = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::Init()
{
	int i;
	Scheduler *scheduler;

	ASSERT(this);

	// {NX
	if (!Device::Init()) {
		return FALSE;
	}

	// FDC擾
	fdc = (FDC*)vm->SearchDevice(MAKEID('F', 'D', 'C', ' '));
	ASSERT(fdc);

	// IOSC擾
	iosc = (IOSC*)vm->SearchDevice(MAKEID('I', 'O', 'S', 'C'));
	ASSERT(iosc);

	// XPW[擾
	scheduler = (Scheduler*)vm->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(scheduler);

	// RTC擾
	rtc = (RTC*)vm->SearchDevice(MAKEID('R', 'T', 'C', ' '));
	ASSERT(rtc);

	// hCuʂ̏
	for (i=0; i<4; i++) {
		drv[i].fdi = new FDI(this);
		drv[i].next = NULL;
		drv[i].seeking = FALSE;
		drv[i].cylinder = 0;
		drv[i].insert = FALSE;
		drv[i].invalid = FALSE;
		drv[i].eject = TRUE;
		drv[i].blink = FALSE;
		drv[i].access = FALSE;
	}

	// ApplyCfg
	fdd.fast = FALSE;

	// ʕ̏
	fdd.motor = FALSE;
	fdd.settle = FALSE;
	fdd.force = FALSE;
	fdd.first = TRUE;
	fdd.selected = 0;
	fdd.hd = TRUE;

	// V[NCxg
	seek.SetDevice(this);
	seek.SetDesc("Seek");
	seek.SetUser(0);
	seek.SetTime(0);
	scheduler->AddEvent(&seek);

	// ]Cxg(ZgOp)
	rotation.SetDevice(this);
	rotation.SetDesc("Rotation Stopped");
	rotation.SetUser(1);
	rotation.SetTime(0);
	scheduler->AddEvent(&rotation);

	// CWFNgCxg(}p)
	eject.SetDevice(this);
	eject.SetDesc("Eject");
	eject.SetUser(2);
	eject.SetTime(0);
	scheduler->AddEvent(&eject);

	// LbV[h
	cache_wb = TRUE;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Cleanup()
{
	int i;

	ASSERT(this);

	// C[Wt@C
	for (i=0; i<4; i++) {
		if (drv[i].fdi) {
			delete drv[i].fdi;
			drv[i].fdi = NULL;
		}
		if (drv[i].next) {
			delete drv[i].next;
			drv[i].next = NULL;
		}
	}

	// {NX
	Device::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Reset()
{
	int i;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// hCuʂ̃Zbg
	for (i=0; i<4; i++) {
		drv[i].seeking = FALSE;
		drv[i].eject = TRUE;
		drv[i].blink = FALSE;
		drv[i].access = FALSE;

		// next΁Aiグ
		if (drv[i].next) {
			delete drv[i].fdi;
			drv[i].fdi = drv[i].next;
			drv[0].fdi->Adjust();
			drv[1].fdi->Adjust();
			drv[i].next = NULL;
			drv[i].insert = TRUE;
			drv[i].invalid = FALSE;
		}

		// invalidȂACWFNg
		if (drv[i].invalid) {
			delete drv[i].fdi;
			drv[i].fdi = new FDI(this);
			drv[i].insert = FALSE;
			drv[i].invalid = FALSE;
		}

		// V_0փV[N
		drv[i].cylinder = 0;
		drv[i].fdi->Seek(0);
	}

	// ʕ̃Zbg(selectedFDCDSRƍ킹鎖)
	fdd.motor = FALSE;
	fdd.settle = FALSE;
	fdd.force = FALSE;
	fdd.first = TRUE;
	fdd.selected = 0;
	fdd.hd = TRUE;

	// V[NCxgȂ(seeking=FALSE)
	seek.SetTime(0);

	// ]EZgOCxgȂ(motor=FALSE, settle=FALSE)
	rotation.SetDesc("Rotation Stopped");
	rotation.SetTime(0);

	// CWFNgCxgȂ(iグinvalid)
	eject.SetTime(0);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::Save(Fileio *fio, int ver)
{
	int i;
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// hCuʕZ[u
	for (i=0; i<4; i++) {
		// TCYZ[u
		sz = sizeof(drv_t);
		if (!fio->Write(&sz, sizeof(sz))) {
			return FALSE;
		}

		// ̂Z[u
		if (!fio->Write(&drv[i], (int)sz)) {
			return FALSE;
		}

		// C[W͔C
		if (drv[i].fdi) {
			if (!drv[i].fdi->Save(fio, ver)) {
				return FALSE;
			}
		}
		if (drv[i].next) {
			if (!drv[i].next->Save(fio, ver)) {
				return FALSE;
			}
		}
	}

	// ʕZ[u
	sz = sizeof(fdd);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (!fio->Write(&fdd, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!seek.Save(fio, ver)) {
		return FALSE;
	}
	if (!rotation.Save(fio, ver)) {
		return FALSE;
	}
	if (!eject.Save(fio, ver)) {
		return FALSE;
	}

	// LbV[hۑ
	if (!fio->Write(&cache_wb, sizeof(cache_wb))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::Load(Fileio *fio, int ver)
{
	int i;
	size_t sz;
	drv_t work;
	BOOL ready;
	Filepath path;
	int media;
	BOOL success;
	BOOL failed;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "[h");

	// hCuʕZ[u
	for (i=0; i<4; i++) {
		// TCY[h
		if (!fio->Read(&sz, sizeof(sz))) {
			return FALSE;
		}
		if (sz != sizeof(drv_t)) {
			return FALSE;
		}

		// ̂[h
		if (!fio->Read(&work, (int)sz)) {
			return FALSE;
		}

		// ݂̃C[Wׂč폜
		if (drv[i].fdi) {
			delete drv[i].fdi;
			drv[i].fdi = NULL;
		}
		if (drv[i].next) {
			delete drv[i].next;
			drv[i].next = NULL;
		}

		// ]
		drv[i] = work;

		// C[Wč\z
		failed = FALSE;
		if (drv[i].fdi) {
			// \z(݂drv[i].fdi͈ӖȂ)
			drv[i].fdi = new FDI(this);

			// [h݂
			success = FALSE;
			if (drv[i].fdi->Load(fio, ver, &ready, &media, path)) {
				// fB̏ꍇ
				if (ready) {
					// I[v݂
					if (drv[i].fdi->Open(path, media)) {
						// 
						drv[i].fdi->Seek(drv[i].cylinder);
						success = TRUE;
					}
					else {
						// s(Z[u鎞_̓fB̂ŁACWFNg)
						failed = TRUE;
					}
				}
				else {
					// fBłȂ΁A[hOK
					success = TRUE;
				}
			}

			// sꍇAÔߍč쐬
			if (!success) {
				delete drv[i].fdi;
				drv[i].fdi = new FDI(this);
			}
		}
		if (drv[i].next) {
			// \z(݂drv[i].next͈ӖȂ)
			drv[i].next = new FDI(this);

			// [h݂
			success = FALSE;
			if (drv[i].next->Load(fio, ver, &ready, &media, path)) {
				// fB̏ꍇ
				if (ready) {
					// I[v݂
					if (drv[i].next->Open(path, media)) {
						// 
						drv[i].next->Seek(drv[i].cylinder);
						success = TRUE;
					}
				}
			}

			// sꍇ́AdeletenextO
			if (!success) {
				delete drv[i].next;
				drv[i].next = NULL;
			}
		}

		// {FDI̍ăI[vɎsꍇACWFNgN
		if (failed) {
			Eject(i, TRUE);
		}
	}

	// ʕ[h
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(fdd)) {
		return FALSE;
	}
	if (!fio->Read(&fdd, (int)sz)) {
		return FALSE;
	}

	// Cxg[h
	if (!seek.Load(fio, ver)) {
		return FALSE;
	}
	if (!rotation.Load(fio, ver)) {
		return FALSE;
	}
	if (!eject.Load(fio, ver)) {
		return FALSE;
	}

	// LbV[h[h(version2.06ȍ~)
	if (ver >= 0x0206) {
		if (!fio->Read(&cache_wb, sizeof(cache_wb))) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL FDD::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);
	LOG0(Log::Normal, "ݒKp");

	// [h
	fdd.fast = config->floppy_speed;
#if defined(FDD_LOG)
	if (fdd.fast) {
		LOG0(Log::Normal, "[h ON");
	}
	else {
		LOG0(Log::Normal, "[h OFF");
	}
#endif	// FDD_LOG

	// LbV[h
	cache_wb = config->cache_fd;
#if defined(FDD_LOG)
	if (cache_wb) {
		LOG0(Log::Normal, "LbV[h:WB");
	}
	else {
		LOG0(Log::Normal, "LbV[h:WT");
	}
#endif	// FDD_LOG

}

//---------------------------------------------------------------------------
//
//	hCu[N擾
//
//---------------------------------------------------------------------------
void FASTCALL FDD::GetDrive(int drive, drv_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT((drive >= 0) && (drive <= 3));

	// hCu[NRs[
	*buffer = drv[drive];
}

//---------------------------------------------------------------------------
//
//	[N擾
//
//---------------------------------------------------------------------------
void FASTCALL FDD::GetFDD(fdd_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// [NRs[
	*buffer = fdd;
}

//---------------------------------------------------------------------------
//
//	FDI擾
//
//---------------------------------------------------------------------------
FDI* FASTCALL FDD::GetFDI(int drive)
{
	ASSERT(this);
	ASSERT((drive == 0) || (drive == 1));

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next;
	}

	// fdi͕K
	ASSERT(drv[drive].fdi);
	return drv[drive].fdi;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::Callback(Event *ev)
{
	DWORD user;
	int i;

	ASSERT(this);
	ASSERT(ev);

	// [Uf[^󂯎
	user = ev->GetUser();

	switch (user) {
		// 0:V[N
		case 0:
			for (i=0; i<4; i++) {
				// V[N
				if (drv[i].seeking) {
					// V[N
					drv[i].seeking = FALSE;
#if defined(FDD_LOG)
					LOG1(Log::Normal, "V[N hCu%d", i);
#endif	// FDD_LOG

					// fBԂɂĕ
					fdc->CompleteSeek(i, IsReady(i));
				}
			}
			// PȂ̂break
			break;

		// 1:]EZgO
		case 1:
			// X^oCȂP
			if (!fdd.settle && !fdd.motor) {
				return FALSE;
			}

			// ZgO
			if (fdd.settle) {
				fdd.settle = FALSE;
				fdd.motor = TRUE;
				fdd.first = TRUE;

				// ]
				Rotation();
			}
			// p
			return TRUE;

		// 2:CWFNgE}
		case 2:
			for (i=0; i<4; i++) {
				// Next΁Aւ
				if (drv[i].next) {
					delete drv[i].fdi;
					drv[i].fdi = drv[i].next;
					drv[0].fdi->Adjust();
					drv[1].fdi->Adjust();
					drv[i].next = NULL;
					Insert(i);
				}

				// invalidŃCWFNg
				if (drv[i].invalid) {
					Eject(i, TRUE);
				}
			}
			// PȂ̂break
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// P
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::Open(int drive, const Filepath& path, int media)
{
	FDI *fdi;

	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));
	ASSERT((media >= 0) && (media < 0x10));

#if defined(FDD_LOG)
	LOG2(Log::Normal, "fBXNI[v hCu%d fBA%d", drive, media);
#endif	// FDD_LOG

	// FDI쐬+I[v
	fdi = new FDI(this);
	if (!fdi->Open(path, media)) {
		delete fdi;
		return FALSE;
	}

	// C[WsăV[N
	fdi->Seek(drv[drive].cylinder);

	// ɎC[W\񂳂Ăꍇ
	if (drv[drive].next) {
		// C[W폜čēxCxg(300ms)
		delete drv[drive].next;
		drv[drive].next = fdi;
		eject.SetTime(300 * 1000 * 2);
		return TRUE;
	}

	// fBAꍇ
	if (drv[drive].insert && !drv[drive].invalid) {
		// C[WɃZbgăCWFNgACxgs(300ms)
		Eject(drive, FALSE);
		drv[drive].next = fdi;
		eject.SetTime(300 * 1000 * 2);
		return TRUE;
	}

	// ʏ͂̂܂܃CT[g
	delete drv[drive].fdi;
	drv[drive].fdi = fdi;
	drv[0].fdi->Adjust();
	drv[1].fdi->Adjust();
	Insert(drive);
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	fBXN}
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Insert(int drive)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

#if defined(FDD_LOG)
	LOG1(Log::Normal, "fBXN} hCu%d", drive);
#endif	// FDD_LOG

	// V[N͈U
	if (drv[drive].seeking) {
		drv[drive].seeking = FALSE;
		fdc->CompleteSeek(drive, FALSE);
	}

	// fBXN}A}Ȃ
	drv[drive].insert = TRUE;
	drv[drive].invalid = FALSE;

	// ݂̃V_փV[N
	ASSERT(drv[drive].fdi);
	drv[drive].fdi->Seek(drv[drive].cylinder);

	// 荞݂N
	iosc->IntFDD(TRUE);
}

//---------------------------------------------------------------------------
//
//	fBXNCWFNg
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Eject(int drive, BOOL force)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// tOĂȂ΁ACWFNg֎~l
	if (!force && !drv[drive].eject) {
		return;
	}

	// ɃCWFNgĂΕsv
	if (!drv[drive].insert && !drv[drive].invalid) {
		return;
	}

#if defined(FDD_LOG)
	LOG1(Log::Normal, "fBXNCWFNg hCu%d", drive);
#endif	// FDD_LOG

	// V[N͈U
	if (drv[drive].seeking) {
		drv[drive].seeking = FALSE;
		fdc->CompleteSeek(drive, FALSE);
	}

	// C[Wt@CFDIɓւ
	ASSERT(drv[drive].fdi);
	delete drv[drive].fdi;
	drv[drive].fdi = new FDI(this);

	// fBXN}ȂA}Ȃ
	drv[drive].insert = FALSE;
	drv[drive].invalid = FALSE;

	// ANZXȂ(LED̓s)
	drv[drive].access = FALSE;

	// next͈U
	if (drv[drive].next) {
		delete drv[drive].next;
		drv[drive].next = NULL;
	}

	// 荞݂N
	iosc->IntFDD(TRUE);
}

//---------------------------------------------------------------------------
//
//	fBXN}
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Invalid(int drive)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

#if defined(FDD_LOG)
	LOG1(Log::Normal, "fBXN} hCu%d", drive);
#endif	// FDD_LOG

	// Ɍ}łΕsv
	if (drv[drive].insert && drv[drive].invalid) {
		return;
	}

	// V[N͈U
	if (drv[drive].seeking) {
		drv[drive].seeking = FALSE;
		fdc->CompleteSeek(drive, FALSE);
	}

	// fBXN}A}
	drv[drive].insert = TRUE;
	drv[drive].invalid = TRUE;

	// next͈U
	if (drv[drive].next) {
		delete drv[drive].next;
		drv[drive].next = NULL;
	}

	// 荞݂N
	iosc->IntFDD(TRUE);

	// Cxgݒ(300ms)
	eject.SetTime(300 * 1000 * 2);
}

//---------------------------------------------------------------------------
//
//	hCuRg[
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Control(int drive, DWORD func)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// bit7 - LED
	if (func & 0x80) {
		if (!drv[drive].blink) {
			// _łȂ_ł
#if defined(FDD_LOG)
			LOG1(Log::Normal, "LED_ł hCu%d", drive);
#endif	// FDD_LOG
			drv[drive].blink = TRUE;
		}
	}
	else {
		drv[drive].blink = FALSE;
	}

	// bit6 - CWFNgL
	if (func & 0x40) {
		if (drv[drive].eject) {
#if defined(FDD_LOG)
			LOG1(Log::Normal, "CWFNg֎~ hCu%d", drive);
#endif	// FDD_LOG
			drv[drive].eject = FALSE;
		}
	}
	else {
		drv[drive].eject = TRUE;
	}

	// bit5 - CWFNg
	if (func & 0x20) {
		Eject(drive, TRUE);
	}
}

//---------------------------------------------------------------------------
//
//	fB
//
//---------------------------------------------------------------------------
void FASTCALL FDD::ForceReady(BOOL flag)
{
	ASSERT(this);

#if defined(FDD_LOG)
	if (flag) {
		LOG0(Log::Normal, "fB ON");
	}
	else {
		LOG0(Log::Normal, "fB OFF");
	}
#endif	// FDD_LOG

	// Zbĝ
	fdd.force = flag;
}

//---------------------------------------------------------------------------
//
//	]ʒu擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDD::GetRotationPos() const
{
	DWORD remain;
	DWORD hus;

	ASSERT(this);

	// ~Ă0
	if (rotation.GetTime() == 0) {
		return 0;
	}

	// ]𓾂
	hus = GetRotationTime();

	// _EJE^𓾂
	remain = rotation.GetRemain();
	if (remain > hus) {
		remain = 0;
	}

	// t
	return (DWORD)(hus - remain);
}

//---------------------------------------------------------------------------
//
//	]Ԏ擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDD::GetRotationTime() const
{
	ASSERT(this);

	// 2HD360rpmA2DD300rpm
	if (fdd.hd) {
		return 333333;
	}
	else {
		return 400000;
	}
}

//---------------------------------------------------------------------------
//
//	]
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Rotation()
{
	DWORD rpm;
	DWORD hus;
	char desc[0x20];

	ASSERT(this);
	ASSERT(!fdd.settle);
	ASSERT(fdd.motor);

	rpm = 2000 * 1000 * 60;
	hus = GetRotationTime();
	rpm /= hus;
	sprintf(desc, "Rotation %drpm", rpm);
	rotation.SetDesc(desc);
	rotation.SetTime(hus);
}

//---------------------------------------------------------------------------
//
//	Ԏ擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDD::GetSearch()
{
	DWORD schtime;

	ASSERT(this);

	// [h64usŒ
	if (fdd.fast) {
		return 128;
	}

	// C[Wt@Cɕ
	schtime = drv[ fdd.selected ].fdi->GetSearch();

	return schtime;
}

//---------------------------------------------------------------------------
//
//	HDݒ
//
//---------------------------------------------------------------------------
void FASTCALL FDD::SetHD(BOOL hd)
{

	ASSERT(this);

	if (hd) {
		if (fdd.hd) {
			return;
		}
#if defined(FDD_LOG)
		LOG0(Log::Normal, "hCuxύX 2HD");
#endif	// FDD_LOG
		fdd.hd = TRUE;
	}
	else {
		if (!fdd.hd) {
			return;
		}
#if defined(FDD_LOG)
		LOG0(Log::Normal, "hCuxύX 2DD");
#endif	// FDD_LOG
		fdd.hd = FALSE;
	}

	// [^쒆ȂAxύX
	if (!fdd.motor || fdd.settle) {
		return;
	}

	// ]Đݒ
	Rotation();
}

//---------------------------------------------------------------------------
//
//	HD擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::IsHD() const
{
	ASSERT(this);

	return fdd.hd;
}

//---------------------------------------------------------------------------
//
//	ANZXLEDݒ
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Access(BOOL flag)
{
	int i;

	ASSERT(this);

	// ׂĉ낷
	for (i=0; i<4; i++) {
		drv[i].access = FALSE;
	}

	// flagオĂ΁AZNghCuL
	if (flag) {
		drv[ fdd.selected ].access = TRUE;
	}
}

//---------------------------------------------------------------------------
//
//	fB`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::IsReady(int drive, BOOL fdc) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// FDC̃ANZXȂ
	if (fdc) {
		// fBȂ烌fB
		if (fdd.force) {
			return TRUE;
		}

		// [^ItȂmbgfB
		if (!fdd.motor) {
			return FALSE;
		}
	}

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next->IsReady();
	}

	// C[Wt@Cɕ
	return drv[drive].fdi->IsReady();
}

//---------------------------------------------------------------------------
//
//	݋֎~`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::IsWriteP(int drive) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next->IsWriteP();
	}

	// C[Wt@Cɕ
	return drv[drive].fdi->IsWriteP();
}

//---------------------------------------------------------------------------
//
//	Read Only`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDD::IsReadOnly(int drive) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next->IsReadOnly();
	}

	// C[Wt@Cɕ
	return drv[drive].fdi->IsReadOnly();
}

//---------------------------------------------------------------------------
//
//	݋֎~
//
//---------------------------------------------------------------------------
void FASTCALL FDD::WriteP(int drive, BOOL flag)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// C[Wt@Cŏ
	drv[drive].fdi->WriteP(flag);
}

//---------------------------------------------------------------------------
//
//	hCuXe[^X擾
//
//	b7 : Insert
//	b6 : Invalid Insert
//	b5 : Inhibit Eject
//	b4 : Blink (Global)
//	b3 : Blink (Current)
//	b2 : Motor
//	b1 : Select
//	b0 : Access
//---------------------------------------------------------------------------
int FASTCALL FDD::GetStatus(int drive) const
{
	int st;

	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// NA
	st = 0;

	// bit7 - }
	if (drv[drive].insert) {
		st |= FDST_INSERT;
	}

	// bit6 - }(}܂)
	if (drv[drive].invalid) {
		st |= FDST_INVALID;
	}

	// bit5 - CWFNgł
	if (drv[drive].eject) {
		st |= FDST_EJECT;
	}

	// bit4 - _(O[oA}܂܂Ȃ)
	if (drv[drive].blink && !drv[drive].insert) {
		st |= FDST_BLINK;

		// bit3 - _()
		if (rtc->GetBlink(drive)) {
			st |= FDST_CURRENT;
		}
	}

	// bit2 - [^
	if (fdd.motor) {
		st |= FDST_MOTOR;
	}

	// bit1 - ZNg
	if (drive == fdd.selected) {
		st |= FDST_SELECT;
	}

	// bit0 - ANZX
	if (drv[drive].access) {
		st |= FDST_ACCESS;
	}

	return st;
}

//---------------------------------------------------------------------------
//
//	[^ݒ{hCuZNg
//
//---------------------------------------------------------------------------
void FASTCALL FDD::SetMotor(int drive, BOOL flag)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// flag=FALSEȂȒP
	if (!flag) {
		// łOFFȂKvȂ
		if (!fdd.motor) {
			return;
		}
#if defined(FDD_LOG)
		LOG0(Log::Normal, "[^OFF");
#endif	// FDD_LOG

		// [^~
		fdd.motor = FALSE;
		fdd.settle = FALSE;

		// ZNghCuݒ
		fdd.selected = drive;

		// X^oCCxgݒ
		rotation.SetDesc("Standby 54000ms");
		rotation.SetTime(54 * 1000 * 2 * 1000);
		return;
	}

	// ZNghCuݒ
	fdd.selected = drive;

	// [^ON܂̓ZgOȂȂ
	if (fdd.motor || fdd.settle) {
		return;
	}

#if defined(FDD_LOG)
	LOG1(Log::Normal, "[^ON hCu%dZNg", drive);
#endif	// FDD_LOG

	// ŃCxgĂ΁AX^oC
	if (rotation.GetTime() != 0) {
		// [^ON
		fdd.settle = FALSE;
		fdd.motor = TRUE;
		fdd.first = TRUE;

		// ]
		Rotation();
		return;
	}

	// [^OFFAZgO
	fdd.motor = FALSE;
	fdd.settle = TRUE;

	// ZgOCxgݒ([h64usAʏ탂[h384ms)
	rotation.SetDesc("Settle 384ms");
	if (fdd.fast) {
		rotation.SetTime(128);
	}
	else {
		rotation.SetTime(384 * 1000 * 2);
	}
}

//---------------------------------------------------------------------------
//
//	V_擾
//
//---------------------------------------------------------------------------
int FASTCALL FDD::GetCylinder(int drive) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// [NԂ
	return drv[drive].cylinder;
}

//---------------------------------------------------------------------------
//
//	fBXN擾
//
//---------------------------------------------------------------------------
void FASTCALL FDD::GetName(int drive, char *buf, int media) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));
	ASSERT(buf);
	ASSERT((media >= -1) && (media < 0x10));

	// next΁AnextD
	if (drv[drive].next) {
		drv[drive].next->GetName(buf, media);
		return;
	}

	// C[Wɕ
	drv[drive].fdi->GetName(buf, media);
}

//---------------------------------------------------------------------------
//
//	pX擾
//
//---------------------------------------------------------------------------
void FASTCALL FDD::GetPath(int drive, Filepath& path) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// next΁AnextD
	if (drv[drive].next) {
		drv[drive].next->GetPath(path);
		return;
	}

	// C[Wɕ
	drv[drive].fdi->GetPath(path);
}

//---------------------------------------------------------------------------
//
//	fBXN擾
//
//---------------------------------------------------------------------------
int FASTCALL FDD::GetDisks(int drive) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next->GetDisks();
	}

	// C[Wɕ
	return drv[drive].fdi->GetDisks();
}

//---------------------------------------------------------------------------
//
//	JgfBAԍ擾
//
//---------------------------------------------------------------------------
int FASTCALL FDD::GetMedia(int drive) const
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

	// next΁AnextD
	if (drv[drive].next) {
		return drv[drive].next->GetMedia();
	}

	// C[Wɕ
	return drv[drive].fdi->GetMedia();
}

//---------------------------------------------------------------------------
//
//	Lu[g
//
//---------------------------------------------------------------------------
void FASTCALL FDD::Recalibrate(DWORD srt)
{
	ASSERT(this);

	// v`FbN
	if (drv[ fdd.selected ].cylinder == 0) {
		fdc->CompleteSeek(fdd.selected, IsReady(fdd.selected));
		return;
	}

#if defined(FDD_LOG)
	LOG1(Log::Normal, "Lu[g hCu%d", fdd.selected);
#endif	// FDD_LOG

	// fdd[drive].cylinderXebvAEg
	StepOut(drv[ fdd.selected ].cylinder, srt);
}

//---------------------------------------------------------------------------
//
//	XebvC
//
//---------------------------------------------------------------------------
void FASTCALL FDD::StepIn(int step, DWORD srt)
{
	int cylinder;

	ASSERT(this);
	ASSERT((step >= 0) && (step < 256));

	// XebvC
	cylinder = drv[ fdd.selected ].cylinder;
	cylinder += step;
	if (cylinder >= 82) {
		cylinder = 81;
	}

	// v`FbN
	if (drv[ fdd.selected ].cylinder == cylinder) {
		// 荞݂o
		fdc->CompleteSeek(fdd.selected, IsReady(fdd.selected));
		return;
	}

#if defined(FDD_LOG)
	LOG2(Log::Normal, "XebvC hCu%d Xebv%d", fdd.selected, step);
#endif	// FDD_LOG

	// V[Nʂ
	SeekInOut(cylinder, srt);
}

//---------------------------------------------------------------------------
//
//	XebvAEg
//
//---------------------------------------------------------------------------
void FASTCALL FDD::StepOut(int step, DWORD srt)
{
	int cylinder;

	ASSERT(this);
	ASSERT((step >= 0) && (step < 256));

	// XebvAEg
	cylinder = drv[ fdd.selected ].cylinder;
	cylinder -= step;
	if (cylinder < 0) {
		cylinder = 0;
	}

	// v`FbN
	if (drv[ fdd.selected ].cylinder == cylinder) {
		fdc->CompleteSeek(fdd.selected, IsReady(fdd.selected));
		return;
	}

#if defined(FDD_LOG)
	LOG2(Log::Normal, "XebvAEg hCu%d Xebv%d", fdd.selected, step);
#endif	// FDD_LOG

	// V[Nʂ
	SeekInOut(cylinder, srt);
}

//---------------------------------------------------------------------------
//
//	V[N
//
//---------------------------------------------------------------------------
void FASTCALL FDD::SeekInOut(int cylinder, DWORD srt)
{
	int step;

	ASSERT(this);
	ASSERT((cylinder >= 0) && (cylinder < 82));

	// XebvvZ
	ASSERT(drv[ fdd.selected ].cylinder != cylinder);
	if (drv[ fdd.selected ].cylinder < cylinder) {
		step = cylinder - drv[ fdd.selected ].cylinder;
	}
	else {
		step = drv[ fdd.selected ].cylinder - cylinder;
	}

	// ɃV[NsĂ܂(stO͗Ă)
	drv[ fdd.selected ].cylinder = cylinder;
	drv[ fdd.selected ].fdi->Seek(cylinder);
	drv[ fdd.selected ].seeking = TRUE;

	// Cxgݒ(SRT * Xebv + 0.64ms)
	if (fdd.fast) {
		// [h64usŒ
		seek.SetTime(128);
	}
	else {
		srt *= step;
		srt += 1280;

		// [^ON̏128ms
		if (fdd.first) {
			srt += (128 * 0x800);
			fdd.first = FALSE;
		}
		seek.SetTime(srt);
	}
}

//---------------------------------------------------------------------------
//
//	[hID
//	̃Xe[^XԂ(G[OR)
//		FDD_NOERROR		G[Ȃ
//		FDD_NOTREADY	mbgfB
//		FDD_MAM			w薧xł̓AtH[}bg
//		FDD_NODATA		w薧xł̓AtH[}bgA
//						܂͗LȃZN^ׂID CRC
//
//---------------------------------------------------------------------------
int FASTCALL FDD::ReadID(DWORD *buf, BOOL mfm, int hd)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT((hd == 0) || (hd == 4));

#if defined(FDD_LOG)
	LOG2(Log::Normal, "[hID hCu%d wbh%d", fdd.selected, hd);
#endif	// FDD_LOG

	// C[WɔC
	return drv[ fdd.selected ].fdi->ReadID(buf, mfm, hd);
}

//---------------------------------------------------------------------------
//
//	[hZN^
//	̃Xe[^XԂ(G[OR)
//		FDD_NOERROR		G[Ȃ
//		FDD_NOTREADY	mbgfB
//		FDD_MAM			w薧xł̓AtH[}bg
//		FDD_NODATA		w薧xł̓AtH[}bgA
//						܂͗LȃZN^SRvȂ
//		FDD_NOCYL		IDCvłAFFłȂZN^
//		FDD_BADCYL		IDCvAFFƂȂĂZN^
//		FDD_IDCRC		IDtB[hCRCG[
//		FDD_DATACRC		DATAtB[hCRCG[
//		FDD_DDAM		f[ebhZN^ł
//
//---------------------------------------------------------------------------
int FASTCALL FDD::ReadSector(BYTE *buf, int *len, BOOL mfm, DWORD *chrn, int hd)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT(len);
	ASSERT(chrn);
	ASSERT((hd == 0) || (hd == 4));

#if defined(FDD_LOG)
	LOG4(Log::Normal, "[hZN^ C:%02X H:%02X R:%02X N:%02X",
						chrn[0], chrn[1], chrn[2], chrn[3]);
#endif	// FDD_LOG

	// C[WɔC
	return drv[ fdd.selected ].fdi->ReadSector(buf, len, mfm, chrn, hd);
}

//---------------------------------------------------------------------------
//
//	CgZN^
//	̃Xe[^XԂ(G[OR)
//		FDD_NOERROR		G[Ȃ
//		FDD_NOTREADY	mbgfB
//		FDD_NOTWRITE	fBA͏݋֎~
//		FDD_MAM			w薧xł̓AtH[}bg
//		FDD_NODATA		w薧xł̓AtH[}bgA
//						܂͗LȃZN^SRvȂ
//		FDD_NOCYL		IDCvłAFFłȂZN^
//		FDD_BADCYL		IDCvAFFƂȂĂZN^
//		FDD_IDCRC		IDtB[hCRCG[
//		FDD_DDAM		f[ebhZN^ł
//
//---------------------------------------------------------------------------
int FASTCALL FDD::WriteSector(const BYTE *buf, int *len, BOOL mfm, DWORD *chrn, int hd, BOOL deleted)
{
	int result;

	ASSERT(this);
	ASSERT(len);
	ASSERT(chrn);
	ASSERT((hd == 0) || (hd == 4));

#if defined(FDD_LOG)
	LOG4(Log::Normal, "CgZN^ C:%02X H:%02X R:%02X N:%02X",
						chrn[0], chrn[1], chrn[2], chrn[3]);
#endif	// FDD_LOG

	// C[WɔC
	result = drv[ fdd.selected ].fdi->WriteSector(buf, len, mfm, chrn, hd, deleted);

	// CgX[LbV
	if (!cache_wb && buf) {
		drv[ fdd.selected ].fdi->Flush();
	}

	return result;
}

//---------------------------------------------------------------------------
//
//	[h_CAO
//
//---------------------------------------------------------------------------
int FASTCALL FDD::ReadDiag(BYTE *buf, int *len, BOOL mfm, DWORD *chrn, int hd)
{
	ASSERT(this);
	ASSERT(len);
	ASSERT(chrn);
	ASSERT((hd == 0) || (hd == 4));

#if defined(FDD_LOG)
	LOG4(Log::Normal, "[h_CAO C:%02X H:%02X R:%02X N:%02X",
						chrn[0], chrn[1], chrn[2], chrn[3]);
#endif	// FDD_LOG

	// C[WɔC
	return drv[ fdd.selected ].fdi->ReadDiag(buf, len, mfm, chrn, hd);
}

//---------------------------------------------------------------------------
//
//	CgID
//
//---------------------------------------------------------------------------
int FASTCALL FDD::WriteID(const BYTE *buf, DWORD d, int sc, BOOL mfm, int hd, int gpl)
{
	int result;
	ASSERT(this);
	ASSERT(sc > 0);
	ASSERT((hd == 0) || (hd == 4));

#if defined(FDD_LOG)
	LOG3(Log::Normal, "CgID hCu%d V_%d wbh%d",
					fdd.selected, drv[ fdd.selected ].cylinder, hd);
#endif	// FDD_LOG

	// C[WɔC
	result = drv[ fdd.selected ].fdi->WriteID(buf, d, sc, mfm, hd, gpl);

	// CgX[LbV
	if (!cache_wb && buf) {
		drv[ fdd.selected ].fdi->Flush();
	}

	return result;
}
