//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2005 ohD(ytanaka@ipc-tokai.or.jp)
//	[ OtBbNVRAM ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "fileio.h"
#include "render.h"
#include "renderin.h"
#include "gvram.h"

//===========================================================================
//
//	OtBbNVRAMnh
//
//===========================================================================
//#define GVRAM_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAMHandler::GVRAMHandler(Render *rend, BYTE *mem, CPU *p)
{
	ASSERT(rend);
	ASSERT(mem);
	ASSERT(p);

	render = rend;
	gvram = mem;
	cpu = p;
}

//===========================================================================
//
//	OtBbNVRAMnh(1024~1024)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAM1024::GVRAM1024(Render *rend, BYTE *mem, CPU *cpu) : GVRAMHandler(rend, mem, cpu)
{
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM1024::ReadByte(DWORD addr)
{
	DWORD offset;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// oCg͓ǂݏoĂ0
	if ((addr & 1) == 0) {
		return 0x00;
	}

	// ʍ
	offset = addr & 0x3ff;

	// ㉺Eŕ
	if (addr & 0x100000) {
		if (addr & 0x400) {
			// y[W3
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] >> 4);
		}
		else {
			// y[W2
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] & 0x0f);
		}
	}
	else {
		if (addr & 0x400) {
			// y[W1
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] >> 4);
		}
		else {
			// y[W0
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] & 0x0f);
		}
	}
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM1024::ReadWord(DWORD addr)
{
	DWORD offset;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);

	// ʍ
	offset = addr & 0x3ff;

	// ㉺Eŕ
	if (addr & 0x100000) {
		if (addr & 0x400) {
			// y[W3
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] >> 4);
		}
		else {
			// y[W2
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] & 0x0f);
		}
	}
	else {
		if (addr & 0x400) {
			// y[W1
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] >> 4);
		}
		else {
			// y[W0
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] & 0x0f);
		}
	}
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM1024::WriteByte(DWORD addr, DWORD data)
{
	DWORD offset;
	DWORD mem;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x100);

	// oCg͏߂Ȃ
	if ((addr & 1) == 0) {
		return;
	}

	// ʍ
	offset = addr & 0x3ff;

	// ㉺Eŕ
	if (addr & 0x100000) {
		if (addr & 0x400) {
			// y[W3
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr] & 0x0f);
			mem |= (data << 4);

			// 
			if (gvram[addr] != mem) {
				gvram[addr] = (BYTE)mem;
				render->GrpMem(addr, 3);
			}
		}
		else {
			// y[W2
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr] != mem) {
				gvram[addr] = (BYTE)mem;
				render->GrpMem(addr, 2);
			}
		}
	}
	else {
		if (addr & 0x400) {
			// y[W1
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr ^ 1] & 0x0f);
			mem |= (data << 4);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 1);
			}
		}
		else {
			// y[W0
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃjuw
			mem = (gvram[addr ^ 1] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 0);
			}
		}
	}
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM1024::WriteWord(DWORD addr, DWORD data)
{
	DWORD offset;
	DWORD mem;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x10000);

	// ʍ
	offset = addr & 0x3ff;

	// ㉺Eŕ
	if (addr & 0x100000) {
		if (addr & 0x400) {
			// y[W3
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr ^ 1] & 0x0f);
			data &= 0x0f;
			mem |= (data << 4);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 3);
			}
		}
		else {
			// y[W2
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr ^ 1] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 2);
			}
		}
	}
	else {
		if (addr & 0x400) {
			// y[W1
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr] & 0x0f);
			data &= 0x0f;
			mem |= (data << 4);

			// 
			if (gvram[addr] != mem) {
				gvram[addr] = (BYTE)mem;
				render->GrpMem(addr, 1);
			}
		}
		else {
			// y[W0
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;

			// ʃju
			mem = (gvram[addr] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr] != mem) {
				gvram[addr] = (BYTE)mem;
				render->GrpMem(addr, 0);
			}
		}
	}
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM1024::ReadOnly(DWORD addr) const
{
	DWORD offset;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// oCg͓ǂݏoĂ0
	if ((addr & 1) == 0) {
		return 0x00;
	}

	// ʍ
	offset = addr & 0x3ff;

	// ㉺Eŕ
	if (addr & 0x100000) {
		if (addr & 0x400) {
			// y[W3
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] >> 4);
		}
		else {
			// y[W2
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr] & 0x0f);
		}
	}
	else {
		if (addr & 0x400) {
			// y[W1
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] >> 4);
		}
		else {
			// y[W0
			addr >>= 1;
			addr &= 0x7fc00;
			addr |= offset;
			return (gvram[addr ^ 1] & 0x0f);
		}
	}
}

//===========================================================================
//
//	OtBbNVRAMnh(16F)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAM16::GVRAM16(Render *rend, BYTE *mem, CPU *cpu) : GVRAMHandler(rend, mem, cpu)
{
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM16::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// AhX̂
	if (addr & 1) {
		if (addr < 0x80000) {
			// y[W0:[h̉ʃoCgb0-b3
			return (gvram[addr ^ 1] & 0x0f);
		}

		if (addr < 0x100000) {
			// y[W1:[h̉ʃoCgb4-b7
			addr &= 0x7ffff;
			return (gvram[addr ^ 1] >> 4);
		}

		if (addr < 0x180000) {
			// y[W2:[h̏ʃoCgb0-b3
			addr &= 0x7ffff;
			return (gvram[addr] & 0x0f);
		}

		// y[W3:[h̏ʃoCgb4-b7
		addr &= 0x7ffff;
		return (gvram[addr] >> 4);
	}

	// AhX͏0
	return 0;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM16::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);

	if (addr < 0x80000) {
		// y[W0:[h̉ʃoCgb0-b3
		return (gvram[addr] & 0x0f);
	}

	if (addr < 0x100000) {
		// y[W1:[h̉ʃoCgb4-b7
		addr &= 0x7ffff;
		return (gvram[addr] >> 4);
	}

	if (addr < 0x180000) {
		// y[W2:[h̏ʃoCgb0-b3
		addr &= 0x7ffff;
		return (gvram[addr ^ 1] & 0x0f);
	}

	// y[W3:[h̏ʃoCgb4-b7
	addr &= 0x7ffff;
	return (gvram[addr ^ 1] >> 4);
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM16::WriteByte(DWORD addr, DWORD data)
{
	DWORD mem;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x100);

	// AhX̂
	if (addr & 1) {
		if (addr < 0x80000) {
			// y[W0:[h̉ʃoCgb0-b3
			mem = (gvram[addr ^ 1] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 0);
			}
			return;
		}

		if (addr < 0x100000) {
			// y[W1:[h̉ʃoCgb4-b7
			addr &= 0x7ffff;
			mem = (gvram[addr ^ 1] & 0x0f);
			mem |= (data << 4);

			// 
			if (gvram[addr ^ 1] != mem) {
				gvram[addr ^ 1] = (BYTE)mem;
				render->GrpMem(addr ^ 1, 1);
			}
			return;
		}

		if (addr < 0x180000) {
			// y[W2:[h̏ʃoCgb0-b3
			addr &= 0x7ffff;
			mem = (gvram[addr] & 0xf0);
			mem |= (data & 0x0f);

			// 
			if (gvram[addr] != mem) {
				gvram[addr] = (BYTE)mem;
				render->GrpMem(addr, 2);
			}
			return;
		}

		// y[W3:[h̏ʃoCgb4-b7
		addr &= 0x7ffff;
		mem = (gvram[addr] & 0x0f);
		mem |= (data << 4);

		// 
		if (gvram[addr] != mem) {
			gvram[addr] = (BYTE)mem;
			render->GrpMem(addr, 3);
		}
		return;
	}

	// AhX͏߂Ȃ
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM16::WriteWord(DWORD addr, DWORD data)
{
	DWORD mem;

	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	if (addr < 0x80000) {
		// y[W0:[h̉ʃoCgb0-b3
		mem = (gvram[addr] & 0xf0);
		mem |= (data & 0x0f);

		// 
		if (gvram[addr] != mem) {
			gvram[addr] = (BYTE)mem;
			render->GrpMem(addr, 0);
		}
		return;
	}
	if (addr < 0x100000) {
		// y[W1:[h̉ʃoCgb4-b7
		addr &= 0x7ffff;
		mem = (gvram[addr] & 0x0f);
		data &= 0x0f;
		mem |= (data << 4);

		// 
		if (gvram[addr] != mem) {
			gvram[addr] = (BYTE)mem;
			render->GrpMem(addr, 1);
		}
		return;
	}
	if (addr < 0x180000) {
		// y[W2:[h̏ʃoCgb0-b3
		addr &= 0x7ffff;
		mem = (gvram[addr ^ 1] & 0xf0);
		mem |= (data & 0x0f);

		// 
		if (gvram[addr ^ 1] != mem) {
			gvram[addr ^ 1] = (BYTE)mem;
			render->GrpMem(addr ^ 1, 2);
		}
		return;
	}

	// y[W3:[h̏ʃoCgb4-b7
	addr &= 0x7ffff;
	mem = (gvram[addr ^ 1] & 0x0f);
	data &= 0x0f;
	mem |= (data << 4);

	// 
	if (gvram[addr ^ 1] != mem) {
		gvram[addr ^ 1] = (BYTE)mem;
		render->GrpMem(addr ^ 1, 3);
	}
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM16::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// AhX̂
	if (addr & 1) {
		if (addr < 0x80000) {
			// y[W0:[h̉ʃoCgb0-b3
			return (gvram[addr ^ 1] & 0x0f);
		}

		if (addr < 0x100000) {
			// y[W1:[h̉ʃoCgb4-b7
			addr &= 0x7ffff;
			return (gvram[addr ^ 1] >> 4);
		}

		if (addr < 0x180000) {
			// y[W2:[h̏ʃoCgb0-b3
			addr &= 0x7ffff;
			return (gvram[addr] & 0x0f);
		}

		// y[W3:[h̏ʃoCgb4-b7
		addr &= 0x7ffff;
		return (gvram[addr] >> 4);
	}

	// AhX͏0
	return 0;
}

//===========================================================================
//
//	OtBbNVRAMnh(256F)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAM256::GVRAM256(Render *rend, BYTE *mem, CPU *p) : GVRAMHandler(rend, mem, p)
{
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM256::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// y[W0
	if (addr < 0x80000) {
		if (addr & 1) {
			// [h̉ʃoCg
			return gvram[addr ^ 1];
		}
		return 0;
	}

	// y[W1
	if (addr < 0x100000) {
		addr &= 0x7ffff;
		if (addr & 1) {
			// [h̏ʃoCg
			return gvram[addr];
		}
		return 0;
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, TRUE);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM256::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);

	// y[W0
	if (addr < 0x80000) {
		// [h̉ʃoCg
		return gvram[addr];
	}

	// y[W1
	if (addr < 0x100000) {
		addr &= 0x7ffff;
		// [h̏ʃoCg
		return gvram[addr ^ 1];
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, TRUE);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM256::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x100);

	// y[W0
	if (addr < 0x80000) {
		if (addr & 1) {
			// [h̉ʃoCg
			if (gvram[addr ^ 1] != data) {
				gvram[addr ^ 1] = (BYTE)data;
				render->GrpMem(addr ^ 1, 0);
			}
		}
		return;
	}

	// y[W1(ubN2)
	if (addr < 0x100000) {
		addr &= 0x7ffff;
		if (addr & 1) {
			// [h̏ʃoCg
			if (gvram[addr] != data) {
				gvram[addr] = (BYTE)data;
				render->GrpMem(addr, 2);
			}
		}
		return;
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, FALSE);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM256::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	// y[W0
	if (addr < 0x80000) {
		// [h̉ʃoCg
		if (gvram[addr] != data) {
			gvram[addr] = (BYTE)data;
			render->GrpMem(addr, 0);
		}
		return;
	}

	// y[W1(ubN2)
	if (addr < 0x100000) {
		addr &= 0x7ffff;
		// [h̏ʃoCg
		if (gvram[addr ^ 1] != data) {
			gvram[addr ^ 1] = (BYTE)data;
			render->GrpMem(addr ^ 1, 2);
		}
		return;
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, FALSE);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM256::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// y[W0
	if (addr < 0x80000) {
		if (addr & 1) {
			// [h̉ʃoCg
			return gvram[addr ^ 1];
		}
		return 0;
	}

	// y[W1
	if (addr < 0x100000) {
		addr &= 0x7ffff;
		if (addr & 1) {
			// [h̏ʃoCg
			return gvram[addr];
		}
		return 0;
	}

	// oXG[
	return 0xff;
}

//===========================================================================
//
//	OtBbNVRAMnh()
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAMNDef::GVRAMNDef(Render *rend, BYTE *mem, CPU *p) : GVRAMHandler(rend, mem, p)
{
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAMNDef::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// y[W
	if (addr & 0x80000) {
		return 0;
	}

	// ʃy[W
	addr &= 0x7ffff;
	if (addr & 1) {
		return gvram[addr ^ 1];
	}
	return 0;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAMNDef::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);

	// y[W
	if (addr & 0x80000) {
		return 0;
	}

	// ʃy[W
	addr &= 0x7ffff;
	return gvram[addr ^ 1];
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAMNDef::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x100);

	// y[W
	if (addr & 0x80000) {
		return;
	}

	// ʃy[W
	addr &= 0x7ffff;
	if (addr & 1) {
		if (gvram[addr ^ 1] != data) {
			gvram[addr ^ 1] = (BYTE)data;
			render->GrpMem(addr ^ 1, 0);
		}
	}
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAMNDef::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	// y[W
	if (addr & 0x80000) {
		return;
	}

	// ʃy[W
	addr &= 0x7ffff;
	if (gvram[addr ^ 1] != data) {
		gvram[addr ^ 1] = (BYTE)data;
		render->GrpMem(addr ^ 1, 0);
	}
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAMNDef::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	// y[W
	if (addr & 0x80000) {
		return 0;
	}

	// ʃy[W
	addr &= 0x7ffff;
	if (addr & 1) {
		return gvram[addr ^ 1];
	}
	return 0;
}

//===========================================================================
//
//	OtBbNVRAMnh(65536F)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAM64K::GVRAM64K(Render *rend, BYTE *mem, CPU *p) : GVRAMHandler(rend, mem, p)
{
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM64K::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	if (addr < 0x80000) {
		return gvram[addr ^ 1];
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, TRUE);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM64K::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);

	if (addr < 0x80000) {
		return *(WORD*)(&gvram[addr]);
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, TRUE);
	return 0xffff;
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM64K::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT(data < 0x100);

	if (addr < 0x80000) {
		if (gvram[addr ^ 1] != data) {
			gvram[addr ^ 1] = (BYTE)data;
			render->GrpMem(addr ^ 1, 0);
		}
		return;
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, FALSE);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM64K::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	if (addr < 0x80000) {
		if (*(WORD*)(&gvram[addr]) != data) {
			*(WORD*)(&gvram[addr]) = (WORD)data;
			render->GrpMem(addr, 0);
		}
		return;
	}

	// oXG[
	cpu->BusErr(addr + 0xc00000, FALSE);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM64K::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT(addr <= 0x1fffff);

	if (addr < 0x80000) {
		return gvram[addr ^ 1];
	}

	// oXG[
	return 0xff;
}

//===========================================================================
//
//	OtBbNVRAM
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
GVRAM::GVRAM(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('G', 'V', 'R', 'M');
	dev.desc = "Graphic VRAM";

	// JnAhXAIAhX
	memdev.first = 0xc00000;
	memdev.last = 0xdfffff;

	// [NGA
	gvram = NULL;
	render = NULL;

	// nh
	handler = NULL;
	hand1024 = NULL;
	hand16 = NULL;
	hand256 = NULL;
	handNDef = NULL;
	hand64K = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL GVRAM::Init()
{
	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// mہANA
	try {
		gvram = new BYTE[ 0x80000 ];
	}
	catch (...) {
		return FALSE;
	}
	if (!gvram) {
		return FALSE;
	}
	memset(gvram, 0, 0x80000);

	// _擾
	render = (Render*)vm->SearchDevice(MAKEID('R', 'E', 'N', 'D'));
	ASSERT(render);

	// nh쐬
	hand1024 = new GVRAM1024(render, gvram, cpu);
	hand16 = new GVRAM16(render, gvram, cpu);
	hand256 = new GVRAM256(render, gvram, cpu);
	handNDef = new GVRAMNDef(render, gvram, cpu);
	hand64K = new GVRAM64K(render, gvram, cpu);

	// f[^
	gvdata.mem = TRUE;
	gvdata.siz = 0;
	gvdata.col = 3;
	gvcount = 0;

	// nh(64K)
	gvdata.type = 4;
	handler = hand64K;

	// NA}XN
	gvdata.mask[0] = 0xfff0;
	gvdata.mask[1] = 0xff0f;
	gvdata.mask[2] = 0xf0ff;
	gvdata.mask[3] = 0x0fff;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::Cleanup()
{
	ASSERT(this);

	// nh
	if (hand64K) {
		delete hand64K;
		hand64K = NULL;
	}
	if (handNDef) {
		delete handNDef;
		handNDef = NULL;
	}
	if (hand256) {
		delete hand256;
		hand256 = NULL;
	}
	if (hand16) {
		delete hand16;
		hand16 = NULL;
	}
	if (hand1024) {
		delete hand1024;
		hand1024 = NULL;
	}
	handler = NULL;

	// 
	if (gvram) {
		delete[] gvram;
		gvram = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::Reset()
{
	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// NAȂ
	gvdata.plane[0] = FALSE;
	gvdata.plane[1] = FALSE;
	gvdata.plane[2] = FALSE;
	gvdata.plane[3] = FALSE;

	// nh(64K)
	gvdata.mem = TRUE;
	gvdata.siz = 0;
	gvdata.col = 3;
	gvdata.type = 4;
	handler = hand64K;

	// ANZXJEg0
	gvcount = 0;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL GVRAM::Save(Fileio *fio, int /*ver*/)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Z[u");

	// Z[u
	if (!fio->Write(gvram, 0x80000)) {
		return FALSE;
	}

	// TCYZ[u
	sz = sizeof(gvram_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&gvdata, (int)sz)) {
		return FALSE;
	}

	// gvcount(version2.04Œǉ)
	if (!fio->Write(&gvcount, sizeof(gvcount))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL GVRAM::Load(Fileio *fio, int ver)
{
	size_t sz;
	int line;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// [h
	if (!fio->Read(gvram, 0x80000)) {
		return FALSE;
	}

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(gvram_t)) {
		return FALSE;
	}

	// ̂[h
	if (!fio->Read(&gvdata, (int)sz)) {
		return FALSE;
	}

	// gvcount(version2.04Œǉ)
	gvcount = 0;
	if (ver >= 0x0204) {
		if (!fio->Read(&gvcount, sizeof(gvcount))) {
			return FALSE;
		}
	}

	// _֒ʒm
	for (line=0; line<0x200; line++) {
		render->GrpAll(line, 0);
		render->GrpAll(line, 1);
		render->GrpAll(line, 2);
		render->GrpAll(line, 3);
	}

	// nhI
	switch (gvdata.type) {
		case 0:
			ASSERT(hand1024);
			handler = hand1024;
			break;
		// 16F^Cv
		case 1:
			ASSERT(hand16);
			handler = hand16;
			break;
		// 256F^Cv
		case 2:
			ASSERT(hand256);
			handler = hand256;
			break;
		// `^Cv
		case 3:
			ASSERT(handNDef);
			handler = handNDef;
			break;
		// 64KF^Cv
		case 4:
			ASSERT(hand64K);
			handler = hand64K;
			break;
		// ̑
		default:
			ASSERT(FALSE);
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::ApplyCfg(const Config* /*config*/)
{
	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::AssertDiag() const
{
	// {NX
	MemDevice::AssertDiag();

	ASSERT(this);
	ASSERT(GetID() == MAKEID('G', 'V', 'R', 'M'));
	ASSERT(memdev.first == 0xc00000);
	ASSERT(memdev.last == 0xdfffff);
	ASSERT(gvram);
	ASSERT(hand1024);
	ASSERT(hand16);
	ASSERT(hand256);
	ASSERT(handNDef);
	ASSERT(hand64K);
	ASSERT(handler);
	ASSERT((gvdata.mem == TRUE) || (gvdata.mem == FALSE));
	ASSERT((gvdata.siz == 0) || (gvdata.siz == 1));
	ASSERT((gvdata.col >= 0) && (gvdata.col <= 3));
	ASSERT((gvdata.type >= 0) && (gvdata.type <= 4));
	ASSERT((gvcount == 0) || (gvcount == 1));
}
#endif	// NDEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// EFCg(0.5EFCg)
	scheduler->Wait(gvcount);
	gvcount ^= 1;

	// nhɔC
	return handler->ReadByte(addr & 0x1fffff);
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT_DIAG();

	// EFCg(0.5EFCg)
	scheduler->Wait(gvcount);
	gvcount ^= 1;

	// nhɔC
	return handler->ReadWord(addr & 0x1fffff);
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// EFCg(0.5EFCg)
	scheduler->Wait(gvcount);
	gvcount ^= 1;

	// nhɔC
	handler->WriteByte(addr & 0x1fffff, data);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);
	ASSERT_DIAG();

	// EFCg(0.5EFCg)
	scheduler->Wait(gvcount);
	gvcount ^= 1;

	// nhɔC
	handler->WriteWord(addr & 0x1fffff, data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL GVRAM::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// nhɔC
	return handler->ReadOnly(addr & 0x1fffff);
}

//---------------------------------------------------------------------------
//
//	GVRAM擾
//
//---------------------------------------------------------------------------
const BYTE* FASTCALL GVRAM::GetGVRAM() const
{
	ASSERT(this);
	ASSERT_DIAG();

	return gvram;
}

//---------------------------------------------------------------------------
//
//	^Cvݒ
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::SetType(DWORD type)
{
	int next;
	int prev;
	BOOL mem;
	DWORD siz;

	ASSERT(this);
	ASSERT_DIAG();

	// ݂̒lۑ
	mem = gvdata.mem;
	siz = gvdata.siz;

	// ݒ
	if (type & 8) {
		gvdata.mem = TRUE;
	}
	else {
		gvdata.mem = FALSE;
	}
	if (type & 4) {
		gvdata.siz = 1;
	}
	else {
		gvdata.siz = 0;
	}
	gvdata.col = type & 3;

	// ݂gvdata.typeL
	prev = gvdata.type;

	// Vtype
	if (gvdata.mem) {
		next = 4;
	}
	else {
		if (gvdata.siz) {
			next = 0;
		}
		else {
			next = gvdata.col + 1;
		}
	}

	// ĂȂ
	if (prev != next) {
		switch (next) {
			// 1024F^Cv
			case 0:
				ASSERT(hand1024);
				handler = hand1024;
				break;
			// 16F^Cv
			case 1:
				ASSERT(hand16);
				handler = hand16;
				break;
			// 256F^Cv
			case 2:
				ASSERT(hand256);
				handler = hand256;
				break;
			// `^Cv
			case 3:
				LOG0(Log::Warning, "OtBbNVRAM `^Cv");
				ASSERT(handNDef);
				handler = handNDef;
				break;
			// 64KF^Cv
			case 4:
				ASSERT(hand64K);
				handler = hand64K;
				break;
			// ̑
			default:
				ASSERT(FALSE);
		}
		gvdata.type = next;
	}

	// ^Cv͎ʃTCYقȂĂ΁A_֒ʒm
	if ((gvdata.mem != mem) || (gvdata.siz != siz)) {
		render->SetVC();
	}
}

//---------------------------------------------------------------------------
//
//	NAݒ
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::FastSet(DWORD mask)
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(GVRAM_LOG)
	LOG1(Log::Normal, "NAv[w %02X", mask);
#endif	// GVRAM_LOG

	if (mask & 0x08) {
		gvdata.plane[3] = TRUE;
	}
	else {
		gvdata.plane[3] = FALSE;
	}

	if (mask & 0x04) {
		gvdata.plane[2] = TRUE;
	}
	else {
		gvdata.plane[2] = FALSE;
	}

	if (mask & 0x02) {
		gvdata.plane[1] = TRUE;
	}
	else {
		gvdata.plane[1] = FALSE;
	}

	if (mask & 0x01) {
		gvdata.plane[0] = TRUE;
	}
	else {
		gvdata.plane[0] = FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	NA
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::FastClr(const CRTC::crtc_t *p)
{
	ASSERT(this);
	ASSERT_DIAG();

	if (gvdata.siz) {
		// 1024x1024
		if (p->hd >= 1) {
			// 1024~1024A512 or 768
			FastClr768(p);
		}
		else {
			// 1024~1024A256
			FastClr256(p);
		}
	}
	else {
		// 512~512
		FastClr512(p);
	}
}

//---------------------------------------------------------------------------
//
//	NA 1024~1024 512/768
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::FastClr768(const CRTC::crtc_t *p)
{
	int y;
	int n;
	int j;
	int k;
	DWORD offset;
	WORD *q;

	ASSERT(this);
	ASSERT_DIAG();

#if defined(GVRAM_LOG)
	LOG0(Log::Normal, "NA 1024x1024 (512/768)");
#endif	// GVRAM_LOG

	// ItZbgyACn𓾂
	y = p->v_scan;
	n = 1;
	if ((p->v_mul == 2) && !(p->lowres)) {
		if (y & 1) {
			return;
		}
		y >>= 1;
	}
	if (p->v_mul == 0) {
		y <<= 1;
		n = 2;
	}

	// C[v
	for (j=0; j<n; j++) {
		// XN[WX^ItZbg𓾂
		offset = (y + p->grp_scrly[0]) & 0x3ff;

		// 㔼Aŕ
		if (offset < 512) {
			// |C^쐬
			q = (WORD*)&gvram[offset << 10];

			// ʃoCgNA
			for (k=0; k<512; k++) {
				*q++ &= 0xff00;
			}

			// tOUp
			render->GrpAll(offset, 0);
			render->GrpAll(offset, 1);
		}
		else {
			// |C^쐬
			offset &= 0x1ff;
			q = (WORD*)&gvram[offset << 10];

			// ʃoCgNA
			for (k=0; k<512; k++) {
				*q++ &= 0x00ff;
			}

			// tOUp
			render->GrpAll(offset, 2);
			render->GrpAll(offset, 3);
		}

		// ̃C
		y++;
	}
}

//---------------------------------------------------------------------------
//
//	NA 1024~1024 256
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::FastClr256(const CRTC::crtc_t *p)
{
#if defined(GVRAM_LOG)
	LOG0(Log::Normal, "NA 1024x1024 (256)");
#endif	// GVRAM_LOG

	// b[u
	FastClr768(p);
}

//---------------------------------------------------------------------------
//
//	NA 512x512
//
//---------------------------------------------------------------------------
void FASTCALL GVRAM::FastClr512(const CRTC::crtc_t *p)
{
	int y;
	int n;
	int i;
	int j;
	int k;
	int w[2];
	DWORD offset;
	WORD *q;

	ASSERT(this);
	ASSERT_DIAG();

#if defined(GVRAM_LOG)
	LOG1(Log::Normal, "NA 512x512 Scan=%d", p->v_scan);
#endif	// GVRAM_LOG

	// ItZbgyACn𓾂
	y = p->v_scan;
	n = 1;
	if ((p->v_mul == 2) && !(p->lowres)) {
		if (y & 1) {
			return;
		}
		y >>= 1;
	}
	if (p->v_mul == 0) {
		y <<= 1;
		n = 2;
	}

	// v[[v
	for (i=0; i<4; i++) {
		if (!gvdata.plane[i]) {
			continue;
		}

		// Zo
		w[0] = p->h_dots;
		w[1] = 0;
		if (((p->grp_scrlx[i] & 0x1ff) + w[0]) > 512) {
			w[1] = (p->grp_scrlx[i] & 0x1ff) + w[0] - 512;
			w[0] = 512 - (p->grp_scrlx[i] & 0x1ff);
		}

		// C[v
		for (j=0; j<n; j++) {
			// XN[WX^ItZbg𓾂
			offset = ((y + p->grp_scrly[i]) & 0x1ff) << 10;
			q = (WORD*)&gvram[offset + ((p->grp_scrlx[i] & 0x1ff) << 1)];

			// NA(1)
			for (k=0; k<w[0]; k++) {
				*q++ &= gvdata.mask[i];
			}
			if (w[1] > 0) {
				// NA(2)
				q = (WORD*)&gvram[offset];
				for (k=0; k<w[1]; k++) {
					*q++ &= gvdata.mask[i];
				}
			}

			// tOUp
			render->GrpAll(offset >> 10, i);

			// ̃C
			y++;
		}

		// C߂
		y -= n;
	}
}

