//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2005 ohD(ytanaka@ipc-tokai.or.jp)
//	[ L[{[h ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "schedule.h"
#include "mfp.h"
#include "mouse.h"
#include "sync.h"
#include "fileio.h"
#include "config.h"
#include "keyboard.h"

//===========================================================================
//
//	L[{[h
//
//===========================================================================
//#define KEYBOARD_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
Keyboard::Keyboard(VM *p) : Device(p)
{
	// foCXID
	dev.id = MAKEID('K', 'E', 'Y', 'B');
	dev.desc = "Keyboard";

	// IuWFNg
	sync = NULL;
	mfp = NULL;
	mouse = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL Keyboard::Init()
{
	int i;
	Scheduler *scheduler;

	ASSERT(this);

	// {NX
	if (!Device::Init()) {
		return FALSE;
	}

	// Sync쐬
	sync = new Sync;

	// MFP擾
	mfp = (MFP*)vm->SearchDevice(MAKEID('M', 'F', 'P', ' '));
	ASSERT(mfp);

	// XPW[擾
	scheduler = (Scheduler*)vm->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(scheduler);

	// }EX擾
	mouse = (Mouse*)vm->SearchDevice(MAKEID('M', 'O', 'U', 'S'));
	ASSERT(mouse);

	// Cxgǉ
	event.SetDevice(this);
	event.SetDesc("Key Repeat");
	event.SetUser(0);
	event.SetTime(0);
	scheduler->AddEvent(&event);

	// [N(ZbgM͂ȂBp[IZbĝ)
	keyboard.connect = TRUE;
	for (i=0; i<0x80; i++) {
		keyboard.status[i] = FALSE;
	}
	keyboard.rep_code = 0;
	keyboard.rep_count = 0;
	keyboard.rep_start = 500 * 1000 * 2;
	keyboard.rep_next = 110 * 1000 * 2;
	keyboard.send_en = TRUE;
	keyboard.send_wait = FALSE;
	keyboard.msctrl = 0;
	keyboard.tv_mode = TRUE;
	keyboard.tv_ctrl = TRUE;
	keyboard.opt2_ctrl = TRUE;
	keyboard.bright = 0;
	keyboard.led = 0;

	// TrueKeypR}hL[
	sync->Lock();
	keyboard.cmdnum = 0;
	keyboard.cmdread = 0;
	keyboard.cmdwrite = 0;
	sync->Unlock();

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::Cleanup()
{
	ASSERT(this);

	// Sync폜
	if (sync) {
		delete sync;
		sync = NULL;
	}

	// {NX
	Device::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::Reset()
{
	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// VXe|[gZbĝŁAsend_wait
	keyboard.send_wait = FALSE;

	// R}hSăNA
	ClrCommand();

	// 0xff𑗐M
	if (keyboard.send_en && !keyboard.send_wait && keyboard.connect) {
		mfp->KeyData(0xff);
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL Keyboard::Save(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(keyboard_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// {̂Z[u
	if (!fio->Write(&keyboard, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!event.Save(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL Keyboard::Load(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "[h");

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(keyboard_t)) {
		return FALSE;
	}

	// {̂[h
	if (!fio->Read(&keyboard, (int)sz)) {
		return FALSE;
	}

	// Cxg[h
	if (!event.Load(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::ApplyCfg(const Config *config)
{
	ASSERT(config);
	LOG0(Log::Normal, "ݒKp");

	// ڑ
	Connect(config->kbd_connect);
}

//---------------------------------------------------------------------------
//
//	ڑ
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::Connect(BOOL connect)
{
	int i;

	ASSERT(this);

	// vĂΖȂ
	if (keyboard.connect == connect) {
		return;
	}

#if defined(KEYBOARD_LOG)
	if (connect) {
		LOG0(Log::Normal, "L[{[hڑ");
	}
	else {
		LOG0(Log::Normal, "L[{[hؒf");
	}
#endif	// KEYBOARD_LOG

	// ڑԂۑ
	keyboard.connect = connect;

	// L[ItɁACxg~߂
	for (i=0; i<0x80; i++) {
		keyboard.status[i] = FALSE;
	}
	keyboard.rep_code = 0;
	event.SetTime(0);

	// ڑĂFF𑗐M
	if (keyboard.connect) {
		mfp->KeyData(0xff);
	}
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::GetKeyboard(keyboard_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// obt@փRs[
	*buffer = keyboard;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL Keyboard::Callback(Event *ev)
{
	ASSERT(this);
	ASSERT(ev);

	// s[gR[hLłȂ΁A~
	if (keyboard.rep_code == 0) {
		ev->SetTime(0);
		return FALSE;
	}

	// L[{[hĂ΁A~
	if (!keyboard.connect) {
		ev->SetTime(0);
		return FALSE;
	}

	// s[gJEgAbv
	keyboard.rep_count++;

	// CN̂ݔs(u[N͏oȂ:maxim.x)
	if (keyboard.send_en && !keyboard.send_wait) {
#if defined(KEYBOARD_LOG)
		LOG2(Log::Normal, "s[g$%02X (%d)", keyboard.rep_code, keyboard.rep_count);
#endif	// KEYBOARD_LOG

		mfp->KeyData(keyboard.rep_code);
	}

	// ̃CxgԂݒ
	ev->SetTime(keyboard.rep_next);
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	CN
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::MakeKey(DWORD code)
{
	ASSERT(this);
	ASSERT((code >= 0x01) && (code <= 0x73));

	// L[{[hĂΓ͂Ȃ
	if (!keyboard.connect) {
		return;
	}

	// ɃCNȂAȂ
	if (keyboard.status[code]) {
		return;
	}

#if defined(KEYBOARD_LOG)
	LOG1(Log::Normal, "CN $%02X", code);
#endif	// KEYBOARD_LOG

	// Xe[^Xݒ
	keyboard.status[code] = TRUE;

	// s[gJn
	keyboard.rep_code = code;
	keyboard.rep_count = 0;
	event.SetTime(keyboard.rep_start);

	// MFPMakef[^M
	if (keyboard.send_en && !keyboard.send_wait) {
		mfp->KeyData(keyboard.rep_code);
	}
}

//---------------------------------------------------------------------------
//
//	u[N
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::BreakKey(DWORD code)
{
	ASSERT(this);
	ASSERT((code >= 0x01) && (code <= 0x73));

	// L[{[hĂΓ͂Ȃ
	if (!keyboard.connect) {
		return;
	}

	// CNԂł邱ƂKvBɃu[NȂ牽Ȃ
	if (!keyboard.status[code]) {
		return;
	}

#if defined(KEYBOARD_LOG)
	LOG1(Log::Normal, "u[N $%02X", code);
#endif	// KEYBOARD_LOG

	// Xe[^Xݒ
	keyboard.status[code] = FALSE;

	// s[g̃L[ȂAs[g艺
	if (keyboard.rep_code == (DWORD)code) {
		keyboard.rep_code = 0;
		event.SetTime(0);
	}

	// MFPփf[^M
	code |= 0x80;
	if (keyboard.send_en && !keyboard.send_wait) {
		mfp->KeyData(code);
	}
}

//---------------------------------------------------------------------------
//
//	L[f[^MEFCg
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::SendWait(BOOL flag)
{
	ASSERT(this);

	keyboard.send_wait = flag;
}

//---------------------------------------------------------------------------
//
//	R}h
//
//---------------------------------------------------------------------------
void FASTCALL Keyboard::Command(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// L[`FbN
	ASSERT(sync);
	ASSERT(keyboard.cmdnum <= 0x10);
	ASSERT(keyboard.cmdread < 0x10);
	ASSERT(keyboard.cmdwrite < 0x10);

	// L[֒ǉ(Syncs)
	sync->Lock();
	keyboard.cmdbuf[keyboard.cmdwrite] = data;
	keyboard.cmdwrite = (keyboard.cmdwrite + 1) & 0x0f;
	keyboard.cmdnum++;
	if (keyboard.cmdnum > 0x10) {
		ASSERT(keyboard.cmdnum == 0x11);
		keyboard.cmdnum = 0x10;
		keyboard.cmdread = (keyboard.cmdwrite + 1) & 0x0f;
	}
	sync->Unlock();

	// L[{[hĂΓ͂Ȃ
	if (!keyboard.connect) {
		return;
	}

#if defined(KEYBOARD_LOG)
	LOG1(Log::Normal, "R}ho $%02X", data);
#endif	// KEYBOARD_LOG

	// fBXvC
	if (data < 0x40) {
		return;
	}

	// L[LED
	if (data >= 0x80) {
		keyboard.led = ~data;
		return;
	}

	// L[s[gJn
	if ((data & 0xf0) == 0x60) {
		keyboard.rep_start = data & 0x0f;
		keyboard.rep_start *= 100 * 1000 * 2;
		keyboard.rep_start += 200 * 1000 * 2;
		return;
	}

	// L[s[gp
	if ((data & 0xf0) == 0x70) {
		keyboard.rep_next = data & 0x0f;
		keyboard.rep_next *= (data & 0x0f);
		keyboard.rep_next *= 5 * 1000 * 2;
		keyboard.rep_next += 30 * 1000 * 2;
		return;
	}

	if ((data & 0xf0) == 0x40) {
		data &= 0x0f;
		if (data < 0x08) {
			// MSCTRL
			keyboard.msctrl = data & 0x01;
			if (keyboard.msctrl) {
				mouse->MSCtrl(TRUE, 2);
			}
			else {
				mouse->MSCtrl(FALSE, 2);
			}
			return;
		}
		else {
			// L[f[^oE֎~
			if (data & 0x01) {
				keyboard.send_en = TRUE;
			}
			else {
				keyboard.send_en = FALSE;
			}
			return;
		}
	}

	// 0x50
	data &= 0x0f;

	switch (data >> 2) {
		// fBXvCL[[h؂ւ
		case 0:
			if (data & 0x01) {
				keyboard.tv_mode = TRUE;
			}
			else {
				keyboard.tv_mode = FALSE;
			}
			return;

		// L[LED邳
		case 1:
			keyboard.bright = data & 0x03;
			return;

		// erRg[LE
		case 2:
			if (data & 0x01) {
				keyboard.tv_ctrl = TRUE;
			}
			else {
				keyboard.tv_ctrl = FALSE;
			}
			return;

		// OPT2Rg[LE
		case 3:
			if (data & 0x01) {
				keyboard.opt2_ctrl = TRUE;
			}
			else {
				keyboard.opt2_ctrl = FALSE;
			}
			return;
	}

	// ʏAɂ͂Ȃ
	ASSERT(FALSE);
}

//---------------------------------------------------------------------------
//
//	R}h擾
//
//---------------------------------------------------------------------------
BOOL Keyboard::GetCommand(DWORD& data)
{
	ASSERT(this);
	ASSERT(sync);
	ASSERT(keyboard.cmdnum <= 0x10);
	ASSERT(keyboard.cmdread < 0x10);
	ASSERT(keyboard.cmdwrite < 0x10);

	// bN
	sync->Lock();

	// f[^ȂFALSE
	if (keyboard.cmdnum == 0) {
		sync->Unlock();
		return FALSE;
	}

	// f[^擾
	data = keyboard.cmdbuf[keyboard.cmdread];
	keyboard.cmdread = (keyboard.cmdread + 1) & 0x0f;
	ASSERT(keyboard.cmdnum > 0);
	keyboard.cmdnum--;

	// AbN
	sync->Unlock();

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	R}hNA
//
//---------------------------------------------------------------------------
void Keyboard::ClrCommand()
{
	ASSERT(this);
	ASSERT(sync);
	ASSERT(keyboard.cmdnum <= 0x10);
	ASSERT(keyboard.cmdread < 0x10);
	ASSERT(keyboard.cmdwrite < 0x10);

	// bN
	sync->Lock();

	// 
	keyboard.cmdnum = 0;
	keyboard.cmdread = 0;
	keyboard.cmdwrite = 0;

	// AbN
	sync->Unlock();
}
