//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ MFP(MC68901) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "cpu.h"
#include "vm.h"
#include "log.h"
#include "event.h"
#include "schedule.h"
#include "keyboard.h"
#include "fileio.h"
#include "sync.h"
#include "mfp.h"

//===========================================================================
//
//	MFP
//
//===========================================================================
//#define MFP_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
MFP::MFP(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('M', 'F', 'P', ' ');
	dev.desc = "MFP (MC68901)";

	// JnAhXAIAhX
	memdev.first = 0xe88000;
	memdev.last = 0xe89fff;

	// SyncIuWFNg
	sync = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL MFP::Init()
{
	int i;
	char buf[0x20];

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// Sync쐬
	sync = new Sync;

	// ^C}Cxg
	for (i=0; i<4; i++) {
		timer[i].SetDevice(this);
		sprintf(buf, "Timer-%c", 'A' + i);
		timer[i].SetDesc(buf);
		timer[i].SetUser(i);
		timer[i].SetTime(0);

		// Timer-B̓CxgJEgŎg@Ȃ߁AO
		if (i != 1) {
			scheduler->AddEvent(&timer[i]);
		}
	}

	// L[{[h擾
	keyboard = (Keyboard*)vm->SearchDevice(MAKEID('K', 'E', 'Y', 'B'));

	// USARTCxg
	// 1(us)x13()x(f[eB50%)x16()x10(bit)Ŗ2400bps
	usart.SetDevice(this);
	usart.SetUser(4);
	usart.SetDesc("USART 2400bps");
	usart.SetTime(8320);
	scheduler->AddEvent(&usart);

	// Zbgł͏ȂWX^ݒ(f[^V[g3.3)
	for (i=0; i<4; i++) {
		if (i == 0) {
			// ^C}A1ɂāAVDISPSTN悤ɂ(DiskX)
			SetTDR(i, 1);
			mfp.tir[i] = 1;
		}
		else {
			// ^C}B,^C}C,^C}D0
			SetTDR(i, 0);
			mfp.tir[i] = 0;
		}
	}
	mfp.tsr = 0;
	mfp.rur = 0;

	// ZbgɃL[{[h$FFM邽߁AInitŏ
	sync->Lock();
	mfp.datacount = 0;
	mfp.readpoint = 0;
	mfp.writepoint = 0;
	sync->Unlock();

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Cleanup()
{
	ASSERT(this);

	// Sync폜
	if (sync) {
		delete sync;
		sync = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Reset()
{
	int i;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// 荞݃Rg[
	mfp.vr = 0;
	mfp.iidx = -1;
	for (i=0; i<0x10; i++) {
		mfp.ier[i] = FALSE;
		mfp.ipr[i] = FALSE;
		mfp.imr[i] = FALSE;
		mfp.isr[i] = FALSE;
		mfp.ireq[i] = FALSE;
	}

	// ^C}
	for (i=0; i<4; i++) {
		timer[i].SetTime(0);
		SetTCR(i, 0);
	}
	mfp.tbr[0] = 0;
	mfp.tbr[1] = 0;
	mfp.sram = 0;
	mfp.tecnt = 0;

	// GPIP (GPIP5͏Hx)
	mfp.gpdr = 0;
	mfp.aer = 0;
	mfp.ddr = 0;
	mfp.ber = (DWORD)~mfp.aer;
	mfp.ber ^= mfp.gpdr;
	SetGPIP(5, 1);

	// USART
	mfp.scr = 0;
	mfp.ucr = 0;
	mfp.rsr = 0;
	mfp.tsr = (DWORD)(mfp.tsr & ~0x01);
	mfp.tur = 0;

	// GPIP(d֘A)
	SetGPIP(1, 1);
	if (vm->IsPowerSW()) {
		SetGPIP(2, 0);
	}
	else {
		SetGPIP(2, 1);
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL MFP::Save(Fileio *fio, int ver)
{
	size_t sz;
	int i;

	ASSERT(this);
	LOG0(Log::Normal, "Z[u");

	// {
	sz = sizeof(mfp_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (!fio->Write(&mfp, (int)sz)) {
		return FALSE;
	}

	// Cxg(^C})
	for (i=0; i<4; i++) {
		if (!timer[i].Save(fio, ver)) {
			return FALSE;
		}
	}

	// Cxg(USART)
	if (!usart.Save(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL MFP::Load(Fileio *fio, int ver)
{
	int i;
	size_t sz;

	ASSERT(this);
	LOG0(Log::Normal, "[h");

	// {
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(mfp_t)) {
		return FALSE;
	}
	if (!fio->Read(&mfp, (int)sz)) {
		return FALSE;
	}

	// Cxg(^C})
	for (i=0; i<4; i++) {
		if (!timer[i].Load(fio, ver)) {
			return FALSE;
		}
	}

	// Cxg(USART)
	if (!usart.Load(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL MFP::ApplyCfg(const Config* /*config*/)
{
	ASSERT(this);

	LOG0(Log::Normal, "ݒKp");
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::ReadByte(DWORD addr)
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// EFCg
		scheduler->Wait(3);

		// 64oCgPʂŃ[v
		addr &= 0x3f;
		addr >>= 1;

		switch (addr) {
			// GPIP
			case 0x00:
				return mfp.gpdr;

			// AER
			case 0x01:
				return mfp.aer;

			// DDR
			case 0x02:
				return mfp.ddr;

			// IER(A)
			case 0x03:
				return GetIER(0);

			// IER(B)
			case 0x04:
				return GetIER(1);

			// IPR(A)
			case 0x05:
				return GetIPR(0);

			// IPR(B)
			case 0x06:
				return GetIPR(1);

			// ISR(A)
			case 0x07:
				return GetISR(0);

			// ISR(B)
			case 0x08:
				return GetISR(1);

			// IMR(A)
			case 0x09:
				return GetIMR(0);

			// IMR(B)
			case 0x0a:
				return GetIMR(1);

			// VR
			case 0x0b:
				return GetVR();

			// ^C}ARg[
			case 0x0c:
				return GetTCR(0);

			// ^C}BRg[
			case 0x0d:
				return GetTCR(1);

			// ^C}C&DRg[
			case 0x0e:
				data = GetTCR(2);
				data <<= 4;
				data |= GetTCR(3);
				return data;

			// ^C}Af[^
			case 0x0f:
				return GetTIR(0);

			// ^C}Bf[^
			case 0x10:
				return GetTIR(1);

			// ^C}Cf[^
			case 0x11:
				return GetTIR(2);

			// ^C}Df[^
			case 0x12:
				return GetTIR(3);

			// SYNCLN^
			case 0x13:
				return mfp.scr;

			// USARTRg[
			case 0x14:
				return mfp.ucr;

			// V[oXe[^X
			case 0x15:
				return mfp.rsr;

			// gX~b^Xe[^X
			case 0x16:
				// TErbg̓NA
				mfp.tsr = (DWORD)(mfp.tsr & ~0x40);
				return mfp.tsr;

			// USARTf[^
			case 0x17:
				Receive();
				return mfp.rur;

			// ȊO
			default:
				LOG1(Log::Warning, "WX^ǂݍ R%02d", addr);
				return 0xff;
		}
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL MFP::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		// oXG[͔Ȃ
		return;
	}

	// 64oCgPʂŃ[v
	addr &= 0x3f;
	addr >>= 1;

	// EFCg
	scheduler->Wait(3);

	switch (addr) {
		// GPIP
		case 0x00:
			// VDISPAND.BŃ`FbNꍇ(MOON FIGHTER)
			SetGPDR(data);
			return;

		// AER
		case 0x01:
			mfp.aer = data;
			mfp.ber = (DWORD)(~data);
			mfp.ber ^= mfp.gpdr;
			IntGPIP();
			return;

		// DDR
		case 0x02:
			mfp.ddr = data;
			if (mfp.ddr != 0) {
				LOG0(Log::Warning, "GPIPo̓fBNV");
			}
			return;

		// IER(A)
		case 0x03:
			SetIER(0, data);
			return;

		// IER(B)
		case 0x04:
			SetIER(1, data);
			return;

		// IPR(A)
		case 0x05:
			SetIPR(0, data);
			return;

		// IPR(B)
		case 0x06:
			SetIPR(1, data);
			return;

		// ISR(A)
		case 0x07:
			SetISR(0, data);
			return;

		// ISR(B)
		case 0x08:
			SetISR(1, data);
			return;

		// IMR(A)
		case 0x09:
			SetIMR(0, data);
			return;

		// IMR(B)
		case 0x0a:
			SetIMR(1, data);
			return;

		// VR
		case 0x0b:
			SetVR(data);
			return;

		// ^C}ARg[
		case 0x0c:
			SetTCR(0, data);
			return;

		// ^C}BRg[
		case 0x0d:
			SetTCR(1, data);
			return;

		// ^C}C&DRg[
		case 0x0e:
			SetTCR(2, (DWORD)(data >> 4));
			SetTCR(3, (DWORD)(data & 0x0f));
			return;

		// ^C}Af[^
		case 0x0f:
			SetTDR(0, data);
			return;

		// ^C}Bf[^
		case 0x10:
			SetTDR(1, data);
			return;

		// ^C}Cf[^
		case 0x11:
			SetTDR(2, data);
			return;

		// ^C}Df[^
		case 0x12:
			SetTDR(3, data);
			return;

		// SYNCLN^
		case 0x13:
			mfp.scr = data;
			return;

		// USARTRg[
		case 0x14:
			if (data != 0x88) {
				LOG1(Log::Warning, "USART p[^G[ %02X", data);
			}
			mfp.ucr = data;
			return;

		// V[oXe[^X
		case 0x15:
			SetRSR(data);
			return;

		// gX~b^Xe[^X
		case 0x16:
			SetTSR(data);
			return;

		// USARTf[^
		case 0x17:
			Transmit(data);
			return;
	}

	LOG2(Log::Warning, "WX^ R%02d <- $%02X",
							addr, data);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL MFP::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::ReadOnly(DWORD addr) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return 0xff;
	}

	// 64oCgPʂŃ[v
	addr &= 0x3f;
	addr >>= 1;

	switch (addr) {
		// GPIP
		case 0x00:
			return mfp.gpdr;

		// AER
		case 0x01:
			return mfp.aer;

		// DDR
		case 0x02:
			return mfp.ddr;

		// IER(A)
		case 0x03:
			return GetIER(0);

		// IER(B)
		case 0x04:
			return GetIER(1);

		// IPR(A)
		case 0x05:
			return GetIPR(0);

		// IPR(B)
		case 0x06:
			return GetIPR(1);

		// ISR(A)
		case 0x07:
			return GetISR(0);

		// ISR(B)
		case 0x08:
			return GetISR(1);

		// IMR(A)
		case 0x09:
			return GetIMR(0);

		// IMR(B)
		case 0x0a:
			return GetIMR(1);

		// VR
		case 0x0b:
			return GetVR();

		// ^C}ARg[
		case 0x0c:
			return mfp.tcr[0];

		// ^C}BRg[
		case 0x0d:
			return mfp.tcr[1];

		// ^C}C&DRg[
		case 0x0e:
			data = mfp.tcr[2];
			data <<= 4;
			data |= mfp.tcr[3];
			return data;

		// ^C}Af[^
		case 0x0f:
			return mfp.tir[0];

		// ^C}Bf[^(_ȒlԂ)
		case 0x10:
			return ((scheduler->GetTotalTime() % 13) + 1);

		// ^C}Cf[^
		case 0x11:
			return mfp.tir[2];

		// ^C}Df[^
		case 0x12:
			return mfp.tir[3];

		// SYNCLN^
		case 0x13:
			return mfp.scr;

		// USARTRg[
		case 0x14:
			return mfp.ucr;

		// V[oXe[^X
		case 0x15:
			return mfp.rsr;

		// gX~b^Xe[^X
		case 0x16:
			return mfp.tsr;

		// USARTf[^
		case 0x17:
			return mfp.rur;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL MFP::GetMFP(mfp_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// f[^Rs[
	*buffer = mfp;
}

//---------------------------------------------------------------------------
//
//	荞
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Interrupt(int level, BOOL enable)
{
	int index;

	ASSERT(this);
	ASSERT((level >= 0) && (level < 0x10));

	index = 15 - level;
	if (enable) {
		// ɗvĂ邩
		if (mfp.ireq[index]) {
			return;
		}

		// Cl[uWX^͂ǂ
		if (!mfp.ier[index]) {
			return;
		}

		// tOUpA荞݃`FbN
		mfp.ireq[index] = TRUE;
		IntCheck();
	}
	else {
		// ɎAv󗝂ꂽォ
		if (!mfp.ireq[index] && !mfp.ipr[index]) {
			return;
		}
		mfp.ireq[index] = FALSE;

		// yfBOA荞݉
		mfp.ipr[index] = FALSE;
		IntCheck();
	}
}

//---------------------------------------------------------------------------
//
//	荞ݗD揇ʔ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::IntCheck()
{
	int i;
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);

	// 荞݃LZ`FbN
	if (mfp.iidx >= 0) {
		// yfBO܂̓}XNŁA荞ݎ艺(f[^V[g2.7.1)
		if (!mfp.ipr[mfp.iidx] || !mfp.imr[mfp.iidx]) {
			cpu->IntCancel(6);
#if defined(MFP_LOG)
			sprintf(buffer, "荞ݎ %s", IntDesc[mfp.iidx]);
			LOG0(Log::Normal, buffer);
#endif	// MFP_LOG
			mfp.iidx = -1;
		}
	}

	// 荞ݔ
	for (i=0; i<0x10; i++) {
		// 荞݃Cl[u
		if (mfp.ier[i]) {
			// 荞݃NGXg邩
			if (mfp.ireq[i]) {
				// yfBOWX^1ɂ(荞ݎ)
				mfp.ipr[i] = TRUE;
				mfp.ireq[i] = FALSE;
			}
		}
		else {
			// Cl[uWX^0ŁA荞݃yfBO
			mfp.ipr[i] = FALSE;
			mfp.ireq[i] = FALSE;
		}
	}

	// 荞݃xN^o
	for (i=0; i<0x10; i++) {
		// 荞݃yfBO
		if (!mfp.ipr[i]) {
			continue;
		}

		// 荞݃}XN
		if (!mfp.imr[i]) {
			continue;
		}

		// T[rXłȂ
		if (mfp.isr[i]) {
			continue;
		}

		// ɗvĂ銄荞݂ʂȂA荞݂
		if (mfp.iidx > i) {
			cpu->IntCancel(6);
#if defined(MFP_LOG)
			sprintf(buffer, "荞ݗD %s", IntDesc[mfp.iidx]);
			LOG0(Log::Normal, buffer);
#endif	// MFP_LOG
			mfp.iidx = -1;
		}

		// xN^o
		if (cpu->Interrupt(6, (mfp.vr & 0xf0) + (15 - i))) {
			// CPUɎ󂯕tꂽBCfbNXL
#if defined(MFP_LOG)
			sprintf(buffer, "荞ݗv %s", IntDesc[i]);
			LOG0(Log::Normal, buffer);
#endif	// MFP_LOG
			mfp.iidx = i;
			break;
		}
	}
}

//---------------------------------------------------------------------------
//
//	荞݉
//
//---------------------------------------------------------------------------
void FASTCALL MFP::IntAck()
{
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);

	// ZbgɁACPU犄荞݂Ԉēꍇ
	if (mfp.iidx < 0) {
		LOG0(Log::Warning, "vĂȂ荞");
		return;
	}

#if defined(MFP_LOG)
	sprintf(buffer, "荞݉ %s", IntDesc[mfp.iidx]);
	LOG0(Log::Normal, buffer);
#endif	// MFP_LOG

	// 荞݂󂯕tꂽByfBO
	mfp.ipr[mfp.iidx] = FALSE;

	// CT[rX(I[gEOI0A}jAEOI1)
	if (mfp.vr & 0x08) {
		mfp.isr[mfp.iidx] = TRUE;
	}
	else {
		mfp.isr[mfp.iidx] = FALSE;
	}

	// CfbNX荞ݖɕύX
	mfp.iidx = -1;

	// ēxA荞݃`FbNs
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IERݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetIER(int offset, DWORD data)
{
	int i;
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));
	ASSERT(data < 0x100);

	// ݒ
	offset <<= 3;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		if (data & 0x80) {
			mfp.ier[i] = TRUE;
		}
		else {
			// IPR(f[^V[g4.3.1)
			mfp.ier[i] = FALSE;
			mfp.ipr[i] = FALSE;
		}

		data <<= 1;
	}

	// 荞ݗD揇
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IER擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetIER(int offset) const
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));

	// ݒ
	offset <<= 3;
	bit = 0;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		bit <<= 1;
		if (mfp.ier[i]) {
			bit |= 0x01;
		}
	}

	return bit;
}

//---------------------------------------------------------------------------
//
//	IPRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetIPR(int offset, DWORD data)
{
	int i;
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));

	// ݒ
	offset <<= 3;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		if (!(data & 0x80)) {
			// IPRCPU1ɂ邱Ƃ͏oȂ
			mfp.ipr[i] = FALSE;
		}

		data <<= 1;
	}

	// 荞ݗD揇
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IPR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetIPR(int offset) const
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));

	// ݒ
	offset <<= 3;
	bit = 0;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		bit <<= 1;
		if (mfp.ipr[i]) {
			bit |= 0x01;
		}
	}

	return bit;
}

//---------------------------------------------------------------------------
//
//	ISRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetISR(int offset, DWORD data)
{
	int i;
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));
	ASSERT(data < 0x100);

	// ݒ
	offset <<= 3;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		if (!(data & 0x80)) {
			mfp.isr[i] = FALSE;
		}

		data <<= 1;
	}

	// 荞ݗD揇
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	ISR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetISR(int offset) const
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));

	// ݒ
	offset <<= 3;
	bit = 0;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		bit <<= 1;
		if (mfp.isr[i]) {
			bit |= 0x01;
		}
	}

	return bit;
}

//---------------------------------------------------------------------------
//
//	IMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetIMR(int offset, DWORD data)
{
	int i;
#if defined(MFP_LOG)
	char buffer[0x40];
#endif	// MFP_LOG

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));
	ASSERT(data < 0x100);

	// ݒ
	offset <<= 3;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		if (data & 0x80) {
			mfp.imr[i] = TRUE;
		}
		else {
			mfp.imr[i] = FALSE;
		}

		data <<= 1;
	}

	// 荞ݗD揇
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IMR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetIMR(int offset) const
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT((offset == 0) || (offset == 1));

	// ݒ
	offset <<= 3;
	bit = 0;

	// 8Arbg
	for (i=offset; i<offset+8; i++) {
		bit <<= 1;
		if (mfp.imr[i]) {
			bit |= 0x01;
		}
	}

	return bit;
}

//---------------------------------------------------------------------------
//
//	VRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetVR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	if (mfp.vr != data) {
		mfp.vr = data;
		LOG1(Log::Detail, "荞݃xN^x[X $%02X", data & 0xf0);

		if (mfp.vr & 0x08) {
			LOG0(Log::Warning, "}jAEOI[h");
		}
	}
}

//---------------------------------------------------------------------------
//
//	VR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetVR() const
{
	ASSERT(this);
	return mfp.vr;
}

//---------------------------------------------------------------------------
//
//	荞ݖ̃e[u
//
//---------------------------------------------------------------------------
const char* MFP::IntDesc[0x10] = {
	"H-SYNC",
	"CIRQ",
	"Timer-A",
	"RxFull",
	"RxError",
	"TxEmpty",
	"TxError",
	"Timer-B",
	"(NoUse)",
	"V-DISP",
	"Timer-C",
	"Timer-D",
	"FMIRQ",
	"POW SW",
	"EXPON",
	"ALARM"
};

//---------------------------------------------------------------------------
//
//	CxgR[obN(fBC[hŎgp)
//
//---------------------------------------------------------------------------
BOOL FASTCALL MFP::Callback(Event *ev)
{
	int channel;
	DWORD low;

	ASSERT(this);
	ASSERT(ev);

	// [Uf[^ʂ𓾂
	channel = (int)ev->GetUser();

	// ^C}
	if ((channel >= 0) && (channel <= 3)) {
		low = (mfp.tcr[channel] & 0x0f);

		// ^C}I
		if (low == 0) {
			// ^C}It
			return FALSE;
		}

		// fBC[h
		if (low & 0x08) {
			// JEg0班xāA荞ݔ
			Interrupt(TimerInt[channel], TRUE);

			// Vbg
			return FALSE;
		}

		// ^C}i߂
		Proceed(channel);
		return TRUE;
	}

	// USART
	ASSERT(channel == 4);
	USART();
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	^C}Cxg(CxgJEg[hŎgp)
//
//---------------------------------------------------------------------------
void FASTCALL MFP::EventCount(int channel, int value)
{
	DWORD edge;
	BOOL flag;

	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 1));
	ASSERT((value == 0) || (value == 1));
	ASSERT((mfp.tbr[channel] == 0) || (mfp.tbr[channel] == 1));

	// CxgJEg[h(^C}A,B̂)
	if ((mfp.tcr[channel] & 0x0f) == 0x08) {
		// GPIP4, GPIP3Ō܂
		if (channel == 0) {
			edge = mfp.aer & 0x10;
		}
		else {
			edge = mfp.aer & 0x08;
		}

		// tOIt
		flag = FALSE;

		// GbW
		if (edge == 1) {
			// GbW1̂ƂA01Ń^C}i߂
			if ((mfp.tbr[channel] == 0) && (value == 1)) {
				flag = TRUE;
			}
		}
		else {
			// GbW0̂ƂA10Ń^C}i߂
			if ((mfp.tbr[channel] == 1) && (value == 0)) {
				flag = TRUE;
			}
		}

		// ^C}i߂
		if (flag) {
			Proceed(channel);
		}
	}

	// TBRXV
	mfp.tbr[channel] = (DWORD)value;
}

//---------------------------------------------------------------------------
//
//	TCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetTCR(int channel, DWORD data)
{
	DWORD prev;
	DWORD now;
	DWORD speed;

	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 3));
	ASSERT(data < 0x100);

	// ^ϊAv`FbN
	now = data;
	now &= 0x0f;
	prev = mfp.tcr[channel];
	if (now == prev) {
		return;
	}
	mfp.tcr[channel] = now;

	// ^C}B0x01̂݉(G~[VȂ)
	if (channel == 1) {
		if ((now != 0x01) && (now != 0x00)) {
			LOG1(Log::Warning, "^C}BRg[ $%02X", now);
		}
		now = 0;
	}

	// ^C}Xgbv
	if (now == 0) {
#if defined(MFP_LOG)
		LOG1(Log::Normal, "^C}%c ~", channel + 'A');
#endif	// MFP_LOG
		timer[channel].SetTime(0);

		// 荞݂̎艺sKv(hL)
		Interrupt(TimerInt[channel], FALSE);

		// Timer-DCPUNbN߂(CH30.SYS)
		if (channel == 3) {
			if (mfp.sram != 0) {
				// CPUNbN߂
				scheduler->SetCPUSpeed(mfp.sram);
				mfp.sram = 0;
			}
		}
		return;
	}

	// pX胂[h̓T|[gĂȂ
	if (now > 0x08) {
		LOG2(Log::Warning, "^C}%c pX胂[h$%02X", channel + 'A', now);
		return;
	}

	// CxgJEg[h
	if (now == 0x08) {
#if defined(MFP_LOG)
		LOG1(Log::Normal, "^C}%c CxgJEg[h", channel + 'A');
#endif	// MFP_LOG
		// xCxg~߂
		timer[channel].SetTime(0);

		// ^C}OFFONȂAJEg[h
		if (prev == 0) {
			mfp.tir[channel] = mfp.tdr[channel];
		}
		return;
	}

	// fBC[hł́AvXP[ݒ
#if defined(MFP_LOG)
	LOG3(Log::Normal, "^C}%c fBC[h %d.%dus",
				channel + 'A', (TimerHus[now] / 2), (TimerHus[now] & 1) * 5);
#endif	// MFP_LOG

	// ^C}OFFONȂAJE^[h(_VDISPSTpb`)
	if (prev == 0) {
		mfp.tir[channel] = mfp.tdr[channel];
	}

	// CxgZbg
	timer[channel].SetTime(TimerHus[now]);

	// Timer-C̏ꍇAINFO.RAMp΍s(CPUxv𖳗肠킹)
	if (channel == 2) {
		if ((now == 3) && (mfp.sram == 0)) {
			speed = cpu->GetPC();
			if ((speed >= 0xed0100) && (speed <= 0xedffff)) {
				// CPUNbN𗎂Ƃ
				speed = scheduler->GetCPUSpeed();
				mfp.sram = speed;
				speed *= 83;
				speed /= 96;
				scheduler->SetCPUSpeed(speed);
			}
		}
		if ((now != 3) && (mfp.sram != 0)) {
			// CPUNbN߂
			scheduler->SetCPUSpeed(mfp.sram);
			mfp.sram = 0;
		}
	}

	// Timer-D̏ꍇACH30.SYSp΍s(CPUxv𖳗肠킹)
	if (channel == 3) {
		if ((now == 7) && (mfp.sram == 0)) {
			speed = cpu->GetPC();
			if ((speed >= 0xed0100) && (speed <= 0xedffff)) {
				// CPUNbN𗎂Ƃ
				speed = scheduler->GetCPUSpeed();
				mfp.sram = speed;
				speed *= 85;
				speed /= 96;
				scheduler->SetCPUSpeed(speed);
			}
		}
	}
}

//---------------------------------------------------------------------------
//
//	TCR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetTCR(int channel) const
{
	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 3));

	return mfp.tcr[channel];
}

//---------------------------------------------------------------------------
//
//	TDRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetTDR(int channel, DWORD data)
{
	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 3));
	ASSERT(data < 0x100);

	mfp.tdr[channel] = data;

	// ^C}B͌Œl̂͂
	if (channel == 1) {
		if (data != 0x0d) {
			LOG1(Log::Warning, "^C}B[hl %02X", data);
		}
	}
}

//---------------------------------------------------------------------------
//
//	TIR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MFP::GetTIR(int channel) const
{
	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 3));

	// ^C}BXM6ł͓ǂݏo֎~(@1us x 14^C})
	if (channel == 1) {
		// (`)
		LOG0(Log::Warning, "^C}B f[^WX^ǂݏo");
		return (DWORD)((scheduler->GetTotalTime() % 13) + 1);
	}

	return mfp.tir[channel];
}

//---------------------------------------------------------------------------
//
//	^C}i߂
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Proceed(int channel)
{
	ASSERT(this);
	ASSERT((channel >= 0) && (channel <= 3));

	// JE^Z
	if (mfp.tir[channel] > 0) {
		mfp.tir[channel]--;
	}
	else {
		mfp.tir[channel] = 0xff;
	}

	// 0ɂȂ烊[hA荞
	if (mfp.tir[channel] == 0) {
#if defined(MFP_LOG)
	LOG1(Log::Normal, "^C}%c I[o[t[", channel + 'A');
#endif	// MFP_LOG

		// [h
		mfp.tir[channel] = mfp.tdr[channel];

		// CxgJEg[h͊荞݃Cxg(XsfBW[II)
		if (mfp.tcr[channel] == 0x08) {
			// GPIPύXɍsAxĊ荞݂o悤ɂ
			// ^IWEX(12ł̓_Av)
			timer[channel].SetTime(36);
		}
		else {
			// ʏ튄荞
			Interrupt(TimerInt[channel], TRUE);
		}
	}
}

//---------------------------------------------------------------------------
//
//	^C}荞݃e[u
//
//---------------------------------------------------------------------------
const int MFP::TimerInt[4] = {
	13,									// Timer-A
	8,									// Timer-B
	5,									// Timer-C
	4									// Timer-D
};

//---------------------------------------------------------------------------
//
//	^C}ԃe[u
//
//---------------------------------------------------------------------------
const DWORD MFP::TimerHus[8] = {
	0,									// ^C}Xgbv
	2,									// 1.0us
	5,									// 2.5us
	8,									// 4us
	25,									// 12.5us
	32,									// 16us
	50,									// 25us
	100									// 50us
};

//---------------------------------------------------------------------------
//
//	GPDRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetGPDR(DWORD data)
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT(data < 0x100);

	// DDR1̃rbĝݗL
	for (i=0; i<8; i++) {
		bit = (DWORD)(1 << i);
		if (mfp.ddr & bit) {
			if (data & bit) {
				SetGPIP(i, 1);
			}
			else {
				SetGPIP(i, 0);
			}
		}
	}
}

//---------------------------------------------------------------------------
//
//	GPIPݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetGPIP(int num, int value)
{
	DWORD data;

	ASSERT(this);
	ASSERT((num >= 0) && (num < 8));
	ASSERT((value == 0) || (value == 1));

	// obNAbv
	data = mfp.gpdr;

	// rbg쐬
	mfp.gpdr &= (DWORD)(~(1 << num));
	if (value == 1) {
		mfp.gpdr |= (DWORD)(1 << num);
	}

	// ĂΊ荞݃`FbN
	if (mfp.gpdr != data) {
		IntGPIP();
	}
}

//---------------------------------------------------------------------------
//
//	GPIP荞݃`FbN
//
//---------------------------------------------------------------------------
void FASTCALL MFP::IntGPIP()
{
	DWORD data;
	int i;

	ASSERT(this);

	// Inside68k̋Lq͋tIMFPf[^V[gɏ]Ǝ̗lB
	// AER0 1->0Ŋ荞
	// AER1 0->1Ŋ荞

	// ~AERGPDRXOR
	data = (DWORD)(~mfp.aer);
	data ^= (DWORD)mfp.gpdr;

	// BERāA10ɕωƂ荞ݔ
	// (pX^C}GPIP4,GPIP3͊荞ݏȂ)
	for (i=0; i<8; i++) {
		if (data & 0x80) {
			if (!(mfp.ber & 0x80)) {
				if (i == 3) {
					// GPIP4B^C}A`FbN
					if ((mfp.tcr[0] & 0x0f) > 0x08) {
						data <<= 1;
						mfp.ber <<= 1;
						continue;
					}
				}
				if (i == 4) {
					// GPIP3B^C}B`FbN
					if ((mfp.tcr[1] & 0x0f) > 0x08) {
						data <<= 1;
						mfp.ber <<= 1;
						continue;
					}
				}

				// 荞ݗv(SORCERIAN X1->88)
				Interrupt(GPIPInt[i], TRUE);
			}
		}

		// 
		data <<= 1;
		mfp.ber <<= 1;
	}

	// BER쐬
	mfp.ber = (DWORD)(~mfp.aer);
	mfp.ber ^= mfp.gpdr;
}

//---------------------------------------------------------------------------
//
//	GPIP荞݃e[u
//
//---------------------------------------------------------------------------
const int MFP::GPIPInt[8] = {
	15,									// H-SYNC
	14,									// CIRQ
	7,									// (NoUse)
	6,									// V-DISP
	3,									// OPM
	2,									// POWER
	1,									// EXPON
	0									// ALARM
};

//---------------------------------------------------------------------------
//
//	V[oXe[^Xݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetRSR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// RÊݐݒ\
	data &= 0x01;

	mfp.rsr &= ~0x01;
	mfp.rsr |= (DWORD)(mfp.rsr | data);
}

//---------------------------------------------------------------------------
//
//	CPUMFP M
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Receive()
{
	ASSERT(this);

	// ^C}B`FbN
	if (mfp.tcr[1] != 0x01) {
		return;
	}
	if (mfp.tdr[1] != 0x0d) {
		return;
	}

	// USARTRg[`FbN
	if (mfp.ucr != 0x88) {
		return;
	}

	// V[ofBZ[ȕꍇAȂ
	if (!(mfp.rsr & 0x01)) {
		return;
	}

	// BF=1̏ꍇAɈf[^͖
	if (!(mfp.rsr & 0x80)) {
		return;
	}

	// f[^BBFAOE0ɐݒ
	mfp.rsr &= (DWORD)~0xc0;
}

//---------------------------------------------------------------------------
//
//	gX~b^Xe[^Xݒ
//
//---------------------------------------------------------------------------
void FASTCALL MFP::SetTSR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// BE,UE,END͒ڃNAłȂ
	mfp.tsr = (DWORD)(mfp.tsr & 0xd0);
	data &= (DWORD)~0xd0;
	mfp.tsr = (DWORD)(mfp.tsr | data);

	// TE=1ŁAUE,ENDNA
	if (mfp.tsr & 0x01) {
		mfp.tsr = (DWORD)(mfp.tsr & ~0x50);
	}
}

//---------------------------------------------------------------------------
//
//	CPUMFP M
//
//---------------------------------------------------------------------------
void FASTCALL MFP::Transmit(DWORD data)
{
	ASSERT(this);

	// ^C}B`FbN
	if (mfp.tcr[1] != 0x01) {
		return;
	}
	if (mfp.tdr[1] != 0x0d) {
		return;
	}

	// USARTRg[`FbN
	if (mfp.ucr != 0x88) {
		return;
	}

	// gX~b^fBZ[ȕꍇAȂ
	if (!(mfp.tsr & 0x01)) {
		return;
	}

	// Xe[^Xyуf[^Zbg
	mfp.tsr = (DWORD)(mfp.tsr & ~0x80);
	mfp.tur = data;
#if defined(MFP_LOG)
	LOG1(Log::Normal, "USARTMf[^t %02X", (BYTE)data);
#endif	// MFP_LOG
	return;
}

//---------------------------------------------------------------------------
//
//	USART
//
//---------------------------------------------------------------------------
void FASTCALL MFP::USART()
{
	ASSERT(this);
	ASSERT((mfp.readpoint >= 0) && (mfp.readpoint < 0x10));
	ASSERT((mfp.writepoint >= 0) && (mfp.writepoint < 0x10));
	ASSERT((mfp.datacount >= 0) && (mfp.datacount <= 0x10));

	// ^C}yUSARTݒ`FbN
	if (mfp.tcr[1] != 0x01) {
		return;
	}
	if (mfp.tdr[1] != 0x0d) {
		return;
	}
	if (mfp.ucr != 0x88) {
		return;
	}

	//
	//	M
	//

	if (!(mfp.tsr & 0x80)) {
		// ŃgX~b^fBZ[uȂAEND
		if (!(mfp.tsr & 0x01)) {
			mfp.tsr = (DWORD)(mfp.tsr & ~0x80);
			mfp.tsr = (DWORD)(mfp.tsr | 0x10);
			LOG0(Log::Warning, "USART MIG[");
			Interrupt(9, TRUE);
			return;
		}

		// obt@GveBAI[g^[AEh
		mfp.tsr = (DWORD)(mfp.tsr | 0x80);
		if (mfp.tsr & 0x20) {
			mfp.tsr = (DWORD)(mfp.tsr & ~0x20);
			SetRSR((DWORD)(mfp.rsr | 0x01));
		}

		// L[{[hփf[^oAMobt@GveB荞
#if defined(MFP_LOG)
		LOG1(Log::Normal, "USARTM %02X", mfp.tur);
#endif	// MFP_LOG
		keyboard->Command(mfp.tur);
		Interrupt(10, TRUE);
	}
	else {
		if (!(mfp.tsr & 0x40)) {
			mfp.tsr = (DWORD)(mfp.tsr | 0x40);
			Interrupt(9, TRUE);
#if defined(MFP_LOG)
			LOG0(Log::Normal, "USART A_[G[");
#endif	// MPF_LOG
		}
	}

	//
	//	M
	//

	// LȎMf[^Ȃ΁AȂ
	if (mfp.datacount == 0) {
		return;
	}

	// V[ofBZ[uȂAȂ
	if (!(mfp.rsr & 0x01)) {
		return;
	}

	// ŃbN
	sync->Lock();

	// ɎMf[^΁AI[o[Ƃăf[^̂Ă
	if (mfp.rsr & 0x80) {
		mfp.rsr |= 0x40;
		mfp.readpoint = (mfp.readpoint + 1) & 0x0f;
		mfp.datacount--;
		sync->Unlock();

		LOG0(Log::Warning, "USART I[o[G[");
		Interrupt(11, TRUE);
		return;
	}

	// f[^MBBFZbgAf[^L
	mfp.rsr |= 0x80;
	mfp.rur = mfp.buffer[mfp.readpoint];
	mfp.readpoint = (mfp.readpoint + 1) & 0x0f;
	mfp.datacount--;
	sync->Unlock();

	Interrupt(12, TRUE);
}

//---------------------------------------------------------------------------
//
//	L[f[^M
//
//---------------------------------------------------------------------------
void FASTCALL MFP::KeyData(DWORD data)
{
	ASSERT(this);
	ASSERT((mfp.readpoint >= 0) && (mfp.readpoint < 0x10));
	ASSERT((mfp.writepoint >= 0) && (mfp.writepoint < 0x10));
	ASSERT((mfp.datacount >= 0) && (mfp.datacount <= 0x10));
	ASSERT(data < 0x100);

	// bN
	sync->Lock();

	// ݃|Cg֊i[
	mfp.buffer[mfp.writepoint] = data;

	// ݃|CgړAJE^{P
	mfp.writepoint = (mfp.writepoint + 1) & 0x0f;
	mfp.datacount++;
	if (mfp.datacount > 0x10) {
		mfp.datacount = 0x10;
	}

	// SyncAbN
	sync->Unlock();
}
