//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2005 ohD(ytanaka@ipc-tokai.or.jp)
//	[ MIDI(YM3802) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "sync.h"
#include "config.h"
#include "fileio.h"
#include "midi.h"

//===========================================================================
//
//	MIDI
//
//===========================================================================
//#define MIDI_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
MIDI::MIDI(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('M', 'I', 'D', 'I');
	dev.desc = "MIDI (YM3802)";

	// JnAhXAIAhX
	memdev.first = 0xeae000;
	memdev.last = 0xeaffff;

	// IuWFNg
	sync = NULL;

	// Mobt@AMobt@
	midi.transbuf = NULL;
	midi.recvbuf = NULL;

	// [Nꕔ(Ȑffp)
	midi.transnum = 0;
	midi.transread = 0;
	midi.transwrite = 0;
	midi.recvnum = 0;
	midi.recvread = 0;
	midi.recvwrite = 0;
	midi.normnum = 0;
	midi.normread = 0;
	midi.normwrite = 0;
	midi.rtnum = 0;
	midi.rtread = 0;
	midi.rtwrite = 0;
	midi.stdnum = 0;
	midi.stdread = 0;
	midi.stdwrite = 0;
	midi.rrnum = 0;
	midi.rrread = 0;
	midi.rrwrite = 0;
	midi.bid = 0;
	midi.ilevel = 4;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL MIDI::Init()
{
	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// Sync쐬
	ASSERT(!sync);
	sync = new Sync;
	ASSERT(sync);

	// Mobt@m
	try {
		midi.transbuf = new mididata_t[TransMax];
	}
	catch (...) {
		midi.transbuf = NULL;
		return FALSE;
	}
	if (!midi.transbuf) {
		return FALSE;
	}

	// Mobt@m
	try {
		midi.recvbuf = new mididata_t[RecvMax];
	}
	catch (...) {
		midi.recvbuf = NULL;
		return FALSE;
	}
	if (!midi.recvbuf) {
		return FALSE;
	}

	// Cxg
	event[0].SetDevice(this);
	event[0].SetDesc("MIDI 31250bps");
	event[0].SetUser(0);
	event[0].SetTime(0);
	event[1].SetDevice(this);
	event[1].SetDesc("Clock");
	event[1].SetUser(1);
	event[1].SetTime(0);
	event[2].SetDevice(this);
	event[2].SetDesc("General Timer");
	event[2].SetUser(2);
	event[2].SetTime(0);

	// {[hȂA荞݃x4
	midi.bid = 0;
	midi.ilevel = 4;

	// Mxꎞ0ms
	recvdelay = 0;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Cleanup()
{
	ASSERT(this);

	// Mobt@폜
	if (midi.recvbuf) {
		delete[] midi.recvbuf;
		midi.recvbuf = NULL;
	}

	// Mobt@폜
	if (midi.transbuf) {
		delete[] midi.transbuf;
		midi.transbuf = NULL;
	}

	// Sync폜
	if (sync) {
		delete sync;
		sync = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Reset()
{
	int i;

	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// ANZXȂ
	midi.access = FALSE;

	// Zbg
	midi.reset = TRUE;

	// WX^Zbg
	ResetReg();

	// Cxg
	for (i=0; i<3; i++) {
		event[i].SetTime(0);
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL MIDI::Save(Fileio *fio, int ver)
{
	size_t sz;
	int i;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(midi_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&midi, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	for (i=0; i<3; i++) {
		if (!event[i].Save(fio, ver)) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL MIDI::Load(Fileio *fio, int ver)
{
	midi_t bk;
	size_t sz;
	int i;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// ݂̃[Nۑ
	bk = midi;

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(midi_t)) {
		return FALSE;
	}

	// ̂[h
	if (!fio->Read(&midi, (int)sz)) {
		return FALSE;
	}

	// Cxg[h
	for (i=0; i<3; i++) {
		if (!event[i].Load(fio, ver)) {
			return FALSE;
		}
	}

	// Cxg𓮓IɒǉE폜
	if (midi.bid == 0) {
		// Cxg폜
		if (scheduler->HasEvent(&event[0])) {
			scheduler->DelEvent(&event[0]);
			scheduler->DelEvent(&event[1]);
			scheduler->DelEvent(&event[2]);
		}
	}
	else {
		// Cxgǉ
		if (!scheduler->HasEvent(&event[0])) {
			scheduler->AddEvent(&event[0]);
			scheduler->AddEvent(&event[1]);
			scheduler->AddEvent(&event[2]);
		}
	}

	// Mobt@𕜋A
	midi.transbuf = bk.transbuf;
	midi.transread = bk.transread;
	midi.transwrite = bk.transwrite;
	midi.transnum = bk.transnum;

	// Mobt@𕜋A
	midi.recvbuf = bk.recvbuf;
	midi.recvread = bk.recvread;
	midi.recvwrite = bk.recvwrite;
	midi.recvnum = bk.recvnum;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");

	if (midi.bid != (DWORD)config->midi_bid) {
		midi.bid = (DWORD)config->midi_bid;

		// Cxg𓮓IɒǉE폜
		if (midi.bid == 0) {
			// Cxg폜
			if (scheduler->HasEvent(&event[0])) {
				scheduler->DelEvent(&event[0]);
				scheduler->DelEvent(&event[1]);
				scheduler->DelEvent(&event[2]);
			}
		}
		else {
			// Cxgǉ
			if (!scheduler->HasEvent(&event[0])) {
				scheduler->AddEvent(&event[0]);
				scheduler->AddEvent(&event[1]);
				scheduler->AddEvent(&event[2]);
			}
		}
	}
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::AssertDiag() const
{
	// {NX
	MemDevice::AssertDiag();

	ASSERT(this);
	ASSERT(GetID() == MAKEID('M', 'I', 'D', 'I'));
	ASSERT(memdev.first == 0xeae000);
	ASSERT(memdev.last == 0xeaffff);
	ASSERT((midi.bid >= 0) && (midi.bid < 2));
	ASSERT((midi.ilevel == 2) || (midi.ilevel == 4));
	ASSERT(midi.transnum <= TransMax);
	ASSERT(midi.transread < TransMax);
	ASSERT(midi.transwrite < TransMax);
	ASSERT(midi.recvnum <= RecvMax);
	ASSERT(midi.recvread < RecvMax);
	ASSERT(midi.recvwrite < RecvMax);
	ASSERT(midi.normnum <= 16);
	ASSERT(midi.normread < 16);
	ASSERT(midi.normwrite < 16);
	ASSERT(midi.rtnum <= 4);
	ASSERT(midi.rtread < 4);
	ASSERT(midi.rtwrite < 4);
	ASSERT(midi.stdnum <= 0x80);
	ASSERT(midi.stdread < 0x80);
	ASSERT(midi.stdwrite < 0x80);
	ASSERT(midi.rrnum <= 4);
	ASSERT(midi.rrread < 4);
	ASSERT(midi.rrwrite < 4);
}
#endif	// NDEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::ReadByte(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	switch (midi.bid) {
		// {[h
		case 0:
			break;

		// {[hID1
		case 1:
			// ͈͓
			if ((addr >= 0xeafa00) && (addr < 0xeafa10)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return 0xff;
				}

				// WX^[h
				addr -= 0xeafa00;
				addr >>= 1;
				return ReadReg(addr);
			}
			break;

		// {[hID2
		case 2:
			// ͈͓
			if ((addr >= 0xeafa10) && (addr < 0xeafa20)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return 0xff;
				}

				// WX^[h
				addr -= 0xeafa10;
				addr >>= 1;
				return ReadReg(addr);
			}
			break;

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// oXG[
	cpu->BusErr(addr, TRUE);

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT_DIAG();

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	switch (midi.bid) {
		// {[h
		case 0:
			break;

		// {[hID1
		case 1:
			// ͈͓
			if ((addr >= 0xeafa00) && (addr < 0xeafa10)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return;
				}

				// WX^[h
				addr -= 0xeafa00;
				addr >>= 1;
				WriteReg(addr, data);
				return;
			}
			break;

		// {[hID2
		case 2:
			// ͈͓
			if ((addr >= 0xeafa10) && (addr < 0xeafa20)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return;
				}

				// WX^[h
				addr -= 0xeafa10;
				addr >>= 1;
				WriteReg(addr, data);
				return;
			}
			break;

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// oXG[
	cpu->BusErr(addr, FALSE);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);
	ASSERT_DIAG();

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::ReadOnly(DWORD addr) const
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	switch (midi.bid) {
		// {[h
		case 0:
			break;

		// {[hID1
		case 1:
			// ͈͓
			if ((addr >= 0xeafa00) && (addr < 0xeafa10)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return 0xff;
				}

				// WX^[h
				addr -= 0xeafa00;
				addr >>= 1;
				return ReadRegRO(addr);
			}
			break;

		// {[hID2
		case 2:
			// ͈͓
			if ((addr >= 0xeafa10) && (addr < 0xeafa20)) {
				// AhX̂݃fR[hĂ
				if ((addr & 1) == 0) {
					return 0xff;
				}

				// WX^[h
				addr -= 0xeafa10;
				addr >>= 1;
				return ReadRegRO(addr);
			}
			break;

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	MIDIANeBu`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL MIDI::IsActive() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// {[hID0ȊOł
	if (midi.bid != 0) {
		// ZbgA1ȏANZXĂ邱
		if (midi.access) {
			return TRUE;
		}
	}

	// ANeBułȂ
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL MIDI::Callback(Event *ev)
{
	DWORD mtr;
	DWORD hus;

	ASSERT(this);
	ASSERT(ev);
	ASSERT_DIAG();

	// ANZXtOFALSEȂAMIDIAvP[V݂͑Ȃ
	if (!midi.access) {
		// ̂߁A擪Ń`FbN
		return TRUE;
	}

	// p[^ɂĐU蕪
	switch (ev->GetUser()) {
		// MIDIM 31250bps
		case 0:
			// MёM
			Receive();
			Transmit();
			return TRUE;

		// MIDINbN(Ԃ)
		case 1:
			// MTRZoĂ
			if (midi.mtr <= 1) {
				mtr = 0x40000;
			}
			else {
				mtr = midi.mtr << 4;
			}

			// R[fBOJE^(8bitJE^)
			midi.srr++;
			midi.srr &= 0xff;
			if (midi.srr == 0) {
				// 荞
				Interrupt(3, TRUE);
			}

			// vCobNJE^(15bit_EJE^)
			if (midi.str != 0xffff8000) {
				midi.str--;
			}
			if (midi.str >= 0xffff8000) {
				// 荞
				Interrupt(2, TRUE);
			}

			// NbNԃJE^fNgA0`FbN
			midi.sct--;
			if (midi.sct != 0) {
				// SCT=1̏ꍇAZɂ؎̂Ă₤߁A␳
				if (midi.sct == 1) {
					hus = mtr - (mtr / midi.scr) * (midi.scr - 1);
					if (event[1].GetTime() != hus) {
						event[1].SetTime(hus);
					}
				}
				return TRUE;
			}
			midi.sct = midi.scr;

			// NbN
			Clock();

			// ML
			if (midi.rcr & 1) {
				// NbN^C}g
				if ((midi.dmr & 0x07) == 0x03) {
					if (midi.dmr & 0x08) {
						// A^CMobt@֑}
						InsertRR(0xf8);
					}
				}
			}

			// obt@`FbN
			CheckRR();

			// ^C}l͖񃍁[h(Mu-1)
			mtr /= midi.scr;
			if (ev->GetTime() != mtr) {
				ev->SetTime(mtr);
			}
			return TRUE;

		// ėp^C}
		case 2:
			General();
			return TRUE;
	}

	// ȊO̓G[Bł؂
	ASSERT(FALSE);
	ev->SetTime(0);
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MR[obN
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Receive()
{
	BOOL recv;
	mididata_t *p;
	DWORD diff;
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// V[oREADY
	midi.rsr &= ~0x01;

	// MfBZ[uȂ牽Ȃ
	if ((midi.rcr & 1) == 0) {
		// MȂ
		return;
	}

	// Mp[^`FbN
	if (midi.rrr != 0x08) {
		// 31250bpsłȂ
		return;
	}
	if ((midi.rmr & 0x32) != 0) {
		// 8bitLN^A1XgbvrbgAmpeBłȂ
		return;
	}

	// MtOOFF
	recv = FALSE;

	// Mobt@̃`FbN
	sync->Lock();
	if (midi.recvnum > 0) {
		// Lȃf[^Bԍ݂
		diff = scheduler->GetTotalTime();
		p = &midi.recvbuf[midi.recvread];
		diff -= p->vtime;
		if (diff > recvdelay) {
			// M
			recv = TRUE;
		}
	}
	sync->Unlock();

	// MȂ̂ł΁AMȂԂ
	if (!recv) {
		if (midi.rcn == 0x3ff) {
			// 327.68msMȂȂ̂ŁAItCƊ荞
			midi.rsr |= 0x04;
			if ((midi.imr & 4) == 0) {
				Interrupt(4, TRUE);
			}
		}
		midi.rcn++;
		if (midi.rcn >= 0x400) {
			midi.rcn = 0;
		}

		// MȂ
		return;
	}

	// MJE^Zbg
	midi.rcn = 0;

	// Mobt@f[^荞
	sync->Lock();
	data = midi.recvbuf[midi.recvread].data;
	midi.recvread = (midi.recvread + 1) & (RecvMax - 1);
	midi.recvnum--;
	sync->Unlock();

	if (data == 0xf8) {
		// A^CMobt@֑}
		if (midi.dmr & 0x08) {
			InsertRR(0xf8);

			// NbN
			Clock();

			// obt@`FbN
			CheckRR();
		}

		// NbNtB^
		if (midi.rcr & 0x10) {
#if defined(MIDI_LOG)
			LOG0(Log::Normal, "MIDINbNtB^ $F8XLbv");
#endif	// MIDI_LOG
			return;
		}
	}
	if ((data >= 0xf9) && (data <= 0xfd)) {
		// A^CMobt@֑}
		if (midi.dmr & 0x08) {
			InsertRR(data);
		}
	}

	// AhXn^
	if (midi.rcr & 2) {
		if (data == 0xf0) {
			// [JID`FbN
			midi.asr = 1;
		}
		if (data == 0xf7) {
			// AhXn^I
			midi.asr = 0;
		}
		switch (midi.asr) {
			// [JID`FbN
			case 1:
				if ((midi.amr & 0x7f) == data) {
#if defined(MIDI_LOG)
					LOG1(Log::Normal, "[JID}b` $%02X", midi.amr & 0x7f);
#endif	// MIDI_LOG
					midi.asr = 2;
				}
				else {
#if defined(MIDI_LOG)
					LOG1(Log::Normal, "[JIDA}b` $%02X", data);
#endif	// MIDI_LOG
					midi.asr = 3;
				}
				break;

			// foCXID`FbN
			case 2:
				if (midi.amr & 0x80) {
					if ((midi.adr & 0x7f) != data) {
#if defined(MIDI_LOG)
						LOG1(Log::Normal, "foCXIDA}b` $%02X", data);
#endif	// MIDI_LOG
						midi.asr = 3;
					}
				}
				break;

			default:
				break;
		}
		if (midi.asr == 3) {
			// AhXn^쒆BXLbv
			return;
		}
	}
	else {
		// AhXn^ fBZ[u
		midi.asr = 0;
	}

	// V[oBUSY
	midi.rsr |= 0x01;

	// Wobt@
	InsertStd(data);
}

//---------------------------------------------------------------------------
//
//	MR[obN
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Transmit()
{
	ASSERT(this);
	ASSERT_DIAG();

	// MReady
	midi.tbs = FALSE;

	// A^CMobt@̃`FbN
	if (midi.rtnum > 0) {
		// A^CMobt@̃f[^𑗐Mobt@֓]
		ASSERT(midi.rtnum <= 4);
		if (midi.tcr & 1) {
			InsertTrans(midi.rtbuf[midi.rtread]);
		}
		midi.rtnum--;
		midi.rtread = (midi.rtread + 1) & 3;

		// MBUSYAMȂ
		midi.tbs = TRUE;
		midi.tcn = 0;
		return;
	}

	// ʏobt@̃`FbN
	if (midi.normnum > 0) {
		// ʏobt@̃f[^𑗐Mobt@֓]
		ASSERT(midi.normnum <= 16);
		if (midi.tcr & 1) {
			InsertTrans(midi.normbuf[midi.normread]);
		}
		midi.normnum--;
		midi.normread = (midi.normread + 1) & 15;

		// ʏobt@NA΁AMobt@GveB荞
		if (midi.normnum == 0) {
			Interrupt(6, TRUE);
		}

		// MBUSYAMȂ
		midi.tbs = TRUE;
		midi.tcn = 0;
		return;
	}

	// ͑MȂBMԂ
	if (midi.tcn == 0xff) {
		// 81.92msM
		if (midi.tcr & 1) {
			if (midi.dmr & 0x20) {
				// ANeBuZVOM(DMR̃rbgwɂ)
				InsertRT(0xfe);
			}
		}
	}
	midi.tcn++;
	if (midi.tcn >= 0x100) {
		midi.tcn = 0x100;
	}
}

//---------------------------------------------------------------------------
//
//	MIDINbNo
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Clock()
{
	ASSERT(this);
	ASSERT_DIAG();

	// NbNJE^
	midi.ctr--;
	if (midi.ctr == 0) {
		// [h
		if (midi.cdr == 0) {
			midi.ctr = 0x80;
		}
		else {
			midi.ctr = midi.cdr;
		}

		// 荞
		if ((midi.imr & 8) == 0) {
			Interrupt(1, TRUE);
		}
	}
}

//---------------------------------------------------------------------------
//
//	ėp^C}R[obN
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::General()
{
	DWORD hus;

	ASSERT(this);
	ASSERT_DIAG();

	// 荞ݔ
	Interrupt(7, TRUE);

	// GTRωĂ΁AԂĐݒ
	hus = midi.gtr << 4;
	if (event[2].GetTime() != hus) {
		event[2].SetTime(hus);
	}
}

//---------------------------------------------------------------------------
//
//	Mobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertTrans(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "Mobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.transbuf[midi.transwrite].data = data;
	midi.transbuf[midi.transwrite].vtime = scheduler->GetTotalTime();
	midi.transwrite = (midi.transwrite + 1) & (TransMax - 1);
	midi.transnum++;
	if (midi.transnum > TransMax) {
		// Mobt@̃I[o[t[͖(I/OG~[VsȂꍇ)
		midi.transnum = TransMax;
		midi.transread = midi.transwrite;
	}

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	Mobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertRecv(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "Mobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.recvbuf[midi.recvwrite].data = data;
	midi.recvbuf[midi.recvwrite].vtime = scheduler->GetTotalTime();
	midi.recvwrite = (midi.recvwrite + 1) & (RecvMax - 1);
	midi.recvnum++;
	if (midi.recvnum > RecvMax) {
		LOG0(Log::Warning, "Mobt@ I[o[t[");
		midi.recvnum = RecvMax;
		midi.recvread = midi.recvwrite;
	}

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	ʏobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertNorm(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "ʏobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.normbuf[midi.normwrite] = data;
	midi.normwrite = (midi.normwrite + 1) & 15;
	midi.normnum++;
	midi.normtotal++;
	if (midi.normnum > 16) {
		LOG0(Log::Warning, "ʏobt@ I[o[t[");
		midi.normnum = 16;
		midi.normread = midi.normwrite;
	}

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	A^CMobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertRT(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "A^CMobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.rtbuf[midi.rtwrite] = data;
	midi.rtwrite = (midi.rtwrite + 1) & 3;
	midi.rtnum++;
	midi.rttotal++;
	if (midi.rtnum > 4) {
		LOG0(Log::Warning, "A^CMobt@ I[o[t[");
		midi.rtnum = 4;
		midi.rtread = midi.rtwrite;
	}

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	ʃobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertStd(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "ʃobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.stdbuf[midi.stdwrite] = data;
	midi.stdwrite = (midi.stdwrite + 1) & 0x7f;
	midi.stdnum++;
	midi.stdtotal++;
	if (midi.stdnum > 0x80) {
#if defined(MIDI_LOG)
		LOG0(Log::Warning, "ʃobt@ I[o[t[");
#endif	// MIDI_LOG
		midi.stdnum = 0x80;
		midi.stdread = midi.stdwrite;

		// MXe[^Xݒ(I[o[t[)
		midi.rsr |= 0x40;
	}

	// AbN
	sync->Unlock();

	// MXe[^Xݒ(Mf[^)
	midi.rsr |= 0x80;

	// 荞ݔ(GveBMf[^)
	if (midi.stdnum == 1) {
		Interrupt(5, TRUE);
	}
}

//---------------------------------------------------------------------------
//
//	A^CMobt@֑}
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::InsertRR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "A^CMobt@} $%02X", data);
#endif	// MIDI_LOG

	// bN
	sync->Lock();

	// f[^}
	midi.rrbuf[midi.rrwrite] = data;
	midi.rrwrite = (midi.rrwrite + 1) & 3;
	midi.rrnum++;
	midi.rrtotal++;
	if (midi.rrnum > 4) {
#if defined(MIDI_LOG)
		LOG0(Log::Warning, "A^CMobt@ I[o[t[");
#endif	// MIDI_LOG
		midi.rrnum = 4;
		midi.rrread = midi.rrwrite;
	}

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	Mobt@擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetTransNum() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// 4oCgf[^ǂ݂̂߁AbNȂ
	// (1̃oXANZXŃf[^m肷邱ƂO)
	return  midi.transnum;
}

//---------------------------------------------------------------------------
//
//	Mobt@f[^擾
//
//---------------------------------------------------------------------------
const MIDI::mididata_t* FASTCALL MIDI::GetTransData(DWORD proceed)
{
	const mididata_t *ptr;
	int offset;

	ASSERT(this);
	ASSERT(proceed < TransMax);
	ASSERT_DIAG();

	// bN
	sync->Lock();

	// 擾
	offset = midi.transread;
	offset += proceed;
	offset &= (TransMax - 1);
	ptr = &(midi.transbuf[offset]);

	// AbN
	sync->Unlock();

	// Ԃ
	return ptr;
}

//---------------------------------------------------------------------------
//
//	Mobt@폜
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::DelTransData(DWORD number)
{
	ASSERT(this);
	ASSERT(number < TransMax);
	ASSERT_DIAG();

	// bN
	sync->Lock();

	// 
	if (number > midi.transnum) {
		number = midi.transnum;
	}

	// read|C^ύX
	midi.transnum -= number;
	midi.transread = (midi.transread + number) & (TransMax - 1);

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	Mobt@NA
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::ClrTransData()
{
	ASSERT(this);
	ASSERT_DIAG();

	// bN
	sync->Lock();

	// NA
	midi.transnum = 0;
	midi.transread = 0;
	midi.transwrite = 0;

	// AbN
	sync->Unlock();
}

//---------------------------------------------------------------------------
//
//	Mobt@f[^ݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetRecvData(const BYTE *ptr, DWORD length)
{
	DWORD lp;

	ASSERT(this);
	ASSERT(ptr);
	ASSERT_DIAG();

	// {[hID0łΉȂ
	if (midi.bid == 0) {
		return;
	}

	// Młꍇ̂݁A󂯕t
	if (midi.rcr & 1) {
		// Mobt@֑}
		for (lp=0; lp<length; lp++) {
			InsertRecv((DWORD)ptr[lp]);
		}
	}

	// MłAłȂɂ炸ATHRUw肪ΑMobt@
	if (midi.trr & 0x40) {
		for (lp=0; lp<length; lp++) {
			InsertTrans((DWORD)ptr[lp]);
		}
	}
}

//---------------------------------------------------------------------------
//
//	MfBCݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetRecvDelay(int delay)
{
	ASSERT(this);
	ASSERT(delay >= 0);
	ASSERT_DIAG();

	// husPʂł邱
	recvdelay = delay;
}

//---------------------------------------------------------------------------
//
//	WX^Zbg
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::ResetReg()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG0(Log::Normal, "WX^Zbg");
#endif	// MIDI_LOG

	// 荞
	midi.vector = -1;

	// MCSWX^()
	midi.wdr = 0;
	midi.rgr = 0xff;

	// MCSWX^(荞)
	// IMRbit1𗧂ĂĂ(X[p[nOI)
	midi.ivr = 0x10;
	midi.isr = 0;
	midi.imr = 0x02;
	midi.ier = 0;

	// MCSWX^(A^CbZ[W)
	midi.dmr = 0;
	midi.dcr = 0;

	// MCSWX^(M)
	midi.rrr = 0;
	midi.rmr = 0;
	midi.rcn = 0;
	midi.amr = 0;
	midi.adr = 0;
	midi.asr = 0;
	midi.rsr = 0;
	midi.rcr = 0;

	// MCSWX^(M)
	midi.trr = 0;
	midi.tmr = 0;
	midi.tbs = FALSE;
	midi.tcn = 0;
	midi.tcr = 0;

	// MCSWX^(FSK)
	midi.fsr = 0;
	midi.fcr = 0;

	// MCSWX^(JE^)
	midi.ccr = 2;
	midi.cdr = 0;
	midi.ctr = 0x80;
	midi.srr = 0;
	midi.scr = 1;
	midi.sct = 1;
	midi.spr = 0;
	midi.str = 0xffff8000;
	midi.gtr = 0;
	midi.mtr = 0;

	// MCSWX^(GPIO)
	midi.edr = 0;
	midi.eor = 0;
	midi.eir = 0;

	// ʏobt@
	midi.normnum = 0;
	midi.normread = 0;
	midi.normwrite = 0;
	midi.normtotal = 0;

	// A^CMobt@
	midi.rtnum = 0;
	midi.rtread = 0;
	midi.rtwrite = 0;
	midi.rttotal = 0;

	// ʃobt@
	midi.stdnum = 0;
	midi.stdread = 0;
	midi.stdwrite = 0;
	midi.stdtotal = 0;

	// A^CMobt@
	midi.rrnum = 0;
	midi.rrread = 0;
	midi.rrwrite = 0;
	midi.rrtotal = 0;

	// Mobt@
	sync->Lock();
	midi.transnum = 0;
	midi.transread = 0;
	midi.transwrite = 0;
	sync->Unlock();

	// Mobt@
	sync->Lock();
	midi.recvnum = 0;
	midi.recvread = 0;
	midi.recvwrite = 0;
	sync->Unlock();

	// Cxg~(ėp^C})
	event[2].SetTime(0);

	// CxgJn(MIDINbN)(OPMDRV3.X)
	event[1].SetDesc("Clock");
	event[1].SetTime(0x40000);

	// GNXN[VuJE^
	ex_cnt[0] = 0;
	ex_cnt[1] = 0;
	ex_cnt[2] = 0;
	ex_cnt[3] = 0;

	// SĂ̊荞݂LZ
	IntCheck();

	// ʏobt@GveB荞(ۂISR̂ݕω)
	Interrupt(6, TRUE);
}

//---------------------------------------------------------------------------
//
//	WX^ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::ReadReg(DWORD reg)
{
	DWORD group;

	ASSERT(this);
	ASSERT(reg < 8);
	ASSERT_DIAG();

	// ANZXtOON
	if (!midi.access) {
		event[0].SetTime(640);
		midi.access = TRUE;
	}

	// EFCg
	scheduler->Wait(2);

	switch (reg) {
		// IVR
		case 0:
#if defined(MIDI_LOG)
			LOG1(Log::Normal, "荞݃xN^ǂݏo $%02X", midi.ivr);
#endif	// MIDI_LOG
			return midi.ivr;

		// RGR
		case 1:
			return midi.rgr;

		// ISR
		case 2:
#if defined(MIDI_LOG)
			LOG1(Log::Normal, "荞݃T[rXWX^ǂݏo $%02X", midi.isr);
#endif	// MIDI_LOG
			return midi.isr;

		// ICR
		case 3:
			LOG1(Log::Normal, "WX^ǂݏo R%d", reg);
			return midi.wdr;
	}

	// O[vAWX^ԍ쐬
	group = midi.rgr & 0x0f;
	if (group >= 0x0a) {
		return 0xff;
	}

	// WX^ԍ
	reg += (group * 10);

	// WX^(WX^4ȍ~)
	switch (reg) {
		// DSR
		case 16:
			return GetDSR();

		// RSR
		case 34:
			return GetRSR();

		// RDR(XV)
		case 36:
			return GetRDR();

		// TSR
		case 54:
			return GetTSR();

		// FSR
		case 64:
			return GetFSR();

		// SRR
		case 74:
			return GetSRR();

		// EIR
		case 96:
			return GetEIR();
	}

	LOG1(Log::Normal, "WX^ǂݏo R%d", reg);
	return midi.wdr;
}

//---------------------------------------------------------------------------
//
//	WX^
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::WriteReg(DWORD reg, DWORD data)
{
	DWORD group;

	ASSERT(this);
	ASSERT(reg < 8);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// ANZXtOON
	if (!midi.access) {
		event[0].SetTime(640);
		midi.access = TRUE;
	}

	// EFCg
	scheduler->Wait(2);

	// ݃f[^ۑ(`̃WX^œǂ߂)
	midi.wdr = data;

	// WX^
	switch (reg) {
		// IVR
		case 0:
			LOG2(Log::Normal, "WX^ R%d <- $%02X", reg, data);
			return;

		// RGR
		case 1:
			// ŏʃrbg̓WX^Zbg
			if (data & 0x80) {
				ResetReg();
			}
			midi.rgr = data;
			return;

		// ISR
		case 2:
			LOG2(Log::Normal, "WX^ R%d <- $%02X", reg, data);
			return;

		// ICR
		case 3:
			SetICR(data);
			return;
	}

	// O[vAWX^ԍ쐬
	group = midi.rgr & 0x0f;
	if (group >= 0x0a) {
		return;
	}

	// WX^ԍ
	reg += (group * 10);

	// WX^(WX^4ȍ~)
	switch (reg) {
		// IOR
		case 4:
			SetIOR(data);
			return;

		// IMR
		case 5:
			SetIMR(data);
			return;

		// IER
		case 6:
			SetIER(data);
			return;

		// DMR
		case 14:
			SetDMR(data);
			return;

		// DCR
		case 15:
			SetDCR(data);
			return;

		// DNR
		case 17:
			SetDNR(data);
			return;

		// RRR
		case 24:
			SetRRR(data);
			return;

		// RMR
		case 25:
			SetRMR(data);
			return;

		// AMR
		case 26:
			SetAMR(data);
			return;

		// ADR
		case 27:
			SetADR(data);
			return;

		// RCR
		case 35:
			SetRCR(data);
			return;

		// TRR
		case 44:
			SetTRR(data);
			return;

		// TMR
		case 45:
			SetTMR(data);
			return;

		// TCR
		case 55:
			SetTCR(data);
			return;

		// TDR
		case 56:
			SetTDR(data);
			return;

		// FCR
		case 65:
			SetFCR(data);
			return;

		// CCR
		case 66:
			SetCCR(data);
			return;

		// CDR
		case 67:
			SetCDR(data);
			return;

		// SCR
		case 75:
			SetSCR(data);
			return;

		// SPR(L)
		case 76:
			SetSPR(data, FALSE);
			return;

		// SPR(H)
		case 77:
			SetSPR(data, TRUE);
			return;

		// GTR(L)
		case 84:
			SetGTR(data, FALSE);
			return;

		// GTR(H)
		case 85:
			SetGTR(data, TRUE);
			return;

		// MTR(L)
		case 86:
			SetMTR(data, FALSE);
			return;

		// MTR(H)
		case 87:
			SetMTR(data, TRUE);
			return;

		// EDR
		case 94:
			SetEDR(data);
			return;

		// EOR
		case 96:
			SetEOR(data);
			return;
	}

	LOG2(Log::Normal, "WX^ R%d <- $%02X", reg, data);
}

//---------------------------------------------------------------------------
//
//	WX^ǂݏo(ReadOnly)
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::ReadRegRO(DWORD reg) const
{
	DWORD group;

	ASSERT(this);
	ASSERT(reg < 8);
	ASSERT_DIAG();

	switch (reg) {
		// IVR
		case 0:
			return midi.ivr;

		// RGR
		case 1:
			return midi.rgr;

		// ISR
		case 2:
			return midi.isr;

		// ICR
		case 3:
			return midi.wdr;
	}

	// O[vAWX^ԍ쐬
	group = midi.rgr & 0x0f;
	if (group >= 0x0a) {
		return 0xff;
	}

	// WX^ԍ
	reg += (group * 10);

	// WX^(WX^4ȍ~)
	switch (reg) {
		// DSR
		case 16:
			return GetDSR();

		// RSR
		case 34:
			return GetRSR();

		// RDR(ReadOnly)
		case 36:
			return GetRDRRO();

		// TSR
		case 54:
			return GetTSR();

		// FSR
		case 64:
			return GetFSR();

		// SRR
		case 74:
			return GetSRR();

		// EIR
		case 96:
			return GetEIR();
	}

	return midi.wdr;
}

//---------------------------------------------------------------------------
//
//	ICRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetICR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "荞݃NAWX^ݒ $%02X", data);
#endif

	// rbg]f[^gāACT[rXNA
	midi.isr &= ~data;

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IORݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetIOR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// 3bit̂ݗL
	data &= 0xe0;

	LOG1(Log::Detail, "荞݃xN^x[X $%02X", data);

	// IVRύX
	midi.ivr &= 0x1f;
	midi.ivr |= data;

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	IMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetIMR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	data &= 0x0f;

	if (midi.imr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "荞݃[hWX^ݒ $%02X", data);
#endif	// MIDI_LOG
		midi.imr = data;

		// 荞݃`FbN
		IntCheck();
	}
}

//---------------------------------------------------------------------------
//
//	IERݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetIER(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (midi.ier != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "荞݋WX^ݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.ier = data;

		// 荞݃`FbN
		IntCheck();
	}
}

//---------------------------------------------------------------------------
//
//	DMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetDMR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lf[^c
	data &= 0x3f;

	if (midi.dmr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "A^CbZ[W[hWX^ݒ $%02X", data);
#endif

		// ݒ
		midi.dmr = data;
	}
}

//---------------------------------------------------------------------------
//
//	DCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetDCR(DWORD data)
{
	DWORD msg;

	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "A^CbZ[WRg[WX^ݒ $%02X", data);
#endif

	// ݒ
	midi.dcr = data;

	// bZ[W쐬
	msg = data & 0x07;
	msg += 0xf8;
	if (msg >= 0xfe) {
		LOG1(Log::Warning, "sȃA^CbZ[W $%02X", msg);
		return;
	}

	// MIDINbNbZ[WDMRbit2ŕ
	if (msg == 0xf8) {
		if ((midi.dmr & 4) == 0) {
			// MIDINbN^C}A$F8MȂǂŃNbNƂ
			return;
		}

		// ̃^C~OMIDINbNMƂ
		if (midi.rcr & 1) {
			if (midi.dmr & 8) {
				// A^CMobt@֑}
				InsertRR(0xf8);

				// NbN
				Clock();

				// obt@`FbN
				CheckRR();
			}
		}
		return;
	}
	else {
		// ȊÕbZ[ẂAA^CMobt@
		if ((data & 0x80) == 0) {
			return;
		}
	}

	// A^CMobt@փf[^}
	InsertRT(msg);
}

//---------------------------------------------------------------------------
//
//	DSR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetDSR() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// FIFO-IRx̃f[^ǂݏoBf[^Ȃ0
	if (midi.rrnum == 0) {
		return 0;
	}

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "FIFO-IRx擾 $%02X", midi.rrbuf[midi.rrread]);
#endif	// MIDI_LOG

	return midi.rrbuf[midi.rrread];
}

//---------------------------------------------------------------------------
//
//	DNRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetDNR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (data & 0x01) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "FIFO-IRxXV");
#endif	// MIDI_LOG

		// A^CMobt@XV
		if (midi.rrnum > 0) {
			midi.rrnum--;
			midi.rrread = (midi.rrread + 1) & 3;
		}

		// A^CMobt@荞݃`FbN
		CheckRR();
	}
}

//---------------------------------------------------------------------------
//
//	RRRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetRRR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x38;

	if (midi.rrr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "M[gݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.rrr = data;
	}

	// M[g`FbN
	if (midi.rrr != 0x08) {
		LOG1(Log::Warning, "M[gُ $%02X", midi.rrr);
	}
}

//---------------------------------------------------------------------------
//
//	RMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetRMR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x3f;

	if (midi.rmr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "Mp[^ݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.rmr = data;
	}

	// Mp[^`FbN
	if ((midi.rmr & 0x32) != 0) {
		LOG1(Log::Warning, "Mp[^ُ $%02X", midi.rmr);
	}
}

//---------------------------------------------------------------------------
//
//	AMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetAMR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (midi.amr != data) {
#if defined(MIDI_LOG)
		if (data & 0x80) {
			LOG1(Log::Normal, "M[JID+foCXID`FbNB[JID $%02X", data & 0x7f);
		}
		else {
			LOG1(Log::Normal, "M[JID̂݃`FbNB[JID $%02X", data & 0x7f);
		}
#endif	// MIDI_LOG

		midi.amr = data;
	}
}

//---------------------------------------------------------------------------
//
//	ADRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetADR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (midi.adr != data) {
#if defined(MIDI_LOG)
		if (data & 0x80) {
			LOG1(Log::Normal, "u[hLXgBfoCXID $%02X", data & 0x7f);
		}
		else {
			LOG1(Log::Normal, "u[hLXgBfoCXID $%02X", data & 0x7f);
		}
#endif	// MIDI_LOG

		midi.adr = data;
	}
}

//---------------------------------------------------------------------------
//
//	RSR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetRSR() const
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "Mobt@Xe[^X擾 $%02X", midi.rsr);
#endif	// MIDI_LOG

	return midi.rsr;
}

//---------------------------------------------------------------------------
//
//	RCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetRCR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0xdf;

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "MRg[ $%02X", data);
#endif	// MIDI_LOG

	// f[^ݒ
	midi.rcr = data;

	// ItCtONA
	if (midi.rcr & 0x04) {
		midi.rsr &= ~0x04;

		// ItC荞݉
		if ((midi.imr & 4) == 0) {
			Interrupt(4, FALSE);
		}
	}

	// u[NtONA
	if (midi.rcr & 0x08) {
		midi.rsr &= ~0x08;

		// u[N荞݉
		if (midi.imr & 4) {
			Interrupt(4, FALSE);
		}
	}

	// I[o[t[tONA
	if (midi.rcr & 0x40) {
		midi.rsr &= ~0x40;
	}

	// Mobt@NA
	if (midi.rcr & 0x80) {
		// ʏobt@
		midi.stdnum = 0;
		midi.stdread = 0;
		midi.stdwrite = 0;

		// A^CMobt@
		midi.rrnum = 0;
		midi.rrread = 0;
		midi.rrwrite = 0;

		// 荞݃It
		Interrupt(5, FALSE);
		Interrupt(0, FALSE);
		if (midi.imr & 8) {
			Interrupt(1, FALSE);
		}
	}

	// AhXn^Cl[u
	if (midi.rcr & 0x02) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "AhXn^ Cl[u");
#endif	// MIDI_LOG
		midi.rsr |= 0x02;
	}
	else {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "AhXn^ fBZ[u");
#endif	// MIDI_LOG
		midi.rsr &= ~0x02;
	}

	// MCl[u
	if (midi.rcr & 0x01) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "MCl[u");
#endif	// MIDI_LOG
	}
	else {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "MfBZ[u");
#endif	// MIDI_LOG
	}
}

//---------------------------------------------------------------------------
//
//	RDR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetRDR()
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// f[^wdr()
	data = midi.wdr;

	// obt@ɗLf[^邩
	if (midi.stdnum > 0) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "ʃobt@擾 $%02X", midi.stdbuf[midi.stdread]);
#endif	// MIDI_LOG

		// obt@f[^𓾂
		data = midi.stdbuf[midi.stdread];

		// obt@
		midi.stdnum--;
		midi.stdread = (midi.stdread + 1) & 0x7f;

		// 荞݉
		Interrupt(5, FALSE);

		// obt@ɂȂ
		if (midi.stdnum == 0) {
			// Mobt@LtO
			midi.rsr &= 0x7f;
		}
		else {
			// Mobt@LtOݒ
			midi.rsr |= 0x80;
		}

		// f[^`FbN(mFpɁAGNXN[VuJE^グ)
		if (data == 0xf0) {
			ex_cnt[2]++;
		}
		if (data == 0xf7) {
			ex_cnt[3]++;
		}
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	RDR擾 (Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetRDRRO() const
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// f[^wdr()
	data = midi.wdr;

	// obt@ɗLf[^邩
	if (midi.stdnum > 0) {
		data = midi.stdbuf[midi.stdread];
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	TRRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetTRR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x78;

	if (midi.trr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "M[gݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.trr = data;
	}

	// M[g`FbN
	if ((midi.trr & 0x38) != 0x08) {
		LOG1(Log::Warning, "M[gُ $%02X", midi.trr);
	}

#if defined(MIDI_LOG)
	if (midi.trr & 0x40) {
		LOG0(Log::Normal, "MIDI THRUݒ");
	}
#endif	// MIDI_LOG
}

//---------------------------------------------------------------------------
//
//	TMRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetTMR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x3f;

	if (midi.tmr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "Mp[^ݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.tmr = data;
	}

	// Mp[^`FbN
	if ((midi.tmr & 0x32) != 0) {
		LOG1(Log::Warning, "Mp[^ُ $%02X", midi.tmr);
	}
}

//---------------------------------------------------------------------------
//
//	TSR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetTSR() const
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// NA
	data = 0;

	// bit7:ʏobt@tO
	if (midi.normnum == 0) {
		data |= 0x80;
	}

	// bit6:ʏobt@f[^LtO
	if (midi.normnum < 16) {
		data |= 0x40;
	}

	// bit2:M(81.920ms)tO
	if (midi.tcn >= 0x100) {
		ASSERT(midi.tcn == 0x100);
		data |= 0x04;
	}

	// bit0:gX~b^BUSYtO
	if (midi.tbs) {
		data |= 0x01;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	TCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetTCR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x8d;

#if defined(MIDI_LOG)
	if (midi.tcr != data) {
		LOG1(Log::Normal, "MRg[ $%02X", data);
	}
#endif	// MIDI_LOG

	// Lȃrbĝݕۑ
	midi.tcr = data;

	// ʏobt@AA^CpMobt@ NA
	if (data & 0x80) {
		// ʏobt@
		midi.normnum = 0;
		midi.normread = 0;
		midi.normwrite = 0;

		// A^CpMobt@
		midi.rtnum = 0;
		midi.rtread = 0;
		midi.rtwrite = 0;

		// 荞݃I
		Interrupt(6, TRUE);
	}

	// M(81.920ms)tO NA
	if (data & 0x04) {
		midi.tcn = 0;
	}

	// MCl[u
#if defined(MIDI_LOG)
	if (data & 0x01) {
		LOG0(Log::Normal, "MCl[u");
	}
	else {
		LOG0(Log::Normal, "MfBZ[u");
	}
#endif	// MIDI_LOG
}

//---------------------------------------------------------------------------
//
//	TDRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetTDR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "Mobt@ $%02X", data);
#endif	// MIDI_LOG

	// f[^`FbN(mFpɁAGNXN[VuJE^グ)
	if (data == 0xf0) {
		ex_cnt[0]++;
	}
	if (data == 0xf7) {
		ex_cnt[1]++;
	}

	// M`FbN(MfBZ[uł߂:WIOtV[)
	if ((midi.trr & 0x38) != 0x08) {
		// M[g31250bpsłȂ
		return;
	}
	if ((midi.tmr & 0x32) != 0) {
		// 8bitf[^A1XgbvrbgAmpeBłȂ
		return;
	}

	// ʏobt@Ƀf[^}
	InsertNorm(data);

	// Mobt@GveB荞݂̓NA
	Interrupt(6, FALSE);
}

//---------------------------------------------------------------------------
//
//	FSR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetFSR() const
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "e[vSYNCXe[^X擾 $%02X", midi.fsr);
#endif	// MIDI_LOG

	return midi.fsr;
}

//---------------------------------------------------------------------------
//
//	FCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetFCR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// Lc
	data &= 0x9f;

	if (midi.fcr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "e[vSYNCR}hݒ $%02X", data);
#endif	// MIDI_LOG

		// ݒ
		midi.fcr = data;
	}
}

//---------------------------------------------------------------------------
//
//	CCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetCCR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "NbNRg[WX^ݒ $%02X", data);
#endif	// MIDI_LOG

	// 2bit̂ݗL
	data &= 0x03;

	// bit1͕K1
	if ((data & 2) == 0) {
		LOG0(Log::Warning, "CLKMNbNgIG[");
	}

	// ݒ
	midi.ccr = data;
}

//---------------------------------------------------------------------------
//
//	CDRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetCDR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// CDRݒ
	if (midi.cdr != (data & 0x7f)) {
		midi.cdr = (data & 0x7f);

#if defined(MIDI_LOG)
		LOG1(Log::Normal, "NbNJE^萔ݒ $%02X", midi.cdr);
#endif	// MIDI_LOG
	}

	// bit7CTRɒl𑦎[h
	if (data & 0x80) {
		// CTRݒ
		midi.ctr = midi.cdr;

#if defined(MIDI_LOG)
		LOG0(Log::Normal, "NbNJE^萔ݒ[h");
#endif	// MIDI_LOG
	}
}

//---------------------------------------------------------------------------
//
//	SRR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetSRR() const
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "R[fBOJE^擾 $%02X", midi.srr);
#endif	// MIDI_LOG

	return midi.srr;
}

//---------------------------------------------------------------------------
//
//	SCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetSCR(DWORD data)
{
	DWORD mtr;
	char desc[16];

	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// ԃ[gݒ
	if (midi.scr != (data & 0x0f)) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "MIDINbNԊݒ $%02X", data & 0x0f);
#endif	// MIDI_LOG
		midi.scr = (data & 0x0f);

		// [g0͐ݒ֎~
		if (midi.scr == 0) {
			LOG0(Log::Warning, "MIDINbNԊ ݒ֎~l");
			midi.scr = 1;
		}

		// CxgĐݒ
		if (midi.mtr <= 1) {
			mtr = 0x40000;
		}
		else {
			mtr = midi.mtr << 4;
		}
		mtr /= midi.scr;
		if (event[1].GetTime() != mtr) {
			event[1].SetTime(mtr);
		}

		// SCT`FbN
		if (midi.sct > midi.scr) {
			midi.sct = midi.scr;
		}
		ASSERT(midi.sct > 0);

		// Cxg
		if (midi.scr == 1) {
			event[1].SetDesc("Clock");
		}
		else {
			sprintf(desc, "Clock (Div%d)", midi.scr);
			event[1].SetDesc(desc);
		}
	}

	// vCobNJE^NA
	if (data & 0x10) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "vCobNJE^NA");
#endif	// MIDI_LOG
		midi.str = 0;
	}

	// vCobNJE^Z
	if (data & 0x20) {
		// Z
		midi.str += midi.spr;

		// I[o[t[`FbN
		if ((midi.str >= 0x8000) && (midi.str < 0x10000)) {
			midi.str = 0x8000;
		}

#if defined(MIDI_LOG)
		LOG2(Log::Normal, "vCobNJE^Z $%04X->$%04X", midi.spr, midi.str);
#endif	// MIDI_LOG
	}
}

//---------------------------------------------------------------------------
//
//	SPRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetSPR(DWORD data, BOOL high)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (high) {
		// 
		midi.spr &= 0x00ff;
		midi.spr |= ((data & 0x7f) << 8);
	}
	else {
		// 
		midi.spr &= 0xff00;
		midi.spr |= data;
	}

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "vCobNJE^萔ݒ $%04X", midi.spr);
#endif	// MIDI_LOG
}

//---------------------------------------------------------------------------
//
//	GTRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetGTR(DWORD data, BOOL high)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (high) {
		// 
		midi.gtr &= 0x00ff;
		midi.gtr |= ((data & 0x3f) << 8);
	}
	else {
		// 
		midi.gtr &= 0xff00;
		midi.gtr |= data;
	}

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "ėp^C}WX^ݒ $%04X", midi.gtr);
#endif	// MIDI_LOG

	// bit7Ńf[^[h
	if (high && (data & 0x80)) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "ėp^C}ݒl[h");
#endif	// MIDI_LOG

		// f[^[hāA^C}Jn
		event[2].SetTime(midi.gtr << 4);
	}
}

//---------------------------------------------------------------------------
//
//	MTRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetMTR(DWORD data, BOOL high)
{
	DWORD mtr;

	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (high) {
		// 
		midi.mtr &= 0x00ff;
		midi.mtr |= ((data & 0x3f) << 8);
	}
	else {
		// 
		midi.mtr &= 0xff00;
		midi.mtr |= data;
	}

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "MIDINbN^C}WX^ݒ $%04X", midi.mtr);
#endif	// MIDI_LOG

	// bit7Ńf[^[h
	if (high) {
		if (midi.mtr <= 1) {
			mtr = 0x40000;
		}
		else {
			mtr = midi.mtr << 4;
		}

		// SCRŊl
		ASSERT(midi.scr != 0);
		event[1].SetTime(mtr / midi.scr);
	}
}

//---------------------------------------------------------------------------
//
//	EDRݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetEDR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (midi.edr != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "GPIOfBNVݒ $%02X", data);
#endif	// MIDI_LOG

		midi.edr = data;
	}
}

//---------------------------------------------------------------------------
//
//	EORݒ
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::SetEOR(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	if (midi.eor != data) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "GPIOo̓f[^ݒ $%02X", data);
#endif	// MIDI_LOG

		midi.eor = data;
	}
}

//---------------------------------------------------------------------------
//
//	EIR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetEIR() const
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "GPIOǂݏo $%02X", midi.eir);
#endif	// MIDI_LOG

	return midi.eir;
}

//---------------------------------------------------------------------------
//
//	A^CMobt@`FbN
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::CheckRR()
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// Mobt@Ƀf[^Ȃ
	if (midi.rrnum == 0) {
		// 荞݃It
		Interrupt(0, FALSE);
		if (midi.imr & 8) {
			Interrupt(1, FALSE);
		}
		return;
	}

	// łÂf[^𓾂
	data = midi.rrbuf[midi.rrread];

	// IRQ-1
	if (data == 0xf8) {
		if (midi.imr & 8) {
			Interrupt(1, TRUE);
			Interrupt(0, FALSE);
			return;
		}
	}

	// IRQ-0
	Interrupt(1, FALSE);
	if ((data >= 0xf9) && (data <= 0xfd)) {
		Interrupt(0, TRUE);
	}
	else {
		Interrupt(0, FALSE);
	}
}

//---------------------------------------------------------------------------
//
//	荞ݗv
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::Interrupt(int type, BOOL flag)
{
	ASSERT(this);
	ASSERT((type >= 0) && (type <= 7));
	ASSERT_DIAG();

	// ISR̊Yrbgω
	if (flag) {
		midi.isr |= (1 << type);
	}
	else {
		midi.isr &= ~(1 << type);
	}

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	荞ACK
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::IntAck(int level)
{
	ASSERT(this);
	ASSERT((level == 2) || (level == 4));
	ASSERT_DIAG();

	// 荞݃xvȂΉȂ
	if (level != (int)midi.ilevel) {
#if defined(MIDI_LOG)
		LOG1(Log::Normal, "荞݃xُ x%d", level);
#endif	// MIDI_LOG
		return;
	}

	// 荞ݗvłȂ΁AȂ
	if (midi.vector < 0) {
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "vĂȂ荞");
#endif	// MIDI_LOG
		return;
	}

#if defined(MIDI_LOG)
	LOG1(Log::Normal, "荞ACK x%d", level);
#endif	// MIDI_LOG

	// vxN^Ȃ
	midi.vector = -1;

	// ̊荞݂XPW[
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	荞݃`FbN
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::IntCheck()
{
	int i;
	int vector;
	DWORD bit;

	ASSERT(this);
	ASSERT_DIAG();

	// 荞݃[hRg[bit1=1łȂƁA68000荞݂ɓKȂ
	if ((midi.imr & 0x02) == 0) {
		// 荞݂΃LZ
		if (midi.vector != -1) {
			cpu->IntCancel(midi.ilevel);
			midi.ivr &= 0xe0;
			midi.ivr |= 0x10;
			midi.vector = -1;
		}
		return;
	}

	// ISR`FbN
	bit = 0x80;
	for (i=7; i>=0; i--) {
		if (midi.isr & bit) {
			if (midi.ier & bit) {
				// 荞ݔBxN^Zo
				vector = midi.ivr;
				vector &= 0xe0;
				vector += (i << 1);

				// ݂̂̂ƈAv
				if (midi.vector != vector) {
					// ɑ̂̂vĂ΁ALZ
					if (midi.vector >= 0) {
						cpu->IntCancel(midi.ilevel);
					}
					// 荞ݗv
					cpu->Interrupt(midi.ilevel, vector);
					midi.vector = vector;
					midi.ivr = (DWORD)vector;

#if defined(MIDI_LOG)
					LOG3(Log::Normal, "荞ݗv x%d ^Cv%d xN^$%02X", midi.ilevel, i, vector);
#endif	// MIDI_LOG
				}
				return;
			}
		}

		// 
		bit >>= 1;
	}

	// 銄荞݂͂Ȃ̂ŁALZ
	if (midi.vector != -1) {
		// 荞݃LZ
		cpu->IntCancel(midi.ilevel);
		midi.vector = -1;
		midi.ivr &= 0xe0;
		midi.ivr |= 0x10;
#if defined(MIDI_LOG)
		LOG0(Log::Normal, "荞݃LZ");
#endif	// MIDI_LOG
	}
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL MIDI::GetMIDI(midi_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT_DIAG();

	// f[^Rs[
	*buffer = midi;
}

//---------------------------------------------------------------------------
//
//	GNXN[VuJEg擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL MIDI::GetExCount(int index) const
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < 4));
	ASSERT_DIAG();

	return ex_cnt[index];
}
