//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ OPM(YM2151) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "mfp.h"
#include "schedule.h"
#include "adpcm.h"
#include "fdd.h"
#include "fileio.h"
#include "config.h"
#include "opm.h"
#include "opmif.h"

//===========================================================================
//
//	OPM
//
//===========================================================================
//#define OPM_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
OPMIF::OPMIF(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('O', 'P', 'M', ' ');
	dev.desc = "OPM (YM2151)";

	// JnAhXAIAhX
	memdev.first = 0xe90000;
	memdev.last = 0xe91fff;

	// [NNA
	mfp = NULL;
	fdd = NULL;
	adpcm = NULL;
	engine = NULL;
	opmbuf = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL OPMIF::Init()
{
	int i;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// MFP擾
	ASSERT(!mfp);
	mfp = (MFP*)vm->SearchDevice(MAKEID('M', 'F', 'P', ' '));
	ASSERT(mfp);

	// ADPCM擾
	ASSERT(!adpcm);
	adpcm = (ADPCM*)vm->SearchDevice(MAKEID('A', 'P', 'C', 'M'));
	ASSERT(adpcm);

	// FDD擾
	ASSERT(!fdd);
	fdd = (FDD*)vm->SearchDevice(MAKEID('F', 'D', 'D', ' '));
	ASSERT(fdd);

	// Cxg쐬
	event[0].SetDevice(this);
	event[0].SetDesc("Timer-A");
	event[1].SetDevice(this);
	event[1].SetDesc("Timer-B");
	for (i=0; i<2; i++) {
		event[i].SetUser(i);
		event[i].SetTime(0);
		scheduler->AddEvent(&event[i]);
	}

	// obt@m
	try {
		opmbuf = new DWORD [BufMax * 2];
	}
	catch (...) {
		return FALSE;
	}
	if (!opmbuf) {
		return FALSE;
	}

	// [NNA
	memset(&opm, 0, sizeof(opm));
	memset(&bufinfo, 0, sizeof(bufinfo));
	bufinfo.max = BufMax;
	bufinfo.sound = TRUE;

	// obt@
	InitBuf(44100);

	// `l}XN̓It
	for (i=0; i<8; i++) {
		chmask[i] = FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::Cleanup()
{
	ASSERT(this);

	if (opmbuf) {
		delete[] opmbuf;
		opmbuf = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::Reset()
{
	int i;

	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// Cxg~߂
	event[0].SetTime(0);
	event[1].SetTime(0);

	// obt@NA
	memset(opmbuf, 0, sizeof(DWORD) * (BufMax * 2));

	// GWw肳Ă΃Zbg
	if (engine) {
		engine->Reset();
	}

	// WX^NA
	for (i=0; i<0x100; i++) {
		if (i == 8) {
			continue;
		}
		if ((i >= 0x60) && (i <= 0x7f)) {
			Output(i, 0x7f);
			continue;
		}
		if ((i >= 0xe0) && (i <= 0xff)) {
			Output(i, 0xff);
			continue;
		}
		Output(i, 0);
	}
	Output(0x19, 0x80);
	for (i=0; i<8; i++) {
		Output(8, i);
		opm.key[i] = i;
	}

	// ̑[NGA
	opm.addr = 0;
	opm.busy = FALSE;
	for (i=0; i<2; i++) {
		opm.enable[i] = FALSE;
		opm.action[i] = FALSE;
		opm.interrupt[i] = FALSE;
		opm.time[i] = 0;
	}
	opm.started = FALSE;

	// 荞ݗvȂ
	mfp->SetGPIP(3, 1);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL OPMIF::Save(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(opm_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// {̂Z[u
	if (!fio->Write(&opm, (int)sz)) {
		return FALSE;
	}

	// obt@̃TCYZ[u
	sz = sizeof(opmbuf_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// obt@Z[u
	if (!fio->Write(&bufinfo, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!event[0].Save(fio, ver)) {
		return FALSE;
	}
	if (!event[1].Save(fio, ver)) {
		return FALSE;
	}

	// `l}XNZ[u(version2.06ȍ~)
	if (!fio->Write(chmask, sizeof(chmask))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL OPMIF::Load(Fileio *fio, int ver)
{
	size_t sz;
	int i;
	int ch;

	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(opm_t)) {
		return FALSE;
	}

	// {̂[h
	if (!fio->Read(&opm, (int)sz)) {
		return FALSE;
	}

	// obt@̃TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(opmbuf_t)) {
		return FALSE;
	}

	// obt@[h
	if (!fio->Read(&bufinfo, (int)sz)) {
		return FALSE;
	}

	// Cxg[h
	if (!event[0].Load(fio, ver)) {
		return FALSE;
	}
	if (!event[1].Load(fio, ver)) {
		return FALSE;
	}

	// `l}XN[h(version2.06ȍ~)
	if (ver >= 0x0206) {
		if (!fio->Read(chmask, sizeof(chmask))) {
			return FALSE;
		}
	}

	// obt@NA
	InitBuf(bufinfo.rate * 100);

	// GWւ̃WX^Đݒ
	if (engine) {
		// WX^:mCYALFOAPMDAAMDACSM
		engine->SetReg(0x0f, opm.reg[0x0f]);
		engine->SetReg(0x14, opm.reg[0x14] & 0x80);
		engine->SetReg(0x18, opm.reg[0x18]);
		engine->SetReg(0x19, opm.reg[0x19]);

		// WX^:WX^
		for (i=0x20; i<0x100; i++) {
			// `l}XNl
			if ((i >= 0x60) && (i <= 0x7f)) {
				// `l擾
				ch = i & 7;
				ASSERT((ch >= 0) && (ch < 8));

				// }XNĂ΁A0x7f֕ύX
				if (chmask[ch]) {
					engine->SetReg(i, 0x7f);
					continue;
				}
			}

			// GW֏o
			engine->SetReg(i, opm.reg[i]);
		}

		// WX^:L[
		for (i=0; i<8; i++) {
			engine->SetReg(8, opm.key[i]);
		}

		// WX^:LFO
		engine->SetReg(1, 2);
		engine->SetReg(1, 0);
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::ApplyCfg(const Config* /*config*/)
{
	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::AssertDiag() const
{
	// {NX
	MemDevice::AssertDiag();

	ASSERT(this);
	ASSERT(GetID() == MAKEID('O', 'P', 'M', ' '));
	ASSERT(memdev.first == 0xe90000);
	ASSERT(memdev.last == 0xe91fff);
	ASSERT(mfp);
	ASSERT(mfp->GetID() == MAKEID('M', 'F', 'P', ' '));
	ASSERT(adpcm);
	ASSERT(adpcm->GetID() == MAKEID('A', 'P', 'C', 'M'));
	ASSERT(fdd);
	ASSERT(fdd->GetID() == MAKEID('F', 'D', 'D', ' '));
	ASSERT(opm.addr < 0x100);
	ASSERT((opm.busy == TRUE) || (opm.busy == FALSE));
	ASSERT((opm.enable[0] == TRUE) || (opm.enable[0] == FALSE));
	ASSERT((opm.enable[1] == TRUE) || (opm.enable[1] == FALSE));
	ASSERT((opm.action[0] == TRUE) || (opm.action[0] == FALSE));
	ASSERT((opm.action[1] == TRUE) || (opm.action[1] == FALSE));
	ASSERT((opm.interrupt[0] == TRUE) || (opm.interrupt[0] == FALSE));
	ASSERT((opm.interrupt[1] == TRUE) || (opm.interrupt[1] == FALSE));
	ASSERT((opm.started == TRUE) || (opm.started == FALSE));
	ASSERT(bufinfo.max == BufMax);
	ASSERT(bufinfo.num <= bufinfo.max);
	ASSERT(bufinfo.read < bufinfo.max);
	ASSERT(bufinfo.write < bufinfo.max);
	ASSERT((bufinfo.sound == TRUE) || (bufinfo.sound == FALSE));
}
#endif	// NDEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL OPMIF::ReadByte(DWORD addr)
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 4oCgPʂŃ[v
		addr &= 3;

		// EFCg
		scheduler->Wait(1);

		if (addr != 0x01) {
			// f[^|[gBUSYƃ^C}̏
			data = 0;

			// BUSY(1񂾂)
			if (opm.busy) {
				data |= 0x80;
				opm.busy = FALSE;
			}

			// ^C}
			if (opm.interrupt[0]) {
				data |= 0x01;
			}
			if (opm.interrupt[1]) {
				data |= 0x02;
			}

			return data;
		}

		// AhX|[gFF
		return 0xff;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
DWORD FASTCALL OPMIF::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT_DIAG();

	return (WORD)(0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 4oCgPʂŃ[v
		addr &= 3;

		// EFCg
		scheduler->Wait(1);

		if (addr == 0x01) {
			// AhXw(BUSYɊւ炸wł邱Ƃɂ)
			opm.addr = data;

			// BUSY~낷BAhXw̑҂͔ȂƂɂ
			opm.busy = FALSE;

			return;
		}
		else {
			// f[^(BUSYɊւ炸߂邱Ƃɂ)
			Output(opm.addr, data);

			// BUSYグ
			opm.busy = TRUE;
			return;
		}
	}
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);
	ASSERT_DIAG();

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL OPMIF::ReadOnly(DWORD addr) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 4oCgPʂŃ[v
		addr &= 3;

		if (addr != 0x01) {
			// f[^|[gBUSYƃ^C}̏
			data = 0;

			// BUSY
			if (opm.busy) {
				data |= 0x80;
			}

			// ^C}
			if (opm.interrupt[0]) {
				data |= 0x01;
			}
			if (opm.interrupt[1]) {
				data |= 0x02;
			}

			return data;
		}

		// AhX|[gFF
		return 0xff;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::GetOPM(opm_t *buffer)
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT_DIAG();

	// f[^Rs[
	*buffer = opm;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL OPMIF::Callback(Event *ev)
{
	int index;

	ASSERT(this);
	ASSERT(ev);
	ASSERT_DIAG();

	// [Uf[^M
	index = ev->GetUser();
	ASSERT((index >= 0) && (index <= 1));

#if defined(OPM_LOG)
	LOG2(Log::Normal, "^C}%c R[obN Time %d", index + 'A', ev->GetTime());
#endif	// OPM_LOG

	// Cl[u쒆ȂA荞݂N
	if (opm.enable[index] && opm.action[index]) {
		opm.action[index] = FALSE;
		opm.interrupt[index] = TRUE;
#if defined(OPM_LOG)
		LOG2(Log::Normal, "^C}%c I[o[t[荞 Time %d", index + 'A', ev->GetTime());
#endif	// OPM_LOG
		mfp->SetGPIP(3, 0);
	}

	// ԂĂ΁AĐݒ
	if (ev->GetTime() != opm.time[index]) {
		ev->SetTime(opm.time[index]);
#if defined(OPM_LOG)
		LOG2(Log::Normal, "^C}%c CxgX^[g Time %d", index + 'A', ev->GetTime());
#endif	// OPM_LOG
	}

	// ^C}ACSM̏ꍇ
	if ((index == 0) && engine) {
		if (opm.reg[0x14] & 0x80) {
			ProcessBuf();
			engine->TimerA();
		}
	}

	// ^C}͉񂵑
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	GWw
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::SetEngine(FM::OPM *p)
{
	int i;
	int ch;

	ASSERT(this);
	ASSERT_DIAG();

	// NULLw肳ꍇ
	engine = p;

	// ADPCM֒ʒm
	if (engine) {
		adpcm->Enable(TRUE);
	}
	else {
		adpcm->Enable(FALSE);
	}

	// enginew肠ȂAOPMWX^o
	if (!engine) {
		return;
	}
	ProcessBuf();

	// WX^:mCYALFOAPMDAAMDACSM
	engine->SetReg(0x0f, opm.reg[0x0f]);
	engine->SetReg(0x14, opm.reg[0x14] & 0x80);
	engine->SetReg(0x18, opm.reg[0x18]);
	engine->SetReg(0x19, opm.reg[0x19]);

	// WX^:WX^
	for (i=0x20; i<0x100; i++) {
		// `l}XNl
		if ((i >= 0x60) && (i <= 0x7f)) {
			// `l擾
			ch = i & 7;
			ASSERT((ch >= 0) && (ch < 8));

			// }XNĂ΁A0x7f֕ύX
			if (chmask[ch]) {
				engine->SetReg(i, 0x7f);
				continue;
			}
		}

		// GW֏o
		engine->SetReg(i, opm.reg[i]);
	}

	// WX^:L[
	for (i=0; i<8; i++) {
		engine->SetReg(8, opm.key[i]);
	}

	// WX^:LFO
	engine->SetReg(1, 2);
	engine->SetReg(1, 0);
}

//---------------------------------------------------------------------------
//
//	WX^o
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::Output(DWORD addr, DWORD data)
{
	int ch;

	ASSERT(this);
	ASSERT(addr < 0x100);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// ꃌWX^̏
	switch (addr) {
		// ^C}A
		case 0x10:
		case 0x11:
			opm.reg[addr] = data;
			CalcTimerA();
			return;

		// ^C}B
		case 0x12:
			opm.reg[addr] = data;
			CalcTimerB();
			return;

		// ^C}
		case 0x14:
			CtrlTimer(data);
			data &= 0x80;
			break;

		// ėp|[gt
		case 0x1b:
			CtrlCT(data);
			opm.reg[addr] = data;
			data &= 0x3f;
			break;

		// L[
		case 0x08:
			opm.key[data & 0x07] = data;
			opm.reg[addr] = data;
			break;

		// ̑
		default:
			opm.reg[addr] = data;
			break;
	}

	// `l}XNl(opm.regւ͒ʏo͂Ƃ)
	if ((addr >= 0x60) && (addr <= 0x7f)) {
		// `l擾
		ch = addr & 7;
		ASSERT((ch >= 0) && (ch < 8));

		// }XNĂ΁A0x7f֕ύX
		if (chmask[ch]) {
			data = 0x7f;
		}
	}

	// }XŇʂWX^OAGW֏o


	// GWw肳Ă΁Aobt@Oďo
	if (engine) {
		// ^C}֘AAłȂ
		if ((addr < 0x10) || (addr > 0x14)) {
			// łȂ(ʏ)
			ProcessBuf();
		}

		// L[ItȊO̓X^[gtOUp
		if ((addr != 0x08) || (data >= 0x08)) {
			opm.started = TRUE;
		}

		// GW֏o
		engine->SetReg(addr, data);
	}
}

//---------------------------------------------------------------------------
//
//	^C}AZo
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::CalcTimerA()
{
	DWORD hus;
	DWORD low;

	ASSERT(this);
	ASSERT_DIAG();

	// husPʂŌvZ
	hus = opm.reg[0x10];
	hus <<= 2;
	low = opm.reg[0x11] & 3;
	hus |= low;
	hus = (0x400 - hus);
	hus <<= 5;
	opm.time[0] = hus;
#if defined(OPM_LOG)
	LOG1(Log::Normal, "^C}A = %d", hus);
#endif	// OPM_LOG

	// ~܂Ă΃X^[g(YST)
	if (event[0].GetTime() == 0) {
		event[0].SetTime(hus);
#if defined(OPM_LOG)
		LOG1(Log::Normal, "^C}A X^[g Time %d", event[0].GetTime());
#endif	// OPM_LOG
	}
}

//---------------------------------------------------------------------------
//
//	^C}BZo
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::CalcTimerB()
{
	DWORD hus;

	ASSERT(this);
	ASSERT_DIAG();

	// husPʂŌvZ
	hus = opm.reg[0x12];
	hus = (0x100 - hus);
	hus <<= 9;
	opm.time[1] = hus;
#if defined(OPM_LOG)
	LOG1(Log::Normal, "^C}B = %d", hus);
#endif	// OPM_LOG

	// ~܂Ă΃X^[g(YST)
	if (event[1].GetTime() == 0) {
		event[1].SetTime(hus);
#if defined(OPM_LOG)
		LOG1(Log::Normal, "^C}B X^[g Time %d", event[1].GetTime());
#endif	// OPM_LOG
	}
}

//---------------------------------------------------------------------------
//
//	^C}
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::CtrlTimer(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// I[o[t[tÕNA
	if (data & 0x10) {
		opm.interrupt[0] = FALSE;
	}
	if (data & 0x20) {
		opm.interrupt[1] = FALSE;
	}

	// A荞݂𗎂Ƃ
	if (!opm.interrupt[0] && !opm.interrupt[1]) {
		mfp->SetGPIP(3, 1);
	}

	// ^C}AANV
	if (data & 0x04) {
		opm.action[0] = TRUE;
	}
	else {
		opm.action[0] = FALSE;
	}

	// ^C}ACl[u
	if (data & 0x01) {
		// 01Ń^C}l[hAȊOł^C}ON
		if (!opm.enable[0]) {
			CalcTimerA();
		}
		opm.enable[0] = TRUE;
	}
	else {
		// (}nb^ENCG)
		event[0].SetTime(0);
		opm.enable[0] = FALSE;
	}

	// ^C}BANV
	if (data & 0x08) {
		opm.action[1] = TRUE;
	}
	else {
		opm.action[1] = FALSE;
	}

	// ^C}BCl[u
	if (data & 0x02) {
		// 01Ń^C}l[hAȊOł^C}ON
		if (!opm.enable[1]) {
			CalcTimerB();
		}
		opm.enable[1] = TRUE;
	}
	else {
		// (}nb^ENCG)
		event[1].SetTime(0);
		opm.enable[1] = FALSE;
	}

	// f[^L
	opm.reg[0x14] = data;

#if defined(OPM_LOG)
	LOG1(Log::Normal, "^C} $%02X", data);
#endif	// OPM_LOG
}

//---------------------------------------------------------------------------
//
//	CT
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::CtrlCT(DWORD data)
{
	DWORD ct;

	ASSERT(this);
	ASSERT_DIAG();

	// CT|[ĝݎo
	data &= 0xc0;

	// ȑÕf[^𓾂
	ct = opm.reg[0x1b];
	ct &= 0xc0;

	// vĂΉȂ
	if (data == ct) {
		return;
	}

	// `FbN(ADPCM)
	if ((data & 0x80) != (ct & 0x80)) {
		if (data & 0x80) {
			adpcm->SetClock(4);
		}
		else {
			adpcm->SetClock(8);
		}
	}

	// `FbN(FDD)
	if ((data & 0x40) != (ct & 0x40)) {
		if (data & 0x40) {
			fdd->ForceReady(TRUE);
		}
		else {
			fdd->ForceReady(FALSE);
		}
	}
}

//---------------------------------------------------------------------------
//
//	obt@e𓾂
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::GetBufInfo(opmbuf_t *buf)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT_DIAG();

	*buf = bufinfo;
}

//---------------------------------------------------------------------------
//
//	`l}XNݒ
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::SetChMask(int ch, BOOL mask)
{
	int addr;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch < 8));
	ASSERT_DIAG();

	chmask[ch] = mask;

	// GWw肳Ă΁A؂ւ
	if (engine) {
		// obt@o
		ProcessBuf();

		// TL̂
		for (addr=0x60; addr<0x80; addr+=8) {
			if (mask) {
				// }XNĂȂA0x7f
				engine->SetReg(addr + ch, 0x7f);
			}
			else {
				// łȂ΁ALe
				engine->SetReg(addr + ch, opm.reg[addr + ch]);
			}
		}
	}
}

//---------------------------------------------------------------------------
//
//	`l}XN擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL OPMIF::GetChMask(int ch)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch < 8));
	ASSERT_DIAG();

	return chmask[ch];
}

//---------------------------------------------------------------------------
//
//	obt@
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::InitBuf(DWORD rate)
{
	ASSERT(this);
	ASSERT(rate > 0);
	ASSERT((rate % 100) == 0);
	ASSERT_DIAG();

	// ADPCMɐɒʒm
	adpcm->InitBuf(rate);

	// JE^A|C^
	bufinfo.num = 0;
	bufinfo.read = 0;
	bufinfo.write = 0;
	bufinfo.under = 0;
	bufinfo.over = 0;

	// TEhԂƁAAgTv
	scheduler->SetSoundTime(0);
	bufinfo.samples = 0;

	// [g
	bufinfo.rate = rate / 100;
}

//---------------------------------------------------------------------------
//
//	obt@O
//
//---------------------------------------------------------------------------
DWORD FASTCALL OPMIF::ProcessBuf()
{
	DWORD stime;
	DWORD sample;
	DWORD first;
	DWORD second;

	ASSERT(this);
	ASSERT_DIAG();

	// GWȂ΃^[
	if (!engine) {
		return bufinfo.num;
	}

	// TEhԂAK؂ȃTv𓾂
	sample = scheduler->GetSoundTime();
	stime = sample;

	sample *= bufinfo.rate;
	sample /= 20000;

	// bufmaxɐ
	if (sample > BufMax) {
		// I[o[Ă̂ŁAZbg
		scheduler->SetSoundTime(0);
		bufinfo.samples = 0;
		return bufinfo.num;
	}

	// ƈvĂ΃^[
	if (sample <= bufinfo.samples) {
		// VN
		while (stime >= 40000) {
			stime -= 20000;
			scheduler->SetSoundTime(stime);
			bufinfo.samples -= bufinfo.rate;
		}
		return bufinfo.num;
	}

	// ƈႤ
	sample -= bufinfo.samples;

	// 1܂2̂ǂ炩`FbN
	first = sample;
	if ((first + bufinfo.write) > BufMax) {
		first = BufMax - bufinfo.write;
	}
	second = sample - first;

	// 1
	memset(&opmbuf[bufinfo.write * 2], 0, first * 8);
	if (bufinfo.sound) {
		engine->Mix((int32*)&opmbuf[bufinfo.write * 2], first);
	}
	bufinfo.write += first;
	bufinfo.write &= (BufMax - 1);
	bufinfo.num += first;
	if (bufinfo.num > BufMax) {
		bufinfo.num = BufMax;
		bufinfo.read = bufinfo.write;
	}

	// 2
	if (second > 0) {
		memset(opmbuf, 0, second * 8);
		if (bufinfo.sound) {
			engine->Mix((int32*)opmbuf, second);
		}
		bufinfo.write += second;
		bufinfo.write &= (BufMax - 1);
		bufinfo.num += second;
		if (bufinfo.num > BufMax) {
			bufinfo.num = BufMax;
			bufinfo.read = bufinfo.write;
		}
	}

	// ς݃Tv։ZA20000husƂɃVN
	bufinfo.samples += sample;
	while (stime >= 40000) {
		stime -= 20000;
		scheduler->SetSoundTime(stime);
		bufinfo.samples -= bufinfo.rate;
	}

	return bufinfo.num;
}

//---------------------------------------------------------------------------
//
//	obt@擾
//
//---------------------------------------------------------------------------
void FASTCALL OPMIF::GetBuf(DWORD *buf, int samples)
{
	DWORD first;
	DWORD second;
	DWORD under;
	DWORD over;

	ASSERT(this);
	ASSERT(buf);
	ASSERT(samples > 0);
	ASSERT(engine);
	ASSERT_DIAG();

	// I[o[`FbN
	over = 0;
	if (bufinfo.num > (DWORD)samples) {
		over = bufinfo.num - samples;
	}

	// A2ځAA_[̗v߂
	first = samples;
	second = 0;
	under = 0;
	if (bufinfo.num < first) {
		first = bufinfo.num;
		under = samples - first;
		samples = bufinfo.num;
	}
	if ((first + bufinfo.read) > BufMax) {
		first = BufMax - bufinfo.read;
		second = samples - first;
	}

	// ǂݎ
	memcpy(buf, &opmbuf[bufinfo.read * 2], (first * 8));
	buf += (first * 2);
	bufinfo.read += first;
	bufinfo.read &= (BufMax - 1);
	bufinfo.num -= first;

	// 2ړǂݎ
	if (second > 0) {
		memcpy(buf, &opmbuf[bufinfo.read * 2], (second * 8));
		bufinfo.read += second;
		bufinfo.read &= (BufMax - 1);
		bufinfo.num -= second;
	}

	// A_[
	if (under > 0) {
		// 1/4Aɍ悤dg
		bufinfo.samples = 0;
		under *= 5000;
		under /= bufinfo.rate;
		scheduler->SetSoundTime(under);

		// L^
		bufinfo.under++;
	}

	// I[o[
	if (over > 0) {
		// 1/4Ax点悤dg
		over *= 5000;
		over /= bufinfo.rate;
		under = scheduler->GetSoundTime();
		if (under < over) {
			scheduler->SetSoundTime(0);
		}
		else {
			under -= over;
			scheduler->SetSoundTime(under);
		}

		// L^
		bufinfo.over++;
	}
}
