//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ PPI(i8255A) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "adpcm.h"
#include "schedule.h"
#include "config.h"
#include "fileio.h"
#include "ppi.h"

//===========================================================================
//
//	PPI
//
//===========================================================================
//#define PPI_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
PPI::PPI(VM *p) : MemDevice(p)
{
	int i;

	// foCXID
	dev.id = MAKEID('P', 'P', 'I', ' ');
	dev.desc = "PPI (i8255A)";

	// JnAhXAIAhX
	memdev.first = 0xe9a000;
	memdev.last = 0xe9bfff;

	// [N
	memset(&ppi, 0, sizeof(ppi));

	// IuWFNg
	adpcm = NULL;
	for (i=0; i<PortMax; i++) {
		joy[i] = NULL;
	}
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL PPI::Init()
{
	int i;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// ADPCM擾
	ASSERT(!adpcm);
	adpcm = (ADPCM*)vm->SearchDevice(MAKEID('A', 'P', 'C', 'M'));
	ASSERT(adpcm);

	// WCXeBbN^Cv
	for (i=0; i<PortMax; i++) {
		ppi.type[i] = 0;
		ASSERT(!joy[i]);
		joy[i] = new JoyDevice(this, i);
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL PPI::Cleanup()
{
	int i;

	ASSERT(this);

	// WCXeBbNfoCX
	for (i=0; i<PortMax; i++) {
		ASSERT(joy[i]);
		delete joy[i];
		joy[i] = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL PPI::Reset()
{
	int i;

	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// |[gC
	ppi.portc = 0;

	// Rg[
	for (i=0; i<PortMax; i++) {
		ppi.ctl[i] = 0;
	}

	// WCXeBbNfoCXɑ΂āAZbgʒm
	for (i=0; i<PortMax; i++) {
		joy[i]->Reset();
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL PPI::Save(Fileio *fio, int ver)
{
	size_t sz;
	int i;
	DWORD type;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(ppi_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&ppi, (int)sz)) {
		return FALSE;
	}

	// foCXZ[u
	for (i=0; i<PortMax; i++) {
		// foCX^Cv
		type = joy[i]->GetType();
		if (!fio->Write(&type, sizeof(type))) {
			return FALSE;
		}

		// foCX
		if (!joy[i]->Save(fio, ver)) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL PPI::Load(Fileio *fio, int ver)
{
	size_t sz;
	int i;
	DWORD type;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(ppi_t)) {
		return FALSE;
	}

	// ̂[h
	if (!fio->Read(&ppi, (int)sz)) {
		return FALSE;
	}

	// version2.00ł΂܂
	if (ver <= 0x200) {
		return TRUE;
	}

	// foCX[h
	for (i=0; i<PortMax; i++) {
		// foCX^Cv𓾂
		if (!fio->Read(&type, sizeof(type))) {
			return FALSE;
		}

		// ݂̃foCXƈvĂȂ΁A蒼
		if (joy[i]->GetType() != type) {
			delete joy[i];
			joy[i] = NULL;

			// PPIɋLĂ^Cv킹ĕύX
			ppi.type[i] = (int)type;

			// č쐬
			joy[i] = CreateJoy(i, ppi.type[i]);
			ASSERT(joy[i]->GetType() == type);
		}

		// foCXŗL
		if (!joy[i]->Load(fio, ver)) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL PPI::ApplyCfg(const Config *config)
{
	int i;

	ASSERT(this);
	ASSERT(config);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");

	// |[g[v
	for (i=0; i<PortMax; i++) {
		// ^CvvȂ玟
		if (config->joy_type[i] == ppi.type[i]) {
			continue;
		}

		// ݂̃foCX폜
		ASSERT(joy[i]);
		delete joy[i];
		joy[i] = NULL;

		// ^CvLAWCXeBbN쐬
		ppi.type[i] = config->joy_type[i];
		joy[i] = CreateJoy(i, config->joy_type[i]);
	}
}

#if defined(_DEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL PPI::AssertDiag() const
{
	ASSERT(this);
	ASSERT(GetID() == MAKEID('P', 'P', 'I', ' '));
	ASSERT(adpcm);
	ASSERT(adpcm->GetID() == MAKEID('A', 'P', 'C', 'M'));
	ASSERT(joy[0]);
	ASSERT(joy[1]);
}
#endif	// _DEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL PPI::ReadByte(DWORD addr)
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(PortMax >= 2);
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return 0xff;
	}

	// 8oCgPʂŃ[v
	addr &= 7;

	// EFCg
	scheduler->Wait(1);

	// fR[h
	addr >>= 1;
	switch (addr) {
		// |[gA
		case 0:
#if defined(PPI_LOG)
			data = joy[0]->ReadPort(ppi.ctl[0]);
			LOG2(Log::Normal, "|[g1ǂݏo Rg[$%02X f[^$%02X",
								ppi.ctl[0], data);
#else
			data = joy[0]->ReadPort(ppi.ctl[0]);
#endif	// PPI_LOG

			// PC7,PC6l
			if (ppi.ctl[0] & 0x80) {
				data &= ~0x40;
			}
			if (ppi.ctl[0] & 0x40) {
				data &= ~0x20;
			}
			return data;

		// |[gB
		case 1:
#if defined(PPI_LOG)
			data = joy[1]->ReadPort(ppi.ctl[1]);
			LOG2(Log::Normal, "|[g2ǂݏo Rg[$%02X f[^$%02X",
								ppi.ctl[1], data);
			return data;
#else
			return joy[1]->ReadPort(ppi.ctl[1]);
#endif	// PPI_LOG

		// |[gC
		case 2:
			return ppi.portc;
	}

	LOG1(Log::Warning, "WX^ǂݍ R%02d", addr);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL PPI::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT_DIAG();

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL PPI::WriteByte(DWORD addr, DWORD data)
{
	DWORD bit;
	int i;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return;
	}

	// 8oCgPʂŃ[v
	addr &= 7;

	// EFCg
	scheduler->Wait(1);

	// |[gCւWrite
	if (addr == 5) {
		// WCXeBbNEADPCMRg[
		SetPortC(data);
		return;
	}

	// [hRg[
	if (addr == 7) {
		if (data < 0x80) {
			// rbgZbgEZbg[h
			i = ((data >> 1) & 0x07);
			bit = (DWORD)(1 << i);
			if (data & 1) {
				SetPortC(DWORD(ppi.portc | bit));
			}
			else {
				SetPortC(DWORD(ppi.portc & ~bit));
			}
			return;
		}

		// [hRg[
		if (data != 0x92) {
			LOG0(Log::Warning, "T|[gO[hw $%02X");
		}
		return;
	}

	LOG2(Log::Warning, "WX^ R%02d <- $%02X",
							addr, data);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL PPI::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);
	ASSERT_DIAG();

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL PPI::ReadOnly(DWORD addr) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(PortMax >= 2);
	ASSERT_DIAG();

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return 0xff;
	}

	// 8oCgPʂŃ[v
	addr &= 7;

	// fR[h
	addr >>= 1;
	switch (addr) {
		// |[gA
		case 0:
			// f[^擾
			data = joy[0]->ReadOnly(ppi.ctl[0]);

			// PC7,PC6l
			if (ppi.ctl[0] & 0x80) {
				data &= ~0x40;
			}
			if (ppi.ctl[0] & 0x40) {
				data &= ~0x20;
			}
			return data;

		// |[gB
		case 1:
			return joy[1]->ReadOnly(ppi.ctl[1]);

		// |[gC
		case 2:
			return ppi.portc;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gCZbg
//
//---------------------------------------------------------------------------
void FASTCALL PPI::SetPortC(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT(PortMax >= 2);
	ASSERT_DIAG();

	// f[^L
	ppi.portc = data;

	// Rg[f[^gݗ(|[gA)...bit0PC4Abit6,7PC6,7
	ppi.ctl[0] = ppi.portc & 0xc0;
	if (ppi.portc & 0x10) {
		ppi.ctl[0] |= 0x01;
	}
#if defined(PPI_LOG)
	LOG1(Log::Normal, "|[g1 Rg[ $%02X", ppi.ctl[0]);
#endif	// PPI_LOG
	joy[0]->Control(ppi.ctl[0]);

	// Rg[f[^gݗ(|[gB)...bit0PC5
	if (ppi.portc & 0x20) {
		ppi.ctl[1] = 0x01;
	}
	else {
		ppi.ctl[1] = 0x00;
	}
#if defined(PPI_LOG)
	LOG1(Log::Normal, "|[g2 Rg[ $%02X", ppi.ctl[1]);
#endif	// PPI_LOG
	joy[1]->Control(ppi.ctl[1]);

	// ADPCMp|bg
	adpcm->SetPanpot(data & 3);

	// ADPCMx䗦
	adpcm->SetRatio((data >> 2) & 3);
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL PPI::GetPPI(ppi_t *buffer)
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT_DIAG();

	// [NRs[
	*buffer = ppi;
}

//---------------------------------------------------------------------------
//
//	WCXeBbNݒ
//
//---------------------------------------------------------------------------
void FASTCALL PPI::SetJoyInfo(int port, const joyinfo_t *info)
{
	ASSERT(this);
	ASSERT((port >= 0) && (port < PortMax));
	ASSERT(info);
	ASSERT(PortMax >= 2);
	ASSERT_DIAG();

	// rāAvĂΉȂ
	if (memcmp(&ppi.info[port], info, sizeof(joyinfo_t)) == 0) {
		return;
	}

	// ۑ
	memcpy(&ppi.info[port], info, sizeof(joyinfo_t));

	// ̃|[gɑΉWCXeBbNfoCXցAʒm
	ASSERT(joy[port]);
	joy[port]->Notify();
}

//---------------------------------------------------------------------------
//
//	WCXeBbN擾
//
//---------------------------------------------------------------------------
const PPI::joyinfo_t* FASTCALL PPI::GetJoyInfo(int port) const
{
	ASSERT(this);
	ASSERT((port >= 0) && (port < PortMax));
	ASSERT(PortMax >= 2);
	ASSERT_DIAG();

	return &(ppi.info[port]);
}

//---------------------------------------------------------------------------
//
//	WCXeBbNfoCX쐬
//
//---------------------------------------------------------------------------
JoyDevice* FASTCALL PPI::CreateJoy(int port, int type)
{
	ASSERT(this);
	ASSERT(type >= 0);
	ASSERT((port >= 0) && (port < PortMax));
	ASSERT(PortMax >= 2);

	// ^Cv
	switch (type) {
		// ڑȂ
		case 0:
			return new JoyDevice(this, port);

		// ATARIW
		case 1:
			return new JoyAtari(this, port);

		// ATARIW+START/SELCT
		case 2:
			return new JoyASS(this, port);

		// TCo[XeBbN(AiO)
		case 3:
			return new JoyCyberA(this, port);

		// TCo[XeBbN(fW^)
		case 4:
			return new JoyCyberD(this, port);

		// MD3{^
		case 5:
			return new JoyMd3(this, port);

		// MD6{^
		case 6:
			return new JoyMd6(this, port);

		// CPSF-SFC
		case 7:
			return new JoyCpsf(this, port);

		// CPSF-MD
		case 8:
			return new JoyCpsfMd(this, port);

		// }WJpbh
		case 9:
			return new JoyMagical(this, port);

		// XPD-1LR
		case 10:
			return new JoyLR(this, port);

		// pbNhppbh
		case 11:
			return new JoyPacl(this, port);

		// BM68pRg[
		case 12:
			return new JoyBM(this, port);

		// ̑
		default:
			ASSERT(FALSE);
			break;
	}

	// ʏAɂ͂Ȃ
	return new JoyDevice(this, port);
}

//===========================================================================
//
//	WCXeBbNfoCX
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyDevice::JoyDevice(PPI *parent, int no)
{
	ASSERT((no >= 0) || (no < PPI::PortMax));

	// ^CvNULL
	id = MAKEID('N', 'U', 'L', 'L');
	type = 0;

	// efoCX(PPI)LA|[gԍݒ
	ppi = parent;
	port = no;

	// E{^ȂAfW^Af[^0
	axes = 0;
	buttons = 0;
	analog = FALSE;
	datas = 0;

	// \
	axis_desc = NULL;
	button_desc = NULL;

	// f[^obt@NULL
	data = NULL;

	// XV`FbNv
	changed = TRUE;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
JoyDevice::~JoyDevice()
{
	// f[^obt@Ή
	if (data) {
		delete[] data;
		data = NULL;
	}
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL JoyDevice::Reset()
{
	ASSERT(this);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyDevice::Save(Fileio *fio, int /*ver*/)
{
	ASSERT(this);
	ASSERT(fio);

	// f[^0ȂZ[uȂ
	if (datas <= 0) {
		ASSERT(datas == 0);
		return TRUE;
	}

	// f[^ۑ
	if (!fio->Write(data, sizeof(DWORD) * datas)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyDevice::Load(Fileio *fio, int /*ver*/)
{
	ASSERT(this);
	ASSERT(fio);

	// f[^0Ȃ烍[hȂ
	if (datas <= 0) {
		ASSERT(datas == 0);
		return TRUE;
	}

	// XV
	changed = TRUE;

	// f[^̂[h
	if (!fio->Read(data, sizeof(DWORD) * datas)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyDevice::ReadPort(DWORD ctl)
{
	ASSERT(this);
	ASSERT((port >= 0) && (port < PPI::PortMax));
	ASSERT(ppi);
	ASSERT(ctl < 0x100);

	// ύXtO`FbN
	if (changed) {
		// tOƂ
		changed = FALSE;

		// f[^쐬
		MakeData();
	}

	// ReadOnlyƓf[^Ԃ
	return ReadOnly(ctl);
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyDevice::ReadOnly(DWORD /*ctl*/) const
{
	ASSERT(this);

	// ڑ
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	Rg[
//
//---------------------------------------------------------------------------
void FASTCALL JoyDevice::Control(DWORD /*ctl*/)
{
	ASSERT(this);
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyDevice::MakeData()
{
	ASSERT(this);
}

//---------------------------------------------------------------------------
//
//	\
//
//---------------------------------------------------------------------------
const char* FASTCALL JoyDevice::GetAxisDesc(int axis) const
{
	ASSERT(this);
	ASSERT(axis >= 0);

	// 𒴂ĂNULL
	if (axis >= axes) {
		return NULL;
	}

	// \e[uȂNULL
	if (!axis_desc) {
		return NULL;
	}

	// \e[uԂ
	return axis_desc[axis];
}

//---------------------------------------------------------------------------
//
//	{^\
//
//---------------------------------------------------------------------------
const char* FASTCALL JoyDevice::GetButtonDesc(int button) const
{
	ASSERT(this);
	ASSERT(button >= 0);

	// {^𒴂ĂNULL
	if (button >= buttons) {
		return NULL;
	}

	// {^\e[uȂNULL
	if (!button_desc) {
		return NULL;
	}

	// {^\e[uԂ
	return button_desc[button];
}

//===========================================================================
//
//	WCXeBbN(ATARIW)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyAtari::JoyAtari(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvATAR
	id = MAKEID('A', 'T', 'A', 'R');
	type = 1;

	// 22{^Af[^1
	axes = 2;
	buttons = 2;
	datas = 1;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyAtari::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);

	// PC41ȂA0xff
	if (ctl & 1) {
		return 0xff;
	}

	// 쐬ς݂̃f[^Ԃ
	return data[0];
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyAtari::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x40;
	}

	// {^B
	if (info->button[1]) {
		data[0] &= ~0x20;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyAtari::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyAtari::ButtonDescTable[] = {
	"A",
	"B"
};

//===========================================================================
//
//	WCXeBbN(ATARIW+START/SELECT)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyASS::JoyASS(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvATSS
	id = MAKEID('A', 'T', 'S', 'S');
	type = 2;

	// 24{^Af[^1
	axes = 2;
	buttons = 4;
	datas = 1;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyASS::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);

	// PC41ȂA0xff
	if (ctl & 1) {
		return 0xff;
	}

	// 쐬ς݂̃f[^Ԃ
	return data[0];
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyASS::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x40;
	}

	// {^B
	if (info->button[1]) {
		data[0] &= ~0x20;
	}

	// START(EƂĕ\)
	if (info->button[2]) {
		data[0] &= ~0x0c;
	}

	// SELECT(㉺Ƃĕ\)
	if (info->button[3]) {
		data[0] &= ~0x03;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyASS::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyASS::ButtonDescTable[] = {
	"A",
	"B",
	"START",
	"SELECT"
};

//===========================================================================
//
//	WCXeBbN(TCo[XeBbNEAiO)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyCyberA::JoyCyberA(PPI *parent, int no) : JoyDevice(parent, no)
{
	int i;

	// ^CvCYBA
	id = MAKEID('C', 'Y', 'B', 'A');
	type = 3;

	// 38{^AAiOAf[^11
	axes = 3;
	buttons = 8;
	analog = TRUE;
	datas = 12;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	for (i=0; i<12; i++) {
		// ACK,L/H,{^
		if (i & 1) {
			data[i] = 0xbf;
		}
		else {
			data[i] = 0x9f;
		}

		// Z^l0x7fƂ
		if ((i >= 2) && (i <= 5)) {
			// AiOf[^H7ɂ
			data[i] &= 0xf7;
		}
	}

	// XPW[擾
	scheduler = (Scheduler*)ppi->GetVM()->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(scheduler);

	// Zbg(Rg[ւꍇɔ)
	Reset();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL JoyCyberA::Reset()
{
	ASSERT(this);
	ASSERT(scheduler);

	// {NX
	JoyDevice::Reset();

	// V[PX
	seq = 0;

	// Rg[0
	ctrl = 0;

	// ԋL
	hus = scheduler->GetTotalTime();
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyCyberA::Save(Fileio *fio, int ver)
{
	ASSERT(this);
	ASSERT(fio);

	// {NX
	if (!JoyDevice::Save(fio, ver)) {
		return FALSE;
	}

	// V[PXZ[u
	if (!fio->Write(&seq, sizeof(seq))) {
		return FALSE;
	}

	// Rg[Z[u
	if (!fio->Write(&ctrl, sizeof(ctrl))) {
		return FALSE;
	}

	// ԂZ[u
	if (!fio->Write(&hus, sizeof(hus))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyCyberA::Load(Fileio *fio, int ver)
{
	ASSERT(this);
	ASSERT(fio);

	// {NX
	if (!JoyDevice::Load(fio, ver)) {
		return FALSE;
	}

	// V[PX[h
	if (!fio->Read(&seq, sizeof(seq))) {
		return FALSE;
	}

	// Rg[[h
	if (!fio->Read(&ctrl, sizeof(ctrl))) {
		return FALSE;
	}

	// Ԃ[h
	if (!fio->Read(&hus, sizeof(hus))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	xݒ(@Ƃ̔rɊÂ)
//	mɂPC4ŏɗĂĂAPA4b5b6܂ł100usȏ
//	Ă镵͋C邪A܂ł̓G~[VȂ
//
//---------------------------------------------------------------------------
#define JOYCYBERA_CYCLE		108

//---------------------------------------------------------------------------
//
//	|[gǂݎ
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyCyberA::ReadPort(DWORD ctl)
{
	DWORD diff;
	DWORD n;

	// V[PX0͖
	if (seq == 0) {
		return 0xff;
	}

	// V[PX12ȏ
	if (seq >= 13) {
		// V[PX0
		seq = 0;
		return 0xff;
	}

	// ύXtO`FbN
	if (changed) {
		// tOƂ
		changed = FALSE;

		// f[^쐬
		MakeData();
	}

	ASSERT((seq >= 1) && (seq <= 12));

	// 擾
	diff = scheduler->GetTotalTime();
	diff -= hus;

	// vZ
	if (diff >= JOYCYBERA_CYCLE) {
		n = diff / JOYCYBERA_CYCLE;
		diff %= JOYCYBERA_CYCLE;

		// V[PXZbg
		if ((seq & 1) == 0) {
			seq--;
		}
		// 2PʂŐi߂
		seq += (2 * n);

		// Ԃ␳
		hus += (JOYCYBERA_CYCLE * n);

		// +1
		if (diff >= (JOYCYBERA_CYCLE / 2)) {
			diff -= (JOYCYBERA_CYCLE / 2);
			seq++;
		}

		// V[PXI[o[΍
		if (seq >= 13) {
			seq = 0;
			return 0xff;
		}
	}
	if (diff >= (JOYCYBERA_CYCLE / 2)) {
		// 㔼ɐݒ
		if (seq & 1) {
			seq++;
		}
	}

	// f[^擾
	return ReadOnly(ctl);
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyCyberA::ReadOnly(DWORD /*ctl*/) const
{
	ASSERT(this);

	// V[PX0͖
	if (seq == 0) {
		return 0xff;
	}

	// V[PX12ȏ
	if (seq >= 13) {
		return 0xff;
	}

	// V[PXɏ]f[^Ԃ
	ASSERT((seq >= 1) && (seq <= 12));
	return data[seq - 1];
}

//---------------------------------------------------------------------------
//
//	Rg[
//
//---------------------------------------------------------------------------
void FASTCALL JoyCyberA::Control(DWORD ctl)
{
	ASSERT(this);
	ASSERT(ctl < 0x100);

	// V[PX0()ƃV[PX11ȍ~
	if ((seq == 0) || (seq >= 11)) {
		// 10ŁAV[PXJn
		if (ctl) {
			// 1ɂ
			ctrl = 1;
		}
		else {
			// 0ɂ
			if (ctrl) {
				// 10ւ̗
				seq = 1;
				hus = scheduler->GetTotalTime();
			}
			ctrl = 0;
		}
		return;
	}

	// V[PX1ȍ~ł́AACK̂ݗL
	ctrl = ctl;
	if (ctl) {
		return;
	}

	// V[PX2PʂŐi߂ʂ
	if ((seq & 1) == 0) {
		seq--;
	}
	seq += 2;

	// ԂL
	hus = scheduler->GetTotalTime();
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyCyberA::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// WCXeBbN擾
	info = ppi->GetJoyInfo(port);

	// data[0]:{^A,{^B,{^C,{^D
	data[0] |= 0x0f;
	if (info->button[0]) {
		data[0] &= ~0x08;
	}
	if (info->button[1]) {
		data[0] &= ~0x04;
	}
	if (info->button[2]) {
		data[0] &= ~0x02;
	}
	if (info->button[3]) {
		data[0] &= ~0x01;
	}

	// data[1]:{^E1,{^E2,{^F,{^G
	data[1] |= 0x0f;
	if (info->button[4]) {
		data[1] &= ~0x08;
	}
	if (info->button[5]) {
		data[1] &= ~0x04;
	}
	if (info->button[6]) {
		data[1] &= ~0x02;
	}
	if (info->button[7]) {
		data[1] &= ~0x01;
	}

	// data[2]:1H
	axis = info->axis[1];
	axis = (axis + 0x800) >> 4;
	data[2] &= 0xf0;
	data[2] |= (axis >> 4);

	// data[3]:2H
	axis = info->axis[0];
	axis = (axis + 0x800) >> 4;
	data[3] &= 0xf0;
	data[3] |= (axis >> 4);

	// data[4]:3H
	axis = info->axis[3];
	axis = (axis + 0x800) >> 4;
	data[4] &= 0xf0;
	data[4] |= (axis >> 4);

	// data[5]:4H(\ƂȂĂB@ł0)
	data[5] &= 0xf0;

	// data[6]:1L
	axis = info->axis[1];
	axis = (axis + 0x800) >> 4;
	data[6] &= 0xf0;
	data[6] |= (axis & 0x0f);

	// data[7]:2L
	axis = info->axis[0];
	axis = (axis + 0x800) >> 4;
	data[7] &= 0xf0;
	data[7] |= (axis & 0x0f);

	// data[8]:3L
	axis = info->axis[3];
	axis = (axis + 0x800) >> 4;
	data[8] &= 0xf0;
	data[8] |= (axis & 0x0f);

	// data[9]:4L(\ƂȂĂB@ł0)
	data[9] &= 0xf0;

	// data[10]:A,B,A',B'
	// A,B̓o[̂̃~j{^AA'B'͒ʏ̉{^
	// o[̂̃~j{^ƂĈ(At^[o[i[II)
	data[10] &= 0xf0;
	data[10] |= 0x0f;
	if (info->button[0]) {
		data[10] &= ~0x08;
	}
	if (info->button[1]) {
		data[10] &= ~0x04;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyCyberA::AxisDescTable[] = {
	"Stick X",
	"Stick Y",
	"Throttle"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyCyberA::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"D",
	"E1",
	"E2",
	"START",
	"SELECT"
};

//===========================================================================
//
//	WCXeBbN(TCo[XeBbNEfW^)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyCyberD::JoyCyberD(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvCYBD
	id = MAKEID('C', 'Y', 'B', 'D');
	type = 4;

	// 36{^Af[^2
	axes = 3;
	buttons = 6;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
	data[1] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyCyberD::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyCyberD::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;
	data[1] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x20;
	}

	// {^B
	if (info->button[1]) {
		data[0] &= ~0x40;
	}

	// XbgUp
	axis = info->axis[2];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x01;
	}
	// XbgDown
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x02;
	}

	// {^C
	if (info->button[2]) {
		data[1] &= ~0x04;
	}

	// {^D
	if (info->button[3]) {
		data[1] &= ~0x08;
	}

	// {^E1
	if (info->button[4]) {
		data[1] &= ~0x20;
	}

	// {^E2
	if (info->button[5]) {
		data[1] &= ~0x40;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyCyberD::AxisDescTable[] = {
	"X",
	"Y",
	"Throttle"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyCyberD::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"D",
	"E1",
	"E2"
};

//===========================================================================
//
//	WCXeBbN(MD3{^)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyMd3::JoyMd3(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvMD3B
	id = MAKEID('M', 'D', '3', 'B');
	type = 5;

	// 24{^Af[^2
	axes = 2;
	buttons = 4;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xf3;
	data[1] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyMd3::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyMd3::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xf3;
	data[1] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x01;
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x02;
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x08;
	}

	// {^B
	if (info->button[1]) {
		data[1] &= ~0x20;
	}

	// {^C
	if (info->button[2]) {
		data[1] &= ~0x40;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x20;
	}

	// X^[g{^
	if (info->button[3]) {
		data[0] &= ~0x40;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyMd3::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyMd3::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"START"
};

//===========================================================================
//
//	WCXeBbN(MD6{^)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyMd6::JoyMd6(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvMD6B
	id = MAKEID('M', 'D', '6', 'B');
	type = 6;

	// 28{^Af[^3
	axes = 2;
	buttons = 8;
	datas = 5;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xf3;
	data[1] = 0xff;
	data[2] = 0xf0;
	data[3] = 0xff;
	data[4] = 0xff;

	// XPW[擾
	scheduler = (Scheduler*)ppi->GetVM()->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(scheduler);

	// Zbg(Rg[ւꍇɔ)
	Reset();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL JoyMd6::Reset()
{
	ASSERT(this);
	ASSERT(scheduler);

	// {NX
	JoyDevice::Reset();

	// V[PXARg[AԂ
	seq = 0;
	ctrl = 0;
	hus = scheduler->GetTotalTime();
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyMd6::Save(Fileio *fio, int ver)
{
	ASSERT(this);
	ASSERT(fio);

	// {NX
	if (!JoyDevice::Save(fio, ver)) {
		return FALSE;
	}

	// V[PXZ[u
	if (!fio->Write(&seq, sizeof(seq))) {
		return FALSE;
	}

	// Rg[Z[u
	if (!fio->Write(&ctrl, sizeof(ctrl))) {
		return FALSE;
	}

	// ԂZ[u
	if (!fio->Write(&hus, sizeof(hus))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL JoyMd6::Load(Fileio *fio, int ver)
{
	ASSERT(this);
	ASSERT(fio);

	// {NX
	if (!JoyDevice::Load(fio, ver)) {
		return FALSE;
	}

	// V[PX[h
	if (!fio->Read(&seq, sizeof(seq))) {
		return FALSE;
	}

	// Rg[[h
	if (!fio->Read(&ctrl, sizeof(ctrl))) {
		return FALSE;
	}

	// Ԃ[h
	if (!fio->Read(&hus, sizeof(hus))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyMd6::ReadOnly(DWORD /*ctl*/) const
{
	ASSERT(this);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);
	ASSERT(data[2] < 0x100);
	ASSERT(data[3] < 0x100);
	ASSERT(data[4] < 0x100);

	// V[PX
	switch (seq) {
		//  CTL=0
		case 0:
			return data[0];

		// 1 CTL=1
		case 1:
			return data[1];

		// 1 CTL=0
		case 2:
			return data[0];

		// 2 CTL=1
		case 3:
			return data[1];

		// 6Bm CTL=0
		case 4:
			return data[2];

		// 6Bm CTL=1
		case 5:
			return data[3];

		// 6Bm CTL=0
		case 6:
			return data[4];

		// 6Bm CTL=1
		case 7:
			return data[1];

		// 6Bm CTL=0
		case 8:
			return data[0];

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	Rg[
//
//---------------------------------------------------------------------------
void FASTCALL JoyMd6::Control(DWORD ctl)
{
	DWORD diff;

	ASSERT(this);
	ASSERT(ctl < 0x100);

	// bit0̂ݕKv
	ctl &= 0x01;

	// KXV
	ctrl = ctl;

	// seq >= 3ȂAŐN1.8ms(3600hus)o߂`FbN
	// o߂Ă΁Aseq=0 or seq=1ɃZbg(LV4)
	if (seq >= 3) {
		diff = scheduler->GetTotalTime();
		diff -= hus;
		if (diff >= 3600) {
			// Zbg
			if (ctl) {
				seq = 1;
				hus = scheduler->GetTotalTime();
			}
			else {
				seq = 0;
			}
			return;
		}
	}

	switch (seq) {
		// V[PXO CTL=0
		case 0:
			// 1ȂAV[PX1ƎԋL
			if (ctl) {
				seq = 1;
				hus = scheduler->GetTotalTime();
			}
			break;

		// ŏ1̌ CTL=1
		case 1:
			// 0ȂAV[PX2
			if (!ctl) {
				seq = 2;
			}
			break;

		// 10̌ CTL=0
		case 2:
			// 1ȂAԃ`FbN
			if (ctl) {
				diff = scheduler->GetTotalTime();
				diff -= hus;
				if (diff <= 2200) {
					// 1.1ms(2200hus)ȉȂ玟̃V[PX(6Bǂݎ)
					seq = 3;
				}
				else {
					// \Ԃ󂢂Ă̂ŁAV[PX1ƓƂ݂Ȃ(3Bǂݎ)
					seq = 1;
					hus = scheduler->GetTotalTime();
				}
			}
			break;

		// 6Bm CTL=1
		case 3:
			if (!ctl) {
				seq = 4;
			}
			break;

		// 6Bm CTL=0
		case 4:
			if (ctl) {
				seq = 5;
			}
			break;

		// 6Bm CTL=1
		case 5:
			if (!ctl) {
				seq = 6;
			}
			break;

		// 6Bm CTL=0
		case 6:
			if (ctl) {
				seq = 7;
			}
			break;

		// 1.8ms̑҂
		case 7:
			if (!ctl) {
				seq = 8;
			}
			break;

		// 1.8ms̑҂
		case 8:
			if (ctl) {
				seq = 7;
			}
			break;

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyMd6::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xf3;
	data[1] = 0xff;
	data[2] = 0xf0;
	data[3] = 0xff;
	data[4] = 0xff;

	// Up(data[0], data[1], data[4])
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
		data[1] &= ~0x01;
		data[4] &= ~0x01;
	}
	// Down(data[0], data[1], data[4])
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
		data[1] &= ~0x02;
		data[4] &= ~0x02;
	}

	// Left(data[1], data[4])
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x04;
		data[4] &= ~0x04;
	}
	// Right(data[1], data[4])
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x08;
		data[4] &= ~0x08;
	}

	// {^B(data[1], data[3], data[4])
	if (info->button[1]) {
		// 3B݊
		data[1] &= ~0x20;

		// (LV4)
		data[3] &= ~0x20;

		// (SFII'patch)
		data[4] &= ~0x40;
	}

	// {^C(data[1], data[3])
	if (info->button[2]) {
		// 3B݊
		data[1] &= ~0x40;

		// (SFII'patch)
		data[3] &= ~0x20;

		// (LV4)
		data[3] &= ~0x40;
	}

	// {^A(data[0], data[2], data[4])
	if (info->button[0]) {
		// 3B݊
		data[0] &= ~0x20;

		// 6B}[J
		data[2] &= ~0x20;

		// (SFII'patch)
		data[4] &= ~0x20;
	}

	// X^[g{^(data[0], data[2])
	if (info->button[6]) {
		// 3B݊
		data[0] &= ~0x40;

		// 6B}[J
		data[2] &= ~0x40;
	}

	// {^X(data[3])
	if (info->button[3]) {
		data[3] &= ~0x04;
	}

	// {^Y(data[3])
	if (info->button[4]) {
		data[3] &= ~0x02;
	}

	// {^Z(data[3])
	if (info->button[5]) {
		data[3] &= ~0x01;
	}

	// MODE{^(data[3])
	if (info->button[7]) {
		data[3] &= ~0x08;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyMd6::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyMd6::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"X",
	"Y",
	"Z",
	"START",
	"MODE"
};

//===========================================================================
//
//	WCXeBbN(CPSF-SFC)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyCpsf::JoyCpsf(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvCPSF
	id = MAKEID('C', 'P', 'S', 'F');
	type = 7;

	// 28{^Af[^2
	axes = 2;
	buttons = 8;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
	data[1] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyCpsf::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyCpsf::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;
	data[1] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^Y
	if (info->button[0]) {
		data[1] &= ~0x02;
	}

	// {^X
	if (info->button[1]) {
		data[1] &= ~0x04;
	}

	// {^B
	if (info->button[2]) {
		data[0] &= ~0x40;
	}

	// {^A
	if (info->button[3]) {
		data[0] &= ~0x20;
	}

	// {^L
	if (info->button[4]) {
		data[1] &= ~0x20;
	}

	// {^R
	if (info->button[5]) {
		data[1] &= ~0x01;
	}

	// X^[g{^
	if (info->button[6]) {
		data[1] &= ~0x40;
	}

	// ZNg{^
	if (info->button[7]) {
		data[1] &= ~0x08;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyCpsf::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyCpsf::ButtonDescTable[] = {
	"Y",
	"X",
	"B",
	"A",
	"L",
	"R",
	"START",
	"SELECT"
};

//===========================================================================
//
//	WCXeBbN(CPSF-MD)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyCpsfMd::JoyCpsfMd(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvCPSM
	id = MAKEID('C', 'P', 'S', 'M');
	type = 8;

	// 28{^Af[^2
	axes = 2;
	buttons = 8;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
	data[1] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyCpsfMd::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyCpsfMd::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;
	data[1] = 0xff;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x20;
	}

	// {^B
	if (info->button[1]) {
		data[0] &= ~0x40;
	}

	// {^C
	if (info->button[2]) {
		data[1] &= ~0x20;
	}

	// {^X
	if (info->button[3]) {
		data[1] &= ~0x04;
	}

	// {^Y
	if (info->button[4]) {
		data[1] &= ~0x02;
	}

	// {^Z
	if (info->button[5]) {
		data[1] &= ~0x01;
	}

	// X^[g{^
	if (info->button[6]) {
		data[1] &= ~0x40;
	}

	// MODE{^
	if (info->button[7]) {
		data[1] &= ~0x08;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyCpsfMd::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyCpsfMd::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"X",
	"Y",
	"Z",
	"START",
	"MODE"
};

//===========================================================================
//
//	WCXeBbN(}WJpbh)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyMagical::JoyMagical(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvMAGI
	id = MAKEID('M', 'A', 'G', 'I');
	type = 9;

	// 26{^Af[^2
	axes = 2;
	buttons = 6;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
	data[1] = 0xfc;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyMagical::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyMagical::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;
	data[1] = 0xfc;

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// Left
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// Right
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x40;
	}

	// {^B
	if (info->button[1]) {
		data[1] &= ~0x40;
	}

	// {^C
	if (info->button[2]) {
		data[0] &= ~0x20;
	}

	// {^D
	if (info->button[3]) {
		data[1] &= ~0x40;
	}

	// {^R
	if (info->button[4]) {
		data[1] &= ~0x08;
	}

	// {^L
	if (info->button[5]) {
		data[1] &= ~0x04;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyMagical::AxisDescTable[] = {
	"X",
	"Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyMagical::ButtonDescTable[] = {
	"A",
	"B",
	"C",
	"D",
	"R",
	"L"
};

//===========================================================================
//
//	WCXeBbN(XPD-1LR)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyLR::JoyLR(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvXPLR
	id = MAKEID('X', 'P', 'L', 'R');
	type = 10;

	// 42{^Af[^2
	axes = 4;
	buttons = 2;
	datas = 2;

	// \e[u
	axis_desc = AxisDescTable;
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
	data[1] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyLR::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);
	ASSERT(data[1] < 0x100);

	// PC4ɂĕ
	if (ctl & 1) {
		return data[1];
	}
	else {
		return data[0];
	}
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyLR::MakeData()
{
	const PPI::joyinfo_t *info;
	DWORD axis;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;
	data[1] = 0xff;

	// EUp
	axis = info->axis[3];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x01;
	}
	// EDown
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x02;
	}

	// ELeft
	axis = info->axis[2];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[1] &= ~0x04;
	}
	// ERight
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[1] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[1] &= ~0x40;
	}

	// {^B
	if (info->button[1]) {
		data[1] &= ~0x20;
	}

	// Up
	axis = info->axis[1];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x01;
	}
	// Down
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x02;
	}

	// ELeft
	axis = info->axis[0];
	if ((axis >= 0xfffff800) && (axis < 0xfffffc00)) {
		data[0] &= ~0x04;
	}
	// ERight
	if ((axis >= 0x00000400) && (axis < 0x00000800)) {
		data[0] &= ~0x08;
	}

	// {^A
	if (info->button[0]) {
		data[0] &= ~0x40;
	}

	// {^B
	if (info->button[1]) {
		data[0] &= ~0x20;
	}
}

//---------------------------------------------------------------------------
//
//	\e[u
//
//---------------------------------------------------------------------------
const char* JoyLR::AxisDescTable[] = {
	"Left-X",
	"Left-Y",
	"Right-X",
	"Right-Y"
};

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyLR::ButtonDescTable[] = {
	"A",
	"B"
};

//===========================================================================
//
//	WCXeBbN(pbNhppbh)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyPacl::JoyPacl(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvPACL
	id = MAKEID('P', 'A', 'C', 'L');
	type = 11;

	// 03{^Af[^1
	axes = 0;
	buttons = 3;
	datas = 1;

	// \e[u
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyPacl::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);

	// PC41ȂA0xff
	if (ctl & 1) {
		return 0xff;
	}

	// 쐬ς݂̃f[^Ԃ
	return data[0];
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyPacl::MakeData()
{
	const PPI::joyinfo_t *info;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;

	// {^A(Left)
	if (info->button[0]) {
		data[0] &= ~0x04;
	}

	// {^B(Jump)
	if (info->button[1]) {
		data[0] &= ~0x20;
	}

	// {^C(Right)
	if (info->button[2]) {
		data[0] &= ~0x08;
	}
}

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyPacl::ButtonDescTable[] = {
	"Left",
	"Jump",
	"Right",
};

//===========================================================================
//
//	WCXeBbN(BM68pRg[)
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
JoyBM::JoyBM(PPI *parent, int no) : JoyDevice(parent, no)
{
	// ^CvBM68
	id = MAKEID('B', 'M', '6', '8');
	type = 12;

	// 06{^Af[^1
	axes = 0;
	buttons = 6;
	datas = 1;

	// \e[u
	button_desc = ButtonDescTable;

	// f[^obt@m
	data = new DWORD[datas];

	// f[^ݒ
	data[0] = 0xff;
}

//---------------------------------------------------------------------------
//
//	|[gǂݎ(Read Only)
//
//---------------------------------------------------------------------------
DWORD FASTCALL JoyBM::ReadOnly(DWORD ctl) const
{
	ASSERT(this);
	ASSERT(ctl < 0x100);
	ASSERT(data[0] < 0x100);

	// PC41ȂA0xff
	if (ctl & 1) {
		return 0xff;
	}

	// 쐬ς݂̃f[^Ԃ
	return data[0];
}

//---------------------------------------------------------------------------
//
//	f[^쐬
//
//---------------------------------------------------------------------------
void FASTCALL JoyBM::MakeData()
{
	const PPI::joyinfo_t *info;

	ASSERT(this);
	ASSERT(ppi);

	// f[^
	info = ppi->GetJoyInfo(port);
	data[0] = 0xff;

	// {^1(C)
	if (info->button[0]) {
		data[0] &= ~0x08;
	}

	// {^2(C+,D-)
	if (info->button[1]) {
		data[0] &= ~0x04;
	}

	// {^3(D)
	if (info->button[2]) {
		data[0] &= ~0x40;
	}

	// {^4(D+,E-)
	if (info->button[3]) {
		data[0] &= ~0x20;
	}

	// {^5(E)
	if (info->button[4]) {
		data[0] &= ~0x02;
	}

	// {^F(Hat)
	if (info->button[5]) {
		data[0] &= ~0x01;
	}
}

//---------------------------------------------------------------------------
//
//	{^\e[u
//
//---------------------------------------------------------------------------
const char* JoyBM::ButtonDescTable[] = {
	"C",
	"C#",
	"D",
	"D#",
	"E",
	"HiHat"
};
