//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ SASI ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "fileio.h"
#include "dmac.h"
#include "iosc.h"
#include "sram.h"
#include "config.h"
#include "disk.h"
#include "memory.h"
#include "scsi.h"
#include "sasi.h"

//===========================================================================
//
//	SASI
//
//===========================================================================
//#define SASI_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SASI::SASI(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('S', 'A', 'S', 'I');
	dev.desc = "SASI (IOSC-2)";

	// JnAhXAIAhX
	memdev.first = 0xe96000;
	memdev.last = 0xe97fff;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::Init()
{
	int i;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// DMAC擾
	dmac = (DMAC*)vm->SearchDevice(MAKEID('D', 'M', 'A', 'C'));
	ASSERT(dmac);

	// IOSC擾
	iosc = (IOSC*)vm->SearchDevice(MAKEID('I', 'O', 'S', 'C'));
	ASSERT(iosc);

	// SRAM擾
	sram = (SRAM*)vm->SearchDevice(MAKEID('S', 'R', 'A', 'M'));
	ASSERT(sram);

	// SCSI擾
	scsi = (SCSI*)vm->SearchDevice(MAKEID('S', 'C', 'S', 'I'));
	ASSERT(scsi);

	// f[^
	memset(&sasi, 0, sizeof(sasi));
	sxsicpu = FALSE;

	// fBXN쐬
	for (i=0; i<SASIMax; i++) {
		sasi.disk[i] = new Disk(this);
	}

	// JgȂ
	sasi.current = NULL;

	// Cxg쐬
	event.SetDevice(this);
	event.SetDesc("Data Transfer");
	event.SetUser(0);
	event.SetTime(0);
	scheduler->AddEvent(&event);

	// ̑
	cache_wb = TRUE;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Cleanup()
{
	int i;

	ASSERT(this);

	// fBXN폜
	for (i=0; i<SASIMax; i++) {
		if (sasi.disk[i]) {
			delete sasi.disk[i];
			sasi.disk[i] = NULL;
		}
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Reset()
{
	Memory *memory;
	Memory::memtype type;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// SCSI^Cv(ɖ₢킹)
	memory = (Memory*)vm->SearchDevice(MAKEID('M', 'E', 'M', ' '));
	ASSERT(memory);
	type = memory->GetMemType();
	switch (type) {
		// SASÎ
		case Memory::None:
		case Memory::SASI:
			sasi.scsi_type = 0;
			break;

		// Ot
		case Memory::SCSIExt:
			sasi.scsi_type = 1;
			break;

		// ̑()
		default:
			sasi.scsi_type = 2;
			break;
	}

	// SCSIgꍇASxSIƂ͔rƂ(SxSI֎~)
	if (sasi.scsi_type != 0) {
		if (sasi.sxsi_drives != 0) {
			sasi.sxsi_drives = 0;
			Construct();
		}
	}

	// XCb`
	if (sasi.memsw) {
		// SASI݂ꍇ̂݁AZbg
		if (sasi.scsi_type < 2) {
			// $ED005A:SASIfBXN
			sram->SetMemSw(0x5a, sasi.sasi_drives);
		}
		else {
			// SASIC^tF[XȂ̂0
			sram->SetMemSw(0x5a, 0x00);
		}

		// SCSI݂Ȃꍇ̂݁ANA
		if (sasi.scsi_type == 0) {
			// $ED006F:SCSItO('V'SCSIL)
			sram->SetMemSw(0x6f, 0x00);
			// $ED0070:SCSI({/Ot)+{SCSI ID
			sram->SetMemSw(0x70, 0x07);
			// $ED0071:SCSIɂSASIG~[VtO
			sram->SetMemSw(0x71, 0x00);
		}
	}

	// CxgZbg
	event.SetUser(0);
	event.SetTime(0);

	// oXZbg
	BusFree();

	// JgfoCXȂ
	sasi.current = NULL;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::Save(Fileio *fio, int ver)
{
	size_t sz;
	int i;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// fBXNtbV
	for (i=0; i<SASIMax; i++) {
		ASSERT(sasi.disk[i]);
		if (!sasi.disk[i]->Flush()) {
			return FALSE;
		}
	}

	// TCYZ[u
	sz = sizeof(sasi_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&sasi, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!event.Save(fio, ver)) {
		return FALSE;
	}

	// pXZ[u
	for (i=0; i<SASIMax; i++) {
		if (!sasihd[i].Save(fio, ver)) {
			return FALSE;
		}
	}
	for (i=0; i<SCSIMax; i++) {
		if (!scsihd[i].Save(fio, ver)) {
			return FALSE;
		}
	}
	if (!scsimo.Save(fio, ver)) {
		return FALSE;
	}

	// version2.02g
	if (!fio->Write(&sxsicpu, sizeof(sxsicpu))) {
		return FALSE;
	}

	// version2.03g
	for (i=0; i<SASIMax; i++) {
		ASSERT(sasi.disk[i]);
		if (!sasi.disk[i]->Save(fio, ver)) {
			return FALSE;
		}
	}

	// version2.06g
	if (!fio->Write(&cache_wb, sizeof(cache_wb))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::Load(Fileio *fio, int ver)
{
	sasi_t buf;
	size_t sz;
	int i;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "[h");

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(sasi_t)) {
		return FALSE;
	}

	// obt@֎̂[h
	if (!fio->Read(&buf, (int)sz)) {
		return FALSE;
	}

	// fBXN
	for (i=0; i<SASIMax; i++) {
		ASSERT(sasi.disk[i]);
		delete sasi.disk[i];
		sasi.disk[i] = new Disk(this);
	}

	// |C^ڂ
	for (i=0; i<SASIMax; i++) {
		buf.disk[i] = sasi.disk[i];
	}

	// ړ
	sasi = buf;
	sasi.mo = NULL;
	sasi.current = NULL;

	// Cxg[h
	if (!event.Load(fio, ver)) {
		return FALSE;
	}

	// pX[h
	for (i=0; i<SASIMax; i++) {
		if (!sasihd[i].Load(fio, ver)) {
			return FALSE;
		}
	}
	for (i=0; i<SCSIMax; i++) {
		if (!scsihd[i].Load(fio, ver)) {
			return FALSE;
		}
	}
	if (!scsimo.Load(fio, ver)) {
		return FALSE;
	}

	// version2.02g
	sxsicpu = FALSE;
	if (ver >= 0x0202) {
		if (!fio->Read(&sxsicpu, sizeof(sxsicpu))) {
			return FALSE;
		}
	}

	// fBXN蒼
	Construct();

	// version2.03g
	if (ver >= 0x0203) {
		for (i=0; i<SASIMax; i++) {
			ASSERT(sasi.disk[i]);
			if (!sasi.disk[i]->Load(fio, ver)) {
				return FALSE;
			}
		}
	}

	// version2.06g
	if (ver >= 0x0206) {
		if(!fio->Read(&cache_wb, sizeof(cache_wb))) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL SASI::ApplyCfg(const Config *config)
{
	int i;

	ASSERT(this);
	ASSERT(config);

	LOG0(Log::Normal, "ݒKp");

	// SASIhCu
	sasi.sasi_drives = config->sasi_drives;

	// XCb`XV
	sasi.memsw = config->sasi_sramsync;

	// SASIt@C
	for (i=0; i<SASIMax; i++) {
		sasihd[i].SetPath(config->sasi_file[i]);
	}

	// peBHt
	sasi.parity = config->sasi_parity;

	// SCSIhCu
	sasi.sxsi_drives = config->sxsi_drives;

	// SCSIt@C
	for (i=0; i<SCSIMax; i++) {
		scsihd[i].SetPath(config->sxsi_file[i]);
	}

	// MODtO
	sasi.mo_first = config->sxsi_mofirst;

	// LbV[h
	cache_wb = config->cache_disk;

	// fBXNč\z
	Construct();
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SASI::ReadByte(DWORD addr)
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// SCSI
	if (sasi.scsi_type >= 2) {
		// 0x40PʂŃ[v
		addr &= 0x3f;

		// ̈`FbN
		if (addr >= 0x20) {
			// SCSÏ
			return scsi->ReadByte(addr - 0x20);
		}
		if ((addr & 1) == 0) {
			// oCg̓fR[hĂȂ
			return 0xff;
		}
		addr &= 0x07;
		if (addr >= 4) {
			return 0xff;
		}
		return 0;
	}

	// AhX̂
	if (addr & 1) {
		// 8oCgPʂŃ[v
		addr &= 0x07;

		// EFCg
		scheduler->Wait(1);

		// f[^WX^
		if (addr == 1) {
			return ReadData();
		}

		// Xe[^XWX^
		if (addr == 3) {
			data = 0;
			if (sasi.msg) {
				data |= 0x10;
			}
			if (sasi.cd) {
				data |= 0x08;
			}
			if (sasi.io) {
				data |= 0x04;
			}
			if (sasi.bsy) {
				data |= 0x02;
			}
			if (sasi.req) {
				data |= 0x01;
			}
			return data;
		}

		// ȊÕWX^Write Only
		return 0xff;
	}

	// AhX̓fR[hĂȂ
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SASI::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL SASI::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);

	// SCSI
	if (sasi.scsi_type >= 2) {
		// 0x40PʂŃ[v
		addr &= 0x3f;

		// ̈`FbN
		if (addr >= 0x20) {
			// SCSÏ
			scsi->WriteByte(addr - 0x20, data);
		}
		// SASḮAȂ
		return;
	}

	// AhX̂
	if (addr & 1) {
		// 8oCgPʂŃ[v
		addr &= 0x07;
		addr >>= 1;

		// EFCg
		scheduler->Wait(1);

		switch (addr) {
			// f[^
			case 0:
				WriteData(data);
				return;

			// SELf[^
			case 1:
				sasi.sel = FALSE;
				WriteData(data);
				return;

			// oXZbg
			case 2:
#if defined(SASI_LOG)
				LOG0(Log::Normal, "oXZbg");
#endif	// SASI_LOG
				BusFree();
				return;

			// SELf[^
			case 3:
				sasi.sel = TRUE;
				WriteData(data);
				return;
		}

		// ɂ͗Ȃ
		ASSERT(FALSE);
		return;
	}

	// AhX̓fR[hĂȂ
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL SASI::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL SASI::ReadOnly(DWORD addr) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// SCSI
	if (sasi.scsi_type >= 2) {
		// 0x40PʂŃ[v
		addr &= 0x3f;

		// ̈`FbN
		if (addr >= 0x20) {
			// SCSÏ
			return scsi->ReadOnly(addr - 0x20);
		}
		if ((addr & 1) == 0) {
			// oCg̓fR[hĂȂ
			return 0xff;
		}
		addr &= 0x07;
		if (addr >= 4) {
			return 0xff;
		}
		return 0;
	}

	// AhX̂
	if (addr & 1) {
		// 8oCgPʂŃ[v
		addr &= 0x07;
		addr >>= 1;

		switch (addr) {
			// f[^WX^
			case 0:
				return 0;

			// Xe[^XWX^
			case 1:
				data = 0;
				if (sasi.msg) {
					data |= 0x10;
				}
				if (sasi.cd) {
					data |= 0x08;
				}
				if (sasi.io) {
					data |= 0x04;
				}
				if (sasi.bsy) {
					data |= 0x02;
				}
				if (sasi.req) {
					data |= 0x01;
				}
				return data;
		}

		// ȊÕWX^Write Only
		return 0xff;
	}

	// AhX̓fR[hĂȂ
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::Callback(Event *ev)
{
	int thres;

	ASSERT(this);
	ASSERT(ev);

	switch (ev->GetUser()) {
		// ʏf[^ zXgRg[
		case 0:
			// Ԓ
			if (ev->GetTime() != 32) {
				ev->SetTime(32);
			}

			// NGXg
			sasi.req = TRUE;
			dmac->ReqDMA(1);

			// DMA]̌ʂɂĂ͑
			return TRUE;

		// ubNf[^ Rg[zXg
		case 1:
			// ԂĐݒ
			if (ev->GetTime() != 48) {
				ev->SetTime(48);
			}

			// DMAANeBułȂ΁ANGXgݒ̂
			if (!dmac->IsAct(1)) {
				if ((sasi.phase == read) || (sasi.phase == write)) {
					sasi.req = TRUE;
				}

				// Cxgp
				return TRUE;
			}

			// 1̃CxgŁA]CPUp[2/3]
			thres = (int)scheduler->GetCPUSpeed();
			thres = (thres * 2) / 3;
			while ((sasi.phase == read) || (sasi.phase == write)) {
				// CPUp[Ȃrőł؂
				if (scheduler->GetCPUCycle() > thres) {
					break;
				}

				// NGXgݒ
				sasi.req = TRUE;

				// SxSI CPUtOݒ肳Ă΁A܂(CPU])
				if (sxsicpu) {
					LOG0(Log::Warning, "SxSI CPU]o");
					break;
				}

				// DMANGXg
				dmac->ReqDMA(1);
				if (sasi.req) {
#if defined(SASI_LOG)
					LOG0(Log::Normal, "DMA or CPU]҂");
#endif
					break;
				}
			}
			return TRUE;

		// ȊO͂肦Ȃ
		default:
			ASSERT(FALSE);
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL SASI::GetSASI(sasi_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// [NRs[
	*buffer = sasi;
}

//---------------------------------------------------------------------------
//
//	fBXNč\z
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Construct()
{
	int i;
	int index;
	int scmo;
	int schd;
	int map[SASIMax];
	Filepath path;
	Filepath mopath;
	BOOL moready;
	BOOL mowritep;
	BOOL moattn;
	SASIHD *sasitmp;
	SCSIHD *scsitmp;

	ASSERT(this);
	ASSERT((sasi.sasi_drives >= 0) && (sasi.sasi_drives <= SASIMax));
	ASSERT((sasi.sxsi_drives >= 0) && (sasi.sxsi_drives <= 7));

	// MOhCȕԂۑAU֎~
	moready = FALSE;
	moattn = FALSE;
	mowritep = FALSE;
	if (sasi.mo) {
		moready = sasi.mo->IsReady();
		if (moready) {
			sasi.mo->GetPath(mopath);
			mowritep = sasi.mo->IsWriteP();
			moattn = sasi.mo->IsAttn();
		}
	}
	sasi.mo = NULL;

	// LSCSI-MO, SCSI-HDZo
	i = (sasi.sasi_drives + 1) >> 1;
	schd = 7 - i;

	// SASI\ȍőSCSIhCu
	if (schd > 0) {
		// ݒSCSIhCuɐ
		if (sasi.sxsi_drives < schd) {
			// SCSIhCum
			schd = sasi.sxsi_drives;
		}
	}

	// MO̊蓖
	if (schd > 0) {
		// hCu1ȏ゠΁AKMO蓖Ă
		scmo = 1;
		schd--;
	}
	else {
		// hCuȂ̂ŁAMO,HDƂ0
		scmo = 0;
		schd = 0;
	}

	// peBȂSxSIgps
	if (!sasi.parity) {
		scmo = 0;
		schd = 0;
	}
#if defined(SASI_LOG)
	LOG3(Log::Normal, "Ċ蓖 SASI-HD:%d SCSI-HD:%d SCSI-MO:%d",
					sasi.sasi_drives, schd, scmo);
#endif	// SASI_LOG

	// }bv쐬(0:NULL 1:SASI-HD 2:SCSI-HD 3:SCSI-MO)
	for (i=0; i<SASIMax; i++) {
		// ׂNULL
		map[i] = 0;
	}
	for (i=0; i<sasi.sasi_drives; i++) {
		// SASI
		map[i] = 1;
	}
	if (scmo > 0) {
		// SCSI
		index = ((sasi.sasi_drives + 1) >> 1) << 1;
		if (sasi.mo_first) {
			// MOD
			map[index] = 3;
			index += 2;

			// SCSI-HD[v
			for (i=0; i<schd; i++) {
				map[index] = 2;
				index += 2;
			}
		}
		else {
			// HDD
			for (i=0; i<schd; i++) {
				map[index] = 2;
				index += 2;
			}

			// ŌMO
			map[index] = 3;
		}
		ASSERT(index <= 16);
	}

	// SCSIn[hfBXN̘AԃCfbNXNA
	index = 0;

	// [v
	for (i=0; i<SASIMax; i++) {
		switch (map[i]) {
			// kfBXN
			case 0:
				// kfBXN
				if (!sasi.disk[i]->IsNULL()) {
					// kfBXNɍւ
					delete sasi.disk[i];
					sasi.disk[i] = new Disk(this);
				}
				break;

			// SASIn[hfBXN
			case 1:
				// SASIn[hfBXN
				if (sasi.disk[i]->IsSASI()) {
					// pX擾Avok
					sasi.disk[i]->GetPath(path);
					if (path.CmpPath(sasihd[i])) {
						// pXvĂ
						break;
					}
				}

				// SASIn[hfBXN쐬ăI[v݂
				sasitmp = new SASIHD(this);
				if (sasitmp->Open(sasihd[i])) {
					// LUNݒ
					sasitmp->SetLUN(i & 1);
					// ւ
					delete sasi.disk[i];
					sasi.disk[i] = sasitmp;
				}
				else {
					// G[
					delete sasitmp;
					delete sasi.disk[i];
					sasi.disk[i] = new Disk(this);
				}
				break;

			// SCSIn[hfBXN
			case 2:
				// SCSIn[hfBXN
				if (sasi.disk[i]->GetID() == MAKEID('S', 'C', 'H', 'D')) {
					// pX擾Avok
					sasi.disk[i]->GetPath(path);
					if (path.CmpPath(scsihd[index])) {
						// pXvĂ
						index++;
						break;
					}
				}

				// SCSIn[hfBXN쐬ăI[v݂
				scsitmp = new SCSIHD(this);
				if (scsitmp->Open(scsihd[index])) {
					// ւ
					delete sasi.disk[i];
					sasi.disk[i] = scsitmp;
				}
				else {
					// G[
					delete scsitmp;
					delete sasi.disk[i];
					sasi.disk[i] = new Disk(this);
				}
				index++;
				break;

			// SCSICfBXN
			case 3:
				// SCSICfBXN
				if (sasi.disk[i]->GetID() == MAKEID('S', 'C', 'M', 'O')) {
					// L
					sasi.mo = (SCSIMO*)sasi.disk[i];
				}
				else {
					// SCSICfBXN쐬ċL
					delete sasi.disk[i];
					sasi.disk[i] = new SCSIMO(this);
					sasi.mo = (SCSIMO*)sasi.disk[i];
				}

				// pĂȂ΁AăI[v
				if (moready) {
					if (!sasi.mo->IsReady()) {
						if (sasi.mo->Open(mopath, moattn)) {
							sasi.mo->WriteP(mowritep);
						}
					}
				}
				break;

			// ̑(蓾Ȃ)
			default:
				ASSERT(FALSE);
				break;
		}
	}

	// BUSYȂAsasi.currentXV
	if (sasi.bsy) {
		// hCu쐬
		ASSERT(sasi.ctrl < 8);
		index = sasi.ctrl << 1;
		if (sasi.cmd[1] & 0x20) {
			index++;
		}

		// Jgݒ
		sasi.current = sasi.disk[index];
		ASSERT(sasi.current);
	}
}

//---------------------------------------------------------------------------
//
//	HD BUSY`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsBusy() const
{
	ASSERT(this);

	// SASI
	if (sasi.bsy) {
		return TRUE;
	}

	// SCSI
	if (scsi->IsBusy()) {
		return TRUE;
	}

	// ǂFALSE
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	BUSYfoCX擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SASI::GetBusyDevice() const
{
	ASSERT(this);

	// SASI
	if (sasi.bsy) {
		// ZNVtF[YIstF[YJn܂ł̊Ԃ
		// ANZXΏۃfoCXm肵Ȃ
		if (!sasi.current) {
			return 0;
		}
		if (sasi.current->IsNULL()) {
			return 0;
		}

		return sasi.current->GetID();
	}

	// SCSI
	return scsi->GetBusyDevice();
}

//---------------------------------------------------------------------------
//
//	f[^ǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SASI::ReadData()
{
	DWORD data;

	ASSERT(this);

	switch (sasi.phase) {
		// Xe[^XtF[YȂAXe[^XnăbZ[WtF[Y
		case status:
			data = sasi.status;
			sasi.req = FALSE;

			// 荞݃Zbg˂
			iosc->IntHDC(FALSE);
			Message();

			// Cxg~
			event.SetTime(0);
			return data;

		// bZ[WtF[YȂAbZ[WnăoXt[
		case message:
			data = sasi.message;
			sasi.req = FALSE;
			BusFree();

			// Cxg~
			event.SetTime(0);
			return data;

		// ǂݍ݃tF[YȂA]Xe[^XtF[Y
		case read:
			// Non-DMA]ȂASxSI CPUtOgOω
			if (!dmac->IsDMA()) {
#if defined(SASI_LOG)
				LOG1(Log::Normal, "f[^INtF[Y CPU] ItZbg=%d", sasi.offset);
#endif	// SASI_LOG
				sxsicpu = !sxsicpu;
			}

			data = (DWORD)sasi.buffer[sasi.offset];
			sasi.offset++;
			sasi.length--;
			sasi.req = FALSE;

			// OX`FbN
			if (sasi.length == 0) {
				// ]ׂubN
				sasi.blocks--;

				// ŏI
				if (sasi.blocks == 0) {
					// Cxg~AXe[^XtF[Y
					event.SetTime(0);
					Status();
					return data;
				}

				// ̃ubNǂ
				sasi.length = sasi.current->Read(sasi.buffer, sasi.next);
				if (sasi.length <= 0) {
					// G[
					Error();
					return data;
				}

				// ̃ubNokA[Nݒ
				sasi.offset = 0;
				sasi.next++;
			}

			// ǂݍ݌p
			return data;
	}

	// Cxg~
	event.SetTime(0);

	// ȊÕIy[V͑z肹
	if (sasi.phase == busfree) {
#if defined(SASI_LOG)
		LOG0(Log::Normal, "oXt[Ԃł̓ǂݍ݁B0Ԃ");
#endif	// SASI_LOG
		return 0;
	}

	LOG0(Log::Warning, "z肵ĂȂf[^ǂݍ");
	BusFree();
	return 0;
}

//---------------------------------------------------------------------------
//
//	f[^
//
//---------------------------------------------------------------------------
void FASTCALL SASI::WriteData(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	switch (sasi.phase) {
		// oXt[tF[YȂAɋ̂̓ZNVtF[Y
		case busfree:
			if (sasi.sel) {
				Selection(data);
			}
#if defined(SASI_LOG)
			else {
				LOG0(Log::Normal, "oXt[ԁASELł̏݁B");
			}
#endif	// SASI_LOG
			event.SetTime(0);
			return;

		// ZNVtF[YȂASEL߂ăR}htF[Y
		case selection:
			if (!sasi.sel) {
				Command();
				return;
			}
			event.SetTime(0);
			break;

		// R}htF[YȂA6/10oCgM
		case command:
			// ŏ̃f[^(ItZbg0)ɂ背OXĐݒ
			sasi.cmd[sasi.offset] = data;
			if (sasi.offset == 0) {
				if ((data >= 0x20) && (data <= 0x3f)) {
					// 10oCgCDB
					sasi.length = 10;
				}
			}
			sasi.offset++;
			sasi.length--;
			sasi.req = FALSE;

			// I肩
			if (sasi.length == 0) {
				// Cxg~ĎstF[Y
				event.SetTime(0);
				Execute();
				return;
			}
			// Cxgp
			return;

		// ݃tF[YȂA]Xe[^XtF[Y
		case write:
			// Non-DMA]ȂASxSI CPUtOgOω
			if (!dmac->IsDMA()) {
#if defined(SASI_LOG)
				LOG1(Log::Normal, "f[^OUTtF[Y CPU] ItZbg=%d", sasi.offset);
#endif	// SASI_LOG
				sxsicpu = !sxsicpu;
			}

			sasi.buffer[sasi.offset] = (BYTE)data;
			sasi.offset++;
			sasi.length--;
			sasi.req = FALSE;

			// I肩
			if (sasi.length > 0) {
				return;
			}

			// WRITE(6)EWRITE(10)EWRITE&VERIFY͕ʏ
			switch (sasi.cmd[0]) {
				case 0x0a:
				case 0x2a:
				case 0x2e:
					break;
				// WRITE DATAȊÕR}h͂ŏI
				default:
					event.SetTime(0);
					Status();
					return;
			}

			// WRITER}hȂÃ݂obt@ŏ
			if (!sasi.current->Write(sasi.buffer, sasi.next - 1)) {
				// G[
				Error();
				return;
			}

			// CgX[LbV
			if (!cache_wb) {
				sasi.current->Flush();
			}

			// ]ubN
			sasi.blocks--;

			// Ō̃ubN
			if (sasi.blocks == 0) {
				// Cxg~ăXe[^XtF[Y
				event.SetTime(0);
				Status();
				return;
			}

			// ̃ubN
			sasi.length = sasi.current->WriteCheck(sasi.next);
			if (sasi.length <= 0) {
				// G[
				Error();
				return;
			}

			// ̃ubN]
			sasi.next++;
			sasi.offset = 0;

			// Cxgp
			return;
	}

	// Cxg~
	event.SetTime(0);

	// ȊÕIy[V͑z肹
	LOG1(Log::Warning, "z肵ĂȂf[^ $%02X", data);
	BusFree();
}

//---------------------------------------------------------------------------
//
//	oXt[tF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::BusFree()
{
	ASSERT(this);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "oXt[tF[Y");
#endif	// SASI_LOG

	// oXZbg
	sasi.msg = FALSE;
	sasi.cd = FALSE;
	sasi.io = FALSE;
	sasi.bsy = FALSE;
	sasi.req = FALSE;

	// oXt[tF[Y
	sasi.phase = busfree;

	// Cxg~
	event.SetTime(0);

	// 荞݃Zbg
	iosc->IntHDC(FALSE);

	// SxSI CPU]tO
	sxsicpu = FALSE;
}

//---------------------------------------------------------------------------
//
//	ZNVtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Selection(DWORD data)
{
	int c;
	int i;

	ASSERT(this);

	// ZNgꂽRg[m
	c = 8;
	for (i=0; i<8; i++) {
		if (data & 1) {
			c = i;
			break;
		}
		data >>= 1;
	}

	// rbgȂ΃oXt[tF[Y
	if (c >= 8) {
		BusFree();
		return;
	}

#if defined(SASI_LOG)
	LOG1(Log::Normal, "ZNVtF[Y Rg[%d", c);
#endif	// SASI_LOG

	// hCu݂Ȃ΃oXt[tF[Y
	if (sasi.disk[(c << 1) + 0]->IsNULL()) {
		if (sasi.disk[(c << 1) + 1]->IsNULL()) {
			BusFree();
			return;
		}
	}

	// BSYグăZNVtF[Y
	sasi.ctrl = (DWORD)c;
	sasi.phase = selection;
	sasi.bsy = TRUE;
}

//---------------------------------------------------------------------------
//
//	R}htF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Command()
{
	ASSERT(this);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "R}htF[Y");
#endif	// SASI_LOG

	// R}htF[Y
	sasi.phase = command;

	// I/O=0, C/D=1, MSG=0
	sasi.io = FALSE;
	sasi.cd = TRUE;
	sasi.msg = FALSE;

	// R}hc蒷6oCg(ꕔR}h10oCgBWriteByteōĐݒ)
	sasi.offset = 0;
	sasi.length = 6;

	// R}hf[^v
	event.SetUser(0);
	event.SetTime(32);
}

//---------------------------------------------------------------------------
//
//	stF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Execute()
{
	int drive;

	ASSERT(this);

	// stF[Y
	sasi.phase = execute;

	// hCu쐬
	ASSERT(sasi.ctrl < 8);
	drive = sasi.ctrl << 1;
	if (sasi.cmd[1] & 0x20) {
		drive++;
	}

	// Jgݒ
	sasi.current = sasi.disk[drive];
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG2(Log::Normal, "stF[Y R}h$%02X hCu%d", sasi.cmd[0], drive);
#endif	// SASI_LOG

#if 0
	// AeVȂG[o(MÕfBAւʒm)
	if (sasi.current->IsAttn()) {
#if defined(SASI_LOG)
	LOG0(Log::Normal, "fBAʒm̂߂̃AeVG[");
#endif	// SASI_LOG

		// AeVNA
		sasi.current->ClrAttn();

		// G[(NGXgZXv)
		Error();
		return;
	}
#endif

	// 0x12ȏ0x2fȉ̃R}h́ASCSIp
	if ((sasi.cmd[0] >= 0x12) && (sasi.cmd[0] <= 0x2f)) {
		ASSERT(sasi.current);
		if (sasi.current->IsSASI()) {
			// SASIȂ̂ŁAR}hG[
			sasi.current->InvalidCmd();
			Error();
			return;
		}
	}

	// R}hʏ
	switch (sasi.cmd[0]) {
		// TEST UNIT READY
		case 0x00:
			TestUnitReady();
			return;

		// REZERO UNIT
		case 0x01:
			Rezero();
			return;

		// REQUEST SENSE
		case 0x03:
			RequestSense();
			return;

		// FORMAT UNIT
		case 0x04:
			Format();
			return;

		// FORMAT UNIT
		case 0x06:
			Format();
			return;

		// REASSIGN BLOCKS
		case 0x07:
			Reassign();
			return;

		// READ(6)
		case 0x08:
			Read6();
			return;

		// WRITE(6)
		case 0x0a:
			Write6();
			return;

		// SEEK(6)
		case 0x0b:
			Seek6();
			return;

		// ASSIGN(SASÎ)
		case 0x0e:
			Assign();
			return;

		// INQUIRY(SCSÎ)
		case 0x12:
			Inquiry();
			return;

		// MODE SENSE(SCSÎ)
		case 0x1a:
			ModeSense();
			return;

		// START STOP UNIT(SCSÎ)
		case 0x1b:
			StartStop();
			return;

		// PREVENT/ALLOW MEDIUM REMOVAL(SCSÎ)
		case 0x1e:
			Removal();
			return;

		// READ CAPACITY(SCSÎ)
		case 0x25:
			ReadCapacity();
			return;

		// READ(10)(SCSÎ)
		case 0x28:
			Read10();
			return;

		// WRITE(10)(SCSÎ)
		case 0x2a:
			Write10();
			return;

		// SEEK(10)(SCSÎ)
		case 0x2b:
			Seek10();
			return;

		// WRITE and VERIFY(SCSÎ)
		case 0x2e:
			Write10();
			return;

		// VERIFY(SCSÎ)
		case 0x2f:
			Verify();
			return;

		// SPECIFY(SASÎ)
		case 0xc2:
			Specify();
			return;
	}

	// ȊO͑ΉĂȂ
	LOG1(Log::Warning, "ΉR}h $%02X", sasi.cmd[0]);
	sasi.current->InvalidCmd();

	// G[
	Error();
}

//---------------------------------------------------------------------------
//
//	Xe[^XtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Status()
{
	ASSERT(this);

#if defined(SASI_LOG)
	LOG1(Log::Normal, "Xe[^XtF[Y Xe[^X$%02X", sasi.status);
#endif	// SASI_LOG

	// Xe[^XtF[Y
	sasi.phase = status;

	// I/O=1 C/D=1 MSG=0
	sasi.io = TRUE;
	sasi.cd = TRUE;
	ASSERT(!sasi.msg);

	// Xe[^Xf[^̈v
	sasi.req = TRUE;

	// 荞݃NGXg
	iosc->IntHDC(TRUE);
}

//---------------------------------------------------------------------------
//
//	bZ[WtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Message()
{
	ASSERT(this);

#if defined(SASI_LOG)
	LOG1(Log::Normal, "bZ[WtF[Y bZ[W$%02X", sasi.message);
#endif	// SASI_LOG

	// bZ[WtF[Y
	sasi.phase = message;

	// I/O=1 C/D=1 MSG=1
	ASSERT(sasi.io);
	ASSERT(sasi.cd);
	sasi.msg = 1;

	// Xe[^Xf[^̈v
	sasi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	ʃG[
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Error()
{
	ASSERT(this);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "G[(Xe[^XtF[Y)");
#endif	// SASI_LOG

	// Xe[^XƃbZ[Wݒ(CHECK CONDITION)
	sasi.status = (sasi.current->GetLUN() << 5) | 0x02;
	sasi.message = 0x00;

	// Cxg~
	event.SetTime(0);

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	TEST UNIT READY
//
//---------------------------------------------------------------------------
void FASTCALL SASI::TestUnitReady()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "TEST UNIT READYR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->TestUnitReady(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REZERO UNIT
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Rezero()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "REZEROR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->Rezero(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REQUEST SENSE
//
//---------------------------------------------------------------------------
void FASTCALL SASI::RequestSense()
{
	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "REQUEST SENSER}h");
#endif	// SASI_LOG

	// hCuŃR}h
	sasi.length = sasi.current->RequestSense(sasi.cmd, sasi.buffer);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

#if defined(SASI_LOG)
	LOG1(Log::Normal, "ZXL[ $%02X", sasi.buffer[2]);
#endif

	// hCuԋpꂽf[^ԐMBI/O=1, C/D=0
	sasi.offset = 0;
	sasi.blocks = 1;
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// G[Xe[^XNA
	sasi.status = 0x00;
	sasi.message = 0x00;

	// CxgݒAf[^̈v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	FORMAT UNIT
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Format()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "FORMAT UNITR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->Format(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REASSIGN BLOCKS
//	SASÎ(KiSCSI)
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Reassign()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "REASSIGN BLOCKSR}h");
#endif	// SASI_LOG

	// SASIȊO̓G[Ƃ(׃Xg𑗐MȂ)
	if (!sasi.current->IsSASI()) {
		sasi.current->InvalidCmd();
		Error();
		return;
	}

	// hCuŃR}h
	status = sasi.current->Reassign(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	READ(6)
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Read6()
{
	DWORD record;

	ASSERT(this);
	ASSERT(sasi.current);

	// R[hԍƃubN擾
	record = sasi.cmd[1] & 0x1f;
	record <<= 8;
	record |= sasi.cmd[2];
	record <<= 8;
	record |= sasi.cmd[3];
	sasi.blocks = sasi.cmd[4];
	if (sasi.blocks == 0) {
		sasi.blocks = 0x100;
	}

#if defined(SASI_LOG)
	LOG2(Log::Normal, "READ(6)R}h R[h=%06X ubN=%d", record, sasi.blocks);
#endif	// SASI_LOG

	// hCuŃR}h
	sasi.length = sasi.current->Read(sasi.buffer, record);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// ̃ubNokA[Nݒ
	sasi.offset = 0;
	sasi.next = record + 1;

	// ǂݍ݃tF[Y, I/O=1, C/D=0
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// CxgݒAf[^̈v
	event.SetUser(1);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	WRITE(6)
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Write6()
{
	DWORD record;

	ASSERT(this);
	ASSERT(sasi.current);

	// R[hԍƃubN擾
	record = sasi.cmd[1] & 0x1f;
	record <<= 8;
	record |= sasi.cmd[2];
	record <<= 8;
	record |= sasi.cmd[3];
	sasi.blocks = sasi.cmd[4];
	if (sasi.blocks == 0) {
		sasi.blocks = 0x100;
	}

#if defined(SASI_LOG)
	LOG2(Log::Normal, "WRITE(6)R}h R[h=%06X ubN=%d", record, sasi.blocks);
#endif	// SASI_LOG

	// hCuŃR}h
	sasi.length = sasi.current->WriteCheck(record);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// ][N
	sasi.next = record + 1;
	sasi.offset = 0;

	// ݃tF[Y, C/D=0
	sasi.phase = write;
	sasi.cd = FALSE;

	// CxgݒAf[^݂̏v
	event.SetUser(1);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	SEEK(6)
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Seek6()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "SEEK(6)R}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->Seek(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	ASSIGN
//	SASÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Assign()
{
	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "ASSIGNR}h");
#endif	// SASI_LOG

	// SASIȊO̓G[(x_j[NR}h)
	if (!sasi.current->IsSASI()) {
		sasi.current->InvalidCmd();
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// 4oCg̃f[^NGXg
	sasi.phase = write;
	sasi.length = 4;
	sasi.offset = 0;

	// C/D0(f[^)
	sasi.cd = FALSE;

	// CxgݒAf[^݂̏v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Inquiry()
{
	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "INQUIRYR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	sasi.length = sasi.current->Inquiry(sasi.cmd, sasi.buffer);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// hCuԋpꂽf[^ԐMBI/O=1, C/D=0
	sasi.offset = 0;
	sasi.blocks = 1;
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// CxgݒAf[^̈v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	MODE SENSE
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::ModeSense()
{
	int length;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "MODE SENSER}h");
#endif	// SASI_LOG

	// hCuŃR}h
	length = sasi.current->ModeSense(sasi.cmd, sasi.buffer);

	// ʕ]
	if (length > 0) {
		// Xe[^XOK
		sasi.status = 0x00;
		sasi.message = 0x00;
		sasi.length = length;
	}
	else {
#if 1
		Error();
		return;
#else
		// Xe[^XG[
		sasi.status = (sasi.current->GetLUN() << 5) | 0x02;
		sasi.message = 0;
		sasi.length = -length;
#endif
	}

	// hCuԋpꂽf[^ԐMBI/O=1, C/D=0
	sasi.offset = 0;
	sasi.blocks = 1;
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// CxgݒAf[^̈v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	START STOP UNIT
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::StartStop()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "START STOP UNITR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->StartStop(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	PREVENT/ALLOW MEDIUM REMOVAL
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Removal()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "PREVENT/ALLOW MEDIUM REMOVALR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->Removal(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	READ CAPACITY
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::ReadCapacity()
{
	int length;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "READ CAPACITYR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	length = sasi.current->ReadCapacity(sasi.cmd, sasi.buffer);

	// ʕ]
	if (length > 0) {
		// Xe[^XOK
		sasi.status = 0x00;
		sasi.message = 0x00;
		sasi.length = length;
	}
	else {
#if 1
		// s(G[)
		Error();
		return;
#else
		// Xe[^XG[
		sasi.status = (sasi.current->GetLUN() << 5) | 0x02;
		sasi.message = 0x00;
		sasi.length = -length;
#endif
	}

	// hCuԋpꂽf[^ԐMBI/O=1, C/D=0
	sasi.offset = 0;
	sasi.blocks = 1;
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// CxgݒAf[^̈v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	READ(10)
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Read10()
{
	DWORD record;

	ASSERT(this);
	ASSERT(sasi.current);

	// R[hԍƃubN擾
	record = sasi.cmd[2];
	record <<= 8;
	record |= sasi.cmd[3];
	record <<= 8;
	record |= sasi.cmd[4];
	record <<= 8;
	record |= sasi.cmd[5];
	sasi.blocks = sasi.cmd[7];
	sasi.blocks <<= 8;
	sasi.blocks |= sasi.cmd[8];

#if defined(SASI_LOG)
	LOG2(Log::Normal, "READ(10)R}h R[h=%08X ubN=%d", record, sasi.blocks);
#endif	// SASI_LOG

	// ubN0͏Ȃ
	if (sasi.blocks == 0) {
		sasi.status = 0x00;
		sasi.message = 0x00;
		Status();
		return;
	}

	// hCuŃR}h
	sasi.length = sasi.current->Read(sasi.buffer, record);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// ̃ubNokA[Nݒ
	sasi.offset = 0;
	sasi.next = record + 1;

	// ǂݍ݃tF[Y, I/O=1, C/D=0
	sasi.phase = read;
	sasi.io = TRUE;
	sasi.cd = FALSE;

	// CxgݒAf[^̈v
	event.SetUser(1);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	WRITE(10)
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Write10()
{
	DWORD record;

	ASSERT(this);
	ASSERT(sasi.current);

	// R[hԍƃubN擾
	record = sasi.cmd[2];
	record <<= 8;
	record |= sasi.cmd[3];
	record <<= 8;
	record |= sasi.cmd[4];
	record <<= 8;
	record |= sasi.cmd[5];
	sasi.blocks = sasi.cmd[7];
	sasi.blocks <<= 8;
	sasi.blocks |= sasi.cmd[8];

#if defined(SASI_LOG)
	LOG2(Log::Normal, "WRITE(10)R}h R[h=%08X ubN=%d", record, sasi.blocks);
#endif	// SASI_LOG

	// ubN0͏Ȃ
	if (sasi.blocks == 0) {
		sasi.status = 0x00;
		sasi.message = 0x00;
		Status();
		return;
	}

	// hCuŃR}h
	sasi.length = sasi.current->WriteCheck(record);
	if (sasi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// ][N
	sasi.next = record + 1;
	sasi.offset = 0;

	// ݃tF[Y, C/D=0
	sasi.phase = write;
	sasi.cd = FALSE;

	// CxgݒAf[^݂̏v
	event.SetUser(1);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	SEEK(10)
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Seek10()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "SEEK(10)R}h");
#endif	// SASI_LOG

	// hCuŃR}h(_ubN͊֌WȂ̂SeekĂ)
	status = sasi.current->Seek(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	VERIFY
//	SCSÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Verify()
{
	BOOL status;

	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "VERIFYR}h");
#endif	// SASI_LOG

	// hCuŃR}h
	status = sasi.current->Verify(sasi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// 
	sasi.status = 0x00;
	sasi.message = 0x00;

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	SPECIFY
//	SASÎ
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Specify()
{
	ASSERT(this);
	ASSERT(sasi.current);

#if defined(SASI_LOG)
	LOG0(Log::Normal, "SPECIFYR}h");
#endif	// SASI_LOG

	// SASIȊO̓G[(x_j[NR}h)
	if (!sasi.current->IsSASI()) {
		sasi.current->InvalidCmd();
		Error();
		return;
	}

	// Xe[^XOK
	sasi.status = 0x00;
	sasi.message = 0x00;

	// 10oCg̃f[^NGXg
	sasi.phase = write;
	sasi.length = 10;
	sasi.offset = 0;

	// C/D0(f[^)
	sasi.cd = FALSE;

	// CxgݒAf[^݂̏v
	event.SetUser(0);
	event.SetTime(48);
}

//---------------------------------------------------------------------------
//
//	MO I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::Open(const Filepath& path)
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->Open(path);
	}

	// LłȂ΃G[
	if (!IsValid()) {
		return FALSE;
	}

	// CWFNg
	Eject(FALSE);

	// mbgfBłȂ΃G[
	if (IsReady()) {
		return FALSE;
	}

	// hCuɔC
	ASSERT(sasi.mo);
	if (sasi.mo->Open(path, TRUE)) {
		// ł΁At@CpXƏ݋֎~ԂL
		sasi.writep = sasi.mo->IsWriteP();
		sasi.mo->GetPath(scsimo);
		return TRUE;
	}

	// s
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MO CWFNg
//
//---------------------------------------------------------------------------
void FASTCALL SASI::Eject(BOOL force)
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		scsi->Eject(force);
		return;
	}

	// fBłȂΉȂ
	if (!IsReady()) {
		return;
	}

	// hCuɒʒm
	ASSERT(sasi.mo);
	sasi.mo->Eject(force);
}

//---------------------------------------------------------------------------
//
//	MO ݋֎~
//
//---------------------------------------------------------------------------
void FASTCALL SASI::WriteP(BOOL flag)
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		scsi->WriteP(flag);
		return;
	}

	// fBłȂΉȂ
	if (!IsReady()) {
		return;
	}

	// CgveNgݒ݂
	ASSERT(sasi.mo);
	sasi.mo->WriteP(flag);

	// ۑ
	sasi.writep = sasi.mo->IsWriteP();
}

//---------------------------------------------------------------------------
//
//	MO ݋֎~擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsWriteP() const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->IsWriteP();
	}

	// fBłȂ΃CgveNgłȂ
	if (!IsReady()) {
		return FALSE;
	}

	// hCuɕ
	ASSERT(sasi.mo);
	return sasi.mo->IsWriteP();
}

//---------------------------------------------------------------------------
//
//	MO Read Only`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsReadOnly() const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->IsReadOnly();
	}

	// fBłȂ΃[hI[łȂ
	if (!IsReady()) {
		return FALSE;
	}

	// hCuɕ
	ASSERT(sasi.mo);
	return sasi.mo->IsReadOnly();
}

//---------------------------------------------------------------------------
//
//	MO Locked`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsLocked() const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->IsLocked();
	}

	// hCu݂邩
	if (!sasi.mo) {
		return FALSE;
	}

	// hCuɕ
	return sasi.mo->IsLocked();
}

//---------------------------------------------------------------------------
//
//	MO Ready`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsReady() const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->IsReady();
	}

	// hCu݂邩
	if (!sasi.mo) {
		return FALSE;
	}

	// hCuɕ
	return sasi.mo->IsReady();
}

//---------------------------------------------------------------------------
//
//	MO L`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASI::IsValid() const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		return scsi->IsValid();
	}

	// |C^Ŕ
	if (sasi.mo) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	MO pX擾
//
//---------------------------------------------------------------------------
void FASTCALL SASI::GetPath(Filepath& path) const
{
	ASSERT(this);

	// SCSILł΁ASCSIɂ܂
	if (sasi.scsi_type != 0) {
		scsi->GetPath(path);
		return;
	}

	if (sasi.mo) {
		if (sasi.mo->IsReady()) {
			// MOfBXN
			sasi.mo->GetPath(path);
			return;
		}
	}

	// LȃpXłȂ̂ŁANA
	path.Clear();
}
