//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ SCC(Z8530) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "cpu.h"
#include "mouse.h"
#include "log.h"
#include "schedule.h"
#include "config.h"
#include "fileio.h"
#include "scc.h"

//===========================================================================
//
//	SCC
//
//===========================================================================
//#define SCC_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SCC::SCC(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('S', 'C', 'C', ' ');
	dev.desc = "SCC (Z8530)";

	// JnAhXAIAhX
	memdev.first = 0xe98000;
	memdev.last = 0xe99fff;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::Init()
{
	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// [NNA(n[hEFAZbgłꕔԂ͎c)
	memset(&scc, 0, sizeof(scc));
	scc.ch[0].index = 0;
	scc.ch[1].index = 1;
	scc.ireq = -1;
	scc.vector = -1;
	clkup = FALSE;

	// }EX擾
	mouse = (Mouse*)vm->SearchDevice(MAKEID('M', 'O', 'U', 'S'));
	ASSERT(mouse);

	// Cxgǉ
	event[0].SetDevice(this);
	event[0].SetDesc("Channel-A");
	event[0].SetUser(0);
	event[0].SetTime(0);
	scheduler->AddEvent(&event[0]);
	event[1].SetDevice(this);
	event[1].SetDesc("Channel-B");
	event[1].SetUser(1);
	event[1].SetTime(0);
	scheduler->AddEvent(&event[1]);

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL SCC::Cleanup()
{
	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::Reset()
{
	int i;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// RESETMɂ郊Zbg(`lZbgƂ͕)
	ResetSCC(&scc.ch[0]);
	ResetSCC(&scc.ch[1]);
	ResetSCC(NULL);

	// x
	for (i=0; i<2; i++ ){
		scc.ch[i].brgen = FALSE;
		scc.ch[i].speed = 0;
		event[i].SetTime(0);
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::Save(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(scc_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (!fio->Write(&scc, sizeof(scc))) {
		return FALSE;
	}

	// CxgZ[u
	if (!event[0].Save(fio, ver)) {
		return FALSE;
	}
	if (!event[1].Save(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::Load(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	LOG0(Log::Normal, "[h");

	// {̂[h
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(scc_t)) {
		return FALSE;
	}
	if (!fio->Read(&scc, sizeof(scc))) {
		return FALSE;
	}

	// Cxg
	if (!event[0].Load(fio, ver)) {
		return FALSE;
	}
	if (!event[1].Load(fio, ver)) {
		return FALSE;
	}

	// xČvZ
	ClockSCC(&scc.ch[0]);
	ClockSCC(&scc.ch[1]);

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL SCC::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);
	LOG0(Log::Normal, "ݒKp");

	// SCCNbN
	if (clkup != config->scc_clkup) {
		// Ă̂Őݒ肵
		clkup = config->scc_clkup;

		// xČvZ
		ClockSCC(&scc.ch[0]);
		ClockSCC(&scc.ch[1]);
	}
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadByte(DWORD addr)
{
	static const DWORD table[] = {
		0, 1, 2, 3, 0, 1, 2, 3, 8, 13, 10, 15, 12, 13, 10, 15
	};
	DWORD reg;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 8oCgPʂŃ[v
		addr &= 7;
		addr >>= 1;

		// EFCg
		scheduler->Wait(3);

		// WX^U蕪
		switch (addr) {
			// `lBR}h|[g
			case 0:
				// WX^
				ASSERT(scc.ch[1].reg <= 7);
				if (scc.ch[1].ph) {
					reg = table[scc.ch[1].reg + 8];
				}
				else {
					reg = table[scc.ch[1].reg];
				}

				// ZNgnC|Cg
				scc.ch[1].reg = 0;
				scc.ch[1].ph = FALSE;

				// f[^[h
				return (BYTE)ReadSCC(&scc.ch[1], reg);

			// `lBf[^|[g
			case 1:
				return (BYTE)ReadRR8(&scc.ch[1]);

			// `lAR}h|[g
			case 2:
				// WX^
				ASSERT(scc.ch[0].reg <= 7);
				if (scc.ch[0].ph) {
					reg = table[scc.ch[0].reg + 8];
				}
				else {
					reg = table[scc.ch[0].reg];
				}

				// ZNgnC|Cg
				scc.ch[0].reg = 0;
				scc.ch[0].ph = FALSE;

				// f[^[h
				return (BYTE)ReadSCC(&scc.ch[0], reg);

			// `lAf[^|[g
			case 3:
				return (BYTE)ReadRR8(&scc.ch[0]);
		}

		// ɂ͗Ȃ
		ASSERT(FALSE);
		return 0xff;
	}

	// AhX$FFԂ
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	return (WORD)(0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteByte(DWORD addr, DWORD data)
{
	DWORD reg;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 8oCgPʂŃ[v
		addr &= 7;
		addr >>= 1;

		// EFCg
		scheduler->Wait(3);

		// WX^U蕪
		switch (addr) {
			// `lBR}h|[g
			case 0:
				// WX^
				ASSERT(scc.ch[1].reg <= 7);
				reg = scc.ch[1].reg;
				if (scc.ch[1].ph) {
					reg += 8;
				}

				// ZNgnC|Cg
				scc.ch[1].reg = 0;
				scc.ch[1].ph = FALSE;

				// f[^Cg
				WriteSCC(&scc.ch[1], reg, (DWORD)data);
				return;

			// `lBf[^|[g
			case 1:
				WriteWR8(&scc.ch[1], (DWORD)data);
				return;

			// `lAR}h|[g
			case 2:
				// WX^
				ASSERT(scc.ch[0].reg <= 7);
				reg = scc.ch[0].reg;
				if (scc.ch[0].ph) {
					reg += 8;
				}

				// ZNgnC|Cg
				scc.ch[0].reg = 0;
				scc.ch[0].ph = FALSE;

				// f[^Cg
				WriteSCC(&scc.ch[0], reg, (DWORD)data);
				return;

			// `lAf[^|[g
			case 3:
				WriteWR8(&scc.ch[0], (DWORD)data);
				return;
		}

		// ɂ͗Ȃ
		ASSERT(FALSE);
		return;
	}

	// AhX̓fR[hĂȂ
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadOnly(DWORD addr) const
{
	static const DWORD table[] = {
		0, 1, 2, 3, 0, 1, 2, 3, 8, 13, 10, 15, 12, 13, 10, 15
	};
	DWORD reg;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) != 0) {
		// 8oCgPʂŃ[v
		addr &= 7;
		addr >>= 1;

		// WX^U蕪
		switch (addr) {
			// `lBR}h|[g
			case 0:
				// WX^
				ASSERT(scc.ch[1].reg <= 7);
				if (scc.ch[1].ph) {
					reg = table[scc.ch[1].reg + 8];
				}
				else {
					reg = table[scc.ch[1].reg];
				}

				// f[^[h
				return (BYTE)ROSCC(&scc.ch[1], reg);

			// `lBf[^|[g
			case 1:
				return (BYTE)ROSCC(&scc.ch[1], 8);

			// `lAR}h|[g
			case 2:
				// WX^
				ASSERT(scc.ch[0].reg <= 7);
				if (scc.ch[0].ph) {
					reg = table[scc.ch[0].reg + 8];
				}
				else {
					reg = table[scc.ch[0].reg];
				}

				// f[^[h
				return (BYTE)ROSCC(&scc.ch[0], reg);

			// `lAf[^|[g
			case 3:
				return (BYTE)ROSCC(&scc.ch[0], 8);
		}

		// ɂ͗Ȃ
		ASSERT(FALSE);
		return 0xff;
	}

	// AhX$FFԂ
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL SCC::GetSCC(scc_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// [NRs[
	*buffer = scc;
}

//---------------------------------------------------------------------------
//
//	[N擾
//
//---------------------------------------------------------------------------
const SCC::scc_t* FASTCALL SCC::GetWork() const
{
	ASSERT(this);

	// [NԂ
	return (const scc_t*)&scc;
}

//---------------------------------------------------------------------------
//
//	xN^擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::GetVector(int type) const
{
	DWORD vector;

	ASSERT(this);
	ASSERT((type >= 0) && (type < 8));

	// NV=1ȂA0Œ
	if (scc.nv) {
		return 0;
	}

	// xN^ݒ
	vector = scc.vbase;

	// bit3-1Abit6-4
	if (scc.vis) {
		if (scc.shsl) {
			// bit6-4
			vector &= 0x8f;
			vector |= (type << 4);
		}
		else {
			// bit3-1
			vector &= 0xf1;
			vector |= (type << 1);
		}
	}

	return vector;
}

//---------------------------------------------------------------------------
//
//	`lZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::ResetSCC(ch_t *p)
{
	int i;

	ASSERT(this);

	// NULL^ƃn[hEFAZbg
	if (p == NULL) {
#if defined(SCC_LOG)
		LOG0(Log::Normal, "n[hEFAZbg");
#endif	// SCC_LOG

		// WR9
		scc.shsl = FALSE;
		scc.mie = FALSE;
		scc.dlc = FALSE;

		// WR11
		return;
	}

#if defined(SCC_LOG)
	LOG1(Log::Normal, "`l%d Zbg", p->index);
#endif	// SCC_LOG

	// RR0
	p->tu = TRUE;
	p->zc = FALSE;

	// WR0
	p->reg = 0;
	p->ph = FALSE;
	p->rxno = TRUE;
	p->txpend = FALSE;

	// RR1
	p->framing = FALSE;
	p->overrun = FALSE;
	p->parerr = FALSE;

	// WR1
	p->rxim = 0;
	p->txie = FALSE;
	p->extie = FALSE;

	// RR3
	for (i=0; i<2; i++) {
		scc.ch[i].rsip = FALSE;
		scc.ch[i].rxip = FALSE;
		scc.ch[i].txip = FALSE;
		scc.ch[i].extip = FALSE;
	}

	// WR3
	p->rxen = FALSE;

	// WR4
	p->stopbit |= 0x01;

	// WR5
	p->dtr = FALSE;
	p->brk = FALSE;
	p->txen = FALSE;
	p->rts = FALSE;
	if (p->index == 1) {
		mouse->MSCtrl(!p->rts, 1);
	}

	// WR14
	p->loopback = FALSE;
	p->aecho = FALSE;
	p->dtrreq = FALSE;

	// WR15
	p->baie = TRUE;
	p->tuie = TRUE;
	p->ctsie = TRUE;
	p->syncie = TRUE;
	p->dcdie = TRUE;
	p->zcie = FALSE;

	// MFIFO
	p->rxfifo = 0;

	// Mobt@
	p->rxnum = 0;
	p->rxread = 0;
	p->rxwrite = 0;
	p->rxtotal = 0;

	// Mobt@
	p->txnum = 0;
	p->txread = 0;
	p->txwrite = 0;
	p->tdf = FALSE;
	p->txtotal = 0;
	p->txwait = FALSE;

	// 荞
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	`lǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadSCC(ch_t *p, DWORD reg)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));


#if defined(SCC_LOG)
	LOG2(Log::Normal, "`l%dǂݏo R%d", p->index, reg);
#endif	// SCC_LOG

	switch (reg) {
		// RR0 - gXe[^X
		case 0:
			return ReadRR0(p);

		// RR1 - XyVRxRfBV
		case 1:
			return ReadRR1(p);

		// RR2 - xN^
		case 2:
			return ReadRR2(p);

		// RR3 - 荞݃yfBO
		case 3:
			return ReadRR3(p);

		// RR8 - Mf[^
		case 8:
			return ReadRR8(p);

		// RR10 - SDLC[v[h
		case 10:
			return 0;

		// RR12 - {[[g
		case 12:
			return p->tc;

		// RR13 - {[[g
		case 13:
			return (p->tc >> 8);

		// RR15 - OXe[^X荞ݐ
		case 15:
			return ReadRR15(p);
	}

#if defined(SCC_LOG)
	LOG1(Log::Normal, "WX^ǂݏo R%d", reg);
#endif	// SCC_LOG

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	RR0ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR0(const ch_t *p) const
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// 
	data = 0;

	// bit7 : Break/AbortXe[^X
	if (p->ba) {
		data |= 0x80;
	}

	// bit6 : MA_[Xe[^X
	if (p->tu) {
		data |= 0x40;
	}

	// bit5 : CTSXe[^X
	if (p->cts) {
		data |= 0x20;
	}

	// bit5 : Sync/HuntXe[^X
	if (p->sync) {
		data |= 0x10;
	}

	// bit3 : DCDXe[^X
	if (p->dcd) {
		data |= 0x08;
	}

	// bit2 : Mobt@GveBXe[^X
	if (!p->txwait) {
		// MEFCg͏Ƀobt@t
		if (!p->tdf) {
			data |= 0x04;
		}
	}

	// bit1 : [JEgXe[^X
	if (p->zc) {
		data |= 0x02;
	}

	// bit0 : Mobt@LXe[^X
	if (p->rxfifo > 0) {
		data |= 0x01;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	RR1ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR1(const ch_t *p) const
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// 
	data = 0x06;

	// bit6 : t[~OG[
	if (p->framing) {
		data |= 0x40;
	}

	// bit5 : I[o[G[
	if (p->overrun) {
		data |= 0x20;
	}

	// bit4 : peBG[
	if (p->parerr) {
		data |= 0x10;
	}

	// bit0 : M
	if (p->txsent) {
		data |= 0x01;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	RR2ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR2(ch_t *p)
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// `l
	if (p->index == 0) {
		// `lAWR2̓ê܂ܕԂ
		data = scc.vbase;
	}
	else {
		// `lB͌ݗvĂ銄荞݃xN^
		data = scc.request;
	}

	// 荞ݗvłȂΉȂ
	if (scc.ireq < 0) {
		return data;
	}

	// ŁÅ荞݂Ɉڂ(MacG~[^)
	ASSERT((scc.ireq >= 0) && (scc.ireq <= 7));
	if (scc.ireq >= 4) {
		p = &scc.ch[1];
	}
	else {
		p = &scc.ch[0];
	}
	switch (scc.ireq & 3) {
		// Rs
		case 0:
			p->rsip = FALSE;
			break;
		// Rx
		case 1:
			p->rxip = FALSE;
			break;
		// Ext
		case 2:
			p->extip = FALSE;
			break;
		// Tx
		case 3:
			p->txip = FALSE;
			break;
	}
	scc.ireq = -1;

	// 荞݃`FbN
	IntCheck();

	// f[^Ԃ
	return data;
}

//---------------------------------------------------------------------------
//
//	RR3ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR3(const ch_t *p) const
{
	int i;
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// `lÂݗLB`lB0Ԃ
	if (p->index == 1) {
		return 0;
	}

	// NA
	data = 0;

	// bit0ext,tx,rxŁA`lB`lȀ
	for (i=0; i<2; i++) {
		data <<= 3;

		// L(v)荞݃rbgׂėĂ
		if (scc.ch[i].extip) {
			data |= 0x01;
		}
		if (scc.ch[i].txip) {
			data |= 0x02;
		}
		if (scc.ch[i].rxip) {
			data |= 0x04;
		}
		if (scc.ch[i].rsip) {
			data |= 0x04;
		}
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	RR8ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR8(ch_t *p)
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// Mf[^邩
	if (p->rxfifo == 0) {
		if (p->index == 0) {
			LOG1(Log::Warning, "`l%d Mf[^ǂ", p->index);
		}
		return 0;
	}

	// KŌɃf[^
#if defined(SCC_LOG)
	LOG2(Log::Normal, "`l%d f[^$%02X", p->index, p->rxdata[2]);
#endif
	data = p->rxdata[2];

	// MFIFOړ
	p->rxdata[2] = p->rxdata[1];
	p->rxdata[1] = p->rxdata[0];

	// 
	p->rxfifo--;

	// U荞݂𗎂ƂBFIFO̎c肪Ύ̃Cxg
	IntSCC(p, rxi, FALSE);

#if defined(SCC_LOG)
	LOG2(Log::Normal, "cFIFO=%d cRxNum=%d", p->rxfifo, p->rxnum);
#endif

	return data;
}

//---------------------------------------------------------------------------
//
//	RR15ǂݏo
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ReadRR15(const ch_t *p) const
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// f[^
	data = 0;

	// bit7 - Break/Abort荞
	if (p->baie) {
		data |= 0x80;
	}

	// bit6 - TxA_[荞
	if (p->tuie) {
		data |= 0x40;
	}

	// bit5 - CTS荞
	if (p->ctsie) {
		data |= 0x20;
	}

	// bit4 - SYNC荞
	if (p->syncie) {
		data |= 0x10;
	}

	// bit3 - DCD荞
	if (p->dcdie) {
		data |= 0x08;
	}

	// bit1 - [JEg荞
	if (p->zcie) {
		data |= 0x02;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	ǂݏô
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::ROSCC(const ch_t *p, DWORD reg) const
{
	DWORD data;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	switch (reg) {
		// RR0 - gXe[^X
		case 0:
			return ReadRR0(p);

		// RR1 - XyVRxRfBV
		case 1:
			return ReadRR1(p);

		// RR2 - xN^
		case 2:
			if (p->index == 0) {
				return scc.vbase;
			}
			return scc.request;

		// RR3 - 荞݃yfBO
		case 3:
			return ReadRR3(p);

		// RR8 - Mf[^
		case 8:
			break;

		// RR10 - SDLC[v[h
		case 10:
			return 0;

		// RR12 - {[[g
		case 12:
			return p->tc;

		// RR13 - {[[g
		case 13:
			return (p->tc >> 8);

		// RR15 - OXe[^X荞ݐ
		case 15:
			return ReadRR15(p);
	}

	// f[^
	data = 0xff;

	// MFIFOLȂAԂ
	if (p->rxfifo > 0) {
		data = p->rxdata[2];
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	`l
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteSCC(ch_t *p, DWORD reg, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

#if defined(SCC_LOG)
	LOG3(Log::Normal, "`l%d R%d <- $%02X", p->index, reg, data);
#endif	// SCC_LOG

	switch (reg) {
		// WR0 - S̃Rg[
		case 0:
			WriteWR0(p, data);
			return;

		// WR1 - 荞݋
		case 1:
			WriteWR1(p, data);
			return;

		// WR2 - xN^x[X
		case 2:
			LOG1(Log::Detail, "荞݃xN^x[X $%02X", data);
			scc.vbase = data;
			return;

		// WR3 - MRg[
		case 3:
			WriteWR3(p, data);
			return;

		// WR4 - ʐM[hݒ
		case 4:
			WriteWR4(p, data);
			return;

		// WR5 - MRg[
		case 5:
			WriteWR5(p, data);
			return;

		// WR8 - Mobt@
		case 8:
			WriteWR8(p, data);
			return;

		// WR9 - 荞݃Rg[
		case 9:
			WriteWR9(data);
			return;

		// WR10 - SDLCRg[
		case 10:
			WriteWR10(p, data);
			return;

		// WR11 - NbNE[qI
		case 11:
			WriteWR11(p, data);

		// WR12 - {[[g
		case 12:
			WriteWR12(p, data);
			return;

		// WR13 - {[[g
		case 13:
			WriteWR13(p, data);
			return;

		// WR14 - NbNRg[
		case 14:
			WriteWR14(p, data);
			return;

		// WR15 - gXe[^X荞݃Rg[
		case 15:
			WriteWR15(p, data);
			return;
	}

	LOG3(Log::Warning, "`l%dWX^ R%d <- $%02X",
								p->index, reg, data);
}

//---------------------------------------------------------------------------
//
//	WR0
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR0(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit2-0 : ANZXWX^
	p->reg = (data & 0x07);

	// bit5-3 : SCCR}h
	data >>= 3;
	data &= 0x07;
	switch (data) {
		// 000:kR[h
		case 0:
			break;

		// 001:ʃWX^I
		case 1:
			p->ph = TRUE;
			break;

		// 010:OXe[^XZbg
		case 2:
#if defined(SCC_LOG)
			LOG1(Log::Normal, "`l%d OXe[^XZbg", p->index);
#endif	// SCC_LOG
			break;

		// 100:Mf[^Zbg
		case 4:
#if defined(SCC_LOG)
			LOG1(Log::Normal, "`l%d Mf[^Zbg", p->index);
#endif	// SCC_LOG
			p->rxno = TRUE;
			break;

		// 101:M荞݃yfBOZbg
		case 5:
#if defined(SCC_LOG)
			LOG1(Log::Normal, "`l%d M荞݃yfBOZbg", p->index);
#endif	// SCC_LOG
			p->txpend = TRUE;
			IntSCC(p, txi, FALSE);

		// 110:MG[Zbg
		case 6:
#if defined(SCC_LOG)
			LOG1(Log::Normal, "`l%d MG[Zbg", p->index);
#endif	// SCC_LOG
			p->framing = FALSE;
			p->overrun = FALSE;
			p->parerr = FALSE;
			IntSCC(p, rsi, FALSE);
			break;

		// 111:ōʃCT[rXZbg
		case 7:
			break;

		// ̑
		default:
			LOG2(Log::Warning, "`l%d SCCT|[gR}h $%02X",
								p->index, data);
			break;
	}
}

//---------------------------------------------------------------------------
//
//	WR1
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR1(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit4-3 : Rx荞݃[h
	p->rxim = (data >> 3) & 0x03;

	// bit2 : peBG[XyVRxRfBV荞݂
	if (data & 0x04) {
		p->parsp = TRUE;
	}
	else {
		p->parsp = FALSE;
	}

	// bit1 : M荞݋
	if (data & 0x02) {
		p->txie = TRUE;
	}
	else {
		p->txie = FALSE;
	}

	// bit0 : OXe[^X荞݋
	if (data & 0x01) {
		p->extie = TRUE;
	}
	else {
		p->extie = FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	WR3
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR3(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit7-6 : MLN^
	p->rxbit = ((data & 0xc0) >> 6) + 5;
	ASSERT((p->rxbit >= 5) && (p->rxbit <= 8));

	// bit5 : I[gCl[u
	if (data & 0x20) {
		p->aen = TRUE;
	}
	else {
		p->aen = FALSE;
	}

	// bit0 : MCl[u
	if (data & 0x01) {
		if (!p->rxen) {
			// G[ԂNA
			p->framing = FALSE;
			p->overrun = FALSE;
			p->parerr = FALSE;
			IntSCC(p, rsi, FALSE);

			// Mobt@NA
			p->rxnum = 0;
			p->rxread = 0;
			p->rxwrite = 0;
			p->rxfifo = 0;

			// MCl[u
			p->rxen = TRUE;
		}
	}
	else {
		p->rxen = FALSE;
	}

	// {[[gČvZ
	ClockSCC(p);
}

//---------------------------------------------------------------------------
//
//	WR4
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR4(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit7-6 : NbN[h
	switch ((data & 0xc0) >> 6) {
		// x1NbN[h
		case 0:
			p->clkm = 1;
			break;
		// x16NbN[h
		case 1:
			p->clkm = 16;
			break;
		// x32NbN[h
		case 2:
			p->clkm = 32;
			break;
		// x64NbN[h
		case 3:
			p->clkm = 64;
			break;
		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// bit3-2 : Xgbvrbg
	p->stopbit = (data & 0x0c) >> 2;
	if (p->stopbit == 0) {
		LOG1(Log::Warning, "`l%d [h", p->index);
	}

	// bit1-0 : peB
	if (data & 1) {
		// 1:peBA2:peB
		p->parity = ((data & 2) >> 1) + 1;
	}
	else {
		// 0:peBȂ
		p->parity = 0;
	}

	// {[[gČvZ
	ClockSCC(p);
}

//---------------------------------------------------------------------------
//
//	WR5
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR5(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit7 : DTR(LowANeBu,p->dtrreq邱)
	if (data & 0x80) {
		p->dtr = TRUE;
	}
	else {
		p->dtr = FALSE;
	}

	// bit6-5 : MLN^rbg
	p->txbit = ((data & 0x60) >> 5) + 5;
	ASSERT((p->txbit >= 5) && (p->txbit <= 8));

	// bit4 : u[NMo
	if (data & 0x10) {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d u[NMON", p->index);
#endif	// SCC_LOG
		p->brk = TRUE;
	}
	else {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d u[NMOFF", p->index);
#endif	// SCC_LOG
		p->brk = FALSE;
	}

	// bit3 : MCl[u
	if (data & 0x08) {
		if (!p->txen) {
			// Mobt@NA
			p->txnum = 0;
			if (!p->txpend) {
				IntSCC(p, txi, TRUE);
			}

			// M[NNA
			p->txsent = FALSE;
			p->tdf = FALSE;
			p->txwait = FALSE;

			// MCl[u
			p->txen = TRUE;
		}
	}
	else {
		p->txen = FALSE;
	}

	// bit1 : RTS
	if (data & 0x02) {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d RTSMON", p->index);
#endif	// SCC_LOG
		p->rts = TRUE;
	}
	else {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d RTSMOFF", p->index);
#endif	// SCC_LOG
		p->rts = FALSE;
	}

	// `lB̏ꍇ́A}EXMSCTRL`
	if (p->index == 1) {
		// RTSLowANeBuA̐ɃCo[^Ă(H}Q)
		mouse->MSCtrl(!(p->rts), 1);
	}
}

//---------------------------------------------------------------------------
//
//	WR8
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR8(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// Mf[^LAMf[^
	p->tdr = data;
	p->tdf = TRUE;

	// M荞݂艺
	IntSCC(p, txi, FALSE);

	// M荞݃yfBOIt
	p->txpend = FALSE;
}

//---------------------------------------------------------------------------
//
//	WR9
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR9(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// bit7-6 : Zbg
	switch ((data & 0xc0) >> 3) {
		case 1:
			ResetSCC(&scc.ch[0]);
			return;
		case 2:
			ResetSCC(&scc.ch[1]);
			return;
		case 3:
			ResetSCC(&scc.ch[0]);
			ResetSCC(&scc.ch[1]);
			ResetSCC(NULL);
			return;
		default:
			break;
	}

	// bit4 : 荞ݕω[hI
	if (data & 0x10) {
#if defined(SCC_LOG)
		LOG0(Log::Warning, "荞݃xN^ω[h bit4-bit6");
#endif	// SCC_LOG
		scc.shsl = TRUE;
	}
	else {
#if defined(SCC_LOG)
		LOG0(Log::Normal, "荞݃xN^ω[h bit3-bit1");
#endif	// SCC_LOG
		scc.shsl = FALSE;
	}

	// bit3 : 荞݃Cl[u
	if (data & 0x08) {
#if defined(SCC_LOG)
		LOG0(Log::Normal, "荞݃Cl[u");
#endif	// SCC_LOG
		scc.mie = TRUE;
	}
	else {
#if defined(SCC_LOG)
		LOG0(Log::Warning, "荞݃fBZ[u");
#endif	// SCC_LOG
		scc.mie = FALSE;
	}

	// bit2 : fBW[`FC֎~
	if (data & 0x04) {
		scc.dlc = TRUE;
	}
	else {
		scc.dlc = FALSE;
	}

	// bit1 : 荞݃xN^Ȃ
	if (data & 0x02) {
		scc.nv = TRUE;
#if defined(SCC_LOG)
		LOG0(Log::Warning, "荞݃ANmbWȂ");
#endif	// SCC_LOG
	}
	else {
#if defined(SCC_LOG)
		LOG0(Log::Normal, "荞݃ANmbW");
#endif	// SCC_LOG
		scc.nv = FALSE;
	}

	// bit0 : 荞݃xN^ϓ
	if (data & 0x01) {
		scc.vis = TRUE;
#if defined(SCC_LOG)
		LOG0(Log::Normal, "荞݃xN^ω");
#endif	// SCC_LOG
	}
	else {
		scc.vis = FALSE;
#if defined(SCC_LOG)
		LOG0(Log::Warning, "荞݃xN^Œ");
#endif	// SCC_LOG
	}

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	WR10
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR10(ch_t* /*p*/, DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	data &= 0x60;
	if (data != 0) {
		LOG0(Log::Warning, "NRZȊO̕ϒ[h");
	}
}

//---------------------------------------------------------------------------
//
//	WR11
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR11(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit7 : RTxCEȂ
	if (data & 0x80) {
		LOG1(Log::Warning, "`l%d bRTxCbSYNCŃNbN쐬", p->index);
	}

	// bit6-5 : MNbNI
	if ((data & 0x60) != 0x40) {
		LOG1(Log::Warning, "`l%d {[[gWFl[^o͂gȂ", p->index);
	}
}

//---------------------------------------------------------------------------
//
//	WR12
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR12(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// 8bit̂
	p->tc &= 0xff00;
	p->tc |= data;

	// {[[gČvZ
	ClockSCC(p);
}

//---------------------------------------------------------------------------
//
//	WR13
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR13(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// 8bit̂
	p->tc &= 0x00ff;
	p->tc |= (data << 8);

	// {[[gČvZ
	ClockSCC(p);
}

//---------------------------------------------------------------------------
//
//	WR14
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR14(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit4 : [J[vobN
	if (data & 0x10) {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d [J[vobN[h", p->index);
#endif	// SCC_LOG
		p->loopback = TRUE;
	}
	else {
		p->loopback = FALSE;
	}

	// bit3 : I[gGR[
	if (data & 0x08) {
#if defined(SCC_LOG)
		LOG1(Log::Normal, "`l%d I[gGR[[h", p->index);
#endif	// SCC_LOG
		p->aecho = TRUE;
	}
	else {
		p->aecho = FALSE;
	}

	// bit2 : DTR/REQZNg
	if (data & 0x04) {
		p->dtrreq = TRUE;
	}
	else {
		p->dtrreq = FALSE;
	}

	// bit1 : {[[gWFl[^NbN
	if (data & 0x02) {
		p->brgsrc = TRUE;
	}
	else {
		p->brgsrc = FALSE;
	}

	// bit0 : {[[gWFl[^Cl[u
	if (data & 0x01) {
		p->brgen = TRUE;
	}
	else {
		p->brgen = FALSE;
	}

	// {[[gČvZ
	ClockSCC(p);
}

//---------------------------------------------------------------------------
//
//	{[[gČvZ
//
//---------------------------------------------------------------------------
void FASTCALL SCC::ClockSCC(ch_t *p)
{
	DWORD len;
	DWORD speed;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// {[[gWFl[^LłȂ΁ACxg~
	if (!p->brgsrc || !p->brgen) {
		event[p->index].SetTime(0);
		p->speed = 0;
		return;
	}

	// NbN
	if (clkup) {
		p->baudrate = 3750000;
	}
	else {
		p->baudrate = 2500000;
	}

	// {[[gݒlƃNbN{A{[[gZo
	speed = (p->tc + 2);
	speed *= p->clkm;
	p->baudrate /= speed;

	// f[^rbgApeBAXgbvrbg2oCgOXZo
	len = p->rxbit + 1;
	if (p->parity != 0) {
		len++;
	}
	len <<= 1;
	switch (p->stopbit) {
		// XgbvrbgȂ
		case 0:
			break;
		// Xgbvrbg1bit
		case 1:
			len += 2;
			break;
		// Xgbvrbg1.5bit
		case 2:
			len += 3;
		// Xgbvrbg2bit
		case 3:
			len += 4;
	}

	// CPSɐݒ肵Ă
	if (clkup) {
		p->cps = 3750000;
	}
	else {
		p->cps = 2500000;
	}
	p->cps /= ((len * speed) >> 1);

	// ŏIxvZ
	p->speed = 100;
	p->speed *= (len * speed);
	if (clkup) {
		p->speed /= 375;
	}
	else {
		p->speed /= 250;
	}

	// ݒ
	if (event[p->index].GetTime() == 0) {
		event[p->index].SetTime(p->speed);
	}
}

//---------------------------------------------------------------------------
//
//	WR15
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WriteWR15(ch_t *p, DWORD data)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(data < 0x100);

	// bit7 - Break/Abort荞
	if (data & 0x80) {
		p->baie = TRUE;
	}
	else {
		p->baie = FALSE;
	}

	// bit6 - TxA_[荞
	if (data & 0x40) {
		p->tuie = TRUE;
	}
	else {
		p->tuie = FALSE;
	}

	// bit5 - CTS荞
	if (data & 0x20) {
		p->ctsie = TRUE;
	}
	else {
		p->ctsie = FALSE;
	}

	// bit4 - SYNC荞
	if (data & 0x10) {
		p->syncie = TRUE;
	}
	else {
		p->syncie = FALSE;
	}

	// bit3 - DCD荞
	if (data & 0x08) {
		p->dcdie = TRUE;
	}
	else {
		p->dcdie = FALSE;
	}

	// bit1 - [JEg荞
	if (data & 0x02) {
		p->zcie = TRUE;
	}
	else {
		p->zcie = FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	荞݃NGXg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::IntSCC(ch_t *p, itype_t type, BOOL flag)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));

	// 荞ݗLȂAyfBOrbgグ
	switch (type) {
		// M荞
		case rxi:
			p->rxip = FALSE;
			if (flag) {
				if ((p->rxim == 1) || (p->rxim == 2)) {
					p->rxip = TRUE;
				}
			}
			break;

		// XyVRxRfBV荞
		case rsi:
			p->rsip = FALSE;
			if (flag) {
				if (p->rxim >= 2) {
					p->rsip = TRUE;
				}
			}
			break;

		// M荞
		case txi:
			p->txip = FALSE;
			if (flag) {
				if (p->txie) {
					p->txip = TRUE;
				}
			}
			break;

		// OXe[^Xω荞
		case exti:
			p->extip = FALSE;
			if (flag) {
				if (p->extie) {
					p->extip = TRUE;
				}
			}
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	荞݃`FbN
//
//---------------------------------------------------------------------------
void FASTCALL SCC::IntCheck()
{
	int i;
	DWORD v;

	ASSERT(this);

	// v荞݂NA
	scc.ireq = -1;

	// }X^荞݋
	if (scc.mie) {

		// `lA`lB̏
		for (i=0; i<2; i++) {
			// Rs,Rx,Tx,Ext̏ŃyfBOrbg
			if (scc.ch[i].rsip) {
				scc.ireq = (i << 2) + 0;
				break;
			}
			if (scc.ch[i].rxip) {
				scc.ireq = (i << 2) + 1;
				break;
			}
			if (scc.ch[i].txip) {
				scc.ireq = (i << 2) + 3;
				break;
			}
			if (scc.ch[i].extip) {
				scc.ireq = (i << 2) + 2;
				break;
			}
		}

		// ^CvāA荞݃xN^쐬
		if (scc.ireq >= 0) {
			// xN^擾
			v = scc.vbase;

			// bit3-1Abit6-4
			if (scc.shsl) {
				// bit6-4
				v &= 0x8f;
				v |= ((7 - scc.ireq) << 4);
			}
			else {
				// bit3-1
				v &= 0xf1;
				v |= ((7 - scc.ireq) << 1);
			}

			// 荞݃xN^L
			scc.request = v;

			// RR2ǂ߂xN^WR9VISɂ炸ɕω
			// f[^V[g̋LqƂ͈قȂ(MacG~[^)
			if (!scc.vis) {
				v = scc.vbase;
			}

			// NV(No Vector)オĂȂΊ荞ݗv
			if (!scc.nv) {
				// ɗvĂOK
				if (scc.vector == (int)v) {
					return;
				}

				// ʂ̃xN^vĂ΁AxLZ
				if (scc.vector >= 0) {
					cpu->IntCancel(5);
				}

				// 荞ݗv
#if defined(SCC_LOG)
				LOG2(Log::Normal, "荞ݗv NGXg$%02X xN^$%02X ", scc.request, v);
#endif	// SCC_LOG
				cpu->Interrupt(5, (BYTE)v);
				scc.vector = (int)v;
			}
			return;
		}
	}

	// v荞݂Ȃ̃xN^𐶐
	scc.request = scc.vbase;

	// bit3-1Abit6-4
	if (scc.shsl) {
		// bit6-4
		scc.request &= 0x8f;
		scc.request |= 0x60;
	}
	else {
		// bit3-1
		scc.request &= 0xf1;
		scc.request |= 0x06;
	}

	// ɗvĂΊ荞݃LZ
	if (scc.vector >= 0) {
		cpu->IntCancel(5);
		scc.vector = -1;
	}
}

//---------------------------------------------------------------------------
//
//	荞݉
//
//---------------------------------------------------------------------------
void FASTCALL SCC::IntAck()
{
	ASSERT(this);

	// ZbgɁACPU犄荞݂Ԉēꍇ
	if (scc.vector < 0) {
		LOG0(Log::Warning, "vĂȂ荞");
		return;
	}

#if defined(SCC_LOG)
	LOG1(Log::Normal, "荞݉ xN^$%02X", scc.vector);
#endif

	// vxN^Ȃ
	scc.vector = -1;

	// ňU荞݂Brxi荞݌p̏ꍇ͎
	// Cxg(mX^WA1907AhL)
}

//---------------------------------------------------------------------------
//
//	f[^M
//
//---------------------------------------------------------------------------
void FASTCALL SCC::Send(int channel, DWORD data)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// BYTEɐ(}EXsigned int̕ϊł邽)
	data &= 0xff;

#if defined(SCC_LOG)
	LOG2(Log::Normal, "`l%d f[^M$%02X", channel, data);
#endif	// SCC_LOG

	// |C^ݒ
	p = &scc.ch[channel];

	// Mobt@֑}
	p->rxbuf[p->rxwrite] = (BYTE)data;
	p->rxwrite = (p->rxwrite + 1) & (sizeof(p->rxbuf) - 1);
	p->rxnum++;
	if (p->rxnum >= sizeof(p->rxbuf)) {
		LOG1(Log::Warning, "`l%d Mobt@I[o[t[", p->index);
		p->rxnum = sizeof(p->rxbuf);
		p->rxread = p->rxwrite;
	}
}

//----------------------------------------------------------------------------
//
//	peBG[
//
//---------------------------------------------------------------------------
void FASTCALL SCC::ParityErr(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	if (!p->parerr) {
		LOG1(Log::Normal, "`l%d peBG[", p->index);
	}

	// tOグ
	p->parerr = TRUE;

	// 荞
	if (p->parsp) {
		if (p->rxim >= 2) {
			IntSCC(p, rsi, TRUE);
		}
	}
}

//---------------------------------------------------------------------------
//
//	t[~OG[
//
//---------------------------------------------------------------------------
void FASTCALL SCC::FramingErr(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	if (!p->framing) {
		LOG1(Log::Normal, "`l%d t[~OG[", p->index);
	}

	// tOグ
	p->framing = TRUE;

	// 荞
	if (p->rxim >= 2) {
		IntSCC(p, rsi, TRUE);
	}
}

//---------------------------------------------------------------------------
//
//	MCl[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::IsRxEnable(int channel) const
{
	const ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	return p->rxen;
}

//---------------------------------------------------------------------------
//
//	Mobt@󂢂Ă邩
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::IsRxBufEmpty(int channel) const
{
	const ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	if (p->rxnum == 0) {
		return TRUE;
	}
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	{[[g`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::IsBaudRate(int channel, DWORD baudrate) const
{
	const ch_t *p;
	DWORD offset;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));
	ASSERT(baudrate >= 75);

	// |C^ݒ
	p = &scc.ch[channel];

	// ^ꂽ{[[g5%vZ
	offset = baudrate * 5;
	offset /= 100;

	// ͈͓ȂOK
	if (p->baudrate < (baudrate - offset)) {
		return FALSE;
	}
	if ((baudrate + offset) < p->baudrate) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	Mf[^rbg擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::GetRxBit(int channel) const
{
	const ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	return p->rxbit;
}

//---------------------------------------------------------------------------
//
//	Xgbvrbg擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::GetStopBit(int channel) const
{
	const ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	return p->stopbit;
}

//---------------------------------------------------------------------------
//
//	peB擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::GetParity(int channel) const
{
	const ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	return p->parity;
}

//---------------------------------------------------------------------------
//
//	f[^M
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCC::Receive(int channel)
{
	ch_t *p;
	DWORD data;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// f[^
	data = 0;

	// f[^
	if (p->txnum > 0) {
		// Mobt@o
		data = p->txbuf[p->txread];
		p->txread = (p->txread + 1) & (sizeof(p->txbuf) - 1);
		p->txnum--;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	Mobt@GveB`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::IsTxEmpty(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// Mobt@󂢂ĂTRUEԂ
	if (p->txnum == 0) {
		return TRUE;
	}
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	Mobt@t`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::IsTxFull(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// Mobt@3/4ȏ㖄܂Ă΁ATRUE
	if (p->txnum >= (sizeof(p->txbuf) * 3 / 4)) {
		return TRUE;
	}
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MubN
//
//---------------------------------------------------------------------------
void FASTCALL SCC::WaitTx(int channel, BOOL wait)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// ȂAtO삪Kv
	if (!wait) {
		if (p->txwait) {
			p->txsent = TRUE;
		}
	}

	// ݒ
	p->txwait = wait;
}

//---------------------------------------------------------------------------
//
//	u[NZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::SetBreak(int channel, BOOL flag)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// 荞݃`FbN
	if (p->baie) {
		// 荞ݐR[h
	}

	// Zbg
	p->ba = flag;
}

//---------------------------------------------------------------------------
//
//	CTSZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::SetCTS(int channel, BOOL flag)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// 荞݃`FbN
	if (p->ctsie) {
		// 荞ݐR[h
	}

	// Zbg
	p->cts = flag;
}

//---------------------------------------------------------------------------
//
//	DCDZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCC::SetDCD(int channel, BOOL flag)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// 荞݃`FbN
	if (p->dcdie) {
		// 荞ݐR[h
	}

	// Zbg
	p->dcd = flag;
}

//---------------------------------------------------------------------------
//
//	u[N擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::GetBreak(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// ̂܂
	return p->brk;
}

//---------------------------------------------------------------------------
//
//	RTS擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::GetRTS(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// ̂܂
	return p->rts;
}

//---------------------------------------------------------------------------
//
//	DTR擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::GetDTR(int channel)
{
	ch_t *p;

	ASSERT(this);
	ASSERT((channel == 0) || (channel == 1));

	// |C^ݒ
	p = &scc.ch[channel];

	// DTR/REQFALSEȂA\tgEFAw
	if (!p->dtrreq) {
		return p->dtr;
	}

	// Mobt@GveBǂ
	if (p->txwait) {
		return FALSE;
	}
	return !(p->tdf);
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCC::Callback(Event *ev)
{
	ch_t *p;
	int channel;

	ASSERT(this);
	ASSERT(ev);

	// `lo
	channel = ev->GetUser();
	ASSERT((channel == 0) || (channel == 1));
	p = &scc.ch[channel];

	// M
	if (p->rxen) {
		EventRx(p);
	}

	// M
	if (p->txen) {
		EventTx(p);
	}

	// xύX
	if (ev->GetTime() != p->speed) {
		ev->SetTime(p->speed);
	}

	// C^[op
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	Cxg(M)
//
//---------------------------------------------------------------------------
void FASTCALL SCC::EventRx(ch_t *p)
{
	BOOL flag;

	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(p->rxen);

	// MtO
	flag = TRUE;

	// DCD̃`FbN
	if (p->dcd && p->aen) {
		// [vobNĂΎMłȂ
		if (!p->loopback) {
			// MȂ
			flag = FALSE;
		}
	}

	// Mobt@Lł΁AFIFOɑ}
	if ((p->rxnum > 0) && flag) {
		if (p->rxfifo >= 3) {
			ASSERT(p->rxfifo == 3);

			// FIFO͈tBI[o[
#if defined(SCC_LOG)
			LOG1(Log::Normal, "`l%d I[o[G[", p->index);
#endif	// SCC_LOG
			p->overrun = TRUE;
			p->rxfifo--;

			if (p->rxim >= 2) {
				IntSCC(p, rsi, TRUE);
			}
		}

		ASSERT(p->rxfifo <= 2);

		// MFIFOɑ}
		p->rxdata[2 - p->rxfifo] = (DWORD)p->rxbuf[p->rxread];
		p->rxread = (p->rxread + 1) & (sizeof(p->rxbuf) - 1);
		p->rxnum--;
		p->rxfifo++;

		// Mg[^
		p->rxtotal++;
	}

	// MFIFOLł΁A荞
	if (p->rxfifo > 0) {
		// 荞ݔ
		switch (p->rxim) {
			// ̂݊荞
			case 1:
				if (!p->rxno) {
					break;
				}
				p->rxno = FALSE;
				// ̂܂ܑ

			// Ɋ荞
			case 2:
				IntSCC(p, rxi, TRUE);
				break;
		}
	}
}

//---------------------------------------------------------------------------
//
//	Cxg(M)
//
//---------------------------------------------------------------------------
void FASTCALL SCC::EventTx(ch_t *p)
{
	ASSERT(this);
	ASSERT(p);
	ASSERT((p->index == 0) || (p->index == 1));
	ASSERT(p->txen);

	// EFCg[hłȂ΁AȎM
	if (!p->txwait) {
		p->txsent = TRUE;
	}

	// CTS̃`FbN
	if (p->cts && p->aen) {
		// eXg[hCTSɊւ炸Mł
		if (!p->aecho && !p->loopback) {
			return;
		}
	}

	// f[^邩
	if (p->tdf) {
		// I[gGR[[hłȂ
		if (!p->aecho) {
			// Mobt@֑}
			p->txbuf[p->txwrite] = (BYTE)p->tdr;
			p->txwrite = (p->txwrite + 1) & (sizeof(p->txbuf) - 1);
			p->txnum++;

			// ӂꂽAÂf[^̂Ă
			if (p->txnum >= sizeof(p->txbuf)) {
				LOG1(Log::Warning, "`l%d Mobt@I[o[t[", p->index);
				p->txnum = sizeof(p->txbuf);
				p->txread = p->txwrite;
			}

			// Mg[^
			p->txtotal++;
		}

		// Ms
		p->txsent = FALSE;

		// Mobt@GveB
		p->tdf = FALSE;

		// [J[vobN[hł΁Af[^M֓
		if (p->loopback && !p->aecho) {
			// Mobt@֑}
			p->rxbuf[p->rxwrite] = (BYTE)p->tdr;
			p->rxwrite = (p->rxwrite + 1) & (sizeof(p->rxbuf) - 1);
			p->rxnum++;

			// ӂꂽAÂf[^珇̂Ă
			if (p->rxnum >= sizeof(p->rxbuf)) {
				LOG1(Log::Warning, "`l%d Mobt@I[o[t[", p->index);
				p->rxnum = sizeof(p->rxbuf);
				p->rxread = p->rxwrite;
			}
		}
	}

	// MyfBOȂΊ荞
	if (!p->txpend) {
		IntSCC(p, txi, TRUE);
	}
}
