//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ XPW[ ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "schedule.h"
#include "vm.h"
#include "log.h"
#include "cpu.h"
#include "event.h"
#include "dmac.h"
#include "core_asm.h"
#include "config.h"
#include "fileio.h"

//===========================================================================
//
//	XPW[
//
//===========================================================================
//#define SCHEDULER_LOG

//---------------------------------------------------------------------------
//
//	CxgEXVAZu
//
//---------------------------------------------------------------------------
#if defined(_MSC_VER) && defined(_M_IX86)
#define SCHEDULER_ASM
#endif	// _MSC_VER

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
Scheduler::Scheduler(VM *p) : Device(p)
{
	int i;

	// foCXID
	dev.id = MAKEID('S', 'C', 'H', 'E');
	dev.desc = "Scheduler";

	// u[N|Cg
	for (i=0; i<BreakMax; i++) {
		breakp[i].use = FALSE;
		breakp[i].addr = 0;
		breakp[i].enable = FALSE;
		breakp[i].time = 0;
		breakp[i].count = 0;
	}

	// 
	sch.total = 0;
	sch.one = 0;
	sch.sound = 0;

	// CPU
	sch.clock = 0;
	sch.speed = 979;
	sch.cycle = 0;
	sch.time = 0;

	// u[N|Cg
	sch.brk = FALSE;
	sch.check = FALSE;

	// Cxg
	sch.first = NULL;
	sch.exec = FALSE;

	// foCX
	cpu = NULL;
	dmac = NULL;

	// ̑
	dma_active = FALSE;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL Scheduler::Init()
{
	ASSERT(this);

	// {NX
	if (!Device::Init()) {
		return FALSE;
	}

	// CPU擾
	ASSERT(!cpu);
	cpu = (CPU*)vm->SearchDevice(MAKEID('C', 'P', 'U', ' '));
	ASSERT(cpu);

	// DMAC擾
	ASSERT(!dmac);
	dmac = (DMAC*)vm->SearchDevice(MAKEID('D', 'M', 'A', 'C'));
	ASSERT(dmac);

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::Cleanup()
{
	ASSERT(this);
	ASSERT_DIAG();

	// {NX
	Device::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::Reset()
{
	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// ԃZbg(sound)
	sch.total = 0;
	sch.one = 0;

	// CPUTCNZbg
	sch.cycle = 0;
	sch.time = 0;

	// CxgsłȂ
	sch.exec = FALSE;

	// DMAsȂ
	dma_active = FALSE;

	// CPUxݒ͖s(INFO.RAM΍􃋁[`̂)
	ASSERT((sch.clock >= 0) && (sch.clock <= 5));
	SetCPUSpeed(ClockTable[sch.clock]);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL Scheduler::Save(Fileio *fio, int /*ver*/)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Z[u");

	// u[N|CgTCYZ[u
	sz = sizeof(breakp);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// u[N|CĝZ[u
	if (!fio->Write(breakp, (int)sz)) {
		return FALSE;
	}

	// XPW[TCYZ[u
	sz = sizeof(scheduler_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// XPW[̂Z[u
	if (!fio->Write(&sch, (int)sz)) {
		return FALSE;
	}

	// TCNe[uZ[u
	if (!fio->Write(CycleTable, sizeof(CycleTable))) {
		return FALSE;
	}

	// dma_activeZ[u(version 2.01)
	if (!fio->Write(&dma_active, sizeof(dma_active))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL Scheduler::Load(Fileio *fio, int ver)
{
	size_t sz;
	Event *first;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x200);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// Cxg|C^ێ
	first = sch.first;

	// u[N|CgTCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(breakp)) {
		return FALSE;
	}

	// u[N|Cĝ[h
	if (!fio->Read(breakp, (int)sz)) {
		return FALSE;
	}

	// XPW[TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(scheduler_t)) {
		return FALSE;
	}

	// XPW[̂[h
	if (!fio->Read(&sch, (int)sz)) {
		return FALSE;
	}

	// TCNe[u[h
	if (!fio->Read(CycleTable, sizeof(CycleTable))) {
		return FALSE;
	}

	// Cxg|C^𕜋A
	sch.first = first;

	// o[W2.01ȏȂAdma_active[h
	if (ver >= 0x0201) {
		if (!fio->Read(&dma_active, sizeof(dma_active))) {
			return FALSE;
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");

	// VXeNbNݒr
	if (sch.clock != config->system_clock) {
		// ݒ肪قȂĂ̂ŁATCNe[uč\z
		sch.clock = config->system_clock;
		ASSERT((sch.clock >= 0) && (sch.clock <= 5));
		SetCPUSpeed(ClockTable[sch.clock]);
	}
}

#if defined(_DEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::AssertDiag() const
{
	ASSERT(this);
	ASSERT(GetID() == MAKEID('S', 'C', 'H', 'E'));
	ASSERT(cpu);
	ASSERT(cpu->GetID() == MAKEID('C', 'P', 'U', ' '));
	ASSERT(dmac);
	ASSERT(dmac->GetID() == MAKEID('D', 'M', 'A', 'C'));
}
#endif	// _DEBUG

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::GetScheduler(scheduler_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT_DIAG();

	// f[^Rs[
	*buffer = sch;
}

//---------------------------------------------------------------------------
//
//	s
//
//---------------------------------------------------------------------------
DWORD FASTCALL Scheduler::Exec(DWORD hus)
{
	int cycle;
	DWORD result;
	DWORD dcycle;

	ASSERT(this);
	ASSERT(hus > 0);
	ASSERT_DIAG();

	// u[N|Cg̏ꍇ
	if (!sch.check) {
		// ŒZ̃CxgT
#if defined(SCHEDULER_ASM)
		sch.one = GetMinEvent(hus);
#else
		sch.one = GetMinRemain(hus);
#endif	// SCHEDULER_ASM

		// sch.one + sch.timeɌTCNɎsĂ邩
		ASSERT((sch.one + sch.time) < 0x1000);
		cycle = CycleTable[sch.one + sch.time];
		if (cycle > sch.cycle) {

			// słTCNTāAs
			cycle -= sch.cycle;
			if (!dma_active) {
				// ʏ
				result = cpu->Exec(cycle);
			}
			else {
				// DMACI[gNGXgL
				dcycle = dmac->AutoDMA(cycle);
				if (dcycle != 0) {
					// ƌ덷oH
					result = cpu->Exec(dcycle);
				}
				else {
					// ׂDMAŏ
					result = cycle;
				}
			}

			// I
			if (result < 0x80000000) {
				// sch.time, sch.cycleXV
				sch.cycle += result;
				sch.time += sch.one;

				// Ԃi߂
				ExecEvent(sch.one);

				if (sch.time < 200) {
					return sch.one;
				}

				// Sync
				while (sch.time >= 200) {
					if ((DWORD)sch.cycle < sch.speed) {
						break;
					}
					sch.time -= 200;
					sch.cycle -= sch.speed;
				}

				// u[N`FbN
				if (!sch.brk) {
					return sch.one;
				}

#if defined(SCHEDULER_LOG)
				LOG0(Log::Normal, "u[N");
#endif	// SCHEDULER_LOG
				sch.brk = FALSE;
				return (DWORD)(sch.one | 0x80000000);
			}
			else {
				// sG[
				result &= 0x7fffffff;

				if ((int)result > cycle) {
					// sch.timeAsch.cycleXV
					sch.time += sch.one;
					sch.cycle += result;

					// Cxgs
					ExecEvent(sch.one);

					while (sch.time >= 200) {
						if ((DWORD)sch.cycle < sch.speed) {
							break;
						}
						sch.time -= 200;
						sch.cycle -= sch.speed;
					}
					// sG[ACxg
					return 0x80000000;
				}
				// SsOcpuG[N
				sch.cycle += result;
				// sG[ACxg
				return 0x80000000;
			}
		}
		else {

			// ͎słȂBԂi߂̂
			sch.time += sch.one;
			ExecEvent(sch.one);

			if (sch.time < 200) {
				return sch.one;
			}

			// sch.timeXV
			while (sch.time >= 200) {
				if ((DWORD)sch.cycle < sch.speed) {
					break;
				}
				sch.time -= 200;
				sch.cycle -= sch.speed;
			}

			// s߂ȂACxg
			return sch.one;
		}

	}

	// [v
	for (;;) {
		result = Trace(hus);

		switch (result) {
			// s߂ȂACxg
			case 0:
				return sch.one;

			// sACxg
			case 1:
				if (sch.brk) {
#if defined(SCHEDULER_LOG)
					LOG0(Log::Normal, "u[N");
#endif	// SCHEDULER_LOG
					sch.brk = FALSE;
					return 0x80000000;
				}
				if (IsBreak(cpu->GetPC()) != -1) {
					OnBreak(cpu->GetPC());
					return 0x80000000;
				}
				return sch.one;

			// sACxg
			case 2:
				if (sch.brk) {
#if defined(SCHEDULER_LOG)
					LOG0(Log::Normal, "u[N");
#endif	// SCHEDULER_LOG
					sch.brk = FALSE;
					return 0x80000000;
				}
				if (IsBreak(cpu->GetPC()) != -1) {
					OnBreak(cpu->GetPC());
					return 0x80000000;
				}
				break;

			// sG[
			case 3:
				if (sch.brk) {
#if defined(SCHEDULER_LOG)
					LOG0(Log::Normal, "u[N");
#endif	// SCHEDULER_LOG
					sch.brk = FALSE;
				}
				return 0x80000000;

			// ȊO
			default:
				ASSERT(FALSE);
				return sch.one;
		}
	}
}

//---------------------------------------------------------------------------
//
//	g[X
//
//---------------------------------------------------------------------------
DWORD FASTCALL Scheduler::Trace(DWORD hus)
{
	int cycle;
	DWORD result;

	ASSERT(this);
	ASSERT(hus > 0);
	ASSERT_DIAG();

	// ŒZ̃CxgT
#if defined(SCHEDULER_ASM)
	sch.one = GetMinEvent(hus);
#else
	sch.one = GetMinRemain(hus);
#endif	// SCHEDULER_ASM

	// sch.one + sch.timeɌTCNɎsĂ邩
	ASSERT((sch.one + sch.time) < 0x1000);
	cycle = CycleTable[sch.one + sch.time];
	if (cycle <= sch.cycle) {
		// ͎słȂBԂi߂
		sch.time += sch.one;
		ExecEvent(sch.one);

		// sch.timeXV
		while (sch.time >= 200) {
			sch.time -= 200;
			sch.cycle -= sch.speed;
		}
		// s߂ȂACxg
		return 0;
	}

	// słTCNT
	cycle -= sch.cycle;

	// 1TCN^ĎsĂ݂
	if (!dma_active) {
		// ʏ
		result = cpu->Exec(1);
	}
	else {
		// DMACI[gNGXgL
		result = dmac->AutoDMA(1);
		if (result != 0) {
			result = cpu->Exec(result);
		}
		else {
			result = 1;
		}
	}
	if (result >= 0x80000000) {
		// sG[
		return 3;
	}

	// result >= cycleȂACxgsł
	if ((int)result >= cycle) {
		// sch.time, sch.cycleXV
		sch.cycle += result;
		sch.time += sch.one;

		// Ԃi߂
		ExecEvent(sch.one);

		while (sch.time >= 200) {
			sch.time -= 200;
			sch.cycle -= sch.speed;
		}
		// sACxg
		return 1;
	}

	// ܂ĂȂ̂ŁACxg܂ł͊Ԃ
	// sch.cycleXV
	sch.cycle += result;

	// sACxg
	return 2;
}

//---------------------------------------------------------------------------
//
//	CPUxݒ
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::SetCPUSpeed(DWORD speed)
{
	int i;
	DWORD cycle;

	ASSERT(this);
	ASSERT(speed > 0);
	ASSERT_DIAG();

	LOG2(Log::Detail, "CPUxݒ %d.%02dMHz", speed / 100, (speed % 100));

	// CPUxL
	sch.speed = speed;

	// 0`2048us܂ŁA0.5usPʂł̑ΉTCNvZ
	for (i=0; i<0x1000; i++) {
		cycle = (DWORD)i;
		cycle *= speed;
		cycle /= 200;
		CycleTable[i] = cycle;
	}
}

//---------------------------------------------------------------------------
//
//	oߎԂ擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL Scheduler::GetPassedTime() const
{
	DWORD hus;

	ASSERT(this);
	ASSERT_DIAG();

	// CxgsȂ0
	if (sch.exec) {
		return 0;
	}

	// sTCNAcpu_cylcle玞ԂZo
	hus = cpu->GetCycle() + sch.cycle;
	hus *= 200;
	hus /= sch.speed;
	hus -= sch.time;

	// one傫΁A
	if (sch.one < hus) {
		hus = sch.one;
	}

	// husPʂŕԂ
	return hus;
}

//---------------------------------------------------------------------------
//
//	u[N|Cgݒ
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::SetBreak(DWORD addr, BOOL enable)
{
	int i;
	BOOL flag;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT_DIAG();

#if defined(SCHEDULER_LOG)
	LOG2(Log::Normal, "u[N|Cgݒ $%06X enable=%d", addr, enable);
#endif	// SCHEDULER_LOG

	flag = FALSE;

	// v`FbN
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].addr == addr) {
				// tOύX̂
				breakp[i].enable = enable;
				flag = TRUE;
				break;
			}
		}
	}

	if (!flag) {
		// 󂫃T[`
		for (i=0; i<BreakMax; i++) {
			if (!breakp[i].use) {
				// Zbg
				breakp[i].use = TRUE;
				breakp[i].addr = addr;
				breakp[i].enable = enable;
				breakp[i].time = 0;
				breakp[i].count = 0;
				break;
			}
		}
	}

	// LtOݒ
	flag = FALSE;
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].enable) {
				// Lȃu[N|Cg
				flag = TRUE;
				break;
			}
		}
	}
	sch.check = flag;
}

//---------------------------------------------------------------------------
//
//	u[N|Cg폜
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::DelBreak(DWORD addr)
{
	int i;
	BOOL flag;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT_DIAG();

#if defined(SCHEDULER_LOG)
	LOG1(Log::Normal, "u[N|Cg폜 $%06X", addr);
#endif	// SCHEDULER_LOG

	// v`FbN
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].addr == addr) {
				// 폜
				breakp[i].use = FALSE;
				break;
			}
		}
	}

	// LtOݒ
	flag = FALSE;
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].enable) {
				// Lȃu[N|Cg
				flag = TRUE;
				break;
			}
		}
	}
	sch.check = flag;
}

//---------------------------------------------------------------------------
//
//	u[N|Cg擾
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::GetBreak(int index, breakpoint_t *buf) const
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < BreakMax));
	ASSERT(buf);
	ASSERT_DIAG();

	// Rs[
	*buf = breakp[index];
}

//---------------------------------------------------------------------------
//
//	u[N|CgLE
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::EnableBreak(int index, BOOL enable)
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < BreakMax));
	ASSERT(breakp[index].use);
	ASSERT_DIAG();

	breakp[index].enable = enable;
}

//---------------------------------------------------------------------------
//
//	u[N񐔃NA
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::ClearBreak(int index)
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < BreakMax));
	ASSERT(breakp[index].use);
	ASSERT_DIAG();

	breakp[index].count = 0;
	breakp[index].time = 0;
}

//---------------------------------------------------------------------------
//
//	u[NAhXύX
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::AddrBreak(int index, DWORD addr)
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < BreakMax));
	ASSERT(addr <= 0xffffff);
	ASSERT(breakp[index].use);
	ASSERT_DIAG();

	breakp[index].addr = addr;
}

//---------------------------------------------------------------------------
//
//	u[NAhX`FbN
//
//---------------------------------------------------------------------------
int FASTCALL Scheduler::IsBreak(DWORD addr, BOOL any) const
{
	int i;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT_DIAG();

	// ŏɃtO
	if (!sch.check) {
		return -1;
	}

	// v`FbN
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].addr == addr) {
				// LECɂȂAL
				if (any || breakp[i].enable) {
					return i;
				}
			}
		}
	}

	// u[N|Cg͂邪Av
	return -1;
}

//---------------------------------------------------------------------------
//
//	u[NAhXKp
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::OnBreak(DWORD addr)
{
	int i;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT(sch.check);
	ASSERT_DIAG();

	// v`FbN
	for (i=0; i<BreakMax; i++) {
		if (breakp[i].use) {
			if (breakp[i].addr == addr) {
				break;
			}
		}
	}
	ASSERT(i < BreakMax);

	// ԃZbgAJEgAbv
	breakp[i].time = GetTotalTime();
	breakp[i].count++;
}

//---------------------------------------------------------------------------
//
//	Cxgǉ
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::AddEvent(Event *event)
{
	Event *p;

	ASSERT(this);
	ASSERT(event);
	ASSERT_DIAG();

#if defined(SCHEDULER_LOG)
	LOG4(Log::Normal, "Cxgǉ Device=%c%c%c%c",
					(char)(event->GetDevice()->GetID() >> 24),
					(char)(event->GetDevice()->GetID() >> 16),
					(char)(event->GetDevice()->GetID() >> 8),
					(char)(event->GetDevice()->GetID()));
	LOG1(Log::Normal, "Cxgǉ %s", event->GetDesc());
#endif	// SCHEDULER_LOG

	// ŏ̃Cxg
	if (!sch.first) {
		// ŏ̃Cxg
		sch.first = event;
		event->SetNextEvent(NULL);

#if defined(SCHEDULER_ASM)
		// ʒm
		NotifyEvent(sch.first);
#endif	// SCHEDULER_ASM
		return;
	}

	// Ō̃CxgT
	p = sch.first;
	while (p->GetNextEvent()) {
		p = p->GetNextEvent();
	}

	// pŌ̃CxgȂ̂ŁAɒǉ
	p->SetNextEvent(event);
	event->SetNextEvent(NULL);

#if defined(SCHEDULER_ASM)
	// ʒm
	NotifyEvent(sch.first);
#endif	// SCHEDULER_ASM
}

//---------------------------------------------------------------------------
//
//	Cxg폜
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::DelEvent(Event *event)
{
	Event *p;
	Event *prev;

	ASSERT(this);
	ASSERT(event);
	ASSERT_DIAG();

#if defined(SCHEDULER_LOG)
	LOG4(Log::Normal, "Cxg폜 Device=%c%c%c%c",
					(char)(event->GetDevice()->GetID() >> 24),
					(char)(event->GetDevice()->GetID() >> 16),
					(char)(event->GetDevice()->GetID() >> 8),
					(char)(event->GetDevice()->GetID()));
	LOG1(Log::Normal, "Cxg폜 %s", event->GetDesc());
#endif	// SCHEDULER_LOG

	// ŏ̃Cxg
	if (sch.first == event) {
		// ŏ̃CxgBnextŏ̃CxgɊ蓖Ă
		sch.first = event->GetNextEvent();
		event->SetNextEvent(NULL);

#if defined(SCHEDULER_ASM)
		// ʒm
		NotifyEvent(sch.first);
#endif	// SCHEDULER_ASM
		return;
	}

	// ̃Cxgv܂Ō
	p = sch.first;
	prev = p;
	while (p) {
		// v`FbN
		if (p == event) {
			prev->SetNextEvent(event->GetNextEvent());
			event->SetNextEvent(NULL);

#if defined(SCHEDULER_ASM)
			// ʒm
			NotifyEvent(sch.first);
#endif	// SCHEDULER_ASM
			return;
		}

		// 
		prev = p;
		p = p->GetNextEvent();
	}

	// ׂẴCxgvȂ(蓾Ȃ)
	ASSERT(FALSE);
}

//---------------------------------------------------------------------------
//
//	CxgL`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL Scheduler::HasEvent(Event *event) const
{
	Event *p;

	ASSERT(this);
	ASSERT(event);
	ASSERT_DIAG();

	// 
	p = sch.first;

	// SẴCxg܂
	while (p) {
		// v`FbN
		if (p == event) {
			return TRUE;
		}

		// 
		p = p->GetNextEvent();
	}

	// ̃Cxg̓`FCɊ܂܂ĂȂ
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	Cxǧ擾
//
//---------------------------------------------------------------------------
int FASTCALL Scheduler::GetEventNum() const
{
	int num;
	Event *p;

	ASSERT(this);
	ASSERT_DIAG();

	// 
	num = 0;
	p = sch.first;

	// SẴCxg܂
	while (p) {
		num++;

		// 
		p = p->GetNextEvent();
	}

	// CxǧԂ
	return num;
}

//---------------------------------------------------------------------------
//
//	ŒZ̃CxgT
//	ʓrAZułp
//
//---------------------------------------------------------------------------
DWORD FASTCALL Scheduler::GetMinRemain(DWORD hus)
{
	Event *p;
	DWORD minimum;
	DWORD remain;

	ASSERT(this);
	ASSERT(hus > 0);
	ASSERT_DIAG();

	// Cxg|C^
	p = sch.first;

	// 
	minimum = hus;

	// [v
	while (p) {
		// c莞Ԏ擾
		remain = p->GetRemain();

		// L
		if (remain == 0) {
			// 
			p = p->GetNextEvent();
			continue;
		}

		// ŏ`FbN
		if (remain >= minimum) {
			p = p->GetNextEvent();
			continue;
		}

		// ŏ
		minimum = remain;
		p = p->GetNextEvent();
	}

	return minimum;
}

//---------------------------------------------------------------------------
//
//	Cxgs
//	ʓrAZułp
//
//---------------------------------------------------------------------------
void FASTCALL Scheduler::ExecEvent(DWORD hus)
{
#if !defined(SCHEDULER_ASM)
	Event *p;
#endif	// !SCHEDULER_ASM

	ASSERT(this);
	ASSERT(hus >= 0);
	ASSERT_DIAG();

	// CxgsJn
	sch.exec = TRUE;

	// g[^ԑATEhԑ
	sch.total += hus;
	sch.sound += hus;

#if defined(SCHEDULER_ASM)
	SubExecEvent(hus);
	sch.exec = FALSE;
#else

	// Cxg|C^
	p = sch.first;

	// CxgāAs
	while (p) {
		p->Exec(hus);
		p = p->GetNextEvent();
	}

	// CxgsI
	sch.exec = FALSE;
#endif
}

//---------------------------------------------------------------------------
//
//	NbNe[u
//
//---------------------------------------------------------------------------
const DWORD Scheduler::ClockTable[] = {
	979,			// 10MHz
	1171,			// 12MHz
	1460,			// 15MHz
	1556,			// 16MHz
	1689,			// 17.4MHz
	1941			// 20MHz
};

//---------------------------------------------------------------------------
//
//	TCNe[u
//
//---------------------------------------------------------------------------
int Scheduler::CycleTable[0x1000];
