//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ SCSI(MB89352) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "memory.h"
#include "sram.h"
#include "config.h"
#include "disk.h"
#include "fileio.h"
#include "filepath.h"
#include "scsi.h"

//===========================================================================
//
//	SCSI
//
//===========================================================================
//#define SCSI_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SCSI::SCSI(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('S', 'C', 'S', 'I');
	dev.desc = "SCSI (MB89352)";

	// JnAhXAIAhX
	memdev.first = 0xea0000;
	memdev.last = 0xea1fff;

	// [N
	memset(&scsi, 0, sizeof(scsi));

	// foCX
	memory = NULL;
	sram = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::Init()
{
	int i;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// 擾
	ASSERT(!memory);
	memory = (Memory*)vm->SearchDevice(MAKEID('M', 'E', 'M', ' '));
	ASSERT(memory);

	// SRAM擾
	ASSERT(!sram);
	sram = (SRAM*)vm->SearchDevice(MAKEID('S', 'R', 'A', 'M'));
	ASSERT(sram);

	// [N
	scsi.bdid = 0x80;
	scsi.ilevel = 2;
	scsi.vector = -1;
	scsi.id = -1;
	for (i=0; i<sizeof(scsi.cmd); i++) {
		scsi.cmd[i] = 0;
	}

	// RtBO
	scsi.memsw = TRUE;
	scsi.mo_first = FALSE;

	// Cxg
	event.SetDevice(this);
	event.SetDesc("Selection");
	event.SetUser(0);
	event.SetTime(0);
	cdda.SetDevice(this);
	cdda.SetDesc("CD-DA 75fps");
	cdda.SetUser(1);
	cdda.SetTime(0);

	// fBXN
	for (i=0; i<DeviceMax; i++) {
		scsi.disk[i] = NULL;
	}
	for (i=0; i<HDMax; i++) {
		scsi.hd[i] = NULL;
	}
	scsi.mo = NULL;
	scsi.cdrom = NULL;

	// ̑
	cache_wb = TRUE;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Cleanup()
{
	int i;

	ASSERT(this);

	// HD폜
	for (i=0; i<HDMax; i++) {
		if (scsi.hd[i]) {
			delete scsi.hd[i];
			scsi.hd[i] = NULL;
		}
	}

	// MO폜
	if (scsi.mo) {
		delete scsi.mo;
		scsi.mo = NULL;
	}

	// CD폜
	if (scsi.cdrom) {
		delete scsi.cdrom;
		scsi.cdrom = NULL;
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Reset()
{
	Memory::memtype type;
	int i;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// SCSI^Cv(ɖ₢킹)
	type = memory->GetMemType();
	switch (type) {
		// SASÎ
		case Memory::None:
		case Memory::SASI:
			scsi.type = 0;
			break;

		// Ot
		case Memory::SCSIExt:
			scsi.type = 1;
			break;

		// ̑()
		default:
			scsi.type = 2;
			break;
	}

	// Cxg𓮓IɒǉE폜
	if (scsi.type == 0) {
		// Cxg폜
		if (scheduler->HasEvent(&event)) {
			scheduler->DelEvent(&event);
			scheduler->DelEvent(&cdda);
		}
	}
	else {
		// Cxgǉ
		if (!scheduler->HasEvent(&event)) {
			scheduler->AddEvent(&event);
			scheduler->AddEvent(&cdda);
		}
	}

	// XCb`
	if (scsi.memsw) {
		// SCSI݂ꍇ̂
		if (scsi.type != 0) {
			// $ED006F:SCSItO('V'SCSIL)
			if (sram->GetMemSw(0x6f) != 'V') {
				sram->SetMemSw(0x6f, 'V');

				// {ID7ɏ
				sram->SetMemSw(0x70, 0x07);
			}

			// $ED0070:{[h+{ID
			if (scsi.type == 1) {
				// Ot
				sram->SetMemSw(0x70, sram->GetMemSw(0x70) | 0x08);
			}
			else {
				// 
				sram->SetMemSw(0x70, sram->GetMemSw(0x70) & ~0x08);
			}

			// $ED0071:SASIG~[VtO
			sram->SetMemSw(0x71, 0x00);
		}
	}

	// fBXNč\z
	Construct();

	// fBXNZbg
	for (i=0; i<HDMax; i++) {
		if (scsi.hd[i]) {
			scsi.hd[i]->Reset();
		}
	}
	if (scsi.mo) {
		scsi.mo->Reset();
	}
	if (scsi.cdrom) {
		scsi.cdrom->Reset();
	}

	// R}hNA
	for (i=0; i<sizeof(scsi.cmd); i++) {
		scsi.cmd[i] = 0;
	}

	// WX^Zbg
	ResetReg();

	// oXt[tF[Y
	BusFree();
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::Save(Fileio *fio, int ver)
{
	size_t sz;
	int i;
	Disk *disk;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// fBXNtbV
	for (i=0; i<HDMax; i++) {
		if (scsi.hd[i]) {
			if (!scsi.hd[i]->Flush()) {
				return FALSE;
			}
		}
	}
	if (scsi.mo) {
		if (!scsi.mo->Flush()) {
			return FALSE;
		}
	}
	if (scsi.cdrom) {
		if (!scsi.cdrom->Flush()) {
			return FALSE;
		}
	}

	// TCYZ[u
	sz = sizeof(scsi_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&scsi, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!event.Save(fio, ver)) {
		return FALSE;
	}
	if (!cdda.Save(fio, ver)) {
		return FALSE;
	}

	// pXZ[u
	for (i=0; i<HDMax; i++) {
		if (!scsihd[i].Save(fio, ver)) {
			return FALSE;
		}
	}

	// fBXNZ[u
	for (i=0; i<HDMax; i++) {
		if (scsi.hd[i]) {
			if (!scsi.hd[i]->Save(fio, ver)) {
				return FALSE;
			}
		}
		else {
			// 킹̂߁A_~[ł悢ۑ(version2.04)
			disk = new SCSIHD(this);
			if (!disk->Save(fio, ver)) {
				delete disk;
				return FALSE;
			}
			delete disk;
		}
	}
	if (scsi.mo) {
		if (!scsi.mo->Save(fio, ver)) {
			return FALSE;
		}
	}
	if (scsi.cdrom) {
		if (!scsi.cdrom->Save(fio, ver)) {
			return FALSE;
		}
	}

	// version2.06ȍ~
	if (!fio->Write(&cache_wb, sizeof(cache_wb))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::Load(Fileio *fio, int ver)
{
	size_t sz;
	int i;
	Disk *disk;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);

	LOG0(Log::Normal, "[h");

	// version2.01ȑÓASCSIT|[gĂȂ
	if (ver <= 0x201) {
		// SCSIȂAhCu0
		scsi.type = 0;
		scsi.scsi_drives = 0;

		// Cxg폜
		if (scheduler->HasEvent(&event)) {
			scheduler->DelEvent(&event);
			scheduler->DelEvent(&cdda);
		}
		return TRUE;
	}

	// fBXN폜AU
	for (i=0; i<HDMax; i++) {
		if (scsi.hd[i]) {
			delete scsi.hd[i];
			scsi.hd[i] = NULL;
		}
	}
	if (scsi.mo) {
		delete scsi.mo;
		scsi.mo = NULL;
	}
	if (scsi.cdrom) {
		delete scsi.cdrom;
		scsi.cdrom = NULL;
	}
	scsi.type = 0;
	scsi.scsi_drives = 0;

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(scsi_t)) {
		return FALSE;
	}

	// ̂[h
	if (!fio->Read(&scsi, (int)sz)) {
		return FALSE;
	}

	// fBXNNA([hɏ㏑邽)
	for (i=0; i<HDMax; i++) {
		scsi.hd[i] = NULL;
	}
	scsi.mo = NULL;
	scsi.cdrom = NULL;

	// Cxg[h
	if (!event.Load(fio, ver)) {
		return FALSE;
	}
	if (ver >= 0x0203) {
		// version2.03ŁACD-DACxgǉꂽ
		if (!cdda.Load(fio, ver)) {
			return FALSE;
		}
	}

	// pX[h
	for (i=0; i<HDMax; i++) {
		if (!scsihd[i].Load(fio, ver)) {
			return FALSE;
		}
	}

	// fBXNč\z
	Construct();

	// fBXN[h(version2.03Œǉ)
	if (ver >= 0x0203) {
		for (i=0; i<HDMax; i++) {
			if (scsi.hd[i]) {
				if (!scsi.hd[i]->Load(fio, ver)) {
					return FALSE;
				}
			}
			else {
				// 킹̂߁A_~[ł悢̂Ń[h(version2.04)
				if (ver >= 0x0204) {
					disk = new SCSIHD(this);
					if (!disk->Load(fio, ver)) {
						delete disk;
						return FALSE;
					}
					delete disk;
				}
			}
		}
		if (scsi.mo) {
			if (!scsi.mo->Load(fio, ver)) {
				return FALSE;
			}
		}
		if (scsi.cdrom) {
			if (!scsi.cdrom->Load(fio, ver)) {
				return FALSE;
			}
		}
	}

	// version2.06g
	if (ver >= 0x0206) {
		if (!fio->Read(&cache_wb, sizeof(cache_wb))) {
			return FALSE;
		}
	}

	// Cxg𓮓IɒǉE폜
	if (scsi.type == 0) {
		// Cxg폜
		if (scheduler->HasEvent(&event)) {
			scheduler->DelEvent(&event);
			scheduler->DelEvent(&cdda);
		}
	}
	else {
		// Cxgǉ
		if (!scheduler->HasEvent(&event)) {
			scheduler->AddEvent(&event);
			scheduler->AddEvent(&cdda);
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ApplyCfg(const Config *config)
{
	int i;

	ASSERT(this);
	ASSERT(config);
	LOG0(Log::Normal, "ݒKp");

	// SCSIhCu
	scsi.scsi_drives = config->scsi_drives;

	// XCb`XV
	scsi.memsw = config->scsi_sramsync;

	// MODtO
	scsi.mo_first = config->scsi_mofirst;

	// SCSIt@C
	for (i=0; i<HDMax; i++) {
		scsihd[i].SetPath(config->scsi_file[i]);
	}

	// fBXNč\z
	Construct();

	// LbV[h
	cache_wb = config->cache_disk;
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::AssertDiag() const
{
	// {NX
	MemDevice::AssertDiag();

	ASSERT(this);
	ASSERT(GetID() == MAKEID('S', 'C', 'S', 'I'));
	ASSERT(memory);
	ASSERT(memory->GetID() == MAKEID('M', 'E', 'M', ' '));
	ASSERT(sram);
	ASSERT(sram->GetID() == MAKEID('S', 'R', 'A', 'M'));
	ASSERT((scsi.type == 0) || (scsi.type == 1) || (scsi.type == 2));
	ASSERT((scsi.vector == -1) || (scsi.vector == 0x6c) || (scsi.vector == 0xf6));
	ASSERT((scsi.id >= -1) && (scsi.id <= 7));
	ASSERT(scsi.bdid != 0);
	ASSERT(scsi.bdid < 0x100);
	ASSERT(scsi.ints < 0x100);
	ASSERT(scsi.mbc < 0x10);
}
#endif	// NDEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::ReadByte(DWORD addr)
{
	const BYTE *rom;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);

	// ʏ̓ǂݍ݂̏ꍇ
	if (addr >= memdev.first) {
		// SASÎ or ̏ꍇA̋Ԃ͌Ȃ
		if (scsi.type != 1) {
			// oXG[
			cpu->BusErr(addr, TRUE);
			return 0xff;
		}
	}

	// AhX}XN
	addr &= 0x1fff;

	// ROMf[^
	if (addr >= 0x20) {
		// EFCg
		scheduler->Wait(1);

		// ROMԂ
		rom = memory->GetSCSI();
		return rom[addr];
	}

	// WX^
	if ((addr & 1) == 0) {
		// AhX̓fR[hĂȂ
		return 0xff;
	}
	addr &= 0x1f;
	addr >>= 1;

	// EFCg
	scheduler->Wait(1);

	// WX^
	switch (addr) {
		// BDID
		case 0:
			return scsi.bdid;

		// SCTL
		case 1:
			return scsi.sctl;

		// SCMD
		case 2:
			return scsi.scmd;

		// U[u
		case 3:
			return 0xff;

		// INTS
		case 4:
			return scsi.ints;

		// PSNS
		case 5:
			return GetPSNS();

		// SSTS
		case 6:
			return GetSSTS();

		// SERR
		case 7:
			return GetSERR();

		// PCTL
		case 8:
			return scsi.pctl;

		// MBC
		case 9:
			return scsi.mbc;

		// DREG
		case 10:
			return GetDREG();

		// TEMP
		case 11:
			return scsi.temp;

		// TCH
		case 12:
			return (BYTE)(scsi.tc >> 16);

		// TCM
		case 13:
			return (BYTE)(scsi.tc >> 8);

		// TCL
		case 14:
			return (BYTE)scsi.tc;

		// U[u
		case 15:
			return 0xff;
	}

	// ʏAɂ͗Ȃ
	LOG1(Log::Warning, "`WX^ǂݍ R%d", addr);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr & 1) == 0);

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::WriteByte(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT(data < 0x100);

	// ʏ̓ǂݍ݂̏ꍇ
	if (addr >= memdev.first) {
		// SASÎ or ̏ꍇA̋Ԃ͌Ȃ
		if (scsi.type != 1) {
			// oXG[
			cpu->BusErr(addr, FALSE);
			return;
		}
	}

	// AhX}XN
	addr &= 0x1fff;

	// ROMf[^
	if (addr >= 0x20) {
		// ROMւ̏
		return;
	}

	// WX^
	if ((addr & 1) == 0) {
		// AhX̓fR[hĂȂ
		return;
	}
	addr &= 0x1f;
	addr >>= 1;

	// EFCg
	scheduler->Wait(1);

	// WX^
	switch (addr) {
		// BDID
		case 0:
			SetBDID(data);
			return;

		// SCTL
		case 1:
			SetSCTL(data);
			return;

		// SCMD
		case 2:
			SetSCMD(data);
			return;

		// U[u
		case 3:
			break;

		// INTS
		case 4:
			SetINTS(data);
			return;

		// SDGC
		case 5:
			SetSDGC(data);
			return;

		// SSTS(Read Only)
		case 6:
			// SCSIhCov1.04̏Cpb`ŁAԉ҂̂߂ɏłlq
			return;

		// SERR(Read Only)
		case 7:
			break;

		// PCTL
		case 8:
			SetPCTL(data);
			return;

		// MBC(Read Only)
		case 9:
			break;

		// DREG
		case 10:
			SetDREG(data);
			return;

		// TEMP
		case 11:
			SetTEMP(data);
			return;

		// TCH
		case 12:
			SetTCH(data);
			return;

		// TCM
		case 13:
			SetTCM(data);
			return;

		// TCL
		case 14:
			SetTCL(data);
			return;

		// U[u
		case 15:
			break;
	}

	// ʏAɂ͗Ȃ
	LOG2(Log::Warning, "`WX^ R%d <- $%02X", addr, data);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::ReadOnly(DWORD addr) const
{
	const BYTE *rom;

	ASSERT(this);

	// ʏ̓ǂݍ݂̏ꍇ
	if (addr >= memdev.first) {
		// SASÎ or ̏ꍇA̋Ԃ͌Ȃ
		if (scsi.type != 1) {
			return 0xff;
		}
	}

	// AhX}XN
	addr &= 0x1fff;

	// ROMf[^
	if (addr >= 0x20) {
		// ROMԂ
		rom = memory->GetSCSI();
		return rom[addr];
	}

	// WX^
	if ((addr & 1) == 0) {
		// AhX̓fR[hĂȂ
		return 0xff;
	}
	addr &= 0x1f;
	addr >>= 1;

	// WX^
	switch (addr) {
		// BDID
		case 0:
			return scsi.bdid;

		// SCTL
		case 1:
			return scsi.sctl;

		// SCMD
		case 2:
			return scsi.scmd;

		// U[u
		case 3:
			break;

		// INTS
		case 4:
			return scsi.ints;

		// PSNS
		case 5:
			return GetPSNS();

		// SSTS
		case 6:
			return GetSSTS();

		// SERR
		case 7:
			return GetSERR();

		// PCTL
		case 8:
			return scsi.pctl;

		// MBC
		case 9:
			return scsi.mbc;

		// DREG
		case 10:
			return scsi.buffer[scsi.offset];

		// TEMP
		case 11:
			return scsi.temp;

		// TCH
		case 12:
			return (BYTE)(scsi.tc >> 16);

		// TCM
		case 13:
			return (BYTE)(scsi.tc >> 8);

		// TCL
		case 14:
			return (BYTE)scsi.tc;

		// U[u
		case 15:
			break;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	f[^擾
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::GetSCSI(scsi_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT_DIAG();

	// [NRs[
	*buffer = scsi;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::Callback(Event* ev)
{
	BOOL success;

	ASSERT(this);
	ASSERT(ev);
	ASSERT_DIAG();

	// CD-DACxgp
	if (ev->GetUser() == 1) {
		if (scsi.cdrom) {
			// CD-ROM֒ʒm
			if (scsi.cdrom->NextFrame()) {
				// ʏ푬x
				ev->SetTime(26666);
			}
			else {
				// 751񂾂A␳
				ev->SetTime(26716);
			}
		}

		// p
		return TRUE;
	}

	// ZNVtF[ŶݗL
	ASSERT(ev->GetUser() == 0);
	if (scsi.phase != selection) {
		// P
		return FALSE;
	}

	// tO쐬
	success = FALSE;
	if (scsi.id != -1) {
		// IDCjVG[^ƃ^[QbgAw肳Ă
		if (scsi.disk[scsi.id]) {
			// LȃfBXN}EgĂ
			success = TRUE;
		}
	}

	// tOɂ蕪
	if (success) {
#if defined(SCSI_LOG)
		LOG1(Log::Normal, "ZNV ID=%d", scsi.id);
#endif	// SCSI_LOG

		// TCfNg
		scsi.tc &= 0x00ffff00;
		scsi.tc -= 0x2800;

		// Rv[g(ZNV)
		Interrupt(4, TRUE);

		// BSY=1
		scsi.bsy = TRUE;

		// ATN=1Ȃ烁bZ[WAEgtF[YAłȂ΃R}htF[Y
		if (scsi.atn) {
			MsgOut();
		}
		else {
			Command();
		}
	}
	else {
#if defined(SCSI_LOG)
		LOG1(Log::Normal, "ZNVs TEMP=$%02X", scsi.temp);
#endif	// SCSI_LOG

		// TC=0Ƃ(^CAEg)
		scsi.tc = 0;

		// ^CAEg(ZNVs)
		Interrupt(2, TRUE);

		// BSY=FALSEȂoXt[
		if (!scsi.bsy) {
			BusFree();
		}
	}

	// P
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	WX^Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ResetReg()
{
	ASSERT(this);

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "WX^Zbg");
#endif	// SCSI_LOG

	// JgID
	scsi.id = -1;

	// 荞
	scsi.vector = -1;

	// WX^
	scsi.bdid = 0x80;
	scsi.sctl = 0x80;
	scsi.scmd = 0;
	scsi.ints = 0;
	scsi.sdgc = 0;
	scsi.pctl = 0;
	scsi.mbc = 0;
	scsi.temp = 0;
	scsi.tc = 0;

	// ]R}h~
	scsi.trans = FALSE;

	// tF[Y(SPC̓oXt[̂)
	scsi.phase = busfree;

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	]Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ResetCtrl()
{
	ASSERT(this);

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "]Zbg");
#endif	// SCSI_LOG

	// ]R}h~
	scsi.trans = FALSE;
}

//---------------------------------------------------------------------------
//
//	oXZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ResetBus(BOOL reset)
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	if (reset) {
		LOG0(Log::Normal, "SCSIoX Zbg");
	}
	else {
		LOG0(Log::Normal, "SCSIoX Zbg");
	}
#endif	// SCSI_LOG

	// L
	scsi.rst = reset;

	// M(oXt[Ɠ)
	scsi.bsy = FALSE;
	scsi.sel = FALSE;
	scsi.atn = FALSE;
	scsi.msg = FALSE;
	scsi.cd = FALSE;
	scsi.io = FALSE;
	scsi.req = FALSE;
	scsi.ack = FALSE;

	// ]R}h~
	scsi.trans = FALSE;

	// tF[Y(SPC̓oXt[̂)
	scsi.phase = busfree;

	// oXZbg荞ݔE
	Interrupt(0, reset);
}

//---------------------------------------------------------------------------
//
//	BDIDݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetBDID(DWORD data)
{
	DWORD bdid;

	ASSERT(this);
	ASSERT(data < 0x100);

	// 3bit̂ݗL
	data &= 0x07;

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "{[hIDݒ ID%d", data);
#endif	// SCSI_LOG

	// lrbg֕ϊ
	bdid = (DWORD)(1 << data);

	// قȂƂ́AfBXNč\zKv
	if (scsi.bdid != bdid) {
		scsi.bdid = bdid;
		Construct();
	}
}

//---------------------------------------------------------------------------
//
//	SCTLݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetSCTL(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "SCTLݒ $%02X", data);
#endif	// SCSI_LOG

	scsi.sctl = data;

	// bit7:Reset & Disable
	if (scsi.sctl & 0x80) {
		// WX^Zbg
		ResetReg();
	}

	// bit6:Reset Control
	if (scsi.sctl & 0x40) {
		// ]Zbg
		ResetCtrl();
	}

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	SCMDݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetSCMD(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// bit4: RST Out
	if (data & 0x10) {
		if ((scsi.scmd & 0x10) == 0) {
			// RST01
			if ((scsi.sctl & 0x80) == 0) {
				// SPCZbgłȂ
				ResetBus(TRUE);
			}
		}
	}
	else {
		if (scsi.scmd & 0x10) {
			// RST10
			if ((scsi.sctl & 0x80) == 0) {
				// SPCZbgłȂ
				ResetBus(FALSE);
			}
		}
	}

	// R}hL
	scsi.scmd = data;

	// bit7-5:Command
	switch (scsi.scmd >> 5) {
		// Bus Release
		case 0:
			BusRelease();
			break;

		// Select
		case 1:
			Select();
			break;

		// Reset ATN
		case 2:
			ResetATN();
			break;

		// Set ATN
		case 3:
			SetATN();
			break;

		// Transfer
		case 4:
			Transfer();
			break;

		// Transfer Pause
		case 5:
			TransPause();
			break;

		// Reset ACK/REQ
		case 6:
			ResetACKREQ();
			break;

		// Set ACK/REQ
		case 7:
			SetACKREQ();
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}
}

//---------------------------------------------------------------------------
//
//	INTSݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetINTS(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "荞݃NA $%02X", data);
#endif	// SCSI_LOG

	// rbg𗧂Ă荞݂艺
	scsi.ints &= ~data;

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	PSNS擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::GetPSNS() const
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// f[^
	data = 0;

	// bit7:REQ
	if (scsi.req) {
		data |= 0x80;
	}

	// bit6:ACK
	if (scsi.ack) {
		data |= 0x40;
	}

	// bit5:ATN
	if (scsi.atn) {
		data |= 0x20;
	}

	// bit4:SEL
	if (scsi.sel) {
		data |= 0x10;
	}

	// bit3:BSY
	if (scsi.bsy) {
		data |= 0x08;
	}

	// bit2:MSG
	if (scsi.msg) {
		data |= 0x04;
	}

	// bit1:C/D
	if (scsi.cd) {
		data |= 0x02;
	}

	// bit0:I/O
	if (scsi.io) {
		data |= 0x01;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	SDGCݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetSDGC(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "SDGCݒ $%02X", data);
#endif	// SCSI_LOG

	scsi.sdgc = data;
}

//---------------------------------------------------------------------------
//
//	SSTS擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::GetSSTS() const
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// f[^
	data = 0;

	// bit6-7:Connected
	if (scsi.phase != busfree) {
		// oXt[ȊÓACjVG[^ƂČ
		data = 0x80;
	}

	// bit5:SPC BUSY
	if (scsi.bsy) {
		data |= 0x20;
	}

	// bit4:Transfer Progress
	if (scsi.trans) {
		data |= 0x10;
	}

	// bit3:Reset
	if (scsi.rst) {
		data |= 0x08;
	}

	// bit2:TC=0
	if (scsi.tc == 0) {
		data |= 0x04;
	}

	// bit0-1:FIFO (00:1`7oCg, 01:Empty, 10:8oCg, 11:gp)
	if (scsi.trans) {
		// ]
		if (scsi.io) {
			// ReadɌAFIFOɃf[^܂\
			if ((scsi.length > 0) && (scsi.tc > 0)) {
				if ((scsi.length >= 8) && (scsi.tc >= 8)) {
					// 8oCgȏ(TS-6BS1mkIII)
					data |= 0x02;
					return data;
				}
				else {
					// 7oCgȉ
					return data;
				}
			}
		}
	}

	// FIFOɃf[^͂Ȃ
	data |= 0x01;
	return data;
}

//---------------------------------------------------------------------------
//
//	SERR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::GetSERR() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// vO]ANGXgMoȂ
	if (scsi.sdgc & 0x20) {
		if (scsi.trans) {
			if (scsi.tc != 0) {
				// Xfer Out
				return 0x20;
			}
		}
	}

	// ȊO0
	return 0x00;
}

//---------------------------------------------------------------------------
//
//	PCTLݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetPCTL(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	scsi.pctl = data;

}

//---------------------------------------------------------------------------
//
//	DREG擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::GetDREG()
{
	DWORD data;

	ASSERT(this);
	ASSERT_DIAG();

	// ]łȂ΁AFFԂ
	if (!scsi.trans) {
		return 0xff;
	}

	// NGXgオĂȂ΁AFFԂ
	if (!scsi.req) {
		return 0xff;
	}

	// TC0ȂAFFԂ
	if (scsi.tc == 0) {
		return 0xff;
	}

	// REQ-ACKnhVF[Nōs
	Xfer(&data);
	XferNext();

	// MBC
	scsi.mbc = (scsi.mbc - 1) & 0x0f;

	// TC
	scsi.tc--;
	if (scsi.tc == 0) {
		// ](scsi.length != 0ȂASCSIoX̓bN)
		TransComplete();
		return data;
	}

	// Xe[^XtF[YɂȂĂ΁A]orG[
	if (scsi.phase == status) {
		// ]
		TransComplete();
		return data;
	}

	// ǂݍ݌p
	return data;
}

//---------------------------------------------------------------------------
//
//	DREGݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetDREG(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// ]łȂ΁AȂ
	if (!scsi.trans) {
		return;
	}

	// NGXgオĂȂ΁AȂ
	if (!scsi.req) {
		return;
	}

	// TC0ȂAȂ
	if (scsi.tc == 0) {
		return;
	}

	// REQ-ACKnhVF[Nōs
	Xfer(&data);
	XferNext();

	// MBC
	scsi.mbc = (scsi.mbc - 1) & 0x0f;

	// TC
	scsi.tc--;
	if (scsi.tc == 0) {
		// ](scsi.length != 0ȂASCSIoX̓bN)
		TransComplete();
		return;
	}

	// Xe[^XtF[YɂȂĂ΁A]orG[
	if (scsi.phase == status) {
		// ]
		TransComplete();
		return;
	}
}

//---------------------------------------------------------------------------
//
//	TEMPݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetTEMP(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	scsi.temp = data;

	// ZNVtF[Y̏ꍇA0x00̃ZbgɂSEL=0ƂȂ
	if (scsi.phase == selection) {
		if (data == 0x00) {
			// SEL=0, BSY=0
			scsi.sel = FALSE;
			scsi.bsy = FALSE;
		}
	}
}

//---------------------------------------------------------------------------
//
//	TCHݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetTCH(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// b23-b16ɐݒ
	scsi.tc &= 0x0000ffff;
	data <<= 16;
	scsi.tc |= data;

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "TCHݒ TC=$%06X", scsi.tc);
#endif	// SCSI_LOG
}

//---------------------------------------------------------------------------
//
//	TCMݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetTCM(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// b15-b8ɐݒ
	scsi.tc &= 0x00ff00ff;
	data <<= 8;
	scsi.tc |= data;

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "TCMݒ TC=$%06X", scsi.tc);
#endif	// SCSI_LOG
}

//---------------------------------------------------------------------------
//
//	TCLݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetTCL(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// b7-b0ɐݒ
	scsi.tc &= 0x00ffff00;
	scsi.tc |= data;

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "TCLݒ TC=$%06X", scsi.tc);
#endif	// SCSI_LOG

	// SCMD$20ȂAZNVp
	if ((scsi.scmd & 0xe0) == 0x20) {
		SelectTime();
	}
}

//---------------------------------------------------------------------------
//
//	oX[X
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::BusRelease()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "Bus ReleaseR}h");
#endif	// SCSI_LOG

	// oXt[
	BusFree();
}

//---------------------------------------------------------------------------
//
//	ZNg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Select()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "SelectR}h");
#endif	// SCSI_LOG

	// SPCZbgASCSIoXZbǵAȂ
	if (scsi.sctl & 0x80) {
		return;
	}
	if (scsi.rst) {
		return;
	}

	// SCTLbit41ȂAA[rg[VtF[YɎs
	if (scsi.sctl & 0x10) {
		Arbitration();
	}

	// PCTLbit01ȂAZNV(^[Qbĝ݋)
	if (scsi.pctl & 1) {
		LOG0(Log::Warning, "ZNVtF[Y");
		BusFree();
		return;
	}

	// ZNVtF[Y
	Selection();
}

//---------------------------------------------------------------------------
//
//	ATNZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ResetATN()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "Reset ATNR}h");
#endif	// SCSI_LOG

	// CjVG[^삷M
	scsi.atn = FALSE;
}

//---------------------------------------------------------------------------
//
//	ATNZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetATN()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "Set ATNR}h");
#endif	// SCSI_LOG

	// CjVG[^삷M
	scsi.atn = TRUE;
}

//---------------------------------------------------------------------------
//
//	]
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Transfer()
{
	ASSERT(this);
	ASSERT_DIAG();
	ASSERT(scsi.req);

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "TransferR}h");
#endif	// SCSI_LOG

	// foCX̏łĂ邩
	if (scsi.length <= 0) {
		// Service Required(tF[YȂ)
		Interrupt(3, TRUE);
		return;
	}

	// tF[YvĂ邩

	// Rg[̏łĂ邩
	if (scsi.tc == 0) {
		// Rv[g(]I)
		Interrupt(4, TRUE);
	}

	// ]R}hJn
	scsi.trans = TRUE;
}

//---------------------------------------------------------------------------
//
//	]f
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::TransPause()
{
	ASSERT(this);
	ASSERT_DIAG();

	// ^[Qbĝݔsł
	LOG0(Log::Warning, "Transfer PauseR}h");
}

//---------------------------------------------------------------------------
//
//	]
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::TransComplete()
{
	ASSERT(this);
	ASSERT(!scsi.ack);
	ASSERT(scsi.trans);
	ASSERT_DIAG();

	// ]
	scsi.trans = FALSE;

	// ]荞
	Interrupt(4, TRUE);
}

//---------------------------------------------------------------------------
//
//	ACKREQZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ResetACKREQ()
{
	ASSERT(this);
	ASSERT_DIAG();

	// ACKオĂꍇ݈̂Ӗ
	if (!scsi.ack) {
		return;
	}

	// ACKオtF[Y͌
	ASSERT((scsi.phase >= command) && (scsi.phase != execute));

	// f[^]ɐi߂
	XferNext();
}

//---------------------------------------------------------------------------
//
//	ACK/REQZbg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SetACKREQ()
{
	ASSERT(this);
	ASSERT_DIAG();

	// REQオĂꍇ݈̂Ӗ
	if (!scsi.req) {
		return;
	}

	// REQオtF[Y͌
	ASSERT((scsi.phase >= command) && (scsi.phase != execute));

	// TEMPWX^-SCSIf[^oXԂ̃f[^]s
	Xfer(&scsi.temp);
}

//---------------------------------------------------------------------------
//
//	f[^]
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Xfer(DWORD *reg)
{
	ASSERT(this);
	ASSERT(reg);
	ASSERT_DIAG();

	// REQオĂ邱
	ASSERT(scsi.req);
	ASSERT(!scsi.ack);
	ASSERT((scsi.phase >= command) && (scsi.phase != execute));

	// CjVG[^삷M
	scsi.ack = TRUE;

	// SCSIf[^oXɃf[^悹
	switch (scsi.phase) {
		// R}htF[Y
		case command:
#if defined(SCSI_LOG)
			LOG1(Log::Normal, "R}htF[Y $%02X", *reg);
#endif	// SCSI_LOG
			// ŏ̃f[^(ItZbg0)ɂ背OXĐݒ
			scsi.cmd[scsi.offset] = *reg;
			if (scsi.offset == 0) {
				if (scsi.temp >= 0x20) {
					// 10oCgCDB
					scsi.length = 10;
				}
			}
			break;

		// bZ[WCtF[Y
		case msgin:
			*reg = scsi.message;
#if defined(SCSI_LOG)
			LOG1(Log::Normal, "bZ[WCtF[Y $%02X", *reg);
#endif	// SCSI_LOG
			break;

		// bZ[WAEgtF[Y
		case msgout:
			scsi.message = *reg;
#if defined(SCSI_LOG)
			LOG1(Log::Normal, "bZ[WAEgtF[Y $%02X", *reg);
#endif	// SCSI_LOG
			break;

		// f[^CtF[Y
		case datain:
			*reg = (DWORD)scsi.buffer[scsi.offset];
			break;

		// f[^AEgtF[Y
		case dataout:
			scsi.buffer[scsi.offset] = (BYTE)*reg;
			break;

		// Xe[^XtF[Y
		case status:
			*reg = scsi.status;
#if defined(SCSI_LOG)
			LOG1(Log::Normal, "Xe[^XtF[Y $%02X", *reg);
#endif	// SCSI_LOG
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}

	// ^[Qbg삷M
	scsi.req = FALSE;
}

//---------------------------------------------------------------------------
//
//	f[^]p
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::XferNext()
{
	BOOL result;

	ASSERT(this);
	ASSERT_DIAG();

	// ACKオĂ邱
	ASSERT(!scsi.req);
	ASSERT(scsi.ack);
	ASSERT((scsi.phase >= command) && (scsi.phase != execute));

	// ItZbgƃOX
	ASSERT(scsi.length >= 1);
	scsi.offset++;
	scsi.length--;

	// CjVG[^삷M
	scsi.ack = FALSE;

	// OX!=0ȂAĂreqZbg
	if (scsi.length != 0) {
		scsi.req = TRUE;
		return;
	}

	// ubNZAUg
	scsi.blocks--;
	result = TRUE;

	// f[^󗝌̏(tF[Y)
	switch (scsi.phase) {
		// f[^CtF[Y
		case datain:
			if (scsi.blocks != 0) {
				// ̃obt@ݒ(offset, lengthZbg邱)
				result = XferIn();
			}
			break;

		// f[^AEgtF[Y
		case dataout:
			if (scsi.blocks == 0) {
				// ̃obt@ŏI
				result = XferOut(FALSE);
			}
			else {
				// ̃obt@ɑ(offset, lengthZbg邱)
				result = XferOut(TRUE);
			}
			break;

		// bZ[WAEgtF[Y
		case msgout:
			if (!XferMsg(scsi.message)) {
				// bZ[WAEgɎsAɃoXt[
				BusFree();
				return;
			}
			// bZ[WCɔAbZ[Wf[^NAĂ
			scsi.message = 0x00;
			break;
	}

	// UgFALSEȂAXe[^XtF[Yֈڍs
	if (!result) {
		Error();
		return;
	}

	// ubN!=0ȂAĂreqZbg
	if (scsi.blocks != 0){
		ASSERT(scsi.length > 0);
		ASSERT(scsi.offset == 0);
		scsi.req = TRUE;
		return;
	}

	// tF[YɈړ
	switch (scsi.phase) {
		// R}htF[Y
		case command:
			// stF[Y
			Execute();
			break;

		// bZ[WCtF[Y
		case msgin:
			// oXt[tF[Y
			BusFree();
			break;

		// bZ[WAEgtF[Y
		case msgout:
			// R}htF[Y
			Command();
			break;

		// f[^CtF[Y
		case datain:
			// Xe[^XtF[Y
			Status();
			break;

		// f[^AEgtF[Y
		case dataout:
			// Xe[^XtF[Y
			Status();
			break;

		// Xe[^XtF[Y
		case status:
			// bZ[WCtF[Y
			MsgIn();
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			break;
	}
}

//---------------------------------------------------------------------------
//
//	f[^]IN
//	offset, lengthĐݒ肷邱
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::XferIn()
{
	ASSERT(this);
	ASSERT(scsi.phase == datain);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// Xe[g[hɂfBXNȂꍇ
	if (!scsi.disk[scsi.id]) {
		// f[^C~
		return FALSE;
	}

	// READnR}hɌ
	switch (scsi.cmd[0]) {
		// READ(6)
		case 0x08:
		// READ(10)
		case 0x28:
			// fBXNǂݎs
			scsi.length = scsi.disk[scsi.id]->Read(scsi.buffer, scsi.next);
			scsi.next++;

			// G[ȂAXe[^XtF[Y
			if (scsi.length <= 0) {
				// f[^C~
				return FALSE;
			}

			// ȂA[Nݒ
			scsi.offset = 0;
			break;

		// ̑(肦Ȃ)
		default:
			ASSERT(FALSE);
			return FALSE;
	}

	// obt@ݒɐ
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	f[^]OUT
//	cont=TRUȄꍇAoffset, lengthĐݒ肷邱
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::XferOut(BOOL cont)
{
	ASSERT(this);
	ASSERT(scsi.phase == dataout);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// Xe[g[hɂfBXNȂꍇ
	if (!scsi.disk[scsi.id]) {
		// f[^AEg~
		return FALSE;
	}

	// MODE SELECT܂́AWRITEn
	switch (scsi.cmd[0]) {
		// MODE SELECT
		case 0x15:
			if (!scsi.disk[scsi.id]->ModeSelect(scsi.buffer, scsi.offset)) {
				// MODE SELECTɎs
				return FALSE;
			}
			break;

		// WRITE(6)
		case 0x0a:
		// WRITE(10)
		case 0x2a:
		// WRITE AND VERIFY
		case 0x2e:
			// ݂s
			if (!scsi.disk[scsi.id]->Write(scsi.buffer, scsi.next - 1)) {
				// ݎs
				return FALSE;
			}

			// CgX[LbV
			if (!cache_wb) {
				scsi.disk[scsi.id]->Flush();
			}

			// ̃ubNKvȂȂ炱܂
			scsi.next++;
			if (!cont) {
				break;
			}

			// ̃ubN`FbN
			scsi.length = scsi.disk[scsi.id]->WriteCheck(scsi.next);
			if (scsi.length <= 0) {
				// ݂łȂ
				return FALSE;
			}

			// ȂA[Nݒ
			scsi.offset = 0;
			break;
	}

	// obt@ۑɐ
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	f[^]MSG
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::XferMsg(DWORD /*msg*/)
{
	ASSERT(this);
	ASSERT(scsi.phase == msgout);
	ASSERT_DIAG();

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	oXt[tF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::BusFree()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "oXt[tF[Y");
#endif	// SCSI_LOG

	// tF[Yݒ
	scsi.phase = busfree;

	// M
	scsi.bsy = FALSE;
	scsi.sel = FALSE;
	scsi.atn = FALSE;
	scsi.msg = FALSE;
	scsi.cd = FALSE;
	scsi.io = FALSE;
	scsi.req = FALSE;
	scsi.ack = FALSE;

	// Xe[^XƃbZ[W(GOOD)
	scsi.status = 0x00;
	scsi.message = 0x00;

	// Cxg~
	event.SetTime(0);

	// PCTLBusfree INT Enable`FbN
	if (scsi.pctl & 0x8000) {
		// Disconnect荞
		Interrupt(5, TRUE);
	}
	else {
		// 荞݃It
		Interrupt(5, FALSE);
	}
}

//---------------------------------------------------------------------------
//
//	A[rg[VtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Arbitration()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "A[rg[VtF[Y");
#endif	// SCSI_LOG

	// tF[Yݒ
	scsi.phase = arbitration;

	// M
	scsi.bsy = TRUE;
	scsi.sel = TRUE;
}

//---------------------------------------------------------------------------
//
//	ZNVtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Selection()
{
	int i;
	DWORD temp;

	ASSERT(this);
	ASSERT_DIAG();

	// tF[Yݒ
	scsi.phase = selection;

	// ^[QbgID
	scsi.id = -1;

	// TEMPWX^ɂ̓CjVG[^ƃ^[QbgÃrbgKv
	temp = scsi.temp;
	if ((temp & scsi.bdid) == scsi.bdid) {
		// CjVG[^̃rbg폜
		temp &= ~(scsi.bdid);

		// ^[QbgT
		for (i=0; i<8; i++) {
			if (temp == (DWORD)(1 << i)) {
				scsi.id = i;
				break;
			}
		}
	}

#if defined(SCSI_LOG)
	if (scsi.id != -1) {
		if (scsi.disk[scsi.id]) {
			LOG1(Log::Normal, "ZNVtF[Y ID=%d (foCX)", scsi.id);
		}
		else {
			LOG1(Log::Normal, "ZNVtF[Y ID=%d (ڑ)", scsi.id);
		}
	}
	else {
		LOG0(Log::Normal, "ZNVtF[Y ()");
	}
#endif	// SCSI_LOG

	// ZNV^C
	SelectTime();
}

//---------------------------------------------------------------------------
//
//	ZNVtF[Y Ԑݒ
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SelectTime()
{
	DWORD tc;

	ASSERT(this);
	ASSERT((scsi.scmd & 0xe0) == 0x20);
	ASSERT_DIAG();

	tc = scsi.tc;
	if (tc == 0) {
		LOG0(Log::Warning, "ZNV^CAEg");
		tc = 0x1000000;
	}
	tc &= 0x00ffff00;
	tc += 15;

	// TCLl
	tc += (((scsi.tc & 0xff) + 7) >> 1);

	// 400nsPʂA500nsP(hus)֕ϊ
	tc = (tc * 4 / 5);

	// tF[Y
	scsi.phase = selection;

	// M
	scsi.sel = TRUE;

	// Cxg^Cݒ
	event.SetDesc("Selection");
	if (scsi.id != -1) {
		if (scsi.disk[scsi.id]) {
			// wID̏ꍇA16usŃZNV𐬌
			event.SetTime(32);
			return;
		}
	}

	// ^CAEg̏ꍇATCʂɐݒ
	event.SetTime(tc);
}

//---------------------------------------------------------------------------
//
//	R}htF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Command()
{
	ASSERT(this);
	ASSERT((scsi.id >= 0) && (scsi.id <= 7));
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "R}htF[Y");
#endif	// SCSI_LOG

	// Xe[g[hɂfBXNȂꍇ
	if (!scsi.disk[scsi.id]) {
		// R}htF[Y~
		BusFree();
		return;
	}

	// tF[Yݒ
	scsi.phase = command;

	// CjVG[^삷M
	scsi.atn = FALSE;

	// ^[Qbg삷M
	scsi.msg = FALSE;
	scsi.cd = TRUE;
	scsi.io = FALSE;

	// f[^]6oCgx1ubN
	scsi.offset = 0;
	scsi.length = 6;
	scsi.blocks = 1;

	// R}hv
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	stF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Execute()
{
	ASSERT(this);
	ASSERT((scsi.id >= 0) && (scsi.id <= 7));
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "stF[Y R}h$%02X", scsi.cmd[0]);
#endif	// SCSI_LOG

	// Xe[g[hɂfBXNȂꍇ
	if (!scsi.disk[scsi.id]) {
		// R}htF[Y~
		Error();
		return;
	}

	// tF[Yݒ
	scsi.phase = execute;

	// f[^]̂߂̏
	scsi.offset = 0;
	scsi.blocks = 1;

	// R}hʏ
	switch (scsi.cmd[0]) {
		// TEST UNIT READY
		case 0x00:
			TestUnitReady();
			return;

		// REZERO
		case 0x01:
			Rezero();
			return;

		// REQUEST SENSE
		case 0x03:
			RequestSense();
			return;

		// FORMAT UNIT
		case 0x04:
			Format();
			return;

		// REASSIGN BLOCKS
		case 0x07:
			Reassign();
			return;

		// READ(6)
		case 0x08:
			Read6();
			return;

		// WRITE(6)
		case 0x0a:
			Write6();
			return;

		// SEEK(6)
		case 0x0b:
			Seek6();
			return;

		// INQUIRY
		case 0x12:
			Inquiry();
			return;

		// MODE SELECT
		case 0x15:
			ModeSelect();
			return;

		// MDOE SENSE
		case 0x1a:
			ModeSense();
			return;

		// START STOP UNIT
		case 0x1b:
			StartStop();
			return;

		// SEND DIAGNOSTIC
		case 0x1d:
			SendDiag();
			return;

		// PREVENT/ALLOW MEDIUM REMOVAL
		case 0x1e:
			Removal();
			return;

		// READ CAPACITY
		case 0x25:
			ReadCapacity();
			return;

		// READ(10)
		case 0x28:
			Read10();
			return;

		// WRITE(10)
		case 0x2a:
			Write10();
			return;

		// SEEK(10)
		case 0x2b:
			Seek10();
			return;

		// WRITE AND VERIFY
		case 0x2e:
			Write10();
			return;

		// VERIFY
		case 0x2f:
			Verify();
			return;

		// READ TOC
		case 0x43:
			ReadToc();
			return;

		// PLAY AUDIO(10)
		case 0x45:
			PlayAudio10();
			return;

		// PLAY AUDIO MSF
		case 0x47:
			PlayAudioMSF();
			return;

		// PLAY AUDIO TRACK
		case 0x48:
			PlayAudioTrack();
			return;
	}

	// ȊO͑ΉĂȂ
	LOG1(Log::Warning, "ΉR}h $%02X", scsi.cmd[0]);
	scsi.disk[scsi.id]->InvalidCmd();

	// G[
	Error();
}

//---------------------------------------------------------------------------
//
//	bZ[WCtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::MsgIn()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "bZ[WCtF[Y");
#endif	// SCSI_LOG

	// tF[Yݒ
	scsi.phase = msgin;

	// ^[Qbg삷M
	scsi.msg = TRUE;
	scsi.cd = TRUE;
	scsi.io = TRUE;

	// f[^]1oCgx1ubN
	scsi.offset = 0;
	scsi.length = 1;
	scsi.blocks = 1;

	// bZ[Wv
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	bZ[WAEgtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::MsgOut()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "bZ[WAEgtF[Y");
#endif	// SCSI_LOG

	// tF[Yݒ
	scsi.phase = msgout;

	// ^[Qbg삷M
	scsi.msg = TRUE;
	scsi.cd = TRUE;
	scsi.io = FALSE;

	// f[^]1oCgx1ubN
	scsi.offset = 0;
	scsi.length = 1;
	scsi.blocks = 1;

	// bZ[Wv
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	f[^CtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::DataIn()
{
	ASSERT(this);
	ASSERT(scsi.length >= 0);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "f[^CtF[Y");
#endif	// SCSI_LOG

	// OX0ȂAXe[^XtF[Y
	if (scsi.length == 0) {
		Status();
		return;
	}

	// tF[Yݒ
	scsi.phase = datain;

	// ^[Qbg삷M
	scsi.msg = FALSE;
	scsi.cd = FALSE;
	scsi.io = TRUE;

	// length, blocks͐ݒς
	ASSERT(scsi.length > 0);
	ASSERT(scsi.blocks > 0);
	scsi.offset = 0;

	// f[^v
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	f[^AEgtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::DataOut()
{
	ASSERT(this);
	ASSERT(scsi.length >= 0);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "f[^AEgtF[Y");
#endif	// SCSI_LOG

	// OX0ȂAXe[^XtF[Y
	if (scsi.length == 0) {
		Status();
		return;
	}

	// tF[Yݒ
	scsi.phase = dataout;

	// ^[Qbg삷M
	scsi.msg = FALSE;
	scsi.cd = FALSE;
	scsi.io = FALSE;

	// length, blocks͐ݒς
	ASSERT(scsi.length > 0);
	ASSERT(scsi.blocks > 0);
	scsi.offset = 0;

	// f[^v
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	Xe[^XtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Status()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "Xe[^XtF[Y");
#endif	// SCSI_LOG

	// tF[Yݒ
	scsi.phase = status;

	// ^[Qbg삷M
	scsi.msg = FALSE;
	scsi.cd = TRUE;
	scsi.io = TRUE;

	// f[^]1oCgx1ubN
	scsi.offset = 0;
	scsi.length = 1;
	scsi.blocks = 1;

	// Xe[^Xv
	scsi.req = TRUE;
}

//---------------------------------------------------------------------------
//
//	荞ݗv
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Interrupt(int type, BOOL flag)
{
	DWORD ints;

	ASSERT(this);
	ASSERT((type >= 0) && (type <= 7));
	ASSERT_DIAG();

	// INTS̃obNAbv
	ints = scsi.ints;

	// INTSWX^ݒ
	if (flag) {
		// 荞ݗv
		scsi.ints |= (1 << type);
	}
	else {
		// 荞݃LZ
		scsi.ints &= ~(1 << type);
	}

	// ωĂ΁A荞݃`FbN
	if (ints != scsi.ints) {
		IntCheck();
	}
}

//---------------------------------------------------------------------------
//
//	荞݃`FbN
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::IntCheck()
{
	int v;
	int lev;

	ASSERT(this);
	ASSERT_DIAG();

	// 荞݃xݒ
	if (scsi.type == 1) {
		// Ot(x2܂4I)
		lev = scsi.ilevel;
	}
	else {
		// (x1Œ)
		lev = 1;
	}

	// vxN^쐬
	v = -1;
	if (scsi.sctl & 0x01) {
		// 荞݋
		if (scsi.ints != 0) {
			// 荞ݗvBxN^쐬
			if (scsi.type == 1) {
				// Ot(xN^$F6)
				v = 0xf6;
			}
			else {
				// (xN^$6C)
				v = 0x6c;
			}
		}
	}

	// vĂOK
	if (scsi.vector == v) {
		return;
	}

	// Ɋ荞ݗvĂ΁ALZ
	if (scsi.vector >= 0) {
		cpu->IntCancel(lev);
	}

	// LZvȂA܂
	if (v < 0) {
#if defined(SCSI_LOG)
		LOG2(Log::Normal, "荞݃LZ x%d xN^$%02X",
							lev, scsi.vector);
#endif	// SCSI_LOG
		scsi.vector = -1;
		return;
	}

	// 荞ݗv
#if defined(SCSI_LOG)
	LOG3(Log::Normal, "荞ݗv x%d xN^$%02X INTS=%02X",
						lev, v, scsi.ints);
#endif	// SCSI_LOG
	cpu->Interrupt(lev, v);
	scsi.vector = v;
}

//---------------------------------------------------------------------------
//
//	荞ACK
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::IntAck(int level)
{
	ASSERT(this);
	ASSERT((level == 1) || (level == 2) || (level == 4));
	ASSERT_DIAG();

	// ZbgɁACPU犄荞݂ԈēꍇA܂͑̃foCX
	if (scsi.vector < 0) {
		return;
	}

	// 荞݃xقȂĂ΁ÃfoCX
	if (scsi.type == 1) {
		// Ot(x2܂4I)
		if (level != scsi.ilevel) {
			return;
		}
	}
	else {
		// (x1Œ)
		if (level != 1) {
			return;
		}
	}

	// 荞݂Ȃ
	scsi.vector = -1;

	// 荞݃`FbN
	IntCheck();
}

//---------------------------------------------------------------------------
//
//	SCSI-ID擾
//
//---------------------------------------------------------------------------
int FASTCALL SCSI::GetSCSIID() const
{
	int id;
	DWORD bdid;

	ASSERT(this);
	ASSERT_DIAG();

	// ݂Ȃ7Œ
	if (scsi.type == 0) {
		return 7;
	}

	// BDID쐬
	ASSERT(scsi.bdid != 0);
	ASSERT(scsi.bdid < 0x100);

	// 
	id = 0;
	bdid = scsi.bdid;

	// rbg琔l֕ϊ
	while ((bdid & 1) == 0) {
		bdid >>= 1;
		id++;
	}

	ASSERT((id >= 0) && (id <= 7));
	return id;
}

//---------------------------------------------------------------------------
//
//	BUSY`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsBusy() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// BSYĂȂFALSE
	if (!scsi.bsy) {
		return FALSE;
	}

	// ZNVtF[YȂFALSE
	if (scsi.phase == selection) {
		return FALSE;
	}

	// BUSY
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	BUSY`FbN
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSI::GetBusyDevice() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// BSYĂȂ0
	if (!scsi.bsy) {
		return 0;
	}

	// ZNVtF[YȂ0
	if (scsi.phase == selection) {
		return 0;
	}

	// ZNVIĂ̂ŁAʏ̓foCX݂
	ASSERT((scsi.id >= 0) && (scsi.id <= 7));
	ASSERT(scsi.disk[scsi.id]);

	// foCXȂ0
	if (!scsi.disk[scsi.id]) {
		return 0;
	}

	return scsi.disk[scsi.id]->GetID();
}

//---------------------------------------------------------------------------
//
//	ʃG[
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Error()
{
	ASSERT(this);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "G[(Xe[^XtF[Y)");
#endif	// SCSI_LOG

	// Xe[^XƃbZ[Wݒ(CHECK CONDITION)
	scsi.status = 0x02;
	scsi.message = 0x00;

	// Cxg~
	event.SetTime(0);

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	TEST UNIT READY
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::TestUnitReady()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "TEST UNIT READYR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->TestUnitReady(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REZERO UNIT
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Rezero()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "REZERO UNITR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Rezero(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REQUEST SENSE
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::RequestSense()
{
	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "REQUEST SENSER}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->RequestSense(scsi.cmd, scsi.buffer);
	ASSERT(scsi.length > 0);

#if defined(SCSI_LOG)
	LOG1(Log::Normal, "ZXL[ $%02X", scsi.buffer[2]);
#endif

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	FORMAT UNIT
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Format()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "FORMAT UNITR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Format(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	REASSIGN BLOCKS
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Reassign()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "REASSIGN BLOCKSR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Reassign(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	READ(6)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Read6()
{
	DWORD record;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// R[hԍƃubN擾
	record = scsi.cmd[1] & 0x1f;
	record <<= 8;
	record |= scsi.cmd[2];
	record <<= 8;
	record |= scsi.cmd[3];
	scsi.blocks = scsi.cmd[4];
	if (scsi.blocks == 0) {
		scsi.blocks = 0x100;
	}

#if defined(SCSI_LOG)
	LOG2(Log::Normal, "READ(6)R}h R[h=%06X ubN=%d", record, scsi.blocks);
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->Read(scsi.buffer, record);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// ̃ubNݒ
	scsi.next = record + 1;

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	WRITE(6)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Write6()
{
	DWORD record;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// R[hԍƃubN擾
	record = scsi.cmd[1] & 0x1f;
	record <<= 8;
	record |= scsi.cmd[2];
	record <<= 8;
	record |= scsi.cmd[3];
	scsi.blocks = scsi.cmd[4];
	if (scsi.blocks == 0) {
		scsi.blocks = 0x100;
	}

#if defined(SCSI_LOG)
	LOG2(Log::Normal, "WRITE(6)R}h R[h=%06X ubN=%d", record, scsi.blocks);
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->WriteCheck(record);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// ̃ubNݒ
	scsi.next = record + 1;

	// f[^AEgtF[Y
	DataOut();
}

//---------------------------------------------------------------------------
//
//	SEEK(6)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Seek6()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "SEEK(6)R}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Seek(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Inquiry()
{
	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "INQUIRYR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->Inquiry(scsi.cmd, scsi.buffer);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	MODE SELECT
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ModeSelect()
{
	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "MODE SELECTR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->SelectCheck(scsi.cmd);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// f[^AEgtF[Y
	DataOut();
}

//---------------------------------------------------------------------------
//
//	MODE SENSE
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ModeSense()
{
	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "MODE SENSER}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->ModeSense(scsi.cmd, scsi.buffer);
	ASSERT(scsi.length >= 0);
	if (scsi.length == 0) {
		LOG1(Log::Warning, "T|[gĂȂMODE SENSEy[W $%02X", scsi.cmd[2]);

		// s(G[)
		Error();
		return;
	}

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	START STOP UNIT
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::StartStop()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "START STOP UNITR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->StartStop(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	SEND DIAGNOSTIC
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::SendDiag()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "SEND DIAGNOSTICR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->SendDiag(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	PREVENT/ALLOW MEDIUM REMOVAL
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Removal()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "PREVENT/ALLOW MEDIUM REMOVALR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Removal(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	READ CAPACITY
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ReadCapacity()
{
	int length;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "READ CAPACITYR}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	length = scsi.disk[scsi.id]->ReadCapacity(scsi.cmd, scsi.buffer);
	ASSERT(length >= 0);
	if (length <= 0) {
		Error();
		return;
	}

	// OXݒ
	scsi.length = length;

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	READ(10)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Read10()
{
	DWORD record;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// R[hԍƃubN擾
	record = scsi.cmd[2];
	record <<= 8;
	record |= scsi.cmd[3];
	record <<= 8;
	record |= scsi.cmd[4];
	record <<= 8;
	record |= scsi.cmd[5];
	scsi.blocks = scsi.cmd[7];
	scsi.blocks <<= 8;
	scsi.blocks |= scsi.cmd[8];

#if defined(SCSI_LOG)
	LOG2(Log::Normal, "READ(10)R}h R[h=%08X ubN=%d", record, scsi.blocks);
#endif	// SCSI_LOG

	// ubN0͏Ȃ
	if (scsi.blocks == 0) {
		Status();
		return;
	}

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->Read(scsi.buffer, record);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// ̃ubNݒ
	scsi.next = record + 1;

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	WRITE(10)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Write10()
{
	DWORD record;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// R[hԍƃubN擾
	record = scsi.cmd[2];
	record <<= 8;
	record |= scsi.cmd[3];
	record <<= 8;
	record |= scsi.cmd[4];
	record <<= 8;
	record |= scsi.cmd[5];
	scsi.blocks = scsi.cmd[7];
	scsi.blocks <<= 8;
	scsi.blocks |= scsi.cmd[8];

#if defined(SCSI_LOG)
	LOG2(Log::Normal, "WRTIE(10)R}h R[h=%08X ubN=%d", record, scsi.blocks);
#endif	// SCSI_LOG

	// ubN0͏Ȃ
	if (scsi.blocks == 0) {
		Status();
		return;
	}

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->WriteCheck(record);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// ̃ubNݒ
	scsi.next = record + 1;

	// f[^AEgtF[Y
	DataOut();
}

//---------------------------------------------------------------------------
//
//	SEEK(10)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Seek10()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

#if defined(SCSI_LOG)
	LOG0(Log::Normal, "SEEK(10)R}h");
#endif	// SCSI_LOG

	// hCuŃR}h
	status = scsi.disk[scsi.id]->Seek(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	VERIFY
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Verify()
{
	BOOL status;
	DWORD record;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// R[hԍƃubN擾
	record = scsi.cmd[2];
	record <<= 8;
	record |= scsi.cmd[3];
	record <<= 8;
	record |= scsi.cmd[4];
	record <<= 8;
	record |= scsi.cmd[5];
	scsi.blocks = scsi.cmd[7];
	scsi.blocks <<= 8;
	scsi.blocks |= scsi.cmd[8];

#if defined(SCSI_LOG)
	LOG2(Log::Normal, "VERIFYR}h R[h=%08X ubN=%d", record, scsi.blocks);
#endif	// SCSI_LOG

	// ubN0͏Ȃ
	if (scsi.blocks == 0) {
		Status();
		return;
	}

	// BytChk=0Ȃ
	if ((scsi.cmd[1] & 0x02) == 0) {
		// hCuŃR}h
		status = scsi.disk[scsi.id]->Seek(scsi.cmd);
		if (!status) {
			// s(G[)
			Error();
			return;
		}

		// Xe[^XtF[Y
		Status();
		return;
	}

	// eXgǂݍ
	scsi.length = scsi.disk[scsi.id]->Read(scsi.buffer, record);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// ̃ubNݒ
	scsi.next = record + 1;

	// f[^AEgtF[Y
	DataOut();
}

//---------------------------------------------------------------------------
//
//	READ TOC
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::ReadToc()
{
	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// hCuŃR}h
	scsi.length = scsi.disk[scsi.id]->ReadToc(scsi.cmd, scsi.buffer);
	if (scsi.length <= 0) {
		// s(G[)
		Error();
		return;
	}

	// f[^CtF[Y
	DataIn();
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO(10)
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::PlayAudio10()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// hCuŃR}h
	status = scsi.disk[scsi.id]->PlayAudio(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// CD-DACxgX^[g(26666~74+26716ŁAv1sec)
	if (cdda.GetTime() == 0) {
		cdda.SetTime(26666);
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO MSF
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::PlayAudioMSF()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// hCuŃR}h
	status = scsi.disk[scsi.id]->PlayAudioMSF(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// CD-DACxgX^[g(26666~74+26716ŁAv1sec)
	if (cdda.GetTime() == 0) {
		cdda.SetTime(26666);
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO TRACK
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::PlayAudioTrack()
{
	BOOL status;

	ASSERT(this);
	ASSERT(scsi.disk[scsi.id]);
	ASSERT_DIAG();

	// hCuŃR}h
	status = scsi.disk[scsi.id]->PlayAudioTrack(scsi.cmd);
	if (!status) {
		// s(G[)
		Error();
		return;
	}

	// CD-DACxgX^[g(26666~74+26716ŁAv1sec)
	if (cdda.GetTime() == 0) {
		cdda.SetTime(26666);
	}

	// Xe[^XtF[Y
	Status();
}

//---------------------------------------------------------------------------
//
//	fBXNč\z
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Construct()
{
	int hd;
	BOOL mo;
	BOOL cd;
	int i;
	int init;
	int index;
	Filepath path;

	ASSERT(this);
	ASSERT_DIAG();

	// UNA
	hd = 0;
	mo = FALSE;
	cd = FALSE;
	for (i=0; i<DeviceMax; i++) {
		scsi.disk[i] = NULL;
	}

	// SCSI݂ȂȂAfBXN폜ďI
	if (scsi.type == 0) {
		for (i=0; i<HDMax; i++) {
			if (scsi.hd[i]) {
				delete scsi.hd[i];
				scsi.hd[i] = NULL;
			}
			if (scsi.mo) {
				delete scsi.mo;
				scsi.mo = NULL;
			}
			if (scsi.cdrom) {
				delete scsi.cdrom;
				scsi.cdrom = NULL;
			}
		}
		return;
	}

	// fBXN
	switch (scsi.scsi_drives) {
		// 0
		case 0:
			break;

		// 1
		case 1:
			// MOD悩AHDD悩ŕ
			if (scsi.mo_first) {
				mo = TRUE;
			}
			else {
				hd = 1;
			}
			break;

		// 2
		case 2:
			// HD,MOƂ1
			hd = 1;
			mo = TRUE;
			break;

		// 3
		case 3:
			// HD,MO,CDƂ1
			hd = 1;
			mo = TRUE;
			cd = TRUE;

		// 4ȏ
		default:
			ASSERT(scsi.scsi_drives <= 7);
			hd = scsi.scsi_drives - 2;
			mo = TRUE;
			cd = TRUE;
			break;
	}

	// HD쐬
	for (i=0; i<HDMax; i++) {
		// 𒴂Ă΁A폜
		if (i >= hd) {
			if (scsi.hd[i]) {
				delete scsi.hd[i];
				scsi.hd[i] = NULL;
			}
			continue;
		}

		// fBXNʂmF
		if (scsi.hd[i]) {
			if (scsi.hd[i]->GetID() != MAKEID('S', 'C', 'H', 'D')) {
				delete scsi.hd[i];
				scsi.hd[i] = NULL;
			}
		}

		// fBXN
		if (scsi.hd[i]) {
			// pX擾Avok
			scsi.hd[i]->GetPath(path);
			if (path.CmpPath(scsihd[i])) {
				// pXvĂ
				continue;
			}

			// pXႤ̂ŁAfBXN폜
			delete scsi.hd[i];
			scsi.hd[i] = NULL;
		}

		// SCSIn[hfBXN쐬ăI[v݂
		scsi.hd[i] = new SCSIHD(this);
		if (!scsi.hd[i]->Open(scsihd[i])) {
			// I[vsB̔ԍscsi.diskNULLƂ
			delete scsi.hd[i];
			scsi.hd[i] = NULL;
		}
	}

	// MO쐬
	if (mo) {
		// MO
		if (scsi.mo) {
			if (scsi.mo->GetID() != MAKEID('S', 'C', 'M', 'O')) {
				delete scsi.mo;
				scsi.mo = NULL;
			}
		}
		if (!scsi.mo) {
			scsi.mo = new SCSIMO(this);
		}
	}
	else {
		// MOȂ
		if (scsi.mo) {
			delete scsi.mo;
			scsi.mo = NULL;
		}
	}

	// CD-ROM쐬
	if (cd) {
		// CD-ROM
		if (scsi.cdrom) {
			if (scsi.cdrom->GetID() != MAKEID('S', 'C', 'C', 'D')) {
				delete scsi.cdrom;
				scsi.cdrom = NULL;
			}
		}
		if (!scsi.cdrom) {
			scsi.cdrom = new SCSICD(this);
		}
	}
	else {
		// CD-ROMȂ
		if (scsi.cdrom) {
			delete scsi.cdrom;
			scsi.cdrom = NULL;
		}
	}

	// zXgEMODlāAfBXNׂ
	init = GetSCSIID();
	index = 0;
	for (i=0; i<DeviceMax; i++) {
		// zXg(CjVG[^)̓XLbv
		if (i == init) {
			continue;
		}

		// MOD̏ꍇAMOׂ
		if (scsi.mo_first) {
			if (mo) {
				ASSERT(scsi.mo);
				scsi.disk[i] = scsi.mo;
				mo = FALSE;
				continue;
			}
		}

		// HDׂ
		if (index < hd) {
			// NULLł悢
			scsi.disk[i] = scsi.hd[index];
			index++;
			continue;
		}

		// MOׂ
		if (mo) {
			ASSERT(scsi.mo);
			scsi.disk[i] = scsi.mo;
			mo = FALSE;
			continue;
		}
	}

	// CDID6ŒBID6tȂID7
	if (cd) {
		ASSERT(scsi.cdrom);
		if (!scsi.disk[6]) {
			scsi.disk[6] = scsi.cdrom;
		}
		else {
			ASSERT(!scsi.disk[7]);
			scsi.disk[7] = scsi.cdrom;
		}
	}

#if defined(SCSI_LOG)
	for (i=0; i<DeviceMax; i++) {
		if (!scsi.disk[i]) {
			LOG1(Log::Detail, "ID=%d : ڑ܂̓CjVG[^", i);
			continue;
		}
		switch (scsi.disk[i]->GetID()) {
			case MAKEID('S', 'C', 'H', 'D'):
				LOG1(Log::Detail, "ID=%d : SCSI n[hfBXN", i);
				break;
			case MAKEID('S', 'C', 'M', 'O'):
				LOG1(Log::Detail, "ID=%d : SCSI MOfBXN", i);
				break;
			case MAKEID('S', 'C', 'C', 'D'):
				LOG1(Log::Detail, "ID=%d : SCSI CD-ROMhCu", i);
				break;
			default:
				LOG2(Log::Detail, "ID=%d : `(%08X)", i, scsi.disk[i]->GetID());
				break;
		}
	}
#endif	//	SCSI_LOG
}

//---------------------------------------------------------------------------
//
//	MO/CD I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::Open(const Filepath& path, BOOL mo)
{
	ASSERT(this);
	ASSERT_DIAG();

	// LłȂ΃G[
	if (!IsValid(mo)) {
		return FALSE;
	}

	// CWFNg
	Eject(FALSE, mo);

	// mbgfBłȂ΃G[
	if (IsReady(mo)) {
		return FALSE;
	}

	// hCuɔC
	if (mo) {
		ASSERT(scsi.mo);
		if (scsi.mo->Open(path, TRUE)) {
			// (MO)
			return TRUE;
		}
	}
	else {
		ASSERT(scsi.cdrom);
		if (scsi.cdrom->Open(path, TRUE)) {
			// (CD)
			return TRUE;
		}
	}

	// s
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MO/CD CWFNg
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::Eject(BOOL force, BOOL mo)
{
	ASSERT(this);
	ASSERT_DIAG();

	// fBłȂΉȂ
	if (!IsReady(mo)) {
		return;
	}

	// hCuɒʒm
	if (mo) {
		ASSERT(scsi.mo);
		scsi.mo->Eject(force);
	}
	else {
		ASSERT(scsi.cdrom);
		scsi.cdrom->Eject(force);
	}
}

//---------------------------------------------------------------------------
//
//	MO ݋֎~
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::WriteP(BOOL flag)
{
	ASSERT(this);
	ASSERT_DIAG();

	// fBłȂΉȂ
	if (!IsReady()) {
		return;
	}

	// CgveNgݒ݂
	ASSERT(scsi.mo);
	scsi.mo->WriteP(flag);
}

//---------------------------------------------------------------------------
//
//	MO ݋֎~擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsWriteP() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// fBłȂ΃CgveNgłȂ
	if (!IsReady()) {
		return FALSE;
	}

	// hCuɕ
	ASSERT(scsi.mo);
	return scsi.mo->IsWriteP();
}

//---------------------------------------------------------------------------
//
//	MO ReadOnly`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsReadOnly() const
{
	ASSERT(this);
	ASSERT_DIAG();

	// fBłȂ΃[hI[łȂ
	if (!IsReady()) {
		return FALSE;
	}

	// hCuɕ
	ASSERT(scsi.mo);
	return scsi.mo->IsReadOnly();
}

//---------------------------------------------------------------------------
//
//	MO/CD Locked`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsLocked(BOOL mo) const
{
	ASSERT(this);
	ASSERT_DIAG();

	if (mo) {
		// hCu݂邩(MO)
		if (!scsi.mo) {
			return FALSE;
		}

		// hCuɕ(MO)
		return scsi.mo->IsLocked();
	}
	else {
		// hCu݂邩(CD)
		if (!scsi.cdrom) {
			return FALSE;
		}

		// hCuɕ(CD)
		return scsi.cdrom->IsLocked();
	}
}

//---------------------------------------------------------------------------
//
//	MO/CD Ready`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsReady(BOOL mo) const
{
	ASSERT(this);
	ASSERT_DIAG();

	if (mo) {
		// hCu݂邩(MO)
		if (!scsi.mo) {
			return FALSE;
		}

		// hCuɕ(MO)
		return scsi.mo->IsReady();
	}
	else {
		// hCu݂邩(CD)
		if (!scsi.cdrom) {
			return FALSE;
		}

		// hCuɕ(CD)
		return scsi.cdrom->IsReady();
	}
}

//---------------------------------------------------------------------------
//
//	MO/CD L`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSI::IsValid(BOOL mo) const
{
	ASSERT(this);
	ASSERT_DIAG();

	if (mo) {
		// |C^Ŕ(MO)
		if (scsi.mo) {
			return TRUE;
		}
	}
	else {
		// |C^Ŕ(CD)
		if (scsi.cdrom) {
			return TRUE;
		}
	}

	// hCuȂ
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MO/CD pX擾
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::GetPath(Filepath& path, BOOL mo) const
{
	ASSERT(this);
	ASSERT_DIAG();

	if (mo) {
		// MO
		if (scsi.mo) {
			// fB
			if (scsi.mo->IsReady()) {
				// pX擾
				scsi.mo->GetPath(path);
				return;
			}
		}
	}
	else {
		// CD
		if (scsi.cdrom) {
			// fB
			if (scsi.cdrom->IsReady()) {
				// pX擾
				scsi.cdrom->GetPath(path);
				return;
			}
		}
	}

	// LȃpXłȂ̂ŁANA
	path.Clear();
}

//---------------------------------------------------------------------------
//
//	CD-DAobt@擾
//
//---------------------------------------------------------------------------
void FASTCALL SCSI::GetBuf(DWORD *buffer, int samples, DWORD rate)
{
	ASSERT(this);
	ASSERT(buffer);
	ASSERT(samples >= 0);
	ASSERT(rate != 0);
	ASSERT((rate % 100) == 0);
	ASSERT_DIAG();

	// C^tF[XL
	if (scsi.type != 0) {
		// CD-ROM݂ꍇɌ
		if (scsi.cdrom) {
			// CD-ROMv
			scsi.cdrom->GetBuf(buffer, samples, rate);
		}
	}
}
