//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ X^eBbNRAM ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "log.h"
#include "filepath.h"
#include "fileio.h"
#include "cpu.h"
#include "schedule.h"
#include "memory.h"
#include "config.h"
#include "sram.h"

//===========================================================================
//
//	X^eBbNRAM
//
//===========================================================================
//#define SRAM_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SRAM::SRAM(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('S', 'R', 'A', 'M');
	dev.desc = "Static RAM";

	// JnAhXAIAhX
	memdev.first = 0xed0000;
	memdev.last = 0xedffff;

	// ̑
	sram_size = 16;
	write_en = FALSE;
	mem_sync = TRUE;
	changed = FALSE;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL SRAM::Init()
{
	Fileio fio;
	int i;
	BYTE data;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// 
	memset(sram, 0xff, sizeof(sram));

	// pX쐬Aǂݍ
	sram_path.SysFile(Filepath::SRAM);
	fio.Load(sram_path, sram, sizeof(sram));

	// GfBA]
	for (i=0; i<sizeof(sram); i+=2) {
		data = sram[i];
		sram[i] = sram[i + 1];
		sram[i + 1] = data;
	}

	// ύXȂ
	ASSERT(!changed);

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::Cleanup()
{
	Fileio fio;
	int i;
	BYTE data;

	ASSERT(this);

	// ύX
	if (changed) {
		// GfBA]
		for (i=0; i<sizeof(sram); i+=2) {
			data = sram[i];
			sram[i] = sram[i + 1];
			sram[i + 1] = data;
		}

		// 
		fio.Save(sram_path, sram, sram_size << 10);
	}

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::Reset()
{
	ASSERT(this);
	ASSERT_DIAG();

	LOG0(Log::Normal, "Zbg");

	// ݋֎~ɏ
	write_en = FALSE;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL SRAM::Save(Fileio *fio, int /*ver*/)
{
	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// SRAMTCY
	if (!fio->Write(&sram_size, sizeof(sram_size))) {
		return FALSE;
	}

	// SRAM{(64KB܂Ƃ߂)
	if (!fio->Write(&sram, sizeof(sram))) {
		return FALSE;
	}

	// ݋tO
	if (!fio->Write(&write_en, sizeof(write_en))) {
		return FALSE;
	}

	// tO
	if (!fio->Write(&mem_sync, sizeof(mem_sync))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SRAM::Load(Fileio *fio, int /*ver*/)
{
	BYTE *buf;

	ASSERT(this);
	ASSERT(fio);
	ASSERT_DIAG();

	LOG0(Log::Normal, "[h");

	// obt@m
	try {
		buf = new BYTE[sizeof(sram)];
	}
	catch (...) {
		buf = NULL;
	}
	if (!buf) {
		return FALSE;
	}

	// SRAMTCY
	if (!fio->Read(&sram_size, sizeof(sram_size))) {
		delete[] buf;
		return FALSE;
	}

	// SRAM{(64KB܂Ƃ߂)
	if (!fio->Read(buf, sizeof(sram))) {
		delete[] buf;
		return FALSE;
	}

	// rƓ]
	if (memcmp(sram, buf, sizeof(sram)) != 0) {
		memcpy(sram, buf, sizeof(sram));
		changed = TRUE;
	}

	// Ƀobt@
	delete[] buf;
	buf = NULL;

	// ݋tO
	if (!fio->Read(&write_en, sizeof(write_en))) {
		return FALSE;
	}

	// tO
	if (!fio->Read(&mem_sync, sizeof(mem_sync))) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);
	ASSERT_DIAG();

	LOG0(Log::Normal, "ݒKp");

	// SRAMTCY
	if (config->sram_64k) {
		sram_size = 64;
#if defined(SRAM_LOG)
		LOG0(Log::Detail, "TCY 64KB");
#endif	// SRAM_LOG
	}
	else {
		sram_size = 16;
	}

	// XCb`
	mem_sync = config->ram_sramsync;
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::AssertDiag() const
{
	// {NX
	MemDevice::AssertDiag();

	ASSERT(this);
	ASSERT(GetID() == MAKEID('S', 'R', 'A', 'M'));
	ASSERT(memdev.first == 0xed0000);
	ASSERT(memdev.last == 0xedffff);
	ASSERT((sram_size == 16) || (sram_size == 32) || (sram_size == 48) || (sram_size == 64));
	ASSERT((write_en == TRUE) || (write_en == FALSE));
	ASSERT((mem_sync == TRUE) || (mem_sync == FALSE));
	ASSERT((changed == TRUE) || (changed == FALSE));
}
#endif	// NDEBUG

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SRAM::ReadByte(DWORD addr)
{
	DWORD size;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// ItZbgZo
	addr -= memdev.first;
	size = (DWORD)(sram_size << 10);

	// `FbN
	if (size <= addr) {
		// oXG[
		cpu->BusErr(memdev.first + addr, TRUE);
		return 0xff;
	}

	// EFCg
	scheduler->Wait(1);

	// ǂݍ(GfBA𔽓])
	return (DWORD)sram[addr ^ 1];
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SRAM::ReadWord(DWORD addr)
{
	DWORD size;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT_DIAG();

	// ItZbgZo
	addr -= memdev.first;
	size = (DWORD)(sram_size << 10);

	// `FbN
	if (size <= addr) {
		// oXG[
		cpu->BusErr(memdev.first + addr, TRUE);
		return 0xff;
	}

	// EFCg
	scheduler->Wait(1);

	// ǂݍ(GfBAɒ)
	return (DWORD)(*(WORD *)&sram[addr]);
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::WriteByte(DWORD addr, DWORD data)
{
	DWORD size;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	// ItZbgZo
	addr -= memdev.first;
	size = (DWORD)(sram_size << 10);

	// `FbN
	if (size <= addr) {
		// oXG[
		cpu->BusErr(memdev.first + addr, FALSE);
		return;
	}

	// ݉\`FbN
	if (!write_en) {
		LOG1(Log::Warning, "݋֎~ $%06X", memdev.first + addr);
		return;
	}

	// EFCg
	scheduler->Wait(1);

	// AhX$09$00 or $10܂ꍇAXCb`XVł΃XLbv
	// (ZbgMemory::Reset珑܂Ă邽߁A㏑ɂjh)
	if ((addr == 0x09) && (data == 0x10)) {
		if (cpu->GetPC() == 0xff03a8) {
			if (mem_sync) {
				LOG2(Log::Warning, "XCb`ύX} $%06X <- $%02X", memdev.first + addr, data);
				return;
			}
		}
	}

	// (GfBA𔽓])
	if (addr < 0x100) {
		LOG2(Log::Detail, "XCb`ύX $%06X <- $%02X", memdev.first + addr, data);
	}
	if (sram[addr ^ 1] != (BYTE)data) {
		sram[addr ^ 1] = (BYTE)data;
		changed = TRUE;
	}
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::WriteWord(DWORD addr, DWORD data)
{
	DWORD size;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);
	ASSERT_DIAG();

	// ItZbgZo
	addr -= memdev.first;
	size = (DWORD)(sram_size << 10);

	// `FbN
	if (size <= addr) {
		// oXG[
		cpu->BusErr(memdev.first + addr, FALSE);
		return;
	}

	// ݉\`FbN
	if (!write_en) {
		LOG1(Log::Warning, "݋֎~ $%06X", memdev.first + addr);
		return;
	}

	// EFCg
	scheduler->Wait(1);

	// (GfBAɒ)
	if (addr < 0x100) {
		LOG2(Log::Detail, "XCb`ύX $%06X <- $%04X", memdev.first + addr, data);
	}
	if (*(WORD *)&sram[addr] != (WORD)data) {
		*(WORD *)&sram[addr] = (WORD)data;
		changed = TRUE;
	}
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL SRAM::ReadOnly(DWORD addr) const
{
	DWORD size;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT_DIAG();

	// ItZbgZo
	addr -= memdev.first;
	size = (DWORD)(sram_size << 10);

	// `FbN
	if (size <= addr) {
		return 0xff;
	}

	// ǂݍ(GfBA𔽓])
	return (DWORD)sram[addr ^ 1];
}

//---------------------------------------------------------------------------
//
//	SRAMAhX擾
//
//---------------------------------------------------------------------------
const BYTE* FASTCALL SRAM::GetSRAM() const
{
	ASSERT(this);
	ASSERT_DIAG();

	return sram;
}

//---------------------------------------------------------------------------
//
//	SRAMTCY擾
//
//---------------------------------------------------------------------------
int FASTCALL SRAM::GetSize() const
{
	ASSERT(this);
	ASSERT_DIAG();

	return sram_size;
}

//---------------------------------------------------------------------------
//
//	݋
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::WriteEnable(BOOL enable)
{
	ASSERT(this);
	ASSERT_DIAG();

	write_en = enable;

	if (write_en) {
		LOG0(Log::Detail, "SRAM݋");
	}
	else {
		LOG0(Log::Detail, "SRAM݋֎~");
	}
}

//---------------------------------------------------------------------------
//
//	XCb`ݒ
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::SetMemSw(DWORD offset, DWORD data)
{
	ASSERT(this);
	ASSERT(offset < 0x100);
	ASSERT(data < 0x100);
	ASSERT_DIAG();

	LOG2(Log::Detail, "XCb`ݒ $%06X <- $%02X", memdev.first + offset, data);
	if (sram[offset ^ 1] != (BYTE)data) {
		sram[offset ^ 1] = (BYTE)data;
		changed = TRUE;
	}
}

//---------------------------------------------------------------------------
//
//	XCb`擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL SRAM::GetMemSw(DWORD offset) const
{
	ASSERT(this);
	ASSERT(offset < 0x100);
	ASSERT_DIAG();

	return (DWORD)sram[offset ^ 1];
}

//---------------------------------------------------------------------------
//
//	NJE^XV
//
//---------------------------------------------------------------------------
void FASTCALL SRAM::UpdateBoot()
{
	WORD *ptr;

	ASSERT(this);
	ASSERT_DIAG();

	// ɕύX
	changed = TRUE;

	// |C^ݒ($ED0044)
	ptr = (WORD *)&sram[0x0044];

	// CNg(Low)
	if (ptr[1] != 0xffff) {
		ptr[1] = (WORD)(ptr[1] + 1);
		return;
	}

	// CNg(High)
	ptr[1] = 0x0000;
	ptr[0] = WORD(ptr[0] + 1);
}
