/* File: keyboard.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Keyboard Subroutines

	Extend the EmuZ-2000 mz2000Key.c by FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.

	Caution !!
	This file is including the old EmuZ-2000 program code.
	These are no copyright information and released open source code.
*/

#include "config.h"

#include <windows.h>
#include <string.h>

#include "common.h"
#include "mz2000.h"

static MZ2000 *mz2000_keyboard;
static volatile u8 mzkeybuf[16] = { 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
			0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff };

typedef struct _mzkeys {
	int	winkey;
	BYTE	port;
	BYTE	data;
	WORD	dmy;
} MZKEYS, *PMZKEYS;

static MZKEYS mzkeys[] = {
	0x00, 0x0F, 0xFF, 0x00, // none 00
	0x00, 0x0F, 0xFF, 0x00, // none 01
	0x00, 0x0F, 0xFF, 0x00, // none 02
	0x00, 0x0F, 0xFF, 0x00, // none 03
	0x00, 0x0F, 0xFF, 0x00, // none 04
	0x00, 0x0F, 0xFF, 0x00, // none 05
	0x00, 0x0F, 0xFF, 0x00, // none 06
	0x00, 0x0F, 0xFF, 0x00, // none 07
	0x08, 0x03, 0x7F, 0x00, // Control 'BREAK' (AT Backspace)
	0x09, 0x0b, 0xFE, 0x00, // Control 'GRAPH' (AT Tab)
	0x00, 0x0F, 0xFF, 0x00, // none 0A
	0x00, 0x0F, 0xFF, 0x00, // none 0B
	0x00, 0x0F, 0xFF, 0x00, // none 0C
	0x0D, 0x03, 0xFB, 0x00, // 'CR'
	0x00, 0x0F, 0xFF, 0x00, // none 0E
	0x00, 0x0F, 0xFF, 0x00, // none 0F

	0x10, 0x0b, 0xFB, 0x00, // Control 'SHIFT'
	0x11, 0x0b, 0xFD, 0x00, // Control 'SFT LOCK' (AT Control)
	0x00, 0x0F, 0xFF, 0x00, // none 12
	0x13, 0x03, 0x7F, 0x00, // Control 'BREAK' (AT Break)
	0x14, 0x0b, 0xFD, 0x00, // Control 'SFT LOCK' (AT CapsLock)
	0x00, 0x0F, 0xFF, 0x00, // none 15
	0x00, 0x0F, 0xFF, 0x00, // none 16
	0x00, 0x0F, 0xFF, 0x00, // none 17
	0x00, 0x0F, 0xFF, 0x00, // none 18
	0x00, 0x0F, 0xFF, 0x00, // none 19 (AT KANJI/ALT+Hankaku/Zenkaku)
	0x00, 0x0F, 0xFF, 0x00, // none 1A
	0x1B, 0x0b, 0xF7, 0x00, // Control 'KANA' (AT Esc)
	0x1C, 0x03, 0xFD, 0x00, // 'SPACE' (AT Henkan)
	0x1D, 0x03, 0xFE, 0x00, // Control 'TAB' (AT Muhenkan)
	0x00, 0x0F, 0xFF, 0x00, // none 1E
	0x00, 0x0F, 0xFF, 0x00, // none 1F

	0x20, 0x03, 0xFD, 0x00, // 'SPACE'
	0x21, 0x03, 0xF7, 0x00, // Cursor 'UP' (AT PgUp)
	0x22, 0x03, 0xEF, 0x00, // Cursor 'DOWN' (AT PgDown)
	0x23, 0x03, 0xFE, 0x00, // Control 'TAB' (AT End)
	0x24, 0x0a, 0xFB, 0x00, // Control 'CLR/HOME' (AT Home)
	0x25, 0x03, 0xDF, 0x00, // Cursor 'LEFT'
	0x26, 0x03, 0xF7, 0x00, // Cursor 'UP'
	0x27, 0x03, 0xBF, 0x00, // Cursor 'RIGHT'
	0x28, 0x03, 0xEF, 0x00, // Cursor 'DOWN'
	0x00, 0x0F, 0xFF, 0x00, // none 29
	0x00, 0x0F, 0xFF, 0x00, // none 2A
	0x00, 0x0F, 0xFF, 0x00, // none 2B
	0x00, 0x0F, 0xFF, 0x00, // none 2C (AT Print Screen)
	0x2D, 0x0a, 0xF7, 0x00, // Control 'INST/DEL' (AT Insert)
	0x2E, 0x0a, 0xF7, 0x00, // Control 'INST/DEL' (AT Delete)
	0x00, 0x0F, 0xFF, 0x00, // none 2F

	0x30, 0x08, 0xFE, 0x00, // '0'
	0x31, 0x08, 0xFD, 0x00, // '1'
	0x32, 0x08, 0xFB, 0x00, // '2'
	0x33, 0x08, 0xF7, 0x00, // '3'
	0x34, 0x08, 0xEF, 0x00, // '4'
	0x35, 0x08, 0xDF, 0x00, // '5'
	0x36, 0x08, 0xBF, 0x00, // '6'
	0x37, 0x08, 0x7F, 0x00, // '7'
	0x38, 0x09, 0xFE, 0x00, // '8'
	0x39, 0x09, 0xFD, 0x00, // '9'
	0x00, 0x0F, 0xFF, 0x00, // none 3A
	0x00, 0x0F, 0xFF, 0x00, // none 3B
	0x00, 0x0F, 0xFF, 0x00, // none 3C
	0x00, 0x0F, 0xFF, 0x00, // none 3D
	0x00, 0x0F, 0xFF, 0x00, // none 3E
	0x00, 0x0F, 0xFF, 0x00, // none 3F

	0x00, 0x0F, 0xFF, 0x00, // none 40
	0x41, 0x04, 0xFD, 0x00, // 'A'
	0x42, 0x04, 0xFB, 0x00, // 'B'
	0x43, 0x04, 0xF7, 0x00, // 'C'
	0x44, 0x04, 0xEF, 0x00, // 'D'
	0x45, 0x04, 0xDF, 0x00, // 'E'
	0x46, 0x04, 0xBF, 0x00, // 'F'
	0x47, 0x04, 0x7F, 0x00, // 'G'
	0x48, 0x05, 0xFE, 0x00, // 'H'
	0x49, 0x05, 0xFD, 0x00, // 'I'
	0x4A, 0x05, 0xFB, 0x00, // 'J'
	0x4B, 0x05, 0xF7, 0x00, // 'K'
	0x4C, 0x05, 0xEF, 0x00, // 'L'
	0x4D, 0x05, 0xDF, 0x00, // 'M'
	0x4E, 0x05, 0xBF, 0x00, // 'N'
	0x4F, 0x05, 0x7F, 0x00, // 'O'

	0x50, 0x06, 0xFE, 0x00, // 'P'
	0x51, 0x06, 0xFD, 0x00, // 'Q'
	0x52, 0x06, 0xFB, 0x00, // 'R'
	0x53, 0x06, 0xF7, 0x00, // 'S'
	0x54, 0x06, 0xEF, 0x00, // 'T'
	0x55, 0x06, 0xDF, 0x00, // 'U'
	0x56, 0x06, 0xBF, 0x00, // 'V'
	0x57, 0x06, 0x7F, 0x00, // 'W'
	0x58, 0x07, 0xFE, 0x00, // 'X'
	0x59, 0x07, 0xFD, 0x00, // 'Y'
	0x5A, 0x07, 0xFB, 0x00, // 'Z'
	0x00, 0x0F, 0xFF, 0x00, // none 5B
	0x00, 0x0F, 0xFF, 0x00, // none 5C
	0x00, 0x0F, 0xFF, 0x00, // none 5D
	0x00, 0x0F, 0xFF, 0x00, // none 5E
	0x00, 0x0F, 0xFF, 0x00, // none 5F

	0x60, 0x02, 0xFE, 0x00, // Ten Key '0'
	0x61, 0x02, 0xFD, 0x00, // Ten Key '1'
	0x62, 0x02, 0xFB, 0x00, // Ten Key '2'
	0x63, 0x02, 0xF7, 0x00, // Ten Key '3'
	0x64, 0x02, 0xEF, 0x00, // Ten Key '4'
	0x65, 0x02, 0xDF, 0x00, // Ten Key '5'
	0x66, 0x02, 0xBF, 0x00, // Ten Key '6'
	0x67, 0x02, 0x7F, 0x00, // Ten Key '7'
	0x68, 0x01, 0xFB, 0x00, // Ten Key '8'
	0x69, 0x01, 0xF7, 0x00, // Ten Key '9'
	0x6A, 0x01, 0xEF, 0x00, // Ten Key 00
	0x6B, 0x01, 0xBF, 0x00, // Ten Key '+'
	0x00, 0x0F, 0xFF, 0x00, // none 6C
	0x6D, 0x01, 0x7F, 0x00, // Ten Key '-'
	0x6E, 0x01, 0xDF, 0x00, // Ten Key '.'
	0x6F, 0x01, 0xEF, 0x00, // Ten Key 00

	0x70, 0x00, 0xFE, 0x00, // F1
	0x71, 0x00, 0xFD, 0x00, // F2
	0x72, 0x00, 0xFB, 0x00, // F3
	0x73, 0x00, 0xF7, 0x00, // F4
	0x74, 0x00, 0xEF, 0x00, // F5
	0x75, 0x00, 0xDF, 0x00, // F6
	0x76, 0x00, 0xBF, 0x00, // F7
	0x77, 0x00, 0x7F, 0x00, // F8
	0x78, 0x01, 0xFE, 0x00, // F9
	0x79, 0x01, 0xFD, 0x00, // F10
	0x7A, 0x01, 0xFD, 0x00, // F10 (AT F11)
	0x7B, 0x0b, 0xF7, 0x00, // Control 'KANA' (AT F12)
	0x00, 0x0F, 0xFF, 0x00, // none 7C
	0x00, 0x0F, 0xFF, 0x00, // none 7D
	0x00, 0x0F, 0xFF, 0x00, // none 7E
	0x00, 0x0F, 0xFF, 0x00, // none 7F

	0x00, 0x0F, 0xFF, 0x00, // none 80
	0x00, 0x0F, 0xFF, 0x00, // none 81
	0x00, 0x0F, 0xFF, 0x00, // none 82
	0x00, 0x0F, 0xFF, 0x00, // none 83
	0x00, 0x0F, 0xFF, 0x00, // none 84
	0x00, 0x0F, 0xFF, 0x00, // none 85
	0x00, 0x0F, 0xFF, 0x00, // none 86
	0x00, 0x0F, 0xFF, 0x00, // none 87
	0x00, 0x0F, 0xFF, 0x00, // none 88
	0x00, 0x0F, 0xFF, 0x00, // none 89
	0x00, 0x0F, 0xFF, 0x00, // none 8A
	0x00, 0x0F, 0xFF, 0x00, // none 8B
	0x00, 0x0F, 0xFF, 0x00, // none 8C
	0x00, 0x0F, 0xFF, 0x00, // none 8D
	0x00, 0x0F, 0xFF, 0x00, // none 8E
	0x00, 0x0F, 0xFF, 0x00, // none 8F

	0x00, 0x0F, 0xFF, 0x00, // none 90 (AT Num Lock)
	0x00, 0x0F, 0xFF, 0x00, // none 91 (AT Scroll Lock)
	0x00, 0x0F, 0xFF, 0x00, // none 92
	0x00, 0x0F, 0xFF, 0x00, // none 93
	0x00, 0x0F, 0xFF, 0x00, // none 94
	0x00, 0x0F, 0xFF, 0x00, // none 95
	0x00, 0x0F, 0xFF, 0x00, // none 96
	0x00, 0x0F, 0xFF, 0x00, // none 97
	0x00, 0x0F, 0xFF, 0x00, // none 98
	0x00, 0x0F, 0xFF, 0x00, // none 99
	0x00, 0x0F, 0xFF, 0x00, // none 9A
	0x00, 0x0F, 0xFF, 0x00, // none 9B
	0x00, 0x0F, 0xFF, 0x00, // none 9C
	0x00, 0x0F, 0xFF, 0x00, // none 9D
	0x00, 0x0F, 0xFF, 0x00, // none 9E
	0x00, 0x0F, 0xFF, 0x00, // none 9F

	0x00, 0x0F, 0xFF, 0x00, // none A0
	0x00, 0x0F, 0xFF, 0x00, // none A1
	0x00, 0x0F, 0xFF, 0x00, // none A2
	0x00, 0x0F, 0xFF, 0x00, // none A3
	0x00, 0x0F, 0xFF, 0x00, // none A4
	0x00, 0x0F, 0xFF, 0x00, // none A5
	0x00, 0x0F, 0xFF, 0x00, // none A6
	0x00, 0x0F, 0xFF, 0x00, // none A7
	0x00, 0x0F, 0xFF, 0x00, // none A8
	0x00, 0x0F, 0xFF, 0x00, // none A9
	0x00, 0x0F, 0xFF, 0x00, // none AA
	0x00, 0x0F, 0xFF, 0x00, // none AB
	0x00, 0x0F, 0xFF, 0x00, // none AC
	0x00, 0x0F, 0xFF, 0x00, // none AD
	0x00, 0x0F, 0xFF, 0x00, // none AE
	0x00, 0x0F, 0xFF, 0x00, // none AF

	0x00, 0x0F, 0xFF, 0x00, // none B0
	0x00, 0x0F, 0xFF, 0x00, // none B1
	0x00, 0x0F, 0xFF, 0x00, // none B2
	0x00, 0x0F, 0xFF, 0x00, // none B3
	0x00, 0x0F, 0xFF, 0x00, // none B4
	0x00, 0x0F, 0xFF, 0x00, // none B5
	0x00, 0x0F, 0xFF, 0x00, // none B6
	0x00, 0x0F, 0xFF, 0x00, // none B7
	0x00, 0x0F, 0xFF, 0x00, // none B8
	0x00, 0x0F, 0xFF, 0x00, // none B9
	0xBA, 0x09, 0xFB, 0x00, // ':
	0xBB, 0x09, 0xF7, 0x00, // ';
	0xBC, 0x07, 0x7F, 0x00, // ','
	0xBD, 0x09, 0xEF, 0x00, // '-
	0xBE, 0x07, 0xBF, 0x00, // '.'
	0xBF, 0x04, 0xFE, 0x00, // '/'

	0xC0, 0x09, 0xDF, 0x00, // '@
	0x00, 0x0F, 0xFF, 0x00, // none C1
	0x00, 0x0F, 0xFF, 0x00, // none C2
	0x00, 0x0F, 0xFF, 0x00, // none C3
	0x00, 0x0F, 0xFF, 0x00, // none C4
	0x00, 0x0F, 0xFF, 0x00, // none C5
	0x00, 0x0F, 0xFF, 0x00, // none C6
	0x00, 0x0F, 0xFF, 0x00, // none C7
	0x00, 0x0F, 0xFF, 0x00, // none C8
	0x00, 0x0F, 0xFF, 0x00, // none C9
	0x00, 0x0F, 0xFF, 0x00, // none CA
	0x00, 0x0F, 0xFF, 0x00, // none CB
	0x00, 0x0F, 0xFF, 0x00, // none CC
	0x00, 0x0F, 0xFF, 0x00, // none CD
	0x00, 0x0F, 0xFF, 0x00, // none CE
	0x00, 0x0F, 0xFF, 0x00, // none CF

	0x00, 0x0F, 0xFF, 0x00, // none D0
	0x00, 0x0F, 0xFF, 0x00, // none D1
	0x00, 0x0F, 0xFF, 0x00, // none D2
	0x00, 0x0F, 0xFF, 0x00, // none D3
	0x00, 0x0F, 0xFF, 0x00, // none D4
	0x00, 0x0F, 0xFF, 0x00, // none D5
	0x00, 0x0F, 0xFF, 0x00, // none D6
	0x00, 0x0F, 0xFF, 0x00, // none D7
	0x00, 0x0F, 0xFF, 0x00, // none D8
	0x00, 0x0F, 0xFF, 0x00, // none D9
	0x00, 0x0F, 0xFF, 0x00, // none DA
	0xDB, 0x09, 0xBF, 0x00, // '[
	0xDC, 0x07, 0xEF, 0x00, // '\'
	0xDD, 0x0a, 0xFE, 0x00, // ']
	0xDE, 0x07, 0xF7, 0x00, // '^'
	0x00, 0x0F, 0xFF, 0x00, // none DF

	0x00, 0x0F, 0xFF, 0x00, // none E0
	0x00, 0x0F, 0xFF, 0x00, // none E1
	0xE2, 0x07, 0xDF, 0x00, // '?'
	0x00, 0x0F, 0xFF, 0x00, // none E3
	0x00, 0x0F, 0xFF, 0x00, // none E4
	0x00, 0x0F, 0xFF, 0x00, // none E5
	0x00, 0x0F, 0xFF, 0x00, // none E6
	0x00, 0x0F, 0xFF, 0x00, // none E7
	0x00, 0x0F, 0xFF, 0x00, // none E8
	0x00, 0x0F, 0xFF, 0x00, // none E9
	0x00, 0x0F, 0xFF, 0x00, // none EA
	0x00, 0x0F, 0xFF, 0x00, // none EB
	0x00, 0x0F, 0xFF, 0x00, // none EC
	0x00, 0x0F, 0xFF, 0x00, // none ED
	0x00, 0x0F, 0xFF, 0x00, // none EE
	0x00, 0x0F, 0xFF, 0x00, // none EF

	0x00, 0x0F, 0xFF, 0x00, // none F0 (AT Eisu)
	0x00, 0x0F, 0xFF, 0x00, // none F1 (AT Hiragana Katakana)
	0x00, 0x0F, 0xFF, 0x00, // none F2 (AT Hiragana Katakana)
	0x00, 0x0F, 0xFF, 0x00, // none F3 (AT Hankaku/Zenkaku)
	0x00, 0x0F, 0xFF, 0x00, // none F4 (AT Hankaku/Zenkaku - 2)
	0x00, 0x0F, 0xFF, 0x00, // none F5 (AT Hiragana Katakana)
	0x00, 0x0F, 0xFF, 0x00, // none F6 (AT Hiragana Katakana)
	0x00, 0x0F, 0xFF, 0x00, // none F7
	0x00, 0x0F, 0xFF, 0x00, // none F8
	0x00, 0x0F, 0xFF, 0x00, // none F9
	0x00, 0x0F, 0xFF, 0x00, // none FA
	0x00, 0x0F, 0xFF, 0x00, // none FB
	0x00, 0x0F, 0xFF, 0x00, // none FC
	0x00, 0x0F, 0xFF, 0x00, // none FD
	0x00, 0x0F, 0xFF, 0x00, // none FE
	0x00, 0x0F, 0xFF, 0x00  // none FF
};

static MZKEYS mzkeyEx1[] = {
	0x25, 0x02, 0xEF, 0x00, // Ten Key '4'
	0x26, 0x01, 0xFB, 0x00, // Ten Key '8'
	0x27, 0x02, 0xBF, 0x00, // Ten Key '6'
	0x28, 0x02, 0xFB, 0x00  // Ten Key '2'
};

static MZKEYS mzkeyEx1Normal[] = {
	0x25, 0x03, 0xDF, 0x00, // Cursor 'LEFT'
	0x26, 0x03, 0xF7, 0x00, // Cursor 'UP'
	0x27, 0x03, 0xBF, 0x00, // Cursor 'RIGHT'
	0x28, 0x03, 0xEF, 0x00  // Cursor 'DOWN'
};

static MZKEYS mzkeyEx2[] = {
	0x25, 0x02, 0xFD, 0x00, // Ten Key '1'
	0x26, 0x02, 0xDF, 0x00, // Ten Key '5'
	0x27, 0x02, 0xF7, 0x00, // Ten Key '3'
	0x28, 0x01, 0xEF, 0x00  // Ten Key '00'
};

static MZKEYS mzkeyEx3[] = {
	0x60, 0x01, 0xEF, 0x00  // Ten Key '00'
};

static MZKEYS mzkeyEx3Normal[] = {
	0x60, 0x02, 0xFE, 0x00  // Ten Key '0'
};

static MZKEYS mzkeyEx4a[] = {
	0x58, 0x03, 0xFD, 0x00, // 'SPACE' (AT Henkan)
	0x59, 0x07, 0xFD, 0x00, // 'Y'
	0x5A, 0x03, 0xFE, 0x00, // Control 'TAB' (AT Muhenkan)
};

static MZKEYS mzkeyEx4b[] = {
	0x1C, 0x07, 0xFE, 0x00, // 'X'
	0x1D, 0x07, 0xFB, 0x00, // 'Z'
	0x00, 0x0F, 0xFF, 0x00, // none 1E
	0x00, 0x0F, 0xFF, 0x00, // none 1F
	0x1C, 0x07, 0xFE, 0x00, // 'X'
};

static MZKEYS mzkeyEx4aNormal[] = {
	0x58, 0x07, 0xFE, 0x00, // 'X'
	0x59, 0x07, 0xFD, 0x00, // 'Y'
	0x5A, 0x07, 0xFB, 0x00, // 'Z'
};

static MZKEYS mzkeyEx4bNormal[] = {
	0x1C, 0x03, 0xFD, 0x00, // 'SPACE' (AT Henkan)
	0x1D, 0x03, 0xFE, 0x00, // Control 'TAB' (AT Muhenkan)
	0x00, 0x0F, 0xFF, 0x00, // none 1E
	0x00, 0x0F, 0xFF, 0x00, // none 1F
	0x20, 0x03, 0xFD, 0x00, // 'SPACE'
};

static MZKEYS mzkeyEx5a[] = {
	0x09, 0x03, 0xFE, 0x00, // Control 'TAB' (AT Muhenkan)
};

static MZKEYS mzkeyEx5b[] = {
	0x1D, 0x0b, 0xFE, 0x00, // Control 'GRAPH' (AT Tab)
};

static MZKEYS mzkeyEx5aNormal[] = {
	0x09, 0x0b, 0xFE, 0x00, // Control 'GRAPH' (AT Tab)
};

static MZKEYS mzkeyEx5bNormal[] = {
	0x1D, 0x03, 0xFE, 0x00, // Control 'TAB' (AT Muhenkan)
};

static MZKEYS mzkeyEx6[] = {
	0x61, 0x02, 0xFE, 0x00, // Ten Key '0'
	0x62, 0x01, 0xEF, 0x00, // Ten Key 00
	0x63, 0x01, 0xDF, 0x00, // Ten Key '.'
};

static MZKEYS mzkeyEx6Normal[] = {
	0x61, 0x02, 0xFD, 0x00, // Ten Key '1'
	0x62, 0x02, 0xFB, 0x00, // Ten Key '2'
	0x63, 0x02, 0xF7, 0x00, // Ten Key '3'
};

static WPARAM convert_tenkeys( WPARAM wp )
{
	switch (wp) {
	case 0x2d :	/* 0/Ins */
		return 0x60;
	case 0x2e :	/* ./Del */
		return 0x6e;
	case 0x23 :	/* 1/End */
		return 0x61;
	case 0x28 :	/* 2/Down */
		return 0x62;
	case 0x22 :	/* 3/PgDn */
		return 0x63;
	case 0x25 :	/* 4/Left */
		return 0x64;
	case 0x0c :	/* 5 */
		return 0x65;
	case 0x27 :	/* 6/Right */
		return 0x66;
	case 0x24 :	/* 7/Home */
		return 0x67;
	case 0x26 :	/* 8/Up */
		return 0x68;
	case 0x21 :	/* 9/PgUp */
		return 0x69;
	default :
		break;
	}
	return wp;
}

static int keyboard_execevent( WPARAM wParam, LPARAM lParam )
{
	int keycode = wParam;
	BYTE bk_keydata;

	if (copypaste_runningpaste())
		return -1;

	//_TRACE( "key:%08x %08x\n", keycode, lParam );
	switch (lParam & 0x80000000) {
	case 0x00000000 :			/* Press */
		if (lParam & 0x20000000)	/* Alt */
			break;
		if (keycode >= 0xf0)		/* Toggle only keys */
			break;
		if (!(lParam & 0x01000000)) {
			/* Bit24: No enhanced 101 keys */
			keycode = convert_tenkeys( keycode );
		}
		bk_keydata = mzkeybuf[mzkeys[keycode].port];
		mzkeybuf[mzkeys[keycode].port] &= mzkeys[keycode].data;
		if (mz2000_keyboard -> keyint_select == mzkeys[keycode].port && mz2000_keyboard -> keyint_func) {
			/* keyboard interrupt */
			if (bk_keydata != mzkeybuf[mzkeys[keycode].port])
				mz2000_keyboard -> keyint_func( (u8)(bk_keydata ^ mzkeybuf[mzkeys[keycode].port]) );
		}
		break;
	case 0x80000000 :	/* Release */
		if (keycode > 0xff)
			break;
		if (!(lParam & 0x01000000)) {
			/* Bit24: No enhanced 101 keys */
			keycode = convert_tenkeys( keycode );
		}
		bk_keydata = mzkeybuf[mzkeys[keycode].port];
		mzkeybuf[mzkeys[keycode].port] |= ~mzkeys[keycode].data;
		break;
	default:
		break;
	}
	return 0;
}

static LRESULT CALLBACK KeyHookProc( int nCode, WPARAM wParam, LPARAM lParam )
{
	if (nCode < 0)
		return CallNextHookEx( mz2000_keyboard -> hKeyHook, nCode, wParam, lParam );
	if (wParam == 0x79) {
		/* F10 */
		keyboard_execevent( wParam, lParam );
	}
	return 0;
}

void keyboard_keyevent( WPARAM wParam, LPARAM lParam )
{
	if (wParam != 0x79) {
		/* not F10 */
		keyboard_execevent( wParam, lParam );
	}
	return;
}

void keyboard_setinterrupt( unsigned char select, void *intfunc )
{
	mz2000_keyboard -> keyint_select = 0xffU;
	mz2000_keyboard -> keyint_func = intfunc;
	mz2000_keyboard -> keyint_select = select;
	return;
}

void keyboard_setdirect( BOOL press_flag, int strobe, int bit )
{
	if (press_flag)
		mzkeybuf[strobe] &= ~(1U << bit);
	else
		mzkeybuf[strobe] |= (1U << bit);
	return;
}

int keyboard_getbitdata( MZ2000 *mz2000, int strobe )
{
	return mzkeybuf[strobe & 0x0f];
}

int keyboard_getallbitdata( MZ2000 *mz2000 )
{
	int i, bitdata = 0xff;

	for (i = 0; i < 12; i++)
		bitdata &= mzkeybuf[i];
	return bitdata;
}

BOOL keyboard_breakkey_released( MZ2000 *mz2000 )
{
	return mzkeybuf[3] & 0x80U ? TRUE : FALSE;
}

BOOL keyboard_releaseall( MZ2000 *mz2000 )
{
	int i;

	for (i = 0; i < 16; i++)
		mzkeybuf[i] = 0xff;
	return TRUE;
}

BOOL keyboard_setkeymode( unsigned char keymode )
{
	if (keymode & 0x01U) {
		/* cursor keys = 2468 */
		memcpy( &mzkeys[0x25], mzkeyEx1, sizeof(MZKEYS)*4 );
	} else if (keymode & 0x02U) {
		/* cursor keys = [00]135 */
		memcpy( &mzkeys[0x25], mzkeyEx2, sizeof(MZKEYS)*4 );
	} else {
		/* cursor keys = normal */
		memcpy( &mzkeys[0x25], mzkeyEx1Normal, sizeof(MZKEYS)*4 );
	}
	if (keymode & 0x04U) {
		/* 0 = [00] */
		memcpy( &mzkeys[0x60], mzkeyEx3, sizeof(MZKEYS) );
	} else
		memcpy( &mzkeys[0x60], mzkeyEx3Normal, sizeof(MZKEYS) );
	if (keymode & 0x08U) {
		keymode &= ~0x10U;
		/* ZX = TAB/SPACE */
		memcpy( &mzkeys[0x58], mzkeyEx4a, sizeof(MZKEYS)*3 );
		/* TAB/SPACE = ZX */
		memcpy( &mzkeys[0x1c], mzkeyEx4b, sizeof(MZKEYS)*5 );
	} else {
		/* ZX = ZX */
		memcpy( &mzkeys[0x58], mzkeyEx4aNormal, sizeof(MZKEYS)*3 );
		/* TAB/SPACE = TAB/SPACE */
		memcpy( &mzkeys[0x1c], mzkeyEx4bNormal, sizeof(MZKEYS)*5 );
	}
	if (keymode & 0x10U) {
		/* GRPH = TAB */
		memcpy( &mzkeys[0x09], mzkeyEx5a, sizeof(MZKEYS) );
		/* TAB = GRAPH */
		memcpy( &mzkeys[0x1d], mzkeyEx5b, sizeof(MZKEYS) );
	} else {
		/* GRPH = GRPH */
		memcpy( &mzkeys[0x09], mzkeyEx5aNormal, sizeof(MZKEYS) );
		if (!(keymode & 0x08U)) {
			/* TAB = TAB */
			memcpy( &mzkeys[0x1d], mzkeyEx5bNormal, sizeof(MZKEYS) );
		}
	}
	if (keymode & 0x20U) {
		/* 1 2 3 = 0 [00] . */
		memcpy( &mzkeys[0x61], mzkeyEx6, sizeof(MZKEYS)*3 );
	} else {
		/* 1 2 3 = normal */
		memcpy( &mzkeys[0x61], mzkeyEx6Normal, sizeof(MZKEYS)*3 );
	}
	return TRUE;
}

BOOL keyboard_hookstart( HWND hWnd, MZ2000 *mz )
{
	if (mz -> hKeyHook)
		keyboard_hookend( hWnd, mz );
	mz -> hKeyHook = SetWindowsHookEx( WH_KEYBOARD,
		(HOOKPROC)KeyHookProc,
		NULL, GetCurrentThreadId() );
	if (!(mz->hKeyHook))
		MessageBox(hWnd, "SetWindowsHooxEx() for Keyboard Error", "Keyboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
	return TRUE;
}

BOOL keyboard_hookend( HWND hWnd, MZ2000 *mz )
{
	if (!UnhookWindowsHookEx( mz->hKeyHook ))
		; // MessageBox( hWnd, "Keyboard Hook Release Error", "Keyboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
	return TRUE;
}

void keyboard_exit( MZ2000 *mz2000 )
{
	return;
}

BOOL keyboard_init( MZ2000 *mz2000 )
{
	mz2000_keyboard = mz2000;
	mz2000_keyboard -> keyint_select = 0xffU;
	mz2000_keyboard -> keyint_func = NULL;
	keyboard_releaseall( mz2000 );
	return TRUE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
