/* File: mz2io.c

	MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Hardware I/O Controls

	Combine the emz2000 mz2io.c, EmuZ-2000 emuz2000ioTrap.c,
		and extend by FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

/*
	8255 (Cassette Control, Display Control, Sound Control)
	$E0	... Pa: output
	$E1	... Pb: input
	$E2	... Pc: output
	$E3	... mode control

	8253 (timer) 31.25KHz Cascade Counters
	  - Implemented only mode 2
	$E4	... C0: mode 2
	$E5	... C1: mode 2
	$E6	... C2: mode 2
	$E7	... mode control
	($F0-$F3 ... CLR#)

	Z80 PIO (Keyboard, Memory mapping, CRTC)
	$E8	... A: output mode 3
	$E9	... mode control A
	$EA	... B: input mode 3
	$EB	... mode control B

	8253 (timer)
	$F0-$F3	... CLR# (write only)

	MZ-2000/2200 CRTC / Color Control (LZ90D01/LZ90D03)
	$F4	... background color
	$F5	... character data color and priority
	$F6	... graphic output page control and enable green display
	$F7	... graphic area change

	MZ-80B/B2 Graphic Page Control
	$F4(-$F7) ... graphic page control

	* Big different ports for MZ-80B
	  - $E0	... 8255 Pa: output
	  - $E8	... Z80 PIO A: output mode 3
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <memory.h>
#include <sys/time.h>
#else
#include <windows.h>
#endif

#include "common.h"
#include "mz2000.h"

static MZ2000 *mz2000_for_io;

#undef TIMER8253_ENABLE_DEBUG

#ifndef TIMER8253_USE_INTERNALTIME
#ifndef WIN32
static z80clk GetTickCount_us( void )
{
	struct timeval tv;
	struct timezone tz;

	gettimeofday( &tv, &tz );
	return ((z80clk)tv.tv_sec) * 1000000 + ((z80clk)tv.tv_usec);
}
#else
static z80clk GetTickCount_us( void )
{
	return (z80clk)GetTickCount() * 1000;
}
#endif
#endif

void mz2io_keyintfunc( u8 KeyBitMask )
{
	/* _TRACE("int! %02x %02x\n", mz2000_for_io->bitMask, KeyBitMask ); */
	if (mz2000_for_io->bitMask & KeyBitMask)
		int_z80( (int)mz2000_for_io->intVect );
	return;
}

static u8 in_e0( u16 port )
{
	return mz2000_for_io->ioinfo->port_e0;
}

static int out_e0( u16 port, u8 value )
{
	//_TRACE("port e0=%02x\n", value);
	/* BUG FIRE screen reverse output to $80 and $d1 */
	/* bit0 REW# (cassette control) */
	if (!(value & 0x01U) && value != 0x80 && value != 0xd1)
		mz2tape_rew( mz2000_for_io );
	/* bit1 FF# (cassette control) */
	if (!(value & 0x02U) && value != 0x80 && value != 0xd1)
		mz2tape_ff( mz2000_for_io );
	/* bit2 PLAY# (cassette control) */
	if (!(value & 0x04U) && value != 0x80 && value != 0xd1) {
		if (mz2000_for_io->ioinfo->port_e2 & 0x40U)
			mz2tape_play( mz2000_for_io );
		else
			mz2tape_record( mz2000_for_io );
	}
	/* bit3 STOP# (cassette control) */
	if (!(value & 0x08U) && value != 0x80 && value != 0xd1)
		mz2tape_stop( mz2000_for_io );
	/* bit4 VID (display control) */
	if ((value & 0x10U) != (mz2000_for_io->ioinfo->port_e0 & 0x10U))
		screen_setreverse( mz2000_for_io, !(value & 0x10U) );
	/* bit5 AREW# (cassette control) */
	/* not supported */
	/* bit6 APLAY# (cassette control) */
	/* not supported */
	/* bit7 APSS-P# (cassette control) */
	/* ignore */
	mz2000_for_io->ioinfo->port_e0 = value;
	return 0;
}

static u8 in_e1(u16 port)
{
	u8 v;
	z80clk t;

	v = mz2000_for_io->ioinfo->port_e1 | ~0x40U;
	if (!keyboard_breakkey_released(mz2000_for_io))
		v &= ~0x80U;	/* bit7 PIO PB7 (break key) */
	if (!(mz2000_for_io -> blank) && (getspeed_z80() > 0)) {
		if (!mz2000_for_io -> vblankspecial) {
			/* Too slowly for MILKY WAY round 3,6,9 */
			t = get_internal_clock() % (get_frameclk_z80() * 16 / MZ2000_REFRESHRATE);
			if (t < (get_frameclk_z80() * 16 / MZ2000_REFRESHRATE) * 20/525)
				v &= ~0x01U;	/* bit0 Blank (display control) */
		} else {
			/* This is magic number for MILKY WAY round 3,6,9 */
			/* H timing is too shortly for JELDA II damage code (cause the hung up) */
			t = get_internal_clock() % 1000;
			if (t < 500)
				v &= ~0x01U;	/* bit0 Blank (display control) */
		}
	}
	mz2tape_updatestatus( mz2000_for_io, &v );
	mz2000_for_io->ioinfo->port_e1 = v;
	return v;
}

static int out_e1( u16 port, u8 value )
{
	return 0;
}

static u8 in_e2( u16 port )
{
	return mz2000_for_io->ioinfo->port_e2;
}

static int out_e2( u16 port, u8 value )
{
	//_TRACE("port e2=%02x\n", value);
	/* bit0 VGATE (display control) */
	if (mz2000_for_io->blank != ((value & 0x01U) ? TRUE : FALSE)) {
		int t;

		t = value & 0x01U ? TRUE : FALSE;
		mz2000_for_io -> blank = t;
		screen_setblank( mz2000_for_io, t );
	}
	/* bit1 NST (memory controller) */
	if (value & 0x02U) {
		value &= ~(0x02U | 0x08U);
		mz2000_for_io -> howipl = 0;
		mz2000_for_io -> needReset = TRUE;
		stop_z80();
		return 0;
	}
	/* bit2 SOUND (sound control) */
	sound_setstate( mz2000_for_io, value & 0x04U ? TRUE : FALSE );
	/* bit3 BST# (IPL reset sw) */
	if (!(value & 0x08U) && value) {	/* Milky Way sound output to $00 */
		value |= 0x08U;
		mz2000_for_io -> needIpl = TRUE;
		stop_z80();
	}
	/* bit4 OPEN# (cassette control) */
	if (!(value & 0x10U) && value)		/* Milky Way sound output to $00 */
		mz2tape_eject( mz2000_for_io, value ? TRUE : FALSE );
	if (!mz2000_for_io -> mzmode) {
		/* bit5 KINH (cassette control, inhibit user controls) */
		mz2tape_setkinh( value & 0x20U ? TRUE : FALSE );
	} else {
		/* bit5 FF,REW latch (cassette control) */
		if (!(mz2000_for_io->ioinfo->port_e0 & 0x02U))
			mz2tape_setrev( TRUE );		/* REW */
		else
			mz2tape_setrev( FALSE );	/* FF */
	}
	/* bit6 REC#,RD/WR (cassette control) */
	/* bit7 WDATA (cassette control) */
	mz2tape_updatewrite( mz2000_for_io, value & 0x80U ? TRUE : FALSE );
	mz2000_for_io->ioinfo->port_e2 = value;
	return 0;
}

static u8 in_e3( u16 port )
{
	return 0xffU;
}

static int out_e3( u16 port, u8 value )
{
	if (!(value & 0x80)) {
		u8 c, e2data = 0x01U << ((value & 0x0eU) >> 1);

		c = mz2000_for_io->ioinfo->port_e2;
		if (!(value & 0x01U))
			c &= ~e2data;
		else
			c |= e2data;
		out_e2( 0xe2U, c );
	} else {
		/* Tonky e3 <- 0x82, but other boot */
		if (value != 0x82U) {
			mz2000_for_io->needIpl = TRUE;
			stop_z80();
		}
	}
	mz2000_for_io -> cpu_peripherals_start |= CPUPERIPHERALS_INIT_8255;
	return 0;
}

static u8 in_e456common( u16 port, int sc )
{
	u16 t;
	z80clk now_time;
	MZ2000 *mz = mz2000_for_io;

	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_8253)) {
		/* Not Initialized 8253 */
		return 0xffU;
	}
	if (sc < 0 || sc >= 3)
		return 0xffU;
#ifndef TIMER8253_USE_INTERNALTIME
	mz -> timer8253_currenttime[sc] = GetTickCount_us();
	now_time = (mz->timer8253_currenttime[sc] - mz->timer8253_starttime[sc]) / (1000 * 1000 / 31250);
#else
	mz -> timer8253_currenttime[sc] = get_internal_clock();
	now_time = (mz->timer8253_currenttime[sc] - mz->timer8253_starttime[sc]) / ((getspeed_z80() > 0 ? getspeed_z80() : DEFAULT_SPEED_KHZ) * 1000 / 31250);
#endif
	switch (sc) {
	case 0 :
		if (mz -> timer8253_count[sc])
			t = (u16)(now_time % mz -> timer8253_count[sc]);
		else
			t = (u16)now_time;
		break;
	case 1 :
		if (mz -> timer8253_count[0])
			now_time = now_time / mz -> timer8253_count[0];
		else
			now_time = now_time / 65536;
		if (mz -> timer8253_count[sc])
			t = (u16)(now_time % mz -> timer8253_count[sc]);
		else
			t = (u16)now_time;
		break;
	case 2 :
		if (mz -> timer8253_count[0])
			now_time = now_time / mz -> timer8253_count[0];
		else
			now_time = now_time / 65536;
		if (mz -> timer8253_count[1])
			now_time = now_time / mz -> timer8253_count[1];
		else
			now_time = now_time / 65536;
		if (mz -> timer8253_count[sc])
			t = (u16)(now_time % mz -> timer8253_count[sc]);
		else
			t = (u16)now_time;
		break;
	default :
		t = 0;
		break;
	}
	t = mz -> timer8253_count[sc] - t;
	switch (mz -> timer8253_rwmode[sc]) {
	case 2 :
#ifdef TIMER8253_ENABLE_DEBUG
		_TRACE("%02x/2=%02x\n", port, t & 0xffU);
#endif
		return t & 0xffU;
	case 3 :
#ifdef TIMER8253_ENABLE_DEBUG
		_TRACE("%02x/3=%02x\n", port, (t >> 8) & 0xffU);
#endif
		return (t >> 8) & 0xffU;
	default : 
		mz -> timer8253_rwmode[sc] = !mz -> timer8253_rwmode[sc];
		if (mz -> timer8253_rwmode[sc]) {
#ifdef TIMER8253_ENABLE_DEBUG
			_TRACE("%02x/0=%02x\n", port, t & 0xffU);
#endif
			return t & 0xffU;
		} else {
#ifdef TIMER8253_ENABLE_DEBUG
			_TRACE("%02x/1=%02x\n", port, (t >> 8) & 0xffU);
#endif
			return (t >> 8) & 0xffU;
		}
	}
}

static int out_e456common( u16 port, u8 value, int sc )
{
	u16 t;
	z80clk tt;
	MZ2000 *mz = mz2000_for_io;

	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_8253)) {
		/* Not Initialized 8253 */
		return 0;
	}
#ifndef TIMER8253_USE_INTERNALTIME
	tt = GetTickCount_us();
#else
	tt = get_internal_clock();
#endif
	mz -> timer8253_starttime[0] = tt;
	mz -> timer8253_starttime[1] = tt;
	mz -> timer8253_starttime[2] = tt;
	t = mz -> timer8253_count[sc];
	switch (mz -> timer8253_rwmode[sc]) {
	case 2 :
		t = value;
		break;
	case 3 :
		t = (value << 8) & 0xff00U;
		break;
	default : 
		mz->timer8253_rwmode[sc] = !mz->timer8253_rwmode[sc];
		if (mz->timer8253_rwmode[sc])
			t = (t & 0xff00U) | (value & 0xffU);
		else
			t = (t & 0xffU) | ((value << 8) & 0xff00U);
		break;
	}
	mz -> timer8253_count[sc] = t;
	return 0;
}

static u8 in_e4( u16 port )
{
	return in_e456common( port, 0 );
}

static int out_e4( u16 port, u8 value )
{
	return out_e456common( port, value, 0 );
}

static u8 in_e5( u16 port )
{
	return in_e456common( port, 1 );
}

static int out_e5( u16 port, u8 value )
{
	return out_e456common( port, value, 1 );
}

static u8 in_e6( u16 port )
{
	return in_e456common( port, 2 );
}

static int out_e6( u16 port, u8 value )
{
	return out_e456common( port, value, 2 );
}

static u8 in_e7( u16 port )
{
	return 0xffU;
}

static int out_e7( u16 port, u8 value )
{
	unsigned int sc;
	MZ2000 *mz = mz2000_for_io;

	/* SC1 SC0 RL1 RL0 M2 M1 M0 BCD */
	sc = ((value >> 6) & 0x03);
	if (sc >= 3) {
		mz2000_for_io -> cpu_peripherals_start |= CPUPERIPHERALS_INIT_8253;
		return 0;
	}
#ifdef TIMER8253_ENABLE_DEBUG
	_TRACE("%02x=%02x (sc=%d)\n", port, value, sc);
#endif
	switch ((value >> 4) & 0x03) {	/* RL */
	case 1 :	/* only low 8bit */
		mz->timer8253_rwmode[sc] = 2;
		break;
	case 2 :	/* only high 8bit */
		mz->timer8253_rwmode[sc] = 3;
		break;
	case 3 :	/* 16bit register low -> high */
		mz->timer8253_rwmode[sc] = 0;
		break;
	default : /*0*/	/* counter latch operation */
		//mz->timer8253_rwmode[sc] = 0;
		break;
	}
	mz2000_for_io -> cpu_peripherals_start |= CPUPERIPHERALS_INIT_8253;
	return 0;
}

static u8 in_e8( u16 port )
{
	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_Z80PIOA)) {
		/* Not Initialized Z80 PIO A (Keyboard, etc.) */
		return 0xffU;
	}
	return mz2000_for_io->ioinfo->port_e8;
}

static int out_e8( u16 port, u8 value )
{
	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_Z80PIOA)) {
		/* Not Initialized Z80 PIO A (Keyboard, etc.) */
		return 0;
	}
	/* bit0,1,2,3 keystrobe */
	/* bit4 keystrobe low */
	mz2000_for_io->ioinfo->strobe_low = !(value & 0x10U);
	/* bit6,7 gram/vram switching */
	if (value & 0x80U) {		/* Enable V-RAM or G-RAM */
		if (value & 0x40U)	/* Select Text V-RAM */
			vram_on( mz2000_for_io );
		else
			gram_on( mz2000_for_io );
	} else
		v_gram_off( mz2000_for_io );
	/* bit5 width 80 (L=40) */
	if ((mz2000_for_io -> ioinfo-> port_e8 & 0x20U) != (value & 0x20U)) {
		mz2000_for_io -> ioinfo -> port_e8 = value;
		screen_setchrsize( mz2000_for_io, value & 0x20U ? TRUE : FALSE );
		screen_region_add_all( mz2000_for_io );
	}
	mz2000_for_io -> ioinfo -> port_e8 = value;
	return 0;
}

static u8 in_e9( u16 port )
{
	return 0xffU;
}

static int out_e9( u16 port, u8 value )
{
	mz2000_for_io -> cpu_peripherals_start |= CPUPERIPHERALS_INIT_Z80PIOA;
	return 0;
}

static u8 in_ea( u16 port )
{
	u8 code;

	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_Z80PIOB)) {
		/* Not Initialized Z80 PIO B (Keyboard) */
		return 0xffU;
	}
	if (mz2000_for_io->ioinfo->strobe_low)
		code = keyboard_getallbitdata( mz2000_for_io );
	else
		code = keyboard_getbitdata( mz2000_for_io, mz2000_for_io->ioinfo->port_e8 & 0x0fU );
	return code;
}

static int out_ea( u16 port, u8 value )
{
	return 0;
}

static u8 in_eb( u16 port )
{
	return 0xff;
}

static int out_eb( u16 port, u8 value )
{
	MZ2000 *mz = mz2000_for_io;

	//_TRACE("eb=%02x\n", value);
	switch (mz->z80pio_modecontrol_state) {
	case 1 :
		if (!(value & 1))
			mz -> intVect = value;
		mz -> z80pio_modecontrol_state = 0;
		break;
	case 2 :
		mz -> bitMask = ~value;
		keyboard_setinterrupt( (unsigned char)(mz->ioinfo->port_e8 & 0x0f), (void *)mz2io_keyintfunc );
		mz -> z80pio_modecontrol_state = 0;
		break;
	default :
		if ((value & 0x0fU) == 0x03U)
			mz -> z80pio_modecontrol_state = 1;
		else if ((value & 0x9fU) == 0x97U)
			mz -> z80pio_modecontrol_state = 2;
		else if (value == 0xcfU) {
			mz -> intVect = 0;
			mz -> bitMask = 0;
			mz -> z80pio_modecontrol_state = 0;
			keyboard_setinterrupt( 0xffU, NULL );
		} else  if (value == 0xffU) {
			;
		}
		break;
	}
	mz2000_for_io -> cpu_peripherals_start |= CPUPERIPHERALS_INIT_Z80PIOB;
	return 0;
}

static u8 in_f0( u16 port )
{
	return 0xff;
}

static int out_f0( u16 port, u8 value )
{
	z80clk t;
	MZ2000 *mz = mz2000_for_io;

#ifndef TIMER8253_USE_INTERNALTIME
	t = GetTickCount_us();
#else
	t = get_internal_clock();
#endif
	mz -> timer8253_starttime[0] = t;
	mz -> timer8253_starttime[1] = t;
	mz -> timer8253_starttime[2] = t;
	return 0;
}

static u8 in_f4( u16 port )
{
	return 0xff;
}

static int out_f4( u16 port, u8 value )
{
	u8 t;

	t = value & 0x07U;
	if (t != mz2000_for_io -> ioinfo -> port_f4) {
		mz2000_for_io -> ioinfo-> port_f4 = t;
		screen_portF4( mz2000_for_io, t );
	}
	return 0;
}

static u8 in_f5( u16 port )
{
	return 0xff;
}

static int out_f5( u16 port, u8 value )
{
	u8 t;

	t = value & 0x0fU;
	if (t != mz2000_for_io -> ioinfo -> port_f5) {
		mz2000_for_io -> ioinfo-> port_f5 = t;
		screen_portF5( mz2000_for_io, t );
	}
	return 0;
}

static u8 in_f6( u16 port )
{
	return 0xff;
}

static int out_f6( u16 port, u8 value )
{
	u8 t;

	t = value & 0x0fU;
	if (t != mz2000_for_io->ioinfo->port_f6) {
		mz2000_for_io->ioinfo->port_f6 = t;
		screen_portF6( mz2000_for_io, t );
	}
	return 0;
}

static u8 in_f7( u16 port )
{
	return 0xff;
}

static int out_f7( u16 port, u8 value )
{
	u8 t = value & 0x03U;

	mz2000_for_io->ioinfo->port_f7 = t;
	if (mz2000_for_io -> gramno != t) {
		mz2000_for_io -> gramno = t;
		if (mz2000_for_io -> vramon && !mz2000_for_io -> selchr)
			gram_on( mz2000_for_io );
	}
	return 0;
}

static int out_e0_80B( u16 port, u8 value )
{
	//_TRACE("port e0(80b)=%02x\n", value);
	/* bit0 Reel MOTOR ON (cassette control) */
	if (value & 0x01U) {
		if (mz2tape_getrev())
			mz2tape_rew( mz2000_for_io );
		else
			mz2tape_ff( mz2000_for_io );
	}
	/* bit1 FF,REW select (cassette control) */
	/* bit2 PLAY# (cassette control) */
	if (value & 0x04U) {
		if (mz2000_for_io->ioinfo->port_e2 & 0x40U)
			mz2tape_play( mz2000_for_io );
		else
			mz2tape_record( mz2000_for_io );
	}
	/* bit3 STOP# (cassette control) */
	if (value & 0x08U)
		mz2tape_stop( mz2000_for_io );
	/* bit4 REVERSE (display control) */
	if ((value & 0x10U) != (mz2000_for_io->ioinfo->port_e0 & 0x10U))
		screen_setreverse( mz2000_for_io, !(value & 0x10U) );
	/* bit5 SFT LOCK (keyboard LED) */
	/* bit6 GRAPH (keyboard LED) */
	/* bit7 KANA (keyboard LED) */
	mz2000_for_io->ioinfo->port_e0 = value;
	return 0;
}

static int out_e8_80B( u16 port, u8 value )
{
	if (!(mz2000_for_io -> cpu_peripherals_start & CPUPERIPHERALS_INIT_Z80PIOA)) {
		/* Not Initialized Z80 PIO A (Keyboard, etc.) */
		return 0;
	}
	/* bit0,1,2,3 keystrobe */
	/* bit4 keystrobe low */
	mz2000_for_io->ioinfo->strobe_low = !(value & 0x10U);
	/* bit6,7 gram/vram switching */
	if (value & 0x80U) {		/* Enable V-RAM and G-RAM */
		if (value & 0x40U)	/* Select Address $5000 */
			v_gram_on80B_50(mz2000_for_io);
		else			/* Select Address $D000 */
			v_gram_on80B_D0(mz2000_for_io);
	} else
		v_gram_off80B(mz2000_for_io);
	/* bit5 width 80 (L=40) */
	if ((mz2000_for_io -> ioinfo-> port_e8 & 0x20U) != (value & 0x20U)) {
		mz2000_for_io -> ioinfo -> port_e8 = value;
		screen_setchrsize( mz2000_for_io, value & 0x20U ? TRUE : FALSE );
		screen_region_add_all( mz2000_for_io );
	}
	mz2000_for_io -> ioinfo -> port_e8 = value;
	return 0;
}

static int out_f4_80B( u16 port, u8 value )
{
	u8 t;

	t = value & 0x07U;
	if (t != mz2000_for_io -> ioinfo -> port_f4) {
		mz2000_for_io -> ioinfo -> port_f4 = t;
		screen_portF6( mz2000_for_io, (unsigned char)((t & 0x06U) >> 1) );
		if (value & 0x01U) {
			mz2000_for_io -> gramno = 1;
			out_e8_80B( 0xe8, mz2000_for_io->ioinfo->port_e8 );
		} else {
			mz2000_for_io -> gramno = 0;
			out_e8_80B( 0xe8, mz2000_for_io->ioinfo->port_e8 );
		}
	}
	return 0;
}

static void reinit_io_0( MZ2000 *mz, int disk_init_flag )
{
	int i;

	/* MZ internal I/O ports */
	if (!(mz -> mzmode))
		set_ioent( 0xe0, &in_e0, &out_e0 );
	else
		set_ioent( 0xe0, &in_e0, &out_e0_80B );
	set_ioent( 0xe1, &in_e1, &out_e1 );
	set_ioent( 0xe2, &in_e2, &out_e2 );
	set_ioent( 0xe3, &in_e3, &out_e3 );
	set_ioent( 0xe4, &in_e4, &out_e4 );
	set_ioent( 0xe5, &in_e5, &out_e5 );
	set_ioent( 0xe6, &in_e6, &out_e6 );
	set_ioent( 0xe7, &in_e7, &out_e7 );
	if (!(mz -> mzmode))
		set_ioent( 0xe8, &in_e8, &out_e8 );
	else
		set_ioent( 0xe8, &in_e8, &out_e8_80B );
	set_ioent( 0xe9, &in_e9, &out_e9 );
	set_ioent( 0xea, &in_ea, &out_ea );
	set_ioent( 0xeb, &in_eb, &out_eb );
	set_ioent( 0xf0, &in_f0, &out_f0 );
	set_ioent( 0xf1, &in_f0, &out_f0 );
	set_ioent( 0xf2, &in_f0, &out_f0 );
	set_ioent( 0xf3, &in_f0, &out_f0 );
	if (!(mz -> mzmode)) {
		set_ioent( 0xf4, &in_f4, &out_f4 );
		set_ioent( 0xf5, &in_f5, &out_f5 );
		set_ioent( 0xf6, &in_f6, &out_f6 );
		set_ioent( 0xf7, &in_f7, &out_f7 );
	} else {
		set_ioent( 0xf4, &in_f4, &out_f4_80B );
		set_ioent( 0xf5, &in_f4, &out_f4_80B );
		set_ioent( 0xf6, &in_f4, &out_f4_80B );
		set_ioent( 0xf7, &in_f4, &out_f4_80B );
	}
	if (!(mz -> mzmode))
		mz -> ioinfo -> port_e0 = 0xf7;
	else
		mz -> ioinfo -> port_e0 = 0x12;
	mz -> ioinfo -> port_e1 = 0xff;
	mz -> ioinfo -> port_e2 = 0x58;
	mz -> ioinfo -> port_e3 = 0x02;
	mz -> ioinfo -> port_e4 =
	mz -> ioinfo -> port_e5 =
	mz -> ioinfo -> port_e6 =
	mz -> ioinfo -> port_e7 = 0;
	if (!(mz -> mzmode)) {
		mz -> ioinfo -> port_e8 = 0xff;
		mz -> ioinfo -> port_f4 = 0;
		mz -> ioinfo -> port_f5 = 7;
		mz -> ioinfo -> port_f6 = 0;
		mz -> ioinfo -> port_f7 = 0;
	} else {
		mz -> ioinfo -> port_e8 = 0x3f;
		mz -> ioinfo -> port_f4 = 0;
		mz -> ioinfo -> port_f5 = 0xff;
		mz -> ioinfo -> port_f6 = 0xff;
		mz -> ioinfo -> port_f7 = 0xff;
	}
	mz -> ioinfo -> strobe_low = 0;
	mz -> selchr = TRUE;
	mz -> gramno = 0;
	mz -> blank = FALSE;

	/* 8255 / 8253 / Z80PIOs are not initialized */
	if (mz -> enable_initreset)
		mz -> cpu_peripherals_start = CPUPERIPHERALS_INIT;
	else
		mz -> cpu_peripherals_start = CPUPERIPHERALS_RESET;

	/* Timer (8253) */
#ifndef TIMER8253_USE_INTERNALTIME
	mz -> timer8253_starttime[0] = GetTickCount_us();
#else
	mz -> timer8253_starttime[0] = get_internal_clock();
#endif
	mz -> timer8253_starttime[1] = mz -> timer8253_starttime[0];
	mz -> timer8253_starttime[2] = mz -> timer8253_starttime[0];
	for (i = 0; i < TIMER8253_COUNTER_MAXNUM; i++) {
		mz -> timer8253_rwmode[i] = 0;
		mz -> timer8253_currenttime[i] = mz -> timer8253_starttime[i];
	}
	mz -> timer8253_count[0] = 0x7a12;
	mz -> timer8253_count[1] = 0xa8c0;
	mz -> timer8253_count[2] = 0x0000;
	mz -> timer8253_fast_mode = 0;

	/* Keyboard (Z80 PIO) */
	mz->intVect = 0;
	mz->bitMask = 0;
	mz->z80pio_modecontrol_state = 0;
	if (!(mz -> mzmode))
		vram_on( mz );
	else
		v_gram_on80B_50( mz );

	/* Screen/CRTC */
	screen_initarch( mz );
	screen_initattr2( mz );

	/* Cassette Tape Deck */
	if (mz2tape_getstatus( NULL, NULL, NULL, NULL, NULL, NULL ) == MZ2TAPE_RUNMODE_RECORD
		|| mz2tape_getstatus( NULL, NULL, NULL, NULL, NULL, NULL ) == MZ2TAPE_RUNMODE_PLAY)
			mz2tape_stop( mz );
	mz2tape_setkinh( FALSE );
	mz2tape_setrev( FALSE );

	/* Floppy Disk Drive Interface */
	if (mz -> enable_fddif) {
		set_ioent(0xd8, &disk_port_in_d8, &disk_port_out_d8);
		set_ioent(0xd9, &disk_port_in_d9, &disk_port_out_d9);
		set_ioent(0xda, &disk_port_in_da, &disk_port_out_da);
		set_ioent(0xdb, &disk_port_in_db, &disk_port_out_db);
		set_ioent(0xdc, &disk_port_in_dc, &disk_port_out_dc);
		set_ioent(0xdd, &disk_port_in_dd, &disk_port_out_dd);
	} else {
		set_ioent(0xd8, NULL, NULL);
		set_ioent(0xd9, NULL, NULL);
		set_ioent(0xda, NULL, NULL);
		set_ioent(0xdb, NULL, NULL);
		set_ioent(0xdc, NULL, NULL);
		set_ioent(0xdd, NULL, NULL);
	}
	if (disk_init_flag)
		disk_reset( mz );
	return;
}

void reinit_io( MZ2000 *mz )
{
	reinit_io_0( mz, TRUE );
}

void refresh_io( MZ2000 *mz )
{
	u8 t;

	if (!(mz -> mzmode)) {
		/* MZ-2000/2200 */
		out_e0( 0xe0, mz->ioinfo->port_e0 );
		out_e2( 0xe2, mz->ioinfo->port_e2 );
		out_e8( 0xe8, mz->ioinfo->port_e8 );
		t = mz->ioinfo->port_f4; mz->ioinfo->port_f4 = ~t;
		out_f4( 0xf4, t );
		t = mz->ioinfo->port_f5; mz->ioinfo->port_f5 = ~t;
		out_f5( 0xf5, t );
		t = mz->ioinfo->port_f6; mz->ioinfo->port_f6 = ~t;
		out_f6( 0xf6, t );
		t = mz->ioinfo->port_f7; mz->ioinfo->port_f7 = ~t;
		out_f7( 0xf7, t );
	} else {
		/* MZ-80B/B2 */
		out_e0_80B( 0xe0, mz->ioinfo->port_e0 );
		out_e2( 0xe2, mz->ioinfo->port_e2 );
		out_e8_80B( 0xe8, mz->ioinfo->port_e8 );
		t = mz->ioinfo->port_f4; mz->ioinfo->port_f4 = ~t;
		out_f4_80B( 0xf4, t );
	}
	screen_initarch( mz );
	screen_region_add_all( mz );
	return;
}

void io_exit( MZ2000 *mz )
{
	if (mz -> ioinfo) {
		FREE( (void *)mz -> ioinfo );
		mz -> ioinfo = NULL;
	}
	return;
}

int io_init( MZ2000 *mz )
{
	mz2000_for_io = mz;

	mz->ioinfo = CALLOC( sizeof(MZIOInfo), 1 );
	if (!mz->ioinfo) {
#ifndef WIN32
		fprintf(stderr, "No enough memory\n");
#else
		_TRACE("No enough memory\n");
#endif
		return FALSE;
	}
	reinit_io_0( mz, FALSE );	/* be called before disk initialize */
	return TRUE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
