/* File: mz2mem.c

	MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Main Memory and Video Memory

	Combine the emz2000 mz2mem.c, EmuZ-2000 emuz2000mem.c,
		and extend by FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "config.h"

#include <stdio.h>
#ifndef WIN32
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#else
#include <windows.h>
#endif

#include "common.h"
#include "mz2000.h"

#define MEMORY_WAITTVRAMCLOCK_R		(mz -> memory_waittvramclock_r)
#define MEMORY_WAITTVRAMCLOCK_W		(mz -> memory_waittvramclock_w)
#define MEMORY_WAITGRAMCLOCK_R		(mz -> memory_waitgramclock_r)
#define MEMORY_WAITGRAMCLOCK_W		(mz -> memory_waitgramclock_w)
#define MEMORY_WAITGRAM80BCLOCK_R	(mz -> memory_waitgram80bclock_r)
#define MEMORY_WAITGRAM80BCLOCK_W	(mz -> memory_waitgram80bclock_w)
#define MEMORY_WAITMAINCLOCK_R		0
#define MEMORY_WAITMAINCLOCK_W		0
#define MEMORY_WAITROMCLOCK_R		1
#define MEMORY_WAITROMCLOCK_W		1

#if MEMORY_WAITMAINCLOCK_R != 0 || MEMORY_WAITMAINCLOCK_W != 0
#define USE_WAIT_ALLMEMORY
#endif

static MZ2000 *mz;

static int wait_function( void )
{
	int r;
	z80clk t;

	if (mz -> blank)
		return 0;
	if (getspeed_z80() > 0) {
		t = get_internal_clock() % get_frameclk_z80();
		if (t < get_frameclk_z80() * 20/525)
			r = 0;
		else
			r = 1;
	} else
		r = 0;
	return r;
}

/* Normal RAM */

#ifdef USE_WAIT_ALLMEMORY
static u8 ram_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITMAINCLOCK_R;

	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->memory[addr];
}

static int ram_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITMAINCLOCK_W;

	z80.internal_clock += w;
	z80.left_clock -= w;
	mz -> memory[addr] = value;
	return TRUE;
}
#endif

static u8 rom_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITROMCLOCK_R;

	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->ipl[addr];
}

static int rom_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITROMCLOCK_W;

	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

/* Text V-RAM */

static u8 vram_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITTVRAMCLOCK_R;

	addr &= (TEXT_CLIP - 1);
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->vram[addr];
}

static int vram_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITTVRAMCLOCK_W;

	addr &= (TEXT_CLIP - 1);
	mz->vram[addr] = value;
	screen_region_add_textvram( mz, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

/* G-RAM R/W and memory map switch for MZ-2000 */

static u8 gram0_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_R;

	z80.internal_clock += w;
	z80.left_clock -= w;
	return 0xffU;
}

static u8 gram1_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_R;

	addr &= GRAM_MAX;
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->gram1[addr];
}

static u8 gram2_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_R;

	addr &= GRAM_MAX;
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->gram2[addr];
}

static u8 gram3_read(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_R;

	addr &= GRAM_MAX;
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->gram3[addr];
}

static int gram0_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_W;

	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

static int gram1_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_W;

	addr &= GRAM_MAX;
	mz->gram1[addr] = value;
	if (addr < GRAM_CLIP)
		screen_region_add_gram2000( mz, 0, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

static int gram2_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_W;

	addr &= GRAM_MAX;
	mz->gram2[addr] = value;
	if (addr < GRAM_CLIP)
		screen_region_add_gram2000( mz, 1, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

static int gram3_write(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAMCLOCK_W;

	addr &= GRAM_MAX;
	mz->gram3[addr] = value;
	if (addr < GRAM_CLIP)
		screen_region_add_gram2000( mz, 2, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

int v_gram_off(MZ2000 *mz)
{
	if (mz->inipl)
		memory_ipl( mz );
	else
		memory_normal( mz );
	return TRUE;
}

int vram_on(MZ2000 *mz)
{
#ifdef USE_WAIT_ALLMEMORY
	int i;
#endif

	if (mz->vramon && !(mz->selchr))
		v_gram_off( mz );
	set_mement( 13, NULL, &vram_read, &vram_write );
#ifdef USE_WAIT_ALLMEMORY
	if (!mz->inipl) {
		for (i = 0; i < 13; i++)
			set_mement( i, NULL, ram_read, ram_write );
		for (i = 14; i < 16; i++)
			set_mement( i, NULL, ram_read, ram_write );
	}
#endif
	mz->vramon = TRUE;
	mz->selchr = TRUE;
	return TRUE;
}

int gram_on(MZ2000 *mz)
{
#ifdef USE_WAIT_ALLMEMORY
	int i;
#endif
	u8 (*readf)(u16);
	int (*writef)(u16 addr,u8 value);

	if (mz->vramon && mz->selchr)
		v_gram_off( mz );
	switch (mz->gramno) {
	case 1:
		readf = &gram1_read;
		writef = &gram1_write;
		break;
	case 2:
		readf = &gram2_read;
		writef = &gram2_write;
		break;
	case 3:
		readf = &gram3_read;
		writef = &gram3_write;
		break;
	default:
		readf = &gram0_read;
		writef = &gram0_write;
		break;
	}
	set_mement( 12, NULL, readf, writef );
	set_mement( 13, NULL, readf, writef );
	set_mement( 14, NULL, readf, writef );
	set_mement( 15, NULL, readf, writef );
#ifdef USE_WAIT_ALLMEMORY
	if (!mz->inipl) {
		for (i = 0; i < 12; i++)
			set_mement( i, NULL, ram_read, ram_write );
	}
#endif
	mz->vramon = TRUE;
	mz->selchr = FALSE;
	return TRUE;
}

/* G-RAM R/W and memory map switch for MZ-80B */

static u8 gram1_read80B(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAM80BCLOCK_R;

	addr &= GRAM_MAX_80B;
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->gram1[addr];
}

static u8 gram2_read80B(u16 addr)
{
	int w = wait_function() * MEMORY_WAITGRAM80BCLOCK_R;

	addr &= GRAM_MAX_80B;
	z80.internal_clock += w;
	z80.left_clock -= w;
	return mz->gram2[addr];
}

static int gram1_write80B(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAM80BCLOCK_W;

	addr &= GRAM_MAX_80B;
	mz->gram1[addr] = value;
	if (addr < GRAM_CLIP_80B)
		screen_region_add_gram80B( mz, 0, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

static int gram2_write80B(u16 addr, u8 value)
{
	int w = wait_function() * MEMORY_WAITGRAM80BCLOCK_W;

	addr &= GRAM_MAX_80B;
	mz->gram2[addr] = value;
	if (addr < GRAM_CLIP_80B)
		screen_region_add_gram80B( mz, 1, addr );
	z80.internal_clock += w;
	z80.left_clock -= w;
	return TRUE;
}

int v_gram_off80B(MZ2000 *mz)
{
	return v_gram_off( mz );
}

int v_gram_on80B_D0(MZ2000 *mz)
{
#ifdef USE_WAIT_ALLMEMORY
	int i;
#endif
	u8 (*readf)(u16);
	int (*writef)(u16 addr,u8 value);

	if (mz -> vramon)
		v_gram_off80B( mz );
	set_mement( 13, NULL, &vram_read, &vram_write );
	switch (mz -> gramno) {
	case 0:
		readf = &gram1_read80B;
		writef = &gram1_write80B;
		break;
	case 1:
	default:
		readf = &gram2_read80B;
		writef = &gram2_write80B;
		break;
	}
	set_mement( 14, NULL, readf, writef );
	set_mement( 15, NULL, readf, writef );
#ifdef USE_WAIT_ALLMEMORY
	if (!mz->inipl) {
		for (i = 0; i < 13; i++)
			set_mement( i, NULL, ram_read, ram_write );
	}
#endif
	mz->vramon = TRUE;
	return TRUE;
}

int v_gram_on80B_50(MZ2000 *mz)
{
#ifdef USE_WAIT_ALLMEMORY
	int i;
#endif
	u8 (*readf)(u16);
	int (*writef)(u16 addr,u8 value);

	if (mz -> vramon)
		v_gram_off80B( mz );
	set_mement( 5, NULL, &vram_read, &vram_write );
	switch (mz -> gramno) {
	case 0:
		readf = &gram1_read80B;
		writef = &gram1_write80B;
		break;
	case 1:
	default:
		readf = &gram2_read80B;
		writef = &gram2_write80B;
		break;
	}
	set_mement( 6, NULL, readf, writef );
	set_mement( 7, NULL, readf, writef );
#ifdef USE_WAIT_ALLMEMORY
	if (!mz->inipl) {
		for (i = 0; i < 5; i++)
			set_mement( i, NULL, ram_read, ram_write );
		for (i = 8; i < 16; i++)
			set_mement( i, NULL, ram_read, ram_write );
	}
#endif
	mz->vramon = TRUE;
	return TRUE;
}

/* Memory System Interfaces */

/* mode: 0 = all / 1 = main / 2 = T/G-RAM */
int memory_clear( MZ2000 *mz, int mode )
{
	if (!mode || mode == 1) {
		if (mz -> iplmemzclr)
			memset( mz -> memory, 0, MZ_MAINMEMORY_SIZE );
		else {
			if (!(mz -> mzmode)) {
				/* MZ-2000/2200 */
				int i, t, mode = FALSE;

				for (i = 0; i < MZ_MAINMEMORY_SIZE; i++) {
					if (!(i % 0x0080U))
						mode = !mode;
					t = (i & 1) ^ mode;
					mz->memory[i] = t ? 0x00U : 0xffU;	/* My MZ-2000 pattern */
				}
			} else {
				/* MZ-80B/B2 */
				int i, t;

				for (i = 0; i < MZ_MAINMEMORY_SIZE; i++) {
					t = i & 1;
					mz->memory[i] = t ? 0x00U : 0xffU;	/* My MZ-2000 pattern */
				}
			}
		}
	}
	if (!mode || mode == 2) {
		memset( mz->vram,  0, MZ_TEXTVRAM_SIZE );
		memset( mz->gram1, 0, MZ_GRAMPAGE_SIZE );
		memset( mz->gram2, 0, MZ_GRAMPAGE_SIZE );
		memset( mz->gram3, 0, MZ_GRAMPAGE_SIZE );
	}
	return TRUE;
}

int memory_ipl(MZ2000 *mz)
{
	int i;

	set_mement( 0, mz->ipl, rom_read, rom_write );
	for (i = 1; i < 8; i++)
		set_mement( i, NULL, NULL, NULL );
	for (i = 0; i < 8; i++)
		set_mement( i+8, mz->memory+i*4096, NULL, NULL );
	mz->inipl = TRUE;
	mz->vramon = FALSE;
	return TRUE;
}

int memory_normal(MZ2000 *mz)
{
	int i;

	for (i = 0; i < 16; i++)
		set_mement( i, mz->memory+i*4096, NULL, NULL );
	mz->inipl = FALSE;
	mz->vramon = FALSE;
	return TRUE;
}

void memory_exit( MZ2000 *arg )
{
	if (mz -> gram3) {
		FREE( mz -> gram3 );
		mz -> gram3 = NULL;
	}
	if (mz -> gram2) {
		FREE( mz -> gram2 );
		mz -> gram2 = NULL;
	}
	if (mz -> gram1) {
		FREE( mz -> gram1 );
		mz -> gram1 = NULL;
	}
	if (mz -> vram) {
		FREE( mz -> vram );
		mz -> vram = NULL;
	}
	if (mz -> memory) {
		FREE( mz -> memory );
		mz -> memory = NULL;
	}
	if (mz -> ipl) {
		FREE( mz -> ipl );
		mz -> ipl = NULL;
	}
	return;
}

int memory_init( MZ2000 *arg )
{
	mz = arg;
	if (!(mz->ipl = (u8 *)CALLOC( MZ_IPLROM_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory (iplrom)\n" );
#else
		_TRACE( "No enough memory (iplrom)\n" );
#endif
		return FALSE;
	}
	if (!(mz->memory = (u8 *)CALLOC( MZ_MAINMEMORY_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory fmain (memory)\n" );
#else
		_TRACE( "No enough memory fmain (memory)\n" );
#endif
		FREE( mz -> ipl );
		mz -> ipl = NULL;
		return FALSE;
	}
	if (!(mz->vram = (u8 *)CALLOC( MZ_TEXTVRAM_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory (vram)\n" );
#else
		_TRACE( "No enough memory (vram)\n" );
#endif
		FREE( mz -> memory );
		mz -> memory = NULL;
		FREE( mz -> ipl );
		mz -> ipl = NULL;
		return FALSE;
	}
	if (!(mz->gram1 = (u8 *)CALLOC( MZ_GRAMPAGE_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory (gram1)\n" );
#else
		_TRACE( "No enough memory (gram1)\n" );
#endif
		FREE( mz -> vram );
		mz -> vram = NULL;
		FREE( mz -> memory );
		mz -> memory = NULL;
		FREE( mz -> ipl );
		mz -> ipl = NULL;
		return FALSE;
	}
	if (!(mz->gram2 = (u8 *)CALLOC( MZ_GRAMPAGE_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory (gram2)\n" );
#else
		_TRACE( "No enough memory (gram2)\n" );
#endif
		FREE( mz -> gram1 );
		mz -> gram1 = NULL;
		FREE( mz -> vram );
		mz -> vram = NULL;
		FREE( mz -> memory );
		mz -> memory = NULL;
		FREE( mz -> ipl );
		mz -> ipl = NULL;
		return FALSE;
	}
	if (!(mz->gram3 = (u8 *)CALLOC( MZ_GRAMPAGE_SIZE, 1 ))) {
#ifndef WIN32
		fprintf( stderr, "No enough memory (gram3)\n" );
#else
		_TRACE( "No enough memory (gram3)\n" );
#endif
		FREE( mz -> gram2 );
		mz -> gram2 = NULL;
		FREE( mz -> gram1 );
		mz -> gram1 = NULL;
		FREE( mz -> vram );
		mz -> vram = NULL;
		FREE( mz -> memory );
		mz -> memory = NULL;
		FREE( mz -> ipl );
		mz -> ipl = NULL;
		return FALSE;
	}
	mz -> selchr = TRUE;
	mz -> gramno = 0;
	memory_clear( mz, 0 );
	memory_ipl( mz );
	return TRUE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
