/* File: mz2tape.c

	SHARP MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Cassette Tape Deck

	Copyright (C) 2002-2019 FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.

	Cf. Referenced EmuZ-2000 emuz2000Tape.cpp Thanks.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef WIN32
#include <windows.h>

#ifndef INVALID_SET_FILE_POINTER
#define INVALID_SET_FILE_POINTER	0xffffffff
#endif
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif

#include "common.h"
#include "mz2000.h"
#include "wavedata.h"
#ifdef WIN32
#include "resource.h"
#else
#include "uigtk.h"
#endif

#ifdef WIN32
#define TAPE_APSS_TIMING	MZ2000_REFRESHRATE * 8 / 5 /* Don't need (,) */
#else
#define TAPE_APSS_TIMING	MZ2000_REFRESHRATE * 7 / 5 /* Don't need (,) */
#endif
#define TAPE_DEFAULTFREQ	22050		/* Frequency 22.05KHz */
#define TAPE_BUFFER_SIZE	1024
#define TAPE_MZHEADER_SIZE	128
#ifdef WIN32
#define	TAPE_MAX_PATH	_MAX_PATH
#else
#define	TAPE_MAX_PATH	PATH_MAX
#endif

struct _mz2tape_filemodelist_struct {
	const char *ext;
	int filemode;
} _mz2tape_filemodelist[] = {
	{ "mzt", MZ2TAPE_FILEMODE_MZT },
	{ "mzf", MZ2TAPE_FILEMODE_MZF },
	{ "mtw", MZ2TAPE_FILEMODE_MTW },
	{ "mti", MZ2TAPE_FILEMODE_MTI },
	{ "wav", MZ2TAPE_FILEMODE_WAV },
	{ "dat", MZ2TAPE_FILEMODE_DAT },
	{ "hex", MZ2TAPE_FILEMODE_HEX },
	{ NULL,  MZ2TAPE_FILEMODE_ERROR }
};

typedef struct {
#ifdef WIN32
	HANDLE hFile;
#else
	FILE *fp;
#endif
	char pathname[TAPE_MAX_PATH];
	volatile MZ2TAPE_RUNMODE runmode;
	int freq;
	int writable;
	volatile int end;
	volatile int kinh;
	volatile int rev;
	volatile u32 pos;
	volatile u32 total;
	volatile u32 offset;

	/* for write close */
	volatile u32 headpos;
	volatile u32 dcpos;
	volatile u32 esize;

	/* z80 time context */
	volatile int have_record, prestate;
	volatile z80clk starttime, pretapetime;
	volatile u32 startoffset;
	volatile u32 recordoffset;
	volatile u8 recorddata;
	volatile int readdata_old;
} MZ2TAPE;

static volatile MZ2TAPE tape;
static unsigned char data_mzhead[TAPE_MZHEADER_SIZE];

/* small cache system for tape read */
static int data_buff_offset = -1;
static int data_buff_size = 0;
static char data_buff[TAPE_BUFFER_SIZE];

int mz2tape_ready( void )
{
#ifdef WIN32
	if (tape.hFile)
#else
	if (tape.fp)
#endif
		return TRUE;
	else
		return FALSE;
}

int mz2tape_getstatus( unsigned long *pos, unsigned long *total,
	unsigned long *offset, int *tapeend,
	unsigned long *tapefreq, unsigned long *deffreq )
{
	if (!mz2tape_ready()) {
		if (total)
			*total = tape.total;
		if (pos)
			*pos = 0UL;
		if (offset)
			*offset = 0UL;
		if (tapeend)
			*tapeend = FALSE;
		if (tapefreq)
			*tapefreq = TAPE_DEFAULTFREQ;
		if (deffreq)
			*deffreq = TAPE_DEFAULTFREQ;
		return -1;
	} else {
		if (total)
			*total = tape.total;
		if (pos)
			*pos = tape.pos;
		if (offset)
			*offset = tape.offset;
		if (tapeend)
			*tapeend = tape.end;
		if (tapefreq)
			*tapefreq = tape.freq;
		if (deffreq)
			*deffreq = TAPE_DEFAULTFREQ;
		return tape.runmode;
	}
}

int mz2tape_getwritable( void )
{
	return tape.writable;
}

int mz2tape_setwritable( int enable )
{
	tape.writable = enable;
	return 0;
}

/* MZ-2000/2200 only, inhibit the control keys */
int mz2tape_getkinh( void )
{
	return tape.kinh;
}

int mz2tape_setkinh( int enable )
{
	tape.kinh = enable;
	return 0;
}

/* MZ-80B/80B2 only, reel motor normal(FF) / reverse(REW) */
int mz2tape_getrev( void )
{
	return tape.rev;
}

int mz2tape_setrev( int enable )
{
	tape.rev = enable;
	return 0;
}

static unsigned long mz2tape_getcurpos( void )
{
#ifdef WIN32
	if (!tape.hFile)
#else
	if (!tape.fp)
#endif
		return 0;
	return tape.pos;
}

static int __mz2tape_clearcache( void )
{
	data_buff_offset = -1;
	data_buff_size = 0;
	memset( data_buff, 0x80, TAPE_BUFFER_SIZE );
	return 0;
}

/* Read the byte data with cache buffer */
static int mz2tape_readbyte( unsigned long pos, unsigned char *readdata )
{
#ifdef WIN32
	DWORD r, dwReadBytes;
#else
	int r;
#endif

	if (readdata)
		*readdata = 0x80U;
#ifdef WIN32
	if (!tape.hFile)
		return FALSE;
#else
	if (!tape.fp)
		return FALSE;
#endif
	if (pos >= tape.total)
		return FALSE;
	if (data_buff_size > 0) {
		if ((int)pos >= data_buff_offset && (int)pos - data_buff_offset < data_buff_size) {
			if (readdata)
				*readdata = data_buff[pos - data_buff_offset];
			return TRUE;
		}
	}
#ifdef WIN32
	r = SetFilePointer( tape.hFile, pos, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER)
		return FALSE;
#else
	r = fseek( tape.fp, pos, SEEK_SET );
	if (r < 0)
		return FALSE;
#endif
	__mz2tape_clearcache();
#ifdef WIN32
	ReadFile( tape.hFile, data_buff, TAPE_BUFFER_SIZE, &dwReadBytes, NULL );
	if (!dwReadBytes)
		return FALSE;
	data_buff_size = dwReadBytes;
#else
	r = fread( data_buff, 1, TAPE_BUFFER_SIZE, tape.fp );
	if (r < 1)
		return FALSE;
	data_buff_size = r;
#endif
	data_buff_offset = pos;

	if (readdata)
		*readdata = data_buff[0]; 
	tape.pos = pos;
	return TRUE;
}

/* Write the byte data without cache. (can't run the fast clock mode)
   It has fill past no wrote area. */
static int mz2tape_writebyte( MZ2000 *mz, unsigned long pos, unsigned char data )
{
	int ct;
#ifdef WIN32
	DWORD r, dwWriteBytes;
#else
	int r;
#endif

#ifdef WIN32
	if (!tape.hFile)
		return FALSE;
#else
	if (!tape.fp)
		return FALSE;
#endif

	/* Fill the past no wrote area */
	ct = 0;
	while (tape.recordoffset < pos) {
#ifdef WIN32
		r = SetFilePointer( tape.hFile, tape.recordoffset, 0, FILE_BEGIN );
		if (r == INVALID_SET_FILE_POINTER)
			return FALSE;
		WriteFile( tape.hFile, (void *)&tape.recorddata, 1, &dwWriteBytes, NULL );
		if (!dwWriteBytes)
			return FALSE;
#else
		r = fseek( tape.fp, tape.recordoffset, SEEK_SET );
		if (r < 0)
			return FALSE;
		r = fwrite( (void *)&tape.recorddata, 1, 1, tape.fp );
		if (r < 1)
			return FALSE;
#endif
		tape.have_record = TRUE;
		tape.pos = tape.recordoffset;
		tape.recordoffset++;
		if (tape.pos >= tape.total)
			tape.total = tape.pos + 1;
		ct++;
		if (ct > 64 * tape.freq / TAPE_DEFAULTFREQ)  {
			/* Detect the long same value */
			/* Emulate the coupling capacitor */
			tape.recorddata = mz -> writeThreshold;
		}
	}

	/* Write the current byte data */
#ifdef WIN32
	r = SetFilePointer( tape.hFile, pos, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER)
		return FALSE;
	WriteFile( tape.hFile, &data, 1, &dwWriteBytes, NULL );
	if (!dwWriteBytes)
		return FALSE;
#else
	r = fseek( tape.fp, pos, SEEK_SET );
	if (r < 0)
		return FALSE;
	r = fwrite( &data, 1, 1, tape.fp );
	if (r < 1)
		return FALSE;
#endif
	tape.have_record = TRUE;
	tape.pos = pos;
	tape.recordoffset = pos;
	tape.recorddata = data;
	if (tape.pos >= tape.total)
		tape.total = tape.pos + 1;
	return TRUE;
}

static int mz2tape_writeclose( MZ2000 *mz )
{
	unsigned int vp[2];
#ifdef WIN32
	DWORD r, dwWriteBytes;
#else
	int r;
#endif

#ifdef WIN32
	if (!tape.hFile)
		return FALSE;
#else
	if (!tape.fp)
		return FALSE;
#endif
	if (!tape.have_record)
		return -1;
	/* fill the past no wrote area */
	mz2tape_updatewrite( mz, -1 );

	/* close the wave file */
	vp[0] = tape.total - tape.offset;
	vp[1] = vp[0] + tape.esize - 8;
#ifdef WIN32
	r = SetFilePointer( tape.hFile, tape.dcpos, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER)
		return FALSE;
	WriteFile( tape.hFile, &vp[0], sizeof(unsigned int), &dwWriteBytes, NULL );
	if (!dwWriteBytes)
		return FALSE;
	r = SetFilePointer( tape.hFile, tape.headpos + 4, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER)
		return FALSE;
	WriteFile( tape.hFile, &vp[1], sizeof(unsigned int), &dwWriteBytes, NULL );
	if (!dwWriteBytes)
		return FALSE;
#else
	r = fseek( tape.fp, tape.dcpos, SEEK_SET );
	if (r < 0)
		return FALSE;
	r = fwrite( &vp[0], 1, sizeof(unsigned int), tape.fp );
	if (r < sizeof(unsigned int))
		return FALSE;
	r = fseek( tape.fp, tape.headpos + 4, SEEK_SET );
	if (r < 0)
		return FALSE;
	r = fwrite( &vp[1], 1, sizeof(unsigned int), tape.fp );
	if (r < sizeof(unsigned int))
		return FALSE;
#endif
	tape.have_record = FALSE;
	return 0;
}

static int mz2tape_eject2( MZ2000 *mz )
{
	mz2tape_stop2( mz );
#ifdef WIN32
	if (tape.hFile) {
		CloseHandle( tape.hFile );
		tape.hFile = NULL;
	}
#else
	if (tape.fp) {
		fclose( tape.fp );
		tape.fp = NULL;
	}
#endif
	tape.runmode = MZ2TAPE_RUNMODE_STOP;
	tape.end = FALSE;
	tape.pos = 0;
	tape.total = 0;
	tape.offset = 0;
	memset( &data_mzhead, 0, TAPE_MZHEADER_SIZE );
	__mz2tape_clearcache();
	return 0;
}

int mz2tape_eject( MZ2000 *mz, int sound_flag )
{
	setspeed_z80( (int)mz->nowZ80Clock );	/* normal speed */
	mz2tape_stop( mz );
	if (sound_flag) {
		if (!mz->mzmode)
			sound_playwav_file( mz, SOUND_WAV_2000EJECT );
		else
			sound_playwav_file( mz, SOUND_WAV_80BEJECT );
	}
	mz2tape_eject2( mz );
	return 0;
}

int mz2tape_stop2( MZ2000 *mz )
{
	setspeed_z80( (int)mz->nowZ80Clock );	/* normal speed */
	sound_reset( mz );
	if (tape.runmode == MZ2TAPE_RUNMODE_RECORD) {
		mz2tape_writeclose( mz );
		tape.pretapetime = 0;
#ifdef WIN32
		CloseHandle( tape.hFile );
		tape.hFile = CreateFile( (char *)tape.pathname, GENERIC_READ, FILE_SHARE_READ,
			NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
		if (tape.hFile == INVALID_HANDLE_VALUE) {
			MessageBox( NULL, "MTI/WAV file read open error",
				"Tape Record", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			mz2tape_eject( mz, FALSE );
			return -1;
		}
#else
		fflush( tape.fp );
		tape.fp = freopen( (char *)tape.pathname, "r", tape.fp );
		if (!tape.fp) {
			uigtk_messagebox( "MTI/WAV file read open error",
				"Tape Record", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			mz2tape_eject( mz, FALSE );
			return -1;
		}
#endif
	}
	tape.runmode = MZ2TAPE_RUNMODE_STOP;
	__mz2tape_clearcache();
	return 0;
}

int mz2tape_stop( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	if (tape.runmode != MZ2TAPE_RUNMODE_STOP) {
		if (!mz->mzmode)
			sound_playwav_file( mz, SOUND_WAV_2000STOP );
		else
			sound_playwav_file( mz, SOUND_WAV_80BSTOP );
	}
	return mz2tape_stop2( mz );
}

int mz2tape_rew( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	if (tape.runmode == MZ2TAPE_RUNMODE_REW)
		return 0;
	if (tape.runmode != MZ2TAPE_RUNMODE_STOP)
		mz2tape_stop2( mz );
	if (!mz->mzmode)
		sound_playwav_file( mz, SOUND_WAV_2000FFST );
	else
		sound_playwav_file( mz, SOUND_WAV_80BPLAY );
	tape.runmode = MZ2TAPE_RUNMODE_REW;
	tape.starttime = get_internal_clock();
	tape.startoffset = mz2tape_getcurpos();
	return 0;
}

int mz2tape_ff( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	if (tape.runmode == MZ2TAPE_RUNMODE_FF)
		return 0;
	if (tape.runmode != MZ2TAPE_RUNMODE_STOP)
		mz2tape_stop2( mz );
	if (!mz->mzmode)
		sound_playwav_file( mz, SOUND_WAV_2000FFST );
	else
		sound_playwav_file( mz, SOUND_WAV_80BPLAY );
	if (mz -> tapeff_ignore) {
		mz2tape_stop( mz );
		return 0;
	}
	tape.runmode = MZ2TAPE_RUNMODE_FF;
	tape.starttime = get_internal_clock();
	tape.startoffset = mz2tape_getcurpos();
	return 0;
}

int mz2tape_play( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	if (tape.runmode != MZ2TAPE_RUNMODE_PLAY) {
		mz2tape_stop( mz );
		if (!mz->mzmode)
			sound_playwav_file( mz, SOUND_WAV_2000PLAY );
		else
			sound_playwav_file( mz, SOUND_WAV_80BPLAY );
	}
	if (!mz -> tapespeaker)
		setspeed_z80( (int)(mz -> z80MaxClock) ); /* max speed */
	tape.runmode = MZ2TAPE_RUNMODE_PLAY;
	tape.starttime = get_internal_clock();
	tape.startoffset = mz2tape_getcurpos();
	return 0;
}

int mz2tape_record( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	__mz2tape_clearcache();
	if (!tape.writable)
		return -1;
	if (tape.runmode != MZ2TAPE_RUNMODE_RECORD) {
		mz2tape_stop( mz );
		if (!mz->mzmode)
			sound_playwav_file( mz, SOUND_WAV_2000PLAY );
		else
			sound_playwav_file( mz, SOUND_WAV_80BPLAY );
	} else
		return 0;
#ifdef WIN32
	CloseHandle( tape.hFile );
	tape.hFile = CreateFile( (char *)tape.pathname, GENERIC_READ | GENERIC_WRITE, 0,
		NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
	if (tape.hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "MTI/WAV file write open error",
			"Tape Record", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		mz2tape_eject( mz, FALSE );
		return -1;
	}
#else
	tape.fp = freopen( (char *)tape.pathname, "r+", tape.fp );
	if (!tape.fp) {
		uigtk_messagebox( "MTI/WAV file write open error",
			"Tape Record", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		mz2tape_eject( mz, FALSE );
		return -1;
	}
#endif
	tape.runmode = MZ2TAPE_RUNMODE_RECORD;
	tape.starttime    = get_internal_clock();
	tape.startoffset  = mz2tape_getcurpos();
	tape.recordoffset = tape.startoffset;
	/* Default is center, because it has the coupling capacitor. */
	tape.recorddata = mz -> writeThreshold;
	tape.pretapetime = 0;
	return 0;
}

int mz2tape_head( MZ2000 *mz )
{
	if (!mz2tape_ready())
		return -1;
	if (tape.runmode != MZ2TAPE_RUNMODE_STOP)
		mz2tape_stop( mz );
	else {
		if (!mz->mzmode)
			sound_playwav_file( mz, SOUND_WAV_2000STOP );
		else
			sound_playwav_file( mz, SOUND_WAV_80BSTOP );
	}
#ifdef WIN32
	if (tape.hFile) {
		SetFilePointer( tape.hFile, tape.offset, 0, FILE_BEGIN );
#else
	if (tape.fp) {
		fseek( tape.fp, tape.offset, SEEK_SET );
#endif
		tape.pos = tape.offset;
	}
	tape.runmode = MZ2TAPE_RUNMODE_STOP;
	tape.end = FALSE;
	return 0;
}

int mz2tape_updatestatus( MZ2000 *mz, unsigned char *port )
{
	int flag;
	u8 v, tdat1, tdat2;
	u32 th, dat, tapepos;
	z80clk t, nowtapetime, ttt;

	v = *port;
	if (!mz2tape_ready()) {
		v |= 0x20U;	/* bit5 no TREADY (cassette control) */
		v |= 0x10U;	/* bit4 no WREADY (cassette control) */
		*port = v;
		return 0;
	}
	switch (tape.runmode) {
	case MZ2TAPE_RUNMODE_PLAY :
		/* Compute the position byte from CPU time for PLAY/RECORD */
		/* It has tapePitch value compatibility for original EmuZ-2000. */
		ttt = (get_internal_clock() - tape.starttime) * tape.freq / TAPE_DEFAULTFREQ;
		nowtapetime = ttt * (mz -> tapePitch + TAPE_PITCH_OFFSET);
		/* Read it */
		flag = mz2tape_readbyte( (u32)(nowtapetime / 10000) + tape.startoffset, &tdat1 );
		if (flag)
			flag = mz2tape_readbyte( (u32)(nowtapetime / 10000) + 1 + tape.startoffset, &tdat2 );
		if (!flag) {
			/* Tape End or I/O error */
			if (!mz->mzmode)
				sound_playwav_file( mz, SOUND_WAV_2000STOP );
			else
				sound_playwav_file( mz, SOUND_WAV_80BSTOP );
			setspeed_z80( (int)mz->nowZ80Clock );	/* Normal speed */
			sound_reset( mz );
			tape.runmode = MZ2TAPE_RUNMODE_STOP;
			tape.end = TRUE;
			tape.pos = tape.total;
			v &= ~0x40U;		/* bit6 RDATA=L */
			if (!(mz -> mzmode))
				v |= 0x08U;	/* bit3 TAPE END (MZ-2000/2200 only) */
		} else {
			/* Linear Interpolation */
			if (tdat2 >= tdat1) {
				t = (tdat2 - tdat1) * (nowtapetime % 10000) / 10000;
				dat = tdat1 + (u32)t;
			} else {
				t = (tdat1 - tdat2) * (nowtapetime % 10000) / 10000;
				dat = tdat1 - (u32)t;
			}

			/* Get the value with Schumidt Trigger */
			th = mz -> tapeThreshold;
			flag = tape . readdata_old;
			if (dat >= th + 3)
				flag = FALSE;	/* L */
			else if (dat <= th)
				flag = TRUE;	/* H */
			else
				;		/* Noise */
			tape . readdata_old = flag;

			/* Set the value to Hardware */
			if (mz -> tapemode_reverse)
				flag = !flag;
			if (flag)
				v |= 0x40U;	/* bit6 RDATA=H */
			else
				v &= ~0x40U;	/* bit6 RDATA=L */
			if (!(mz -> mzmode))
				v &= ~0x08U;	/* bit3 NO TAPE END (MZ-2000/2200 only) */
		}
		if (mz -> tapespeaker) {
			if (getspeed_z80() != (int)mz->nowZ80Clock)
				setspeed_z80( (int)mz->nowZ80Clock );
			sound_setstate( mz, v & 0x40U ? TRUE : FALSE );
		} else {
			if (getspeed_z80() != (int)mz->z80MaxClock)
				setspeed_z80( (int)mz->z80MaxClock);
		}
		break;
	case MZ2TAPE_RUNMODE_RECORD :
		if (!(mz -> mzmode))
			v &= ~0x08U;		/* bit3 NO TAPE END (MZ-2000/2200 only) */
		break;
	case MZ2TAPE_RUNMODE_FF :
	case MZ2TAPE_RUNMODE_REW :
		/* Compute the position byte from CPU time for FF/REW */
		if (tape.runmode == MZ2TAPE_RUNMODE_FF) {
			ttt = (get_internal_clock() - tape.starttime) * tape.freq / TAPE_DEFAULTFREQ;
			nowtapetime = ttt * TAPE_APSS_TIMING * (mz -> tapePitch + TAPE_PITCH_OFFSET);
		} else {
			ttt = (tape.starttime - get_internal_clock()) * tape.freq / TAPE_DEFAULTFREQ;
			nowtapetime = ttt * TAPE_APSS_TIMING * (mz -> tapePitch + TAPE_PITCH_OFFSET);
		}
		if (nowtapetime / 10000 + tape.startoffset < 0)
			tape.pos = 0;
		else
			tape.pos = (u32)(nowtapetime / 10000 + tape.startoffset);
		tapepos = tape.pos;
		/* Read it for APSS */
		flag = mz2tape_readbyte( tapepos, &tdat1 );
		if (flag)
			flag = mz2tape_readbyte( tapepos + 1, &tdat2 );
		if (tape.pos < tape.offset) {
			/* Tape Head */
			if (!mz->mzmode)
				sound_playwav_file( mz, SOUND_WAV_2000STOP );
			else
				sound_playwav_file( mz, SOUND_WAV_80BSTOP );
			tape.runmode = MZ2TAPE_RUNMODE_STOP;
			tape.end = FALSE;
			tape.pos = tape.offset;
			if (!(mz -> mzmode))
				v &= ~0x08U;	/* bit3 NO TAPE END (MZ-2000/2200 only) */
		} else if (!flag || tape.pos >= tape.total) {
			/* Tape End */
			if (!mz->mzmode)
				sound_playwav_file( mz, SOUND_WAV_2000STOP );
			else
				sound_playwav_file( mz, SOUND_WAV_80BSTOP );
			tape.runmode = MZ2TAPE_RUNMODE_STOP;
			tape.end = TRUE;
			tape.pos = tape.total;
			if (!(mz -> mzmode))
				v |= 0x08U;	/* bit3 TAPE END (MZ-2000/2200 only) */
		} else {
			tape.end = FALSE;

			/* Get the value */
			dat = tdat1;
			if (dat >= mz -> tapeThreshold - 3)
				flag = FALSE;	/* L */
			else
				flag = TRUE;	/* H */
			tape . readdata_old = flag;

			/* Set the value to Hardware */
			if (flag)
				v |= 0x40U;	/* bit6 RDATA=H */
			else
				v &= ~0x40U;	/* bit6 RDATA=L */
			if (!(mz -> mzmode))
				v &= ~0x08U;	/* bit3 NO TAPE END (MZ-2000/2200 only) */
		}
		setspeed_z80( (int)mz->nowZ80Clock );	/* Normal speed */
		break;
	default :
		v &= ~0x40U;	/* bit6 RDATA=L */
		setspeed_z80( (int)mz->nowZ80Clock );	/* Normal speed */
		break;
	}
	v &= ~0x20U;		/* bit5 TREADY (cassette control) */
	if (tape.writable)
		v &= ~0x10U;	/* bit4 WREADY (cassette control) */
	else
		v |= 0x10U;	/* bit4 no WREADY (cassette control) */
	*port = v;
	return 0;
}

int mz2tape_updatewrite( MZ2000 *mz, int state )
{
	int flag, thv;
	int nstate, prestate;
	u8 tdat1, tdat2, dat;
	z80clk t, nowtapetime, ttt;

	if (!mz2tape_ready())
		return 0;
	if (tape.runmode != MZ2TAPE_RUNMODE_RECORD)
		return 0;
	prestate = (mz -> tapemode_reverse) ? !tape.prestate : tape.prestate;
	if (!prestate) {
		thv = mz -> writeThreshold + mz -> amplitudeRange;
		if (thv >= 0xff)
			thv = 0xff;
		tdat1 = (u8)thv;
	} else {
		thv = mz -> writeThreshold - mz -> amplitudeRange;
		if (thv <= 0)
			thv = 0;
		tdat1 = (u8)thv;
	}
	if (state >= 0)
		nstate = (mz -> tapemode_reverse) ? !state : state;
	else
		nstate = prestate;
	if (!nstate) {
		thv = mz -> writeThreshold + mz -> amplitudeRange;
		if (thv >= 0xff)
			thv = 0xff;
		tdat2 = (u8)thv;
	} else {
		thv = mz -> writeThreshold - mz -> amplitudeRange;
		if (thv <= 0)
			thv = 0;
		tdat2 = (u8)thv;
	}
	/* Compute the position byte from CPU time for PLAY/RECORD */
	/* It has tapePitch value compatibility for original EmuZ-2000. */
	ttt = (get_internal_clock() - tape.starttime) * tape.freq / TAPE_DEFAULTFREQ;
	nowtapetime = ttt * (mz -> tapePitch + TAPE_PITCH_OFFSET);
	/* Write it */
	dat = tdat2;
	flag = TRUE;
	if (mz -> tapewrite_interpolation) {
		if (state != tape.prestate) {
			if (tape.pretapetime) {
				tdat1 = tape.recorddata;
				if (tdat2 >= tdat1) {
					t = (tdat2 - tdat1) * (nowtapetime % 10000) / 10000;
					dat = tdat1 + (u32)t;
				} else {
					t = (tdat1 - tdat2) * (nowtapetime % 10000) / 10000;
					dat = tdat2 + (u32)t;
				}
			}
		}
	}
	flag = mz2tape_writebyte( mz, (u32)(nowtapetime / 10000) + tape.startoffset, dat );
	dat = tdat2;
	if (flag)
		flag = mz2tape_writebyte( mz, (u32)(nowtapetime / 10000) + tape.startoffset + 1, dat );
	if (!flag) {
		/* I/O error */
		mz2tape_stop( mz );
	}
	if (mz -> tapespeaker)
		sound_setstate( mz, state );
	tape.pretapetime = nowtapetime;
	tape.prestate = state;
	return 0;
}

int mz2tape_updatetimer( void )
{
	u32 offset_bk;
	z80clk nowtapetime, ttt;
	MZ2000 *mz = mz2000_global;
	static int lct = 0;

	if (!mz2tape_ready())
		return 0;
	switch (tape.runmode) {
	case MZ2TAPE_RUNMODE_FF :
	case MZ2TAPE_RUNMODE_REW :
		lct++;
		if (lct > 100 / MZ2000_REFRESHRATE - 1) {
			if (tape.runmode == MZ2TAPE_RUNMODE_FF) {
				if (!mz->mzmode)
					sound_playwav_file( mz, SOUND_WAV_2000FF );
				else
					sound_playwav_file( mz, SOUND_WAV_80BFF );
			} else {
				if (!mz->mzmode)
					sound_playwav_file( mz, SOUND_WAV_2000REW );
				else
					sound_playwav_file( mz, SOUND_WAV_80BREW );
			}
			lct = 0;
		}
		if (tape.runmode == MZ2TAPE_RUNMODE_FF) {
			if (get_internal_clock() - tape.starttime >= 0 && get_internal_clock() - tape.starttime < 4000 * Z80_TIMER_MSEC)
				break;
		} else {
			if (tape.starttime - get_internal_clock() >= 0 && tape.starttime - get_internal_clock() < 4000 * Z80_TIMER_MSEC)
				break;
		}
		/* no read status, normal FF/REW */
		offset_bk = mz2tape_getcurpos();
		if (tape.runmode == MZ2TAPE_RUNMODE_FF) {
			ttt = (get_internal_clock() - tape.starttime) * tape.freq / TAPE_DEFAULTFREQ;
			nowtapetime = ttt * TAPE_APSS_TIMING * (mz -> tapePitch + TAPE_PITCH_OFFSET);
		} else {
			ttt = (tape.starttime - get_internal_clock()) * tape.freq / TAPE_DEFAULTFREQ;
			nowtapetime = ttt * TAPE_APSS_TIMING * (mz -> tapePitch + TAPE_PITCH_OFFSET);
		}
		if (nowtapetime / 10000 + tape.startoffset < 0)
			tape.pos = 0;
		else
			tape.pos = (u32)(nowtapetime / 10000 + tape.startoffset);
		if (tape.pos < tape.offset) {
			/* Tape Head */
			if (!mz->mzmode)
				sound_playwav_file( mz, SOUND_WAV_2000STOP );
			else
				sound_playwav_file( mz, SOUND_WAV_80BSTOP );
			tape.runmode = MZ2TAPE_RUNMODE_STOP;
			tape.end = FALSE;
			tape.pos = tape.offset;
		} else if (tape.pos >= tape.total) {
			/* Tape End */
			if (!mz->mzmode)
				sound_playwav_file( mz, SOUND_WAV_2000STOP );
			else
				sound_playwav_file( mz, SOUND_WAV_80BSTOP );
			tape.runmode = MZ2TAPE_RUNMODE_STOP;
			tape.end = TRUE;
			tape.pos = tape.total;
		} else {
			tape.end = FALSE;
			if (offset_bk != mz2tape_getcurpos()) {
				tape.starttime = get_internal_clock();
				tape.startoffset = mz2tape_getcurpos();
			}
		}
		setspeed_z80( (int)mz->nowZ80Clock );	/* Normal speed */
		break;
	default :
		break;
	}
	return 0;
}

#ifdef WIN32
static int _intelhex_getln( char *p, int max, HANDLE hFile )
{
	int i = 0;
	char c;
	DWORD dwReadSize;

	while (1)  {
		ReadFile( hFile, &c, 1, &dwReadSize, NULL );
		if (dwReadSize != 1) {
			if (!i) {
				p[0] = '\n';
				p[1] = '\0';
				return 0;
			}
			p[0] = '\0';
			break;
		}
		if (c == '\r')
			continue;
		if (c == '\n' || c == '\0')
			break;
		p[i] = c;
		i++;		
	}
	p[i] = '\n';
	i++;
	p[i] = '\0';
	i++;
	return i - 1;
}

static int _intelhex_read( HANDLE *hFile, unsigned char *p, unsigned int maxsize, char *errbuf )
#else
static int _intelhex_read( FILE *fp, unsigned char *p, unsigned int maxsize, char *errbuf )
#endif
{
	int j, r, n, hextype, lineno;
	unsigned int seg, adr, adr_up, adr_low, c, sum, t = 0;
	char buf[LNMAX], *cp;

	seg = 0U;
	lineno = 0;
#ifdef WIN32
	while (_intelhex_getln( buf, LNMAX, hFile ))
#else
	while (fgets( buf, LNMAX, fp ))
#endif
	{
		lineno++;
		buf[LNMAX - 1] = '\0';
		cp = buf;
		if (*cp++ != ':')
			continue;
		r = sscanf( cp, "%02x%02x%02x%02x", &n, &adr_up, &adr_low, &hextype );
		cp += 2 + 4 + 2;
		if (r < 4) {
			sprintf( errbuf, "File read error, length < 4 (line:%d)", lineno );
			return -1;
		}
		adr = seg + ((adr_up << 8) | adr_low);
		sum = n + adr_up + adr_low + hextype;
		sum = sum & 0xffU;
		for (j = 0; j < n; j++) {
			if (adr+j < 0 || adr+j >= maxsize) {
				sprintf( errbuf, "File read error, over address range (line:%d)" , lineno );
				return -1;
			}
			if (sscanf( cp, "%02x", &c ) != 1) {
				sprintf( errbuf, "File read error, no hex value found (line:%d)" , lineno );
				return -1;
			}
			switch (hextype) {
			case 0 :
				p[adr + j] = c;
				break;
			case 2 :
				if (!j)
					t = c;
				else
					seg = (c << 12) || (t << 4);
				break;
			case 4 :
				if (!j)
					t = c;
				else
					seg = (c << 24) || (t << 16);
				break;
			default :
				break;
			}
			sum = (sum + c) & 0xffU;
			cp += 2;
		}
		sum = sum & 0xffU;
		sum = (0x100U - sum) & 0xffU;
		if (sscanf( cp, "%02x", &c ) != 1) {
			sprintf( errbuf, "File read error, check sum value error (line:%d)", lineno );
			return -1;
		}
		cp += 2;
		if (sum != c) {
			sprintf( errbuf, "File read error, check sum error (line:%d)", lineno );
			return -1;
		}
		switch (hextype) {
		case 0 :
		case 2 :
		case 4 :
			break;
		case 1 :
			goto readend;
		default :
			sprintf( errbuf, "File read warning, skip unknown type %d (line:%d)", hextype, lineno );
			break;
		}
	}
readend:
#ifndef WIN32
	if (ferror(fp)) {
		sprintf( errbuf, "File read error, I/O fail (line:%d)", lineno );
		return -1;
	}
#endif
	return 0;
}

static int _mz2tape_exec_data( int filemode, MZ2000 *mz, const char *filename )
{
	int r;
	u32 dwReadSize, size;
	char errbuf[LNMAX];
#ifdef WIN32
	HANDLE hFile;
#else
	FILE *fp;
#endif

	mz2tape_eject2( mz );
#ifdef WIN32
	hFile = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ,
		NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "MZT/MZF/DAT/HEX file open error",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
#else
	fp = fopen( filename, "r" );
	if (!fp) {
		uigtk_messagebox( "MZT/MZF/DAT/HEX file open error",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		return -1;
	}
#endif
	switch (filemode) {
	case MZ2TAPE_FILEMODE_MZT :
	case MZ2TAPE_FILEMODE_MZF :
#ifdef WIN32
		ReadFile( hFile, data_mzhead, TAPE_MZHEADER_SIZE, &dwReadSize, NULL );
		if (dwReadSize != TAPE_MZHEADER_SIZE) {
			MessageBox( NULL, "MZT/MZF file read error - 1 (Broken Header)",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
#else
		if (fread( data_mzhead, 1, TAPE_MZHEADER_SIZE, fp ) != TAPE_MZHEADER_SIZE) {
			uigtk_messagebox( "MZT/MZF file read error - 1 (Broken Header)",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
#endif
		if (data_mzhead[0x00] != 1) {
#ifdef WIN32
			MessageBox( NULL, "MZT/MZF file mode error",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
#else
			uigtk_messagebox( "MZT/MZF file mode error",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
#endif
			return -1;
		}
		memcpy( &(mz->memory[0x4f00]), data_mzhead, TAPE_MZHEADER_SIZE );
		//strncpy( tape_iplmsg, (char *)&(mz->memory[0x4f01]), TAPE_IPLMSG_MAXNUM );
		//tape_iplmsg[TAPE_IPLMSG_MAXNUM] = '\0';
		memset( &(mz->memory[0x7fe0]), 0, 0x20 );
		size = (u32)((data_mzhead[0x13] * 256) + data_mzhead[0x12]);
#ifdef WIN32
		ReadFile( hFile, mz -> memory, size, &dwReadSize, NULL );
		if (dwReadSize != size) {
			MessageBox( NULL, "MZT/MZF file read error - 2 (Broken Body)",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
#else
		if (fread( mz -> memory, 1, size, fp ) != size) {
			uigtk_messagebox( "MZT/MZF file read error - 2 (Broken Body)",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
#endif
		break;
	case MZ2TAPE_FILEMODE_DAT :
#ifdef WIN32
		ReadFile( hFile, mz -> memory, MZ_MAINMEMORY_SIZE, &dwReadSize, NULL );
#else
		dwReadSize = fread( mz -> memory, MZ_MAINMEMORY_SIZE, 1, fp );
#endif
		break;
	case MZ2TAPE_FILEMODE_HEX :
		strncpy( errbuf, "Unknown error", LNMAX );
		errbuf[LNMAX - 1] = '\0';
		mz -> memory[0] = 0x18;
		mz -> memory[1] = 0xfe;
#ifdef WIN32
		r = _intelhex_read( hFile, mz -> memory, MZ_MAINMEMORY_SIZE, errbuf );
		if (r < 0) {
			MessageBox( NULL, errbuf,
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
#else
		r = _intelhex_read( fp, mz -> memory, MZ_MAINMEMORY_SIZE, errbuf );
		if (r < 0) {
			uigtk_messagebox( errbuf,
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
#endif
		break;
	}
#ifdef WIN32
	CloseHandle( hFile );
#else
	fclose( fp );
#endif
	return 0;
}

#ifdef WIN32
static int _mz2tape_openwave( MZ2000 *mz, HANDLE hFile,
	unsigned int _offset, unsigned int _filesize, int _writable )
#else
static int _mz2tape_openwave( MZ2000 *mz, FILE *fp,
	unsigned int _offset, unsigned int _filesize, int _writable )
#endif
{
	int i, err_type;
	unsigned int vp[5], len, offset, size;
	char buf[LNMAX];
	WAVEFORMATEX wvfmt;
#ifdef WIN32
	DWORD r, dwReadSize;
#else
	int r;
#endif

	offset = _offset;
	size = _filesize - offset;
	_TRACE("tape: offset=%04x wavesize=%04x\n", offset, size );
#ifdef WIN32
	for (i = 0; i < 10; i++) {
		ReadFile( hFile, buf, 1, &dwReadSize, NULL );
		if (dwReadSize != 1) {
			MessageBox( NULL, "WAV file read error - 3A (Eyecatch RIFF(or T-P) not found)",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
		if (buf[0] == 'R' || buf[0] == 0x36)
			break;
		offset++;
		--size;
	}
	_TRACE("tape: offset=%04x wavesize=%04x (repaired)\n", offset, size );
	r = SetFilePointer( hFile, offset, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER) {
		MessageBox( NULL, "File read error - 3B (Broken file, FSEEK error)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
	ReadFile( hFile, vp, sizeof(unsigned int) * 5, &dwReadSize, NULL );
	if (dwReadSize != sizeof(unsigned int) * 5) {
		MessageBox( NULL, "File read error - 3C (Broken file, truncated)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
#else
	for (i = 0; i < 10; i++) {
		if (fread( buf, 1, 1, fp ) != 1) {
			uigtk_messagebox( "WAV File read error - 3A (Eyecatch RIFF(or T/P) not found)",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
		if (buf[0] == 'R' || buf[0] == 0x36)
			break;
		offset++;
		--size;
	}
	_TRACE("tape: offset=%04x wavesize=%04x (repaired)\n", offset, size );
	r = fseek( fp, offset, SEEK_SET );
	if (r < 0) {
		uigtk_messagebox( "File read error - 3B (Broken file, FSEEK error)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	if (fread( vp, 1, sizeof(unsigned int) * 5, fp ) != sizeof(unsigned int) * 5) {
		uigtk_messagebox( "File read error - 3C (Broken file, truncated)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
#endif
	tape.headpos = offset;
	offset += sizeof(unsigned int) * 5;
	if (vp[0] != 'FFIR') {
		if (((unsigned char *)vp)[0x00] == 0x36) {
			/* threshold and pitch stored format */
			mz -> tapePitch = (u32)((unsigned char *)vp)[0x01];
			mz -> tapeThreshold = (u32)((unsigned char *)vp)[0x02];
		} else {
#ifdef WIN32
			MessageBox( NULL, "File read error - 4 (No eyecatch RIFF(or T-P))",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
#else
			uigtk_messagebox( "File read error - 4 (No eyecatch RIFF(or T-P))",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
#endif
			return -1;
		}
	}
	err_type = -1;
#if 0
	if (vp[1] + 8 > (unsigned long)size) {
		_TRACE( "struct info size %04x > file size %04x\n", vp[1] + 8, size );
		if (err_type < 0)
			err_type = 1;
	}
#endif
	/* WAVE chunk */
	if (vp[2] != 'EVAW') {
		if (err_type < 0)
			err_type = 2;
	}

	/* WAVE format chunk */
	if (vp[3] != ' tmf') {
		if (err_type < 0)
			err_type = 3;
	}
	len = vp[4];
	if (len <= 0 || len > LNMAX) {
		if (err_type < 0)
			err_type = 4;
	}
#ifdef WIN32
	if (err_type >= 0) {
		_snprintf( buf, LNMAX, "WAV file read error - 5 (Broken file, type = %d)", err_type );
		buf[LNMAX - 1] = '\0';
		MessageBox( NULL, buf, "Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
	memset( buf, 0, LNMAX );
	ReadFile( hFile, buf, len, &dwReadSize, NULL );
	if (dwReadSize != len) {
		MessageBox( NULL, "File read error - 6 (Broken file, truncated)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
#else
	if (err_type >= 0) {
		snprintf( buf, LNMAX, "WAV file read error - 5 (Broken file, type = %d)", err_type );
		uigtk_messagebox( buf, "Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	memset( buf, 0, LNMAX );
	if (fread( buf, 1, len, fp ) != len) {
		uigtk_messagebox( "File read error - 6 (Broken file, truncated)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
#endif
	offset += len;
	err_type = -1;
	memcpy( &wvfmt, buf, sizeof(WAVEFORMATEX) );
	if (wvfmt.wFormatTag != 1) {
		if (err_type < 0)
			err_type = 1;
	}
#if 0
	if (wvfmt.nSamplesPerSec != TAPE_DEFAULTFREQ) {
		if (err_type < 0)
			err_type = 2;
	}
#endif
	if (wvfmt.wBitsPerSample != 8) {	/* 8bit */
		if (err_type < 0)
			err_type = 3;
	}
	if (wvfmt.nChannels != 1) {		/* Mono */
		if (err_type < 0)
			err_type = 4;
	}
	if (wvfmt.nBlockAlign / wvfmt.nChannels != wvfmt.wBitsPerSample / 8) {
		if (err_type < 0)
			err_type = 5;
	}
#ifdef WIN32
	if (err_type >= 0) {
		_snprintf( buf, LNMAX, "WAV file read error - 7 (only mono 8bit PCM, error type = %d)", err_type );
		buf[LNMAX - 1] = '\0';
		MessageBox( NULL, buf, "Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
#else
	if (err_type >= 0) {
		snprintf( buf, LNMAX, "WAV file read error - 7 (only mono 8bit PCM, error type = %d)", err_type );
		uigtk_messagebox( buf, "Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
#endif

	/* skip unknown header or search 'data' */
	while (1) {
#ifdef WIN32
		ReadFile( hFile, vp, sizeof(unsigned int) * 2, &dwReadSize, NULL );
		if (dwReadSize != sizeof(unsigned int) * 2) {
			MessageBox( NULL, "File read error - 8 (Broken file, truncated)",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
#else
		if (fread( vp, 1, sizeof(unsigned int) * 2, fp ) != sizeof(unsigned int) * 2) {
			uigtk_messagebox( "File read error - 8 (Broken file, truncated)",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
#endif
		offset += sizeof(unsigned int) * 2;
		len = vp[1];
		if (vp[0] == 'atad') {
			tape.dcpos = offset - sizeof(unsigned int);
			tape.esize = offset - tape.offset;
			break;
		}

		/* skip unknown header */
		offset += len;
#ifdef WIN32
		r = SetFilePointer( hFile, offset, 0, FILE_BEGIN );
		if (r == INVALID_SET_FILE_POINTER) {
			MessageBox( NULL, "File read error - 9 (Broken file, truncated)",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			CloseHandle( hFile );
			return -1;
		}
#else
		r = fseek( fp, offset, SEEK_SET );
		if (r < 0) {
			uigtk_messagebox( "File read error - 9 (Broken file, truncated)",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
			fclose( fp );
			return -1;
		}
#endif
	}
	tape.writable = _writable;
#ifdef WIN32
	tape.hFile = hFile;
#else
	tape.fp = fp;
#endif
	tape.total = size;
	tape.offset = offset;
	tape.pos = tape.offset;
	tape.have_record = FALSE;
	tape.freq = wvfmt.nSamplesPerSec;
	return 0;
}

static int _mz2tape_exec_mtw( MZ2000 *mz, const char *filename )
{
	u32 datasize, size, offset;
#ifdef WIN32
	HANDLE hFile;
	DWORD r, dwReadSize;
#else
	int r;
	FILE *fp;
#endif

	mz2tape_eject2( mz );
#ifdef WIN32
	hFile = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ,
		NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "MTW file open error",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	size = GetFileSize( hFile, NULL );
	ReadFile( hFile, data_mzhead, TAPE_MZHEADER_SIZE, &dwReadSize, NULL );
	if (dwReadSize != TAPE_MZHEADER_SIZE) {
		MessageBox( NULL, "MTW file read error - 1 (Broken header)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
#else
	fp = fopen( filename, "r" );
	if (!fp) {
		uigtk_messagebox( "MTW file open error",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		return -1;
	}
	r = fseek( fp, 0L, SEEK_END );
	if (r < 0) {
		uigtk_messagebox( "MTW file read error - 0A (FSEEK error)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	size = ftell( fp );
	r = fseek( fp, 0L, SEEK_SET );
	if (r < 0) {
		uigtk_messagebox( "MTW file read error - 0B (FSEEK error)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	if (fread( data_mzhead, 1, TAPE_MZHEADER_SIZE, fp ) != TAPE_MZHEADER_SIZE) {
		uigtk_messagebox( "MTW file read error - 1 (Broken header)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
#endif
	if (data_mzhead[0x00] != 1) {
#ifdef WIN32
		MessageBox( NULL, "MTW file mode error",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
#else
		uigtk_messagebox( "MTW file mode error",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
#endif
		return -1;
	}
	datasize = (u32)((data_mzhead[0x13] * 256) + data_mzhead[0x12]);
	if (data_mzhead[0x20] == 0x36) {
		/* threshold and pitch stored format */
		mz -> tapePitch = (u32)data_mzhead[0x21];
		mz -> tapeThreshold = (u32)data_mzhead[0x22];
		_TRACE( "pitch:%d threshold:%d\n", mz -> tapePitch, mz -> tapeThreshold );
	}
#ifdef WIN32
	ReadFile( hFile, mz -> memory, datasize, &dwReadSize, NULL );
	if (dwReadSize != datasize) {
		MessageBox( NULL, "MTW file read error - 2 (Broken body)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
		return -1;
	}
#else
	if (fread( mz -> memory, 1, datasize, fp ) != datasize) {
		uigtk_messagebox( "MTW file read error - 2 (Broken body)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
#endif
	strncpy( (char *)tape.pathname, filename, TAPE_MAX_PATH );
	tape.pathname[TAPE_MAX_PATH - 1] = '\0';
	offset = TAPE_MZHEADER_SIZE + datasize;
#ifdef WIN32
	r = SetFilePointer( hFile, offset, 0, FILE_BEGIN );
	if (r == INVALID_SET_FILE_POINTER) {
		MessageBox( NULL, "MTW file read error - 3 (FSEEK error)",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseHandle( hFile );
	}
	return _mz2tape_openwave( mz, hFile, offset, size, FALSE );
#else
	r = fseek( fp, offset, SEEK_SET );
	if (r < 0) {
		uigtk_messagebox( "MTW file read error - 3 (FSEEK error)",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	return _mz2tape_openwave( mz, fp, offset, size, FALSE );
#endif
}

static int _mz2tape_open_mti( MZ2000 *mz, const char *filename, int readonly )
{
	int writable;
	unsigned int size;
#ifdef WIN32
	HANDLE hFile;
	DWORD attr;
#else
	int r;
	FILE *fp;
	struct stat stbuf;
#endif
	mz2tape_eject2( mz );
#ifdef WIN32
	attr = GetFileAttributes( filename );
	if (!(attr & FILE_ATTRIBUTE_READONLY))
		writable = TRUE;
	else
		writable = FALSE;
	if (readonly)
		writable = FALSE;
	hFile = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ,
		NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "MTI/WAV file open error",
			"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	size = GetFileSize( hFile, NULL );
	strncpy( (char *)tape.pathname, filename, TAPE_MAX_PATH );
	tape.pathname[TAPE_MAX_PATH - 1] = '\0';
	return _mz2tape_openwave( mz, hFile, 0, size, writable );
#else
	r = access( filename, R_OK | W_OK );
	if (!r) {
		r = stat( filename, &stbuf );
		if (!r) {
			if ((stbuf.st_mode & S_IRUSR) && (stbuf.st_mode & S_IWUSR))
				r = 0;
			else
				r = -1;
		}
	}
	if (!r)
		writable = TRUE;
	else
		writable = FALSE;
	if (readonly)
		writable = FALSE;
	fp = fopen( filename, "r" );
	if (!fp) {
		uigtk_messagebox( "MTI/WAV file open error",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	r = fseek( fp, 0L, SEEK_END );
	if (r < 0) {
		uigtk_messagebox( "MTI/WAV file feek() error - 1",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	size = ftell( fp );
	r = fseek( fp, 0L, SEEK_SET );
	if (r < 0) {
		uigtk_messagebox( "MTI/WAV file fseek() error - 2",
			"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		fclose( fp );
		return -1;
	}
	strncpy( (char *)tape.pathname, filename, TAPE_MAX_PATH );
	tape.pathname[TAPE_MAX_PATH - 1] = '\0';
	return _mz2tape_openwave( mz, fp, 0, size, writable );
#endif
}

static int _mz2tape_exec_chk( const char *filename )
{
	int i, r;
	const char *cp;

	r = MZ2TAPE_FILEMODE_ERROR;
	cp = strrchr( filename, '.' );
	if (!cp)
		return r;
	cp++;
	for (i = 0; _mz2tape_filemodelist[i].ext; i++) {
#ifdef WIN32
		if (!stricmp( cp, _mz2tape_filemodelist[i].ext ))
#else
		if (!strcasecmp( cp, _mz2tape_filemodelist[i].ext ))
#endif
		{
			r = _mz2tape_filemodelist[i].filemode;
			break;
		}
	}
	if (r >= 0) {
#ifdef WIN32
		HANDLE hFile;

		hFile = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ,
			NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
		if (hFile == INVALID_HANDLE_VALUE)
			r = -2;
		else
			CloseHandle( hFile );
#else
		FILE *fp;

		fp = fopen( filename, "r" );
		if (!fp)
			r = -2;
		else
			fclose( fp );
#endif
	}
	return r;
}

int mz2tape_exec( MZ2000 *mz, const char *filename, int readonly )
{
	int filemode, r = MZ2TAPE_FILEMODE_ERROR;

	filemode = _mz2tape_exec_chk( filename );
	if (filemode < 0) {
		if (filemode < -1) {
#ifdef WIN32
			MessageBox( NULL, "File open error",
				"Tape Change", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
#else
			uigtk_messagebox( "File open error",
				"Tape Change", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
#endif
			return MZ2TAPE_FILEMODE_ERROR;
		} else
			return MZ2TAPE_FILEMODE_ERRNDLG;
	}
	/* strcpy( mz->nowPlayFileName, filename ); */
	mz2tape_eject2( mz );
	if (filemode == MZ2TAPE_FILEMODE_MTI || filemode == MZ2TAPE_FILEMODE_WAV) {
		r = _mz2tape_open_mti( mz, filename, readonly );
		if (r >= 0)
			return filemode;
		else
			return r;
	}
	iplreset_mz( mz );
#ifdef WIN32
	Sleep( 100 );
#endif
	switch (filemode) {
	case MZ2TAPE_FILEMODE_MZT :
	case MZ2TAPE_FILEMODE_MZF :
	case MZ2TAPE_FILEMODE_DAT :
	case MZ2TAPE_FILEMODE_HEX :
		r = _mz2tape_exec_data( filemode, mz, filename );
		break;
	case MZ2TAPE_FILEMODE_MTW :
		r = _mz2tape_exec_mtw( mz, filename );
		break;
	default :
		break;
	}
	if (r >= 0) {
		mz -> howipl = 0;
		reset_mz2000( mz );
		screen_setchrsize( mz, FALSE );
		screen_initattr( mz );
		screen_putmsg( mz, 0, "IPL is loading" );
		screen_putmsgn( mz, 15, (char *)&data_mzhead[1], TAPE_IPLMSG_MAXNUM );
		screen_region_add_all( mz );
#ifdef WIN32
		Sleep( 100 );
#else
		usleep( 100 * 1000 );
#endif
		return filemode;
	} else
		return r;
}

void tape_exit( MZ2000 *mz )
{
	mz2tape_eject2( mz );
	return;
}

int tape_init( MZ2000 *mz )
{
	memset( (void *)&tape, 0, sizeof(tape) );
	tape.recorddata = 0x00U;
	mz2tape_eject2( mz );
	return TRUE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
