/* File: sound.cpp

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Sound Modules

	Copyright (C) 2002-2019 FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.

	Cf. Referenced EmuZ-2000 emuz2000pwm.cpp Thanks.
*/

extern "C" {
#include "config.h"
}

#include <windows.h>
#include <dsound.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <io.h>

extern "C" {
#include "common.h"
#include "mz2000.h"
#include "wavefile.h"
}

/* 8bit output values */
#define SOUND_PULSE8_M		0x80U
#define SOUND_PULSE8_VAL	0x60U
/* 16bit output values */
#define SOUND_PULSE16_M		0x0000U	/* only lower=upper=0, because a byte referenced. */
#define SOUND_PULSE16_VAL	0x6000U

/* for Direct Sound - I */
#define SOUND_DSBUFFER_MAXNUM	(1 + 5 + 12)	/* sound buffer each ch.s */
	/* (1 + sizeof(sound_wavedata)/sizeof(SOUND_WAVEDATA)) */
static BOOL dsound_available = FALSE;
static LPDIRECTSOUND lpds = NULL;
static LPDIRECTSOUNDBUFFER lpdsbPRIMARY = NULL;
static LPDIRECTSOUNDBUFFER lpdsb[SOUND_DSBUFFER_MAXNUM];
static DWORD sound_writeindex = 0;
static HANDLE sound_hmutex = NULL;

/* for Direct Sound - II */
static int sound_volume1;
static int sound_volume2;
static int sound_samplingrate;
static int sound_bytes;
static int sound_delaytime;
static int sound_datasize;
static int sound_bufsize;
static int sound_multiply;
static DWORD sound_dsplaybuffer;
static DWORD sound_dsplayoffset;
static DWORD sound_dsplayleftmax;
static DWORD sound_dsplayleftmin;

/* for playdata */
static unsigned char *sound_databuf = NULL;
static unsigned char *sound_attrbuf = NULL;
static unsigned char *sound_playbuf = NULL;
static unsigned char *sound_clearbuf = NULL;
static z80clk sound_baseclock = 0;
static BYTE sound_playdata8  = SOUND_PULSE8_M;
static WORD sound_playdata16 = SOUND_PULSE16_M;
static int sound_lastonoff = 0;
static int sound_lastoffset = 0;
static int g_sound_ct = 0;

/* for status */
int sound_stat_value1;
int sound_stat_value2;
int sound_stat_value3;
/* for sound time interpolations */
static int sound_stat_oldvalue1 = -1;
static int sound_stat_oldvalue2 = -1;

/* for wavefiles */
typedef struct {
	const char *fn;
	char *p;
	int len;
	int device_type;	/* 1 = TAPE / 2 = FD / 3 = QD / 0 = others */
} SOUND_WAVEDATA;

static SOUND_WAVEDATA sound_wavedata[] = {
	{"fddmotor.wav",	NULL, -1, 2},
	{"fddmotor_st.wav",	NULL, -1, 2},
	{"fddeject.wav",	NULL, -1, 2},
	{"fddseek.wav",		NULL, -1, 2},
	{"fddseek1.wav",	NULL, -1, 2},
	{"cmteject2000.wav",	NULL, -1, 1},
	{"cmtplay2000.wav",	NULL, -1, 1},
	{"cmtstop2000.wav",	NULL, -1, 1},
	{"cmtff2000.wav",	NULL, -1, 1},
	{"cmtrew2000.wav",	NULL, -1, 1},
	{"cmtffst2000.wav",	NULL, -1, 1},
	{"cmtapss2000.wav",	NULL, -1, 1},
	{"cmteject80b.wav",	NULL, -1, 1},
	{"cmtplay80b.wav",	NULL, -1, 1},
	{"cmtstop80b.wav",	NULL, -1, 1},
	{"cmtff80b.wav",	NULL, -1, 1},
	{"cmtrew80b.wav",	NULL, -1, 1}
};

void sound_reset( MZ2000 *mz )
{
	int ch = 0;
	DWORD dwBytes1, dwBytes2, dwReadOffset;
	LPVOID lpvPtr1, lpvPtr2;
	HRESULT hr;

	if (!dsound_available)
		return;
	if (!sound_hmutex)
		return;
	WaitForSingleObject( sound_hmutex, INFINITE );
	if (sound_bytes == 1) {
		FillMemory( sound_databuf, sound_bufsize, SOUND_PULSE8_M );
		FillMemory( sound_playbuf, sound_bufsize, SOUND_PULSE8_M );
	} else {
		FillMemory( sound_databuf, sound_bufsize, SOUND_PULSE16_M );
		FillMemory( sound_playbuf, sound_bufsize, SOUND_PULSE16_M );
	}
	FillMemory( sound_attrbuf, sound_bufsize, 0 );
	sound_lastoffset = 0;
	sound_playdata8  = SOUND_PULSE8_M;
	sound_playdata16 = SOUND_PULSE16_M;
	sound_baseclock = get_internal_clock();
	if (lpdsb[ch]) {
		hr = lpdsb[ch] -> Lock( 0, sound_dsplaybuffer, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
		if (hr == DSERR_BUFFERLOST) {
			lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
			sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
			hr = lpdsb[ch] -> Lock( 0, sound_dsplaybuffer, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
		}
		if FAILED(hr) {
			lpdsb[ch] -> GetCurrentPosition( &sound_writeindex, NULL );
			sound_writeindex = (sound_writeindex + sound_dsplayoffset) % sound_dsplaybuffer;
			ReleaseMutex( sound_hmutex );
			return;
		}
		CopyMemory( lpvPtr1, sound_clearbuf, dwBytes1 );
		if (dwBytes2)
			CopyMemory( lpvPtr2, sound_clearbuf + dwBytes1, dwBytes2 );
		if FAILED(lpdsb[ch] -> Unlock( lpvPtr1, dwBytes1, lpvPtr2, dwBytes2 )) {
			;
		}
		lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
		sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
		if (dwReadOffset < sound_writeindex)
			sound_stat_value1 = sound_writeindex - dwReadOffset;
		else
			sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
	}
	ReleaseMutex( sound_hmutex );
	return;
}

void sound_setstate( MZ2000 *mz, BOOL onoff )
{
	int pos, posmod;
	z80clk posll, t;

	if (!dsound_available)
		return;
	if (!mz -> soundmode || mz -> playStop || mz -> pause)
		return;
	if (getspeed_z80() >= DEFAULT_SPEED_KHZ * 20)	/* >= 80MHz */
		return;
	WaitForSingleObject( sound_hmutex, INFINITE );
	posll = (get_internal_clock() - sound_baseclock) * (sound_datasize / sound_bytes);
	t = get_frameclk_z80() * sound_multiply;
	pos = (int)(posll / t);
	/* pos *= sound_bytes; ... because setstate() use pos/2 */
	if (sound_bytes == 1) {
		posmod = (int)((posll % t) * SOUND_PULSE8_VAL / t);
		//if (pos >= sound_bufsize - 10)
		//	_TRACE("pos over\n");
		if (pos > sound_bufsize - 10)
			pos = sound_bufsize - 10;
		if (pos < 0)
			pos = 0;
		if (pos < sound_lastoffset)
			sound_lastoffset -= sound_datasize;
	} else {
		posmod = (int)((posll % t) * SOUND_PULSE16_VAL / t);
		//if (pos >= sound_bufsize - 10)
		//	_TRACE("pos over\n");
		if (pos * 2 > sound_bufsize - 10)
			pos = (sound_bufsize - 10) / 2;
		if (pos < 0)
			pos = 0;
		if (pos < sound_lastoffset)
			sound_lastoffset -= sound_datasize / 2;
	}
#if 0
	{
		static tcnt = 0;

		tcnt = (tcnt + 1) % 200;
		if (!tcnt)
			_TRACE("%d: %d\n", pos - sound_lastoffset, onoff );
	}
#endif
	if (sound_lastonoff != onoff) {
		if (sound_bytes == 1) {
			if (onoff) {
				sound_databuf[pos] = SOUND_PULSE8_M + SOUND_PULSE8_VAL - posmod * 2;
				sound_attrbuf[pos] = 1;
				sound_databuf[pos + 1] = SOUND_PULSE8_M + SOUND_PULSE8_VAL;
				sound_attrbuf[pos + 1] = 1;
			} else {
				sound_databuf[pos] = SOUND_PULSE8_M - SOUND_PULSE8_VAL + posmod * 2;
				sound_attrbuf[pos] = 1;
				sound_databuf[pos + 1] = SOUND_PULSE8_M - SOUND_PULSE8_VAL;
				sound_attrbuf[pos + 1] = 1;
			}
		} else {
			WORD t;

			if (onoff) {
				t = SOUND_PULSE16_VAL - posmod * 2;
				sound_databuf[pos * 2    ] = t & 0xff;
				sound_databuf[pos * 2 + 1] = (t >> 8) & 0xff;
				sound_attrbuf[pos] = 1;
				t = SOUND_PULSE16_VAL;
				sound_databuf[pos * 2 + 2] = t & 0xff;
				sound_databuf[pos * 2 + 3] = (t >> 8) & 0xff;
				sound_attrbuf[pos + 1] = 1;
			} else {
				t = posmod * 2 - SOUND_PULSE16_VAL;
				sound_databuf[pos * 2    ] = t & 0xff;
				sound_databuf[pos * 2 + 1] = (t >> 8) & 0xff;
				sound_attrbuf[pos] = 1;
				t = (WORD)(0 - SOUND_PULSE16_VAL);
				sound_databuf[pos * 2 + 2] = t & 0xff;
				sound_databuf[pos * 2 + 3] = (t >> 8) & 0xff;
				sound_attrbuf[pos + 1] = 1;
			}
		}
	}
	sound_lastoffset = pos;
	sound_lastonoff = onoff;
	ReleaseMutex( sound_hmutex );
	return;
}

void sound_tickprogress( MZ2000 *mz )
{
	int i, j, pos, tpause, ch = 0;
	int n, posdiff, posnct;
	BYTE c, c1, c2, *cp;
	DWORD dwBytes1, dwBytes2, dwReadOffset;
	LPVOID lpvPtr1, lpvPtr2;
	HRESULT hr;
	z80clk posll, tclk;

	if (!dsound_available) {
		sound_stat_oldvalue1 = -1;
		sound_stat_oldvalue2 = -1;
		return;
	}
	g_sound_ct++;
	if (g_sound_ct >= sound_multiply || mz -> playStop || mz -> pause)
		g_sound_ct = 0;
	else
		return;
	WaitForSingleObject( sound_hmutex, INFINITE );
	if (!lpdsb[ch]) {
		ReleaseMutex( sound_hmutex );
		sound_stat_oldvalue1 = -1;
		sound_stat_oldvalue2 = -1;
		return;
	}
	if (!mz -> soundmode || mz -> playStop || mz -> pause
			|| (mz -> howipl > 0 && mz -> howipl != 10))
		tpause = TRUE;
	else
		tpause = FALSE;
	tclk = get_internal_clock();
	if (!tpause) {
		posll = (tclk - sound_baseclock) * (sound_datasize / sound_bytes)
			/ (get_frameclk_z80() * sound_multiply);
		pos = (int)posll * sound_bytes;
	} else
		pos = sound_datasize;
	lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
	if (dwReadOffset < sound_writeindex) {
		if (sound_writeindex - dwReadOffset < sound_dsplayleftmin
				|| sound_writeindex - dwReadOffset > sound_dsplayleftmax)
			sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
	} else {
		if (sound_writeindex + sound_dsplaybuffer - dwReadOffset < sound_dsplayleftmin
				|| sound_writeindex + sound_dsplaybuffer - dwReadOffset > sound_dsplayleftmax)
			sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
	}
	if (dwReadOffset < sound_writeindex)
		sound_stat_value1 = sound_writeindex - dwReadOffset;
	else
		sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
	if (pos > sound_bufsize)
		pos = sound_bufsize;
#if 0	/* fast!! but... */
	if (!pos) {
#if 0
		lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
		sound_writeindex = (dwReadOffset + sound_dsplayoffsete) % sound_dsplaybuffer;
		sound_stat_oldvalue2 = sound_stat_oldvalue1;
		if (dwReadOffset < sound_writeindex)
			sound_stat_value1 = sound_writeindex - dwReadOffset;
		else
			sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
		sound_stat_oldvalue1 = sound_stat_value1;
#endif
		ReleaseMutex( sound_hmutex );
		return;
	}
#endif
	hr = lpdsb[ch] -> Lock( sound_writeindex, pos, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
	if (hr == DSERR_BUFFERLOST) {
		lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
		sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
		sound_stat_oldvalue2 = sound_stat_oldvalue1;
		if (dwReadOffset < sound_writeindex)
			sound_stat_value1 = sound_writeindex - dwReadOffset;
		else
			sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
		sound_stat_oldvalue1 = sound_stat_value1;
		hr = lpdsb[ch] -> Lock( sound_writeindex, pos, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
	}
	if FAILED(hr) {
		ReleaseMutex( sound_hmutex );
		sound_stat_oldvalue1 = -1;
		sound_stat_oldvalue2 = -1;
		return;
	}
	posdiff = 0;
	posnct = 0;
	n = 0;
	if (mz2000_global -> enable_soundtminterpolations) {
		if (sound_stat_oldvalue1 > 0 && sound_stat_oldvalue2 > 0) {
			DWORD tlen = dwBytes1 + dwBytes2;

#if 1	/* absolute */
			posdiff = sound_stat_oldvalue1 - sound_dsplayoffset;
#else	/* relative */
			posdiff = sound_stat_oldvalue1 - sound_stat_oldvalue2;
#endif
			if (posdiff) {
#if 1	/* normals */
				n = (int)(tlen + 2) / abs(posdiff);
#else	/* max=1 */
				n = (int)(tlen + 2) / 2;
#endif
				if (abs(n < 3))
					n = 3;
				if (posdiff < 0)
					n = -n;
			}
		}
#if 0
		_TRACE("oval1=%d oval2=%d sbuf-ofs.=%d diff=%d n=%d dwBytes=%d\n",
			sound_stat_oldvalue1, sound_stat_oldvalue2, sound_dsplayoffset,
			posdiff, n, (int)(dwBytes1+dwBytes2) );
#endif
	}
	if (sound_bytes == 1) {
		if (!tpause) {
			c = sound_playdata8;
			cp = sound_playbuf;
			for (i = 0, j = 0; i < (int)(dwBytes1 + dwBytes2); j++) {
				if (i < sound_bufsize) {
					if (sound_attrbuf[i])
						c = sound_databuf[i];
				} else
					c = SOUND_PULSE8_M;
				if (n && !(j % n) && i < (int)(dwBytes1 + dwBytes2) - 2) {
					if (n >= 0) {
						*cp++ = c;
						*cp++ = c;
						posnct++;
						i += 2;
					} else
						--posnct;
				} else {
					*cp++ = c;
					i++;
				}
			}
			sound_playdata8 = c;
		} else {
			c = SOUND_PULSE8_M;
			cp = sound_playbuf;
			for (i = 0; i < (int)(dwBytes1 + dwBytes2); i++)
				*cp++ = c;
			sound_playdata8 = c;
		}
		pos += posnct;
		if (pos < sound_bufsize  && !tpause) {
			int tt;

			memcpy( sound_databuf, sound_databuf + pos, sound_bufsize - pos );
			memcpy( sound_attrbuf, sound_attrbuf + pos, sound_bufsize - pos );
			tt = sound_bufsize - pos;
			memset( sound_databuf + tt, SOUND_PULSE8_M, sound_bufsize - tt );
			memset( sound_attrbuf + tt, 0, sound_bufsize - tt );
		} else {
			memset( sound_databuf, SOUND_PULSE8_M, sound_bufsize );
			memset( sound_attrbuf, 0, sound_bufsize );
		}
	} else {
		int tpos = pos / 2;

		if (!tpause) {
			c2 = sound_playdata16 & 0xff;		/* lower */
			c1 = (sound_playdata16 >> 8) & 0xff;	/* upper */
			cp = sound_playbuf;
			for (i = 0, j = 0; i < (int)(dwBytes1 + dwBytes2)/2; j+=2 /* sound_bytes */) {
				if (i < sound_bufsize/2) {
					if (sound_attrbuf[i]) {
						c2 = sound_databuf[i * 2];
						c1 = sound_databuf[i * 2 + 1];
					}
				} else {
					c2 = SOUND_PULSE16_M;
					c1 = SOUND_PULSE16_M;
				}
				if (n && !(j % n) && i < (int)(dwBytes1 + dwBytes2)/2 - 2) {
					if (n >= 0) {
						*cp++ = c2;
						*cp++ = c1;
						*cp++ = c2;
						*cp++ = c1;
						posnct += 2;
						i += 2;
					} else
						posnct -= 2;
				} else {
					*cp++ = c2;
					*cp++ = c1;
					i++;
				}
			}
			sound_playdata16 = c2 | (c1 << 8);
		} else {
			c = SOUND_PULSE16_M;
			cp = sound_playbuf;
			for (i = 0; i < (int)(dwBytes1 + dwBytes2); i++)
				*cp++ = c;
			sound_playdata16 = c;
		}
		pos += posnct;
		if (pos < sound_bufsize  && !tpause) {
			int tt, tt2;

			memcpy( sound_databuf, sound_databuf + pos, sound_bufsize - pos );
			memcpy( sound_attrbuf, sound_attrbuf + tpos, sound_bufsize - tpos );
			tt  = sound_bufsize - pos;
			tt2 = sound_bufsize / 2 - tpos;
			memset( sound_databuf + tt, SOUND_PULSE16_M, sound_bufsize - tt );
			memset( sound_attrbuf + tt2, 0, sound_bufsize - tt2 );
		} else {
			memset( sound_databuf, SOUND_PULSE16_M, sound_bufsize );
			memset( sound_attrbuf, 0, sound_bufsize / 2 );
		}
	}
	memcpy( lpvPtr1, sound_playbuf, dwBytes1 );
	if (dwBytes2)
		memcpy( lpvPtr2, sound_playbuf + dwBytes1, dwBytes2 );
	if FAILED(lpdsb[ch] -> Unlock( lpvPtr1, dwBytes1, lpvPtr2, dwBytes2 )) {
		;
	}
	sound_writeindex = (sound_writeindex + dwBytes1 + dwBytes2 - posnct) % sound_dsplaybuffer;
	if (dwReadOffset < sound_writeindex) {
		if (sound_writeindex - dwReadOffset < sound_dsplayleftmin
				|| sound_writeindex - dwReadOffset > sound_dsplayleftmax)
			sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
	} else {
		if (sound_writeindex + sound_dsplaybuffer - dwReadOffset < sound_dsplayleftmin
				|| sound_writeindex + sound_dsplaybuffer - dwReadOffset > sound_dsplayleftmax)
			sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
	}
	sound_stat_oldvalue2 = sound_stat_oldvalue1;
	if (dwReadOffset < sound_writeindex)
		sound_stat_value1 = sound_writeindex - dwReadOffset;
	else
		sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
	sound_stat_oldvalue1 = sound_stat_value1;
	sound_baseclock = tclk;
	ReleaseMutex( sound_hmutex );
	return;
}

static BOOL sound_stopmzsound( MZ2000 *mz )
{
	int ch = 0;
	DWORD stat;

	WaitForSingleObject( sound_hmutex, INFINITE );
	if (lpdsb[ch]) {
		if FAILED(lpdsb[ch] -> GetStatus( &stat )) {
			ReleaseMutex( sound_hmutex );
			return FALSE;
		}
		if (stat & (DSBSTATUS_PLAYING | DSBSTATUS_LOOPING))
			lpdsb[ch] -> Stop();
	}
	sound_writeindex = 0;
	ReleaseMutex( sound_hmutex );
	return TRUE;
}

static BOOL sound_startmzsound( MZ2000 *mz )
{
	int ch = 0;
	DWORD dwReadOffset;
	WAVEFORMATEX wfmt;
	DSBUFFERDESC dsbdesc;

	ZeroMemory( &wfmt, sizeof(wfmt) );
	wfmt.wFormatTag		= WAVE_FORMAT_PCM;
	wfmt . nChannels	= 1;			/* mono/stereo */
	wfmt . nSamplesPerSec	= sound_samplingrate;	/* f */
	wfmt . nBlockAlign	= sound_bytes * 1;	/* one data size */
	wfmt . wBitsPerSample	= sound_bytes * 8;	/* 8bit/16bit */
	wfmt . cbSize		= 0;			/* Extended data size */
	wfmt . nAvgBytesPerSec	= (wfmt . nBlockAlign) * (wfmt . nSamplesPerSec);
	ZeroMemory( &dsbdesc, sizeof(dsbdesc) );
	dsbdesc.dwSize		= sizeof(dsbdesc);
#if DIRECTSOUND_VERSION >= 0x700
	dsbdesc.dwFlags		= DSBCAPS_STATIC | DSBCAPS_GLOBALFOCUS;
#else
	dsbdesc.dwFlags		= DSBCAPS_STATIC | DSBCAPS_CTRLDEFAULT | DSBCAPS_GLOBALFOCUS;
#endif
	dsbdesc.dwBufferBytes	= sound_dsplaybuffer;
	dsbdesc.lpwfxFormat	= &wfmt;
	if FAILED(lpds -> CreateSoundBuffer( &dsbdesc, &lpdsb[ch], NULL )) {
		MessageBox( NULL, "Sound Initialize Error... Use no sound mode.", "SCREEN_CRTC2", MB_OK | MB_ICONEXCLAMATION );
		return FALSE;
	}
	lpdsb[ch] -> SetFrequency( 0 );
	lpdsb[ch] -> SetPan( 0 );
	lpdsb[ch] -> SetVolume( sound_volume1 );
	lpdsb[ch] -> SetCurrentPosition( 0 );
	lpdsb[ch] -> Play( 0, 0, DSBPLAY_LOOPING );
	lpdsb[ch] -> GetCurrentPosition( &dwReadOffset, NULL );
	sound_writeindex = (dwReadOffset + sound_dsplayoffset) % sound_dsplaybuffer;
	if (dwReadOffset < sound_writeindex)
		sound_stat_value1 = sound_writeindex - dwReadOffset;
	else
		sound_stat_value1 = sound_writeindex + sound_dsplaybuffer - dwReadOffset;
	return TRUE;
}

BOOL sound_playwav_file( MZ2000 *mz, SOUND_WAVLIST no )
{
	int ch, len;
	char *p;
	DWORD dwBytes1, dwBytes2, dwOffset;
	LPVOID lpvPtr1, lpvPtr2;
	HRESULT hr;

	if (no < 0 || no >= sizeof(sound_wavedata)/sizeof(SOUND_WAVEDATA))
		return -1;
	if (!sound_wavedata[no].p || sound_wavedata[no].len < 0)
		return -1;
	ch = no + 1;
	if (!lpdsb[ch])
		return -1;
	if (!(mz -> soundmode))
		return 0;
	if (sound_wavedata[no].device_type == 1 && !(mz -> enable_soundtape))
		return 0;
	if (sound_wavedata[no].device_type == 2 && !(mz -> enable_soundfdd))
		return 0;
	len = sound_wavedata[no].len;
	p = sound_wavedata[no].p;
	lpdsb[ch] -> Stop();
	lpdsb[ch] -> SetCurrentPosition( 0 );
	lpdsb[ch] -> GetCurrentPosition( NULL, &dwOffset );
	//_TRACE("play %d woffset=%d len=%d\n", no, dwOffset, len);
	hr = lpdsb[ch] -> Lock( dwOffset, len, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
	if (hr == DSERR_BUFFERLOST) {
		lpdsb[ch] -> Restore();
		hr = lpdsb[ch] -> Lock( dwOffset, len, &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0 );
	}
	if SUCCEEDED(hr) {
		CopyMemory( lpvPtr1, p, dwBytes1 );
		if (dwBytes2)
			CopyMemory( lpvPtr2, p + dwBytes1, dwBytes2 );
		if FAILED(lpdsb[ch] -> Unlock( lpvPtr1, dwBytes1, lpvPtr2, dwBytes2 )) {
			;
		}
		lpdsb[ch] -> Play( 0, 0, 0 );
	}
	return 0;
}

static char *sound_loadwav_file( WAVEDATA_STRUCT *ws_p, const char *fn )
{
	int r;
	unsigned long len;
	char *p, *rp;

	r = wavefile_ropen( WAVEFILE_ATTR_DEFAULT, ws_p, fn );
	if (r < 0) {
		_TRACE( "%s: read open error (result=%d) \"%s\"\n", module_name, r, fn );
		return NULL;
	}
	len = wavedata_getlenbytes( ws_p );
	p = (char *)MALLOC( len );
	if (!p) {
		_TRACE( "%s: memory allocation error (%lu bytes).\n", module_name, len );
		wavefile_rclose( ws_p );
		return NULL;
	}
	rp = p;
	while (len > 0) {
		r = _read( wavedata_getfd(ws_p), p, len );
		if (r > 0) {
			len -= r;
			p += r;
		}
		if (r < 0) {
			_TRACE( "%s: read error r=%d (%s)\n", module_name, r, strerror(errno) );
			wavefile_rclose( ws_p );
			return NULL;
		}
	}
	r = wavefile_rclose( ws_p );
	if (r < 0) {
		_TRACE( "%s: read close error\n", module_name );
		return NULL;
	}
	return rp;
}

static void sound_freewav( void )
{
	int i;
	for (i = 0; i < sizeof(sound_wavedata)/sizeof(SOUND_WAVEDATA); i++) {
		if (sound_wavedata[i].p) {
			/* sound buffers are stopped already */
			FREE( sound_wavedata[i].p );
			sound_wavedata[i].p   = NULL;
			sound_wavedata[i].len = -1;
		}
	}
	return;
}

static BOOL sound_loadwav( MZ2000 *mz )
{
	int i, len, ch;
	char *p, fnbuf[_MAX_PATH];
	WAVEDATA_STRUCT ws;
	WAVEFORMATEX wfmt;
	DSBUFFERDESC dsbdesc;

	for (i = 0; i < sizeof(sound_wavedata)/sizeof(SOUND_WAVEDATA); i++) {
		//_TRACE("%d: %s\n", i, sound_wavedata[i].fn);
		wavedata_initstruct( &ws );
		_snprintf( fnbuf, _MAX_PATH, "%s\\%s", app_currentpath, sound_wavedata[i].fn );
		fnbuf[_MAX_PATH - 1] = '\0';
	        p = sound_loadwav_file( &ws, fnbuf );
		if (!p)
			continue;
		//_TRACE("f=%d ch=%d bytes=%d\n", ws.f, ws.ch, ws.bytes ); 
		ch = i + 1;
		len = wavedata_getlenbytes( &ws );
		ZeroMemory( &wfmt, sizeof(wfmt) );
 		wfmt . wFormatTag	= WAVE_FORMAT_PCM;
		wfmt . nChannels	= ws . ch;		/* mono/stereo */
		wfmt . nSamplesPerSec	= ws . f;		/* f */
		wfmt . nBlockAlign	= ws . bytes * ws . ch;	/* one data size */
		wfmt . wBitsPerSample	= ws . bytes * 8;	/* 8bit/16bit */
		wfmt . cbSize		= 0;			/* extended data size */
		wfmt . nAvgBytesPerSec	= (wfmt . nBlockAlign) * (wfmt . nSamplesPerSec);
		ZeroMemory( &dsbdesc, sizeof(dsbdesc) );
		dsbdesc.dwSize		= sizeof(dsbdesc);
#if DIRECTSOUND_VERSION >= 0x700
		dsbdesc.dwFlags		= DSBCAPS_STATIC | DSBCAPS_GLOBALFOCUS;
#else
		dsbdesc.dwFlags		= DSBCAPS_STATIC | DSBCAPS_CTRLDEFAULT | DSBCAPS_GLOBALFOCUS;
#endif
		dsbdesc.dwBufferBytes	= len;
		dsbdesc.lpwfxFormat	= &wfmt;
		if FAILED(lpds -> CreateSoundBuffer( &dsbdesc, &lpdsb[ch], NULL )) {
			FREE( p );
			continue;
		}
		lpdsb[ch] -> SetFrequency( 0 );
		lpdsb[ch] -> SetPan( 0 );
		lpdsb[ch] -> SetVolume( sound_volume2 );
		sound_wavedata[i].p = p;
		sound_wavedata[i].len = len;
		//_TRACE("ok\n");
	}
	return TRUE;
}

BOOL sound_enablechk( void )
{
	return dsound_available;
}

void sound_exit_directsound( MZ2000 *mz )
{
	int i;

	if (!dsound_available) {
		dsound_available = FALSE;
		if (sound_clearbuf) {
			FREE( sound_clearbuf );
			sound_clearbuf = NULL;
		}
		if (sound_playbuf) {
			FREE( sound_playbuf );
			sound_playbuf = NULL;
		}
		if (sound_attrbuf) {
			FREE( sound_attrbuf );
			sound_attrbuf = NULL;
		}
		if (sound_databuf) {
			FREE( sound_databuf );
			sound_databuf = NULL;
		}
		if (sound_hmutex) {
			CloseHandle( sound_hmutex );
			sound_hmutex = NULL;
		}
		return;
	}
	sound_stopmzsound( mz );
	for (i = 0; i < SOUND_DSBUFFER_MAXNUM; i++) {
		if (lpdsb[i]) {
			lpdsb[i] -> Stop();
			lpdsb[i] -> Release();
			lpdsb[i] = NULL;
		}
	}
	sound_freewav();
	if (lpdsbPRIMARY) {
		lpdsbPRIMARY -> Release();
		lpdsbPRIMARY = NULL;
	}
	if (lpds) {
		lpds -> Release();
		lpds = NULL;
	}
	if (sound_clearbuf) {
		FREE( sound_clearbuf );
		sound_clearbuf = NULL;
	}
	if (sound_playbuf) {
		FREE( sound_playbuf );
		sound_playbuf = NULL;
	}
	if (sound_attrbuf) {
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
	}
	if (sound_databuf) {
		FREE( sound_databuf );
		sound_databuf = NULL;
	}
	return;
}

BOOL sound_init_directsound( HWND hwnd, MZ2000 *mz )
{
	int i, t;
	DSBUFFERDESC dsbdesc;

	dsound_available = FALSE;
	sound_datasize = (sound_samplingrate * sound_multiply * MZ2000_REFRESHRATE)
		/ 1000 * sound_bytes;
	sound_bufsize = (sound_datasize + 512);
	t = (sound_samplingrate * sound_delaytime / 1000) * sound_bytes;
#if 1	/* new buffer values */
	sound_dsplaybuffer = (sound_datasize + t) * 2;
	sound_dsplayleftmax = t + t / 2;
	sound_dsplayleftmin = t - t / 8;
	while (sound_dsplayleftmax % 2 /* max sound_bytes */)
		sound_dsplayleftmax++;
	while (sound_dsplayleftmin % 2 /* max sound_bytes */)
		sound_dsplayleftmin++;
//	sound_dsplayoffset = t;		/* <= Ver.0.95.5 */
	sound_dsplayoffset = (sound_dsplayleftmax + sound_dsplayleftmin) / 2;/* changed for sound time interpolations */
	while (sound_dsplayoffset % 2 /* max sound_bytes */)
		sound_dsplayoffset++;
#else	/* old calcuration algorithms (left for debugs), caution of soundbytes=2 odd values */
	t = sound_datasize;
	sound_dsplaybuffer = t * 4;
	sound_dsplayoffset = t * 2 - t / 4;
	sound_dsplayleftmax = t * 3 + t / 4 * 3;
	sound_dsplayleftmin = t / 2;
#endif

	sound_stat_value1 = 0;
	sound_stat_value2 = sound_dsplayleftmin;
	sound_stat_value3 = sound_dsplayleftmax;

	sound_databuf = (unsigned char *)MALLOC( sound_bufsize );
	if (!sound_databuf)
		return FALSE;
	FillMemory( sound_databuf, sound_bufsize, sound_bytes == 1 ? SOUND_PULSE8_M : SOUND_PULSE16_M );
	sound_attrbuf = (unsigned char *)MALLOC( sound_bufsize );
	if (!sound_attrbuf) {
		FREE( sound_databuf );
		sound_databuf = NULL;
		return FALSE;
	}
	FillMemory( sound_attrbuf, sound_bufsize, 0 );
	sound_playbuf = (unsigned char *)MALLOC( sound_bufsize );
	if (!sound_playbuf) {
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
		FREE( sound_databuf );
		sound_databuf = NULL;
		return FALSE;
	}
	FillMemory( sound_playbuf, sound_bufsize, sound_bytes == 1 ? SOUND_PULSE8_M : SOUND_PULSE16_M );
	sound_clearbuf = (unsigned char *)MALLOC( sound_dsplaybuffer );
	if (!sound_clearbuf) {
		FREE( sound_playbuf );
		sound_playbuf = NULL;
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
		FREE( sound_databuf );
		sound_databuf = NULL;
		return FALSE;
	}
	FillMemory( sound_clearbuf, sound_dsplaybuffer, sound_bytes == 1 ? SOUND_PULSE8_M : SOUND_PULSE16_M );
	if FAILED(DirectSoundCreate( NULL, &lpds, NULL )) {
		FREE( sound_clearbuf );
		sound_clearbuf = NULL;
		FREE( sound_playbuf );
		sound_playbuf = NULL;
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
		FREE( sound_databuf );
		sound_databuf = NULL;
		return FALSE;
	}
	if FAILED(lpds -> SetCooperativeLevel( hwnd, DSSCL_NORMAL )) {
		FREE( sound_clearbuf );
		sound_clearbuf = NULL;
		FREE( sound_playbuf );
		sound_playbuf = NULL;
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
		FREE( sound_databuf );
		sound_databuf = NULL;
		lpds -> Release();
		lpds = NULL;
		return FALSE;
	}
	ZeroMemory( &dsbdesc, sizeof(dsbdesc) );
	dsbdesc.dwSize  = sizeof(dsbdesc);
	dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER;
	if FAILED(lpds -> CreateSoundBuffer( &dsbdesc, &lpdsbPRIMARY, NULL )) {
		FREE( sound_clearbuf );
		sound_clearbuf = NULL;
		FREE( sound_playbuf );
		sound_playbuf = NULL;
		FREE( sound_attrbuf );
		sound_attrbuf = NULL;
		FREE( sound_databuf );
		sound_databuf = NULL;
		lpds -> Release();
		lpds = NULL;
		return FALSE;
	}
	for (i = 0; i < SOUND_DSBUFFER_MAXNUM; i++)
		lpdsb[i] = NULL;
	sound_startmzsound( mz );
	sound_loadwav( mz );
	dsound_available = TRUE;
	return TRUE;
}

BOOL sound_updateparams( HWND hwnd, MZ2000 *mz )
{
	BOOL r;

	WaitForSingleObject( sound_hmutex, INFINITE );
	sound_exit_directsound( mz );
	Sleep( 1000 );
	dsound_available = FALSE;
	sound_volume1 = (DSBVOLUME_MAX - DSBVOLUME_MIN + 1)
		* mz -> sound_volume1 / 100 + DSBVOLUME_MIN; /* MZ pulse sound */
	sound_volume2 = (DSBVOLUME_MAX - DSBVOLUME_MIN + 1)
		* mz -> sound_volume2 / 100 + DSBVOLUME_MIN; /* wave sound effects */
	sound_samplingrate = mz -> sound_freq;
	sound_bytes        = mz -> sound_bit / 8;
	sound_multiply  = (mz -> sound_playbuffer_ms + MZ2000_REFRESHRATE - 1) / MZ2000_REFRESHRATE;
	sound_delaytime = mz -> sound_delaytime_ms;
	r = sound_init_directsound( hwnd, mz );
	ReleaseMutex( sound_hmutex );
	return r;
}

void sound_exit( MZ2000 *mz )
{
	sound_exit_directsound( mz );
	if (sound_hmutex) {
		CloseHandle( sound_hmutex );
		sound_hmutex = NULL;
	}
	return;
}

BOOL sound_init( HWND hwnd, MZ2000 *mz )
{
	dsound_available = FALSE;
	sound_volume1 = (DSBVOLUME_MAX - DSBVOLUME_MIN + 1)
		* mz -> sound_volume1 / 100 + DSBVOLUME_MIN; /* MZ pulse sound */
	sound_volume2 = (DSBVOLUME_MAX - DSBVOLUME_MIN + 1)
		* mz -> sound_volume2 / 100 + DSBVOLUME_MIN; /* wave sound effects */
	sound_samplingrate = mz -> sound_freq;
	sound_bytes        = mz -> sound_bit / 8;
	sound_multiply  = (mz -> sound_playbuffer_ms + MZ2000_REFRESHRATE - 1) / MZ2000_REFRESHRATE;
	sound_delaytime = mz -> sound_delaytime_ms;
	sound_stat_value1 = 0;
	sound_stat_value2 = 50;
	sound_stat_value3 = 100;
	sound_hmutex = CreateMutex( NULL, FALSE, "EmuZ_2000_Sound_Mutex" );
	if (!sound_hmutex)
		return FALSE;
	return sound_init_directsound( hwnd, mz );
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
