/* File: mz1e08f.c

                SHARP MZ-80B/80B2/2000/2200 Emulator for Win32
                         EmuZ-2000  Extend Board DLL
                                << MZ-1E08F >>
                           Main Module [mz1e08f.C]

		Copyright (C) 2017 Toshio Fukui
		Cf. Referenced S.Tsuneoka's MZ-1E08 plugin. Thanks.

		2017/04/07 Ver.0.95.0 First version.
*/

/*
	MZ-1E08 I/O Ports (Printer I/F)
	0xFE	... Control Signals (in/out)
			(In)
			bit 0	- BUSY (RDA#)
			bit 1	- SELECT (STATUS#)
			bit 2	- GND
			bit 3	- GND
			bit 4-7	- <no connect>
			(Out)
			bit 0-5 - <no connect>	// generally, output to 0
			bit 6	- PRIME# (IRT)	// reversable by external switch
			bit 7	- STROBE# (RDP)	// reversable by external switch
			(Out - EmuZ-2000 special)
			0xFF	- end of document/print
	0xFF	... Data (out only)
			bit 0-7 - RD1-RD8

	() ... MZ signal name
*/

#include <windows.h>
#include <stdio.h>

#include "resource.h"

#include "../interface/emuz2000ioBoard.h"
#include "../interface/plugin_ex.h"
#include "PrinterFileDlg.h"

#define CHARCODEFILE	"emuz2000_charcode.txt"

#define MZ1E08_IOBASE		0xfe
#define MZ1E08_IO_CONTROL	MZ1E08_IOBASE
#define MZ1E08_IO_DATA		(MZ1E08_IOBASE + 1)

#define	LNMAX	512

BYTE InportFunction( WORD port );
int OutportFunction( WORD port, BYTE data );

/* Board Information */
EMUZ2000EBIH mz1e08f_ebih = {
	"D257188F-B91C-490C-8F19-CA8D984E106D",
	"MZ-1E08(to File) by SHARP",
	"Printer I/F (to File)",
	"Copyright (C) 2017 Toshio Fukui",
	"http://ja.osdn.net/users/tfukui0/pf/emuz20000/",
	"tfukui0@users.sourceforge.jp",
	0x00000000,	/* reserved */
	InportFunction,
	OutportFunction,
	MZ1E08_IOBASE,
	2,		/* use 2 I/O ports 0xfe-0xff */
	USE_INFODIALOG
};

static HINSTANCE g_hInstance = NULL;
static DWORD g_dwId = 0;

static int printer_reset = FALSE;
static int printer_busy = FALSE;
static BYTE printer_tdata = 0U;
static BYTE printer_data = 0U;
static BYTE printer_predata = 0U;

int enable_nocodeconv = FALSE;
int enable_erasesharpcode = TRUE;
int enable_convcrlf = TRUE;
FILE *output_fp = NULL;
char output_fn[MAX_PATH];

/* code tables from copy and paste */
static unsigned char copypaste_codetable[256 * 2];

/* Check the S-JIS 1st character */
#define iskanji(_X) \
	(((_X) >= 0x80U && (_X) < 0xa0U) || ((_X) >= 0xe0U && (_X) < 0xfdU))

BOOL APIENTRY DllMain( HANDLE hInstance, DWORD dwReasonBeingCalled, LPVOID lpReserved )
{
	UNREFERENCED_PARAMETER( lpReserved );

	switch (dwReasonBeingCalled) {
	case DLL_PROCESS_ATTACH:
		g_hInstance = hInstance;
		break;
	case DLL_PROCESS_DETACH:
		hInstance = NULL;
		break;
	}
	return TRUE;
}

BYTE InportFunction( WORD port )
{
	BYTE c = 0xff;

	switch (port) {
	case MZ1E08_IO_CONTROL :
		c = 0xf0;
		if (printer_reset || printer_busy)
			c |= 0x01;	/* BUSY(RDA#) = H  */
		if (output_fp)
			c |= 0x02;	/* SELECT(STATUS#) = H */
		break;
	case MZ1E08_IO_DATA :
		break;
	}
	return c;
}

static void prtout( unsigned char ch )
{
	if (enable_nocodeconv || !ch || ch == 0x0d || ch == 0x0a) {
		/* direct output (raw) */
		if (ch)
			fputc( ch, output_fp );
	} else {
		/* output widh code convert */
		unsigned char c;

		c = copypaste_codetable[ch * 2];
		if (c)
			fputc( c, output_fp );
		c = copypaste_codetable[ch * 2 + 1];
		if (c)
			fputc( c, output_fp );
	}
	return;
}

int OutportFunction( WORD port, BYTE data )
{
	switch (port) {
	case MZ1E08_IO_CONTROL :
		if (data == 0xff) {
			/* EmuZ-2000 special, end of print */
			if (output_fp) {
				fclose( output_fp );
				output_fp = NULL;
			}
		} else {
			if (data & 0x40)	/* PRIME# -> L (IRT -> H) */
				printer_reset = TRUE;
			else
				printer_reset = FALSE;
			if (!(data & 0x40) && data & 0x80) {	/* STROBE# -> L (RDP -> H) */
				int flag;

				/* output data */
				printer_busy = TRUE;
				printer_predata = printer_data;
				printer_data = printer_tdata;
				flag = enable_erasesharpcode
					&& (printer_data == 0x03 ||printer_data == 0x04	|| printer_data == 0x1b);
				if (output_fp && !flag) {
					if (enable_convcrlf) {
						if (printer_predata != 0x0d && printer_data == 0x0a)
							prtout( 0x0d );
					}
					prtout( printer_data );
				}
			} else
				printer_busy = FALSE;
		}
		break;
	case MZ1E08_IO_DATA :
		printer_tdata = data;
		break;
	}
	return 0;
}

static int copypaste_loadtable( void )		/* S-JIS only */
{
	int i;
	unsigned char c1, c2;
	FILE *fp;

	fp  = fopen( CHARCODEFILE, "rb" );
	if (!fp)
		return -1;
	for (i = 0; i < 256; i++) {
		c1 = getc( fp );
		while (c1 == '\n' || c1 == '\r' || c1 == '\t')
			c1 = getc( fp );
		if (c1 == EOF) {
			fclose( fp );
			return -2;
		}
		c2 = getc( fp );
		if (c2 == EOF) {
			fclose( fp );
			return -3;
		}
		if (c2 == '\n' || c2 == '\r' || c2 == '\t') {
			fclose( fp );
			return -4;
		}
		if (!iskanji(c1) && iskanji(c2)) {
			fclose( fp );
			return -5;
		}
		if (!iskanji(c1)) {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = '\0';
		} else {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = c2;
		}
	}
	fclose( fp );
	return 0;
}

BOOL APIENTRY IntializeBoard( VOID )
{
	int i, t, r;
	char msgbuf[LNMAX];

	enable_nocodeconv = FALSE;
	t = emuz2000_readProfileDword( g_dwId, "NoCodeConvertMode" );
	if (t == 10 || t == 11)
		enable_nocodeconv = t - 10;
	enable_erasesharpcode = TRUE;
	t = emuz2000_readProfileDword( g_dwId, "EraseSharpCodeMode" );
	if (t == 10 || t == 11)
		enable_erasesharpcode = t - 10;
	enable_convcrlf = TRUE;
	t = emuz2000_readProfileDword( g_dwId, "ConvCrlfMode" );
	if (t == 10 || t == 11)
		enable_convcrlf = t - 10;
	t = emuz2000_readProgileString( g_dwId, "OutputFileName", output_fn, sizeof(output_fn) );
	if (!t) {
		GetCurrentDirectory( MAX_PATH, output_fn );
		output_fn[MAX_PATH - 1] = '\0';
		strncat( output_fn, "\\", MAX_PATH - strlen(output_fn) - 1 );
		output_fn[MAX_PATH - 1] = '\0';
	}
	output_fp = NULL;
	printer_tdata = 0U;
	printer_data = 0U;
	printer_predata = 0U;
	printer_reset = FALSE;
	printer_busy  = FALSE;

	/* initialize code table */
	for (i = 0; i < 256; i++) {
		if (i >= ' ' && i < 0x7e)
			copypaste_codetable[i * 2    ] = i;
		else
			copypaste_codetable[i * 2    ] = ' ';
		copypaste_codetable[i * 2 + 1] = '\0';
	}
	r = copypaste_loadtable();
	switch (r) {
	case 0 :
		break;
	case -1 :
		MessageBox( NULL, CHARCODEFILE " Open failed",
			MZ_MESSAGEBOX_TITLE, MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	default :
		_snprintf( msgbuf, LNMAX, CHARCODEFILE " Read error or illegal data (code=%d)", -r );
		msgbuf[LNMAX - 1] = '\0';
		MessageBox( NULL, msgbuf,
			MZ_MESSAGEBOX_TITLE, MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	}
	return TRUE;
}

BOOL APIENTRY ReIntializeBoard( VOID )
{
	printer_reset = FALSE;
	printer_busy  = FALSE;
	return TRUE;
}

VOID APIENTRY FinalizeBoard( VOID )
{
	if (output_fp) {
		fclose( output_fp );
		output_fp = NULL;
	}
	emuz2000_writeProfileDword( g_dwId, "NoCodeConvertMode", enable_nocodeconv + 10 );
	emuz2000_writeProfileDword( g_dwId, "EraseSharpCodeMode", enable_erasesharpcode + 10 );
	emuz2000_writeProfileDword( g_dwId, "ConvCrlfMode", enable_convcrlf + 10 );
	emuz2000_writeProgileString(g_dwId, "OutputFileName", output_fn );
	return;
}

BOOL APIENTRY BoardInfomationService( PEMUZ2000EBIH *info, DWORD id )
{
	*info = &mz1e08f_ebih;
	g_dwId = id;
	return TRUE;
}

VOID APIENTRY ReportCpuClock( int iClock )
{
	return;
}

#ifdef EMUZ2000EX
BOOL WINAPI StateSaveOutputDataEx( int *ver, LPDWORD lpdwSize, LPBYTE *lpbBuff )
{
	*lpdwSize = 0;
	*lpbBuff = NULL;
	return TRUE;
}
#else
BOOL APIENTRY StateSaveOutputData( LPDWORD lpdwSize, LPBYTE lpbBuff )
{
	lpdwSize = 0;
	lpbBuff = NULL;
	return TRUE;
}
#endif

#ifdef EMUZ2000EX
BOOL WINAPI StateSaveInputDataEx( int ver, DWORD dwSize, PBYTE lpbBuff )
{
	return TRUE;
}
#else
BOOL APIENTRY StateSaveInputData( DWORD dwSize, PBYTE lpbBuff )
{
	return TRUE;
}
#endif

VOID APIENTRY InfomationDialog( HWND hWnd )
{
	DialogBox( g_hInstance, MAKEINTATOM(IDD_INFODLG), hWnd, DlgProcPrinterSettings );
	return;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
