/******************************************************************************
*               SHARP MZ-80B/80B2/2000/2200 Emulator for Win32
*                        EmuZ-2000  Extend Board DLL
*                         << MZ-1E24F (to File)  >>
*                          Main Module [mz1e24f.C]
*
*		Copyright (C) 2017 Toshio Fukui
*
*               2017/04/08 Ver0.10 Create new project.
******************************************************************************/

/*
	MZ-1E24/8B03 I/O Ports (RS-232C Serial I/F)
	0xB0	... Z80 SIO A Data	(in/out)
	0xB1	... Z80 SIO A Control	(in/out)
	0xB2	... Z80 SIO B Data	(in/out)
	0xB3	... Z80 SIO B Control	(in/out)

	... and IM 2 interrupt signales are connected.
	    (this plug-in is no support)
*/

/*
	Z80 SIO Initialize Sample from My 64K CP/M BIOS

 388 INITSIO2
 389  A=$18 PORT(C)=A    ; WR0 channel reset
 390  A=$01 PORT(C)=A    ; WR1
 391  A=$00 PORT(C)=A    ; WR1 disable interrupt
 392  A=4   PORT(C)=A    ; WR4
 393  A=$44 PORT(C)=A    ; WR4 x16, 8bit, stop bit 1, non parity
 394  A=5   PORT(C)=A    ; WR5
 395  A=$EA PORT(C)=A    ; WR5 DTR, send 8bit, RTS, start send
 396  A=3   PORT(C)=A    ; WR3
 397  A=$C1 PORT(C)=A    ; WR3 recv 8bit, start recv C1
 398  RET
 399 INITSIO C=SIOACTRL !INITSIO2
 400         C=SIOBCTRL !INITSIO2
*/

#include <windows.h>
#include <stdio.h>

#include "resource.h"

#include "../interface/emuz2000ioBoard.h"
#include "../interface/plugin_ex.h"
#include "SerialDlg.h"

#define CHARCODEFILE	"emuz2000_charcode.txt"

#define MZ1E24_IOBASE	0xb0
#define Z80SIOA_DATA	MZ1E24_IOBASE
#define Z80SIOA_CTRL	(MZ1E24_IOBASE + 1)
#define Z80SIOB_DATA	(MZ1E24_IOBASE + 2)
#define Z80SIOB_CTRL	(MZ1E24_IOBASE + 3)

#define Z80SIO_CH_MAXNUM	2
#define Z80SIO_CH_A		0
#define Z80SIO_CH_B		1
#define Z80SIO_WR_MAXNUM	8	// WR0 - WR7
#define Z80SIO_RR_MAXNUM	3	// RR0 - RR2

#define SIO_RECV_WAITCT	5
#define SIO_SEND_WAITCT	5

#define	LNMAX	512

#ifdef WIN32
#ifdef _DEBUG
static void _TRACE( char *s, ... )
{
	static FILE *fp = NULL;
	char ___debug_buf[LNMAX];
	va_list arg_ptr;

	va_start( arg_ptr, s );
	_vsnprintf(___debug_buf, LNMAX, s, arg_ptr);
	___debug_buf[LNMAX - 1] = '\0';
	va_end( arg_ptr );
	//OutputDebugString(___debug_buf);
	if (!fp)
		fp = fopen( "MZ-1E24f_tracelog.txt", "w" );
	fputs( ___debug_buf, fp );
	fflush( fp );
	return;
}
#else
static void _TRACE( char *s, ... )
{
	return;
}
#endif /* _DEBUG */
#else
#ifdef NDEBUG
#define _TRACE(X...)
#else
static inline int _TRACE_0( const char *file, const char *func, int line, const char *s, ... )
{
	char ___msg_buf[LNMAX];
	va_list arg_ptr;

	printf("%s(%s/L=%d):", file, func, line);
#ifdef USE_PROFILER
	profiler_timedisp( stdout, &profiler_debug_timectxt );
	printf(": ");
#endif
	va_start( arg_ptr, s );
	vsnprintf(___msg_buf, LNMAX, s, arg_ptr);
	va_end( arg_ptr );
	printf("%s", ___msg_buf);
	fflush(stdout);
	return TRUE;
}
#if defined(__STDC_VERSION__) && __STDC_VERSION__ >= 199901L
#define _TRACE(_X...) _TRACE_0(__FILE__, __func__, __LINE__, _X)
#elif defined(__GNUC__)
#define _TRACE(_X...) _TRACE_0(__FILE__, __PRETTY_FUNCTION__, __LINE__, _X)
#else
#define _TRACE(_X...) _TRACE_0(__FILE__, "", __LINE__, _X)
#endif
#endif /* NDEBUG */
#endif /* WIN32 */

BYTE InportFunction( WORD port );
int OutportFunction( WORD port, BYTE data );

/* Board Information */
EMUZ2000EBIH mz1e24f_ebih = {
	"1087FD7C-9AA6-471B-9371-CAA3B1B60B31",
	"MZ-1E24(to File) by SHARP",
	"RS-232C Serial I/F",
	"Copyright (C) 2017 Toshio Fukui",
	"http://ja.osdn.net/users/tfukui0/pf/emuz20000/",
	"tfukui0@users.sourceforge.jp",
	0x00000000,	/* reserved */
	InportFunction,
	OutportFunction,
	MZ1E24_IOBASE,
	4,		/* use 4 I/O ports 0xb0-0xb3 */
	USE_INFODIALOG
};

static HINSTANCE g_hInstance = NULL;
static DWORD g_dwId = 0;

static unsigned char sio_wr[Z80SIO_CH_MAXNUM][Z80SIO_WR_MAXNUM];
static unsigned char sio_rr[Z80SIO_CH_MAXNUM][Z80SIO_RR_MAXNUM];
static int sio_recv_ct[Z80SIO_CH_MAXNUM];
static int sio_send_ct[Z80SIO_CH_MAXNUM];
static int sio_recvchar_have = FALSE;
static unsigned char sio_recvchar = 0x00;

int enable_nocodeconv = FALSE;
int enable_erasectrlchars = FALSE;
int enable_eoftochar = FALSE;
int file_channel = Z80SIO_CH_A;
FILE *input_fp = NULL;
char input_fn[MAX_PATH];
FILE *output_fp = NULL;
char output_fn[MAX_PATH];

/* code tables from copy and paste */
static unsigned char copypaste_codetable[256 * 2];

/* Check the S-JIS 1st character */
#define iskanji(_X) \
	(((_X) >= 0x80U && (_X) < 0xa0U) || ((_X) >= 0xe0U && (_X) < 0xfdU))

BOOL APIENTRY DllMain( HANDLE hInstance, DWORD dwReasonBeingCalled, LPVOID lpReserved )
{
	UNREFERENCED_PARAMETER( lpReserved );

	switch (dwReasonBeingCalled) {
	case DLL_PROCESS_ATTACH:
		g_hInstance = hInstance;
		break;
	case DLL_PROCESS_DETACH:
		hInstance = NULL;
		break;
	}
	return TRUE;
}

static int copypaste_paste_conv( unsigned char ch1, unsigned char ch2 )
{
	int i;
	unsigned char c1, c2;

	if (!ch1 && ch2 == 0x0a)	/* LF */
		return 0x0d;
	if (!ch1 && ch2 == 0x0d)	/* CR */
		return -1;
	if (!ch1 && ch2 == 0x09)	/* TAB */
		return 0x1a;
	if (!ch1 && ch2 == ' ')		/* SPACE */
		return 0x20;
	if (!ch1 && !ch2)		/* '\0' */
		return -1;
	for (i = 0; i < 256; i++) {
		c1 = copypaste_codetable[i * 2 + 1];
		c2 = copypaste_codetable[i * 2];
		if (c1 == ch1 && c2 == ch2)
			return i != 0 ? i : -1;
	}
	return -1;
}

int recvdata_reset( void )
{
	sio_recvchar_have = FALSE;
	sio_recvchar = 0x00;
	return 0;
}

static int recvdata_readchar( void )
{
	int r;
	unsigned char ch;

	if (sio_recvchar_have)
		return 0;
	if (!input_fp) {
		if (enable_eoftochar) {
			sio_recvchar_have = TRUE;
			sio_recvchar = 0x1a;
			return 0;
		} else
			return -1;
	}
	while (1) {
		r = fgetc( input_fp );
		if (r == EOF) {
			if (enable_eoftochar) {
				sio_recvchar_have = TRUE;
				sio_recvchar = 0x1a;
				return 0;
			} else
				return -1;
		}
		ch = r;
		if (enable_erasectrlchars && (ch != '\t' && ch != '\r' && ch != '\n'
				&& ch != 0x1a && ch != 0x1b && ch < 0x20)) {
			/* ignore character */
			continue;
		}
		break;
	}
	sio_recvchar_have = TRUE;
	sio_recvchar = r;
	return 0;
}

static unsigned char recvdata( void )
{
	int r;
	unsigned char ch, ch1, ch2;

	recvdata_readchar();
	if (!sio_recvchar_have)
		return 0x00;
	ch = sio_recvchar;
	sio_recvchar_have = FALSE;
	if (enable_nocodeconv || !ch || ch == '\t' || ch == '\r' || ch == '\n' || ch == 0x1a || ch == 0x1b) {
		/* direct input (raw) */
		recvdata_readchar();	/* read for next char */
		return ch;
	}
	ch1 = ch;
	if (iskanji(ch1)) {
		recvdata_readchar();
		if (!sio_recvchar_have)
			return 0x00;
		ch2 = sio_recvchar;
		sio_recvchar_have = FALSE;
	} else
		ch2 = 0;
	recvdata_readchar();	/* read for next char */
	r = copypaste_paste_conv( ch2, ch1 );
	if (r < 0 || r > 255)
		return 'x';
	return r;
}

static void senddata( unsigned char ch )
{
	if (enable_erasectrlchars && (ch != '\t' && ch != '\r' && ch != '\n'
			&& ch != 0x1a && ch != 0x1b && ch < 0x20)) {
		/* ignore character */
		return;
	}
	if (enable_nocodeconv || !ch || ch == '\t' || ch == '\r' || ch == '\n' || ch == 0x1a || ch == 0x1b) {
		/* direct output (raw) */
		fputc( ch, output_fp );
	} else {
		/* output widh code convert */
		unsigned char c;

		c = copypaste_codetable[ch * 2];
		if (c)
			fputc( c, output_fp );
		c = copypaste_codetable[ch * 2 + 1];
		if (c)
			fputc( c, output_fp );
	}
	return;
}

BYTE InportFunction( WORD port )
{
	int ch, rr;
	BYTE c = 0xff;

	switch (port) {
	case Z80SIOA_CTRL :
	case Z80SIOB_CTRL :
		ch = (port == Z80SIOA_CTRL) ? Z80SIO_CH_A : Z80SIO_CH_B;
		rr = sio_wr[ch][0] & 0x03;
		switch (rr) {
		case 0 :	/* RR0 */
			if (!(sio_rr[ch][0] & 0x01)) {	/* RR0 bit 0 */
				/* receive data not ready */
				if (sio_recv_ct[ch] > 0)
					--sio_recv_ct[ch];
				if (sio_recv_ct[ch] <= 0) {
					sio_recv_ct[ch] = 0;
					if (file_channel == ch && input_fp) {
						if (!sio_recvchar_have)
							recvdata_readchar();
						if (sio_recvchar_have)
							sio_rr[ch][0] |= 0x01; /* set receive data ready */
					} else {
						if (enable_eoftochar) {
							if (!sio_recvchar_have)
								recvdata_readchar();
							if (sio_recvchar_have)
								sio_rr[ch][0] |= 0x01; /* set receive data ready */
						}
					}
				}
			}
			if (!(sio_rr[ch][0] & 0x04)) {	/* RR0 bit 2 */
				/* transfer buffer not empty */
				if (sio_send_ct[ch] > 0)
					--sio_send_ct[ch];
				if (sio_send_ct[ch] <= 0) {
					sio_send_ct[ch] = 0;
					sio_rr[ch][0] |= 0x04; /* set transfer buffer empty */
				}
			}
			c = sio_rr[ch][0];
			break;
		case 1 :	/* RR1 */
			c = sio_rr[ch][1];
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 2 :	/* RR2 (only Ch.B) */
			if (ch == Z80SIO_CH_B) {
				c = sio_rr[ch][2];
			}
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		default :
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		}
		break;
	case Z80SIOA_DATA :
	case Z80SIOB_DATA :
		ch = (port == Z80SIOA_DATA) ? Z80SIO_CH_A : Z80SIO_CH_B;
		if (!(sio_rr[ch][0] & 0x01)) {	/* RR0 bit 0 */
			/* receive data not ready */
			c = 0;
			_TRACE("recvdata: 00 (not ready)\n");
			break;
		}
		if (file_channel == ch) {
			c = recvdata();
			_TRACE("recvdata: %02x\n", c);
		} else {
			if (enable_eoftochar) {
				c = 0x1a;
				_TRACE("recvdata: 1a (file channel error)\n");
			} else {
				c = 0;
				_TRACE("recvdata: 00 (file channel error)\n");
			}
		}
		sio_rr[ch][0] &= ~0x01;		/* reset receive data ready */
		sio_recv_ct[ch] = SIO_RECV_WAITCT; 
		break;
	default :
		break;
	}
	return c;
}

int OutportFunction( WORD port, BYTE data )
{
	int ch, rr;

	switch (port) {
	case Z80SIOA_CTRL :
	case Z80SIOB_CTRL :
		ch = (port == Z80SIOA_CTRL) ? Z80SIO_CH_A : Z80SIO_CH_B;
		rr = sio_wr[ch][0] & 0x03;
		switch (rr) {
		case 0 :	/* WR0 */
			sio_wr[ch][0] = data;
			break;
		case 1 :	/* WR1 */
			sio_wr[ch][1] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 2 :	/* WR2 (only Ch.B) */
			if (ch == Z80SIO_CH_B) {
				sio_wr[ch][2] = data;
			}
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 3 :	/* WR3 */
			sio_wr[ch][3] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 4 :	/* WR4 */
			sio_wr[ch][4] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 5 :	/* WR5 */
			sio_wr[ch][5] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 6 :	/* WR6 */
			sio_wr[ch][6] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		case 7 :	/* WR7 */
			sio_wr[ch][7] = data;
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		default :
			sio_wr[ch][0] &= ~0x03;		/* Next REGS = RR0/WR0 */
			break;
		}
		break;
	case Z80SIOA_DATA :
	case Z80SIOB_DATA :
		ch = (port == Z80SIOA_DATA) ? Z80SIO_CH_A : Z80SIO_CH_B;
		if (!(sio_rr[ch][0] & 0x04)) {	/* RR0 bit 2 */
			/* transfer buffer not empty */
			break;
		}
		if (file_channel == ch && output_fp) {
			_TRACE("senddata: %02x\n", data);
			senddata( data );
		} else {
			_TRACE("senddata: %02x (file channel error)\n", data);
		}
		sio_rr[ch][0] &= ~0x04;		/* reset transfer buffer empty */
		sio_send_ct[ch] = SIO_SEND_WAITCT; 
		break;
	default :
		break;
	}
	return 0;
}

static int copypaste_loadtable( void )		/* S-JIS only */
{
	int i;
	unsigned char c1, c2;
	FILE *fp;

	fp  = fopen( CHARCODEFILE, "rb" );
	if (!fp)
		return -1;
	for (i = 0; i < 256; i++) {
		c1 = getc( fp );
		while (c1 == '\n' || c1 == '\r' || c1 == '\t')
			c1 = getc( fp );
		if (c1 == EOF) {
			fclose( fp );
			return -2;
		}
		c2 = getc( fp );
		if (c2 == EOF) {
			fclose( fp );
			return -3;
		}
		if (c2 == '\n' || c2 == '\r' || c2 == '\t') {
			fclose( fp );
			return -4;
		}
		if (!iskanji(c1) && iskanji(c2)) {
			fclose( fp );
			return -5;
		}
		if (!iskanji(c1)) {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = '\0';
		} else {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = c2;
		}
	}
	fclose( fp );
	return 0;
}

BOOL APIENTRY IntializeBoard( VOID )
{
	int i, t, r;
	char msgbuf[LNMAX];

	enable_nocodeconv = FALSE;
	t = emuz2000_readProfileDword( g_dwId, "NoCodeConvertMode" );
	if (t == 10 || t == 11)
		enable_nocodeconv = t - 10;
	enable_erasectrlchars = FALSE;
	t = emuz2000_readProfileDword( g_dwId, "EraseCtrlCharactersMode" );
	if (t == 10 || t == 11)
		enable_erasectrlchars = t - 10;
	enable_eoftochar = FALSE;
	t = emuz2000_readProfileDword( g_dwId, "EOFtoCharacterMode" );
	if (t == 10 || t == 11)
		enable_eoftochar = t - 10;

	t = emuz2000_readProgileString( g_dwId, "InputFileName", input_fn, sizeof(input_fn) );
	if (!t) {
		GetCurrentDirectory( MAX_PATH, input_fn );
		input_fn[MAX_PATH - 1] = '\0';
		strncat( input_fn, "\\", MAX_PATH - strlen(input_fn) - 1 );
		input_fn[MAX_PATH - 1] = '\0';
	}
	t = emuz2000_readProgileString( g_dwId, "OutputFileName", output_fn, sizeof(output_fn) );
	if (!t) {
		GetCurrentDirectory( MAX_PATH, output_fn );
		output_fn[MAX_PATH - 1] = '\0';
		strncat( output_fn, "\\", MAX_PATH - strlen(output_fn) - 1 );
		output_fn[MAX_PATH - 1] = '\0';
	}
	file_channel = Z80SIO_CH_A;
	t = emuz2000_readProfileDword( g_dwId, "SelectFileChannel" );
	if (t == Z80SIO_CH_A + 10 || t == Z80SIO_CH_B + 10)
		file_channel = t - 10;

	/* initialize file */
	input_fp = NULL;
	output_fp = NULL;

	/* initialize Z80 SIO */
	ReIntializeBoard();

	/* initialize code table */
	for (i = 0; i < 256; i++) {
		if (i >= ' ' && i < 0x7e)
			copypaste_codetable[i * 2    ] = i;
		else
			copypaste_codetable[i * 2    ] = ' ';
		copypaste_codetable[i * 2 + 1] = '\0';
	}
	r = copypaste_loadtable();
	switch (r) {
	case 0 :
		break;
	case -1 :
		MessageBox( NULL, CHARCODEFILE " Open failed",
			MZ_MESSAGEBOX_TITLE, MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	default :
		_snprintf( msgbuf, LNMAX, CHARCODEFILE " Read error or illegal data (code=%d)", -r );
		msgbuf[LNMAX - 1] = '\0';
		MessageBox( NULL, msgbuf,
			MZ_MESSAGEBOX_TITLE, MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	}
	_TRACE("MZ-1E24 InitializeBoard()\n");
	return TRUE;
}

BOOL APIENTRY ReIntializeBoard( VOID )
{
	int i, ch;

	/* initialize Z80 SIO */
	memset( sio_wr, 0, sizeof(sio_wr) );
	memset( sio_rr, 0, sizeof(sio_rr) );
	for (i = 0; i < Z80SIO_CH_MAXNUM; i++) {
		ch = i;
		sio_send_ct[ch] = 0;
		sio_rr[ch][0] |= 0x02;	/* set transfer buffer empty */
		sio_recv_ct[ch] = 0;
		sio_rr[ch][0] &= ~0x01;	/* reset receive data ready */
	}
	recvdata_reset();
	return TRUE;
}

VOID APIENTRY FinalizeBoard( VOID )
{
	if (input_fp) {
		fclose( input_fp );
		input_fp = NULL;
	}
	if (output_fp) {
		fclose( output_fp );
		output_fp = NULL;
	}
	emuz2000_writeProfileDword( g_dwId, "NoCodeConvertMode", enable_nocodeconv + 10 );
	emuz2000_writeProfileDword( g_dwId, "EraseCtrlCharactersMode", enable_erasectrlchars + 10 );
	emuz2000_writeProfileDword( g_dwId, "EOFtoCharacterMode", enable_eoftochar + 10 );
	emuz2000_writeProgileString( g_dwId, "InputFileName", input_fn );
	emuz2000_writeProgileString( g_dwId, "OutputFileName", output_fn );
	emuz2000_writeProfileDword( g_dwId, "SelectFileChannel", file_channel + 10 );
	return;
}

BOOL APIENTRY BoardInfomationService( PEMUZ2000EBIH *info, DWORD id )
{
	*info = &mz1e24f_ebih;
	g_dwId = id;
	return TRUE;
}

VOID APIENTRY ReportCpuClock( int iClock )
{
	return;
}

#ifdef EMUZ2000EX
BOOL WINAPI StateSaveOutputDataEx( int *ver, LPDWORD lpdwSize, LPBYTE *lpbBuff )
{
	*lpdwSize = 0;
	*lpbBuff = NULL;
	return TRUE;
}
#else
BOOL APIENTRY StateSaveOutputData( LPDWORD lpdwSize, LPBYTE lpbBuff )
{
	lpdwSize = 0;
	lpbBuff = NULL;
	return TRUE;
}
#endif

#ifdef EMUZ2000EX
BOOL WINAPI StateSaveInputDataEx( int ver, DWORD dwSize, PBYTE lpbBuff )
{
	return TRUE;
}
#else
BOOL APIENTRY StateSaveInputData( DWORD dwSize, PBYTE lpbBuff )
{
	return TRUE;
}
#endif

VOID APIENTRY InfomationDialog( HWND hWnd )
{
	DialogBox( g_hInstance, MAKEINTATOM(IDD_INFODLG), hWnd, DlgProcSerialSettings );
	return;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
