/* File: z80operand.c

	SHARP MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Z-80 Operands Modules

	Combine the emz2000 z80operand.c, EmuZ-2000 z80operand.c,
		and modified by FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include <stdio.h>
#include <stdlib.h>

#ifndef WIN32
#include "config.h"
#include "common.h"
#else
#include "../mz2000/config.h"
#include "../mz2000/common.h"
#endif

#include "z80.h"
#include "z80operand.h"

#define NOMEM	0xff
#define NOIO	0xff

u8 memread8(u16 addr)
{
	int segment = addr >> 12;
	int v;

	if (z80.mem[segment].readf != NULL)
		v = z80.mem[segment].readf(addr);
	else if (z80.mem[segment].ptr != NULL)
		v = z80.mem[segment].ptr[addr & 0xfff];
	else
		v = NOMEM;
	return v;
}

u16 memread16(u16 addr)
{
	int segment1 = addr >> 12;
	int segment2 = (addr+1) >> 12;
	int vh, vl;

	if (z80.mem[segment1].readf != NULL)
		vl = z80.mem[segment1].readf(addr);
	else if (z80.mem[segment1].ptr != NULL)
		vl = z80.mem[segment1].ptr[addr & 0xfff];
	else
		vl = NOMEM;

	if (z80.mem[segment2].readf != NULL)
		vh = z80.mem[segment2].readf(addr+1);
	else if (z80.mem[segment2].ptr != NULL)
		vh = z80.mem[segment2].ptr[(addr+1) & 0xfff];
	else
		vh = NOMEM;
	return (vh << 8) + vl;
}

int memwrite8(u16 addr, u8 value)
{
	int segment = addr >> 12;

	if (z80.mem[segment].writef != NULL)
		z80.mem[segment].writef(addr, value);
	else if (z80.mem[segment].ptr != NULL)
		z80.mem[segment].ptr[addr & 0xfff] = value;
	return value;
}

int memwrite16(u16 addr, u16 value)
{
	int segment1 = addr >> 12;
	int segment2 = (addr+1) >> 12;
	int vh = value >> 8;
	int vl = value & 0xff;

	if (z80.mem[segment1].writef != NULL)
		z80.mem[segment1].writef(addr, vl);
	else if (z80.mem[segment1].ptr != NULL)
		z80.mem[segment1].ptr[addr & 0xfff] = vl;

	if (z80.mem[segment2].writef != NULL)
		z80.mem[segment2].writef(addr+1, vh);
	else if (z80.mem[segment2].ptr != NULL)
		z80.mem[segment2].ptr[(addr+1) & 0xfff] = vh;
	return value;
}

u8 ioread(u16 port)
{
	int v;

#ifndef IO16BIT
	port &= 0xff;
#endif
	if (z80.io[port].readf != NULL)
		v = z80.io[port].readf(port);
	else
		v = NOIO;
	return v;
}

int iowrite(u16 port, u8 value)
{
#ifndef IO16BIT
	port &= 0xff;
#endif
	if (z80.io[port].writef != NULL)
		z80.io[port].writef(port, value);
	return 0;
}

/* executing instruction except for DDCB, FDCB */
int exec_instruction(int group)
{
	int code;
	int clk;

	fetchPC(code);
	clk = z80.instructs[group][code]();
	wrtReg(R, (readReg(R)+1)&0x7f);
	return clk;
}

/* executing instruction DDCB, FDCB */
int exec_instruction2(int group)
{
	int code = memread8(readReg(PC)+1);
	int clk;

	clk = z80.instructs[group][code]();
	wrtReg(R, (readReg(R)+1)&0x7f);
	fetchPC(code);
	return clk;
}

int exec_interrupt(int onbus)
{
	int clk = 0;
	u16 regI;

	if (z80.external_signal & RESET_SIGNAL) {
		wrtReg(PC,0);
		wrtReg(R,0); wrtReg(I,0);
		resIFF1(); resIFF2(); wrtIM(0);
		resHALT();
		clk = 0;
		z80.external_signal = NO_SIGNAL;
	} else if (z80.external_signal & NMI_SIGNAL) {
		copyIFF1toIFF2();
		resIFF1();
		CALL_N(0x66);
		clk = 11;
		z80.external_signal = NO_SIGNAL;
	} else {
		if (bitIFF1()) {
			if (readIM() == 2) {
				regI = ((readI()*256)+onbus);
				CALL_N(memread16(regI));
				clk = 13;
			} else if (readIM() == 1) {
				CALL_N(0x38);
				clk = 13;
			} else {
				clk = z80.instructs[GROUP_XX][onbus]();
				wrtReg(R, (readReg(R)+1)&0x7f);
			}
			resIFF1();
			resIFF2();
			z80.external_signal = NO_SIGNAL;
		}
	}
	return clk;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
