/* File: copypaste.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Copy and Paste

	Copyright (C) 2002-2019 FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "config.h"

#include <windows.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <io.h>

#include "common.h"
#include "mz2000.h"
#include "util.h"

#define SCREENCOPYBUFF_MAX	8192	/* >= (80+2)*25 * sizeof(SHORT) + 1 */

/* Check the S-JIS 1st character */
#define iskanji(_X) \
	(((_X) >= 0x80U && (_X) < 0xa0U) || ((_X) >= 0xe0U && (_X) < 0xfdU))

/* Paste Keyboard Modifier Flags */
#define KNONE		0x0000U		/* null key (this code is F1. It's ignored.) */
#define KSHIFT		0x8000U		/* modifier keys SHIFT */
#define KGRAPH		0x4000U		/* modifier keys GRPH lock */
#define KKANA		0x2000U		/* modifier keys KANA lock */
#define KSPCR		0x0100U		/* special input 'carridge return' mark */

/* Paste Keyboard Table for MONITOR MZ-1Z001M/MZ-2Z004A/SB-1520/SB-1521
	(ordered by ascii code)

	Value: (Keystrobe << 4) | Bit | ModifierFlags
*/
static unsigned short copypaste_pastekbddata[256] = {
	0x0000U,				// 0x00
	0x0000U, 0x0000U, 0x0000U,		// 0x01-0x04 D,U,R,L
	0x0000U,
	0x0000U, 0x0000U,			// 0x05-0x06 HOME,CLR
	0x0000U,
	0x0000U, 0x0000U, 0x0000U, 0x0000U,	// 0x08
	0x0000U,
	0x32,					// 0x0D CR
	0x0000U, 0x0000U,

	0x0000U, 0x0000U, 0x0000U, 0x0000U,	// 0x10
	0x0000U, 0x0000U, 0x0000U, 0x0000U,
	0x0000U, 0x0000U,			// 0x18
	0x30,					// 0x1A TAB
				   0x0000U,
	0x0000U, 0x0000U,
	0x0000U,		// 0x1E square
	0x0000U,		// 0x1F checkerflag

	0x31,			// 0x20 SPACE
	0x81 | KSHIFT,		// 0x21 !
	0x82 | KSHIFT,		// 0x22 "
	0x83 | KSHIFT,		// 0x23 #
	0x84 | KSHIFT,		// 0x24 $
	0x85 | KSHIFT,		// 0x25 %
	0x86 | KSHIFT,		// 0x26 &
	0x87 | KSHIFT,		// 0x27 '
	0x90 | KSHIFT,		// 0x28 (
	0x91 | KSHIFT,		// 0x29 )
	0x92 | KSHIFT,		// 0x2A *
	0x93 | KSHIFT,		// 0x2B +
	0x77,			// 0x2C ,
	0x94,			// 0x2D -
	0x76,			// 0x2E .
	0x40,			// 0x2F /

	0x80,			// 0x30 0
	0x81,			// 0x31 1
	0x82,			// 0x32 2
	0x83,			// 0x33 3
	0x84,			// 0x34 4
	0x85,			// 0x35 5
	0x86,			// 0x36 6
	0x87,			// 0x37 7
	0x90,			// 0x38 8
	0x91,			// 0x39 9
	0x92,			// 0x3A :
	0x93,			// 0x3B ;
	0x77 | KSHIFT,		// 0x3C <
	0x94 | KSHIFT,		// 0x3D =
	0x76 | KSHIFT,		// 0x3E >
	0x75,			// 0x3F ?

	0x95,			// 0x40 @
	0x41,			// 0x41 A
	0x42,			// 0x42 B
	0x43,			// 0x43 C
	0x44,			// 0x44 D
	0x45,			// 0x45 E
	0x46,			// 0x46 F
	0x47,			// 0x47 G
	0x50,			// 0x48 H
	0x51,			// 0x49 I
	0x52,			// 0x4A J
	0x53,			// 0x4B K
	0x54,			// 0x4C L
	0x55,			// 0x4D M
	0x56,			// 0x4E N
	0x57,			// 0x4F O

	0x60,			// 0x50 P
	0x61,			// 0x51 Q
	0x62,			// 0x52 R
	0x63,			// 0x53 S
	0x64,			// 0x54 T
	0x65,			// 0x55 U
	0x66,			// 0x56 V
	0x67,			// 0x57 W
	0x70,			// 0x58 X
	0x71,			// 0x59 Y
	0x72,			// 0x5A Z
	0x96,			// 0x5B [
	0x74,			// 0x5C Reverse Slash
	0xa0,			// 0x5D ]
	0x73,			// 0x5E ^
	0x80 | KSHIFT,		// 0x5F _(upper line)

	0x95 | KSHIFT,		// 0x60 `
	0x41 | KSHIFT,		// 0x61 a
	0x42 | KSHIFT,		// 0x62 b
	0x43 | KSHIFT,		// 0x63 c
	0x44 | KSHIFT,		// 0x64 d
	0x45 | KSHIFT,		// 0x65 e
	0x46 | KSHIFT,		// 0x66 f
	0x47 | KSHIFT,		// 0x67 g
	0x50 | KSHIFT,		// 0x68 h
	0x51 | KSHIFT,		// 0x69 i
	0x52 | KSHIFT,		// 0x6A j
	0x53 | KSHIFT,		// 0x6B k
	0x54 | KSHIFT,		// 0x6C l
	0x55 | KSHIFT,		// 0x6D m
	0x56 | KSHIFT,		// 0x6E n
	0x57 | KSHIFT,		// 0x6F o

	0x60 | KSHIFT,		// 0x70 p
	0x61 | KSHIFT,		// 0x71 q
	0x62 | KSHIFT,		// 0x72 r
	0x63 | KSHIFT,		// 0x73 s
	0x64 | KSHIFT,		// 0x74 t
	0x65 | KSHIFT,		// 0x75 u
	0x66 | KSHIFT,		// 0x76 v
	0x67 | KSHIFT,		// 0x77 w
	0x70 | KSHIFT,		// 0x78 x
	0x71 | KSHIFT,		// 0x79 y
	0x72 | KSHIFT,		// 0x7A z
	0x96 | KSHIFT,		// 0x7B {
	0x74 | KSHIFT,		// 0x7C |
	0xa0 | KSHIFT,		// 0x7D }
	0x73 | KSHIFT,		// 0x7E ~
	KSPCR,			// 0x7F CarridgeReturn

	0x50 | KGRAPH,		// 0x80 graph
	0x75 | KGRAPH,		// 0x81 down
	0x75 | KSHIFT,		// 0x82 upper
	0x40 | KGRAPH,		// 0x83 right
	0x40 | KSHIFT,		// 0x84 left
	0x70 | KGRAPH,		// 0x85 graph
	0x43 | KGRAPH,		// 0x86 graph
	0x66 | KGRAPH,		// 0x87 graph
	0x42 | KGRAPH,		// 0x88 graph
	0x51 | KGRAPH,		// 0x89 graph
	0x65 | KGRAPH,		// 0x8A graph
	0x60 | KGRAPH,		// 0x8B graph
	0x57 | KGRAPH,		// 0x8C graph
	0x53 | KGRAPH,		// 0x8D graph
	0x72 | KGRAPH,		// 0x8E graph
	0x54 | KGRAPH,		// 0x8F graph

	0x52 | KGRAPH,		// 0x90 graph
	0x55 | KGRAPH,		// 0x91 graph (yen)
	0x65 | KGRAPH | KSHIFT,	// 0x92 rev U
	0x0000U,		// 0x93 graph (circle full)
	0x56 | KGRAPH,		// 0x94 graph (circle)
	0x67 | KGRAPH,		// 0x95 graph
	0x63 | KGRAPH,		// 0x96 graph
	0x61 | KGRAPH,		// 0x97 graph
	0x41 | KGRAPH,		// 0x98 graph
	0x47 | KGRAPH,		// 0x99 graph
	0x44 | KGRAPH,		// 0x9A graph
	0x46 | KGRAPH,		// 0x9B graph
	0x64 | KGRAPH,		// 0x9C graph
	0x71 | KGRAPH,		// 0x9D graph
	0x45 | KGRAPH,		// 0x9E graph
	0x62 | KGRAPH,		// 0x9F graph

	0x66 | KGRAPH | KSHIFT,	// 0xA0 rev V
	0x76 | KKANA | KSHIFT,	// 0xA1 kana period
	0x96 | KKANA | KSHIFT,	// 0xA2 kana kakko st
	0xa0 | KKANA | KSHIFT,	// 0xA3 kana kakko ed
	0x67 | KGRAPH | KSHIFT,	// 0xA4 rev W
	0x70 | KGRAPH | KSHIFT,	// 0xA5 rev X
	0x74 | KKANA,		// 0xA6 wo
	0x83 | KKANA | KSHIFT,	// 0xA7 l-a
	0x45 | KKANA | KSHIFT,	// 0xA8 l-i
	0x84 | KKANA | KSHIFT,	// 0xA9 l-u
	0x85 | KKANA | KSHIFT,	// 0xAA l-e
	0x86 | KKANA | KSHIFT,	// 0xAB l-i
	0x87 | KKANA | KSHIFT,	// 0xAC l-ya
	0x90 | KKANA | KSHIFT,	// 0xAD l-yu
	0x91 | KKANA | KSHIFT,	// 0xAE l-yo
	0x72 | KKANA | KSHIFT,	// 0xAF l-tsu

	0x71 | KGRAPH | KSHIFT,	// 0xB0 rev Y
	0x83 | KKANA,		// 0xB1 a
	0x45 | KKANA,		// 0xB2 i
	0x84 | KKANA,		// 0xB3 u
	0x85 | KKANA,		// 0xB4 e
	0x86 | KKANA,		// 0xB5 o
	0x64 | KKANA,		// 0xB6 ka
	0x47 | KKANA,		// 0xB7 ki
	0x50 | KKANA,		// 0xB8 ku
	0x92 | KKANA,		// 0xB9 ke
	0x42 | KKANA,		// 0xBA ko
	0x70 | KKANA,		// 0xBB sa
	0x44 | KKANA,		// 0xBC shi
	0x62 | KKANA,		// 0xBD su
	0x60 | KKANA,		// 0xBE se
	0x43 | KKANA,		// 0xBF so

	0x61 | KKANA,		// 0xC0 ta
	0x41 | KKANA,		// 0xC1 chi
	0x72 | KKANA,		// 0xC2 tsu
	0x67 | KKANA,		// 0xC3 te
	0x63 | KKANA,		// 0xC4 to
	0x65 | KKANA,		// 0xC5 na
	0x51 | KKANA,		// 0xC6 ni
	0x81 | KKANA,		// 0xC7 nu
	0x77 | KKANA,		// 0xC8 ne
	0x53 | KKANA,		// 0xC9 no
	0x46 | KKANA,		// 0xCA ha
	0x66 | KKANA,		// 0xCB hi
	0x82 | KKANA,		// 0xCC hu
	0x73 | KKANA,		// 0xCD fe
	0x94 | KKANA,		// 0xCE fo
	0x52 | KKANA,		// 0xCF ma

	0x56 | KKANA,		// 0xD0 mi
	0xa0 | KKANA,		// 0xD1 mu
	0x40 | KKANA,		// 0xD2 me
	0x55 | KKANA,		// 0xD3 mo
	0x87 | KKANA,		// 0xD4 ya
	0x90 | KKANA,		// 0xD5 yu
	0x91 | KKANA,		// 0xD6 yo
	0x57 | KKANA,		// 0xD7 ra
	0x54 | KKANA,		// 0xD8 ri
	0x76 | KKANA,		// 0xD9 ru
	0x93 | KKANA,		// 0xDA re
	0x75 | KKANA,		// 0xDB ro
	0x80 | KKANA,		// 0xDC wa
	0x71 | KKANA,		// 0xDD n
	0x95 | KKANA,		// 0xDE tenten
	0x96 | KKANA,		// 0xDF maru

	0x72 | KGRAPH | KSHIFT,	// 0xE0 rev Z
	0x41 | KGRAPH | KSHIFT,	// 0xE1 rev A
	0x42 | KGRAPH | KSHIFT,	// 0xE2 rev B
	0x43 | KGRAPH | KSHIFT,	// 0xE3 rev C
	0x44 | KGRAPH | KSHIFT,	// 0xE4 rev D
	0x45 | KGRAPH | KSHIFT,	// 0xE5 rev E
	0x46 | KGRAPH | KSHIFT,	// 0xE6 rev F
	0x47 | KGRAPH | KSHIFT,	// 0xE7 rev G
	0x50 | KGRAPH | KSHIFT,	// 0xE8 rev H
	0x51 | KGRAPH | KSHIFT,	// 0xE9 rev I
	0x52 | KGRAPH | KSHIFT,	// 0xEA rev J
	0x53 | KGRAPH | KSHIFT,	// 0xEB rev K
	0x54 | KGRAPH | KSHIFT,	// 0xEC rev L
	0x55 | KGRAPH | KSHIFT,	// 0xED rev M
	0x56 | KGRAPH | KSHIFT,	// 0xEE rev N
	0x57 | KGRAPH | KSHIFT,	// 0xEF rev O

	0x80 | KGRAPH | KSHIFT,	// 0xF0 rev 0
	0x81 | KGRAPH | KSHIFT,	// 0xF1 rev 1
	0x82 | KGRAPH | KSHIFT,	// 0xF2 rev 2
	0x83 | KGRAPH | KSHIFT,	// 0xF3 rev 3
	0x84 | KGRAPH | KSHIFT,	// 0xF4 rev 4
	0x85 | KGRAPH | KSHIFT,	// 0xF5 rev 5
	0x86 | KGRAPH | KSHIFT,	// 0xF6 rev 6
	0x87 | KGRAPH | KSHIFT,	// 0xF7 rev 7
	0x90 | KGRAPH | KSHIFT,	// 0xF8 rev 8
	0x91 | KGRAPH | KSHIFT,	// 0xF9 rev 9
	0x60 | KGRAPH | KSHIFT,	// 0xFA rev P
	0x61 | KGRAPH | KSHIFT,	// 0xFB rev Q
	0x62 | KGRAPH | KSHIFT,	// 0xFC rev R
	0x63 | KGRAPH | KSHIFT,	// 0xFD rev S
	0x64 | KGRAPH | KSHIFT,	// 0xFE rev T
	0x77 | KGRAPH,		// 0xFF graph (pi)
};

/* copy and paste variables */
static unsigned char copypaste_codetable[256 * 2];

/* paste constants */
int paste_mode = PASTE_MONITOR_DEFAULT;
int paste_wait = PASTE_WAIT_DEFAULT;
int paste_waitcr = PASTE_WAITCR_DEFAULT;

/* paste variables */
static volatile int paste_state = -1;
static volatile int paste_wct = 0;
static volatile int paste_rawchar = 0;
static volatile unsigned short paste_kdt = 0;
static volatile unsigned short paste_kmod = 0;
static volatile unsigned char *paste_buf = NULL;
static volatile unsigned char *paste_p = NULL;

int copypaste_loadtable( void )		/* S-JIS only */
{
	int i;
	unsigned char c1, c2;
	FILE *fp;

	fp  = fopen( CHARCODEFILE, "rb" );
	if (!fp)
		return -1;
	for (i = 0; i < 256; i++) {
		c1 = getc( fp );
		while (c1 == '\n' || c1 == '\r' || c1 == '\t')
			c1 = getc( fp );
		if (c1 == EOF) {
			fclose( fp );
			return -2;
		}
		c2 = getc( fp );
		if (c2 == EOF) {
			fclose( fp );
			return -3;
		}
		if (c2 == '\n' || c2 == '\r' || c2 == '\t') {
			fclose( fp );
			return -4;
		}
		if (!iskanji(c1) && iskanji(c2)) {
			fclose( fp );
			return -5;
		}
		if (!iskanji(c1)) {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = '\0';
		} else {
			copypaste_codetable[i * 2    ] = c1;
			copypaste_codetable[i * 2 + 1] = c2;
		}
	}
	fclose( fp );
	return 0;
}

BOOL copypaste_havepaste( HWND hwnd )
{
	char *pClipMemory;
	HGLOBAL hClipMemory;

	if (!IsClipboardFormatAvailable( CF_OEMTEXT ))
		return FALSE;
	if (!OpenClipboard( NULL ))
		return FALSE;
	hClipMemory = GetClipboardData( CF_OEMTEXT );
	if (!hClipMemory) {
		CloseClipboard();
		return FALSE;
	}
	pClipMemory = (char *)GlobalLock( hClipMemory );
	if (!pClipMemory) {
		CloseClipboard();
		return FALSE;
	}
	GlobalUnlock( hClipMemory );
	CloseClipboard();
	return TRUE;
}

int copypaste_pastestart( HWND hWnd )
{
	char *pClipMemory, *cp;
	HGLOBAL hClipMemory;

	if (!IsClipboardFormatAvailable( CF_OEMTEXT ))
		return -1;
	if (!OpenClipboard( NULL )) {
		MessageBox( NULL, "OpenClipboard() Failed.",
			"Paste from Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	hClipMemory = GetClipboardData( CF_OEMTEXT );
	if (!hClipMemory) {
		MessageBox( NULL, "GetClipboardData() Failed.",
			"Paste from Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseClipboard();
		return -1;
	}
	pClipMemory = (char *)GlobalLock( hClipMemory );
	if (!pClipMemory) {
		MessageBox( NULL, "GlobalLock() Failed.",
			"Paste from Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseClipboard();
		return -1;
	}
	cp = MALLOC( strlen( pClipMemory ) + 1 );
	if (cp) {
		strncpy( cp, pClipMemory, strlen( pClipMemory ) );
		cp[strlen(pClipMemory)] = '\0';
	}
	GlobalUnlock( hClipMemory );
	CloseClipboard();
	if (!cp)
		return -1;

	keyboard_releaseall( mz2000_global );
	if (paste_buf)
		FREE( (void *)paste_buf );
	paste_buf = cp;
	paste_p = cp;
	paste_kmod  = KNONE;
	paste_rawchar = 0;
	paste_wct   = 0;
	paste_state = 0;

	return 0;
}

int copypaste_pasteend( BOOL force_flag )
{
	if (force_flag)
		paste_state = -1;
	else
		paste_state = 1000;
	paste_p = NULL;
	if (paste_buf) {
		FREE( (void *)paste_buf );
		paste_buf = NULL;
	}
	return 0;
}

static int _copypaste_paste_conv( unsigned char ch1, unsigned char ch2 )
{
	int i;
	unsigned char c1, c2;

	if (!ch1 && ch2 == 0x0a)	/* LF */
		return 0x0d;
	if (!ch1 && ch2 == 0x0d)	/* CR */
		return -1;
	if (!ch1 && ch2 == 0x09)	/* TAB */
		return 0x1a;
	if (!ch1 && ch2 == ' ')		/* SPACE */
		return 0x20;
	if (!ch1 && !ch2)		/* '\0' */
		return -1;
	for (i = 0; i < 256; i++) {
		c1 = copypaste_codetable[i * 2 + 1];
		c2 = copypaste_codetable[i * 2];
		if (c1 == ch1 && c2 == ch2)
			return i != 0 ? i : -1;
	}
	return -1;
}

void copypaste_pastetimer( void )
{
	int r;
	unsigned char ch1, ch2;

	if (paste_state < 0)
		return;
	if (paste_wct > 0) {
		--paste_wct;
		return;
	}
retry:	switch (paste_state) {
	case 0 :
		paste_state = 100;
		goto retry;

	/* get a character and execute it */
	case 100 :	/* normal */
		if (!paste_p) {
			paste_state = 1000;
			goto retry;
		}
		ch1 = *paste_p;
		if (!ch1) {
			paste_p = NULL;
			goto retry;
		}
		paste_p++;
		if (iskanji(ch1)) {
			ch2 = *paste_p;
			if (!ch2) {
				paste_p = NULL;
				goto retry;
			}
			paste_p++;
		} else
			ch2 = 0;
		r = _copypaste_paste_conv( ch2, ch1 );
		if (r < 0 || r > 255)
			break;	/* ignore character */
		paste_rawchar = r;
		paste_kdt = copypaste_pastekbddata[r];
		paste_state = 110;
		goto retry;

	/* convert the special keys */
	case 110 :
		switch (paste_mode) {
		case PASTE_MONITOR_PUPMON :
			switch (paste_rawchar) {
			case 0x01 :	/* down */
				paste_kdt = 0x34 | KGRAPH;
				break;
			case 0x02 :	/* up */
				paste_kdt = 0x33 | KGRAPH;
				break;
			case 0x03 :	/* right */
				paste_kdt = 0x36 | KGRAPH;
				break;
			case 0x04 :	/* left */
				paste_kdt = 0x35 | KGRAPH;
				break;
			case 0x05 :	/* home */
				paste_kdt = 0xa2 | KGRAPH;
				break;
			case 0x06 :	/* clr */
				paste_kdt = 0xa2 | KGRAPH | KSHIFT;
				break;
			case 0x1e :	/* square full */
				paste_kdt = 0x76 | KGRAPH;          /* . */
				break;
			case 0x1f :	/* checker flag */
				paste_kdt = 0x76 | KGRAPH | KSHIFT; /* . */
				break;
			case 0x93 :	/* circle full */
				paste_kdt = 0x73 | KGRAPH | KSHIFT; /* ^ */
				break;
			default :
				break;
			}
			break;
		case PASTE_MONITOR_TS :
			switch (paste_rawchar) {
			case 0x01 :	/* down */
				paste_kdt = 0x34 | KGRAPH | KSHIFT;
				break;
			case 0x02 :	/* up */
				paste_kdt = 0x33 | KGRAPH | KSHIFT;
				break;
			case 0x03 :	/* right */
				paste_kdt = 0x36 | KGRAPH | KSHIFT;
				break;
			case 0x04 :	/* left */
				paste_kdt = 0x35 | KGRAPH | KSHIFT;
				break;
			case 0x05 :	/* home */
				paste_kdt = 0xa2 | KGRAPH;
				break;
			case 0x06 :	/* clr */
				paste_kdt = 0xa2 | KGRAPH | KSHIFT;
				break;
			case 0x1e :	/* square full */
				paste_kdt = 0x73 | KGRAPH; /* ^ */
				break;
			case 0x1f :	/* checker flag */
				paste_kdt = 0x74 | KGRAPH; /* \ */
				break;
			case 0x7f :	/* function CR */
				paste_kdt = 0x81 | KGRAPH; /* 1 */
				break;
			case 0x93 :	/* circle full */
				paste_kdt = 0x80 | KGRAPH; /* 0 */
				break;
			default :
				break;
			}
			break;
		case PASTE_MONITOR_HUBASIC :
			switch (paste_rawchar) {
			case 0x93 :	/* circle full */
				paste_kdt = 0x73 | KGRAPH; /* ^ */
				break;
			case 0x7f :	/* function CR */
				paste_kdt = 0x76 | KGRAPH; /* . */
				break;
			default :
				break;
			}
			break;
		case PASTE_MONITOR_DBBASIC :
			switch (paste_rawchar) {
			case 0x7f :	/* function CR */
				paste_kdt = 0x0000U;
				break;
			default :
				break;
			}
			break;
		default : /* normal */
			break;
		}
		if (paste_kdt == 0x0000U)
			paste_state = 0;
		else
			paste_state = 300;
		goto retry;

	/* modifier keys */
	case 300 :	/* if want to modes change to KANA or GRAPH, release the shift keys */
		if ((paste_kmod & KSHIFT) &&
			((paste_kmod & KGRAPH) != (paste_kdt & KGRAPH)
				|| (paste_kmod & KKANA) != (paste_kdt & KKANA))) {
			keyboard_setdirect( FALSE, 0x0b, 0x02 );
			paste_kmod &= ~KSHIFT;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 301 :	/* if want to KANA and GRPH were selected, push it for deselect it */
		if ((paste_kmod & KGRAPH) && (paste_kdt & KKANA)) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x00 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 302 :	/* if want to KANA and GRPH were selected, release it */
		if ((paste_kmod & KGRAPH) && (paste_kdt & KKANA)) {
			keyboard_setdirect( FALSE, 0x0b, 0x00 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod &= ~KGRAPH;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 303 :	/* if want to GRPH and KANA were selected, push it for deselect it (require dB-BASIC) */
		if ((paste_kmod & KKANA) && (paste_kdt & KGRAPH)) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x03 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 304 :	/* if want to GRPH and KANA were selected, release it (require dB-BASIC) */
		if ((paste_kmod & KKANA) && (paste_kdt & KGRAPH)) {
			keyboard_setdirect( FALSE, 0x0b, 0x03 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod &= ~KKANA;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 305 :	/* if want to KANA, push it for select it */
		if ((paste_kmod & KKANA) != (paste_kdt & KKANA)) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x03 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 306 :	/* if want to KANA, release it for select it */
		if ((paste_kmod & KKANA) != (paste_kdt & KKANA)) {
			keyboard_setdirect( FALSE, 0x0b, 0x03 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod ^= KKANA;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 307 :	/* if want to GRPH, push it for select it */
		if ((paste_kmod & KGRAPH) != (paste_kdt & KGRAPH)) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x00 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 308 :	/* if want to GRPH, release it for select it */
		if ((paste_kmod & KGRAPH) != (paste_kdt & KGRAPH)) {
			keyboard_setdirect( FALSE, 0x0b, 0x00 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod ^= KGRAPH;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 309 :	/* SHIFT keys */
		if ((paste_kmod & KSHIFT) != (paste_kdt & KSHIFT)) {
			if (paste_kdt & KSHIFT) {
				keyboard_setdirect( TRUE, 0x0b, 0x02 );
				paste_kmod |= KSHIFT;
			} else {
				keyboard_setdirect( FALSE, 0x0b, 0x02 );
				paste_kmod &= ~KSHIFT;
			}
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 310 :
		paste_state = 500;
		goto retry;

	/* main keys want to press and release */
	case 500 :
		if (paste_kdt == KSPCR) {
			paste_state = 510;
			goto retry;
		}
		paste_state++;
	case 501 :	/* press a key */
		keyboard_setdirect( TRUE, (paste_kdt & 0xf0) >> 4, paste_kdt & 0x0f );
		paste_state++;
		paste_wct = paste_wait;
		break;
	case 502 :	/* release a key */
		keyboard_setdirect( FALSE, (paste_kdt & 0xf0) >> 4, paste_kdt & 0x0f );
		paste_state = 0;
		if ((paste_kdt & 0xff) == 0x32)	{ /* CR */
			paste_wct = paste_waitcr;
			if (paste_mode != PASTE_MONITOR_HUBASIC && paste_mode != PASTE_MONITOR_DBBASIC)
				paste_kmod = 0;
		} else
			paste_wct = paste_wait;
		break;
	case 510 :	/* press the SFTLOCK and GRPH keys (function CR 0x7f) */
		keyboard_setdirect( TRUE, 0x0b, 0x00 ); /* GRPH */
		keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
		paste_state++;
		paste_wct = paste_wait;
		break;
	case 511 :	/* release the SFTLOCK and GRPH keys (function CR 0x7f) */
		keyboard_setdirect( FALSE, 0x0b, 0x00 ); /* GRPH */
		keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
		paste_state = 0;
		paste_wct = paste_wait;
		break;

	/* entering to ending process, reset to the modifier keys */
	case 1000 :	/* Shift keys */
		if (paste_kmod & KSHIFT) {
			keyboard_setdirect( FALSE, 0x0b, 0x02 );
			paste_kmod &= ~KSHIFT;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 1001 :	/* push the GRPH */
		if (paste_kmod & KGRAPH) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x00 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 1002 :	/* release the GRPH */
		if (paste_kmod & KGRAPH) {
			keyboard_setdirect( FALSE, 0x0b, 0x00 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod &= ~KGRAPH;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 1003 :	/* push the KANA */
		if (paste_kmod & KKANA) {
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( TRUE, 0x0b, 0x01 ); /* SFTLOCK */
			keyboard_setdirect( TRUE, 0x0b, 0x03 );
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 1004 :	/* release the KANA */
		if (paste_kmod & KKANA) {
			keyboard_setdirect( FALSE, 0x0b, 0x03 );
			if (paste_mode == PASTE_MONITOR_HUBASIC)
				keyboard_setdirect( FALSE, 0x0b, 0x01 ); /* SFTLOCK */
			paste_kmod &= ~KKANA;
			paste_state++;
			paste_wct = paste_wait;
			break;
		}
		paste_state++;
	case 1005 :
		keyboard_releaseall( mz2000_global );
		paste_state = -1;
		goto retry;
	default :
		break;
	}
	return;
}

int copypaste_runningpaste( void )
{
	return paste_state >= 0;
}

static void _copypaste_copy_sub( char *buf, int *index, unsigned char ch )
{
	if (*index >= SCREENCOPYBUFF_MAX - 1)
		return;
	buf[*index] = ch;
	(*index)++;
	return;
}

static void _copypaste_copy_conv( char *buf, int *index, unsigned char ch )
{
	unsigned char c;

	c = copypaste_codetable[ch * 2];
	if (c)
		_copypaste_copy_sub( buf, index, c );
	c = copypaste_codetable[ch * 2 + 1];
	if (c)
		_copypaste_copy_sub( buf, index, c );
	return;
}

int copypaste_copy( HWND hWnd )
{
	int i, j, k, index, lmax;
	char *pText, buf[SCREENCOPYBUFF_MAX];
	unsigned char ch;
	HGLOBAL hText;

	/* make data */
	index = 0;
	if (!(mz2000_global -> ioinfo -> port_e8 & 0x20U))
		lmax = 40;
	else
		lmax = 80;
	for (j = 0; j < 25; j++) {
		for (i = 0; i < lmax; i++) {
			ch = mz2000_global -> vram[i + j * lmax];
			if (ch)
				_copypaste_copy_conv( buf, &index, ch );
			else {
				int flag = FALSE;
				for (k = i + 1; k < lmax; k++) {
					if (mz2000_global -> vram[k + j * lmax])
						flag = TRUE;
				}
				if (flag)
					_copypaste_copy_conv( buf, &index, 0 );
				else
					break;
			}
		}
		_copypaste_copy_sub( buf, &index, '\r' );
		_copypaste_copy_sub( buf, &index, '\n' );
	}
	_copypaste_copy_sub( buf, &index, 0 );
	buf[SCREENCOPYBUFF_MAX - 1] = '\0';

	/* set clipboard */
	if (!(hText = GlobalAlloc( GHND, strlen(buf) + 1 ))) {
		MessageBox( NULL, "Memory Allocation Failed for Clipboard",
			"Copy to clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	pText = (char *)GlobalLock( hText );
	strncpy( pText, buf, strlen(buf) + 1 );
	buf[strlen(buf)] = '\0';
	GlobalUnlock(hText);

	if (!OpenClipboard( NULL )) {
		MessageBox( NULL, "OpenClipboard() Failed",
			"Copy to Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	if (!EmptyClipboard()) {
		MessageBox( NULL, "EmptyClipboard() Failed",
			"Copy to Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseClipboard();
		return -1;
	}
	if (!SetClipboardData( CF_OEMTEXT, hText )) {
		MessageBox( NULL, "SetClipboardData() failed",
			"Copy to Clipboard", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		CloseClipboard();
		return -1;
	}
	CloseClipboard();
	return 0;
}

void copypaste_exit( void )
{
	MZ2000 *mz = mz2000_global;

	if (mz && mz2000_global) {
		char buf[LNMAX];
		const char *section = INISECTIONNAME_BASIC;

		_snprintf( buf, LNMAX, "%d", paste_mode );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		WriteUserProfileString( section, "paste_environment", buf, app_defaultinifile );
		_snprintf( buf, LNMAX, "%d", paste_wait );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		WriteUserProfileString( section, "paste_wait", buf, app_defaultinifile );
		_snprintf( buf, LNMAX, "%d", paste_waitcr );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		WriteUserProfileString( section, "paste_wait_cr", buf, app_defaultinifile );
	}
	return;
}

int copypaste_init( void )
{
	int i, r;
	char buf[LNMAX], msgbuf[LNMAX];
	const char *section = INISECTIONNAME_BASIC;

	/* initializing states variables */
	paste_state	= -1;
	paste_wct	= 0;
	paste_rawchar	= 0;
	paste_kmod	= KNONE;
	paste_kdt	= 0;
	paste_buf	= NULL;
	paste_p		= NULL;

	/* initialzing user settings variables */
	paste_mode = PASTE_MONITOR_DEFAULT;
	if (GetUserProfileString( section, "paste_environment", "", buf, LNMAX, app_defaultinifile ) > 0)
		paste_mode = atoi( buf );
	paste_wait = PASTE_WAIT_DEFAULT;
	if (GetUserProfileString( section, "paste_wait", "", buf, LNMAX, app_defaultinifile ) > 0)
		paste_wait = atoi( buf );
	paste_waitcr = PASTE_WAITCR_DEFAULT;
	if (GetUserProfileString( section, "paste_wait_cr", "", buf, LNMAX, app_defaultinifile ) > 0)
		paste_waitcr = atoi( buf );

	/* initializing a code table */
	for (i = 0; i < 256; i++) {
		if (i >= ' ' && i < 0x7e)
			copypaste_codetable[i * 2    ] = i;
		else
			copypaste_codetable[i * 2    ] = ' ';
		copypaste_codetable[i * 2 + 1] = '\0';
	}
	r = copypaste_loadtable();
	switch (r) {
	case 0 :
		break;
	case -1 :
		MessageBox( NULL, CHARCODEFILE " Open Failed",
			"Initializing Copy/Paste", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	default :
		_snprintf( msgbuf, LNMAX, CHARCODEFILE " Read Failed or Illegal Data (code=%d)", -r );
		msgbuf[LNMAX - 1] = '\0';
		MessageBox( NULL, msgbuf,
			"Initializing Copy/Paste", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	}
	return 0;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
