/* File: deviceDlg.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Device Configurations Dialog

	Copyright (C) 2011-2019 FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "config.h"

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <string.h>

#include "common.h"
#include "mz2000.h"
#include "deviceDlg.h"
#include "resource.h"

static int current_monitor;

static BOOL DlgCenterWindow( HWND hDlg, HWND hWndParent )
{
	RECT rect;
	int tx, ty, dx, dy, sx, sy;

	/* get the target center positions */
	if (!hWndParent)
		hWndParent = GetWindow( hDlg, GW_OWNER );
	if (!hWndParent)
		return FALSE;
	GetWindowRect( hWndParent, &rect );
	tx = (rect.left + rect.right) / 2;
	ty = (rect.top + rect.bottom) / 2;

	/* move to center positions */
	GetWindowRect( hDlg, &rect );
	dx = tx - (rect.left + rect.right) / 2;
	dy = ty - (rect.top + rect.bottom) / 2;
	rect.left   += dx;
	rect.right  += dx;
	rect.top    += dy;
	rect.bottom += dy;

	/* move to inside of screen */
	sx = GetSystemMetrics( SM_CXSCREEN );
	sy = GetSystemMetrics( SM_CYSCREEN );
	if (rect.right >= sx) {
		int dd = sx - rect.right;
		rect.left  += dd;
		rect.right += dd;
	}
	if (rect.bottom >= sy) {
		int dd = sy - rect.bottom;
		rect.top    += dd;
		rect.bottom += dd;
	}
	if (rect.left < 0) {
		rect.right += -rect.left;
		rect.left = 0;
	}
	if (rect.top < 0) {
		rect.bottom += -rect.top;
		rect.top = 0;
	}

	/* update the real window positions */
	SetWindowPos( hDlg, NULL,
		rect.left, rect.top, 0, 0,
		SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE );
	return TRUE;
}

static BOOL DialogCommand( HWND hDlg, int id, HWND hWndCtl, UINT CodeNotify )
{
	int i, t;
	char buf[LNMAX];
	DWORD r;
	HWND hSliderSoundvol1 = GetDlgItem( hDlg, IDC_SLIDER_SOUNDVOL1 );
	HWND hSliderSoundvol2 = GetDlgItem( hDlg, IDC_SLIDER_SOUNDVOL2 );

	switch (id) {
	case IDC_COMBO_DEVICE :
		/* display the switchings */
		r = SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
			CB_GETCURSEL, 0, 0 );
		if (r != CB_ERR && r >= 0 && r != (unsigned)current_monitor) {
			/* display the devices */
			current_monitor = r;
			SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
				CB_SETCURSEL, current_monitor, 0 );
			/* display the resolutions */
			SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
				CB_RESETCONTENT, 0, 0 );
			SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
				CB_ADDSTRING, 0, (LPARAM)"Auto" );
			for (i = 0; i < screen_info[current_monitor] . res_maxnum; i++) {
				_snprintf( buf, LNMAX, "%d x %d (%dHz)",
					screen_info[current_monitor] . sx[i],
					screen_info[current_monitor] . sy[i],
					screen_info[current_monitor] . srefresh[i] );
				buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
				SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
					CB_ADDSTRING, 0, (LPARAM)buf );
			}
			SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
				CB_SETCURSEL, 0, 0 );
		}
		return TRUE;
	case IDOK :
		/* the sound devices */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDBUFFER, WM_GETTEXT, 10, (LPARAM)buf );
		if (sscanf( buf, "%d", &t ) != 1 || t < 10 || t > 1000) {
			MessageBox( NULL, "Sound Play Buffers (Time) value range error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> sound_playbuffer_ms = t;
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDDELAY, WM_GETTEXT, 10, (LPARAM)buf );
		if (sscanf( buf, "%d", &t ) != 1 || t < 10 || t > 1000) {
			MessageBox( NULL, "Sound Delay Times value range error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> sound_delaytime_ms = t;
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDFREQ, WM_GETTEXT, 10, (LPARAM)buf );
		if (sscanf( buf, "%d", &t ) != 1 || t < 1000 || t > 192000) {
			MessageBox( NULL, "Sound Frequency value range error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> sound_freq = t;
		r = SendDlgItemMessage( hDlg, IDC_COMBO_SOUNDBIT, CB_GETCURSEL, 0, 0 );
		if (r == CB_ERR || r < 0) {
			MessageBox( NULL, "Sound bits select error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> sound_bit = r == 1 ? 8 : 16;
		r = SendMessage( hSliderSoundvol1, TBM_GETPOS, 0, 0 );
		mz2000_global -> sound_volume1 = (char)(100 - r);
		r = SendMessage( hSliderSoundvol2, TBM_GETPOS, 0, 0 );
		mz2000_global -> sound_volume2 = (char)(100 - r);

		/* the displaies */
		r = SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
			CB_GETCURSEL, 0, 0 );
		if (r == CB_ERR || r < 0) {
			MessageBox( NULL, "Display device select error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> fullscreen_device = (char)r;
		r = SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
			CB_GETCURSEL, 0, 0 );
		if (r == CB_ERR || r < 0) {
			MessageBox( NULL, "Display resolution select error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return FALSE;
		}
		mz2000_global -> fullscreen_resolution = (char)r;
		r = SendDlgItemMessage( hDlg, IDC_COMBO_BPP,
			CB_GETCURSEL, 0, 0 );
		if (r == CB_ERR || r < 0 || r >= 1 + 1) {
			MessageBox( hDlg, "Display BPP select error",
				"Device Configurations", MB_OK | MB_ICONEXCLAMATION | MB_TASKMODAL );
			return FALSE;
		}
		mz2000_global -> fullscreen_bpp = !r ? SCREEN_BPP_16 : SCREEN_BPP_32;

		EndDialog( hDlg, TRUE );
		return TRUE;
	case IDCANCEL :
		EndDialog( hDlg, FALSE );
		return TRUE;
	case IDC_BUTTON_DEFAULT :
		/* the sound devices */
		_snprintf( buf, LNMAX, "%d", DEFAULT_SOUND_PLAYBUFFER_MS );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDBUFFER, WM_SETTEXT, 0, (LPARAM)buf );
		_snprintf( buf, LNMAX, "%d", DEFAULT_SOUND_DELAYTIME_MS );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDDELAY, WM_SETTEXT, 0, (LPARAM)buf );
		_snprintf( buf, LNMAX, "%d", DEFAULT_SOUND_FREQ );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDFREQ, WM_SETTEXT, 0, (LPARAM)buf );
		SendDlgItemMessage( hDlg, IDC_COMBO_SOUNDBIT,
			CB_SETCURSEL, DEFAULT_SOUND_BIT == 8 ? 1 : 0, 0 );
		PostMessage( hSliderSoundvol1, TBM_SETPOS, TRUE, 100 - DEFAULT_SOUND_VOLUME1 );
		PostMessage( hSliderSoundvol2, TBM_SETPOS, TRUE, 100 - DEFAULT_SOUND_VOLUME2 );
		/* the displaies */
		SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
			CB_SETCURSEL, SCREEN_DEVICE_DEFAULT, 0 );
		SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
			CB_SETCURSEL, SCREEN_RESOLUTION_AUTO, 0 );
		SendDlgItemMessage( hDlg, IDC_COMBO_BPP,
			CB_SETCURSEL, 0, 0 );
		break;
	}
	return TRUE;
}

BOOL WINAPI DlgProcDeviceConfig( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
	DWORD r;
	int i;
	char buf[LNMAX];
	HWND hSliderSoundvol1 = GetDlgItem( hDlg, IDC_SLIDER_SOUNDVOL1 );
	HWND hSliderSoundvol2 = GetDlgItem( hDlg, IDC_SLIDER_SOUNDVOL2 );

	switch (msg) {
	case WM_INITDIALOG :
		/* set the window attributes */
		DlgCenterWindow( hDlg, NULL );

		/* the sound devices */
		_snprintf( buf, LNMAX, "%d", mz2000_global -> sound_playbuffer_ms );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDBUFFER, WM_SETTEXT, 0, (LPARAM)buf );
		_snprintf( buf, LNMAX, "%d", mz2000_global -> sound_delaytime_ms );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDDELAY, WM_SETTEXT, 0, (LPARAM)buf );
		_snprintf( buf, LNMAX, "%d", mz2000_global -> sound_freq );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		SendDlgItemMessage( hDlg, IDC_EDIT_SOUNDFREQ, WM_SETTEXT, 0, (LPARAM)buf );
		/* the sound bits */
		SendDlgItemMessage( hDlg, IDC_COMBO_SOUNDBIT,
			CB_ADDSTRING, 0, (LPARAM)"16" );
		SendDlgItemMessage( hDlg, IDC_COMBO_SOUNDBIT,
			CB_ADDSTRING, 0, (LPARAM)"8" );
		SendDlgItemMessage( hDlg, IDC_COMBO_SOUNDBIT,
			CB_SETCURSEL, mz2000_global -> sound_bit == 8 ? 1 : 0, 0 );
		/* the volumes */
		PostMessage( hSliderSoundvol1, TBM_SETRANGEMIN, TRUE, 0 );
		PostMessage( hSliderSoundvol1, TBM_SETRANGEMAX, TRUE, 100 );
		PostMessage( hSliderSoundvol1, TBM_SETPOS, TRUE, 100 - mz2000_global -> sound_volume1 );
		PostMessage( hSliderSoundvol2, TBM_SETRANGEMIN, TRUE, 0 );
		PostMessage( hSliderSoundvol2, TBM_SETRANGEMAX, TRUE, 100 );
		PostMessage( hSliderSoundvol2, TBM_SETPOS, TRUE, 100 - mz2000_global -> sound_volume2 );
		sprintf( buf, "%ld", mz2000_global -> sound_volume1 );
		SetWindowText( GetDlgItem( hDlg, IDC_EDIT_SOUNDVOL1 ), buf );
		sprintf( buf, "%ld", mz2000_global -> sound_volume2 );
		SetWindowText( GetDlgItem( hDlg, IDC_EDIT_SOUNDVOL2 ), buf );

		/* the displaies */
		/* the display devices */
		for (i = 0; i < screeninfo_maxnum; i++) {
			SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
				CB_ADDSTRING, 0, (LPARAM)screen_info[i] . DriverName );
		}
		current_monitor = mz2000_global -> fullscreen_device;
		SendDlgItemMessage( hDlg, IDC_COMBO_DEVICE,
			CB_SETCURSEL, current_monitor, 0 );
		/* the display resolutions */
		SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
			CB_ADDSTRING, 0, (LPARAM)"Auto" );
		for (i = 0; i < screen_info[current_monitor] . res_maxnum; i++) {
			if (!screen_info[current_monitor] . srefresh[i]) {
				_snprintf( buf, LNMAX, "%d x %d",
					screen_info[current_monitor] . sx[i],
					screen_info[current_monitor] . sy[i] );
			} else {
				_snprintf( buf, LNMAX, "%d x %d (%dHz)",
					screen_info[current_monitor] . sx[i],
					screen_info[current_monitor] . sy[i],
					screen_info[current_monitor] . srefresh[i] );
			}
			buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
			SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
				CB_ADDSTRING, 0, (LPARAM)buf );
		}
		SendDlgItemMessage( hDlg, IDC_COMBO_RESOLUTION,
			CB_SETCURSEL, mz2000_global -> fullscreen_resolution, 0 );
		/* the display BPP */
		SendDlgItemMessage( hDlg, IDC_COMBO_BPP,
			CB_ADDSTRING, 0, (LPARAM)"16" );
		SendDlgItemMessage( hDlg, IDC_COMBO_BPP,
			CB_ADDSTRING, 0, (LPARAM)"32" );
		SendDlgItemMessage( hDlg, IDC_COMBO_BPP,
			CB_SETCURSEL, mz2000_global -> fullscreen_bpp == SCREEN_BPP_16 ? 0 : 1, 0 );
		return TRUE;
	case WM_NOTIFY :
		r = SendMessage( hSliderSoundvol1, TBM_GETPOS, 0, 0 );
		sprintf( buf, "%ld", 100 - r );
		SetWindowText( GetDlgItem( hDlg, IDC_EDIT_SOUNDVOL1 ), buf );
		r = SendMessage( hSliderSoundvol2, TBM_GETPOS, 0, 0 );
		sprintf( buf, "%ld", 100 - r );
		SetWindowText( GetDlgItem( hDlg, IDC_EDIT_SOUNDVOL2 ), buf );
		return TRUE;
	case WM_COMMAND :
		return HANDLE_WM_COMMAND( hDlg, wParam, lParam, DialogCommand );
	}
	return FALSE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
