/* File: disk.c

	SHARP MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000" (FDD)

	Interfaces:	MZ-1E05 (FDC Fujitsu MB8876A)
			MZ-8BFI/MZ-80FI (FDC Fujitsu MB8866)
			MZ-2500 Internal FDD Interface
			and comptaibles
	(Fujitsu FDCs are second source of Western Digital FD179X series)

	Floppy Disk Drives: 5.25'2D drives (MZ-1F07 / MZ-1F02 /
			MZ-80BF / MZ-80BFK / MZ-80FB / MZ-80FBK)
			MZ-2500 Internal 3.5'2DD drives
			and compatibles

	Support Disk Image Formats:
			D88(D20/D77) / DSK /
			2D (320KB/640KB/(or any) 256 bytes length 16 sectors raw images,
				reverse bit(*), reverse side(*))
			(*) ... de facto standard format

	Copyright (C) 2002-2019 FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

/*
	MZ-1E05 I/O Ports (It's accesable only 2D/2DD MFM)
	0xD8	... FDC command (out) / FDC status (in) register
	0xD9	... FDC track register (in/out)
	0xDA	... FDC sector register (in/out)
	0xDB	... FDC data register (in/out)
	0xDC	... A drive select and motor (out only)
			bit 0-1 - drive no.
			bit 2   - drive selects
			bit 7   - enable the motor
	0xDD	... side (out only)
			bit 0   - side select

	(Unimplemented)
	- Index hole is enabled at random.
	- Many wait sequences are omitted. (fast running mode only)
	- No disk media and motor on and access it and FDC into loop.
	  When disk is inserted before motor off, this program can't recovery it.
	- I don't know the specification of .DSK format
		* support only CRC error
		* write track (format) is failed (status = WriteFault)
*/

#include "config.h"

#ifdef WIN32
#include <windows.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#ifndef WIN32
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif

#include "common.h"
#include "mz2000.h"
#ifndef WIN32
#include "uigtk.h"
#endif
#include "util.h"

/* trace information for debugging build */
#undef ENBALE_TRACE_DISKIO
#undef ENBALE_TRACE_DISKIO_STATUS
#undef ENBALE_TRACE_DISKCMD
#undef ENBALE_TRACE_READID

/* Max Drive Numbers */
#define DISK_DRIVE_MAXNUM	MZ_DISK_DRIVE_MAX

/* Max Track Numbers */
/* SHARP MZ-1F07 is accessable 0-83 */
/* SHARP MZ-80BF and EPSON TF-10/20(?) are accessable 0-79 */
/* SHARP MZ-80BF early lot is accessable only 0-69 */
#define DISK_TRACK_2D_MAXNUM	84
/* SHARP MZ-2500 internal 3.5'2DD or other 2DD drives accessable 0-163 */
#define DISK_TRACK_2DD_MAXNUM	164

/* Disk Buffer Size */
#define DISK_IMAGEBUFF_SIZE	(2*1024*1024)	/* 2MB */
#define DISK_IMAGEREADABLE_SIZE	(1*1024*1024)	/* 1MB for image files size check */
/* Track Buffer Size */
#define DISK_TRACKBUFF_SIZE	0x2000
/*  1-Track Size (2D 320KB-360KB / 2DD 640KB-720KB FD) */
#define DISK_TRACK_BYTES	0x1850

/* Max ID Table */
#define DISK_SECTOR_MAXNUM	256

/* Other Parameters */
#ifndef WIN32
#define DISK_WAIT_CLOCK1	DEFAULT_DISKWAITCLOCK6ms
#define DISK_WAIT_CLOCK2	DEFAULT_DISKWAITCLOCK12ms
#define DISK_WAIT_CLOCK3	DEFAULT_DISKWAITCLOCK20ms
#define DISK_WAIT_CLOCK4	DEFAULT_DISKWAITCLOCK30ms
#else
#define DISK_WAIT_CLOCK1	((os_runmode != WOV_NT && os_runmode != WOV_2000) ?  24000 : 15000) /* 6ms */
#define DISK_WAIT_CLOCK2	((os_runmode != WOV_NT && os_runmode != WOV_2000) ?  48000 : 30000) /* 12ms */
#define DISK_WAIT_CLOCK3	((os_runmode != WOV_NT && os_runmode != WOV_2000) ?  80000 : 50000) /* 20ms */
#define DISK_WAIT_CLOCK4	((os_runmode != WOV_NT && os_runmode != WOV_2000) ? 120000 : 75000) /* 30ms */
#endif
#define DISK_MOTOR_STOPTIME	(300/MZ2000_REFRESHRATE)	/* 300ms */
#define REQUESTDATA_WAIT_CT	10
#define INDEXHOLE_VALUE		1024

/* Disk Images Format Specifications */
/* .2D Format */
/* When execute format to very small image, and use 1-track format to >>image size,
   fill the data with this */
#if 1
#define DISK_2D_FILLDATA	0x00	/* zero, because padding data for truncated disks are it */
#elif 1
#define DISK_2D_FILLDATA	0xe5	/* de facto standard initialize data, CP/M, etc */
#elif 1
#define DISK_2D_FILLDATA	0x40	/* MZ_MODE = 0xbf, it's sharp format initialize data */
#else
#define DISK_2D_FILLDATA	0xff	/* MZ_MODE = 0x00, it's NEC PC-8001/8801 FDD BIOS initialize data */
#endif

/* .D88 format images */
#define D88_DISKNAME_MAXLEN	16
#define D88_TRACKS_MAXNUM	164
#define D88_FIRSTHEAD_SIZE	0x20
#define D88_HEADDER_SIZE	0x02b0U	/* D88_FIRSTHEAD_SIZE + D88_TRACKS_MAXNUM * 4bytes */
#define D88_OFFSET_WRITEPROTECT	0x1aU	/* 0x00(off) or 0x10(on) */
#define D88_OFFSET_MEDIATYPE	0x1bU	/* 0x00=2D, 0x10=2DD, 0x20=2HD(No support) */
#define D88_OFFSET_DISKSIZE	0x1cU
#define D88_OFFSET_TRACKTABLE	0x20U
/* .DSK format images */
/* #define DSK_DISKEYECACHE_OFFS	0 */
#define DSK_DISKEYECACHE_LEN		34
#define DSK_DISKNAME_OFFS		0x22U	/* name of creator */
#define DSK_DISKNAME_MAXLEN		14
#define DSK_DISKTRACKS_OFFS		0x30U
#define DSK_DISKSIDES_OFFS		0x31U
#define DSKNM_TRACKSIZE_OFFS		0x32U
#define DSKEX_DISKTRKTBL_OFFS		0x34U
#define DSK_DISKTRKSTABLES_OFFS		0x0100U
#define DSK_TRKSINFO_LEN		0x0100U
/* #define DSK_TRKSEYECATCH_OFFS	0 */
#define DSK_TRKSEYECATCH_LEN		12
#define DSK_TRKS_MAXSECTORS_OFFS	0x15
#define DSK_TRKS_SCTINFOLIST_OFFS	0x18
#define DSK_SCTINFO_ONELEN		0x0008U
#define DSKEX_SCTINFO_DATARATE_OFFS	0x12
#define DSKEX_SCTINFO_RECMODE_OFFS	0x13

typedef enum {
	DISK_DATAFMT_NONE = 0,
	DISK_DATAFMT_ERRORS,
	DISK_DATAFMT_2D,
	DISK_DATAFMT_D88,
	DISK_DATAFMT_DSK,
	DISK_DATAFMT_DSKEX,
	DISK_DATAFMT_MAXNUM
} DISK_DATAFMT_TYPE;

typedef enum {
	DISK_CMD_NONE_I = 0,	/* Init. or end Type-I commands */
	DISK_CMD_NONE_II_III,	/* end Type-II/III commands */
	DISK_CMD_NONE_II_III_WRITE, /* end Type-II/III write commands */
	DISK_CMD_SEEK,		/* TYPE-I */
	DISK_CMD_READ,		/* TYPE-II */
	DISK_CMD_READMULTI,
	DISK_CMD_WRITE,
	DISK_CMD_WRITEMULTI,
	DISK_CMD_READADDRESS,	/* TYPE-III */
	DISK_CMD_READTRACK,
	DISK_CMD_WRITETRACK,
	DISK_CMD_MAXNUM
} DISK_CMD_TYPE;

/*
	Cyclic Redundancy Check CRC-16-CCITT

	Cyclic Code Generator Polynomial
		GF(X) = X^16+X^12+X^5+1
	Where X=2,
		GF(2) = 2^16+2^12+2^5+1 = 0x11021 (69665)
*/
#define DISK_CRC_GENERATOR_POLYNOMIAL_VALUE	0x11021
#define DISK_CRC_GENERATOR_POLYNOMIAL_USE \
	((unsigned short)(DISK_CRC_GENERATOR_POLYNOMIAL_VALUE & 0xffff))

typedef struct {
	/* disk data status */
	volatile int attr;
	volatile int writable;
	volatile int modified;
#ifdef WIN32
	volatile DWORD disksize;
#else
	volatile unsigned long disksize;
#endif
	volatile unsigned char *disk_buffer;
#ifdef WIN32
	volatile char filename[_MAX_PATH];
#else
	volatile char filename[PATH_MAX];
#endif
	volatile char diskname[D88_DISKNAME_MAXLEN + 1];
	volatile u8 maxtrack;

	/* current fdc status */
	int cmd;
	int substatus;
	int index;
	u8 cylinder;
	u8 m_track;	/* track(cylinder+head) infor mation for next multi-sector R/W or R ID */
	u8 sector;
	u8 maxsector;
	u8 status;
	u8 sidechk;		/* bits 1,0 = C,S */
	int trackchkfail;	/* for Seek Commands, which is V=1 */
	volatile unsigned char *dadr;		/* data address pointer by search_adr() */
	volatile unsigned char *dadr_id;	/* id information for dadr */
	unsigned char did[6];
	int fm_mode;		/* FDD running mode (MFM/FM) */
		/* MZ-80B/2000/2200 FDD I/F can access only MFM, it's fixed FALSE(0) */
	int deleted_data;	/* current reading data info (boolean) */
	int crc_error;		/*	//		     (boolean) */
	int have_sector_data;	/*	//		     (sector size, ex. 256) */
	int waitct;
	int seektime;		/* 0-3 for 6/12/20/30ms */
	z80clk oldclk;
} DISK_STRUCT;

static const int sctlen[] = {
	0x080,	/* N=0 */
	0x100,	/* N=1 */
	0x200,	/* N=2 */
	0x400	/* N=3 */
};

static MZ2000 *gmz = NULL;
static volatile DISK_STRUCT disk_struct[DISK_DRIVE_MAXNUM];
static unsigned char *disk_tempbuff = NULL;
/* running mode */
static int disk_mode_execfast = TRUE;	/* not used flag */
static int disk_mode_read_ignorecrc = FALSE;
static int disk_mode_format_erasesct17crc = FALSE;
/* for Internal IPL */
static int current_drive = -1;
char disk_iplmsg[DISK_IPLMSG_MAXNUM + 1];
/* FDC internal information */
static int disk_step_direction = TRUE;
static unsigned long disk_wait_clocks[4];
/* FDC CRC */
static unsigned short crc_table[256];
/* for FDD interface (not FDC) */
static int disk_enablemotor = FALSE;
static int disk_motorcount = 0;
/* track buffers for ReadTrack / WriteTrack type-III commands */
static int disk_trkbytes = 0;
static unsigned char *disk_trkbuff = NULL;
/* sectors in track information */
static int disk_idnum;
/*   0 1 2 3 4    5    6   7       (size=8)
     C H R N CRC1 CRC2 DDM DataCRC           */
static unsigned char disk_idlist[256][8];
static unsigned char *disk_idlist_dtptr[256];

int disk_updatetimer( void )
{
	static int lct = 0;

	lct++;
	if (lct > 100 / MZ2000_REFRESHRATE - 1) {
		if (gmz && (disk_enablemotor || (disk_motorcount > 0)))
			sound_playwav_file( gmz, SOUND_WAV_FDMOTOR );
		lct = 0;
		if (!disk_enablemotor && disk_motorcount > 0) {
			disk_motorcount -= 10;
			if (disk_motorcount < 0)
				disk_motorcount = 0;
		}
	}
	return 0;
}

static int check_writeprotect( int drive )
{
	if (!disk_struct[drive].writable)
		return TRUE;
	if (disk_struct[drive].attr != DISK_DATAFMT_D88)
		return FALSE;
	if (!disk_struct[drive].disk_buffer)
		return TRUE;
	/* check D88 writable bytes */
	return disk_struct[drive].disk_buffer[D88_OFFSET_WRITEPROTECT] ? TRUE : FALSE;
}

static void make_crctable( void )
{
	int i, j;

	for (i = 0; i < 256; i++){
		unsigned short c = i << 8;

		for (j = 0; j < 8; j++)
			c = (c << 1) ^ ((c & 0x8000) ? DISK_CRC_GENERATOR_POLYNOMIAL_USE : 0);
		crc_table[i] = c;
        }
	return;
}

static unsigned int calccrc( unsigned char *buf, int len )
{
	int i;
	unsigned short c = 0xffff;

	for (i = 0; i < len; i++)
		c = (c << 8) ^ crc_table[(c >> 8) ^ buf[i]];
	return c;
}

static int setcrc_in_id( int drive, unsigned char *did )
{
	unsigned short crc;
	unsigned char crc_for_id[8];

	crc_for_id[0] = 0xA1;	/* ID MFM (A1 A1 A1 FE C H R N CRC1 CRC2) */
	crc_for_id[1] = 0xA1;
	crc_for_id[2] = 0xA1;
	crc_for_id[3] = 0xFE;
	memcpy( &crc_for_id[4], did, 4 );
	crc = calccrc( crc_for_id, 8 );
	did[4] = crc >> 8;	/* CRC1 */
	did[5] = crc & 0xffU;	/* CRC2 */
	return 0;
}

/*
	Search the data addresses pointer and set the data to disk_struct[drive].*
		The track parameters are not the MZ sides. It's the standard track numbers.

	return: N
		<0	... error (value is only -1)
		0	...  128bytes (sctlen[0])
		1	...  256bytes (sctlen[1])
		2	...  512bytes (sctlen[2])
		3	... 1024bytes (sctlen[3])
		4-255	... MB8877 executes N=N & 0x03
			    When disk format data size > sctlen[x],
			    This FDD program update only the MB8877 sector size in the image file.
			    (NEC uPD765 FDC can use N>=4, it has >1024 length data. Ignore oversizes.)
*/
static int search_adr( int drive, int trk, int sct, u8 sidechk_param )
{
	int offset, i, flag, max, tval;
	int maxtrks, maxsides;
	unsigned char *t, *dt;
	unsigned long tmp;

	disk_struct[drive].have_sector_data = 0;
	disk_struct[drive].dadr    = NULL;
	disk_struct[drive].dadr_id = NULL;
	disk_struct[drive].have_sector_data = 0;
	disk_struct[drive].maxsector = 0;
	if (trk < 0 || trk >= disk_struct[drive].maxtrack)
		return -1;

	/* search the random sectors */
	switch (disk_struct[drive].attr) {
	case DISK_DATAFMT_2D :
		if (sct < 1 || sct > 16)
			return -1;		/* no sectors */
		if (disk_struct[drive].disksize < (unsigned)(trk*16*sctlen[1] + (sct-1)*sctlen[1])) {
			/* over disk data length or truncated disks */
			return -1;
		}
		disk_struct[drive].maxsector = 16;
		disk_struct[drive].dadr = &(disk_struct[drive].disk_buffer[
				trk*16*sctlen[1] + (sct-1)*sctlen[1]]);
		disk_struct[drive].have_sector_data = 256;
		disk_struct[drive].deleted_data = FALSE;
		disk_struct[drive].crc_error    = FALSE;
		disk_struct[drive].did[0] = trk >> 1;	/* C */
		disk_struct[drive].did[1] = trk & 1;	/* H */
		disk_struct[drive].did[2] = sct;	/* R */
		disk_struct[drive].did[3] = 1;		/* N */
		setcrc_in_id( drive, (unsigned char *)disk_struct[drive].did );
		return 1;				/* N=1 */
	case DISK_DATAFMT_D88 :
		/* track table adrs. */
		if (disk_struct[drive].disksize < (unsigned long)D88_OFFSET_TRACKTABLE + trk * sizeof(int) + 4)
			return -1;		/* broken */
		offset = *(unsigned int *)&(disk_struct[drive].disk_buffer[
			D88_OFFSET_TRACKTABLE + trk * sizeof(int)]);
		if (!offset)
			return -1;		/* broken */
		if (disk_struct[drive].disksize < (unsigned long)offset + 4)
			return -1;		/* broken */

		/* track data adrs. */
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset]);
		if (t == (unsigned char *)&(disk_struct[drive].disk_buffer[offset+sizeof(int)]))
			return -1;		/* ?? ... this code is unknown ... */
		i = 0;
		max = 0;
		while (1) {
			if (disk_struct[drive].disksize < (unsigned long)&t[0x10] - (unsigned long)disk_struct[drive].disk_buffer)
				return -1;		/* broken */
			if (!i) {
				max = t[4] | (t[5] << 8);
				disk_struct[drive].maxsector = max;
			}
			if (!max) {
				disk_struct[drive].maxsector = 0;
				return -1;	/* unformat ? */
			}
			flag = TRUE;
#if 0			/* check only seek commands, which is V=1 */
			if (t[0] != trk >> 1)	/* C */
				flag = FALSE;
#endif
			if (sidechk_param) {	/* H */
				if ((t[1] & 0x01U) != (sidechk_param & 0x01U))
				/*  if (t[1] != (trk & 1)) ... not use */
					flag = FALSE;
			}
			if (t[2] != sct)	/* R */
				flag = FALSE;
			if ((t[6] ? TRUE : FALSE) != disk_struct[drive].fm_mode)
				flag = FALSE;
			disk_struct[drive].deleted_data = t[7] ? TRUE : FALSE;
			disk_struct[drive].crc_error = FALSE;
			switch (t[8] & 0xf0) {	/* NEC PC-9801 DISK BIOS status */
			case 0x00 :	/* normal ends */
				break;
#if 0
			case 0xa0 :	/* ID CRC errors */
			case 0xb0 :	/* Data CRC errors */
			case 0xe0 :	/* ID Missing Address Mark */
			case 0xf0 :	/* Data Missing Address Mark */
			/* Perhaps, disk images has none ... */
			case 0x10 :	/* DDAM or Write Protect, but DDAM is t[7] */
			case 0x20 :	/* Hardware errors */
			case 0x30 :	/*   //   */
			case 0x40 :	/*   //   */
			case 0x50 :	/*   //   */
			case 0x60 :	/*   //   */
			case 0x70 :	/*   //   */
			case 0x80 :	/*   //   */
			case 0x90 :	/*   //   */
			case 0xc0 :	/* No sectors, Seek errors */
			case 0xd0 :	/* Bad Cylinders */
#endif
			default :
				disk_struct[drive].crc_error = TRUE;
				break;

			}
			disk_struct[drive].have_sector_data = t[0x0e] | (t[0x0f] << 8);
			memcpy( (void *)disk_struct[drive].did, &t[0], 4 );
			setcrc_in_id( drive, (unsigned char *)disk_struct[drive].did );
#if 0
			if (flag) {
				_TRACE("DISK ID TRK:%d/SCT:%d", trk, sct);
				for (k = 0; k < 6; k++) {
					_TRACE(" %02x", disk_struct[drive].did[k]);
				}
				_TRACE("\n");
			}
#endif
			if (flag) {
				disk_struct[drive].dadr = &t[0x10];
				disk_struct[drive].dadr_id = t;
				return t[3];	/* N */
			}
			t += (disk_struct[drive].have_sector_data + 0x10);
			i++;
			if (i >= max)
				break;
		}
		disk_struct[drive].have_sector_data = 0;
		return -1;
	case DISK_DATAFMT_DSK :
	case DISK_DATAFMT_DSKEX :
		/* track table adrs. */
		if (disk_struct[drive].disksize < (unsigned long)DSK_DISKTRKSTABLES_OFFS)
			return -1;		/* broken */
		tval = disk_struct[drive].disk_buffer[DSK_DISKTRACKS_OFFS];
		maxtrks = tval;
		tval = disk_struct[drive].disk_buffer[DSK_DISKSIDES_OFFS];
		if (tval != 1 && tval != 2)
			return -1;	/* only side == 1 or 2 */
		maxsides = tval;
		if (trk >= maxtrks * 2)
			return -1;	/* trk is over ranges */

		offset = DSK_DISKTRKSTABLES_OFFS;
		for (i = 0; i < trk; i++) {
			if (disk_struct[drive].attr != DISK_DATAFMT_DSKEX) {
				tmp = (*((unsigned char *)&(disk_struct[drive].disk_buffer[
					DSKNM_TRACKSIZE_OFFS + 1])) << 8) |
				    *((unsigned char *)&(disk_struct[drive].disk_buffer[DSKNM_TRACKSIZE_OFFS]));
			} else {
				tmp = *(unsigned char *)&(disk_struct[drive].disk_buffer[
					DSKEX_DISKTRKTBL_OFFS + i]) * 0x100;
			}
			offset += tmp;
			if (i == trk -1 && !tmp) {	/* unformat a track */
				offset = 0;
				break;
			}
			if (disk_struct[drive].disksize < (unsigned long)offset)
				return -1;		/* broken */
			if (maxsides == 1)
				i++;			/* 1D or 1DD */
		}
		//_TRACE("dsk %d %08x\n", trk, offset);
		if (!offset) {
			disk_struct[drive].maxsector = 0;
			return -1;		/* no track data */
		}
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset]);
		if (memcmp( t, "Track-Info\r\n", DSK_TRKSEYECATCH_LEN ))
			return -1;		/* eyecache compare error */

		/* track information blocks */
		max = *(unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKS_MAXSECTORS_OFFS]);
		disk_struct[drive].maxsector = max;
		if (!max)
			return -1;		/* unformat ? */
		i = 0;
		/* t = head of sector information list */
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKS_SCTINFOLIST_OFFS]);
		/* dt = sector data */
		dt = (unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKSINFO_LEN]);
		while (1) {
			//_TRACE("searching ... %d/%d %08x %08x\n", i, max, t, dt);
			if (disk_struct[drive].disksize < (unsigned long)dt - (unsigned long)disk_struct[drive].disk_buffer)
				return -1;	/* broken */

			flag = TRUE;
			if (sidechk_param) {	/* H */
				if ((t[1] & 0x01U) != (sidechk_param & 0x01U))
					flag = FALSE;
			}
			if (t[2] != sct)	/* R */
				flag = FALSE;
			if (disk_struct[drive].attr == DISK_DATAFMT_DSKEX) {
				tval = disk_struct[drive].disk_buffer[offset + DSKEX_SCTINFO_DATARATE_OFFS];
				if (!(!tval || tval == 1))
					flag = FALSE;	/* only double density */
				tval = disk_struct[drive].disk_buffer[offset + DSKEX_SCTINFO_RECMODE_OFFS];
				if (!(!tval || tval == 2))
					flag = FALSE;	/* only unknown or MFM */
			}
			disk_struct[drive].deleted_data = !!(t[5] & 0x40U); /* deleted data */
			disk_struct[drive].crc_error    = !!(t[4] != 0 || (t[5] != 0 && t[5] != 0x40U)); /* CRC errors */
			disk_struct[drive].have_sector_data = sctlen[t[3] & 0x03];
			memcpy( (void *)disk_struct[drive].did, &t[0], 4 );
			setcrc_in_id( drive, (unsigned char *)disk_struct[drive].did );
			if (flag) {
				//_TRACE("trk:%d-%d len=%02x %08x\n", trk, sct, t[3], dt);
				disk_struct[drive].dadr = dt;
				disk_struct[drive].dadr_id = t;
				return t[3];	/* N */
			}
			t += DSK_SCTINFO_ONELEN;
			dt += disk_struct[drive].have_sector_data;
			i++;
			if (i >= max)
				break;
		}
		return -1;
	default :
		break;
	}
	return -1;
}

/*
	Update the status of the FD images after search_adr() and the modified FD data

	disk_struct[].deleted_data;
	disk_struct[].d88_status;
*/
static int update_fdimg_status( int drive )
{
	volatile unsigned char *t;

	switch (disk_struct[drive].attr) {
	case DISK_DATAFMT_2D :
		break;
	case DISK_DATAFMT_D88 :
		if (disk_struct[drive].have_sector_data <= 0)
			break;
		if (!disk_struct[drive].dadr)
			break;
		if (!disk_struct[drive].dadr_id)
			break;
		t = disk_struct[drive].dadr_id;
		/* MFM/FM t[6] is set by search_adr() */
		/* deleted data */
		t[7] = disk_struct[drive].deleted_data ? 0x10 : 0x00;
		/* status (00=normal ends, others = NEC PC-9801 DISK BIOS status) */
		t[8] = disk_struct[drive].crc_error ? 0xb0 : 0;
		break;
	case DISK_DATAFMT_DSK :
	case DISK_DATAFMT_DSKEX :
		if (disk_struct[drive].have_sector_data <= 0)
			break;
		if (!disk_struct[drive].dadr)
			break;
		if (!disk_struct[drive].dadr_id)
			break;
		t = disk_struct[drive].dadr_id;
		if (disk_struct[drive].deleted_data)
			t[5] = 0x40U;
		t[4] =  disk_struct[drive].crc_error ? 0x20U : 0;
		t[5] |= disk_struct[drive].crc_error ? 0x20U : 0;
		/* ignored disk_struct[drive].deleted_data */
		break;
	default :
		break;
	}
	return 0;
}

/*
	Make the id list to disk_idlist[][], disk_idlist_dtptr[], disk_idnum in tracks.
		The track parameters are not the MZ sides. It's the standard track numbers.

	return: >=0	... ID numbers (no errors)
		<0	... errors (value is only -1)
*/
static int make_idtbl( int drive, int trk )
{
	int offset, i, flag, max, have_sector_data;
	int maxtrks, maxsides;
	unsigned char *t, *dt, tval;
	unsigned long tmp;

	disk_idnum = 0;
	if (trk < 0 || trk >= disk_struct[drive].maxtrack)
		return -1;

	/* search the random sectors */
	switch (disk_struct[drive].attr) {
	case DISK_DATAFMT_2D :
		if (disk_struct[drive].disksize < (unsigned)(trk*16*sctlen[1])) {
			/* over disk data length or truncated disks  */
			/* unformat data regions */
			return -1;
		}
		for (i = 0; i < 16; i++) {
			disk_idlist_dtptr[i] = (unsigned char *)
				&(disk_struct[drive].disk_buffer[trk*16*sctlen[1] + i*sctlen[1]]);
			disk_idlist[i][0] = trk >> 1;	/* C */
			disk_idlist[i][1] = trk & 1;	/* H */
			disk_idlist[i][2] = i;		/* R */
			disk_idlist[i][3] = 1;		/* N */
			setcrc_in_id( drive, disk_idlist[i] );
			disk_idlist[i][6] = FALSE;	/* No CRC errors */
			disk_idlist[i][7] = FALSE;	/* No Deleted Data Mark */
			disk_idnum++;			/* 16 < DISK_SECTOR_MAXNUM */
		}
#if 0
		_TRACE( "DISK ID TRK:%d (2D)\n", trk );
		{ int j, k;
		for (j = 0; j < disk_idnum; j++) {
			_TRACE("%3d:", j);
			for (k = 0; k < 8; k++)
				_TRACE(" %02x", disk_idlist[j][k]);
			_TRACE(" %08x", disk_idlist_dtptr[j]);
			_TRACE("\n");
		}}
#endif
		return disk_idnum;
	case DISK_DATAFMT_D88 :
		/* track table adrs. */
		if (disk_struct[drive].disksize < (unsigned long)D88_OFFSET_TRACKTABLE + trk * sizeof(int) + 4)
			return -1;		/* broken */
		offset = *(unsigned int *)&(disk_struct[drive].disk_buffer[
			D88_OFFSET_TRACKTABLE + trk * sizeof(int)]);
		if (!offset)
			return -1;		/* broken */
		if (disk_struct[drive].disksize < (unsigned long)offset + 4)
			return -1;		/* broken */

		/* track data adrs. */
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset]);
		if (t == (unsigned char *)&(disk_struct[drive].disk_buffer[offset+sizeof(int)]))
			return -1;		/* ?? ... this code is unknown ... */
		i = 0;
		max = 0;
		while (1) {
			if (disk_struct[drive].disksize < (unsigned long)&t[0x10] - (unsigned long)disk_struct[drive].disk_buffer)
				return -1;		/* broken */
			if (!i)
				max = t[4] | (t[5] << 8);
			if (!max)
				return -1;	/* unformat ? */
			flag = TRUE;
			if ((t[6] ? TRUE : FALSE) != disk_struct[drive].fm_mode)
				flag = FALSE;
			disk_idlist[i][6] = t[7] ? TRUE : FALSE;	/* deleted data */
			disk_idlist[i][7] = FALSE;			/* No CRC errors */
			switch (t[8] & 0xf0) {	/* NEC PC-9801 DISK BIOS status */
			case 0x00 :	/* normal ends */
				break;
#if 0
			case 0xa0 :	/* ID CRC errors */
			case 0xb0 :	/* Data CRC errors */
			case 0xe0 :	/* ID Missing Address Mark */
			case 0xf0 :	/* Data Missing Address Mark */
			/* Perhaps, disk images has none ... */
			case 0x10 :	/* DDAM or Write Protect, but DDAM is t[7] */
			case 0x20 :	/* Hardware errors */
			case 0x30 :	/*   //   */
			case 0x40 :	/*   //   */
			case 0x50 :	/*   //   */
			case 0x60 :	/*   //   */
			case 0x70 :	/*   //   */
			case 0x80 :	/*   //   */
			case 0x90 :	/*   //   */
			case 0xc0 :	/* No sectors, Seek errors */
			case 0xd0 :	/* Bad cylinders */
#endif
			default :
				disk_idlist[i][7] = TRUE;		/* CRC errors */
				break;

			}
			have_sector_data = t[0x0e] | (t[0x0f] << 8);
			memcpy( (void *)disk_idlist[i], &t[0], 4 );
			setcrc_in_id( drive, disk_idlist[i] );
			disk_idlist_dtptr[i] = &t[0x10];
			if (flag) {
				/* This ID is enabled */
				disk_idnum++;
			}
			t += (have_sector_data + 0x10);
			i++;
			if (i >= max)
				break;
		}
#if 0
		_TRACE( "DISK ID TRK:%d (D88)\n", trk );
		{ int j, k;
		for (j = 0; j < disk_idnum; j++) {
			_TRACE("%3d:", j);
			for (k = 0; k < 8; k++)
				_TRACE(" %02x", disk_idlist[j][k]);
			_TRACE(" %08x", disk_idlist_dtptr[j]);
			_TRACE("\n");
		}}
#endif
		return disk_idnum;
	case DISK_DATAFMT_DSK :
	case DISK_DATAFMT_DSKEX :
		/* track table adrs. */
		if (disk_struct[drive].disksize < (unsigned long)DSK_DISKTRKSTABLES_OFFS)
			return -1;		/* broken */
		tval = disk_struct[drive].disk_buffer[DSK_DISKTRACKS_OFFS];
		maxtrks = tval;
		tval = disk_struct[drive].disk_buffer[DSK_DISKSIDES_OFFS];
		if (tval != 1 && tval != 2)
			return -1;	/* only side == 1 or 2 */
		maxsides = tval;
		if (trk >= maxtrks * 2)
			return -1;	/* trk is over ranges */

		offset = DSK_DISKTRKSTABLES_OFFS;
		for (i = 0; i < trk; i++) {
			if (disk_struct[drive].attr != DISK_DATAFMT_DSKEX) {
				tmp = (*((unsigned char *)&(disk_struct[drive].disk_buffer[
					DSKNM_TRACKSIZE_OFFS + 1])) << 8) |
				    *((unsigned char *)&(disk_struct[drive].disk_buffer[DSKNM_TRACKSIZE_OFFS]));
			} else {
				tmp = *(unsigned char *)&(disk_struct[drive].disk_buffer[
					DSKEX_DISKTRKTBL_OFFS + i]) * 0x100;
			}
			offset += tmp;
			if (i == trk -1 && !tmp) {	/* unformat a track */
				offset = 0;
				break;
			}
			if (disk_struct[drive].disksize < (unsigned long)offset)
				return -1;		/* broken */
			if (maxsides == 1)
				i++;			/* 1D or 1DD */
		}
		//_TRACE("dsk %d %08x\n", trk, offset);
		if (!offset) {
			return -1;		/* no track data */
		}
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset]);
		if (memcmp( t, "Track-Info\r\n", DSK_TRKSEYECATCH_LEN ))
			return -1;		/* eyecache compare error */

		/* track information blocks */
		max = *(unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKS_MAXSECTORS_OFFS]);
		if (!max)
			return -1;		/* unformat ? */
		i = 0;
		/* t = head of sector information list */
		t = (unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKS_SCTINFOLIST_OFFS]);
		/* dt = sector data */
		dt = (unsigned char *)&(disk_struct[drive].disk_buffer[offset + DSK_TRKSINFO_LEN]);
		while (1) {
			//_TRACE("searching ... %d/%d %08x %08x\n", i, max, t, dt);
			if (disk_struct[drive].disksize < (unsigned long)dt - (unsigned long)disk_struct[drive].disk_buffer)
				return -1;	/* broken */

			flag = TRUE;
			disk_idlist[i][6] = !!(t[5] & 0x40U); /* deleted data */
			disk_idlist[i][7] = !!(t[4] != 0 || (t[5] != 0 && t[5] != 0x40U)); /* CRC errors */
			have_sector_data = sctlen[t[3] & 0x03];
			memcpy( (void *)disk_idlist[i], &t[0], 4 );
			setcrc_in_id( drive, disk_idlist[i] );
			disk_idlist_dtptr[i] = dt;
			if (flag) {
				/* this ID is enabled */
				disk_idnum++;
			}
			t += DSK_SCTINFO_ONELEN;
			dt += have_sector_data;
			i++;
			if (i >= max)
				break;
		}
#if 0
		_TRACE( "DISK ID TRK:%d (DSK)\n", trk );
		{ int j, k;
		for (j = 0; j < disk_idnum; j++) {
			_TRACE("%3d:", j);
			for (k = 0; k < 8; k++)
				_TRACE(" %02x", disk_idlist[j][k]);
			_TRACE(" %08x", disk_idlist_dtptr[j]);
			_TRACE("\n");
		}}
#endif
		return disk_idnum;
	default :
		break;
	}
	return -1;
}


/*
	Make the id list for checking cyliders
		The track parameters are not the MZ sides. It's the standard track numbers.

	return: >=0	... ID numbers (no errors)
		<0	... errors (value is only -1)
*/
int make_idtbl_checkcylinder( int drive, int trk, int cylinder_for_checks )
{
	int maxid, i, cyl;

	maxid = make_idtbl( drive, trk );
	if (maxid <= 0)		/* errors or sectors not found */
		return -1;
	//for (i = 0; i < maxid; i++)
	i = rand() % maxid;
	{
		cyl = disk_idlist[i][0]; /* C */
		if (cyl != cylinder_for_checks)
			return -1;
	}
	return maxid;
}

static int make_trackimage_setbyte( unsigned char **p, unsigned char c )
{
	if ((unsigned long)(*p) - (unsigned long)disk_trkbuff < (unsigned long)disk_trkbytes)
		*(*p)++ = c;
	return 0;
}

/*
	Make the track images for the read tracks. (physical format types are only ISO)

	It's supported only the MFM images.
*/
static int make_trackimage( int drive, int trk )
{
	int i, j, len, total;
	int gap1, gap2, gap3;
	unsigned char *p, *bp;
	unsigned short crc;

	if (make_idtbl( drive, trk ) < 0)
		;	/* ignore, because make unformatted data */
	memset( disk_trkbuff, 0x4e, DISK_TRACKBUFF_SIZE );
	p = disk_trkbuff;
	disk_trkbytes = DISK_TRACK_BYTES;
	if (disk_idnum < 1)
		return 0;	/* unformat data */

	/*
	    allbytes = gap1+(28+gap2+24+datalen+gap3)*disk_idnum+gap4 < DISK_TRACK_BYTES(about 0x1850)

	    Ex. sectors(disk_idnum)=16 N=1(datalen=256)

		gap1 = 0x32;		//only first
		gap2 = 0x22;
		gap3 = 0x54;
		gap4 = automatic	//only last
	*/

	/* calculating data size without gaps */
	total = 0;
	for (i = 0; i < disk_idnum; i++) {
		total += 28;	/* ID */
		total += 24;	/* Sync + AM2 + CRC(data) */
		total += sctlen[disk_idlist[i][3] & 0x03];
	}
	/* determine to gaps */
	gap1 = 0x30;
	gap2 = 0x20;
	total -= gap1;
	total -= 0x20; /* gap4 */
	total -= gap2 * disk_idnum;
	total = DISK_TRACK_BYTES - total;
	if (total < 0)
		total = 1;
	total /= disk_idnum;
	if (total < 5)
		total = 5;
	gap3 = total;

	/* make track data */
	/* GAP1 */
	len = gap1;
	for (j = 0; j < len; j++)
		make_trackimage_setbyte( &p, 0x4e );
	for (i = 0; i < disk_idnum; i++) {
		/* ID size=0x12(18)+(3+1)+(4+2)=28 */
		/* SYNC */
		len = 0x12;
		for (j = 0; j < len; j++)
			make_trackimage_setbyte( &p, 0x00 );
		/* AM1 */
		for (j = 0; j < 3; j++)
			make_trackimage_setbyte( &p, 0xa1 );
		make_trackimage_setbyte( &p, 0xfe );
		/* ID */
		make_trackimage_setbyte( &p, disk_idlist[i][0] ); /* C */
		make_trackimage_setbyte( &p, disk_idlist[i][1] ); /* H */
		make_trackimage_setbyte( &p, disk_idlist[i][2] ); /* R */
		make_trackimage_setbyte( &p, disk_idlist[i][3] ); /* N */
		/* ID (CRC) */
		make_trackimage_setbyte( &p, disk_idlist[i][4] ); /* CRC1 */
		make_trackimage_setbyte( &p, disk_idlist[i][5] ); /* CRC2 */
		/* GAP2 */
		len = gap2;
		for (j = 0; j < len; j++)
			make_trackimage_setbyte( &p, 0x4e );
		/* DATA size=0x12(18)+(3+1)+datalen+2=24+datalen */
		/* SYNC */
		len = 0x12;
		for (j = 0; j < len; j++)
			make_trackimage_setbyte( &p, 0x00 );
		/* set pointer for calcurating CRC */
		bp = p;
		/* AM2 */
		for (j = 0; j < 3; j++)
			make_trackimage_setbyte( &p, 0xa1 );
		if (!disk_idlist[i][7])
			make_trackimage_setbyte( &p, 0xfb );	/* data mark */
		else
			make_trackimage_setbyte( &p, 0xf8 );	/* deleted data mark */
		/* DATA */
		for (j = 0; j < sctlen[disk_idlist[i][3] & 0x03]; j++)
			make_trackimage_setbyte( &p, (disk_idlist_dtptr[i])[j] );
		/* CRC */
		crc = calccrc( bp, (unsigned long)p - (unsigned long)bp + 1 );
		if (disk_idlist[i][6])
			crc ^= 0xffffU;	/* make CRC error */
		make_trackimage_setbyte( &p, (unsigned char)(crc >> 8) );	/* CRC1 */
		make_trackimage_setbyte( &p, (unsigned char)(crc & 0xffU) );	/* CRC2 */
		/* GAP3 */
		len = gap3;
		for (j = 0; j < len; j++)
			make_trackimage_setbyte( &p, 0x4e );
	}
	return 0;
}

/*
	Anaylyze the track images for the write track and set the data to disk_idlist.
 
	It's supported only the MFM images.
*/
static int analyze_trackimage( int drive, int trk )
{
	int i, len;
	int ct_sync, have_id, ddm_flag;
	unsigned char *p;

	disk_idnum = 0;
	p = disk_trkbuff;
	ct_sync = 0;
	have_id = FALSE;
	i = 0;
	while ((unsigned long)p - (unsigned long)disk_trkbuff < (unsigned long)disk_trkbytes) {
		if (*p++ != 0xf5) {
			ct_sync = 0;
			continue;
		}
		ct_sync++;
		if (ct_sync < 3)
			continue;
		/* found F5 F5 F5 (A1 A1 A1 with missing clock) */
		switch (*p) {
		case 0xfe : /* ID */
			if (have_id) {
				/* no data. which only ID, ignore it */
				/* generally, it is used to the copy protect */
			}
			p++;
			memset( disk_idlist[i], 0, 8 );
			disk_idlist[i][0] = *p++; /* C */
			disk_idlist[i][1] = *p++; /* H */
			disk_idlist[i][2] = *p++; /* R */
			disk_idlist[i][3] = *p++; /* N */
			disk_idlist[i][4] = 0;	  /* CRC1 */
			disk_idlist[i][5] = 0;	  /* CRC2 */
			if (*p == 0xf7)
				p++;
			else {
				p++;
				p++;	/* ignore special F7 position, etc. */
			}
			have_id = TRUE;
			break;
		case 0xfb : /* Data Mark */
		case 0xf8 : /* Deleted Data Mark */
			ddm_flag = (*p == 0xf8);
			p++;
			if (!have_id)
				continue;
			disk_idlist[i][7] = ddm_flag;
			disk_idlist_dtptr[i] = p;		/* ignore F5-F7 special data */
			len = sctlen[disk_idlist[i][3] & 0x03];
			p += len;
			if (*p != 0xf7)				/* check format CRC errors */
				disk_idlist[i][6] = TRUE;
			p++;
			if ((unsigned long)p - (unsigned long)disk_trkbuff > (unsigned long)disk_trkbytes)
				disk_idlist[i][6] = TRUE;	/* set CRC errors */
			i++;
			break;
		default :
			p++;
			continue;
		}
	}
	disk_idnum = i;
	if (disk_mode_format_erasesct17crc && disk_idnum == 17) {
		i = 16;
		if (disk_idlist[i][2] == 17 && disk_idlist[i][6]) {
			/* 17 sectors and CRC errors, erase the last sector */
			disk_idnum = 16;
		}
	}

#if 0
	_TRACE( "DISK ID TRK:%d (DSK)\n", trk );
	{ int j, k;
	for (j = 0; j < disk_idnum; j++) {
		_TRACE("%3d:", j);
		for (k = 0; k < 8; k++)
			_TRACE(" %02x", disk_idlist[j][k]);
		_TRACE(" %08x", disk_idlist_dtptr[j]);
		_TRACE("\n");
	}}
#endif
	return 0;
}

static int trackcopy_d88( int drive, unsigned char **dpp, unsigned char *sp )
{
	int i, max, have_sector_data;
	unsigned char *t, *td;

	i = 0;
	max = 0;
	t = sp;
	td = *dpp;
	while (1) {
		if (disk_struct[drive].disksize < (unsigned long)&t[0x10] - (unsigned long)disk_struct[drive].disk_buffer) {
			/* truncated disks, this is broken track, ignore this track... */
			return -1;
		}
		if (!i)
			max = t[4] | (t[5] << 8);
		if (!max)
			return -1;	/* this is broken track, ignore this track... */
		memcpy( td, t, 0x10 );
		have_sector_data = t[0x0e] | (t[0x0f] << 8);
		t  += 0x10;
		td += 0x10;
		memcpy( td, t, have_sector_data );
		t  += have_sector_data;
		td += have_sector_data;
		i++;
		if (i >= max)
			break;
	}
	*dpp = td;
	return 0;
}

/*
	Change the disk format with disk_idlist in a track.
*/
static int change_format_1trk( int drive, int trk )
{
	int i, j, len, flag;
	unsigned char *sp, *dp, *sp_s, *dp_s, *odp, *t;
	unsigned int dlen, *sp_trk, *dp_trk;		/* for 32bit offsets */

	switch (disk_struct[drive].attr) {
	case DISK_DATAFMT_2D :
		flag = FALSE;
		if (disk_idnum != 16 && disk_idnum != 17)
			return -1;
		for (i = 0; i < disk_idnum; i++) {
			if (disk_idlist[i][0] != trk / 2)	/* C */
				return -1;
			if (disk_idlist[i][1] != trk % 2)	/* H */
				return -1;
			if (disk_idlist[i][2] < 1 || disk_idlist[i][2] > 17) { 	/* R */
				/* interleave format is ok */
				return -1;
			}
			if (disk_idlist[i][3] != 1)		/* N */
				return -1;
			if (disk_idlist[i][6] && disk_idlist[i][2] != 17) /* CRC and N<>17 */
				return -1;
			if (disk_idlist[i][7])			/* DDM */
				return -1;
		}
		if (disk_idnum == 17) {		/* sector 17 is only CRC */
			i = 16;
			if (disk_idlist[i][2] != 17)
				return -1;
			if (!disk_idlist[i][6])
				return -1;
		}
		if ((unsigned int)(trk*16*sctlen[1] + 16*sctlen[1]) > disk_struct[drive].disksize) {
			/* extend file */
			memset( (unsigned char *)&(disk_struct[drive].disk_buffer[disk_struct[drive].disksize]),
				DISK_2D_FILLDATA, (trk*16*sctlen[1] + 16*sctlen[1]) - disk_struct[drive].disksize + 1 );
			disk_struct[drive].disksize = trk*16*sctlen[1] + 16*sctlen[1];
		}
		for (i = 0; i < 16; i++) {
			memcpy( (unsigned char *)&(disk_struct[drive].disk_buffer[trk*16*sctlen[1] + i*sctlen[1]]),
				disk_idlist_dtptr[i], sctlen[1] );
		}
		return 0;				/* N=1 */
	case DISK_DATAFMT_D88 :
		memset( disk_tempbuff, 0, DISK_IMAGEBUFF_SIZE );
		sp_s = sp = (unsigned char *)disk_struct[drive].disk_buffer;
		dp_s = dp = disk_tempbuff;
		/* copy D88 first header */
		memcpy( dp, sp, D88_FIRSTHEAD_SIZE );
		/* make ready to track tables in D88 header data */
		sp_trk = (unsigned int *)(sp + D88_OFFSET_TRACKTABLE);
		dp_trk = (unsigned int *)(dp + D88_OFFSET_TRACKTABLE);
		dp += D88_HEADDER_SIZE;
		/* copy the data 0 - (trk-1) */
		for (j = 0; j < trk; j++) {
			if (sp_trk[j]) {
				sp = &sp_s[sp_trk[j]];
				odp = dp;
				if (trackcopy_d88( drive, &dp, sp ) >= 0) {
					dp = (unsigned char *)(((unsigned long)dp + 3) & ~3);
					dp_trk[j] = (unsigned int)((unsigned long)odp - (unsigned long)dp_s);
				} else {
					/* broken track, unformat it */
					dp_trk[j] = 0;
				}
			} else {
				/* unformat track (no sectors) */
				dp_trk[j] = 0;
			}
		}
		/* make the data in a track */
		j = trk;
		if (disk_idnum > 0) {
			odp = dp;
			t = dp;
			for (i = 0; i < disk_idnum; i++) {
				t = dp;
				memset( t, 0, 0x10 );
				t[0] = disk_idlist[i][0];	/* C */
				t[1] = disk_idlist[i][1];	/* H */
				t[2] = disk_idlist[i][2];	/* R */
				t[3] = disk_idlist[i][3];	/* N */
				t[4] = disk_idnum & 0xff;	/* total sectors in this track */
				t[5] = disk_idnum >> 8;
				t[6] = 0x00;			/* MFM=0x00 / FM=0x40 */
				t[7] = disk_idlist[i][7] ? 0x10 : 0x00; /* DDM */
				t[8] = disk_idlist[i][6] ? 0xb0 : 0x00; /* Data CRC */
				len = sctlen[disk_idlist[i][3] & 0x03];
				t[0x0e] = len & 0xff;
				t[0x0f] = len >> 8;
				dp += 0x10;
				memcpy( dp, disk_idlist_dtptr[i], len );
				dp += len;
			}
			dp_trk[j] = (unsigned int)((unsigned long)odp - (unsigned long)dp_s);
		} else {
			/* this is a unformat track (no sectors) */
			dp_trk[j] = 0;
		}
		/* copy the data (trk+1) - D88_TRACKS_MAXNUM */
		for (j = trk + 1; j < D88_TRACKS_MAXNUM; j++) {
			if (sp_trk[j]) {
				sp = &sp_s[sp_trk[j]];
				odp = dp;
				if (trackcopy_d88( drive, &dp, sp ) >= 0) {
					dp = (unsigned char *)(((unsigned long)dp + 3) & ~3);
					dp_trk[j] = (unsigned int)((unsigned long)odp - (unsigned long)dp_s);
				} else {
					/* broken track, unformat it */
					dp_trk[j] = 0;
				}
			} else {
				/* this is a unformat track (no sectors) */
				dp_trk[j] = 0;
			}
		}
		dlen = (unsigned int)((unsigned long)dp - (unsigned long)dp_s);
		*(unsigned int *)(&dp_s[D88_OFFSET_DISKSIZE]) = dlen;
		/* copy from the temporary buffers to the real buffers */
		memcpy( (unsigned char *)disk_struct[drive].disk_buffer, disk_tempbuff, DISK_IMAGEBUFF_SIZE );
		disk_struct[drive].disksize = dlen;
		return 0;
	case DISK_DATAFMT_DSK :
	case DISK_DATAFMT_DSKEX :
		/* The DSK type disks are not supported to write track. */
		return -1;
	default :
		break;
	}
	return 0;
}

/*
	I/O port 0xD8-0xDB implements for FDC
*/
u8 disk_port_in_d8( u16 port )
{
	z80clk cclk, dwclk;
	int drive = gmz->ioinfo->port_dc & 0x03;
	u8 val = disk_struct[drive].status;

	/* FDC status */
	/*
		Common bit 7-6 ... Common informations
			7             6
			NotReady      WriteProtect
		Type-I/IV bit 5-0
			5             4           3      2        1       0
			HeadEngaged   SeekErr     CRCErr TRACK00  INDEX   Busy
		Type-II/III bit 5-0
			5             4           3      2        1       0
			              RecNotFound CRCErr LostData DataReq Busy
			RD:RecType (Deleted Data Mark=1)
			WT:WriteFault
	*/
	switch (disk_struct[drive].cmd) {
	/* end of commands */
	case DISK_CMD_NONE_I :
		if ((gmz->ioinfo->port_dc & 0x84U) == 0x84U) {	/* A drive was selected */
			/* index hole */
			if (!(rand() % INDEXHOLE_VALUE))
				val |= 0x02U;			/* status index hole */
		} else {					/* no drive selected */
			val |= 0x80U;				/* status not ready */
		}
		break;
	case DISK_CMD_NONE_II_III :
	case DISK_CMD_NONE_II_III_WRITE :
		if ((gmz->ioinfo->port_dc & 0x84U) == 0x84U) {	/* A drive was selected */
			;
		} else {					/* no drive selected */
			val |= 0x80U;				/* status not ready */
		}
		if (disk_struct[drive].cmd != DISK_CMD_NONE_II_III_WRITE)
			val &= ~0x40U;				/* fixed 0, no write protect information */
		break;
	/* Type - I FDC Commands */
	case DISK_CMD_SEEK :
		cclk = get_internal_clock();
		dwclk = disk_wait_clocks[disk_struct[drive].seektime];
		if (disk_struct[drive].waitct > 1) {
			if (cclk - disk_struct[drive].oldclk < dwclk)	/* seek wait */
				break;
			//if (disk_struct[drive].waitct > 2)
				sound_playwav_file( gmz, SOUND_WAV_FDSEEK );
			//else
			//	sound_playwav_file( gmz, SOUND_WAV_FDSEEK1 );
			disk_struct[drive].oldclk = get_internal_clock();
			--disk_struct[drive].waitct;
			break;
		}
		if (cclk - disk_struct[drive].oldclk < dwclk)	/* seek wait */
			break;
		disk_struct[drive].waitct = 0;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		if (!disk_struct[drive].cylinder)
			disk_struct[drive].status |= 0x24U;	/* status engaged and track00 */
		else
			disk_struct[drive].status |= 0x20U;	/* status engaged */
		if (disk_struct[drive].trackchkfail)
			disk_struct[drive].status |= 0x10U;	/* status seek error */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		break;
	/* Type - II / III FDC Commands */
	case DISK_CMD_READ :
	case DISK_CMD_READMULTI :
	case DISK_CMD_READADDRESS :
	case DISK_CMD_READTRACK :
		if (disk_struct[drive].substatus > 0) {
			--disk_struct[drive].substatus;
			if (!disk_struct[drive].substatus)
				disk_struct[drive].status |= 0x02U; /* status data request */
		}
		val &= ~0x40U;					/* fixed 0, no write protect information */
		break;
	case DISK_CMD_WRITE :
	case DISK_CMD_WRITEMULTI :
	case DISK_CMD_WRITETRACK :
		if (disk_struct[drive].substatus > 0) {
			--disk_struct[drive].substatus;
			if (!disk_struct[drive].substatus)
				disk_struct[drive].status |= 0x02U; /* status data request */
		}
		break;
	default :
		_TRACE("??? ... In FDC status, unsupported FDC command running ... CMD: %02x\n", disk_struct[drive].cmd);
		break;
	}
#ifdef ENBALE_TRACE_DISKIO_STATUS
	//When it's the enable, many messages are occurred and very slow...
	if (disk_struct[drive].cmd != DISK_CMD_SEEK && disk_struct[drive].cmd != DISK_CMD_NONE_I_IV)
	_TRACE( "ind8(status): ~%02x (cmd = %d)\n", val, disk_struct[drive].cmd );
#endif
	return ~val;
}

int disk_port_out_d8( u16 port, u8 value )
{
	u8 tt, val;
	int t, ddam;
	int drive = gmz->ioinfo->port_dc & 0x03;
	int side =  gmz->ioinfo->port_dd & 1;
	int sector = ~(gmz->ioinfo->port_da);

#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "outd8(cmd): ~%02x (&0xf0=%02x)\n", ~value & 0xff, ~value & 0xf0 );
#endif
	gmz->ioinfo->port_d8 = value;
	val = ~value;

	/* FDC commands */
	if ((val & 0xf0) == 0x00) {
		/* Restore (-> 0) type-I  (0 0 0 0 h V r1 r0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Restore (uhVr1r0=%02x)\n", val & 0x0f);
#endif
		disk_step_direction = FALSE;			/* seek to -TRK */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].trackchkfail = FALSE;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			disk_struct[drive].waitct = disk_struct[drive].cylinder;
			disk_struct[drive].seektime = val & 0x03;
			disk_struct[drive].cmd = DISK_CMD_SEEK;
			disk_struct[drive].cylinder = 0;
			gmz->ioinfo->port_d9 = ~(disk_struct[drive].cylinder);
			disk_struct[drive].status |= 0x24U;	/* status engaged and track00 */
			disk_struct[drive].status |= 0x01U;	/* status busy */
			disk_struct[drive].oldclk = get_internal_clock();
			if (!disk_struct[drive].waitct)
				disk_struct[drive].waitct = 1;
			else {
				if (disk_struct[drive].waitct == 1)
					sound_playwav_file( gmz, SOUND_WAV_FDSEEK1 );
				else
					sound_playwav_file( gmz, SOUND_WAV_FDSEEK );
			}

			/* check cylinders after seek */
			if (val & 0x04) { /* V=1 */
				t = (disk_struct[drive].cylinder) * 2 + side;
				if (make_idtbl_checkcylinder( drive, t, disk_struct[drive].cylinder ) < 0) {
					disk_struct[drive].trackchkfail = TRUE;
					disk_struct[drive].status |= 0x10U;	/* status seek error */
				} else
					disk_struct[drive].trackchkfail = FALSE;
			}
		}
	} else if ((val & 0xf0) == 0x10) {
		/* Seek (trk -> datareg) type-I  (0 0 0 1 h V r1 r0) */
		tt = ~(gmz->ioinfo->port_db);
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Seek (hVr1r0=%02x) to cyl:%02d\n", val & 0x0f, tt);
#endif
		if (tt > disk_struct[drive].cylinder)
			disk_step_direction = TRUE;		/* seek to +TRK */
		if (tt < disk_struct[drive].cylinder)
			disk_step_direction = FALSE;		/* seek to -TRK */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].trackchkfail = FALSE;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84 &&	/* A drive was selected */
				disk_struct[drive].cylinder != tt) {
			disk_struct[drive].waitct = abs( tt - disk_struct[drive].cylinder );
			disk_struct[drive].seektime = val & 0x03;
			disk_struct[drive].cmd = DISK_CMD_SEEK;
			if (!disk_struct[drive].cylinder)
				disk_struct[drive].status |= 0x24; /* status engaged and track00 */
			else
				disk_struct[drive].status |= 0x20; /* status engaged */
			disk_struct[drive].status |= 0x01U;	/* status busy */
			disk_struct[drive].cylinder = tt;
			gmz->ioinfo->port_d9 = ~tt;
			disk_struct[drive].oldclk = get_internal_clock();
			if (disk_struct[drive].waitct == 1)
				sound_playwav_file( gmz, SOUND_WAV_FDSEEK1 );
			else
				sound_playwav_file( gmz, SOUND_WAV_FDSEEK );

			/* check cylinders after seek */
			if (val & 0x04) { /* V=1 */
				t = (disk_struct[drive].cylinder) * 2 + side;
				if (make_idtbl_checkcylinder( drive, t, disk_struct[drive].cylinder ) < 0) {
					disk_struct[drive].trackchkfail = TRUE;
					disk_struct[drive].status |= 0x10U;	/* status seek error */
				} else
					disk_struct[drive].trackchkfail = FALSE;
			}
		}
	} else if ((val & 0xe0) == 0x40 || ((val & 0xe0) == 0x20 && disk_step_direction)) {
		/* Step  (seek +/-1) type-I  (0 0 1 u h V r1 r0) */
		/* Step In (seek +1) type-I  (0 1 0 u h V r1 r0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Step or in +1 (uhVr1r0=%02x)\n", val & 0x1f);
#endif
		disk_step_direction = TRUE;			/* seek to +TRK */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].trackchkfail = FALSE;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			disk_struct[drive].waitct = 1;
			disk_struct[drive].seektime = val & 0x03;
			disk_struct[drive].cmd = DISK_CMD_SEEK;
			if (!disk_struct[drive].cylinder)
				disk_struct[drive].status |= 0x24; /* status engaged and track00 */
			else
				disk_struct[drive].status |= 0x20; /* status engaged */
			disk_struct[drive].status |= 0x01U;	/* status busy */
			disk_struct[drive].cylinder++;
			if (disk_struct[drive].cylinder >= disk_struct[drive].maxtrack / 2)
				disk_struct[drive].cylinder = disk_struct[drive].maxtrack / 2 - 1;
			sound_playwav_file( gmz, SOUND_WAV_FDSEEK1 );
			disk_struct[drive].oldclk = get_internal_clock();
			if (val & 0x10) /* u=1 */
				gmz->ioinfo->port_d9 = ~(disk_struct[drive].cylinder);

			/* check cylinders after seek */
			if (val & 0x04) { /* V=1 */
				t = (disk_struct[drive].cylinder) * 2 + side;
				if (make_idtbl_checkcylinder( drive, t, disk_struct[drive].cylinder ) < 0) {
					disk_struct[drive].trackchkfail = TRUE;
					disk_struct[drive].status |= 0x10U;	/* status seek error */
				} else
					disk_struct[drive].trackchkfail = FALSE;
			}
		}
	} else if ((val & 0xe0) == 0x60 || ((val & 0xe0) == 0x20 && !disk_step_direction)) {
		/* Step   (seek +/-1) type-I  (0 0 1 u h V r1 r0) */
		/* Step Out (seek -1) type-I  (0 1 1 u h V r1 r0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Step or out -1 (uhVr1r0=%02x)\n", val & 0x1f);
#endif
		disk_step_direction = FALSE;			/* seek to -TRK */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].trackchkfail = FALSE;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			disk_struct[drive].waitct = 1;
			disk_struct[drive].seektime = val & 0x03;
			disk_struct[drive].cmd = DISK_CMD_SEEK;
			if (!disk_struct[drive].cylinder)
				disk_struct[drive].status |= 0x24; /* status engaged and track00 */
			else
				disk_struct[drive].status |= 0x20; /* status engaged */
			disk_struct[drive].status |= 0x01U;	/* status busy */
			if (disk_struct[drive].cylinder > 0) {
				--disk_struct[drive].cylinder;
				sound_playwav_file( gmz, SOUND_WAV_FDSEEK1 );
				disk_struct[drive].oldclk = get_internal_clock();
			}
			if (val & 0x10) /* u=1 */
				gmz->ioinfo->port_d9 = ~(disk_struct[drive].cylinder);

			/* check cylinders after seek */
			if (val & 0x04) { /* V=1 */
				t = (disk_struct[drive].cylinder) * 2 + side;
				if (make_idtbl_checkcylinder( drive, t, disk_struct[drive].cylinder ) < 0) {
					disk_struct[drive].trackchkfail = TRUE;
					disk_struct[drive].status |= 0x10U;	/* status seek error */
				} else
					disk_struct[drive].trackchkfail = FALSE;
			}
		}
	} else if ((val & 0xe0) == 0x80) {
		/* Read type-II  (1 0 0 m S E C 0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Read (mSEC0=%02x)\n", val & 0x1f);
#endif
		disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].sidechk = 0U;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			/* set disk status */
			disk_struct[drive].cmd = (val & 0x10) /* m=1 */ ?
				DISK_CMD_READMULTI : DISK_CMD_READ;
			disk_struct[drive].index = 0;
			disk_struct[drive].dadr = NULL;
			t = (disk_struct[drive].cylinder) * 2 + side;
			disk_struct[drive].m_track = t;		/* for next sector (multiple sectors) */
			disk_struct[drive].sector = sector;
			disk_struct[drive].sidechk = ((val >> 3) & 0x01U) | (val & 0x02U); /* bits1,0=C,S */
			if (search_adr( drive, t, disk_struct[drive].sector, disk_struct[drive].sidechk ) >= 0) {
				disk_struct[drive].status |= 0x01U;	/* status busy */
				disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
				if (disk_struct[drive].deleted_data)
					disk_struct[drive].status |= 0x20U; /* status deleted data */
			} else {
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
				disk_struct[drive].status |= 0x10U;	/* status record not found */
			}
		}
	} else if ((val & 0xe0) == 0xa0) {
		/* Write type-II  (1 0 1 m S E C a0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Write (mSECa0=%02x)\n", val & 0x1f);
#endif
		disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].sidechk = 0U;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			/* set disk status */
			disk_struct[drive].cmd = (val & 0x10) /* m=1 */ ?
				DISK_CMD_WRITEMULTI : DISK_CMD_WRITE;
			disk_struct[drive].index = 0;
			disk_struct[drive].dadr = NULL;
			disk_struct[drive].deleted_data = (val & 1) /* a0=1 */ ?
				TRUE /* Deleted Data Mark = F8 */ : FALSE /* Data Mark = FB */;
			t = (disk_struct[drive].cylinder) * 2 + side;
			disk_struct[drive].m_track = t;		/* for next sector (multiple sectors) */
			disk_struct[drive].sector = sector;
			ddam = disk_struct[drive].deleted_data;
			disk_struct[drive].sidechk = ((val >> 3) & 0x01U) | (val & 0x02U); /* bits1,0=C,S */
			if (search_adr( drive, t, disk_struct[drive].sector, disk_struct[drive].sidechk ) >= 0) {
				disk_struct[drive].deleted_data = ddam; /* ddam by FDC commands */
				disk_struct[drive].crc_error = TRUE;
				disk_struct[drive].status |= 0x01U;	/* status busy */
				disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
			} else {
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
				disk_struct[drive].status |= 0x10U;	/* status record not found */
			}
		}
	} else if ((val & 0xf0) == 0xc0) {
		/* Read Address type-III  (1 1 0 0 0 E 0 0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE( "FDC Read Address (E=%d)\n", val & 0x04 ? 1 : 0 );
#endif
		disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].sidechk = 0U;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			/* set disk status */
			disk_struct[drive].cmd = DISK_CMD_READADDRESS;
			disk_struct[drive].sidechk = 0U;
			disk_struct[drive].index = 0;
			disk_struct[drive].dadr = disk_trkbuff;
			t = (disk_struct[drive].cylinder) * 2 + side;
			disk_struct[drive].m_track = t;		/* for read id debugging messages */
			make_trackimage( drive, t );
			if (1 /* disk_idnum > 0 */) {
				disk_struct[drive].status |= 0x01U;	/* status busy */
				disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
			} else {
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
				disk_struct[drive].status |= 0x10U;	/* status record not found */
			}
		}
	} else if ((val & 0xf0) == 0xe0) {
		/* Read Track type-III   (1 1 1 0 0 E 0 0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE( "FDC Read Track (E=%d)\n", val & 0x04 ? 1 : 0 );
#endif
		disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].sidechk = 0U;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			/* set disk status */
			disk_struct[drive].cmd = DISK_CMD_READTRACK;
			disk_struct[drive].index = 0;
			disk_struct[drive].dadr = disk_trkbuff;
			t = (disk_struct[drive].cylinder) * 2 + side;
			make_trackimage( drive, t );
			disk_struct[drive].status |= 0x01U;	/* status busy */
			disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
		}
	} else if ((val & 0xf0) == 0xf0) {
		/* Write Track type-III  (1 1 1 1 0 E 0 0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE( "FDC Write Track (E=%d)\n", val & 0x04 ? 1 : 0 );
#endif
		disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
		disk_struct[drive].status &= ~0x3fU;		/* status clear */
		disk_struct[drive].sidechk = 0U;
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* A drive was selected */
			/* set disk status */
			disk_struct[drive].cmd = DISK_CMD_WRITETRACK;
			disk_struct[drive].index = 0;
			disk_struct[drive].dadr = disk_trkbuff;
			t = (disk_struct[drive].cylinder) * 2 + side;
			disk_struct[drive].m_track = t;		/* for write track post processings */
			memset( disk_trkbuff, 0x4e, DISK_TRACKBUFF_SIZE );
			disk_trkbytes = DISK_TRACK_BYTES;
			disk_struct[drive].status |= 0x01U;	/* status busy */
			disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
		}
	} else if ((val & 0xf0) == 0xd0) {
		/* Force Interrupt type-IV  (1 1 0 1 I3 I2 I1 I0) */
		/*     The interrupts are not used by MZ-80B/2000/2200 (I3-I0) */
#ifdef ENBALE_TRACE_DISKCMD
		_TRACE("FDC Force Interrupt (I3-I0=%02x)\n", val & 0x0f);
#endif
		switch (disk_struct[drive].cmd) {
		case DISK_CMD_NONE_I :
		case DISK_CMD_SEEK :
			disk_struct[drive].cmd = DISK_CMD_NONE_I;
			disk_struct[drive].status &= ~0x3fU;	/* status clear */
			if (!disk_struct[drive].cylinder)
				disk_struct[drive].status |= 0x24; /* status engaged and track00 */
			else
				disk_struct[drive].status |= 0x20; /* status engaged */
			break;
		case DISK_CMD_NONE_II_III_WRITE :
		case DISK_CMD_WRITE :
		case DISK_CMD_WRITEMULTI :
		case DISK_CMD_WRITETRACK :
			disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
			disk_struct[drive].status &= ~0x3fU;	/* status clear */
			disk_struct[drive].sidechk = 0U;
			break;
		default :
			disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
			disk_struct[drive].status &= ~0x3fU;	/* status clear */
			disk_struct[drive].sidechk = 0U;
			break;
		}
	} else {
		_TRACE("??? ... Unsupported FDC commands CMD:%02x\n", val);
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status &= ~0x3fU;	/* status clear */
		disk_struct[drive].sidechk = 0U;
	}
	return 0;
}

u8 disk_port_in_d9( u16 port )
{
	/* Tracks (in) */
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "ind9(track): ~%02x\n", ~(gmz->ioinfo->port_d9) & 0xff );
#endif
	return gmz->ioinfo->port_d9;
}

int disk_port_out_d9( u16 port, u8 value )
{
	/* Tracks (out) */
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "outd9(track): ~%02x\n", ~value & 0xff );
#endif
	gmz->ioinfo->port_d9 = value;
	return 0;
}

u8 disk_port_in_da( u16 port )
{
	/* Sectors (in) */
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "inda(sector): ~%02x\n", ~(gmz->ioinfo->port_da) & 0xff );
#endif
	return gmz->ioinfo->port_da;
}

int disk_port_out_da( u16 port, u8 value )
{
	/* Sectors (out) */
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "outda(sector): ~%02x\n", ~value & 0xff );
#endif
	gmz->ioinfo->port_da = value;
	return 0;
}

u8 disk_port_in_db( u16 port )
{
	int drive = gmz->ioinfo->port_dc & 0x03;
	u8 val = ~gmz->ioinfo->port_db;

	/* Data Regs */
	if ((gmz->ioinfo->port_dc & 0x84) != 0x84) {	/* All drives were deselected */
#ifdef ENBALE_TRACE_DISKIO
		_TRACE( "indb(datareg): ~%02x (disabled)\n", ~(gmz->ioinfo->port_db) & 0xff );
#endif
		return gmz->ioinfo->port_db;
	}
	/* A drive was selected */
	if ((disk_struct[drive].cmd == DISK_CMD_READ || disk_struct[drive].cmd == DISK_CMD_READMULTI)
			&& (!disk_struct[drive].substatus)) {
		if (!disk_struct[drive].dadr) {
			disk_struct[drive].status &= ~0x3fU;		/* status clear */
			disk_struct[drive].status |= 0x08U;		/* status CRC error */
			disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
			val = 0;
			gmz->ioinfo->port_db = ~val;
#ifdef ENBALE_TRACE_DISKIO
			_TRACE( "indb(datareg): %02x (no data)\n", ~(gmz->ioinfo->port_db) & 0xff );
#endif
			return gmz->ioinfo->port_db;
		}
		/* read or multisector read */
		if (disk_struct[drive].index < disk_struct[drive].have_sector_data) {
			val = disk_struct[drive].dadr[disk_struct[drive].index];
			disk_struct[drive].index++;
		}
		if (disk_struct[drive].index >= disk_struct[drive].have_sector_data) {
			int eod = FALSE;

			if (disk_struct[drive].deleted_data)
				disk_struct[drive].status |= 0x20U;	/* status deleted data */
			if (disk_struct[drive].crc_error) {
				if (!disk_mode_read_ignorecrc) {
					disk_struct[drive].status |= 0x08U;	/* status CRC error */
					eod = TRUE;
				}
			}
			disk_struct[drive].status &= ~0x03;		/* status clear data request and busy */
			if ((disk_struct[drive].cmd == DISK_CMD_READ) || eod) {
				/* single sector or error */
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
			} else {
				/* multiple sectors, read next record */
				disk_struct[drive].sector++;
				gmz->ioinfo->port_da = ~disk_struct[drive].sector;
				disk_struct[drive].index = 0;
				disk_struct[drive].dadr = NULL;
				disk_struct[drive].status &= ~0x3fU;	/* status clear */
				if (search_adr( drive, disk_struct[drive].m_track, disk_struct[drive].sector,
						disk_struct[drive].sidechk ) >= 0) {
					disk_struct[drive].status |= 0x01U; /* status busy */
					if (disk_struct[drive].deleted_data)
						disk_struct[drive].status |= 0x20U; /* status deleted data */
					disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
				} else {
					disk_struct[drive].status &= ~0x03; /* status clear data request and busy */
					disk_struct[drive].status |= 0x10U; /* status record not found */
					disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
				}
			}
		}
	} else if (disk_struct[drive].cmd == DISK_CMD_READADDRESS && !disk_struct[drive].substatus) {
		/* read address */
		if (disk_struct[drive].index >= 6 * disk_idnum)
			disk_struct[drive].index = 0;
		if (disk_struct[drive].index < 6 * disk_idnum) {
			int i = disk_struct[drive].index;

			val = disk_idlist[i / 6][i % 6];
			if (!(i % 6)) {
				/* sector reg <- cylinder */
				/* It's used by DISK CONTROL PROGRAM I/O'83/6 and TAKE-DOS Oh!MZ'85/5 */
				gmz->ioinfo->port_da = ~disk_idlist[i/6][0];
			}
			disk_struct[drive].index++;
		}
#ifdef ENBALE_TRACE_READID
		//When it's enable, many messages are occurred and very slow...
		_TRACE( "FDC Read ID: %02x Trk(Cyl+Head):%02d\n", val, disk_struct[drive].m_track );
#endif
	} else if ((disk_struct[drive].cmd == DISK_CMD_READTRACK) && (!disk_struct[drive].substatus)) {
		if (disk_struct[drive].index < disk_trkbytes) {
			val = disk_struct[drive].dadr[disk_struct[drive].index];
			disk_struct[drive].index++;
		}
		if (disk_struct[drive].index >= disk_trkbytes) {
			disk_struct[drive].status &= ~0x03;		/* status clear data request and busy */
			disk_struct[drive].cmd = DISK_CMD_NONE_II_III;
		}
	}

	gmz->ioinfo->port_db = ~val;
#ifdef ENBALE_TRACE_DISKIO
	//When it's enable, many messages are occurred and very slow...
	//if (disk_struct[drive].cmd == DISK_CMD_READADDRESS)
	if (disk_struct[drive].index < 10 || disk_struct[drive].index > 252)
	_TRACE( "indb(datareg): ~%02x (index=%d)\n", ~(gmz->ioinfo->port_db) & 0xff, disk_struct[drive].index );
#endif
	return gmz->ioinfo->port_db;
}

int disk_port_out_db( u16 port, u8 value )
{
	u8 val;
	int ddam, drive = gmz->ioinfo->port_dc & 0x03;

	/* Data Reg */
#ifdef ENBALE_TRACE_DISKIO
	//When it's enable, many messages are occurred and very slow...
	//_TRACE( "outdb(datareg): ~%02x\n", ~value & 0xff );
#endif
	gmz->ioinfo->port_db = value;
	val = ~value;

	if ((gmz->ioinfo->port_dc & 0x84) != 0x84)	/* All drives were deselected */
		return 0;
	/* A drive was selected */
	if ((disk_struct[drive].cmd == DISK_CMD_WRITE || disk_struct[drive].cmd == DISK_CMD_WRITEMULTI)
			&& (!disk_struct[drive].substatus)) {
		if (!disk_struct[drive].dadr) {
			disk_struct[drive].status &= ~0x3fU;	/* status clear */
			disk_struct[drive].status |= 0x10U;	/* status record not found */
			disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
			return 0;
		}
		/* write or multisector write */
		if (disk_struct[drive].index < disk_struct[drive].have_sector_data) {
			if (!(check_writeprotect(drive))) {
				disk_struct[drive].dadr[disk_struct[drive].index] = val;
				disk_struct[drive].modified = TRUE;
			}
			disk_struct[drive].index++;
		}
		if (disk_struct[drive].index >= disk_struct[drive].have_sector_data) {
			disk_struct[drive].crc_error = FALSE;
			update_fdimg_status( drive );
			disk_struct[drive].status &= ~0x03;	/* status clear data request and busy */
			if (disk_struct[drive].cmd == DISK_CMD_WRITE) {
				/* single sector */
				disk_struct[drive].status &= ~0x3fU;	/* status clear */
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
			} else {
				/* multiple sectors, make ready for next sector */
				disk_struct[drive].sector++;
				gmz->ioinfo->port_da = ~disk_struct[drive].sector;
				disk_struct[drive].index = 0;
				disk_struct[drive].dadr = NULL;
				disk_struct[drive].status &= ~0x3fU;	/* status clear */
				ddam = disk_struct[drive].deleted_data;
				if (search_adr( drive, disk_struct[drive].m_track, disk_struct[drive].sector,
						disk_struct[drive].sidechk ) >= 0) {
					disk_struct[drive].deleted_data = ddam; /* ddam by FDC command  */ 
					disk_struct[drive].crc_error = TRUE;
					disk_struct[drive].status |= 0x01U;	/* status busy */
					disk_struct[drive].substatus = REQUESTDATA_WAIT_CT;
				} else {
					disk_struct[drive].status |= 0x10U;	/* status record not found */
					disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
				}
			}
		}
	} else if ((disk_struct[drive].cmd == DISK_CMD_WRITETRACK) && (!disk_struct[drive].substatus)) {
		if (disk_struct[drive].index < disk_trkbytes) {
			if (!(check_writeprotect(drive)))
				disk_struct[drive].dadr[disk_struct[drive].index] = val;
			disk_struct[drive].index++;
		}
		if (disk_struct[drive].index >= disk_trkbytes) {
			analyze_trackimage( drive, disk_struct[drive].m_track );
			if (!(check_writeprotect(drive)) && change_format_1trk( drive, disk_struct[drive].m_track ) >= 0) {
				/* success */
				disk_struct[drive].modified = TRUE;
				disk_struct[drive].status &= ~0x3fU;	/* status clear */
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
			} else {
				disk_struct[drive].status &= ~0x3fU;	/* status clear */
				disk_struct[drive].status |= 0x20U;	/* status write fault */
				disk_struct[drive].cmd = DISK_CMD_NONE_II_III_WRITE;
			}
		}
	}
	return 0;
}

u8 disk_port_in_dc( u16 port )
{
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "indc: unused I/O val=ff <error>\n" );
#endif
	return 0xff;
}

/*
	I/O port 0xDC (not FDC)

	bit 0-1	drive number
	bit 2	drive selects
	bit 7	enable the motor
*/
int disk_port_out_dc( u16 port, u8 value )
{
	int drive;

#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "outdc(ds/motor): %02x\n", value );
#endif
	if (value == gmz->ioinfo->port_dc)
		return 0;

	drive = value & 0x03;
	gmz->ioinfo->port_dc = value;
	if (value & 0x80) {	/* motor on */
		if (!disk_enablemotor) {
			disk_enablemotor = TRUE;
			if (disk_motorcount <= 0)
				sound_playwav_file( gmz, SOUND_WAV_FDMOTOR_ST );
		}
		disk_motorcount = DISK_MOTOR_STOPTIME;
	} else {
		/* motor off */
		disk_enablemotor = FALSE;
	}
	if ((gmz->ioinfo->port_dc & 0x84) == 0x84) {	/* enable motors and A drive was selected */
		/* initialize disk state and status */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status = 0xc0;			/* status write protect and not ready */
		if (disk_struct[drive].attr) {
			disk_struct[drive].status &= ~0x80U;		/* status disk ready */
			if (!(check_writeprotect(drive)))
				disk_struct[drive].status &= ~0x40U;	/* status not write protect */
		}
	} else if ((gmz->ioinfo->port_dc & 0x84) == 0x80) { /* enable motors and A drive was deselected */
		;
	} else { /* motor off */
		disk_struct[drive].cmd = DISK_CMD_NONE_I;
		disk_struct[drive].status = 0x80;			/* status disk not ready */
	}
	return 0;
}

u8 disk_port_in_dd( u16 port )
{
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "indd: unused I/O val=ff <error>\n" );
#endif
	return 0xff;
}

/*
	I/O port 0xDD (not FDC)

	bit 0 ... side
*/
int disk_port_out_dd( u16 port, u8 value )
{
	/* side */
	gmz->ioinfo->port_dd = value;
#ifdef ENBALE_TRACE_DISKIO
	_TRACE( "outdd(side): %02x\n", value );
#endif
	return 0;
}

const char *disk_getdefpath( MZ2000 *mz, int drive )
{
	int i;

	if (strlen( (char *)disk_struct[drive].filename ) > 0)
		return (char *)disk_struct[drive].filename;
	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		if (strlen( (char *)disk_struct[i].filename ) > 0)
			return (char *)disk_struct[i].filename;
	}
	return "";
}

int disk_getexist( MZ2000 *mz, int drive )
{
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return DISK_STATUS_NONE;
	if (disk_struct[drive].modified)
		return DISK_STATUS_WCHANGED;
	return DISK_STATUS_INSERT;
}

int disk_getwritable( MZ2000 *mz, int drive )
{
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return FALSE;
	if (check_writeprotect( drive ))
		return FALSE;
	return TRUE;
}

int disk_getaccess( MZ2000 *mz, int drive )
{
	if (!gmz)
		return FALSE;
	if ((gmz->ioinfo->port_dc & 0x03) == drive) {
		if ((gmz->ioinfo->port_dc & 0x84) == 0x84)
			return TRUE;
	}
	return FALSE;
}

int disk_getmodified( MZ2000 *mz, int drive )
{
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return FALSE;
	if (!disk_struct[drive].modified)
		return FALSE;
	return TRUE;
}

char *disk_getfnptr( MZ2000 *mz, int drive )
{
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return "";
	return (char *)disk_struct[drive].filename;
}

int disk_writeback( MZ2000 *mz, int drive )
{
	char buf[LNMAX];
#ifndef WIN32
	FILE *fp;
	long size;
#else
	DWORD size;
	HANDLE file_handle;
#endif

	/* modified check */
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return FALSE;
	if (!disk_struct[drive].modified)
		return FALSE;

	/* write the image file */
#ifndef WIN32
	fp = fopen( (char *)disk_struct[drive].filename, "w" );
	if (!fp) {
		snprintf( buf, LNMAX, "Drive %d image file \"%s\" write open error ", drive, disk_struct[drive].filename );
		uigtk_messagebox( buf, "DISK UNIT", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		return FALSE;
	}
	size = fwrite( (char *)disk_struct[drive].disk_buffer, 1, disk_struct[drive].disksize, fp );
	fclose( fp );
	if (size != disk_struct[drive].disksize) {
		snprintf( buf, LNMAX, "Drive %d image file \"%s\" write data error ", drive, disk_struct[drive].filename );
		uigtk_messagebox( buf, "DISK UNIT", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		return FALSE;
	}
#else
	/* write image */
	file_handle = CreateFile( (char *)disk_struct[drive].filename,
		GENERIC_WRITE, FILE_SHARE_WRITE, NULL, CREATE_ALWAYS,
		FILE_ATTRIBUTE_NORMAL, NULL );
	if (file_handle == INVALID_HANDLE_VALUE) {
		sprintf( buf, "Drive %d image file \"%s\" write open error ", drive, disk_struct[drive].filename );
		MessageBox( mz->ghWnd, buf, "DISK UNIT", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return FALSE;
	}
	WriteFile( file_handle, (unsigned char *)disk_struct[drive].disk_buffer, disk_struct[drive].disksize,
		&size, NULL );
	CloseHandle( file_handle ) ;
	if (size != disk_struct[drive].disksize) {
		sprintf( buf, "Drive %d image file \"%s\" write data error ", drive, disk_struct[drive].filename );
		MessageBox( mz->ghWnd, buf, "DISK UNIT", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return FALSE;
	}
#endif
	disk_struct[drive].modified = FALSE;
	return TRUE;
}

int disk_eject( MZ2000 *mz, int drive, DISK_EJECT_TYPE mode )
{
	int r;

	/* modified check */
	if (disk_struct[drive].attr == DISK_DATAFMT_NONE)
		return TRUE;
	if (disk_getmodified( mz, drive ) && (mode != DISK_EJECT_FORCE)) {
		char buf[LNMAX];

#ifndef WIN32
		snprintf( buf, LNMAX, "Drive %d image was modified. Writeback to \"%s\" ?",
			drive + 1, disk_struct[drive].filename );
		r = uigtk_messagebox( buf, "DISK UNIT", UIGTK_BUTTON_YESNOCANC | UIGTK_ICON_QUESTION );
		if (r < 0)
			return FALSE;
		if (r == UIGTK_IDYES)
			disk_writeback( mz, drive );
#else
		_snprintf( buf, LNMAX, "Drive %d image was modified. Writeback to \"%s\" ?",
			drive + 1, disk_struct[drive].filename );
		buf[LNMAX - 1] = '\0';	/* VC6 is not C99 */
		r = MessageBox( mz->ghWnd, buf, "DISK UNIT", MB_YESNOCANCEL | MB_DEFBUTTON3 | MB_ICONQUESTION | MB_APPLMODAL );
		if (r == IDCANCEL)
			return FALSE;
		if (r == IDYES)
			disk_writeback( mz, drive );
#endif
	}
	if (mode == DISK_EJECT_CHECKONLY)
		return TRUE;
	disk_struct[drive].modified = FALSE;
	disk_struct[drive].attr = DISK_DATAFMT_NONE;
	disk_struct[drive].writable = FALSE;
	disk_struct[drive].dadr = NULL;
	sound_playwav_file( gmz, SOUND_WAV_FDEJECT );
	return TRUE;
}

/*
	d88flag ... no raw file = 1. i.e. D88/DSK = TRUE, 2D = FALSE
*/
static int disk_read( MZ2000 *mz, int drive, const char *filename, int d88flag, int nomsg, int readonly )
{
	char *cp;
#ifdef WIN32
	DWORD size, attr;
	HANDLE file_handle;

	/* read the image file */
	attr = GetFileAttributes( filename );
	file_handle = CreateFile( filename,
		GENERIC_READ, FILE_SHARE_READ, NULL,OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, NULL );
	if (file_handle == INVALID_HANDLE_VALUE) {
		if (!nomsg) {
			MessageBox( mz->ghWnd, "Disk image file read open error",
				"DISK UNIT", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		} else {
			/* start up argument */
			strcpy( (char*)disk_struct[drive].filename, filename );
		}
		return FALSE;
	}
	size = GetFileSize( file_handle, NULL );
	memset( (unsigned char *)disk_struct[drive].disk_buffer, 0, DISK_IMAGEBUFF_SIZE );
	if (size + sctlen[3] + 4096 > DISK_IMAGEREADABLE_SIZE) {	/* add data for truncated disks */
		CloseHandle( file_handle );
		if (!nomsg) {
			MessageBox( mz->ghWnd, "The disk image file is too big.",
				"DISK UNIT", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		} else {
			/* start up argument */
			strcpy( (char *)disk_struct[drive].filename, filename );
		}
		return FALSE;
	}
	ReadFile( file_handle, (char *)disk_struct[drive].disk_buffer, size,
		&size, NULL );
	CloseHandle( file_handle );
	if (!(attr & FILE_ATTRIBUTE_READONLY))
		disk_struct[drive].writable = TRUE;
	else
		disk_struct[drive].writable = FALSE;
#else /* UNIX */
	int r;
	FILE *fp;
	long size;
	struct stat stbuf;

	/* read the image file */
	r = access( filename, R_OK | W_OK );
	if (!r) {
		r = stat( filename, &stbuf );
		if (!r) {
			if ((stbuf.st_mode & S_IRUSR) && (stbuf.st_mode & S_IWUSR))
				r = 0;
			else
				r = -1;
		}
	}
	fp = fopen( filename, "r" );
	if (!fp) {
		if (!nomsg) {
			uigtk_messagebox( "Disk image file read open error",
				"DISK UNIT", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR );
		} else {
			/* start up argument */
			strcpy( (char *)disk_struct[drive].filename, filename );
		}
		return FALSE;
	}
	fseek( fp, 0L, SEEK_END );
	size = ftell( fp );
	rewind( fp );
	memset( (unsigned char *)disk_struct[drive].disk_buffer, 0, DISK_IMAGEBUFF_SIZE );
	if (size + sctlen[3] + 4096 > DISK_IMAGEREADABLE_SIZE) {	/* add data for truncated disks */
		fclose( fp );
		if (!nomsg) {
			uigtk_messagebox( "The disk image file is too big.",
				"DISK UNIT", UIGTK_BUTTON_OKONLY | UIGTK_ICON_ERROR);
		} else {
			/* start up argument */
			strcpy( (char *)disk_struct[drive].filename, filename );
		}
		return FALSE;
	}
	size = fread( (unsigned char *)disk_struct[drive].disk_buffer, 1, size, fp );
	fclose( fp );
	if (!r)
		disk_struct[drive].writable = TRUE;
	else
		disk_struct[drive].writable = FALSE;
#endif /* WIN32 */
	/* force readonly */
	if (readonly)
		disk_struct[drive].writable = FALSE;

	/* set attribute */
	if (d88flag) {	/* D88 or DSK */
		cp = (char *)disk_struct[drive].disk_buffer;
		if (!memcmp( cp, "MV - CPCEMU Disk-File\r\nDisk-Info\r\n", DSK_DISKEYECACHE_LEN )) {
			/* DSK */
			disk_struct[drive].attr = DISK_DATAFMT_DSK;
		}
		if (!memcmp( cp, "EXTENDED CPC DSK File\r\nDisk-Info\r\n", DSK_DISKEYECACHE_LEN )) {
			/* DSKEX */
			disk_struct[drive].attr = DISK_DATAFMT_DSKEX;
		} else {
			/* D88 */
			disk_struct[drive].attr = DISK_DATAFMT_D88;
		}
	} else {	/* 2D */
		/* 256bytes length 16sectors raw image */
		disk_struct[drive].attr = DISK_DATAFMT_2D;
	}
	disk_struct[drive].disksize = size;
	disk_struct[drive].modified = FALSE;
	memset( (char *)disk_struct[drive].diskname, 0, D88_DISKNAME_MAXLEN + 1 );
	switch (disk_struct[drive].attr) {
	case DISK_DATAFMT_D88 :
		disk_struct[drive].maxtrack = disk_struct[drive].disk_buffer[D88_OFFSET_MEDIATYPE] ?
				DISK_TRACK_2DD_MAXNUM :
				DISK_TRACK_2D_MAXNUM;
		strncpy( (char *)disk_struct[drive].diskname, (char *)disk_struct[drive].disk_buffer, D88_DISKNAME_MAXLEN );
		disk_struct[drive].diskname[D88_DISKNAME_MAXLEN] = '\0';
	case DISK_DATAFMT_DSK :
	case DISK_DATAFMT_DSKEX :
		disk_struct[drive].maxtrack = DISK_TRACK_2DD_MAXNUM;
		strncpy( (char *)disk_struct[drive].diskname, (char *)&disk_struct[drive].disk_buffer[DSK_DISKNAME_OFFS],
			DSK_DISKNAME_MAXLEN );
		disk_struct[drive].diskname[DSK_DISKNAME_MAXLEN] = '\0';
		break;
	case DISK_DATAFMT_ERRORS :
		disk_struct[drive].maxtrack = DISK_TRACK_2DD_MAXNUM;
		break;
	default : /* .2D extention sectors=16/length=256bytes raw format */
		disk_struct[drive].maxtrack = DISK_TRACK_2DD_MAXNUM;
		break;
	}
	strcpy( (char *)disk_struct[drive].filename, filename );
	return TRUE;
}

/*
	d88flag ... no raw file = 1. i.e. D88/DSK = TRUE, 2D = FALSE
*/
int disk_change( MZ2000 *mz, int drive, const char *filename, int d88flag, int nomsg, int readonly )
{
	int r;

	if (disk_getmodified( mz, drive ) && !nomsg) {
		char buf[LNMAX];

#ifndef WIN32
		snprintf( buf, LNMAX, "Drive %d image was modified. Writeback to \"%s\" ?",
			drive + 1, disk_struct[drive].filename );
		r = uigtk_messagebox( buf, "DISK UNIT", UIGTK_BUTTON_YESNOCANC | UIGTK_ICON_QUESTION);
		if (r < 0)
			return FALSE;
		if (r == UIGTK_IDYES)
			disk_writeback( mz, drive );
#else /* WIN32 */
		sprintf( buf, "Drive %d image was modified. Writeback to \"%s\" ?",
			drive + 1, disk_struct[drive].filename );
		r = MessageBox( mz->ghWnd, buf, "DISK UNIT", MB_YESNOCANCEL | MB_DEFBUTTON3 | MB_ICONQUESTION | MB_APPLMODAL );
		if (r == IDCANCEL)
			return FALSE;
		if (r == IDYES)
			disk_writeback( mz, drive );
#endif
	}

	/* image file read */
	if (!disk_read( mz, drive, filename, d88flag, nomsg, readonly ))
		return FALSE;
	return TRUE;
}

/*
	IPL executer for internal IPL program

	Return:	Result code
		0  ... No errors
		-1 ... Boot sectors are not found
		-2 ... Not master (Information block error)
		-3 ... Loaging error
*/
static int exec_ipl( MZ2000 *mz )
{
	int i, j, n, flag = TRUE;
	int trkmz, trk, sct;
	int size, rec, drive;
	unsigned char *p;
	volatile unsigned char *adr;

	/* read IPL sectors to $4f00 */
	drive = current_drive;
	/* setting I/O and read disk */
	trkmz = 0;				/* this value has MZ side(head) */
	trk = (trkmz & 0xfe) | ((~trkmz) & 1);	/* reverse surface */
	sct = 1;
	disk_struct[drive].cylinder = trk / 2;
	disk_struct[drive].sector   = sct;
	gmz->ioinfo->port_d9 = ~(disk_struct[drive].cylinder);
	gmz->ioinfo->port_da = ~disk_struct[drive].sector;
	gmz->ioinfo->port_dd = trk % 2; /* head */
	trkmz = 0;				/* this value has MZ side(head) */
	trk = (trkmz & 0xfe) | ((~trkmz) & 1);	/* reverse surface */
	if (search_adr( drive, trk, sct, 0U ) != 1) {
		/* records not found */ 
		return -1;
	}
	if (disk_struct[drive].crc_error && !disk_mode_read_ignorecrc)
		return -1;
	if (disk_struct[drive].have_sector_data != 256)
		return -1;
	if (disk_struct[drive].deleted_data)
		return -1;

	adr = disk_struct[drive].dadr;
	for (i = 0; i < sctlen[1]; i++)
		mz->memory[0x4f00+i] = ~adr[i];

	/* check information block */
	p = &(mz->memory[0x4f00]);
	if (strncmp( (char *)p, "\x001IPLPRO", 7 ))
		flag = FALSE;
	if (!flag)
		return -2;
	strncpy( disk_iplmsg, (char *)&(mz->memory[0x4f07]), DISK_IPLMSG_MAXNUM );
	disk_iplmsg[DISK_IPLMSG_MAXNUM] = '\0';

	/* clear stack information ($7fe0-$7fff) */
	memset( &(mz->memory[0x7fe0]), 0, 0x20 );
	/* write drive no */
	mz->memory[0x7fec] = current_drive;

	/* exec IPL */
	size = p[0x14] | (p[0x15] << 8);
	if (!size)
		size = 0x10000;
	rec  = p[0x1e] | (p[0x1f] << 8);
	for (i = 0; i < size; ) {
		/* setting I/O and read disk */
		trkmz = rec >> 4;			/* this value has MZ side(head) */
		trk = (trkmz & 0xfe) | ((~trkmz) & 1);	/* reverse surface */
		sct = (rec & 15) + 1;
		disk_struct[drive].cylinder = trk / 2;
		disk_struct[drive].sector   = sct;
		gmz->ioinfo->port_d9 = ~(disk_struct[drive].cylinder);
		gmz->ioinfo->port_da = ~disk_struct[drive].sector;
		gmz->ioinfo->port_dd = trk % 2;	/* head */
		n = search_adr( drive, trk, sct, 0 );
		if (n < 0 || n > 3) {
			/* record not founds */
			return -3;
		}
		if (disk_struct[drive].crc_error && !disk_mode_read_ignorecrc)
			return -3;
		if (disk_struct[drive].have_sector_data < 128)
			return -3;
		if (disk_struct[drive].deleted_data)
			return -3;
		adr = disk_struct[current_drive].dadr;
		for (j = 0; j < sctlen[n & 0x03]; j++)
			mz->memory[i+j] = ~adr[j];
		i += sctlen[n & 0x03];
		rec++;
	}
	return TRUE;
}

/*
	d88flag ... no raw file = 1. i.e. D88/DSK = TRUE, 2D = FALSE
*/
int disk_exec( MZ2000 *mz, int drive, const char *filename, int d88flag, int readonly )
{
	int r;

	/* read a image file */
	if (filename && d88flag >= 0) {
#ifndef WIN32
		if (!disk_change( mz, drive, filename, d88flag, TRUE, readonly ))
#else
		if (!disk_change( mz, drive, filename, d88flag, FALSE, readonly ))
#endif
			return FALSE;
	}

	/* execute a IPL program (to MZ memory) */
	current_drive = drive;
	r = exec_ipl( mz );
	if (r <= 0)
		return r;
	return TRUE;
}

/*
	Disk module running mode settings

	parameters:
		the value is boolean (0 or 1),
			== -1 ... no changes and read it
			<= -2 ... no changes and no read it

		exec_fast	... fast running mode (not implemented, fast mode only)
		ignore_crc	... ignore read CRC errors
		erase_sct17crc	... erase sector 17 CRC to 16 sectors

	return:	last == -1 parameter value (boolean)
		-1 ... no parameter -1
*/
int disk_setparam( int exec_fast, int ignore_crc, int erase_sct17crc )
{
	int r = -1;

	if (exec_fast >= 0)
		disk_mode_execfast = exec_fast ? TRUE : FALSE;
	else if (exec_fast == -1)
		r = disk_mode_execfast;
	if (ignore_crc >= 0)
		disk_mode_read_ignorecrc = ignore_crc ? TRUE : FALSE;
	else if (exec_fast == -1)
		r = disk_mode_read_ignorecrc;
	if (erase_sct17crc >= 0)
		disk_mode_format_erasesct17crc = erase_sct17crc ? TRUE : FALSE;
	else if (exec_fast == -1)
		r = disk_mode_format_erasesct17crc;
	return r;
}

void disk_reset( MZ2000 *mz )
{
	int i;

	current_drive = -1;
	disk_enablemotor = FALSE;
	disk_motorcount  = 0;
	disk_step_direction = TRUE;
	gmz -> ioinfo -> port_d8 = 0xff;
	gmz -> ioinfo -> port_d9 = 0xff;
	gmz -> ioinfo -> port_da = 0xff;
	gmz -> ioinfo -> port_db = 0xff;
	gmz -> ioinfo -> port_dc = 0;
	gmz -> ioinfo -> port_dd = 0;
	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		disk_struct[i].cmd = DISK_CMD_NONE_I;
		disk_struct[i].status = 0x00;	/* status disk not ready */
		disk_struct[i].cylinder = 0;
		disk_struct[i].m_track = 0;
		disk_struct[i].sector = 1;
		disk_struct[i].index = 0;
		disk_struct[i].fm_mode = FALSE;
		disk_struct[i].deleted_data = FALSE;
		disk_struct[i].crc_error = FALSE;
	}
	disk_trkbytes = 0;
	memset( disk_trkbuff, 0, DISK_TRACKBUFF_SIZE );
	return;
}

void disk_exit( MZ2000 *mz )
{
	int i;

	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		disk_struct[i].attr = DISK_DATAFMT_NONE;
		disk_struct[i].modified = FALSE;
		disk_struct[i].writable = FALSE;
		disk_struct[i].disksize = 0;
		if (disk_struct[i].disk_buffer) {
			FREE( (void *)disk_struct[i].disk_buffer );
			disk_struct[i].disk_buffer = NULL;
		}
	}
	if (disk_trkbuff) {
		FREE( disk_trkbuff );
		disk_trkbuff = NULL;
		disk_trkbytes = 0;
	}
	if (disk_tempbuff) {
		FREE( disk_tempbuff );
		disk_tempbuff = NULL;
	}
	if (mz && gmz && mz2000_global) {
		int i;
		const char *section = INISECTIONNAME_BASIC;

		for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
			char tbuf[256];

#ifdef WIN32
			_snprintf( tbuf, sizeof(tbuf), "drive%d_filename", i + 1 );
			tbuf[256-1] = '\0';
#else
			snprintf( tbuf, sizeof(tbuf), "drive%d_filename", i + 1 );
#endif
			WriteUserProfileString( section, tbuf, (char *)disk_struct[i].filename, app_defaultinifile );
		}
	}
	return;
}

int disk_init( MZ2000 *mz )
{
	int i;
	const char *section = INISECTIONNAME_BASIC;

	gmz = mz;
	make_crctable();
	disk_wait_clocks[0] = DISK_WAIT_CLOCK1;	/* 6ms */
	disk_wait_clocks[1] = DISK_WAIT_CLOCK2;	/* 12ms */
	disk_wait_clocks[2] = DISK_WAIT_CLOCK3;	/* 20ms */
	disk_wait_clocks[3] = DISK_WAIT_CLOCK4;	/* 30ms */
	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		memset( (char *)&disk_struct[i], 0, sizeof(DISK_STRUCT) );
		disk_struct[i].attr = DISK_DATAFMT_NONE;
		disk_struct[i].writable = FALSE;
		disk_struct[i].modified = FALSE;
		disk_struct[i].disksize = 0;
	}
	disk_trkbuff = MALLOC( DISK_TRACKBUFF_SIZE );
	if (!disk_trkbuff) {
		fprintf(stderr, "No enough memory (disk track buffers)\n");
		return FALSE;
	}
	memset( disk_trkbuff, 0, DISK_TRACKBUFF_SIZE );
	disk_tempbuff = MALLOC( DISK_IMAGEBUFF_SIZE );
	if (!disk_tempbuff) {
		fprintf(stderr, "No enough memory (disk temporary buffers)\n");
		return FALSE;
	}
	memset( disk_tempbuff, 0, DISK_IMAGEBUFF_SIZE );
	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		disk_struct[i].disk_buffer = MALLOC( DISK_IMAGEBUFF_SIZE );
		if (!disk_struct[i].disk_buffer) {
			int j;

			fprintf(stderr, "No enough memory (disk disk_struct.diskbuffer %d/4)\n", i+1);
			for (j= 0; j < i - 1; j++) {
				FREE( (void *)disk_struct[j].disk_buffer );
				disk_struct[j].disk_buffer = NULL;
			}
			FREE( disk_trkbuff );
			disk_trkbuff = NULL;
			FREE( disk_tempbuff );
			disk_tempbuff = NULL;
			return FALSE;
		}
		memset( (unsigned char *)disk_struct[i].disk_buffer, 0, DISK_IMAGEBUFF_SIZE );
	}
	disk_trkbytes = 0;
	disk_reset( mz );
	disk_setparam( TRUE, FALSE, FALSE );
	for (i = 0; i < DISK_DRIVE_MAXNUM; i++) {
		char tbuf[256];
#ifdef WIN32
		char buff[_MAX_PATH];

		_snprintf( tbuf, sizeof(tbuf), "drive%d_filename", i + 1 );
		tbuf[256-1] = '\0';
#else /* UNIX */
		char buff[PATH_MAX];

		snprintf( tbuf, sizeof(tbuf), "drive%d_filename", i + 1 );
#endif
		if (GetUserProfileString( section, tbuf, "", buff, sizeof(buff), app_defaultinifile ) > 0) {
#ifdef WIN32
			strncpy( (char *)disk_struct[i].filename, buff, _MAX_PATH );
			buff[_MAX_PATH - 1] = '\0';
#else /* UNIX */
			strncpy( (char *)disk_struct[i].filename, buff, PATH_MAX );
			buff[PATH_MAX - 1] = '\0';
#endif
		}
	}
	return TRUE;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
