/* File: main.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Main Routines

	Combine the emz2000 mz2000.c(main.c), EmuZ-2000 emuz2000Core.c,
		and extend by FUKUI, Toshio. and TSUNEOKA, Shinji.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "config.h"

#include <windows.h>
#include <stdio.h>

#include "common.h"
#include "mz2000.h"
#include "util.h"
#include "../Plugin/plugin.h"

char const *_copyright =
	"emz2000/EmuZ-2000 TF-EDITION Ver." EMZ2000_VERSION "\n"
	"Copyright (C) " EMZ2000_COPYRIGHT " FUKUI, Toshio.\n"
	"Copyright (C) 1999-2010\n"
	"\tKASUKAWA, Takeya. (emz2000 Ver0.7)\n"
	"\tYAMAMOTO, Michiaki. (EmuZ-2000 Ver4.00pre)\n"
	"\tTSUNEOKA, Shinji. (Plugin Supports, INI, etc)\n";

char module_name[LNMAX];
char commandline_param[1024];
char app_currentpath[MAX_PATH];
char app_defaultinifile[MAX_PATH];

void setplaystop( MZ2000 *mz, int flag )
{
	if (mz -> playStop == flag)
		return;
	mz -> playStop = flag;
	sound_reset( mz );
	Sleep( 100 );
	sound_reset( mz );
	return;
}

int reset_mz2000_sub( MZ2000 *mz2000 )
{
	stop_z80();
	setspeed_z80( mz2000 -> nowZ80Clock );
	screen_initattr2( mz2000 );
	screen_setchrsize( mz2000, TRUE );
	mz2000 -> greenGraphOn = TRUE;
	mz2000 -> greenTextOn = TRUE;
	mz2000 -> mzChgGraphMode = TRUE;
	mz2000 -> viewMsg = TRUE;
	reinit_io( mz2000 );
	plugin_reinitialize_board();
	memory_normal( mz2000 );
	if (!(mz2000 -> mzmode))
		vram_on( mz2000 );
	else
		v_gram_on80B_50( mz2000 );
	keyboard_releaseall( mz2000 );
	sound_reset( mz2000 );
	return TRUE;
}

int reset_mz2000( MZ2000 *mz2000 )
{
	mz2000 -> playStop = TRUE;
	mz2000 -> pause = FALSE;
	if (mz2000 -> howipl > 0)
		return iplreset_mz( mz2000 );
	reset_mz2000_sub( mz2000 );
	reset_z80();
	Sleep( 100 );
	return TRUE;
}

int iplreset_mz( MZ2000 *mz2000 )
{
	mz2000 -> playStop = TRUE;
	mz2000 -> pause = FALSE;
	stop_z80();
	setspeed_z80( mz2000 -> nowZ80Clock );
	screen_initattr( mz2000 );
	screen_setchrsize( mz2000, TRUE );
	mz2000 -> greenGraphOn = TRUE;
	mz2000 -> greenTextOn = TRUE;
	mz2000 -> mzChgGraphMode = TRUE;
	mz2000 -> viewMsg = TRUE;
	reinit_io( mz2000 );
	plugin_reinitialize_board();
	memory_normal( mz2000 );
	keyboard_releaseall( mz2000 );
	if (mz2000 -> iplmemclr)
		memory_clear( mz2000, 1 );	/* main memory */
	mz2000 -> howipl = 1;
	reset_z80();
	sound_reset( mz2000 );
	Sleep( 100 );
	return TRUE;
}

int iplreset_mz2000( MZ2000 *mz2000 )
{
	mz2000 -> mzmode = FALSE;
	return iplreset_mz( mz2000 );
}

int iplreset_mz80b( MZ2000 *mz2000 )
{
	mz2000 -> mzmode = TRUE;
	return iplreset_mz( mz2000 );
}

static void load_iplrom( MZ2000 *mz2000 )
{
	char *filename;
	FILE *fp;

	if (!(mz2000 -> iplrom2000 = MALLOC( MZ_IPLROM_SIZE ))) {
		fprintf( stderr, "No enough memory\n" );
		return;
	}
	if (!(mz2000 -> iplrom80b = MALLOC( MZ_IPLROM_SIZE ))) {
		fprintf( stderr, "No enough memory\n" );
		FREE( mz2000 -> iplrom2000 );
		mz2000 -> iplrom2000 = NULL;
		return;
	}
	if (!(mz2000 -> iplmon = MALLOC( MZ_IPLROM_SIZE * 2 ))) {
		fprintf( stderr, "No enough memory\n" );
		FREE( mz2000 -> iplrom2000 );
		FREE( mz2000 -> iplrom80b );
		mz2000 -> iplrom2000 = NULL;
		mz2000 -> iplrom80b = NULL;
		return;
	}

	/* MZ-2000/2200 IPL ROM */
	filename = MZ20IPLROMFILE;
	fp = fopen( filename, "rb" );
	memset( mz2000 -> iplrom2000, 0, MZ_IPLROM_SIZE );
	if (fp) {
		if (fread( mz2000 -> iplrom2000, 1, MZ_IPLROM_SIZE, fp ) != MZ_IPLROM_SIZE) {
			fprintf( stderr, "IPL ROM for MZ-2000 \"%s\" read error.\n", filename );
			FREE( mz2000 -> iplrom2000 );
			mz2000 -> iplrom2000 = NULL;
		}
		fclose( fp );
	} else {
		FREE( mz2000 -> iplrom2000 );
		mz2000 -> iplrom2000 = NULL;
	}

	/* MZ-80B/80B2 IPL ROM */
	filename = MZ8BIPLROMFILE;
	fp = fopen( filename, "rb" );
	memset( mz2000 -> iplrom80b, 0, MZ_IPLROM_SIZE );
	if (fp) {
		if (fread( mz2000 -> iplrom80b, 1, MZ_IPLROM_SIZE, fp ) != MZ_IPLROM_SIZE) {
			fprintf( stderr, "IPL ROM for MZ-80B \"%s\" read error.\n", filename );
			FREE( mz2000 -> iplrom80b );
			mz2000 -> iplrom80b = NULL;
		}
		fclose( fp );
	} else {
		FREE( mz2000 -> iplrom80b );
		mz2000 -> iplrom80b = NULL;
	}

	/* Monitor for Internal IPL */
	filename = MZMONIPLFILE;
	fp = fopen( filename, "rb" );
	memset( mz2000 -> iplmon, 0, MZ_IPLROM_SIZE * 2 );
	if (fp) {
		if (fread( mz2000 -> iplmon, 1, MZ_IPLROM_SIZE * 2, fp ) <= MZ_IPLROM_SIZE) {
			fprintf( stderr, "Monitor for Internal IPL \"%s\" read error.\n", filename );
			FREE( mz2000 -> iplmon );
			mz2000 -> iplmon = NULL;
		}
		fclose( fp );
		if (mz2000 -> iplmon[0] == 0x01 && mz2000 -> iplmon[1] != 0x18) {
			/* skip MZT header */
			memcpy( mz2000 -> iplmon, &(mz2000 -> iplmon[0x0080]), MZ_IPLROM_SIZE * 2 - 0x80 );
		}
		/* MZ-1Z001M or SB-1520 */
		if (mz2000 -> iplmon[0x0080] != 0xe9 && mz2000 -> iplmon[0x0080] != 0x3e) {
			fprintf( stderr, "Monitor for Internal IPL \"%s\" data error.\n", filename );
			FREE( mz2000 -> iplmon );
			mz2000 -> iplmon = NULL;
		}
	} else {
		FREE( mz2000 -> iplmon );
		mz2000 -> iplmon = NULL;
	}
	return;
}

int mainloop_mz2000( MZ2000 *mz2000 )
{
	int i, r;
	static int runmode = -1;
	static int drno = 0;
	static int ct = 0;
	static unsigned int size = 0U;

	reset_z80();
	while (!playDestroy) {
		switch (mz2000->howipl) {
		case 0 : /* no need */
			break;
		case 1 : /* internal function */
			/* very easy IPL loader */
			reset_z80();
			mz2000 -> needReset = FALSE;
			mz2000 -> needIpl = FALSE;
			memory_ipl( mz2000 );
			if (!mz2000 -> mzmode) {
				if (mz2000 -> iplrom2000) {
					memcpy( mz2000 -> ipl, mz2000 -> iplrom2000, MZ_IPLROM_SIZE );
					mz2000 -> howipl = 10;
					continue;
				}
			} else {
				if (mz2000 -> iplrom80b) {
					memcpy( mz2000 -> ipl, mz2000 -> iplrom80b, MZ_IPLROM_SIZE );
					mz2000 -> howipl = 10;
					continue;
				}
			}
			/* Internal IPL */
			screen_initattr( mz2000 );
			screen_setchrsize( mz2000, FALSE );
			screen_region_add_all( mz2000 );
			mz2000 -> ipl[0] = 0x76;
			mz2000 -> ipl[1] = 0x18;
			mz2000 -> ipl[2] = 0xfd;
			ct = 0;
			mz2000 -> howipl = 2;
			/* Initialize the CPU peripherals */
			mz2000 -> cpu_peripherals_start = CPUPERIPHERALS_INIT;
			continue;
		case 2 :
			{
				int n;
				u8 (*readf)( u16 port );

				mz2000 -> playStop = FALSE;
				get_ioent( 0xd8, &readf, NULL );
				if (readf) {
					n = keyboard_getbitdata( mz2000, 0x04 );
					if (!(n & 0x08)) {	/* C */
						mz2000 -> howipl = 3;
						continue;
					}
					if (!(n & 0x40)) {	/* F */
						mz2000 -> howipl = 92;
						continue;
					}
				}
				n = keyboard_getbitdata( mz2000, 0x04 );
				if (!(n & 0x01)) {	/* / */
					mz2000 -> howipl = 20;
					continue;
				}
				ct++;
				if (ct < 10) {
					Sleep( 100 );
					break;
				}
				if (!readf)
					mz2000 -> howipl = 3;
				else
					mz2000 -> howipl = 4;
			}
			continue;
		case 3 :
			if (!mz2tape_ready()) {
				mz2000 -> howipl = 70;
				break;
			}
			if (!mz2000 -> iplmon) {
				mz2000 -> howipl = 60;
				break;
			}
			stop_z80();
			runmode = mz2000 -> mzmode;
			mz2000 -> mzmode = (mz2000 -> iplmon[0x0080] != 0xe9);
			reset_mz2000_sub( mz2000 );
			reset_z80();
			Sleep( 100 );
			memcpy( mz2000 -> memory, mz2000 -> iplmon, MZ_IPLROM_SIZE * 2 );
			mz2000 -> memory[0x00a4] = 0x00;
			mz2000 -> memory[0x00a5] = 0x00;
			mz2000 -> memory[0x00a6] = 0x00;
			mz2000 -> memory[0x00ae] = 0x76;
			mz2000 -> memory[0x00af] = 0x18;
			mz2000 -> memory[0x00b0] = 0xfd;
			mz2000 -> memory[0x00b1] = 0x76;
			mz2000 -> memory[0x00b2] = 0x18;
			mz2000 -> memory[0x00b3] = 0xfd;
			mz2000 -> memory[0x01dc] = 0x76;
			mz2000 -> memory[0x01f5] = 0x76;
			mz2000 -> memory[0x02aa] = 0x01;
			mz2000 -> memory[0x02b9] = 0x21;
			mz2000 -> memory[0x02ba] = 0x00;
			mz2000 -> memory[0x02bb] = 0x80;
			screen_initattr( mz2000 );
			screen_setchrsize( mz2000, FALSE );
			z80.halt_state = FALSE;
			mz2000 -> howipl = 11;
			continue;
		case 4 :
			mz2000->howipl = 5;
			drno = 0;
			continue;
		case 5 :
			i = drno;
			if (!disk_getexist( mz2000, i ))
				i = -1;
			if (i >= 0 && disk_getexist( mz2000, i )) {
				stop_z80();
				memory_normal( mz2000 );
				r = disk_exec( mz2000, i, NULL, -1, FALSE );
				switch (r) {
				case -1 :	/* Boot sector not found */
					mz2000->howipl = 80;
					continue;
				case -2 :	/* Not master */
					mz2000->howipl = 81;
					continue;
				case -3 :	/* Loading error */
					mz2000->howipl = 99;
					continue;
				default :
					reset_mz2000_sub( mz2000 );
					reset_z80();
					screen_initattr( mz2000 );
					screen_setchrsize( mz2000, FALSE );
					screen_putmsg(  mz2000,  0, "IPL is loading" );
					screen_putmsgn( mz2000, 15, disk_iplmsg, DISK_IPLMSG_MAXNUM );
					screen_region_add_all( mz2000 );
					Sleep( 500 );
					z80.halt_state = FALSE;
					mz2000->howipl = 0;
					continue;
				}
			}  else {
				mz2000->howipl = 73;
				break;
			}
			break;
		case 10 : /* Z-80 IPL program */
			break;
		case 11 : /* Z-80 Tape Load Monitor */
			if (z80.halt_state) {
				//_TRACE("%04x\n", z80.pc);
				switch (z80.pc) {
				case 0x00ae+1 :
					screen_initattr( mz2000 );
					screen_setchrsize( mz2000, FALSE );
					screen_putmsg(  mz2000,  4, "IPL is looking for a program" );
					screen_region_add_all( mz2000 );
					mz2000 -> memory[0x0000] = 0xc3;
					mz2000 -> memory[0x0001] = 0xcf;
					mz2000 -> memory[0x0002] = 0x01;
					reset_z80();
					z80.halt_state = FALSE;
					continue;
				case 0x01dc+1 :
					size = (mz2000 -> memory[0x1152]) | (mz2000 -> memory[0x1153] << 8);
					if (mz2000 -> memory[0x1140] == 0x01 && size < 0x8000U) {
						screen_initattr( mz2000 );
						screen_setchrsize( mz2000, FALSE );
						screen_putmsg(  mz2000,  0, "IPL is loading" );
						screen_putmsgn( mz2000, 15, &(mz2000->memory[0x1141]), 16 );
						screen_region_add_all( mz2000 );
						mz2000 -> memory[0x0000] = 0xc3;
						mz2000 -> memory[0x0001] = 0xed;
						mz2000 -> memory[0x0002] = 0x01;
						memcpy( &mz2000 -> memory[0x4f00], &mz2000 -> memory[0x1140], 0x80 );
						reset_z80();
						z80.halt_state = FALSE;
					} else {
						/* File mode error */
						stop_z80();
						mz2tape_rew( mz2000 );
						mz2000 -> mzmode = runmode;
						reset_mz2000_sub( mz2000 );
						reset_z80();
						z80.halt_state = FALSE;
						screen_initattr( mz2000 );
						screen_setchrsize( mz2000, FALSE );
						memory_ipl( mz2000 );
						mz2000->howipl = 82;
					}
					continue;
				case 0x01f5+1 :
					stop_z80();
					mz2tape_rew( mz2000 );
					memcpy( mz2000 -> memory, &(mz2000 -> memory[0x8000]), size );
					mz2000 -> mzmode = runmode;
					reset_mz2000_sub( mz2000 );
					reset_z80();
					z80.halt_state = FALSE;
					mz2000->howipl = 0;
					continue;
				case 0x00b1+1 :
				default :
					stop_z80();
					mz2tape_rew( mz2000 );
					mz2000 -> mzmode = runmode;
					reset_mz2000_sub( mz2000 );
					reset_z80();
					memory_ipl( mz2000 );
					z80.halt_state = FALSE;
					screen_initattr( mz2000 );
					screen_setchrsize( mz2000, FALSE );
					mz2000->howipl = 99;
					continue;
				}
			}
			break;
		case 20 : /* ROM or S-RAM */
			{
				unsigned char c;
				unsigned short adr;
				int (*writef8)( u16 port, u8 value );
				u8 (*readf9)( u16 port );
				int (*writef9)( u16 port, u8 value );

				stop_z80();
				get_ioent( 0xf8, NULL,    &writef8 );
				get_ioent( 0xf9, &readf9, &writef9 );
				if (!writef8 || !writef9 || !readf9) {
					mz2000 -> howipl = 25;
					break;
				}
				adr = 0x8000U;
				writef8( 0xf8, (u8)((adr >> 8) & 0xffU) );
				writef9( 0xf9, (u8)(adr        & 0xffU) );
				c = readf9( 0xf9 );
				if (c) {
					mz2000 -> howipl = 25;
					break;
				}
				memory_normal( mz2000 );
				for (; adr; adr++) {
					writef8( 0xf8, (u8)((adr >> 8) & 0xffU) );
					writef9( 0xf9, (u8)(adr        & 0xffU) );
					c = readf9( 0xf9 );
					mz2000 -> memory[adr & 0x7fffU] = c;
				}
				reset_mz2000_sub( mz2000 );
				reset_z80();
				z80.halt_state = FALSE;
				mz2000 -> howipl = 0;
				continue;
			}
			break;
		case 25 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 4, "No S-RAM/ROM I/O F8,F9 or No Data" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 60 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 6, "No Monitor for Tape Load" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 70 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 10, "Make ready CMT" );
			screen_region_add_all( mz2000 );
			mz2tape_eject( mz2000, TRUE );
			ct = 0;
			mz2000 -> howipl = 71;
			break;
		case 71 :
			ct++;
			if (ct < 10) {
				Sleep( 100 );
				break;
			}
			mz2000 -> howipl = 72;
			continue;
		case 72 :
			{
				int n;

				if (mz2tape_ready()) {
					mz2000 -> howipl = 3;
					continue;
				}
				n = keyboard_getbitdata( mz2000, 0x03 );
				if (!(n & 0x80)) {	/* Break */
					mz2000 -> howipl = 90;
					continue;
				}
			}
			break;
		case 73 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			ct = 0;
			mz2000 -> howipl = 74;
			break;
		case 74 :
			ct++;
			if (ct < 10) {
				Sleep( 100 );
				break;
			}
			screen_putmsg( mz2000, 10, "Make ready FD" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 80 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 7, "Boot sector not found" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 81 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 7, "This disk is not master" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 82 :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 10, "File mode error" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		case 90 :
			{
				u8 (*readf)( u16 port );

				get_ioent( 0xd8, &readf, NULL );
				if (!readf) {
					screen_putmsg( mz2000, 40*2+4, "Pressing S key starts the CMT" );
					screen_putmsg( mz2000, 40*3+9, "(/:S-RAM or ROM)" );
				} else {
					screen_putmsg( mz2000, 40*2+10, "Press F or C" );
					screen_putmsg( mz2000, 40*4+11, "F:Floppy diskette" );
					screen_putmsg( mz2000, 40*5+11, "C:Cassette tape" );
					screen_putmsg( mz2000, 40*6+10, "(/:S-RAM or ROM)" );
				}
				screen_region_add_all( mz2000 );
				mz2000 -> ipl[0] = 0x76;
				mz2000 -> ipl[1] = 0x18;
				mz2000 -> ipl[2] = 0xfd;
				mz2000 -> howipl = 91;
			}
			continue;
		case 91 :
			{
				int n;
				u8 (*readf)( u16 port );

				mz2000 -> playStop = FALSE;
				get_ioent( 0xd8, &readf, NULL );
				if (!readf) {
					n = keyboard_getbitdata( mz2000, 0x06 );
					if (!(n & 0x08)) {	/* S */
						mz2000 -> howipl = 3;
						continue;
					}
				} else {
					n = keyboard_getbitdata( mz2000, 0x04 );
					if (!(n & 0x08)) {	/* C */
						mz2000 -> howipl = 3;
						continue;
					}
					if (!(n & 0x40)) {	/* F */
						mz2000 -> howipl = 92;
						continue;
					}
				}
				n = keyboard_getbitdata( mz2000, 0x04 );
				if (!(n & 0x01)) {	/* / */
					mz2000 -> howipl = 20;
					continue;
				}
			}
			break;
		case 92 :
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 10, "Drive No? (1-4)" );
			screen_region_add_all( mz2000 );
			mz2000 -> ipl[0] = 0x76;
			mz2000 -> ipl[1] = 0x18;
			mz2000 -> ipl[2] = 0xfd;
			mz2000 -> howipl = 93;
			continue;
		case 93 :
			{
				int n;

				mz2000 -> playStop = FALSE;
				n = keyboard_getbitdata( mz2000, 0x02 );
				if ((~n) & 0x1e) {
					if (!(n & 0x02))
						drno = 0;
					if (!(n & 0x04))
						drno = 1;
					if (!(n & 0x08))
						drno = 2;
					if (!(n & 0x10))
						drno = 3;
					mz2000->howipl = 5;
					continue;
				}
				n = keyboard_getbitdata( mz2000, 0x08 );
				if ((~n) & 0x1e) {
					if (!(n & 0x02))
						drno = 0;
					if (!(n & 0x04))
						drno = 1;
					if (!(n & 0x08))
						drno = 2;
					if (!(n & 0x10))
						drno = 3;
					mz2000->howipl = 5;
					continue;
				}
			}
			break;
		default :
			reset_z80();
			memory_ipl( mz2000 );
			screen_initattr( mz2000 );
			screen_putmsg( mz2000, 10, "Loading error" );
			screen_region_add_all( mz2000 );
			mz2000 -> howipl = 90;
			continue;
		}
		if (!(mz2000 -> pause) && !(mz2000 -> playStop) && !playDestroy
				&& !(mz2000 -> needReset) && !(mz2000 -> needIpl)) {
			run_z80_2( !(!mz2000 -> howipl || mz2000 -> howipl == 10) );
			if (!(!mz2000 -> howipl || mz2000 -> howipl == 10 || mz2000 -> howipl == 11))
				z80.internal_clock += z80.speed_khz;
		}
		if (mz2000 -> needReset) {
			mz2000 -> needReset = FALSE;
			reset_mz2000( mz2000 );
			setplaystop( mz2000, FALSE );
		}
		if (mz2000 -> needIpl) {
			mz2000 -> needIpl = FALSE;
			iplreset_mz( mz2000 );
			setplaystop( mz2000, FALSE );
		}
		while (mz2000 -> pause || mz2000 -> playStop) {
			if (mz2000 -> needReset || mz2000 -> needIpl)
				break;
			if (playDestroy)
				break;
			Sleep( 100 );
		}
	}
	return 0;
}

VOID set_directory( MZ2000 *mz )
{
	char moduleFile[MAX_PATH] = "";
	char drive[_MAX_DRIVE] = "";
	char dir[_MAX_DIR] = "";
	char fileName[_MAX_FNAME] = "";
	char ext[_MAX_EXT] = "";
	char commandline[1024] = "";
	char *p = NULL;

	/* set current directory and ini file path */
	if ( GetModuleFileName( NULL, moduleFile, sizeof(moduleFile)) > 0 ) {
		_splitpath(moduleFile, drive, dir, fileName, ext) ;
		_makepath( app_currentpath, drive, dir, "", "" );
		*(app_currentpath + strlen(app_currentpath) - 1) = '\0';
		SetCurrentDirectory( app_currentpath );
		_makepath( app_defaultinifile, drive, dir, fileName, INIFILEEXT );
	}
	else {
#ifdef WIN32
		GetCurrentDirectory( sizeof(app_currentpath), app_currentpath );
		strcpy( app_defaultinifile, app_currentpath ) ;
		GetFullPathName( INIFILENAME, sizeof(app_defaultinifile), app_defaultinifile, NULL ) ;
#else
		snprintf( app_defaultinifile, PATH_MAX, "%s/%s", app_currentpath, INIFILENAME );
#endif
	}

#ifdef	ENABLE_REGISTORIES
	/* set classes root registory */
	sprintf(commandline, "%s \"%%1\"", moduleFile);
	SetRegistory(HKEY_CLASSES_ROOT, ".mze", "", "EmuZ2000TF");
	SetRegistory(HKEY_CLASSES_ROOT, "EmuZ2000TF\\shell", "", "open");
	SetRegistory(HKEY_CLASSES_ROOT, "EmuZ2000TF\\shell\\open\\command", "", commandline);
#endif

	/* get command line params */
	*commandline_param = '\0';
	strcpy( commandline, GetCommandLine() );
	p = StrIStr( commandline + strlen( moduleFile ) + 2, "\"" );
	if(p != NULL){
		strcpy( commandline_param, p + 1 );
		p = StrIStr( commandline_param, "\"" );
		if(p != NULL){
			*p = '\0';
		}
	}
}

static void load_configuration( MZ2000 *mz )
{
	const char *section = INISECTIONNAME_BASIC;
	char *file = app_defaultinifile;
#ifdef WIN32
	char buff[_MAX_PATH];
#else
	char buff[PATH_MAX];
#endif
	int buffsize = sizeof(buff);

	/* UI Language */
	if(GetUserProfileString(section, "UI_Language", "", buff, buffsize, file) > 0) {
		mz -> language = atoi(buff);
	}

	/* Architecture / Clock */
	if(GetUserProfileString(section, "mzmode", "", buff, buffsize, file) > 0) {
		mz -> mzmode = atoi(buff);
	}
	if(GetUserProfileString(section, "nowZ80Clock", "", buff, buffsize, file) > 0) {
		mz -> nowZ80Clock = atoi(buff);
	}
	if(GetUserProfileString(section, "z80MaxClock", "", buff, buffsize, file) > 0) {
		mz -> z80MaxClock = atoi(buff);
	}

	/* Sound */
	if(GetUserProfileString(section, "soundmode", "", buff, buffsize, file) > 0) {
		mz -> soundmode = atoi(buff);
	}
	if(GetUserProfileString(section, "enable_soundtminterpolations", "", buff, buffsize, file) > 0) {
		mz -> enable_soundtminterpolations = atoi(buff);
	}
	if(GetUserProfileString(section, "enable_soundtape", "", buff, buffsize, file) > 0) {
		mz -> enable_soundtape = atoi(buff);
	}
	if(GetUserProfileString(section, "enable_soundfdd", "", buff, buffsize, file) > 0) {
		mz -> enable_soundfdd = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_bit", "", buff, buffsize, file) > 0) {
		mz -> sound_bit = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_freq", "", buff, buffsize, file) > 0) {
		mz -> sound_freq = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_playbuffer_ms", "", buff, buffsize, file) > 0) {
		mz -> sound_playbuffer_ms = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_delaytime_ms", "", buff, buffsize, file) > 0) {
		mz -> sound_delaytime_ms = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_volume1", "", buff, buffsize, file) > 0) {
		mz -> sound_volume1 = atoi(buff);
	}
	if(GetUserProfileString(section, "sound_volume2", "", buff, buffsize, file) > 0) {
		mz -> sound_volume2 = atoi(buff);
	}

	/* Other Hardware */
	if(GetUserProfileString(section, "displaymode", "", buff, buffsize, file) > 0) {
		mz -> displayMode = atoi(buff);
	}
	if(GetUserProfileString(section, "screen_highreso", "", buff, buffsize, file) > 0) {
		mz -> screen_highreso = atoi(buff);
	}
	if(GetUserProfileString(section, "vblankspecial", "", buff, buffsize, file) > 0) {
		mz -> vblankspecial = atoi(buff);
	}
	if(GetUserProfileString(section, "tapespeaker", "", buff, buffsize, file) > 0) {
		mz -> tapespeaker = atoi(buff);
	}
	if(GetUserProfileString(section, "enable_fddif", "", buff, buffsize, file) > 0) {
		mz -> enable_fddif = atoi(buff);
	}
	if(GetUserProfileString(section, "fd_exec_fast", "", buff, buffsize, file) > 0) {
		mz -> fd_exec_fast = atoi(buff);
	}
	if(GetUserProfileString(section, "fd_ignore_crc", "", buff, buffsize, file) > 0) {
		mz -> fd_ignore_crc = atoi(buff);
	}
	if(GetUserProfileString(section, "fd_erase_sct17crc", "", buff, buffsize, file) > 0) {
		mz -> fd_erase_sct17crc = atoi(buff);
	}
	if(GetUserProfileString(section, "droptofd2", "", buff, buffsize, file) > 0) {
		mz -> droptofd2 = atoi(buff);
	}
	if(GetUserProfileString(section, "dropautoipl", "", buff, buffsize, file) > 0) {
		mz -> dropautoipl = atoi(buff);
	}
	if(GetUserProfileString(section, "dropautoipltape", "", buff, buffsize, file) > 0) {
		mz -> dropautoipltape = atoi(buff);
	}
	if(GetUserProfileString(section, "forcereadonly", "", buff, buffsize, file) > 0) {
		mz -> forcereadonly = atoi(buff);
	}
	if(GetUserProfileString(section, "enable_autoinitio_reset", "", buff, buffsize, file) > 0) {
		mz -> enable_initreset = atoi(buff);
	}
	if(GetUserProfileString(section, "iplmemclr", "", buff, buffsize, file) > 0) {
		mz -> iplmemclr = atoi(buff);
	}
	if(GetUserProfileString(section, "iplmemzclr", "", buff, buffsize, file) > 0) {
		mz -> iplmemzclr = atoi(buff);
	}
	if(GetUserProfileString(section, "font_select", "", buff, buffsize, file) > 0) {
		mz -> font_select = atoi(buff);
	}
	if(GetUserProfileString(section, "font_english", "", buff, buffsize, file) > 0) {
		mz -> font_english = atoi(buff);
	}
	if(GetUserProfileString(section, "consoler_cblank", "", buff, buffsize, file) > 0) {
		mz -> consoler_cblank = atoi(buff);
	}
	if(GetUserProfileString(section, "keyboard_mode", "", buff, buffsize, file) > 0) {
		mz -> keyboard_mode = atoi(buff);
	}
	if(GetUserProfileString(section, "fullscreen_device", "", buff, buffsize, file) > 0) {
		mz -> fullscreen_device = atoi(buff);
	}
	if(GetUserProfileString(section, "fullscreen_resolution", "", buff, buffsize, file) > 0) {
		mz -> fullscreen_resolution = atoi(buff);
	}
	if(GetUserProfileString(section, "fullscreen_bpp", "", buff, buffsize, file) > 0) {
		mz -> fullscreen_bpp = atoi(buff);
	}

	/* Green Monitor Color */
	if(GetUserProfileString(section, "greencolor_normal_foreground_red", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nfr = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_normal_foreground_green", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nfg = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_normal_foreground_blue", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nfb = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_normal_background_red", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nbr = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_normal_background_green", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nbg = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_normal_background_blue", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_nbb = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_foreground_red", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rfr = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_foreground_green", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rfg = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_foreground_blue", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rfb = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_background_red", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rbr = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_background_green", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rbg = (unsigned char)c;
	}
	if(GetUserProfileString(section, "greencolor_reverse_background_blue", "", buff, buffsize, file) > 0) {
		int c = atoi(buff);
		if (c < 0)
			c = 0;
		if (c > 255)
			c = 255;
		mz -> greencolor_rbb = (unsigned char)c;
	}

	/* Memory Wait and Frame Clock Adjust */
	if(GetUserProfileString(section, "frameclock_alpha", "", buff, buffsize, file) > 0) {
		mz -> frameclock_alpha = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waittvramclock_r", "", buff, buffsize, file) > 0) {
		mz -> memory_waittvramclock_r = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waittvramclock_w", "", buff, buffsize, file) > 0) {
		mz -> memory_waittvramclock_w = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waitgramclock_r", "", buff, buffsize, file) > 0) {
		mz -> memory_waitgramclock_r = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waitgramclock_w", "", buff, buffsize, file) > 0) {
		mz -> memory_waitgramclock_w = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waitgram80bclock_r", "", buff, buffsize, file) > 0) {
		mz -> memory_waitgram80bclock_r = atoi(buff);
	}
	if(GetUserProfileString(section, "memory_waitgram80bclock_w", "", buff, buffsize, file) > 0) {
		mz -> memory_waitgram80bclock_w = atoi(buff);
	}

	/* State Save */
	if(GetUserProfileString(section, "state_filename", "", buff, buffsize, file) > 0) {
#ifdef WIN32
		strncpy( (char *)mz -> sfilename, buff, _MAX_PATH );
		mz -> sfilename[_MAX_PATH - 1] = '\0';
#else
		strncpy( (char *)mz -> sfilename, buff, PATH_MAX );
		mz -> sfilename[PATH_MAX - 1] = '\0';
#endif
	}

	/* Tape */
	if(GetUserProfileString(section, "tapePitch", "", buff, buffsize, file) > 0) {
		mz -> tapePitch = atoi(buff);
	}
	if(GetUserProfileString(section, "tapeThreshold", "", buff, buffsize, file) > 0) {
		mz -> tapeThreshold = atoi(buff);
	}
	if(GetUserProfileString(section, "amplitudeRange", "", buff, buffsize, file) > 0) {
		mz -> amplitudeRange = atoi(buff);
	}
	if(GetUserProfileString(section, "writeThreshold", "", buff, buffsize, file) > 0) {
		mz -> writeThreshold = atoi(buff);
	}
	if(GetUserProfileString(section, "tapemode_reverse", "", buff, buffsize, file) > 0) {
		mz -> tapemode_reverse = atoi(buff);
	}
	if(GetUserProfileString(section, "tapeff_ignore", "", buff, buffsize, file) > 0) {
		mz -> tapeff_ignore = atoi(buff);
	}
	if(GetUserProfileString(section, "tapewrite_interpolation", "", buff, buffsize, file) > 0) {
		mz -> tapewrite_interpolation = atoi(buff);
	}
	if(GetUserProfileString(section, "tape_filename", "", buff, buffsize, file) > 0) {
#ifdef WIN32
		strncpy( (char *)mz -> tfilename, buff, _MAX_PATH );
		mz -> tfilename[_MAX_PATH - 1] = '\0';
#else
		strncpy( (char *)mz -> tfilename, buff, PATH_MAX );
		mz -> tfilename[PATH_MAX - 1] = '\0';
#endif
	}

	/* Tape Counter */
	if(GetUserProfileString(section, "tapecounter_mode", "", buff, buffsize, file) > 0) {
		mz -> tapecounter = atoi(buff);
	}
	if(GetUserProfileString(section, "tape_countspeed_2000", "", buff, buffsize, file) > 0) {
		mz -> tapectspeed_2000 = atoi(buff);
	}
	if(GetUserProfileString(section, "tape_countspeed_80B", "", buff, buffsize, file) > 0) {
		mz -> tapectspeed_80B = atoi(buff);
	}
	if(GetUserProfileString(section, "tape_countspeed_2500", "", buff, buffsize, file) > 0) {
		mz -> tapectspeed_2500 = atoi(buff);
	}

	/* QD */
	if(GetUserProfileString(section, "qd_filename", "", buff, buffsize, file) > 0) {
#ifdef WIN32
		strncpy( (char *)mz -> qfilename, buff, _MAX_PATH );
		mz -> qfilename[_MAX_PATH - 1] = '\0';
#else
		strncpy( (char *)mz -> qfilename, buff, PATH_MAX );
		mz -> qfilename[PATH_MAX - 1] = '\0';
#endif
	}

#ifdef ENABLE_D3D
	/* Direct 3D */
	if(GetUserProfileString(section, "draw_type", "", buff, buffsize, file) > 0) {
		int c = atoi( buff );
		gbD3D = (c == 0) ? FALSE : TRUE;
		gbD3DF = ( c & 2) ? TRUE : FALSE;
	}
	if(GetUserProfileString(section, "rpi_type", "", buff, buffsize, file) > 0) {
		gdwRPI = (DWORD)atoi( buff );
	}
#endif
	return;
}

void save_configuration( MZ2000 *mz )
{
	const char *section = INISECTIONNAME_BASIC;
	char *file = app_defaultinifile;
	char buff[256];

	if (!mz || !mz2000_global)
		return;

	/* UI Language */
	sprintf(buff, "%d", mz -> language);
	WriteUserProfileString(section, "UI_Language", buff, file);

	/* Architecture / Clock */
	sprintf(buff, "%d", mz -> mzmode);
	WriteUserProfileString(section, "mzmode", buff, file);
	sprintf(buff, "%d", mz -> nowZ80Clock);
	WriteUserProfileString(section, "nowZ80Clock", buff, file);
	sprintf(buff, "%d", mz -> z80MaxClock);
	WriteUserProfileString(section, "z80MaxClock", buff, file);

	/* Sound */
	sprintf(buff, "%d", mz -> soundmode);
	WriteUserProfileString(section, "soundmode", buff, file);
	sprintf(buff, "%d", mz -> enable_soundtminterpolations);
	WriteUserProfileString(section, "enable_soundtminterpolations", buff, file);
	sprintf(buff, "%d", mz -> enable_soundtape);
	WriteUserProfileString(section, "enable_soundtape", buff, file);
	sprintf(buff, "%d", mz -> enable_soundfdd);
	WriteUserProfileString(section, "enable_soundfdd", buff, file);
	sprintf(buff, "%d", mz -> sound_bit);
	WriteUserProfileString(section, "sound_bit", buff, file);
	sprintf(buff, "%d", mz -> sound_freq);
	WriteUserProfileString(section, "sound_freq", buff, file);
	sprintf(buff, "%d", mz -> sound_playbuffer_ms);
	WriteUserProfileString(section, "sound_playbuffer_ms", buff, file);
	sprintf(buff, "%d", mz -> sound_delaytime_ms);
	WriteUserProfileString(section, "sound_delaytime_ms", buff, file);
	sprintf(buff, "%d", mz -> sound_volume1);
	WriteUserProfileString(section, "sound_volume1", buff, file);
	sprintf(buff, "%d", mz -> sound_volume2);
	WriteUserProfileString(section, "sound_volume2", buff, file);

	/* Other Hardware */
	sprintf(buff, "%d", mz -> displayMode);
	WriteUserProfileString(section, "displaymode", buff, file);
	sprintf(buff, "%d", mz -> screen_highreso);
	WriteUserProfileString(section, "screen_highreso", buff, file);
	sprintf(buff, "%d", mz -> vblankspecial);
	WriteUserProfileString(section, "vblankspecial", buff, file);
	sprintf(buff, "%d", mz -> tapespeaker);
	WriteUserProfileString(section, "tapespeaker", buff, file);
	sprintf(buff, "%d", mz -> enable_fddif);
	WriteUserProfileString(section, "enable_fddif", buff, file);
	sprintf(buff, "%d", mz -> fd_exec_fast);
	WriteUserProfileString(section, "fd_exec_fast", buff, file);
	sprintf(buff, "%d", mz -> fd_ignore_crc);
	WriteUserProfileString(section, "fd_ignore_crc", buff, file);
	sprintf(buff, "%d", mz -> fd_erase_sct17crc);
	WriteUserProfileString(section, "fd_erase_sct17crc", buff, file);
	sprintf(buff, "%d", mz -> forcereadonly);
	WriteUserProfileString(section, "forcereadonly", buff, file);
	sprintf(buff, "%d", mz -> droptofd2);
	WriteUserProfileString(section, "droptofd2", buff, file);
	sprintf(buff, "%d", mz -> dropautoipl);
	WriteUserProfileString(section, "dropautoipl", buff, file);
	sprintf(buff, "%d", mz -> dropautoipltape);
	WriteUserProfileString(section, "dropautoipltape", buff, file);
	sprintf(buff, "%d", mz -> enable_initreset);
	WriteUserProfileString(section, "enable_autoinitio_reset", buff, file);
	sprintf(buff, "%d", mz -> iplmemclr);
	WriteUserProfileString(section, "iplmemclr", buff, file);
	sprintf(buff, "%d", mz -> iplmemzclr);
	WriteUserProfileString(section, "iplmemzclr", buff, file);
	sprintf(buff, "%d", mz -> font_select);
	WriteUserProfileString(section, "font_select", buff, file);
	sprintf(buff, "%d", mz -> font_english);
	WriteUserProfileString(section, "font_english", buff, file);
	sprintf(buff, "%d", mz -> consoler_cblank);
	WriteUserProfileString(section, "consoler_cblank", buff, file);
	sprintf(buff, "%d", mz -> keyboard_mode);
	WriteUserProfileString(section, "keyboard_mode", buff, file);
	sprintf(buff, "%d", mz -> fullscreen_device);
	WriteUserProfileString(section, "fullscreen_device", buff, file);
	sprintf(buff, "%d", mz -> fullscreen_resolution);
	WriteUserProfileString(section, "fullscreen_resolution", buff, file);
	sprintf(buff, "%d", mz -> fullscreen_bpp);
	WriteUserProfileString(section, "fullscreen_bpp", buff, file);

	/* Green Monitor Color */
	sprintf(buff, "%d", mz -> greencolor_nfr);
	WriteUserProfileString(section, "greencolor_normal_foreground_red", buff, file);
	sprintf(buff, "%d", mz -> greencolor_nfg);
	WriteUserProfileString(section, "greencolor_normal_foreground_green", buff, file);
	sprintf(buff, "%d", mz -> greencolor_nfb);
	WriteUserProfileString(section, "greencolor_normal_foreground_blue", buff, file);
	sprintf(buff, "%d", mz -> greencolor_nbr);
	WriteUserProfileString(section, "greencolor_normal_background_red", buff, file);
	sprintf(buff, "%d", mz -> greencolor_nbg);
	WriteUserProfileString(section, "greencolor_normal_background_green", buff, file);
	sprintf(buff, "%d", mz -> greencolor_nbb);
	WriteUserProfileString(section, "greencolor_normal_background_blue", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rfr);
	WriteUserProfileString(section, "greencolor_reverse_foreground_red", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rfg);
	WriteUserProfileString(section, "greencolor_reverse_foreground_green", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rfb);
	WriteUserProfileString(section, "greencolor_reverse_foreground_blue", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rbr);
	WriteUserProfileString(section, "greencolor_reverse_background_red", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rbg);
	WriteUserProfileString(section, "greencolor_reverse_background_green", buff, file);
	sprintf(buff, "%d", mz -> greencolor_rbb);
	WriteUserProfileString(section, "greencolor_reverse_background_blue", buff, file);

	/* Memory Wait and Frame Clock Adjust */
	sprintf(buff, "%d", mz -> frameclock_alpha);
	WriteUserProfileString(section, "frameclock_alpha", buff, file);
	sprintf(buff, "%d", mz -> memory_waittvramclock_r);
	WriteUserProfileString(section, "memory_waittvramclock_r", buff, file);
	sprintf(buff, "%d", mz -> memory_waittvramclock_w);
	WriteUserProfileString(section, "memory_waittvramclock_w", buff, file);
	sprintf(buff, "%d", mz -> memory_waitgramclock_r);
	WriteUserProfileString(section, "memory_waitgramclock_r", buff, file);
	sprintf(buff, "%d", mz -> memory_waitgramclock_w);
	WriteUserProfileString(section, "memory_waitgramclock_w", buff, file);
	sprintf(buff, "%d", mz -> memory_waitgram80bclock_r);
	WriteUserProfileString(section, "memory_waitgram80bclock_r", buff, file);
	sprintf(buff, "%d", mz -> memory_waitgram80bclock_w);
	WriteUserProfileString(section, "memory_waitgram80bclock_w", buff, file);

	/* State Save */
	WriteUserProfileString(section, "state_filename", (char *)mz -> sfilename, file);

	/* Tape */
	sprintf(buff, "%d", mz -> tapePitch);
	WriteUserProfileString(section, "tapePitch", buff, file);
	sprintf(buff, "%d", mz -> tapeThreshold);
	WriteUserProfileString(section, "tapeThreshold", buff, file);
	sprintf(buff, "%d", mz -> amplitudeRange);
	WriteUserProfileString(section, "amplitudeRange", buff, file);
	sprintf(buff, "%d", mz -> writeThreshold);
	WriteUserProfileString(section, "writeThreshold", buff, file);
	sprintf(buff, "%d", mz -> tapemode_reverse);
	WriteUserProfileString(section, "tapemode_reverse", buff, file);
	sprintf(buff, "%d", mz -> tapewrite_interpolation);
	WriteUserProfileString(section, "tapewrite_interpolation", buff, file);
	sprintf(buff, "%d", mz -> tapeff_ignore);
	WriteUserProfileString(section, "tapeff_ignore", buff, file);
	WriteUserProfileString(section, "tape_filename", (char *)mz -> tfilename, file);

	/* Tape Counter */
	sprintf(buff, "%d", mz -> tapecounter);
	WriteUserProfileString(section, "tapecounter_mode", buff, file);
	sprintf(buff, "%d", mz -> tapectspeed_2000);
	WriteUserProfileString(section, "tape_countspeed_2000", buff, file);
	sprintf(buff, "%d", mz -> tapectspeed_80B);
	WriteUserProfileString(section, "tape_countspeed_80B", buff, file);
	sprintf(buff, "%d", mz -> tapectspeed_2500);
	WriteUserProfileString(section, "tape_countspeed_2500", buff, file);

	/* QD */
	WriteUserProfileString(section, "qd_filename", (char *)mz -> qfilename, file);

#ifdef ENABLE_D3D
	/* Direct 3D */
	sprintf(buff, "%d", 
		!gbD3D ? 0 : (gbD3DF ? 2 : 1) );
	WriteUserProfileString(section, "draw_type", buff, file);
	
	sprintf(buff, "%d", gdwRPI );
	WriteUserProfileString(section, "rpi_type", buff, file);
#endif
	return;
}

MZ2000 *init_mz2000( void )
{
	MZ2000 *mz2000;

	strncpy( module_name, EMZ2000_MODULENAME, LNMAX );
	module_name[LNMAX - 1] = '\0';
	mz2000 = (MZ2000 *)CALLOC( sizeof(MZ2000), 1 );
	if (!mz2000)
		return NULL;
	/* Clock */
	mz2000 -> nowZ80Clock	= DEFAULT_SPEED_KHZ;	/* Default CPU speed */
	mz2000 -> z80MaxClock	= DEFAULT_SPEED_KHZ * 100; /* 400MHz */
	/* Architecture */
	mz2000 -> mzmode	= FALSE;		/* MZ-2000 */
	/* Sound */
	mz2000 -> soundmode	= TRUE;			/* Sound On */
	mz2000 -> enable_soundtminterpolations = TRUE;	/* Time Interpolations On */
	mz2000 -> enable_soundtape = TRUE;		/* TAPE Sound On */
	mz2000 -> enable_soundfdd = TRUE;		/* FDD Sound On */
	mz2000 -> sound_bit	= DEFAULT_SOUND_BIT;	/* Sound Output Bit */
	mz2000 -> sound_freq	= DEFAULT_SOUND_FREQ;	/* Sound Output Freq. */
	mz2000 -> sound_playbuffer_ms = DEFAULT_SOUND_PLAYBUFFER_MS; /* Sound Play Buffer (ms) */
	mz2000 -> sound_delaytime_ms  = DEFAULT_SOUND_DELAYTIME_MS;  /* Sound Delay Time (ms) */
	mz2000 -> sound_volume1 = DEFAULT_SOUND_VOLUME1; /* Sound Volume MZ Pulse Sound */
	mz2000 -> sound_volume2 = DEFAULT_SOUND_VOLUME2; /* Sound Volume Effect Waves */

	/* Other Hardware */
	mz2000 -> screen_highreso = TRUE;		/* Screen set to 15KHz */
	mz2000 -> vblankspecial	= FALSE;		/* V-Blank Mode Normal */
	mz2000 -> tapespeaker	= FALSE;		/* Tape Sound Off */
	mz2000 -> enable_fddif	= FALSE;		/* Enable FDD I/F */
	mz2000 -> fd_exec_fast	= FALSE;		/* In FD, no fast running mode */
	mz2000 -> fd_ignore_crc = FALSE;		/* In FD, ignore read crc error */
	mz2000 -> fd_erase_sct17crc = FALSE;		/* In FD, erase sector 17 crc to 16 sectors */
	mz2000 -> forcereadonly	= FALSE;		/* Force Read Only Off */
	mz2000 -> droptofd2	= FALSE;		/* Drop to FD2 */
	mz2000 -> dropautoipl	= TRUE;			/* Drop Auto IPL On */
	mz2000 -> dropautoipltape = TRUE;		/* Drop Auto IPL TAPE Off */
	mz2000 -> enable_initreset = TRUE;		/* Auto Initialize I/O on Reset */
	mz2000 -> iplmemclr	= TRUE;			/* IPL reset memory init On */
	mz2000 -> iplmemzclr	= FALSE;		/* Memory init is zero clear On */
	mz2000 -> font_select	= TEXTFONT_200LINE;	/* Font Select 200line */
	mz2000 -> font_english	= FALSE;		/* Font English mode Off */
	mz2000 -> consoler_cblank = FALSE;		/* Console Rev. Blank on Color Display */
	mz2000 -> keyboard_mode = 0;			/* Reset the keyboard mode */
	mz2000 -> fullscreen_device = SCREEN_DEVICE_DEFAULT; /* Fullscreen Device */
	mz2000 -> fullscreen_resolution = SCREEN_RESOLUTION_AUTO; /* Fullscreen Resolution */
	mz2000 -> fullscreen_bpp = SCREEN_BPP_DEFAULT;	/* Fullscreen BPP */
	mz2000 -> language	= UILANG_AUTOMATIC;	/* Language : Automatic */
	/* Green Monitor Color */
	mz2000 -> greencolor_nfr = MZGREENCOLORN_FR_DEFAULT;
	mz2000 -> greencolor_nfg = MZGREENCOLORN_FG_DEFAULT;
	mz2000 -> greencolor_nfb = MZGREENCOLORN_FB_DEFAULT;
	mz2000 -> greencolor_nbr = MZGREENCOLORN_BR_DEFAULT;
	mz2000 -> greencolor_nbg = MZGREENCOLORN_BG_DEFAULT;
	mz2000 -> greencolor_nbb = MZGREENCOLORN_BB_DEFAULT;
	mz2000 -> greencolor_rfr = MZGREENCOLORR_FR_DEFAULT;
	mz2000 -> greencolor_rfg = MZGREENCOLORR_FG_DEFAULT;
	mz2000 -> greencolor_rfb = MZGREENCOLORR_FB_DEFAULT;
	mz2000 -> greencolor_rbr = MZGREENCOLORR_BR_DEFAULT;
	mz2000 -> greencolor_rbg = MZGREENCOLORR_BG_DEFAULT;
	mz2000 -> greencolor_rbb = MZGREENCOLORR_BB_DEFAULT;
	/* Memory Wait and Frame Clock Adjust */
	mz2000 -> frameclock_alpha = MZCLOCK_ADJUSTVALUE;
	mz2000 -> memory_waittvramclock_r	= MZMEMORY_WAITTVRAMCLOCK_R;
	mz2000 -> memory_waittvramclock_w	= MZMEMORY_WAITTVRAMCLOCK_W;
	mz2000 -> memory_waitgramclock_r	= MZMEMORY_WAITGRAMCLOCK_R;
	mz2000 -> memory_waitgramclock_w	= MZMEMORY_WAITGRAMCLOCK_W;
	mz2000 -> memory_waitgram80bclock_r	= MZMEMORY_WAITGRAM80BCLOCK_R;
	mz2000 -> memory_waitgram80bclock_w	= MZMEMORY_WAITGRAM80BCLOCK_W;
	/* Tape Couner */
	mz2000 -> tapecounter		= TAPE_COUNTERMODE_DEFAULT;
	mz2000 -> tapectspeed_2000	= TAPECOUNTER_DEFAULTSPEED_2000;
	mz2000 -> tapectspeed_80B	= TAPECOUNTER_DEFAULTSPEED_80B;
	mz2000 -> tapectspeed_2500	= TAPECOUNTER_DEFAULTSPEED_2500;
	/* Tape */
	mz2000 -> tapePitch		= DEFAULT_TAPE_PITCH;
	mz2000 -> tapeThreshold		= DEFAULT_TAPE_THRESHOLD;
	mz2000 -> amplitudeRange	= DEFAULT_TAPE_AMPLITUDERANGE;
	mz2000 -> writeThreshold	= DEFAULT_TAPE_WRITETHRESHOLD;
	mz2000 -> tapemode_reverse	= FALSE;		/* Tape mode reverse */
	mz2000 -> tapeff_ignore		= FALSE;		/* Ignore Tape FF */
	mz2000 -> tapewrite_interpolation = FALSE;		/* Write Interpolation On/Off */
#ifdef ENABLE_D3D
	/* Direct 3D */
	gbD3D = TRUE;
	gbD3F = FALSE;
	gdRPI = 0;
#endif

	set_directory( mz2000 );
	load_configuration( mz2000 );

	if (!z80_init( mz2000 -> nowZ80Clock )) {
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!io_init( mz2000 )) {
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!memory_init( mz2000 )) {
		io_exit( mz2000 );
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!screen_init( mz2000 )) {
		memory_exit( mz2000 );
		io_exit( mz2000 );
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!keyboard_init( mz2000 )) {
		screen_exit( mz2000 );
		memory_exit( mz2000 );
		io_exit( mz2000 );
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!tape_init( mz2000 )) {
		/* sound_exit(); */
		keyboard_exit( mz2000 );
		screen_exit( mz2000 );
		memory_exit( mz2000 );
		io_exit( mz2000 );
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	if (!disk_init( mz2000 )) {
		/* sound_exit(); */
		tape_exit( mz2000 );
		keyboard_exit( mz2000 );
		screen_exit( mz2000 );
		memory_exit( mz2000 );
		io_exit( mz2000 );
		z80_exit();
		FREE( (void *)mz2000 );
		return NULL;
	}
	disk_setparam( mz2000 -> fd_exec_fast, mz2000 -> fd_ignore_crc, mz2000 -> fd_erase_sct17crc );
	load_iplrom( mz2000 );
	reset_z80();
	iplreset_mz( mz2000 );
	mz2000 -> howipl = 1;	/* internal function */
	return mz2000;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
