/* File: state.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		States Save/Load Module

	Copyright (C) 2002-2019 FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.

	Caution !!
	Import the old EMUZ file functions include the old EmuZ-2000 program code.
	These are released open source code without copyright information.
*/

#include "config.h"

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "common.h"
#include "resource.h"
#include "mz2000.h"
#include "../Plugin/plugin.h"

#define ENABLE_STATEEMUZ_IMPORT

#ifdef ENABLE_STATEEMUZ_IMPORT
/* EmuZ-2000 Ver.0.400pre base structures */

#define	emuzold_statesave10 "EmuZ-2000 State Save 1.0"
#define	emuzold_statesave11 "EmuZ-2000 State Save 1.1"

typedef struct emuzold_mzioinfo {
	u8 port_d8;
	u8 port_d9;
	u8 port_da;
	u8 port_db;
	u8 port_dc;
	u8 port_dd;
	u8 port_de;
	u8 port_df;
	u8 port_e0;
	u8 port_e1;
	u8 port_e2;
	u8 port_e3;
	u8 port_e4;
	u8 port_e5;
	u8 port_e6;
	u8 port_e7;
	u8 port_e8;
	u8 port_e9;
	u8 port_ea;
	u8 port_eb;
	u8 port_f4;
	u8 port_f5;
	u8 port_f6;
	u8 port_f7;
	int strobe_low;
} EMUZOLD_MZIOInfo;

typedef struct emuzold_mz2000 {
	u8 *ipl;
	u8 *memory;
	u8 *vram;
	u8 *gram1, *gram2, *gram3;
	int inipl, vramon, selchr, gramno;
	int vramChg,gram1Chg,gram2Chg,gram3Chg;
	int viewChg, viewMsg, mzChgGraphMode;
	int howipl;	/* 0: no need, 1: internal function, 2: z80 program */
	EMUZOLD_MZIOInfo *ioinfo;
	int displayMode;  //0:green 1:color
	int greenGraphOn; //0:GreenMode Graph off 1:GreenMode Graph on
	int greenTextOn;  //0:GreenMode Text off  1:Green Mode Text on
	int displayAttr;  //0:t-Front..   1:G-Front (displayMode+dispAttr)=... 
	int textColor;    //Text Color Number
	int backColor;    //Back Color Number
	int needIpl;      //1 Active
	short needReset;  //1 Active
	short mzMode;     //0:MZ-2000/2200  1:MZ-80B
	u8 graphMask[4];

	int idmy0;
	int idmy1;
	int idmy2;
	int idmy3;
	int idmy4;
	int idmy5;

	HANDLE z80handle;
	HANDLE viewHandle;
	HANDLE loopmzHandle;
	DWORD z80thArg, z80ID;
	DWORD viewArg, viewID, loopmzID;

	HINSTANCE ghInstance;
	HHOOK hKeyHook;

	int changeClockDialog;

	HWND	ghWnd;

	int nowZ80Clock;
	volatile int playStop;

	BYTE intVect;
	BYTE bitMask;

	void *gdimem;
	HDC Buffer;
	HBITMAP gbmOldMonoBitmap;
	HPALETTE hpalApp;
	volatile DWORD viewMessageVal;

	int pwmsrc;
	
	DWORD tapePich;
	DWORD tapeThreshold;
	DWORD changeTapeDialog;
	int tapeLoad;

	LONGLONG z80MaxTime;

	DWORD fontSelect;
	BYTE mzFont[256][16];

	RGBQUAD	palletTableTemp[4][16];

	DWORD dwPause;
	DWORD dwDebugPrint;
	BYTE dwDebugPrintDrawCount;
	BYTE pwmMode;

	BYTE keycodeEx;

	BYTE playMode;

	DWORD oldFlg;
	DWORD viewDebug;
	u8 *dvram;
	u8 *dgram;
	BYTE nowPlayFileName[256];
} EMUZOLD_MZ2000;

struct emuzold_z80mement {
	u8 *ptr;
	u8 (*readf)(u16 addr);
	int (*writef)(u16 addr, u8 value);
};

struct emuzold_z80ioent {
	u8 (*readf)(u16 port);
	int (*writef)(u16 port, u8 value);
};

typedef struct emuzold_z80 {
	struct emuzold_z80mement mem[16];
#ifdef IO16BIT
	struct emuzold_z80ioent io[65536];
#else
	struct emuzold_z80ioent io[256];
#endif
	u8 a,b,c,d,e,f,h,l;
	u8 a2,b2,c2,d2,e2,f2,h2,l2;
	u16 ix,iy,sp,pc;
	u8 r,i,iff1,iff2,im;
	z80clk internal_clock;
	z80clk left_clock;
	int speed_khz;
	int halt_state;
	int timer_wait;
	HANDLE timer_cond;
	LONG timer_mutex;
	int external_signal;
	int onbus;
	int stop;
	LONG z80_mutex;
#ifdef TRACE
	int trace;
#endif
	int (*(*instructs[7]))();
} EMUZOLD_Z80;

static int import_stateemuz_main( MZ2000 *mz, const char *file )
{
	int i, version11_flag;
	HANDLE hFile;
	DWORD dwReadSize;
	EMUZOLD_Z80 tz80;
	EMUZOLD_MZ2000 tmz;
	EMUZOLD_MZIOInfo tio;
	BYTE tmp[LNMAX];
	z80clk tt;

	hFile = CreateFile( file,
		GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "Load states failed (CreateFile)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	ReadFile( hFile, tmp, sizeof(emuzold_statesave11), &dwReadSize, NULL );
	version11_flag = FALSE;
	if (dwReadSize != sizeof(emuzold_statesave10) || strncmp( tmp, emuzold_statesave10, sizeof(emuzold_statesave10) )) {
		if (strncmp( tmp, emuzold_statesave11, sizeof(emuzold_statesave11) )) {
			CloseHandle( hFile );
			MessageBox( NULL, "Load states failed (Eyecache or version error)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
			return -1;
		}
		version11_flag = TRUE;
		/* skip the version 1.1 extend data */
		ReadFile( hFile, &tmp, 0x30, &dwReadSize, NULL );
		if (dwReadSize != 0x30) {
			CloseHandle( hFile );
			return -2;
		}
		MessageBox( NULL, "Warning, I found the \"EmuZ-2000 States Save 1.1\", skip the timer(8253) and keyboard interrupt information.", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
	}
	ReadFile( hFile, &tz80, sizeof(EMUZOLD_Z80), &dwReadSize, NULL );
	if (dwReadSize != sizeof(EMUZOLD_Z80)) {
		CloseHandle( hFile );
		return -2;
	}
	ReadFile( hFile, &tmz, sizeof(EMUZOLD_MZ2000), &dwReadSize, NULL );
	if (dwReadSize != sizeof(EMUZOLD_MZ2000)) {
		CloseHandle( hFile );
		return -2;
	}

	/* update the system structures */
	mz -> mzmode = tmz . mzMode;
	mz -> howipl = 0;
	reset_mz2000_sub( mz );
	reinit_io( mz );

	/* Z80 structure */
	memcpy( (void *)&z80.a, &tz80.a, ((DWORD)&tz80.internal_clock - (DWORD)&tz80.a) );
	z80.halt_state = tz80.halt_state;
	z80.external_signal = tz80.external_signal;
	z80.onbus = tz80.onbus;
	z80.stop = tz80.stop;
	z80.internal_clock = tz80.internal_clock;
	z80.left_clock = tz80.left_clock;
	z80.speed_khz = tz80.speed_khz;

	/* MZ structure */
	mz -> inipl  = tmz . inipl;
	mz -> vramon = tmz . vramon;
	mz -> selchr = tmz . selchr;
	mz -> gramno = tmz . gramno;
	mz -> blank  = FALSE;
	mz -> nowZ80Clock = tmz .nowZ80Clock;
	mz -> howipl = 0; 
	mz -> cpu_peripherals_start = CPUPERIPHERALS_INIT;
	/* 8253 support only RESET */
	tt = get_internal_clock();
	for (i = 0; i < TIMER8253_COUNTER_MAXNUM; i++) {
		mz -> timer8253_starttime[i] = tt;
		mz -> timer8253_currenttime[i] = tt;
		mz -> timer8253_rwmode[i] = 0;
	}
	mz -> timer8253_count[0] = 0x7a12;
	mz -> timer8253_count[1] = 0xa8c0;
	mz -> timer8253_count[2] = 0x0000;
	mz -> timer8253_fast_mode = 0;
	/* Disable interrupt for Z80 PIO (Keyboard) */
	mz -> keyint_select = 0xffU;
	mz -> keyint_func   = NULL;
	mz -> intVect = tmz . intVect;
	mz -> bitMask = tmz . bitMask;
	/* Screen */
	mz -> displayMode  = tmz . displayMode;
	mz -> greenGraphOn = tmz . greenGraphOn;
	mz -> greenTextOn  = tmz . greenTextOn;
	mz -> displayAttr  = tmz . displayAttr;
	mz -> textColor    = tmz . textColor;
	mz -> backColor    = tmz . backColor;
	for (i = 0; i <= 3 + 1; i++)
		mz -> graphMask[i] = tmz . graphMask[i];

	/* IO structure */
	ReadFile( hFile, &tio, sizeof(EMUZOLD_MZIOInfo), &dwReadSize, NULL );
	if (dwReadSize != sizeof(EMUZOLD_MZIOInfo)) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}

	ReadFile( hFile, mz->memory, MZ_MAINMEMORY_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_MAINMEMORY_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->vram, MZ_TEXTVRAM_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram1, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram2, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram3, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->dvram, MZ_TEXTVRAM_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->dgram, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	CloseHandle( hFile );

	mz -> ioinfo -> port_d8 = tio . port_d8;
	mz -> ioinfo -> port_d9 = tio . port_d9;
	mz -> ioinfo -> port_da = tio . port_da;
	mz -> ioinfo -> port_db = tio . port_db;
	mz -> ioinfo -> port_dc = tio . port_dc;
	mz -> ioinfo -> port_dd = tio . port_dd;
	mz -> ioinfo -> port_e0 = tio . port_e0;
	mz -> ioinfo -> port_e1 = tio . port_e1;
	mz -> ioinfo -> port_e2 = tio . port_e2;
	mz -> ioinfo -> port_e3 = tio . port_e3;
	mz -> ioinfo -> port_e4 = tio . port_e4;
	mz -> ioinfo -> port_e5 = tio . port_e5;
	mz -> ioinfo -> port_e6 = tio . port_e6;
	mz -> ioinfo -> port_e7 = tio . port_e7;
	mz -> ioinfo -> port_e8 = tio . port_e8;
	mz -> ioinfo -> port_e9 = tio . port_e9;
	mz -> ioinfo -> port_ea = tio . port_ea;
	mz -> ioinfo -> port_eb = tio . port_eb;
	mz -> ioinfo -> port_f4 = tio . port_f4;
	mz -> ioinfo -> port_f5 = tio . port_f5;
	mz -> ioinfo -> port_f6 = tio . port_f6;
	mz -> ioinfo -> port_f7 = tio . port_f7;
	mz -> ioinfo -> strobe_low = tio . strobe_low;

	PaletteChange( mz );
	refresh_io( mz );
	screen_sethighreso( mz, mz -> screen_highreso );
	mz -> vramChg = mz -> gram1Chg = mz -> gram2Chg = mz -> gram3Chg = TRUE;
	return 0;
}
#endif /* ENABLE_STATEEMUZ_IMPORT */

static int save_state_main( MZ2000 *mz, const char *file )
{
	int i;
	HANDLE hFile;
	DWORD dwWriteSize;
	unsigned char head[16], info[16+PLUGIN_GUIDLENGTH_MAXNUM];
	unsigned int val;

	hFile = CreateFile( file,
		GENERIC_WRITE, FILE_SHARE_WRITE, NULL, CREATE_ALWAYS,
		FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "Save states failed (CreateFile)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	memset( head, 0, sizeof(head) );
	head[0] = 'E';
	head[1] = 'M';
	head[2] = 'U';
	head[3] = 'Z';
	head[4] = 1;	/* Type:    0=All, 1=Windows only, 2=Linux/UNIX only */
	head[5] = 0;	/* reserved */
	head[6] = 2;	/* Sub Version: 0<=0.8f, 1>=0.8g, 2>=0.92 */
	head[7] = 0;	/* Version:     0=First */
	WriteFile( hFile, head, sizeof(head), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(head)) {
		CloseHandle( hFile );
		return -2;
	}
	/* structure Z80 */
	val = sizeof( z80 );
	WriteFile( hFile, (void *)&val, sizeof(val), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, (void *)&z80, sizeof(Z80), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(Z80)) {
		CloseHandle( hFile );
		return -2;
	}
	/* structure MZ */
	val = sizeof( MZ2000 );
	WriteFile( hFile, (void *)&val, sizeof(val), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, (void *)mz, sizeof(MZ2000), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(MZ2000)) {
		CloseHandle( hFile );
		return -2;
	}
	/* structure MZIOInfo */
	val = sizeof( MZIOInfo );
	WriteFile( hFile, (void *)&val, sizeof(val), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, (void *)mz->ioinfo, sizeof(MZIOInfo), &dwWriteSize, NULL );
	if (dwWriteSize != sizeof(MZIOInfo)) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->memory, MZ_MAINMEMORY_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_MAINMEMORY_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->vram, MZ_TEXTVRAM_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->gram1, MZ_GRAMPAGE_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->gram2, MZ_GRAMPAGE_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->gram3, MZ_GRAMPAGE_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->dvram, MZ_TEXTVRAM_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		return -2;
	}
	WriteFile( hFile, mz->dgram, MZ_GRAMPAGE_SIZE, &dwWriteSize, NULL );
	if (dwWriteSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		return -2;
	}

	/* Chunks for DLL work area */
	memset( &info, 0, sizeof(info) );
	info[0] = 'I';
	info[1] = 'N';
	info[2] = 'F';
	info[3] = 'O';
	for (i = 0; i < PLUGIN_BOARD_MAXNUM; i++) {
		DWORD size;
		int ver;
		char *buf, *ident;

		ident = NULL;
		buf = NULL;
		if (plugin_get_boardstateinfo( i, &ident, &ver, &size, &buf ) && ident && size > 0 && buf) {
			*((int *)(&info[4]))   = ver;	/* ver */
			*((int *)(&info[12]))  = 0;
			*((DWORD *)(&info[8])) = size;	/* size */
			strncpy( &info[16], ident, PLUGIN_GUIDLENGTH_MAXNUM );
			info[16 + PLUGIN_GUIDLENGTH_MAXNUM - 1] = '\0';
			WriteFile( hFile, info, sizeof(info), &dwWriteSize, NULL );
			if (dwWriteSize != sizeof(info)) {
				CloseHandle( hFile );
				return -2;
			}
			WriteFile( hFile, buf, size, &dwWriteSize, NULL );
			if (dwWriteSize != size) {
				CloseHandle( hFile );
				return -2;
			}
		}
	}
	*((int *)(&info[4])) = 0;	/* ver */
	*((int *)(&info[12]))  = 0;
	*((DWORD *)(&info[8])) = 0;	/* size (DATA END) */
	WriteFile( hFile, info, 16, &dwWriteSize, NULL );
	if (dwWriteSize != 16) {
		CloseHandle( hFile );
		return -2;
	}
	CloseHandle( hFile );
	return 0;
}

static int load_state_main( MZ2000 *mz, const char *file )
{
	HANDLE hFile;
	DWORD dwReadSize;
	Z80 tz80;
	MZ2000 tmz;
	unsigned char head[16], thead[16];
	unsigned char info[16], tinfo[16];
	unsigned int val;
	unsigned int ver, subver;

	hFile = CreateFile( file,
		GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, NULL );
	if (hFile == INVALID_HANDLE_VALUE) {
		MessageBox( NULL, "Load states failed (CreateFile)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	memset( head, 0, sizeof(head) );
	head[0] = 'E';
	head[1] = 'M';
	head[2] = 'U';
	head[3] = 'Z';
	ReadFile( hFile, thead, sizeof(head), &dwReadSize, NULL );
	if (dwReadSize != sizeof(head) || strncmp( thead, head, 4 )) {
		CloseHandle( hFile );
		MessageBox( NULL, "Load states failed (Eyecache error)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	if (thead[5] || !thead[4]) {	/* Type == 0 */
		CloseHandle( hFile );
		MessageBox( NULL, "Load states failed (Architecture error, common type not supported)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	if (thead[5] || thead[4] != 1) {	/* Not type 1 */
		CloseHandle( hFile );
		MessageBox( NULL, "Load states failed (Architecture error, no Windows)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	ver    = thead[7];
	subver = thead[6];
	if (ver != 0 || subver >= 3)	 {	/* >= Version 0.3 */
		CloseHandle( hFile );
		MessageBox( NULL, "Load states failed (Unsupported version, >=0.3)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		return -1;
	}
	/* structure Z80 */
	ReadFile( hFile, (void *)&val, sizeof(val), &dwReadSize, NULL );
	if (dwReadSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	if (val != sizeof(z80)) {
		CloseHandle( hFile );
		return -3;
	}
	ReadFile( hFile, (void *)&tz80, sizeof(Z80), &dwReadSize, NULL );
	if (dwReadSize != sizeof(Z80)) {
		CloseHandle( hFile );
		return -2;
	}
	/* structure MZ */
	ReadFile( hFile, (void *)&val, sizeof(val), &dwReadSize, NULL );
	if (dwReadSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	if (val != sizeof(MZ2000)) {
		CloseHandle( hFile );
		return -3;
	}
	ReadFile( hFile, (void *)&tmz, sizeof(MZ2000), &dwReadSize, NULL );
	if (dwReadSize != sizeof(MZ2000)) {
		CloseHandle( hFile );
		return -2;
	}
	/* structure MZIOInfo - I */
	ReadFile( hFile, (void *)&val, sizeof(val), &dwReadSize, NULL );
	if (dwReadSize != sizeof(val)) {
		CloseHandle( hFile );
		return -2;
	}
	if (val != sizeof(MZIOInfo)) {
		CloseHandle( hFile );
		return -3;
	}
	/* update the system structures */
	mz -> mzmode = tmz . mzmode;
	reset_mz2000_sub( mz );
	reinit_io( mz );
	memcpy( (void *)&z80.a, (void *)&tz80.a, (u32)&tz80.left_clock - (u32)&tz80.a );
	memcpy( (void *)&mz->inipl, (void *)&tmz.inipl, (u32)&tmz.playStop - (u32)&tmz.inipl );
	memcpy( (void *)&mz->cpu_peripherals_start, (void *)&tmz.cpu_peripherals_start,
		(u32)&tmz.cpu_peripherals_end - (u32)&tmz.cpu_peripherals_start );
	if (mz -> keyint_func)
		mz -> keyint_func = mz2io_keyintfunc;
	memcpy( (void *)&mz->displayMode, (void *)&tmz.displayMode, ((u32)&tmz.screen_dummy - (u32)&tmz.displayMode) );
	if (!ver && !subver) {
		/* Version 0.0: force set the 8253 parameters */
		mz -> timer8253_count[0] = 0x7a12;
		mz -> timer8253_count[1] = 0xa8c0;
		mz -> timer8253_count[2] = 0x0000;
		mz -> timer8253_fast_mode = 0;
	}
	if (!ver && subver <= 1) {
		/* Version 0.1: force set the peripherals initialize */
		mz -> cpu_peripherals_start = CPUPERIPHERALS_INIT;
	} else {
		/* Restore the peripherals status */
		mz -> cpu_peripherals_start = tmz . cpu_peripherals_start;
	}

	/* structure MZIOInfo - II */
	ReadFile( hFile, (void *)mz->ioinfo, sizeof(MZIOInfo), &dwReadSize, NULL );
	if (dwReadSize != sizeof(MZIOInfo)) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->memory, MZ_MAINMEMORY_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_MAINMEMORY_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->vram, MZ_TEXTVRAM_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram1, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram2, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->gram3, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->dvram, MZ_TEXTVRAM_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_TEXTVRAM_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}
	ReadFile( hFile, mz->dgram, MZ_GRAMPAGE_SIZE, &dwReadSize, NULL );
	if (dwReadSize != MZ_GRAMPAGE_SIZE) {
		CloseHandle( hFile );
		mz -> needIpl = TRUE;
		stop_z80();
		return -2;
	}

	/* Chunks for DLL work area */
	memset( &info, 0, sizeof(info) );
	info[0] = 'I';
	info[1] = 'N';
	info[2] = 'F';
	info[3] = 'O';
	while (1) {
		DWORD size;
		int ver;
		char *buf, ident[PLUGIN_GUIDLENGTH_MAXNUM];

		ReadFile( hFile, tinfo, 16, &dwReadSize, NULL );
		if (dwReadSize != 16 || strncmp( tinfo, info, 4 )) {
			CloseHandle( hFile );
			mz -> needIpl = TRUE;
			stop_z80();
			return -2;
		}
		ver  = *((int *)(&tinfo[4]));
		size = *((DWORD *)(&info[8]));
		if (!size)
			break;
		ReadFile( hFile, ident, PLUGIN_GUIDLENGTH_MAXNUM, &dwReadSize, NULL );
		if (dwReadSize != PLUGIN_GUIDLENGTH_MAXNUM) {
			CloseHandle( hFile );
			mz -> needIpl = TRUE;
			stop_z80();
			return -2;
		}
		ident[PLUGIN_GUIDLENGTH_MAXNUM - 1] = '\0';
		buf = MALLOC( size );
		if (!buf) {
			CloseHandle( hFile );
			mz -> needIpl = TRUE;
			stop_z80();
			return -4;
		}
		ReadFile( hFile, buf, size, &dwReadSize, NULL );
		if (dwReadSize != size) {
			CloseHandle( hFile );
			mz -> needIpl = TRUE;
			stop_z80();
			return -2;
		}
		plugin_set_boardstateinfo( ident, ver, size, buf );
		FREE( buf );
	}
	CloseHandle( hFile );

	PaletteChange( mz );
	refresh_io( mz );
	screen_sethighreso( mz, mz -> screen_highreso );
	mz -> vramChg = mz -> gram1Chg = mz -> gram2Chg = mz -> gram3Chg = TRUE;
	return 0;
}

int save_state( HWND hWnd, MZ2000 *mz, int force_newname )
{
	int r;
	char *cp;
	char szFileName[MAX_PATH] = "";
	char szFileTitle[MAX_PATH] = "";
	OPENFILENAME ofn;

	setplaystop( mz, TRUE );
	if (mz -> howipl > 0) {
		MessageBox( NULL, "Save states failed (running IPL)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		setplaystop( mz, FALSE );
		return -1;
	}
	if (!(mz -> sfilename_havewrite) || force_newname) {
		strncpy( szFileName, (void *)mz -> sfilename, MAX_PATH );
		szFileName[MAX_PATH - 1] = '\0';
		memset(&ofn, 0, sizeof(OPENFILENAME));
		ofn.lStructSize = sizeof(OPENFILENAME);
		ofn.hwndOwner = hWnd;
		ofn.lpstrFilter = 
			"States save data (*.mze)\0*.mze\0"
			"All Files (*.*)\0*.*\0\0";
		ofn.lpstrFile = szFileName;
		ofn.lpstrFileTitle = szFileTitle;
		ofn.nMaxFile = MAX_PATH;
		ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
		cp = strrchr( szFileName, '.' );
		if (cp) {
			cp++;
			if (!stricmp( cp, "mze" )) {
				ofn.nFilterIndex = 1; 
				ofn.lpstrDefExt = "mze";
			} else if (!stricmp( cp, "emuz" )) {
				--cp;
				*cp = '\0';
				ofn.nFilterIndex = 1; 
				ofn.lpstrDefExt = "mze";
			} else {
				ofn.nFilterIndex = 2; 
				ofn.lpstrDefExt = "";
			}
		} else {
			ofn.nFilterIndex = 1; 
			ofn.lpstrDefExt = "mze";
		}
		ofn.lpstrTitle = "Select MZ states file";
		if (!GetSaveFileName(&ofn)) {
			setplaystop( mz, FALSE );
			return -1;
		}
		strncpy( (void *)mz -> sfilename, szFileName, MAX_PATH );
		mz -> sfilename[MAX_PATH - 1] = '\0';
		mz -> sfilename_havewrite = TRUE;
	}
	cp = strrchr( (void *)mz -> sfilename, '.' );
	if (cp && !stricmp( cp + 1, "emuz" )) {
		MessageBox( NULL, "Old 'emuz' states files are support importing only", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		setplaystop( mz, FALSE );
		return -1;
	}
	r = save_state_main( mz, (void *)mz -> sfilename );
	switch (r) {
	case 0 :	/* Normal End */
	case -1 :	/* Already MessageBox was displayed */
		break;
	case -2 :
		MessageBox( NULL, "Save states failed (WriteFile)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	default :
		if (r > 0)
			break;
		MessageBox( NULL, "Save states failed (Unknown error)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	}
	setplaystop( mz, FALSE );
	return 0;
}

int load_state( HWND hWnd, MZ2000 *mz, int no_fileopendlg )
{
	int r;
	char *cp, msgbuf[LNMAX];
	char szFileName[MAX_PATH] = "";
	char szFileTitle[MAX_PATH] = "";
	OPENFILENAME ofn;

	setplaystop( mz, TRUE );
	if (!no_fileopendlg) {
		strncpy( szFileName, (void *)mz -> sfilename, MAX_PATH );
		szFileName[MAX_PATH - 1] = '\0';
		memset(&ofn, 0, sizeof(OPENFILENAME));
		ofn.lStructSize = sizeof(OPENFILENAME);
		ofn.hwndOwner = hWnd;
		ofn.lpstrFilter = 
			"States save data (*.mze)\0*.mze\0"
#ifdef ENABLE_STATEEMUZ_IMPORT
			"old EmuZ-2000 states save data (*.emuz)\0*.emuz\0"
#endif
			"All Files (*.*)\0*.*\0\0";
		ofn.lpstrFile = szFileName;
		ofn.lpstrFileTitle = szFileTitle;
		ofn.nMaxFile = MAX_PATH;
		ofn.Flags = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
		cp = strrchr( szFileName, '.' );
		if (cp) {
			cp++;
			if (!stricmp( cp, "mze" )) {
				ofn.nFilterIndex = 1;
				ofn.lpstrDefExt = "mze";
#ifdef ENABLE_STATEEMUZ_IMPORT
			} else if (!stricmp( cp, "emuz" )) {
				ofn.nFilterIndex = 2;
				ofn.lpstrDefExt = "emuz";
#endif
			} else {
#ifdef ENABLE_STATEEMUZ_IMPORT
				ofn.nFilterIndex = 3;
#else
				ofn.nFilterIndex = 2;
#endif
				ofn.lpstrDefExt = "";
			}
		} else {
			ofn.nFilterIndex = 1; 
			ofn.lpstrDefExt = "mze";
		}
		ofn.lpstrTitle = "Select MZ states file";
		if (!GetOpenFileName(&ofn)) {
			setplaystop( mz, FALSE );
			return -1;
		}
		strncpy( (void *)mz -> sfilename, szFileName, MAX_PATH );
		mz -> sfilename[MAX_PATH - 1] = '\0';
	}
	mz -> sfilename_havewrite = FALSE;
#ifdef ENABLE_STATEEMUZ_IMPORT
	r = -9999;
	cp = strrchr( (void *)mz -> sfilename, '.' );
	if (cp)
		cp++;
	if (cp) {
		if (!stricmp( cp, "emuz" ))
			r = import_stateemuz_main( mz, (void *)mz -> sfilename );
		else
			r = load_state_main( mz, (void *)mz -> sfilename );
	}
#else
	r = load_state_main( mz, mz -> sfilename );
#endif
	switch (r) {
	case 0 :	/* Normal End */
	case -1 :	/* Already MessageBox was displayed */
		break;
	case -2 :
		MessageBox( NULL, "Load states failed (ReadFile)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	case -3 :
		MessageBox( NULL, "Load states failed (Structure size error)", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	case -4 :
		MessageBox( NULL, "Load states failed (Chunk malloc())", "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	default :
		if (r > 0)
			break;
		_snprintf( msgbuf, LNMAX, "Load states failed (Unknown error: %d)\n", r );
		msgbuf[LNMAX - 1] = '\0'; /* VC9 is not C99 */
		MessageBox( NULL, msgbuf, "States", MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL );
		break;
	}
	setplaystop( mz, FALSE );
	return 0;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
