/* File: wavefile.c

	SHARP MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Wave Files Accesses

	Copyright (C) 2001-2019 FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifndef WIN32
#include <unistd.h>
#else
#include <windows.h>
#include <io.h>
#endif

#include "common.h"
#include "wavedata.h"
#include "wavefile.h"

#ifdef TRACE
#undef TRACE
#endif

#ifndef WIN32
#define OPEN		open
#define CLOSE		close
#define READ		read
#define WRITE		write
#define LSEEK		lseek
#define UNLINK		unlink
#define OFF_T		off_t
#else
#define OPEN		_open
#define CLOSE		_close
#define READ		_read
#define WRITE		_write
#define LSEEK		_lseeki64
#define UNLINK		_unlink
#define FILELENGTH	_filelengthi64
#define OFF_T		_int64
#endif
#define TRACE		_TRACE

int wavefile_ropen( unsigned int attr, WAVEDATA_STRUCT *p, const char *fn )
{
	unsigned int vp[5], len;
	char rbuf[256];
	OFF_T size;
	WAVEFORMATEX wvfmt;
#ifndef WIN32
	int r;
#endif

	p -> attr = attr;
#ifndef WIN32
	p -> fd = OPEN( fn, O_RDONLY );
#else
	p -> fd = OPEN( fn, _O_RDONLY | _O_BINARY );
#endif
	if (p -> fd < 0)
		return WAVEFILE_RESULT_OPENFAIL;
#ifndef WIN32
	size = lseek( p -> fd, 0, SEEK_END );
	if (size == -1) {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_IOERROR;
	}
	r = lseek( p -> fd, 0, SEEK_SET );
	if (r == -1) {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_IOERROR;
	}
#else
	size = FILELENGTH( p -> fd );
#endif
	if (size > 0xffffffff) {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_UNSUPPORTEDFMT;
	}
	if (READ( p -> fd, vp, sizeof(unsigned int) * 5 ) < 0) {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_IOERROR;
	}

	/* RIFF chunk */
	if (vp[0] != 'FFIR') {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_UNSUPPORTEDFMT;
	}
#if 0
	if (vp[1] + 8 != (unsigned long)size) {
		CLOSE(p -> fd);
		TRACE( "error 1 %u <> %lu\n", vp[1] + 8, (unsigned long)size );
		return WAVEFILE_RESULT_HEADERR_1;
	}
#endif

	/* WAVE chunk */
	if (vp[2] != 'EVAW') {
		CLOSE(p -> fd);
		TRACE( "error 2\n" );
		return WAVEFILE_RESULT_HEADERR_2;
	}

	/* WAVE format chunk */
	if (vp[3] != ' tmf') {
		CLOSE(p -> fd);
		TRACE( "error 3\n" );
		return WAVEFILE_RESULT_HEADERR_3;
	}
	len = vp[4];
	if (len <= 0 || len > 256) {
		CLOSE(p -> fd);
		TRACE( "error 4 %d\n", len );
		return WAVEFILE_RESULT_HEADERR_4;
	}
	memset( rbuf, 0, 256 );
	if (READ( p -> fd, &rbuf, len ) < 0) {
		CLOSE(p -> fd);
		TRACE( "error 5\n" );
		return WAVEFILE_RESULT_HEADERR_5;
	}
	memcpy( &wvfmt, rbuf, sizeof(WAVEFORMATEX) );
	if (wvfmt.wFormatTag != 1) {
		CLOSE(p -> fd);
		TRACE( "error u1\n" );
		return WAVEFILE_RESULT_UNSUPPORTEDFMT_1;
	}
	if (wvfmt.nChannels < 1 || wvfmt.nChannels > 2) {
		CLOSE(p -> fd);
		TRACE( "error u2\n" );
		return WAVEFILE_RESULT_UNSUPPORTEDFMT_2;
	}
	if (wvfmt.wBitsPerSample != 8 && wvfmt.wBitsPerSample != 16) {
		CLOSE(p -> fd);
		TRACE( "error u3\n" );
		return WAVEFILE_RESULT_UNSUPPORTEDFMT_3;
	}
	if (wvfmt.nBlockAlign / wvfmt.nChannels != wvfmt.wBitsPerSample / 8) {
		CLOSE(p -> fd);
		TRACE( "error 6\n" );
		return WAVEFILE_RESULT_HEADERR_6;
	}

	/* skip unknown header or search 'data' */
	while (1) {
		if (READ( p -> fd, vp, sizeof(unsigned int) * 2 ) < 1) {
			CLOSE(p -> fd);
			TRACE( "error 2\n" );
			return WAVEFILE_RESULT_IOERROR;
		}
		len = vp[1];
		if (vp[0] == 'atad')
			break;

		/* skip unknown header */
		vp[1] = 0;
		if (LSEEK( p -> fd, len, SEEK_CUR ) < 0) {
			CLOSE(p -> fd);
			return WAVEFILE_RESULT_IOERROR;
		}
	}
	
	/* WAVE data chunk */
	TRACE("f=%d, rate=%d, bit=%d, channel=%d\n",
		wvfmt.nSamplesPerSec, wvfmt.nAvgBytesPerSec,
		wvfmt.wBitsPerSample, wvfmt.nChannels );
	p -> ch = wvfmt.nChannels;
	p -> f = wvfmt.nSamplesPerSec;
	p -> bytes = wvfmt.wBitsPerSample / 8;
	p -> len = len / (p -> bytes * p -> ch);
	if (len % (p -> bytes * p -> ch) != 0) {
		CLOSE(p -> fd);
		return WAVEFILE_RESULT_DATALENERR;
	}
	return 0;
}

int wavefile_rclose( WAVEDATA_STRUCT *p )
{
	if (p -> fd < 0)
		return WAVEFILE_RESULT_IOERROR;
	CLOSE( p -> fd );
	return 0;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
