/* File: plugin.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Plugins Supports

	Unit:	MZ-1U01 (MZ-2000)
		MZ-8BK  (MZ-80B)
		The other system has the internal slots.

	Combine the EmuZ-2000 Board_SDK / emuz2000ioBoard.h, emuz2000Debug.h,
		and extend by FUKUI, Toshio and TSUNEOKA, Shinji.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "../MZ2000/config.h"

#include <windows.h>
#include <stdio.h>

#include "../MZ2000/common.h"
#include "../MZ2000/mz2000.h"
#include "../MZ2000/util.h"
#include "../z80/z80operand.h"
#include "plugin.h"
#include "plugin_ex.h"

#define MAX_BOARD	PLUGIN_BOARD_MAXNUM

static EMUZ2000SERVICES _services;
static EMUZ2000PLUGIN _plugin[MAX_BOARD];

static PEMUZ2000PLUGIN get_plugin(int id)
{
	PEMUZ2000PLUGIN plugin = NULL;

	if (id >= MAX_BOARD){
		return NULL;
	}
	plugin = &_plugin[id];
	if (plugin->hDll == NULL){
		return NULL;
	}
	return plugin;
}

VOID plugin_initialize_board(VOID)
{
	int id = 0;

	/* set service entry points */
	_services.emuz2000_getCpuMode = emuz2000_getCpuMode;
	_services.emuz2000_getCpuClock = emuz2000_getCpuClock;
	_services.emuz2000_getMainMemoryPointer = emuz2000_getMainMemoryPointer;
	_services.emuz2000_getTextVramPointer = emuz2000_getTextVramPointer;
	_services.emuz2000_getGraphicRamPointers = emuz2000_getGraphicRamPointers;
	_services.emuz2000_setViewChange = emuz2000_setViewChange;
	_services.emuz2000_sendKeyMessage = emuz2000_sendKeyMessage;
	_services.emuz2000_signalInterrupt = emuz2000_signalInterrupt;
	_services.emuz2000_inputPort = emuz2000_inputPort;
	_services.emuz2000_outputPort = emuz2000_outputPort;
	_services.emuz2000_readProfileDword = emuz2000_readProfileDword;
	_services.emuz2000_readProfileString = emuz2000_readProfileString;
	_services.emuz2000_writeProfileDword = emuz2000_writeProfileDword;
	_services.emuz2000_writeProfileString = emuz2000_writeProfileString;
	_services.emuz2000_getMemorySegmentInfo = emuz2000_getMemorySegmentInfo;
	_services.emuz2000_setMemorySegmentInfo = emuz2000_setMemorySegmentInfo;
	_services.emuz2000_ext_Sound_Create = emuz2000_ext_Sound_Create;
	_services.emuz2000_ext_Sound_Play = emuz2000_ext_Sound_Play;
	_services.emuz2000_ext_Sound_Stop = emuz2000_ext_Sound_Stop;
	/* extended services */
	_services.emuz2000_ex_getVersion = emuz2000_ex_getVersion;
	_services.emuz2000_ex_setIoEntries = emuz2000_ex_setIoEntries;

	/* init board information */
	memset(_plugin, 0, sizeof(_plugin));
	for(id = 0; id <= MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = &_plugin[id];
		char *file = app_defaultinifile;
		char key[MAX_KEYNAME];
		char dllName[MAX_PATH];
		sprintf(key, "%d", id);
		if (GetUserProfileString(INISECTIONNAME_PLUGIN, key, "", dllName, sizeof(dllName), file) > 0){
			HINSTANCE hDll = NULL;
			PLUGINBOARDINFORMATIONSERVICE lpfnBoardInformationService = NULL;
			PLUGININITIALIZEBOARDEX lpfnInitializeBoardEx = NULL;
			if ((hDll = LoadLibrary(dllName)) != NULL){
				lpfnBoardInformationService = (PLUGINBOARDINFORMATIONSERVICE)GetProcAddress(hDll, "BoardInfomationService");
				lpfnInitializeBoardEx = (PLUGININITIALIZEBOARDEX)GetProcAddress(hDll, "InitializeBoardEx");
				if ((lpfnBoardInformationService == NULL) || (lpfnInitializeBoardEx == NULL)){
					FreeLibrary(hDll);
					continue;
				}
				lpfnBoardInformationService(&(plugin->pebih), id);
				plugin->hDll = hDll;
				lpfnInitializeBoardEx(&_services);
			}
		}
	}

	plugin_reinitialize_board();
	plugin_report_cpuClock();
}

VOID plugin_reinitialize_board(VOID)
{
	PLUGINREINITIALIZEBOARD lpfnReInitializeBoard = NULL;
	int id;

	for(id = 0; id < MAX_BOARD ; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		int i;
		if (plugin == NULL){
			continue;
		}

		/* set io mapping */
		for (i = 0; i < plugin->pebih->ioMapper.nCount; i++){
			int port = plugin->pebih->ioMapper.ucStartPort + i;
			set_ioent(port, plugin->pebih->ioMapper.fInportFunc, plugin->pebih->ioMapper.fOutportFunc);
		}

		/* call ReInitializeBoard */
		lpfnReInitializeBoard = (PLUGINREINITIALIZEBOARD)GetProcAddress(plugin->hDll, "ReIntializeBoard");
		if (lpfnReInitializeBoard == NULL){
			continue;
		}
		lpfnReInitializeBoard();
	}
}

VOID plugin_finalize_board(VOID)
{
	PLUGINFINALIZEBOARD lpfnFinalizeBoard = NULL;
	int id = 0;

	for(id = 0; id < MAX_BOARD ; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnFinalizeBoard = (PLUGINFINALIZEBOARD)GetProcAddress(plugin->hDll, "FinalizeBoard");
		if (lpfnFinalizeBoard == NULL){
			continue;
		}
		lpfnFinalizeBoard();
		FreeLibrary(plugin->hDll);
	}
	memset(&_plugin, 0, sizeof(_plugin));
}

VOID plugin_report_cpuClock(VOID)
{
	PLUGINREPORTCPUCLOCK lpfnReportCpuClock = NULL;
	int id = 0;

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnReportCpuClock = (PLUGINREPORTCPUCLOCK)GetProcAddress(plugin->hDll, "ReportCpuClock");
		if (lpfnReportCpuClock == NULL){
			continue;
		}
		lpfnReportCpuClock(emuz2000_getCpuClock(id));
	}
}

VOID plugin_show_informationDialog(HWND hWnd, DWORD id)
{
	PLUGININFORMATIONDIALOG lpfnInformationDialog = NULL;
	PEMUZ2000PLUGIN plugin = get_plugin(id);

	if (plugin == NULL){
		return;
	}

	lpfnInformationDialog = (PLUGININFORMATIONDIALOG)GetProcAddress(plugin->hDll, "InfomationDialog");
	if (lpfnInformationDialog == NULL){
		return;
	}
	lpfnInformationDialog(hWnd);
}

PEMUZ2000EBIH plugin_get_boardInformation(DWORD id)
{
	PEMUZ2000PLUGIN plugin = get_plugin(id);

	if (plugin == NULL){
		return NULL;
	}
	return plugin->pebih;
}

BOOL plugin_sound_dataUpload(int id, DWORD size, PUCHAR buf)
{
	PLUGINSOUNDDATAUPLOAD lpfnSoundDataUpload = NULL;
	PEMUZ2000PLUGIN plugin = get_plugin(id);

	if(plugin == NULL){
		return FALSE;
	}
	lpfnSoundDataUpload = (PLUGINSOUNDDATAUPLOAD)GetProcAddress(plugin->hDll, "SoundDataUpload");
	if (lpfnSoundDataUpload == NULL){
		return FALSE;
	}
	return lpfnSoundDataUpload(size, buf);
}

BOOL plugin_get_boardstateinfo(int id, char **ident, int *ver, DWORD *size, PUCHAR *buf)
{
	PLUGINSTATESAVEOUTPUTDATAEX lpfnStateSaveOutputDataEx = NULL;
	PEMUZ2000PLUGIN plugin = get_plugin(id);

	if(plugin == NULL){
		return FALSE;
	}

	lpfnStateSaveOutputDataEx = (PLUGINSTATESAVEOUTPUTDATAEX)GetProcAddress(plugin->hDll, "StateSaveOutputDataEx");
	if(lpfnStateSaveOutputDataEx == NULL){
		return FALSE;
	}

	*ident = plugin->pebih->szGuidCode;
	return lpfnStateSaveOutputDataEx(ver, size, buf);
}

BOOL plugin_set_boardstateinfo(char *ident, int ver, DWORD size, PUCHAR buf)
{
	int id = 0;
	PLUGINSTATESAVEINPUTDATAEX lpStateSaveInputDataEx = NULL;

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if(plugin == NULL){
			continue;
		}
		if(StrICmp(ident, plugin->pebih->szGuidCode) != 0){
			continue;
		}

		lpStateSaveInputDataEx = (PLUGINSTATESAVEINPUTDATAEX)GetProcAddress(plugin->hDll, "StateSaveInputDataEx");
		if (lpStateSaveInputDataEx == NULL){
			return FALSE;
		}
		return lpStateSaveInputDataEx(ver, size, buf);
	}

	return FALSE;
}

char plugin_iplmsg[PLUGIN_IPLMSG_MAXNUM + 1];

BOOL plugin_device_getinterface( int type )
{
	int id = 0;
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	BOOL fExists = FALSE;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, 0, PLUGIN_DEVICE_VERB_GETINTERFACE, 0, 0, 0, (LPVOID)&fExists) == TRUE){
			return fExists;
		}
	}
	return FALSE;
}

char *plugin_device_getfnptr( int type, int drive )
{
	return NULL;
}

int plugin_device_getexist( int type, int drive )
{
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	int id = 0;
	BOOL fExist = FALSE;

	if (type != PLUGIN_DEVICETYPE_QD){
		return PLUGIN_DEVICESTATUS_NONE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_GETEXIST, 0, 0, 0, (LPVOID)&fExist) == TRUE){
			if(fExist == FALSE){
				return PLUGIN_DEVICESTATUS_NONE;
			}
			return PLUGIN_DEVICESTATUS_INSERT;
		}
	}
	return PLUGIN_DEVICESTATUS_NONE;
}

BOOL plugin_device_getwritable( int type, int drive )
{
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	int id = 0;
	BOOL fWritable = FALSE;

	if (type != PLUGIN_DEVICETYPE_QD){
		return fWritable;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_GETWRITABLE, 0, 0, 0, (LPVOID)&fWritable) == TRUE){
			return fWritable;
		}
	}
	return fWritable;
}

BOOL plugin_device_getaccess( int type, int drive )
{
	int id = 0;
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	BOOL fBusy = FALSE;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_GETACCESS, 0, 0, 0, (LPVOID)&fBusy) == TRUE){
			return fBusy;
		}
	}
	return FALSE;
}

BOOL plugin_device_getmodified( int type, int drive )
{
	if (type != PLUGIN_DEVICETYPE_QD)
		return FALSE;
	/* QD don't use the function */
	return FALSE;
}

BOOL plugin_device_getrunningstatus( int type, int drive )
{
	int id = 0;
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	BOOL fBusy = FALSE;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_GETRUNNINGSTATUS, 0, 0, 0, (LPVOID)&fBusy) == TRUE){
			return fBusy;
		}
	}
	return FALSE;
}

BOOL plugin_device_writeback( int type, int drive )
{
	if (type != PLUGIN_DEVICETYPE_QD)
		return FALSE;
	/* QD don't use the function */
	return FALSE;
}

BOOL plugin_device_eject( int type, int drive, DISK_EJECT_TYPE mode )
{
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	int id = 0;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_EJECT, mode, 0, 0, NULL) == TRUE){
			return TRUE;
		}
	}
	return FALSE;
}

BOOL plugin_device_change( int type, int drive, const char *filename, int ftype, int nomsg, int readonly )
{
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	int id = 0;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_CHANGE, 0, (DWORD)ftype, (DWORD)nomsg, (LPVOID)filename) == TRUE){
			return TRUE;
		}
	}
	return FALSE;
}

/* Windows: filename = NULL, ftype = 0, readonly=FALSE only */
BOOL plugin_device_exec( int type, int drive, const char *filename, int ftype, int readonly )
{
	PSTRAGEDEVICECALLBACK lpfnStrageDeviceCallback = NULL;
	int id = 0;

	if (type != PLUGIN_DEVICETYPE_QD){
		return FALSE;
	}

	for(id = 0; id < MAX_BOARD; id++){
		PEMUZ2000PLUGIN plugin = get_plugin(id);
		if (plugin == NULL){
			continue;
		}
		lpfnStrageDeviceCallback = (PSTRAGEDEVICECALLBACK)GetProcAddress(plugin->hDll, "StrageDeviceCallback");
		if(lpfnStrageDeviceCallback == NULL){
			continue;
		}
		if(lpfnStrageDeviceCallback(type, drive, PLUGIN_DEVICE_VERB_EXEC, 0, (DWORD)ftype, (DWORD)readonly, (LPVOID)filename) == TRUE){
			return TRUE;
		}
	}
	return FALSE;
}

/* EmuZ2000 Services */

int WINAPI emuz2000_ex_getVersion(DWORD id)	/* TF-EDITION only service */
{
	if (!mz2000_global)
		return -1;
	return EMZ2000_VERSION_VALUE & 0xff;	/* current version 8 bit only used */ 
}

VOID WINAPI emuz2000_ex_setIoEntries(DWORD id, PMAPIOSTRUCT pIoMapper)	/* TF-EDITION only service */
{
	int i;

	for (i = 0; i < pIoMapper->nCount; i++){
		int port = pIoMapper->ucStartPort + i;
		set_ioent(port, pIoMapper->fInportFunc, pIoMapper->fOutportFunc);
	}
}

short WINAPI emuz2000_getCpuMode(DWORD id)
{
	if (!mz2000_global)
		return -1;
	return mz2000_global -> mzmode;		/* FALSE=2000 / TRUE=80B */
}

int WINAPI emuz2000_getCpuClock(DWORD id)
{
	if (!mz2000_global)
		return -1;
	return mz2000_global -> nowZ80Clock; /* KHz */
}

PVOID WINAPI emuz2000_getMainMemoryPointer(DWORD id)
{
	if (!mz2000_global)
		return NULL;
	return mz2000_global -> memory;
}

PVOID WINAPI emuz2000_getTextVramPointer(DWORD id)
{
	if (!mz2000_global)
		return NULL;
	return mz2000_global -> vram;
}

VOID WINAPI emuz2000_getGraphicRamPointers(DWORD id, PVOID g1, PVOID g2, PVOID g3)
{
	if (!mz2000_global)
		return;
	*(void **)g1 = mz2000_global -> gram1;
	*(void **)g2 = mz2000_global -> gram2;
	*(void **)g3 = mz2000_global -> gram3;
	return;
}

VOID WINAPI emuz2000_setViewChange(DWORD id)
{
	screen_region_add_all( mz2000_global );
	return;
}

VOID WINAPI emuz2000_sendKeyMessage(DWORD id, UCHAR keyCode, UCHAR flg)
{
	/* No compatible for Linux. Require ? What is this ? */
	return;
}

VOID WINAPI emuz2000_signalInterrupt(DWORD id, UCHAR code)
{
	/* What is this ? TRUE ? */
	int_z80( code );
	return;
}

UCHAR WINAPI emuz2000_inputPort(DWORD id, USHORT port)
{
	return ioread( port );
}

VOID WINAPI emuz2000_outputPort(DWORD id, USHORT port, UCHAR data)
{
	iowrite( port, data );
	return;
}

DWORD WINAPI emuz2000_readProfileDword(DWORD id, LPCTSTR lpKey)
{
	PEMUZ2000PLUGIN plugin = get_plugin(id);
	char *file = app_defaultinifile;
	char buff[32];

	if(plugin == NULL){
		return 0;
	}
	GetUserProfileString(plugin->pebih->szGuidCode, lpKey, "", buff, sizeof(buff), file);
	return (DWORD)atol(buff);
}

DWORD WINAPI emuz2000_readProfileString(DWORD id, LPCTSTR lpKey, LPTSTR	lpGetBuf, DWORD nGetMaxSize)
{
	PEMUZ2000PLUGIN plugin = get_plugin(id);
	char *file = app_defaultinifile;

	if(plugin == NULL){
		return 0;
	}
	return GetUserProfileString(plugin->pebih->szGuidCode, lpKey, "", lpGetBuf, nGetMaxSize, file);
}

BOOL WINAPI emuz2000_writeProfileDword(DWORD id, LPCTSTR lpKey, DWORD value)
{
	PEMUZ2000PLUGIN plugin = get_plugin(id);
	char *file = app_defaultinifile;
	char buff[32];

	if(plugin == NULL){
		return FALSE;
	}
	sprintf(buff, "%d", value);
	if(WriteUserProfileString(plugin->pebih->szGuidCode, lpKey, buff, file) != 0){
		return FALSE;
	}
	return TRUE;
}

BOOL WINAPI emuz2000_writeProfileString(DWORD id, LPCTSTR lpKey, LPCTSTR lpString)
{
	PEMUZ2000PLUGIN plugin = get_plugin(id);
	char *file = app_defaultinifile;

	if(plugin == NULL){
		return FALSE;
	}
	if(WriteUserProfileString(plugin->pebih->szGuidCode, lpKey, lpString, file) != 0){
		return FALSE;
	}
	return TRUE;
}

BOOL WINAPI emuz2000_getMemorySegmentInfo(DWORD id, PMAPMEMORYSTRUCT pmms)
{
	u8 *ptr;
	u8 (*readf)(u16 addr);
	int (*writef)(u16 addr, u8 value);

	get_mement( pmms -> seg, &ptr, &readf, &writef );
	pmms -> ptr    = ptr;
	pmms -> readf  = readf;
	pmms -> writef = writef;
	return TRUE;
}

BOOL WINAPI emuz2000_setMemorySegmentInfo(DWORD id, PMAPMEMORYSTRUCT pmms)
{
	return set_mement( pmms -> seg, pmms -> ptr, pmms -> readf, pmms -> writef );
}

BOOL WINAPI emuz2000_ext_Sound_Create(DWORD id, DWORD no)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_ext_Sound_Play(DWORD id, DWORD no)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_ext_Sound_Stop(DWORD id, DWORD no)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setPcBreadPoint(DWORD id, USHORT adr)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setMemoryWriteBreak(DWORD id, USHORT adr)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setMemoryReadBreak(DWORD id, USHORT adr)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setIoWriteBreak(DWORD id, USHORT port)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setIoReadBreak(DWORD id, USHORT port)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_emunBreadPoints(DWORD id, DWORD mode, PBRKINFO info)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_clearBreakPoint(DWORD id, DWORD mode, USHORT adr)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_getCpuInformation(DWORD id, PCPUINFO info)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_setInternalBreak(DWORD id, USHORT adr)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

BOOL WINAPI emuz2000_dbg_playControl(DWORD id, DWORD control)
{
	return FALSE;	/* No support in the original EmuZ-2000 */
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
