/* File: z80.c

	SHARP MZ-2000/2200/80B/B2 Emulator "EmuZ-2000"
		Z-80 Main Routines

	Combine the emz2000 z80.c, EmuZ-2000 z80.c,
		and extend by FUKUI, Toshio.
	This file is part of the EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#include "../mz2000/config.h"

#include <windows.h>
#include <stdio.h>

#include "../mz2000/common.h"
#include "../mz2000/mz2000.h"
#include "z80.h"
#include "z80operand.h"

#undef TRACE

#ifdef TRACE	/* windows ver., no use z80.trace */
static int traceflag = FALSE;
static int bpenable = FALSE;
static u16 bpoint = 0x00f0;
#endif

Z80 z80;
char is_peven[256];
int daa_array[2048];
static HANDLE hMutex = NULL;

static void pthread_mutex_init( PULONG exchange, PULONG ptr )
{
	hMutex = CreateMutex( NULL, FALSE, "EmuZ_2000_Z80_Core_Mutex" );
	*exchange = 0;
	return;
}

static void pthread_mutex_lock( PULONG exchange )
{
	WaitForSingleObject( hMutex, INFINITE );
	InterlockedExchange( exchange, 1 );
	return;
}

static void pthread_mutex_unlock(PULONG exchange)
{
	InterlockedExchange( exchange, 0 );
	ReleaseMutex( hMutex );
	return;
}

static void pthread_cond_init( HANDLE *hHandle, PLONG ptr )
{
	*hHandle = CreateSemaphore( NULL, 0, 0x7fffffff, NULL );
	return;
}

static void pthread_cond_wait( HANDLE *hHandle, PULONG exchange )
{
	pthread_mutex_unlock( exchange );
	WaitForSingleObject( *hHandle, INFINITE );
	pthread_mutex_lock( exchange );
	return;
}

static void pthread_cond_signal( HANDLE *hHandle )
{
	ReleaseSemaphore( *hHandle, 1, NULL );
	return;
}

int z80_timer( void )
{
	pthread_mutex_lock( &(LONG)(z80.timer_mutex) );
	if (z80.timer_wait) {
		z80.timer_wait = 0;
		pthread_cond_signal( &(HANDLE)(z80.timer_cond) );
	}
	pthread_mutex_unlock( &(LONG)(z80.timer_mutex) );
	return 1;
}

int z80_init( int speed_khz )
{
	int i;

	memset( (void *)&z80, 0, sizeof(z80) );
	for (i = 0; i < 16; i++) {
		z80.mem[i].ptr = NULL;
		z80.mem[i].readf = NULL;
		z80.mem[i].writef = NULL;
	}
	for (i = 0; i < 256; i++) {
		z80.io[i].readf = NULL;
		z80.io[i].writef = NULL;
	}
#ifdef IO16BIT
	for (i = 256; i < 65536; i++) {
		z80.io[i].readf = NULL;
		z80.io[i].writef = NULL;
	}
#endif
	z80.a = z80.b = z80.c = z80.d = z80.e = z80.f = z80.h = z80.l = 0;
	z80.a2 = z80.b2 = z80.c2 = z80.d2 = z80.e2 = z80.f2 = z80.h2 = z80.l2 = 0;
	z80.ix = z80.iy = z80.sp = 0;
	z80.internal_clock = 0;
	z80.left_clock = 0;
	setspeed_z80(speed_khz);
	z80.onbus = 0;
#ifdef TRACE
	z80.trace = 0;
#endif

	for (i = 0; i < 256; i++) {
		int j,n = 0;
		for (j = 0; j < 8; j++) {
			if (((i >> j) & 1) == 1) n++;
		}
		is_peven[i] = ((n == 0)||(n == 2)||(n == 4)||(n == 6)||(n == 8));;
	}

	for (i = 0; i < 2048; i++) {
		int v = i & 0xff;
		int ih = (i >> 4) & 0x0f;
		int il = i & 0x0f;
		switch (i>>8) {
		case 0:
			if ((ih <= 8)&&(il >= 10)) v += 0x06;
			else if ((ih >= 10)&&(il <= 9)) v += 0x60;
			else if ((ih >= 9)&&(il >= 10)) v += 0x66;
			break;
		case 1:	/* H */
			if ((ih <= 9)&&(il <= 3)) v += 0x06;
			else if ((ih >= 10)&&(il <= 3)) v += 0x66;
			break;
		case 2:	/* C */
			if ((ih <= 2)&&(il <= 9)) v = (v + 0x60) | 0x0100;
			else if ((ih <= 2)&&(il >= 10)) v = (v + 0x66) | 0x0100;
			break;
		case 3: /* H,C */
			if ((ih <= 3)&&(il >= 3)) v = (v + 0x66) | 0x100;
			break;
		case 5:	/* H,N */
			if ((ih <= 8)&&(il >= 6)) v -= 0x06;
			if (ih > 9)
				v += 0x9a;
			break;
		case 6:	/* C,N */
			if ((ih >= 7)&&(il <= 9)) v += 0xa0;
			break;
		case 7:	/* H,C,N */
			if ((ih >= 6)&&(il >= 6)) v += 0x9a;
			break;
		}
		daa_array[i] = v;
	}
	z80.instructs[GROUP_XX] = instr_xx;
	z80.instructs[GROUP_CB] = instr_cb;
	z80.instructs[GROUP_ED] = instr_ed;
	z80.instructs[GROUP_DD] = instr_dd;
	z80.instructs[GROUP_FD] = instr_fd;
	z80.instructs[GROUP_DDCB] = instr_ddcb;
	z80.instructs[GROUP_FDCB] = instr_fdcb;

	z80.timer_wait = 0;
	pthread_mutex_init( &(LONG)(z80.timer_mutex), NULL );
	pthread_cond_init( &(HANDLE)(z80.timer_cond), NULL );
	return 1;
}

void z80_exit( void )
{
	if (z80.timer_cond) {
		CloseHandle(z80.timer_cond);
		z80.timer_cond = NULL;
	}
	if (hMutex) {
		CloseHandle(hMutex);
		hMutex = NULL;
	}
	return;
}

static volatile int main_z80( int enable_halt_exit )
{
	int r, clk;

	r = TRUE;
	pthread_mutex_lock( &(LONG)(z80.z80_mutex) );
	while (z80.left_clock > 0) {
		if (z80.halt_state && enable_halt_exit)
			z80.stop = TRUE;
		if (z80.stop) {
			r = FALSE;
			break;
		}
		if (z80.external_signal) {
			z80.halt_state = 0;
			clk = exec_interrupt(z80.onbus);
			z80.internal_clock += clk;
			z80.left_clock -= clk;
		}
		if (z80.speed_khz && z80.left_clock > z80.speed_khz) {
			z80.left_clock = 0;
			break;
		}
		clk = z80.halt_state ? 4 : exec_instruction(GROUP_XX);
		z80.internal_clock += clk;
		z80.left_clock -= clk;
		//z80.left_clock = z80.left_clock >= clk ? z80.left_clock - clk : 0;
#ifdef TRACE
		if (traceflag)
			print_registers();
		if (bpenable && z80.pc == bpoint)
			print_registers();
#endif
	}
	pthread_mutex_unlock( &(LONG)(z80.z80_mutex) );
	return r;
}

int run_z80_2( int enable_halt_exit )
{
	/* reset_z80(); */
	z80.stop = 0;
	while (1) {
		if (!z80.speed_khz)
			z80.left_clock += DEFAULT_SPEED_KHZ * Z80_TIMER_MSEC;
		else {
			pthread_mutex_lock( &(LONG)(z80.timer_mutex) );
			z80.timer_wait = 1;
			pthread_cond_wait( &(HANDLE)(z80.timer_cond), &(LONG)(z80.timer_mutex) );
			z80.timer_wait = 0;
			pthread_mutex_unlock( &(LONG)(z80.timer_mutex) );
			z80.left_clock += z80.speed_khz;
		}
		if (!main_z80( enable_halt_exit ))
			break;
	}
	return 1;
}

int run_z80( void )
{
	return run_z80_2( FALSE );
}

int reset_z80( void )
{
	pthread_mutex_lock( &(LONG)(z80.z80_mutex) );
	z80.onbus = 0;
	z80.external_signal |= RESET_SIGNAL;
	pthread_mutex_unlock( &(LONG)(z80.z80_mutex) );
	return 1;
}

int int_z80( int onbus )
{
	if (z80.external_signal & INT_SIGNAL)
		return FALSE;
	z80.onbus = onbus;
	z80.external_signal |= INT_SIGNAL;
	return TRUE;
}

int nmi_z80( void )
{
	pthread_mutex_lock( &(LONG)(z80.z80_mutex) );
	z80.onbus = 0;
	z80.external_signal |= NMI_SIGNAL;
	pthread_mutex_unlock( &(LONG)(z80.z80_mutex) );
	return 1;
}

int stop_z80( void )
{
	if (!z80.stop) {
		z80.stop = 1;
		Sleep( 100 );
	}
	return 1;
}

z80clk get_internal_clock( void )
{
	return z80.internal_clock;
}

int setspeed_z80( int speed_khz )
{
	int n;

	if (!mz2000_global) {
		z80.speed_khz = speed_khz * Z80_TIMER_MSEC;
		return 1;
	}
	// Force add for sound
	n = z80.speed_khz;
	if (!speed_khz)
		z80.speed_khz = 0;
	else {
		z80.speed_khz = speed_khz * Z80_TIMER_MSEC + 
			(mz2000_global -> enable_soundtminterpolations ?
			mz2000_global -> frameclock_alpha : 0);
	}
	if (n != z80.speed_khz)
		z80.left_clock = 0;
	return 1;
}

int getspeed_z80( void )
{
	if (!mz2000_global)
		return z80.speed_khz / Z80_TIMER_MSEC;
	// Force add for sound
	if (!z80.speed_khz)
		return 0;
	else
		return (z80.speed_khz - (mz2000_global -> enable_soundtminterpolations ?
			mz2000_global -> frameclock_alpha : 0)) / Z80_TIMER_MSEC;
}

z80clk get_frameclk_z80( void )
{
	return (z80clk)z80.speed_khz;
}

int set_ioent( int port,
		u8 (*readf)(u16 port),
		int (*writef)(u16 port, u8 value) )
{
#ifndef IO16BIT
	port &= 0xff;
#endif
	z80.io[port].readf = readf;
	z80.io[port].writef = writef;
	return 1;
}

int get_ioent( int port,
		u8 (**readf)(u16 port),
		int (**writef)(u16 port, u8 value) )
{
#ifndef IO16BIT
	port &= 0xff;
#endif
	if (readf)
		*readf  = z80.io[port].readf;
	if (writef)
		*writef = z80.io[port].writef;
	return 1;
}

int set_mement( int segment, u8 *ptr,
		u8 (*readf)(u16 addr),
		int (*writef)(u16 addr,u8 value) )
{
	segment &= 0xf;
	z80.mem[segment].ptr = ptr;
	z80.mem[segment].readf = readf;
	z80.mem[segment].writef = writef;
	return 1;
}

int get_mement( int segment, u8 **ptr,
		u8 (**readf)(u16 addr),
		int (**writef)(u16 addr,u8 value) )
{
	segment &= 0xf;
	if (ptr)
		*ptr    = z80.mem[segment].ptr;
	if (readf)
		*readf  = z80.mem[segment].readf;
	if (writef)
		*writef = z80.mem[segment].writef;
	return 1;
}

#ifdef TRACE
void trace_on( void )
{
	z80.trace = 1;
}

void trace_off( void )
{
	z80.trace = 0;
}
#endif

void print_registers( void )
{
	char buf[256] ;
	OutputDebugString("A  B  C  D  E  H  L  F   IX   IY   SP   PC   Clock\n");
	sprintf(buf,"%02x %02x %02x %02x %02x %02x %02x %02x %04x %04x %04x %04x %d\n", 
	       z80.a, z80.b, z80.c, z80.d, z80.e, z80.h, z80.l, z80.f,
	       z80.ix, z80.iy, z80.sp, z80.pc, z80.internal_clock);
	OutputDebugString(buf);
	sprintf(buf,"%02x %02x %02x %02x %02x %02x %02x %02x\n", 
	       z80.a2, z80.b2, z80.c2, z80.d2, z80.e2, z80.h2, z80.l2, z80.f2);
	OutputDebugString(buf);
	sprintf(buf,"Memory: %04x %02x %02x %02x %02x %02x ...\n",
	       z80.pc, memread8(z80.pc), memread8(z80.pc+1), memread8(z80.pc+2),
	       memread8(z80.pc+3), memread8(z80.pc+4));
	OutputDebugString(buf);

	sprintf(buf,"Stack: %04x %04x %04x %04x %04x ...\n",
	       memread16(z80.sp), memread16(z80.sp+2), memread8(z80.sp+4),
	       memread8(z80.sp+6), memread8(z80.sp+8));
	OutputDebugString(buf);
	Sleep(10);
	return;
}

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
