/* File: z80operand.h

	SHARP MZ-2000/2200/80B/B2 Emulator "emz2000 / EmuZ-2000"
		Z-80 Operands Modules Headers

	Combine the emz2000 z80operand.h, EmuZ-2000 z80operand.h,
		and modified by FUKUI, Toshio.
	This file is part of the emz2000 / EmuZ-2000 software.
	See copyright notice in the COPYING file.
*/

#ifndef __Z80OPERAND_H_INCLUDED
#define __Z80OPERAND_H_INCLUDED

#include "z80.h"

#ifdef WIN32
#pragma warning( disable : 4244 4761)
#endif

/* Z80 structure */
#define Z8 (z80)

/*
 * defines for reading and identifying registers
 */
/* 8bits registers */
#define readA() (Z8.a)
#define readB() (Z8.b)
#define readC() (Z8.c)
#define readD() (Z8.d)
#define readE() (Z8.e)
#define readF() (Z8.f)
#define readH() (Z8.h)
#define readL() (Z8.l)
#define readA2() (Z8.a2)
#define readB2() (Z8.b2)
#define readC2() (Z8.c2)
#define readD2() (Z8.d2)
#define readE2() (Z8.e2)
#define readF2() (Z8.f2)
#define readH2() (Z8.h2)
#define readL2() (Z8.l2)

/* 16bits registers */
#define readIX() (Z8.ix)
#define readIY() (Z8.iy)
#define readPC() (Z8.pc)
#define readSP() (Z8.sp)
#define readAF() ((readA() << 8) + readF())
#define readBC() ((readB() << 8) + readC())
#define readDE() ((readD() << 8) + readE())
#define readHL() ((readH() << 8) + readL())
#define readAF2() ((readA2() << 8) + readF2())
#define readBC2() ((readB2() << 8) + readC2())
#define readDE2() ((readD2() << 8) + readE2())
#define readHL2() ((readH2() << 8) + readL2())

/* 8bits special registers */
#define readI() (Z8.i)
#define readR() (Z8.r)
#define readIXL() (readIX() & 0xff)
#define readIXH() ((readIX() >> 8)&0xff)
#define readIYL() (readIY() & 0xff)
#define readIYH() ((readIY() >> 8)&0xff)

/* general use */
#define readReg(r) (read ## r())

/*
 * defines for writing registers
 */
/* 8bits registers */
#define wrtA(v) (Z8.a = (v))
#define wrtB(v) (Z8.b = (v))
#define wrtC(v) (Z8.c = (v))
#define wrtD(v) (Z8.d = (v))
#define wrtE(v) (Z8.e = (v))
#define wrtF(v) (Z8.f = (v))
#define wrtH(v) (Z8.h = (v))
#define wrtL(v) (Z8.l = (v))
#define wrtA2(v) (Z8.a2 = (v))
#define wrtB2(v) (Z8.b2 = (v))
#define wrtC2(v) (Z8.c2 = (v))
#define wrtD2(v) (Z8.d2 = (v))
#define wrtE2(v) (Z8.e2 = (v))
#define wrtF2(v) (Z8.f2 = (v))
#define wrtH2(v) (Z8.h2 = (v))
#define wrtL2(v) (Z8.l2 = (v))

/* 16bits registers */
#define wrtIX(v) (Z8.ix = (v))
#define wrtIY(v) (Z8.iy = (v))
#define wrtPC(v) (Z8.pc = (v))
#define wrtSP(v) (Z8.sp = (v))
#define wrtAF(v) wr=(v),Z8.a = (wr >> 8),Z8.f = (wr & 0xff)
#define wrtBC(v) wr=(v),Z8.b = (wr >> 8),Z8.c = (wr & 0xff)
#define wrtDE(v) wr=(v),Z8.d = (wr >> 8),Z8.e = (wr & 0xff)
#define wrtHL(v) wr=(v),Z8.h = (wr >> 8),Z8.l = (wr & 0xff)
#define wrtAF2(v) wr=(v),Z8.a2 = (wr >> 8),Z8.f2 = (wr & 0xff)
#define wrtBC2(v) wr=(v),Z8.b2 = (wr >> 8),Z8.c2 = (wr & 0xff)
#define wrtDE2(v) wr=(v),Z8.d2 = (wr >> 8),Z8.e2 = (wr & 0xff)
#define wrtHL2(v) wr=(v),Z8.h2 = (wr >> 8),Z8.l2 = (wr & 0xff)

/* 8bits special registers */
#define wrtI(v) (Z8.i = (u8)(v))
#define wrtR(v) (Z8.r = (u8)(v))
#define wrtIXL(v) (Z8.ix = (readIX() & 0xff00) + (v))
#define wrtIXH(v) (Z8.ix = ((v) << 8) + (readIX() & 0xff))
#define wrtIYL(v) (Z8.iy = (readIY() & 0xff00) + (v))
#define wrtIYH(v) (Z8.iy = ((v) << 8) + (readIY() & 0xff))

/* general use */
#define wrtReg(r,v) (wrt ## r (v))

/*
 * defines for flags
 */
/* set flags */
#define setFlgS()  (wrtReg(F,readReg(F)|0x80))
#define setFlgZ()  (wrtReg(F,readReg(F)|0x40))
#define setFlg5()  (wrtReg(F,readReg(F)|0x20))
#define setFlgH()  (wrtReg(F,readReg(F)|0x10))
#define setFlg3()  (wrtReg(F,readReg(F)|0x08))
#define setFlgPV() (wrtReg(F,readReg(F)|0x04))
#define setFlgN()  (wrtReg(F,readReg(F)|0x02))
#define setFlgC()  (wrtReg(F,readReg(F)|0x01))

/* reset flags */
#define resFlgS()  (wrtReg(F,readReg(F)&0x7f))
#define resFlgZ()  (wrtReg(F,readReg(F)&0xbf))
#define resFlg5()  (wrtReg(F,readReg(F)&0xdf))
#define resFlgH()  (wrtReg(F,readReg(F)&0xef))
#define resFlg3()  (wrtReg(F,readReg(F)&0xf7))
#define resFlgPV() (wrtReg(F,readReg(F)&0xfb))
#define resFlgN()  (wrtReg(F,readReg(F)&0xfd))
#define resFlgC()  (wrtReg(F,readReg(F)&0xfe))

/* bit flags */
#define bitFlgS()  (readReg(F)&0x80)
#define bitFlgZ()  (readReg(F)&0x40)
#define bitFlg5()  (readReg(F)&0x20)
#define bitFlgH()  (readReg(F)&0x10)
#define bitFlg3()  (readReg(F)&0x08)
#define bitFlgPV() (readReg(F)&0x04)
#define bitFlgN()  (readReg(F)&0x02)
#define bitFlgC()  (readReg(F)&0x01)

/* general use */
#define setFlg(f) (setFlg ## f ())
#define resFlg(f) (resFlg ## f ())
#define bitFlg(f) (bitFlg ## f ())
#define conFlg(f,cond) ((cond)? setFlg(f) : resFlg(f))
#define valFlgSZ53P(v) (wrtReg(F, (readReg(F)& 0x13) | ((v)& 0xa8) | (((v)==0)?0x40:0x00) | ((is_peven[v])?0x04:0x00)))
#define valFlgSZ53(v) (wrtReg(F, (readReg(F)& 0x17) | ((v)& 0xa8) | (((v)==0)?0x40:0x00)))
#define valFlgS53(v) (wrtReg(F, (readReg(F)& 0x57) | ((v)& 0xa8)))
#define valFlg53(v) (wrtReg(F, (readReg(F)& 0xd7) | ((v)& 0x28)))

/*
 * defines for system values
 */
#define setIFF1() (Z8.iff1 = 1)
#define setIFF2() (Z8.iff2 = 1)
#define resIFF1() (Z8.iff1 = 0)
#define resIFF2() (Z8.iff2 = 0)
#define bitIFF1() (Z8.iff1)
#define bitIFF2() (Z8.iff2)
#define conIFF1(c) ((c)? setIFF1():resIFF1())
#define conIFF2(c) ((c)? setIFF2():resIFF2())
#define copyIFF1toIFF2() (Z8.iff2 = Z8.iff1)
#define copyIFF2toIFF1() (Z8.iff1 = Z8.iff2)
#define readIM() (Z8.im)
#define wrtIM(v) (Z8.im = (v))
#define setHALT() (Z8.halt_state = 1)
#define resHALT() (Z8.halt_state = 0)

/*
 *  increment / decrement register or memory
 */
u8 t ;
#define incRegL(r) t=readReg(r),t++,wrtReg(r,t)
#define decRegL(r) t=readReg(r),t--,wrtReg(r,t)
#define incReg(r) (wrtReg(r,readReg(r)+1))
#define decReg(r) (wrtReg(r,readReg(r)-1))
#define addReg(r,v) (wrtReg(r,readReg(r)+v))
#define subReg(r,v) (wrtReg(r,readReg(r)-v))
#define incmem(a) (memwrite8(a,memread8(a)+1))
#define decmem(a) (memwrite8(a,memread8(a)-1))

/*
 *  set / reset bit of register or memory
 */
#define setReg(r,d) (wrtReg(r,readReg(r)|(1<<(d))))
#define resReg(r,d) (wrtReg(r,readReg(r)&(~(1<<(d)))))
#define setmem(a,d) (memwrite8(a,memread8(a)|(1<<(d))))
#define resmem(a,d) (memwrite8(a,memread8(a)&(~(1<<(d)))))

/*
 *  PC registers
 */
#define fetchPC(v) {v=memread8(readReg(PC));incReg(PC);}
#define fetch2PC(v) {v=memread16(readReg(PC));addReg(PC,2);}
//#define fetchPC(v) v=memread8(readReg(PC)),incReg(PC)
//#define fetch2PC(v) v=memread16(readReg(PC)),addReg(PC,2)

/*
 *  LD instructions
 */
int idx_temp;
/* LD reg, reg */
#define LD_R8_R8(r1,r2) (wrtReg(r1,readReg(r2)))
#define LD_R16_R16(r1,r2) (wrtReg(r1,readReg(r2)))
/* LD reg, n */
#define LD_R8_N(r1,n2) (wrtReg(r1,(n2)))
#define LD_R16_N(r1,n2) (wrtReg(r1,(n2)))
/* LD reg, (reg) */
#define LD_R8_RM(r1,r2) (wrtReg(r1,memread8(readReg(r2))))
#define LD_R16_RM(r1,r2) (wrtReg(r1,memread16(readReg(r2))))
/* LD reg, (reg+d) */
#define LD_R8_IDX(r1,r2,d) /*(wrtReg(r1,memread8(readReg(r2)+(d))))*/{\
	if( d < 0x80 ) (wrtReg(r1,memread8(readReg(r2)+(d))));\
	else idx_temp = ((0xff-d)+1),(wrtReg(r1,memread8(readReg(r2)-idx_temp)));\
}
/* LD reg, (nn) */
#define LD_R8_M(r1,a2) (wrtReg(r1,memread8(a2)))
#define LD_R16_M(r1,a2) (wrtReg(r1,memread16(a2)))
/* LD (nn), reg */
#define LD_M_R8(a1,r2) (memwrite8(a1,readReg(r2)))
#define LD_M_R16(a1,r2) (memwrite16(a1,readReg(r2)))
/* LD (reg), reg */
#define LD_RM_R8(r1,r2) (memwrite8(readReg(r1),readReg(r2)))
#define LD_RM_R16(r1,r2) (memwrite16(readReg(r1),readReg(r2)))
/* LD (reg), n */
#define LD_RM_N(r1,n2) (memwrite8(readReg(r1),(n2)))
/* LD (reg+d), reg */
#define LD_IDX_R8(r1,d,r2) /*(memwrite8(readReg(r1)+(d),readReg(r2)))*/{\
	if( d < 0x80 ) (memwrite8(readReg(r1)+(d),readReg(r2)));\
	else idx_temp = ((0xff-d)+1),(memwrite8(readReg(r1)-idx_temp,readReg(r2)));\
}
/* LD (reg+d), n */
#define LD_IDX_N(r1,d,n2) /*(memwrite8(readReg(r1)+(d),(n2)))*/{\
	if( d < 0x80 )	(memwrite8(readReg(r1)+(d),(n2)));\
	else idx_temp = ((0xff-d)+1),(memwrite8(readReg(r1)-idx_temp,(n2)));\
}

#define LD_R8_RI(r1,r2) {\
	int v = LD_R8_R8(r1,r2);\
	valFlgSZ53(v);\
	resFlg(H); resFlg(N);\
	conFlg(PV,bitIFF2());\
}

/*
 *  EX instructions
 */
#define EX_R8_R8(r1,r2) {int t=readReg(r1);wrtReg(r1,readReg(r2));wrtReg(r2,t);}
#define EX_R16_R16(r1,r2) {int t=readReg(r1);wrtReg(r1,readReg(r2));wrtReg(r2,t);}
#define EX_RM_R8(r1,r2) {int a=readReg(r1),t=memread16(a);memwrite16(a,readReg(r2));wrtReg(r2,t);}
#define EX_RM_R16(r1,r2) {int a=readReg(r1),t=memread16(a);memwrite16(a,readReg(r2));wrtReg(r2,t);}
#define EXX() {EX_R16_R16(BC,BC2);EX_R16_R16(DE,DE2);EX_R16_R16(HL,HL2);}

/*
 *  PUSH/POP instructions
 */
#define PUSH_R16(r) {subReg(SP,2);memwrite16(readReg(SP),readReg(r));}
#define POP_R16(r) {wrtReg(r,memread16(readReg(SP)));addReg(SP,2);}

/*
 *  ADD/ADC instructions
 */
#define ADD8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)+(v2);\
	int ts=((s8)(v1))+((s8)(v2));\
	valFlgSZ53(t&0xff);\
	conFlg(H,(((v1)&0xf)+((v2)&0xf)>0xf));\
	conFlg(PV,(ts>0x7f)||(ts<-0x80));\
	resFlg(N);\
	conFlg(C,t>0xff);\
	wrtReg(r1,t&0xff);\
}
#define ADC8(r1,v2) {\
	int v1=readReg(r1);\
	int v3=(bitFlg(C))?1:0;\
	int t=(v1)+(v2)+(v3);\
	int ts=((s8)(v1))+((s8)(v2))+(v3);\
	valFlgSZ53(t&0xff);\
	conFlg(H,(((v1)&0xf)+((v2)&0xf)+(v3)>0xf));\
	conFlg(PV,(ts>0x7f)||(ts<-0x80));\
	resFlg(N);\
	conFlg(C,t>0xff);\
	wrtReg(r1,t&0xff);\
}
#define ADD16(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)+(v2);\
	valFlg53((t>>8)&0xff);\
	conFlg(H,(((v1)&0xfff)+((v2)&0xfff)>0xfff));\
	resFlg(N);\
	conFlg(C,t>0xffff);\
	wrtReg(r1,t&0xffff);\
}
#define ADC16(r1,v2) {\
	int v1=readReg(r1);\
	int v3=(bitFlg(C))?1:0;\
	int t=(v1)+(v2)+(v3);\
	int ts=((s16)(v1))+((s16)(v2))+(v3);\
	valFlgS53((t>>8)&0xff);\
	conFlg(Z,(t&0xffff)==0);\
	conFlg(H,(((v1)&0xfff)+((v2)&0xfff)+(v3)>0xfff));\
	conFlg(PV,(ts>0x7fff)||(ts<-0x8000));\
	resFlg(N);\
	conFlg(C,t>0xffff);\
	wrtReg(r1,t&0xffff);\
}
#define ADD_R8_R8(r1,r2) {ADD8(r1,readReg(r2));}
#define ADD_R8_N(r1,n2)  {ADD8(r1,(n2));}
#define ADD_R8_RM(r1,r2) {ADD8(r1,memread8(readReg(r2)));}
//#define ADD_R8_IDX(r1,r2,d) {ADD8(r1,memread8(readReg(r2)+(d)));}
#define ADD_R8_IDX(r1,r2,d) {\
	if(d<0x80) { ADD8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); ADD8(r1,memread8(readReg(r2)-idx_temp));}\
}

#define ADD_R16_R16(r1,r2) {ADD16(r1,readReg(r2));}
#define ADC_R8_R8(r1,r2) {ADC8(r1,readReg(r2));}
#define ADC_R8_N(r1,n2)  {ADC8(r1,(n2));}
#define ADC_R8_RM(r1,r2) {ADC8(r1,memread8(readReg(r2)));}
//#define ADC_R8_IDX(r1,r2,d) {ADC8(r1,memread8(readReg(r2)+(d)));}
#define ADC_R8_IDX(r1,r2,d) {\
	if(d<0x80) { ADC8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); ADC8(r1,memread8(readReg(r2)-idx_temp));}\
}

#define ADC_R16_R16(r1,r2) {ADC16(r1,readReg(r2));}

/*
 *  SUB/SBC instructions
 */
#define SUB8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)-(v2);\
	int ts=((s8)(v1))-((s8)(v2));\
	valFlgSZ53(t&0xff);\
	conFlg(H,(((v1)&0xf)-((v2)&0xf)<0));\
	conFlg(PV,(ts>0x7f)||(ts<-0x80));\
	setFlg(N);\
	conFlg(C,t<0);\
	wrtReg(r1,t&0xff);\
}
#define SBC8(r1,v2) {\
	int v1=readReg(r1);\
	int v3=(bitFlg(C))?1:0;\
	int t=(v1)-(v2)-(v3);\
	int ts=((s8)(v1))-((s8)(v2))-(v3);\
	valFlgSZ53(t&0xff);\
	conFlg(H,(((v1)&0xf)-((v2)&0xf)-(v3)<0));\
	conFlg(PV,(ts>0x7f)||(ts<-0x80));\
	setFlg(N);\
	conFlg(C,t<0);\
	wrtReg(r1,t&0xff);\
}
#define SBC16(r1,v2) {\
	int v1=readReg(r1);\
	int v3=(bitFlg(C))?1:0;\
	int t=(v1)-(v2)-(v3);\
	int ts=((s16)(v1))-((s16)(v2))-(v3);\
	valFlgS53((t>>8)&0xff);\
	conFlg(Z,(t&0xffff)==0);\
	conFlg(H,(((v1)&0xfff)-((v2)&0xfff)-(v3)<0));\
	conFlg(PV,(ts>0x7fff)||(ts<-0x8000));\
	setFlg(N);\
	conFlg(C,t<0);\
	wrtReg(r1,t&0xffff);\
}
#define SUB_R8_R8(r1,r2) {SUB8(r1,readReg(r2));}
#define SUB_R8_N(r1,n2)  {SUB8(r1,(n2));}
#define SUB_R8_RM(r1,r2) {SUB8(r1,memread8(readReg(r2)));}
//#define SUB_R8_IDX(r1,r2,d) {SUB8(r1,memread8(readReg(r2)+(d)));}
#define SUB_R8_IDX(r1,r2,d) {\
	if(d<0x80) {SUB8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); SUB8(r1,memread8(readReg(r2)-idx_temp))}\
}

#define SBC_R8_R8(r1,r2) {SBC8(r1,readReg(r2));}
#define SBC_R8_N(r1,n2)  {SBC8(r1,(n2));}
#define SBC_R8_RM(r1,r2) {SBC8(r1,memread8(readReg(r2)));}
//#define SBC_R8_IDX(r1,r2,d) {SBC8(r1,memread8(readReg(r2)+(d)));}
#define SBC_R8_IDX(r1,r2,d) {\
	if(d<0x80) {SBC8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); SBC8(r1,memread8(readReg(r2)-idx_temp))}\
}


#define SBC_R16_R16(r1,r2) {SBC16(r1,readReg(r2));}
#define SUB_R8(r1) SUB_R8_R8(A,r1)
#define SUB_N(n1) SUB_R8_N(A,n1)
#define SUB_RM(r1) SUB_R8_RM(A,r1)
#define SUB_IDX(r1,d) SUB_R8_IDX(A,r1,d)

/*
 *  CP instructions
 */
#define CP8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)-(v2);\
	int ts=(s8)(((s8)(v1))-((s8)(v2)));\
	valFlg53(v2);\
	conFlg(S,t&0x80);\
	conFlg(Z,(t&0xff)==0);\
	conFlg(H,(((v1)&0xf)-((v2)&0xf)<0));\
	conFlg(PV,(ts>0x7f)||(ts<-0x80));\
	setFlg(N);\
	conFlg(C,t<0);\
}
#define CP_R8_R8(r1,r2) {CP8(r1,readReg(r2));}
#define CP_R8_N(r1,n2)  {CP8(r1,(n2));}
#define CP_R8_RM(r1,r2) {CP8(r1,memread8(readReg(r2)));}
//#define CP_R8_IDX(r1,r2,d) {CP8(r1,memread8(readReg(r2)+(d)));}
#define CP_R8_IDX(r1,r2,d) {\
	if(d<0x80) {CP8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); CP8(r1,memread8(readReg(r2)-idx_temp))}\
}


#define CP_R8(r1) CP_R8_R8(A,r1)
#define CP_N(n1) CP_R8_N(A,n1)
#define CP_RM(r1) CP_R8_RM(A,r1)
#define CP_IDX(r1,d) CP_R8_IDX(A,r1,d)

/*
 *  INC/DEC instructions
 */
#define INCflg(v) {\
	valFlgSZ53((v)&0xff);\
	conFlg(H,((v)&0xf)==0);\
	conFlg(PV,(v)==0x80);\
	resFlg(N);\
}
#define DECflg(v) {\
	valFlgSZ53((v)&0xff);\
	conFlg(H,((v)&0xf)==0xf);\
	conFlg(PV,(v)==0x7f);\
	setFlg(N);\
}
#define INC_R8(r1) {incReg(r1);INCflg(readReg(r1));}
#define INC_R8IL(r1) {incRegL(r1);INCflg(readReg(r1));}
#define INC_RM(r1) {int t=incmem(readReg(r1));INCflg(t);}
//#define INC_IDX(r1,d) {int t=incmem(readReg(r1)+d);INCflg(t);}
#define INC_IDX(r1,d) {\
	if(d<0x80) {int t=incmem(readReg(r1)+d);INCflg(t);}\
	else {int t = 0;idx_temp = ((0xff-d)+1);t=incmem(readReg(r1)-idx_temp);INCflg(t);}\
}


#define INC_R16(r1) {incReg(r1);}
#define DEC_R8(r1) {decReg(r1);DECflg(readReg(r1));}
#define DEC_R8IL(r1) {decRegL(r1);DECflg(readReg(r1));}
#define DEC_RM(r1) {int t=decmem(readReg(r1));DECflg(t);}
//#define DEC_IDX(r1,d) {int t=decmem(readReg(r1)+d);DECflg(t);}
#define DEC_IDX(r1,d) {\
	if(d<0x80) {int t=decmem(readReg(r1)+d);DECflg(t);}\
	else {int t = 0;idx_temp = ((0xff-d)+1);t=decmem(readReg(r1)-idx_temp);DECflg(t);}\
}


#define DEC_R16(r1) {decReg(r1);}

/*
 *  JP/JR/DJNZ instructions
 */
#define JP_N(a) {wrtReg(PC,(a));}
#define JP_R16(r) {wrtReg(PC,readReg(r));}
#define JP_NZ_N(a) {if (!bitFlg(Z)) JP_N(a);}
#define JP_Z_N(a) {if (bitFlg(Z)) JP_N(a);}
#define JP_NC_N(a) {if (!bitFlg(C)) JP_N(a);}
#define JP_C_N(a) {if (bitFlg(C)) JP_N(a);}
#define JP_PO_N(a) {if (!bitFlg(PV)) JP_N(a);}
#define JP_PE_N(a) {if (bitFlg(PV)) JP_N(a);}
#define JP_P_N(a) {if (!bitFlg(S)) JP_N(a);}
#define JP_M_N(a) {if (bitFlg(S)) JP_N(a);}
#define JP_FLG_N(f,a) {JP_ ## f ## _N(a);}

#define JR_N(a) {wrtReg(PC,readReg(PC)+(signed char)(a));}
#define JR_NZ_N(a) {if (!bitFlg(Z)) JR_N(a);}
#define JR_Z_N(a) {if (bitFlg(Z)) JR_N(a);}
#define JR_NC_N(a) {if (!bitFlg(C)) JR_N(a);}
#define JR_C_N(a) {if (bitFlg(C)) JR_N(a);}
#define JR_FLG_N(f,a) {JR_ ## f ## _N(a);}

#define DJNZ_N(a) {decReg(B);if (readReg(B)!=0) JR_N(a);}

/*
 *  CALL/RST/RET instructions
 */
#define CALL_N(a) {PUSH_R16(PC);JP_N(a);}
#define CALL_NZ_N(a) {if (!bitFlg(Z)) CALL_N(a);}
#define CALL_Z_N(a) {if (bitFlg(Z)) CALL_N(a);}
#define CALL_NC_N(a) {if (!bitFlg(C)) CALL_N(a);}
#define CALL_C_N(a) {if (bitFlg(C)) CALL_N(a);}
#define CALL_PO_N(a) {if (!bitFlg(PV)) CALL_N(a);}
#define CALL_PE_N(a) {if (bitFlg(PV)) CALL_N(a);}
#define CALL_P_N(a) {if (!bitFlg(S)) CALL_N(a);}
#define CALL_M_N(a) {if (bitFlg(S)) CALL_N(a);}
#define CALL_FLG_N(f,a) {CALL_ ## f ## _N(a);}
#define RST_N(a) CALL_N(a)

#define RET()  {POP_R16(PC);}
#define RET_NZ() {if (!bitFlg(Z)) RET();}
#define RET_Z() {if (bitFlg(Z)) RET();}
#define RET_NC() {if (!bitFlg(C)) RET();}
#define RET_C() {if (bitFlg(C)) RET();}
#define RET_PO() {if (!bitFlg(PV)) RET();}
#define RET_PE() {if (bitFlg(PV)) RET();}
#define RET_P() {if (!bitFlg(S)) RET();}
#define RET_M() {if (bitFlg(S)) RET();}
#define RET_FLG(f) {RET_ ## f ();}

#define RETI() {RET();}
#define RETN() {copyIFF2toIFF1();RET();}

/*
 *  AND/OR/XOR instructions
 */
#define AND8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)&(v2);\
	wrtReg(F,0x10);\
	valFlgSZ53P(t);\
	wrtReg(r1,t);\
}
#define OR8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)|(v2);\
	wrtReg(F,0);\
	valFlgSZ53P(t);\
	wrtReg(r1,t);\
}
#define XOR8(r1,v2) {\
	int v1=readReg(r1);\
	int t=(v1)^(v2);\
	wrtReg(F,0);\
	valFlgSZ53P(t);\
	wrtReg(r1,t);\
}
#define AND_R8_R8(r1,r2) {AND8(r1,readReg(r2));}
#define AND_R8_N(r1,n2) {AND8(r1,(n2));}
#define AND_R8_RM(r1,r2)  {AND8(r1,memread8(readReg(r2)));}
//#define AND_R8_IDX(r1,r2,d)  {AND8(r1,memread8(readReg(r2)+(d)));}
#define AND_R8_IDX(r1,r2,d)  {\
	if(d<0x80) {AND8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); AND8(r1,memread8(readReg(r2)-idx_temp));}\
}


#define OR_R8_R8(r1,r2) {OR8(r1,readReg(r2));}
#define OR_R8_N(r1,n2) {OR8(r1,(n2));}
#define OR_R8_RM(r1,r2)  {OR8(r1,memread8(readReg(r2)));}
//#define OR_R8_IDX(r1,r2,d)  {OR8(r1,memread8(readReg(r2)+(d)));}
#define OR_R8_IDX(r1,r2,d)  {\
	if(d<0x80) {OR8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); OR8(r1,memread8(readReg(r2)-idx_temp))}\
}


#define XOR_R8_R8(r1,r2) {XOR8(r1,readReg(r2));}
#define XOR_R8_N(r1,n2) {XOR8(r1,(n2));}
#define XOR_R8_RM(r1,r2)  {XOR8(r1,memread8(readReg(r2)));}
//#define XOR_R8_IDX(r1,r2,d)  {XOR8(r1,memread8(readReg(r2)+(d)));}
#define XOR_R8_IDX(r1,r2,d)  {\
	if(d<0x80) {XOR8(r1,memread8(readReg(r2)+(d)));}\
	else {idx_temp = ((0xff-d)+1); XOR8(r1,memread8(readReg(r2)-idx_temp));}\
}


#define AND_R8(r1) AND_R8_R8(A,r1)
#define AND_N(n1)  AND_R8_N(A,n1)
#define AND_RM(r1) AND_R8_RM(A,r1)
#define AND_IDX(r1,d) AND_R8_IDX(A,r1,d)
#define OR_R8(r1) OR_R8_R8(A,r1)
#define OR_N(n1)  OR_R8_N(A,n1)
#define OR_RM(r1) OR_R8_RM(A,r1)
#define OR_IDX(r1,d) OR_R8_IDX(A,r1,d)
#define XOR_R8(r1) XOR_R8_R8(A,r1)
#define XOR_N(n1)  XOR_R8_N(A,n1)
#define XOR_RM(r1) XOR_R8_RM(A,r1)
#define XOR_IDX(r1,d) XOR_R8_IDX(A,r1,d)

/*
 *  SET/RES/BIT instructions
 */
#define SET_N_R8(n,r1) {setReg(r1,(n));}
#define SET_N_RM(n,r1) {setmem(readReg(r1),(n));}
//#define SET_N_IDX(n,r1,d) {setmem(readReg(r1)+(d),(n));}
#define SET_N_IDX(n,r1,d) {\
	if(d<0x80) {setmem(readReg(r1)+(d),(n)); }\
	else {idx_temp = ((0xff-d)+1); setmem(readReg(r1)-idx_temp,(n));}\
}

//#define SET_LD_N_IDX(n,r1,d,r2) {int v=setmem(readReg(r1)+(d),(n));wrtReg(r2,v);}
#define SET_LD_N_IDX(n,r1,d,r2) {\
	if(d<0x80) {int v=setmem(readReg(r1)+(d),(n));wrtReg(r2,v); }\
	else{int v;idx_temp = ((0xff-d)+1); v=setmem(readReg(r1)-idx_temp,(n));wrtReg(r2,v);}\
}

#define RES_N_R8(n,r1) {resReg(r1,(n));}
#define RES_N_RM(n,r1) {resmem(readReg(r1),(n));}
//#define RES_N_IDX(n,r1,d) {resmem(readReg(r1)+(d),(n));}
#define RES_N_IDX(n,r1,d) {\
	if(d<0x80) { resmem(readReg(r1)+(d),(n));}\
	else{idx_temp = ((0xff-d)+1); resmem(readReg(r1)-idx_temp,(n));}\
}

//#define RES_LD_N_IDX(n,r1,d,r2) {int v=resmem(readReg(r1)+(d),(n));wrtReg(r2,v);}
#define RES_LD_N_IDX(n,r1,d,r2) {\
	if(d<0x80) { int v=resmem(readReg(r1)+(d),(n));wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=resmem(readReg(r1)-idx_temp,(n));wrtReg(r2,v);}\
}

#define BIT(v1,n) {\
	u8 t=((v1)&(1<<(n)));\
	conFlg(Z,(t==0));\
	conFlg(PV,(t==0));\
	conFlg(S,((n)==7)&&(t));\
	conFlg(5,((n)==5)&&(t));\
	conFlg(3,((n)==3)&&(t));\
	setFlg(H);resFlg(N);\
}
#define BIT_N_R8(n,r1) {BIT(readReg(r1),(n));}
#define BIT_N_RM(n,r1) {BIT(memread8(readReg(r1)),(n));}
//#define BIT_N_IDX(n,r1,d) {BIT(memread8(readReg(r1)+(d)),(n));}
#define BIT_N_IDX(n,r1,d) {\
	if(d<0x80) { BIT(memread8(readReg(r1)+(d)),(n));}\
	else{idx_temp = ((0xff-d)+1); BIT(memread8(readReg(r1)-idx_temp),(n));}\
}

/*
 *  rotate instructions
 */
#define RLCA() {\
	int v=readReg(A);\
	int t=(v)&0x80;\
	wrtReg(A,((v)<<1)|((t)?0x01:0));\
	wrtReg(F,readReg(F)&0xc4);\
	conFlg(C,t);\
	valFlg53(v);\
}
#define RLA() {\
	int v=readReg(A);\
	int t=(v)&0x80;\
	wrtReg(A,((v)<<1)|((bitFlg(C))?0x01:0));\
	wrtReg(F,readReg(F)&0xc4);\
	conFlg(C,t);\
	valFlg53(v);\
}
#define RRCA() {\
	int v=readReg(A);\
	int t=(v)&0x01;\
	wrtReg(A,((v)>>1)|((t)?0x80:0));\
	wrtReg(F,readReg(F)&0xc4);\
	conFlg(C,t);\
	valFlg53(v);\
}
#define RRA() {\
	int v=readReg(A);\
	int t=(v)&0x01;\
	wrtReg(A,((v)>>1)|((bitFlg(C))?0x80:0));\
	wrtReg(F,readReg(F)&0xc4);\
	conFlg(C,t);\
	valFlg53(v);\
}
#define RLC(v) {\
	int t=(v)&0x80;\
	(v)=((v)<<1)|((t)?0x01:0);\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define RL(v) {\
	int t=(v)&0x80;\
	(v)=((v)<<1)|((bitFlg(C))?0x01:0);\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define RRC(v) {\
	int t=(v)&0x01;\
	(v)=((v)>>1)|((t)?0x80:0);\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define RR(v) {\
	int t=(v)&0x01;\
	(v)=((v)>>1)|((bitFlg(C))?0x80:0);\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define RLC_R8(r) {int v=readReg(r);RLC(v);wrtReg(r,v);}
#define RLC_RM(r) {int v=memread8(readReg(r));RLC(v);memwrite8(readReg(r),v);}
//#define RLC_IDX(r,d) {int v=memread8(readReg(r)+(d));RLC(v);memwrite8(readReg(r)+(d),v);}
#define RLC_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));RLC(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);RLC(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define RLC_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));RLC(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define RLC_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));RLC(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);RLC(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define RL_R8(r) {int v=readReg(r);RL(v);wrtReg(r,v);}
#define RL_RM(r) {int v=memread8(readReg(r));RL(v);memwrite8(readReg(r),v);}
//#define RL_IDX(r,d) {int v=memread8(readReg(r)+(d));RL(v);memwrite8(readReg(r)+(d),v);}
#define RL_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));RL(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);RL(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define RL_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));RL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define RL_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));RL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);RL(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define RRC_R8(r) {int v=readReg(r);RRC(v);wrtReg(r,v);}
#define RRC_RM(r) {int v=memread8(readReg(r));RRC(v);memwrite8(readReg(r),v);}
//#define RRC_IDX(r,d) {int v=memread8(readReg(r)+(d));RRC(v);memwrite8(readReg(r)+(d),v);}
#define RRC_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));RRC(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);RRC(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define RRC_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));RRC(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define RRC_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));RRC(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);RRC(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define RR_R8(r) {int v=readReg(r);RR(v);wrtReg(r,v);}
#define RR_RM(r) {int v=memread8(readReg(r));RR(v);memwrite8(readReg(r),v);}
//#define RR_IDX(r,d) {int v=memread8(readReg(r)+(d));RR(v);memwrite8(readReg(r)+(d),v);}
#define RR_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));RR(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);RR(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define RR_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));RR(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define RR_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));RR(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)idx_temp);RR(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define RLD() {\
	int v2=memread8(readReg(HL));\
	int v1=readReg(A);\
	int t=((v1&0xf0) | (v2>>4));\
	wrtReg(A,t);\
	memwrite8(readReg(HL), (((v2&0x0f)<<4) | (v1&0x0f)));\
	valFlgSZ53P(t);\
	resFlg(H);resFlg(N);\
}
#define RRD() {\
	int v2=memread8(readReg(HL));\
	int v1=readReg(A);\
	int t=((v1&0xf0) | (v2&0x0f));\
	wrtReg(A,t);\
	memwrite8(readReg(HL), (((v1&0x0f)<<4) | (v2>>4)));\
	valFlgSZ53P(t);\
	resFlg(H);resFlg(N);\
}
/*
 *  shift instructions
 */
#define SLA(v) {\
	int t=(v)&0x80;\
	(v)=(v)<<1;\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define SLL(v) {\
	int t=(v)&0x80;\
	(v)=((v)<<1) | 1;\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define SRA(v) {\
	int t1=(v)&0x01;\
	int t2=(v)&0x80;\
	(v)=((v)>>1) | ((t2)?0x80:0);\
	wrtReg(F,0);\
	conFlg(C,t1);\
	valFlgSZ53P(v);\
}
#define SRL(v) {\
	int t=(v)&0x01;\
	(v)=(v)>>1;\
	wrtReg(F,0);\
	conFlg(C,t);\
	valFlgSZ53P(v);\
}
#define SLA_R8(r) {int v=readReg(r);SLA(v);wrtReg(r,v);}
#define SLA_RM(r) {int v=memread8(readReg(r));SLA(v);memwrite8(readReg(r),v);}
//#define SLA_IDX(r,d) {int v=memread8(readReg(r)+(d));SLA(v);memwrite8(readReg(r)+(d),v);}
#define SLA_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));SLA(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);SLA(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define SLA_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));SLA(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define SLA_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));SLA(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);SLA(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define SLL_R8(r) {int v=readReg(r);SLL(v);wrtReg(r,v);}
#define SLL_RM(r) {int v=memread8(readReg(r));SLL(v);memwrite8(readReg(r),v);}
//#define SLL_IDX(r,d) {int v=memread8(readReg(r)+(d));SLL(v);memwrite8(readReg(r)+(d),v);}
#define SLL_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));SLL(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);SLL(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define SLL_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));SLL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define SLL_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));SLL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)idx_temp);SLL(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define SRA_R8(r) {int v=readReg(r);SRA(v);wrtReg(r,v);}
#define SRA_RM(r) {int v=memread8(readReg(r));SRA(v);memwrite8(readReg(r),v);}
//#define SRA_IDX(r,d) {int v=memread8(readReg(r)+(d));SRA(v);memwrite8(readReg(r)+(d),v);}
#define SRA_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));SRA(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);SRA(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define SRA_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));SRA(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define SRA_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));SRA(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);SRA(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}

#define SRL_R8(r) {int v=readReg(r);SRL(v);wrtReg(r,v);}
#define SRL_RM(r) {int v=memread8(readReg(r));SRL(v);memwrite8(readReg(r),v);}
//#define SRL_IDX(r,d) {int v=memread8(readReg(r)+(d));SRL(v);memwrite8(readReg(r)+(d),v);}
#define SRL_IDX(r,d) {\
	if(d<0x80) { int v=memread8(readReg(r)+(d));SRL(v);memwrite8(readReg(r)+(d),v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r)-idx_temp);SRL(v);memwrite8(readReg(r)-idx_temp,v);}\
}

//#define SRL_LD_IDX(r1,d,r2) {int v=memread8(readReg(r1)+(d));SRL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}
#define SRL_LD_IDX(r1,d,r2) {\
	if(d<0x80) { int v=memread8(readReg(r1)+(d));SRL(v);memwrite8(readReg(r1)+(d),v);wrtReg(r2,v);}\
	else{int v;idx_temp = ((0xff-d)+1); v=memread8(readReg(r1)-idx_temp);SRL(v);memwrite8(readReg(r1)-idx_temp,v);wrtReg(r2,v);}\
}


/*
 * HALT/DI/EI/IM instructions
 */
#define HALT() {setHALT();}
#define DI() {resIFF1();resIFF2();}
#define EI() {setIFF1();setIFF2();}
#define IM_N(v) {wrtIM(v);}

/*
 * SCF/CCF instructions
 */
#define SCF() {valFlg53(readReg(A));resFlg(H);resFlg(N);setFlg(C);}
#define CCF() {valFlg53(readReg(A));conFlg(H,bitFlg(C));resFlg(N);conFlg(C,!bitFlg(C));}

/*
 * CPL/NEG instructions
 */
#define CPL_R8(r) {wrtReg(r,~readReg(r));valFlg53(readReg(r));setFlg(H);setFlg(N);}
#define NEG_R8(r) {wrtReg(r,-readReg(r));valFlgSZ53(readReg(r));conFlg(H,(readReg(r)&0x0f)!=0);conFlg(PV,readReg(r)==0x80);setFlg(N);conFlg(C,readReg(r)>0);}
#define CPL() CPL_R8(A)
#define NEG() NEG_R8(A)

/*
 *  LDI/LDD/LDIR/LDDR instructions
 */
#define LDI() {\
	int t=memwrite8(readReg(DE),memread8(readReg(HL)));\
	incReg(DE);\
	incReg(HL);\
	decReg(BC);\
	resFlg(H); resFlg(N);\
	conFlg(PV,readReg(BC)!=0);\
	conFlg(5,(t+readReg(A))&0x02);\
	conFlg(3,(t+readReg(A))&0x08);\
}
#define LDIR() {LDI();if(bitFlg(PV))subReg(PC,2);}
#define LDD() {\
	int t=memwrite8(readReg(DE),memread8(readReg(HL)));\
	decReg(DE);\
	decReg(HL);\
	decReg(BC);\
	resFlg(H); resFlg(N);\
	conFlg(PV,readReg(BC)!=0);\
	conFlg(5,(t+readReg(A))&0x02);\
	conFlg(3,(t+readReg(A))&0x08);\
}
#define LDDR() {LDD();if(bitFlg(PV))subReg(PC,2);}

/*
 *  CPI/CPD/CPIR/CPDR instructions
 */
#define CPI() {\
	int v1=readReg(A);\
	int v2=memread8(readReg(HL));\
	int t=(v1)-(v2);\
	incReg(HL);\
	decReg(BC);\
	conFlg(5,(t-readReg(H)) & 0x02);\
	conFlg(3,(t-readReg(H)) & 0x08);\
	conFlg(S,t&0x80);\
	conFlg(Z,(t&0xff)==0);\
	conFlg(H,(((v1)&0xf)-((v2)&0xf)<0));\
	conFlg(PV,(readReg(BC)!=0));\
	setFlg(N);\
}
#define CPIR() {CPI();if(!bitFlg(Z)&&bitFlg(PV))subReg(PC,2);}
#define CPD() {\
	int v1=readReg(A);\
	int v2=memread8(readReg(HL));\
	int t=(v1)-(v2);\
	decReg(HL);\
	decReg(BC);\
	conFlg(5,(t-readReg(H)) & 0x02);\
	conFlg(3,(t-readReg(H)) & 0x08);\
	conFlg(S,t&0x80);\
	conFlg(Z,(t&0xff)==0);\
	conFlg(H,(((v1)&0xf)-((v2)&0xf)<0));\
	conFlg(PV,(readReg(BC)!=0));\
	setFlg(N);\
}
#define CPDR() {CPD();if(!bitFlg(Z)&&bitFlg(PV))subReg(PC,2);}

/*
 *  OUT/IN instructions
 */
#define OUT_RM_R8(r1,r2) (iowrite(readReg(r1),readReg(r2)))
#define OUT_M_R8(n1,r2) (iowrite(n1+readReg(r2)*256,readReg(r2)))
#define OUT_RM_N(r1,n2) (iowrite(readReg(r1),n2))
#define IN_R8_M(r1,n2) (wrtReg(r1,ioread(n2+readReg(r1)*256)))
#define IN_R8_RM(r1,r2) {\
	int t=ioread(readReg(r2));\
	wrtReg(r1,t);\
	valFlgSZ53P(t);\
	resFlg(H); resFlg(N);\
}
#define IN_RM(r1) {\
	valFlgSZ53P(ioread(readReg(r1)));\
	resFlg(H); resFlg(N);\
}

/*
 *  OTDR/OTIR/OUTD/OUTI instructions
 */
#define OUTI() {\
	int v=memread8(readReg(HL));\
	iowrite(readReg(BC),v);\
	incReg(HL);\
	decReg(B);\
	valFlgSZ53(readReg(B));\
	conFlg(C,v+readReg(C)+1);\
	conFlg(H,((v)&0xf)+((readReg(C)+1)&0xf)>0xf);\
}
#define OTIR() {OUTI();if(!bitFlg(Z))subReg(PC,2);}
#define OUTD() {\
	int v=memread8(readReg(HL));\
	iowrite(readReg(BC),v);\
	decReg(HL);\
	decReg(B);\
	valFlgSZ53(readReg(B));\
	conFlg(C,v+readReg(C)-1);\
	conFlg(H,((v)&0xf)+((readReg(C)-1)&0xf)>0xf);\
}
#define OTDR() {OUTD();if(!bitFlg(Z))subReg(PC,2);}

/*
 *  INDR/INIR/IND/INI instructions
 */
#define INI() {\
	int v=ioread(readReg(BC));\
	memwrite8(readReg(HL),v);\
	incReg(HL);\
	decReg(B);\
	valFlgSZ53(readReg(B));\
	conFlg(C,v+readReg(C)+1);\
	conFlg(H,((v)&0xf)+((readReg(C)+1)&0xf)>0xf);\
}
#define INIR() {INI();if(!bitFlg(Z))subReg(PC,2);}
#define IND() {\
	int v=ioread(readReg(BC));\
	memwrite8(readReg(HL),v);\
	decReg(HL);\
	decReg(B);\
	valFlgSZ53(readReg(B));\
	conFlg(C,v+readReg(C)-1);\
	conFlg(H,((v)&0xf)+((readReg(C)-1)&0xf)>0xf);\
}
#define INDR() {IND();if(!bitFlg(Z))subReg(PC,2);}

/*
 *  DAA instructions
 */
#define DAA_R8(r1) {\
	int t=readReg(r1)+(bitFlg(H)?0x100:0)+(bitFlg(C)?0x200:0)+(bitFlg(N)?0x400:0);\
	int v=daa_array[t];\
	wrtReg(r1,v&0xff);\
	conFlg(C,v&0x100);\
	valFlgSZ53P(v&0xff);\
}
#define DAA() DAA_R8(A)

/*
 *  NOP instructions
 */
#define NOP() {}

extern char is_peven[256];
extern int daa_array[2048];

extern u8 memread8(u16 addr);
extern u16 memread16(u16 addr);
extern int memwrite8(u16 addr, u8 value);
extern int memwrite16(u16 addr, u16 value);
extern u8 ioread(u16 port);
extern int iowrite(u16 port, u8 value);

#define GROUP_XX 0
#define GROUP_CB 1
#define GROUP_ED 2
#define GROUP_DD 3
#define GROUP_FD 4
#define GROUP_DDCB 5
#define GROUP_FDCB 6

extern int (*(instr_xx[]))();
extern int (*(instr_cb[]))();
extern int (*(instr_ed[]))();
extern int (*(instr_dd[]))();
extern int (*(instr_fd[]))();
extern int (*(instr_ddcb[]))();
extern int (*(instr_fdcb[]))();

extern int exec_instruction(int group);
extern int exec_instruction2(int group);
extern int exec_interrupt(int onbus);

#endif /* __Z80OPERAND_H_INCLUDED */

/*
	Local Variables:
	mode:c++
	c-set-style:"k&r"
	c-basic-offset:8
	tab-width:8
	End:
*/
