#include <ctype.h>
#include <gtk/gtk.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

//#define WIN32_LEAN_AND_MEAN

#include "emu.h"

/* cycle_count_delta is a (usually negative) number telling what the time is relative
 * to the next timer interrupt that will be executed.
 * (cycle_count + cycle_count_delta) is the total number of cycles executed so far */
u64 cycle_count = 0;
int cycle_count_delta = 0;

int throttle_delay = 10; /* in milliseconds */

u32 cpu_events;

bool exiting;
bool do_translate = true;
bool emulate_cas;

bool emulate_ti84_keypad;
volatile u16 key_map[KEYPAD_ROWS];

bool turbo_mode;
bool show_speed;

jmp_buf restart_after_exception;

static FILE *boot2_file = NULL;
static char *boot2_filename = NULL;


const char log_type_tbl[] = LOG_TYPE_TBL;
int log_enabled[MAX_LOG];
FILE *log_file[MAX_LOG];
void logprintf(int type, char *str, ...) {
	if (log_enabled[type]) {
		va_list va;
		va_start(va, str);
		vfprintf(log_file[type], str, va);
		va_end(va);
	}
}

void warn(char *fmt, ...) {
	va_list va;
	fprintf(stderr, "Warning at PC=%08X: ", arm.reg[15]);
	va_start(va, fmt);
	vfprintf(stderr, fmt, va);
	va_end(va);
	fprintf(stderr, "\n");
}

__attribute__((noreturn))
void error(char *fmt, ...) {
	va_list va;
	fprintf(stderr, "Error at PC=%08X: ", arm.reg[15]);
	va_start(va, fmt);
	vfprintf(stderr, fmt, va);
	va_end(va);
	fprintf(stderr, "\n\tBacktrace:\n");
	backtrace(arm.reg[11]);
	debugger();
	exit(1);
}

void throttle_timer_on() {
	gui_calc_timer_on();
}
void throttle_timer_off() {
	gui_calc_timer_off();
}

int exec_hack() {
	u32 pc = arm.reg[15];
	//if (pc == 0x1194D0A8) {
	//	arm.reg[0] = 1000000;	// navnet log level
	//	return 0;
	//} else
	if (pc == usblink_addr_schedule) {
		usblink_hook_schedule();
		return 1;
	} else if (pc == usblink_addr_submit_read_buffer) {
		usblink_hook_submit_read_buffer();
		return 1;
	} else if (pc == usblink_addr_submit_write_buffer) {
		usblink_hook_submit_write_buffer();
		return 1;
	}
	return 0;
}

void prefetch_abort(u32 mva, u8 status) {
	logprintf(LOG_CPU, "Prefetch abort: address=%08x status=%02x\n", mva, status);
	arm.reg[15] += 4;
	// Fault address register not changed
	arm.instruction_fault_status = status;
	cpu_exception(EX_PREFETCH_ABORT);
	if (mva == arm.reg[15])
		error("Abort occurred with exception vectors unmapped");
	longjmp(restart_after_exception, 0);
}

void data_abort(u32 mva, u8 status) {
	logprintf(LOG_CPU, "Data abort: address=%08x status=%02x\n", mva, status);
	fix_pc_for_fault();
	arm.reg[15] += 8;
	arm.fault_address = mva;
	arm.data_fault_status = status;
	cpu_exception(EX_DATA_ABORT);
	longjmp(restart_after_exception, 0);
}

static void emu_reset() {
	memset(&arm, 0, sizeof arm);
	memset(arm.reg, 0x55, sizeof arm.reg);
	arm.cpsr_low28 = MODE_SVC | 0xC0;
	cpu_events &= ~(EVENT_IRQ | EVENT_RESET);
	if (boot2_file) {
		/* Start from BOOT2. (needs to be re-loaded on each reset since
		 * it can get overwritten in memory) */
		fseek(boot2_file, 0, SEEK_SET);
		fread(MEM_PTR(0x11800000), 1, 0x800000, boot2_file);
		arm.reg[15] = 0x11800000;
		if (*(u8 *)MEM_PTR(0x11800003) < 0xE0) {
			printf("%s does not appear to be an uncompressed BOOT2 image.\n", boot2_filename);
			return;
		}

		/* To enter maintenance mode (home+enter+P), address A4012ECC
		 * must contain an array indicating those keys before BOOT2 starts */
		memcpy(MEM_PTR(0xA4012ECC), (void *)key_map, 0x12);
	} else {
		/* Start from BOOT1. */
		arm.reg[15] = 0;
	}
	arm.control = 0x00050078;
	addr_cache_flush();
	flush_translations();

	setjmp(restart_after_exception);
}

void	emu_iterate(int count) {
	while (count-- > 0) {
		if (cycle_count_delta < 0) {
			if (cpu_events & EVENT_RESET) {
				printf("CPU reset\n");
				emu_reset();
				return;
			}

			if (cpu_events & EVENT_IRQ) {
				if (cpu_events & EVENT_WAITING)
					arm.reg[15] += 4; // Skip over wait instruction
				logprintf(LOG_IRQS, "Dispatching an IRQ\n");
				arm.reg[15] += 4;
				cpu_exception(EX_IRQ);
			}
			cpu_events &= ~EVENT_WAITING;

			if (arm.cpsr_low28 & 0x20)
				cpu_thumb_loop();
			else
				cpu_arm_loop();
			continue;
		}

		// Timer ticks are approx. 32 kHz
		cycle_count       += 2812;
		cycle_count_delta -= 2812;

		if (++timer[0].ticks > timer[0].ticks_per_count) {
			timer[0].ticks = 0;
			if (--timer[0].count < 0) {
				timer[0].count = (timer[0].counts_per_int - 1) & 0xFFFF;
				if (reg_900A0004 & 0x80) // just guessing here...
					irq_activate(1 << IRQ_TIMER1);
			}
		}

		if (++timer[1].ticks > timer[1].ticks_per_count) {
			timer[1].ticks = 0;
			if (--timer[1].count < 0) {
				timer[1].count = (timer[1].counts_per_int - 1) & 0xFFFF;
				irq_activate(1 << IRQ_TIMER2);
			}
		}

		static int interval_ticks;
		if (--interval_ticks < 0) {
			/* Throttle interval (defined arbitrarily as 100Hz) - used for
			 * keeping the emulator speed down, and other miscellaneous stuff
			 * that needs to be done periodically */
			static int intervals;
			intervals++;
			interval_ticks += 320;

			//if (_kbhit()) {
				//char c = _getch();
				//if (c == 4)
					//debugger();
				//else
					//serial_byte_in(c);
			//}


			//LARGE_INTEGER interval_end;
			//QueryPerformanceCounter(&interval_end);

			//{	// Update graphics (frame rate is arbitrary)
				//static LARGE_INTEGER prev;
				//LONGLONG time = interval_end.QuadPart - prev.QuadPart;
				//if (time >= (perffreq.QuadPart >> 5)) {
					//InvalidateRect(hwndMain, NULL, FALSE);
					//prev = interval_end;
				//}
			//}

			//if (show_speed) {
				//// Show speed
				//static int prev_intervals;
				//static LARGE_INTEGER prev;
				//LONGLONG time = interval_end.QuadPart - prev.QuadPart;
				//if (time >= (perffreq.QuadPart >> 1)) {
					//double speed = (double)perffreq.QuadPart * (intervals - prev_intervals) / time;
					//char buf[40];
					//sprintf(buf, "nspire_emu - %.1f%%", speed);
					//SetWindowText(hwndMain, buf);
					//prev_intervals = intervals;
					//prev = interval_end;
				//}
			//}

			//if (!turbo_mode)
				//WaitForSingleObject(hTimerEvent, INFINITE);

			//if (log_enabled[LOG_ICOUNT]) {
				//static LARGE_INTEGER interval_start;
				//LARGE_INTEGER next_start;
				//QueryPerformanceCounter(&next_start);

				//static u64 prev_cycles;
				//u64 cycles = cycle_count + cycle_count_delta;
				//logprintf(LOG_ICOUNT, "Time=%7d (CPU=%7d Idle=%7d) Insns=%8d Rate=%9I64d\n",
					//next_start.LowPart - interval_start.LowPart,
					//interval_end.LowPart - interval_start.LowPart,
					//next_start.LowPart - interval_end.LowPart,
					//(u32)(cycles - prev_cycles),
					//(cycles - prev_cycles) * perffreq.QuadPart / (interval_end.LowPart - interval_start.LowPart));
				//prev_cycles = cycles;
				//interval_start = next_start;
			//}
		}
	}
}


int main(int argc, char **argv) {
	int i;
	FILE *f;
	static char *boot1_filename = NULL, *flash_filename = NULL;
	bool new_flash_image = false;
	char *preload_boot2 = NULL, *preload_diags = NULL, *preload_os = NULL;

	gtk_init(&argc, &argv);

	for (i = 1; i < argc; i++) {
		char *arg = argv[i];
		if (*arg == '/' || *arg == '-') {
			arg++;
			switch (toupper(*arg++)) {
				case '1':
					if (*arg == '=') arg++;
					boot1_filename = arg;
					break;
				case 'B':
					if (*arg == '=') arg++;
					boot2_filename = arg;
					boot2_file = fopen(boot2_filename, "rb");
					if (!boot2_file) {
						perror(boot2_filename);
						return 1;
					}
					break;
				case 'C':
					if (*arg) goto usage;
					emulate_cas = true;
					break;
				case 'D':
					if (*arg) goto usage;
					cpu_events |= EVENT_DEBUG_STEP;
					break;
				case 'F':
					if (*arg == '=') arg++;
					flash_filename = arg;
					break;
				case 'K':
					if (*arg) goto usage;
					emulate_ti84_keypad = true;
					break;
				case 'L': {
					char *p = strchr(log_type_tbl, toupper(*arg++));
					int type = p - log_type_tbl;
					if (!p) goto usage;
					log_enabled[type] = 1;
					if (arg[0] == '=') {
						char *p2 = strchr(log_type_tbl, toupper(arg[1]));
						int type2 = p2 - log_type_tbl;
						if (!p2) goto usage;
						log_file[type] = log_file[type2];
					} else if (strcmp(arg, "-") == 0) {
						log_file[type] = stdout;
					} else {
						log_file[type] = fopen(arg, "wb");
					}
					if (!log_file[type]) {
						perror(arg);
						exit(1);
					}
					break;
				}
				case 'N':
					if (*arg) goto usage;
					new_flash_image = true;
					break;
				case 'P': {
					char **pp;
					switch (toupper(*arg++)) {
						case 'B': pp = &preload_boot2; break;
						case 'D': pp = &preload_diags; break;
						case 'O': pp = &preload_os; break;
						default: goto usage;
					}
					if (*arg == '=') arg++;
					*pp = arg;
					break;
				}
				default:
usage:
					printf(
						"nspire emulator v0.26\n"
						"  /1=boot1      - location of BOOT1 image\n"
						"  /B=boot2      - location of decompressed BOOT2 image\n"
						"  /C            - emulate CAS hardware version\n"
						"  /D            - enter debugger at start\n"
						"  /F=file       - flash image filename\n"
						"  /K            - emulate TI-84+ keypad\n"
						"  /N            - create new flash image\n"
						"  /PB=boot2.img - preload flash with BOOT2 (.img file)\n"
						"  /PO=osfile    - preload flash with OS (.tnc/.tno file)\n");
					return 1;
			}
		} else {
			goto usage;
		}
	}

	switch ((boot1_filename != NULL) + (boot2_filename != NULL) + new_flash_image) {
		case 0: goto usage;
		case 1: break;
		default: printf("Must use exactly one of /1, /B, or /N.\n"); return 0;
	}

	if (new_flash_image) {
		flash_initialize(preload_boot2, preload_diags, preload_os);
		flash_file = fopen(flash_filename, "wb");
		if (!flash_file) {
			perror(flash_filename);
			return 1;
		}
		if (!fwrite(flash_data, sizeof flash_data, 1, flash_file)) {
			printf("Could not write flash data to %s\n", flash_filename);
			return 1;
		}
		fclose(flash_file);
		printf("Created flash image %s\n", flash_filename);
		return 0;
	}

	if (flash_filename) {
		if (preload_boot2 || preload_diags || preload_os) {
			printf("Can't preload to an existing flash image\n");
			return 1;
		}
		flash_file = fopen(flash_filename, "r+b");
		if (!flash_file) {
			perror(flash_filename);
			return 1;
		}
		if (!fread(flash_data, sizeof flash_data, 1, flash_file)) {
			printf("Could not read flash image from %s\n", flash_filename);
			return 1;
		}
	} else {
		flash_initialize(preload_boot2, preload_diags, preload_os);
	}

	memset(MEM_PTR(0x00000000), -1, 0x80000);
	for (i = 0x00000; i < 0x80000; i += 4) {
		RAM_FLAGS(&rom_00[i]) = RF_READ_ONLY;
	}
	if (boot1_filename) {
		/* Load the ROM */
		f = fopen(boot1_filename, "rb");
		if (!f) {
			perror(boot1_filename);
			return 1;
		}
		fread(MEM_PTR(0x00000000), 1, 0x80000, f);
		fclose(f);
	} else {
		/* Little hack to get OS 1.1 to work */
		*(u32 *)MEM_PTR(0x0000001C) = 0x1234;
		*(u32 *)MEM_PTR(0x00001234) = 0xE12FFF1E; /* bx lr */
	}

	memory_initialize();
	addr_cache_init();

	gui_initialize();

	//FILE *untrans = fopen("untrans.out", "wb");

	emu_reset();

	gtk_main();
	return 0;
}
