#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//#define WIN32_LEAN_AND_MEAN
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "id.h"
#include "emu.h"


#define	LCD_W			320
#define	LCD_H			240
#define	LCD_FREQ	50

#define	RGBA(c)		((c) << 24 | ((c) & 0xff00) << 8 | \
										((c) >> 8 & 0xff00) | ((c) >> 24 & 0xff))
//#define	RGBA(c)	(c)		// no byte order change for little endian targets


char target_folder[256];


static const struct key_desc {
	unsigned int keycode;
	unsigned char ext; /* 1 = non-extended key only, 2 = extended key only, 3 = either */
	unsigned char nspire_key;
	unsigned char ti84_key;
	struct rect {
		int x, y;
		int w, h;
	} hitbox[2];
} key_table[] = {
	{ GDK_KP_Enter,        2, 0x00, 0x10, { { 348, 780, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_Return,          1, 0x01, 0x10, { { 311, 749, 36, 36 }, { 309, 756, 50, 30 } } },
	{ GDK_space,           3, 0x02, 0x40, { { 292, 780, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_KP_Subtract,     3, 0x03, 0x20, { { 255, 749, 36, 36 }, { 250, 756, 50, 30 } } },
	{ GDK_z,               3, 0x04, 0x31, { { 236, 780, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_period,          3, 0x05, 0x30, { { 199, 749, 36, 36 }, { 191, 756, 50, 30 } } },
	{ GDK_y,               3, 0x06, 0x41, { { 180, 780, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_0,               3, 0x07, 0x40, { { 143, 749, 36, 36 }, { 132, 756, 50, 30 } } },
	{ GDK_x,               3, 0x08, 0x51, { { 124, 780, 18, 18 }, {  73, 717, 50, 30 } } },	// x sto
	{ GDK_F4,              3, 0x0A, 0x61, { {  68, 780, 18, 18 }, { 250, 386, 50, 30 } } },	// theta trace
	{ GDK_comma,           3, 0x10, 0x44, { { 348, 736, 18, 18 }, { 132, 600, 50, 30 } } },
	{ GDK_plus,            3, 0x11, 0x11, { { 311, 705, 36, 36 }, { 309, 717, 50, 30 } } },
	{ GDK_w,               3, 0x12, 0x12, { { 292, 736, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_3,               3, 0x13, 0x21, { { 255, 705, 36, 36 }, { 250, 717, 50, 30 } } },
	{ GDK_v,               3, 0x14, 0x22, { { 236, 736, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_2,               3, 0x15, 0x31, { { 199, 705, 36, 36 }, { 191, 717, 50, 30 } } },
	{ GDK_u,               3, 0x16, 0x32, { { 180, 736, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_1,               3, 0x17, 0x41, { { 143, 705, 36, 36 }, { 132, 717, 50, 30 } } },
	{ GDK_t,               3, 0x18, 0x42, { { 124, 736, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_End,             2, 0x19, 0x37, { {  87, 705, 36, 36 }, { 191, 483, 50, 30 } } },	// e^x stat
	{ GDK_F5,              3, 0x1A, 0x60, { {  68, 736, 18, 18 }, { 309, 386, 50, 30 } } },	// pi graph
	{ GDK_question,        3, 0x20, 0x14, { { 348, 692, 18, 18 }, {   0,   0,  0,  0 } } }, // ?
	{ GDK_minus,           3, 0x21, 0x12, { { 311, 661, 36, 36 }, { 309, 678, 50, 30 } } },
	{ GDK_s,               3, 0x22, 0x52, { { 292, 692, 18, 18 }, {  73, 678, 50, 30 } } },	// s ln
	{ GDK_6,               3, 0x23, 0x22, { { 255, 661, 36, 36 }, { 250, 678, 50, 30 } } },
	{ GDK_r,               3, 0x24, 0x13, { { 236, 692, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_5,               3, 0x25, 0x32, { { 199, 661, 36, 36 }, { 191, 678, 50, 30 } } },
	{ GDK_q,               3, 0x26, 0x23, { { 180, 692, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_4,               3, 0x27, 0x42, { { 143, 661, 36, 36 }, { 132, 678, 50, 30 } } },
	{ GDK_p,               3, 0x28, 0x33, { { 124, 692, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_Next,            2, 0x29, 0x36, { {  87, 661, 36, 36 }, { 191, 522, 50, 30 } } },	// 10^x prgm
	{ GDK_Prior,           2, 0x2A, 0x46, { {  68, 692, 18, 18 }, { 132, 522, 50, 30 } } },	// EE apps
	{ GDK_colon,           3, 0x30, 0xFF, { { 348, 648, 18, 18 }, {   0,   0,  0,  0 } } }, // :
	{ GDK_asterisk,        3, 0x31, 0x13, { { 311, 617, 36, 36 }, { 309, 639, 50, 30 } } },
	{ GDK_o,               3, 0x32, 0x43, { { 292, 648, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_9,               3, 0x33, 0x23, { { 255, 617, 36, 36 }, { 250, 639, 50, 30 } } },
	{ GDK_n,               3, 0x34, 0x53, { { 236, 648, 18, 18 }, {  73, 639, 50, 30 } } },	// log
	{ GDK_8,               3, 0x35, 0x33, { { 199, 617, 36, 36 }, { 191, 639, 50, 30 } } },
	{ GDK_m,               3, 0x36, 0x14, { { 180, 648, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_7,               3, 0x37, 0x43, { { 143, 617, 36, 36 }, { 132, 639, 50, 30 } } },
	{ GDK_l,               3, 0x38, 0x24, { { 124, 648, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_VoidSymbol,      0, 0x39, 0x54, { {  87, 617, 36, 36 }, {  73, 600, 50, 30 } } },	// x^2
	{ GDK_Insert,          3, 0x3A, 0x26, { {  68, 648, 18, 18 }, { 250, 522, 50, 30 } } },	// i vars
	{ GDK_quotedbl,        3, 0x40, 0xFF, { { 348, 604, 18, 18 }, {   0,   0,  0,  0 } } }, // "
	{ GDK_slash,           3, 0x41, 0x14, { { 311, 573, 36, 36 }, { 309, 600, 50, 30 } } },
	{ GDK_k,               3, 0x42, 0x34, { { 292, 604, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_VoidSymbol,      0, 0x43, 0x25, { { 255, 573, 36, 36 }, { 250, 561, 50, 30 } } },	// tan
	{ GDK_j,               3, 0x44, 0x44, { { 236, 604, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_VoidSymbol,      0, 0x45, 0x35, { { 199, 573, 36, 36 }, { 191, 561, 50, 30 } } },	// cos
	{ GDK_i,               3, 0x46, 0x54, { { 180, 604, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_VoidSymbol,      0, 0x47, 0x45, { { 143, 573, 36, 36 }, { 132, 561, 50, 30 } } },	// sin
	{ GDK_h,               3, 0x48, 0x15, { { 124, 604, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_dead_circumflex, 0, 0x49, 0x15, { {  87, 573, 36, 36 }, { 309, 561, 50, 30 } } },	// ^
	{ GDK_greater,         0, 0x4A, 0xFF, { {  68, 604, 18, 18 }, {   0,   0,  0,  0 } } },	// >
	{ GDK_apostrophe,      0, 0x50, 0xFF, { { 348, 560, 18, 18 }, {   0,   0,  0,  0 } } },	// '
	{ GDK_F2,              3, 0x51, 0x63, { { 311, 529, 36, 36 }, { 132, 386, 50, 30 } } },	// cat wind
	{ GDK_g,               3, 0x52, 0x25, { { 292, 560, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_parenright,      3, 0x53, 0x24, { { 255, 529, 36, 36 }, { 250, 600, 50, 30 } } }, // )
	{ GDK_f,               3, 0x54, 0x35, { { 236, 560, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_parenleft,       3, 0x55, 0x34, { { 199, 529, 36, 36 }, { 191, 600, 50, 30 } } }, // (
	{ GDK_e,               3, 0x56, 0x45, { { 180, 560, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_F3,              3, 0x57, 0x62, { { 143, 529, 36, 36 }, { 191, 386, 50, 30 } } },	// var zoom
	{ GDK_d,               3, 0x58, 0x55, { { 124, 560, 18, 18 }, {  73, 561, 50, 30 } } },	// x^-1
	{ GDK_Shift_R,         3, 0x59, 0x16, { {   0,   0,  0,  0 }, { 309, 522, 50, 30 } } },	// clear
	{ GDK_Shift_L,         3, 0x59, 0x65, { {  87, 529, 36, 36 }, {  73, 444, 50, 30 } } },	// 2nd
	{ GDK_less,            0, 0x5A, 0xFF, { {  68, 560, 18, 18 }, {   0,   0,  0,  0 } } },	// <
	{ GDK_Delete,          3, 0x60, 0x67, { { 348, 516, 18, 18 }, { 191, 444, 50, 30 } } },	// flag del
	{ GDK_KP_Begin,        3, 0x61, 0xFF, { { 199, 420, 36, 36 }, {   0,   0,  0,  0 } } }, /* numeric keypad 5 */
	{ GDK_c,               3, 0x62, 0x36, { { 292, 516, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_Home,            2, 0x63, 0x56, { { 306, 392, 44, 30 }, {  73, 522, 50, 30 } } },	// math
	{ GDK_b,               3, 0x64, 0x46, { { 236, 516, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_F1,              3, 0x65, 0x64, { { 306, 436, 44, 30 }, {  73, 386, 50, 30 } } },	// menu y=
	{ GDK_a,               3, 0x66, 0x56, { { 180, 516, 18, 18 }, {   0,   0,  0,  0 } } },
	{ GDK_Escape,          3, 0x67, 0x66, { {  83, 392, 44, 30 }, { 132, 444, 50, 30 } } },	// mode
	{ GDK_bar,             3, 0x68, 0xFF, { { 124, 516, 18, 18 }, {   0,   0,  0,  0 } } }, // |
	{ GDK_Tab,             3, 0x69, 0xFF, { {  83, 436, 44, 30 }, {   0,   0,  0,  0 } } },
	{ GDK_Up,              3, 0x70, 0x03, { { 199, 382, 36, 36 }, { 284, 444, 40, 30 } } },
	{ GDK_KP_Prior,        1, 0x71, 0x46, { { 237, 382, 36, 36 }, {   0,   0,  0,  0 } } }, /* numeric keypad 9 */
	{ GDK_Right,           3, 0x72, 0x02, { { 237, 420, 36, 36 }, { 331, 444, 29, 69 } } },
	{ GDK_KP_Next,         1, 0x73, 0x36, { { 237, 458, 36, 36 }, {   0,   0,  0,  0 } } }, /* numeric keypad 3 */
	{ GDK_Down,            3, 0x74, 0x00, { { 199, 458, 36, 36 }, { 284, 483, 40, 30 } } },
	{ GDK_KP_End,          1, 0x75, 0x37, { { 161, 458, 36, 36 }, {   0,   0,  0,  0 } } }, /* numeric keypad 1 */
	{ GDK_Left,            3, 0x76, 0x01, { { 161, 420, 36, 36 }, { 249, 444, 29, 69 } } },
	{ GDK_KP_Home,         1, 0x77, 0x56, { { 161, 382, 36, 36 }, {   0,   0,  0,  0 } } }, /* numeric keypad 7 */
	{ GDK_BackSpace,       3, 0x78, 0xFF, { { 311, 485, 36, 36 }, {   0,   0,  0,  0 } } },
	{ GDK_Control_L,       3, 0x79, 0x57, { {  87, 485, 36, 36 }, {  73, 483, 50, 30 } } },	// alpha
	{ GDK_equal,           3, 0x7A, 0x47, { {  68, 516, 18, 18 }, { 132, 483, 50, 30 } } },	// = X
	{ GDK_VoidSymbol,      0, 0x94, 0x94, { {  87, 749, 36, 36 }, {  73, 756, 50, 30 } } },	// on
};

static const unsigned int fullspeed = 65000;		// FIXME: 65 khz, arbitrary constant to approximate 100% speed

static struct	calc_skin {		// calc skin definition
	GdkPixbuf *pxb;			// pixel data
	int lcd_x, lcd_y;		// lcd position
	//const struct key_desc *keypad;
} skin;

static struct gui_cursor {
	GdkPixmap *pxm;				// pixel data
	int x, y;							// position on drawing area
	int type;							// 0 = keyboard press, 1 = mouse click

	union {
		struct key_desc key;
	} u;
} keypress[4];		// FIXME: constant

static unsigned int n_keypresses = 0;		// count simultaneous keypresses

static struct	gui_calc {
	GtkWidget *wgt;		// drawing area
	GdkPixmap *pxm;		// final layer before drawing area (scaled)
	float zoom;				// scale factor
	int w, h;					// size (unscaled)
	u32 bg;						// background color
	const struct calc_skin *skin;		// current skin or NULL

	struct {							// calc screen
		GdkPixbuf *pxb;	// pixel data (unscaled)
		int x, y;				// position on drawing area (scaled)
	} lcd;

	struct {
		unsigned int id;
		unsigned int freq;			// only updated by speed increase / decrease
		unsigned int rt_freq;		// real time frequency mesured by gui_emu_run
		double time;
		double cpu_time;
		double idle_time;
		u32 insns;
		u32 rate;
	} timer;
} ti;

static struct gui {
	GtkWidget *win;		// main window
	GtkWidget *menu;	// main menu
	GtkBuilder *builder;
	//struct gui_calc *calc;
	//struct gui_cursor *cursor;
} ui;


static inline void	gui_gdk_color(GdkColor *c, u32 color) {
	if (!c) return;
	color >>= 8;
	c->red = (color >> 8 & 0xff00) | color >> 16;
	c->green = (color >> 8 & 0xff) | (color & 0xff00);
	c->blue = (color << 8 & 0xff00) | (color & 0xff);
}


static inline void	gui_calc_key_press(unsigned char key) {		// FIXME: move elsewhere
	int row, col;

	row = key >> 4;
	col = key & 15;
	key_map[row] |= 1 << col;
}


static inline void	gui_calc_key_release(unsigned char key) {		// FIXME: move elsewhere
	int row, col;

	row = key >> 4;
	col = key & 15;
	key_map[row] &= ~(1 << col);
}


static const struct calc_skin	*gui_load_skin(const char *path) {
	GError *error;

	error = NULL;
	skin.pxb = gdk_pixbuf_new_from_file(path, &error);
	if (error)
		return NULL;
	skin.lcd_x = 56;		// FIXME: constants
	skin.lcd_y = 83;		//
	return &skin;
}


static gboolean	gui_calc_log(gpointer data) {
	// FIXME
	//if (_kbhit()) {
		//char c = _getch();
		//if (c == 4)
			//debugger();
		//else
			//serial_byte_in(c);
	//}

	if (log_enabled[LOG_ICOUNT])
		logprintf(LOG_ICOUNT, "Time=%.6f (CPU=%.6f Idle=%.6f) Insns=%8u Rate=%9d\n",
			ti.timer.time, ti.timer.cpu_time, ti.timer.idle_time,
			ti.timer.insns, ti.timer.rate);
	return TRUE;
}


static gboolean	gui_show_speed(gpointer data) {
	if (show_speed) {
		double speed = (double)ti.timer.rt_freq / fullspeed;
		char buf[40];
		sprintf(buf, "nspire_emu - %.1f%%", 100 * speed);
		gtk_window_set_title(GTK_WINDOW(ui.win), (gchar *)buf);
	} else
		gtk_window_set_title(GTK_WINDOW(ui.win), "nspire_emu");
	return TRUE;
}


static gboolean	gui_emu_run(gpointer data) {
	static GTimer *gtimer = NULL;
	static double cpu_prev_start;
	static double cpu_prev_stop;
	static u64 cycles_prev;
	double cpu_start;
	double cpu_stop;
	u64 cycles;
	int counter;

	counter = data ? *(int *)data : 1;
	if (!gtimer)
		gtimer = g_timer_new();

	cpu_start = g_timer_elapsed(gtimer, NULL);
	emu_iterate(counter);
	cpu_stop = g_timer_elapsed(gtimer, NULL);
	cycles = cycle_count + cycle_count_delta;

	ti.timer.time = cpu_stop - cpu_prev_stop;
	ti.timer.idle_time = cpu_start - cpu_prev_stop;
	ti.timer.cpu_time = cpu_stop - cpu_start;
	ti.timer.insns = (u32)(cycles - cycles_prev);
	ti.timer.rate = ti.timer.insns / ti.timer.cpu_time;
	ti.timer.rt_freq = counter / ti.timer.time;

	cpu_prev_start = cpu_start;
	cpu_prev_stop = cpu_stop;
	cycles_prev = cycles;
	return TRUE;
}


static void	gui_cursor_show(struct gui_cursor *cursor) {
	const u32 color = 0x24ffff80;
	const int size = 3;
	GdkPixbuf *pxb;
	GdkGC *gc;
	GdkColor c;
	struct rect r;
	u32 *p;
	int i;

	if (!cursor) return;

	r = cursor->u.key.hitbox[emulate_ti84_keypad];
	if (!r.w || !r.h)
		return;
	if (!GTK_WIDGET_REALIZED(ti.wgt))
		gtk_widget_realize(ti.wgt);

	cursor->x = r.x * ti.zoom + 0.5f;
	cursor->y = r.y * ti.zoom + 0.5f;
	r.w = r.w * ti.zoom + 0.5f;
	r.h = r.h * ti.zoom + 0.5f;

	if (cursor->pxm)
		g_object_unref(cursor->pxm);
	cursor->pxm = gdk_pixmap_new(GDK_DRAWABLE(ti.wgt->window), r.w, r.h, -1);

	gc = gdk_gc_new(GDK_DRAWABLE(cursor->pxm));
	gdk_draw_drawable(GDK_DRAWABLE(cursor->pxm), gc,
		GDK_DRAWABLE(ti.pxm), cursor->x, cursor->y, 0, 0, r.w, r.h);
	pxb = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, r.w, r.h);
	p = (u32 *)gdk_pixbuf_get_pixels(pxb);
	for (i = 0; i < r.w * r.h; i++)
		*p++ = RGBA(color);
	gdk_draw_pixbuf(GDK_DRAWABLE(cursor->pxm), NULL,
		pxb, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0, 0);
	g_object_unref(pxb);

	gui_gdk_color(&c, color);
	gdk_gc_set_rgb_fg_color(gc, &c);
	gdk_gc_set_line_attributes(gc, 2 * size * ti.zoom + 0.5f,
		GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
	gdk_draw_rectangle(GDK_DRAWABLE(cursor->pxm), gc, FALSE, 0, 0, r.w, r.h);
	g_object_unref(gc);

	gtk_widget_queue_draw_area(ti.wgt, cursor->x, cursor->y, r.w, r.h);
}


static void	gui_cursor_hide(struct gui_cursor *c) {
	int w, h;

	if (!c || !c->pxm) return;
	gdk_drawable_get_size(GDK_DRAWABLE(c->pxm), &w, &h);
	g_object_unref(c->pxm);
	c->pxm = NULL;
	gtk_widget_queue_draw_area(ti.wgt, c->x, c->y, w, h);
}


static void	gui_register_keypress(const struct key_desc *k, int type) {
	struct gui_cursor *cursor;

	if (!k) return;
	gui_calc_key_press((&k->nspire_key)[emulate_ti84_keypad]);
	cursor = &keypress[n_keypresses++];
	cursor->type = type;
	cursor->u.key = *k;
	if (ti.skin)
		gui_cursor_show(cursor);
}


static void	gui_calc_lcd_resize() {
	GdkPixbuf *pxb;
	int w, h;

	w = LCD_W * ti.zoom + 0.5f;
	h = LCD_H * ti.zoom + 0.5f;
	pxb = gdk_pixbuf_scale_simple(ti.lcd.pxb, w, h, GDK_INTERP_NEAREST);
	gdk_draw_pixbuf(GDK_DRAWABLE(ti.pxm), NULL,
		pxb, 0, 0, ti.lcd.x, ti.lcd.y, -1, -1, GDK_RGB_DITHER_NONE, 0, 0);
	g_object_unref(pxb);
	gtk_widget_queue_draw_area(ti.wgt, ti.lcd.x, ti.lcd.y, w, h);
}


static void	gui_calc_resize() {
	GdkPixbuf *pxb;
	GdkGC *gc;
	GdkColor c;
	int w, h;

	if (!GTK_WIDGET_REALIZED(ti.wgt))
		gtk_widget_realize(ti.wgt);

	ti.w = ti.skin ? gdk_pixbuf_get_width(ti.skin->pxb) : LCD_W;
	ti.h = ti.skin ? gdk_pixbuf_get_height(ti.skin->pxb) : LCD_H;
	w = ti.w * ti.zoom + 0.5f;
	h = ti.h * ti.zoom + 0.5f;

	if (ti.pxm)
		g_object_unref(ti.pxm);
	ti.pxm = gdk_pixmap_new(GDK_DRAWABLE(ti.wgt->window), w, h, -1);

	gc = gdk_gc_new(GDK_DRAWABLE(ti.pxm));
	gui_gdk_color(&c, ti.bg);
	gdk_gc_set_rgb_fg_color(gc, &c);
	gdk_draw_rectangle(GDK_DRAWABLE(ti.pxm), gc, TRUE, 0, 0, w, h);
	g_object_unref(gc);		// FIXME: necessary ?

	if (ti.skin) {
		pxb = gdk_pixbuf_scale_simple(ti.skin->pxb, w, h, GDK_INTERP_NEAREST);
		gdk_draw_pixbuf(GDK_DRAWABLE(ti.pxm), NULL,
			pxb, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NONE, 0, 0);
		g_object_unref(pxb);
		ti.lcd.x = ti.skin->lcd_x * ti.zoom + 0.5f;
		ti.lcd.y = ti.skin->lcd_y * ti.zoom + 0.5f;
	} else {
		ti.lcd.x = 0;
		ti.lcd.y = 0;
	}
	gui_calc_lcd_resize();
	gtk_widget_queue_draw(ti.wgt);
}


void	gui_calc_timer_on() {
	static int counter;
	unsigned int delay;

	if (ti.timer.id)
		g_source_remove(ti.timer.id);

		// We adjust delay to avoid blocking the UI or the CPU
	delay = 7 * fullspeed / ti.timer.freq + 3;
	counter = ti.timer.freq * delay / 1000;
	ti.timer.id = g_timeout_add(delay, gui_emu_run, &counter);
	//printf("counter=%d delay=%u freq=%u\n", counter, delay, ti.timer.freq);
}


void	gui_calc_timer_off() {
	if (ti.timer.id)
		g_source_remove(ti.timer.id);
	ti.timer.id = g_idle_add(gui_emu_run, NULL);
}


static gboolean	gui_calc_lcd_refresh() {
	static const u32 palette[16] = {
		0x000000, 0x222623, 0x2e332f, 0x39403a, 0x444c46, 0x505952, 0x5b665e, 0x677369,
		0x728075, 0x7d8c81, 0x89998c, 0x94a698, 0xa0b2a4, 0xabbfaf, 0xb6ccbb, 0xc3dac8,
	};
	const u8 *p = lcd_framebuffer;
	u8 *pixels;
	int r, w, h;

	pixels = gdk_pixbuf_get_pixels(ti.lcd.pxb);
	r = gdk_pixbuf_get_rowstride(ti.lcd.pxb);

	for (h = 0; h < LCD_H; h++) {
		u32 *pp = (u32 *)&pixels[h * r];
		for (w = LCD_W / 2; w-- > 0; p++) {
			*pp++ = RGBA(palette[*p >> 4] << 8 | 0xff);		// take care of endianness
			*pp++ = RGBA(palette[*p & 15] << 8 | 0xff);		//
		}
	}
	gui_calc_lcd_resize();
	return TRUE;
}


static void	gui_calc_new() {
	const char *path;

	path = emulate_ti84_keypad ? "skin84.png" : "skin.png";
	ti.skin = gui_load_skin(path);

	ti.timer.freq = fullspeed;
	gui_calc_timer_on();

	ti.lcd.pxb = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, LCD_W, LCD_H);
	g_timeout_add(1000 / LCD_FREQ, gui_calc_lcd_refresh, NULL);

	ti.wgt = GTK_WIDGET(gtk_builder_get_object(ui.builder, "calculator"));
	ti.zoom = 1.0f;				// FIXME: constants
	ti.bg = 0x424242ff;		//
		// The first call to gui_on_configure() will take care of the rest
}


static void	gui_ui_new(const char *path) {
	GtkAccelGroup *accel;
	GtkWidget *item;
	GError *error;

	ui.builder = gtk_builder_new();

	error = NULL;
	gtk_builder_add_from_file(ui.builder, path, &error);
	if (error) {
		printf("%s\n", error->message);
		exit(1);
	}

	ui.win = GTK_WIDGET(gtk_builder_get_object(ui.builder, "window"));
	ui.menu = GTK_WIDGET(gtk_builder_get_object(ui.builder, "menu"));

		// FIXME: find a way to set key accels in glade
	accel = gtk_accel_group_new();
	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_show_speed"));
	gtk_widget_add_accelerator(item, "activate", accel, GDK_F6, 0, GTK_ACCEL_VISIBLE);
	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_screenshot"));
	gtk_widget_add_accelerator(item, "activate", accel, GDK_F7, 0, GTK_ACCEL_VISIBLE);
	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_speed_up"));
	gtk_widget_add_accelerator(item, "activate", accel, GDK_F8, 0, GTK_ACCEL_VISIBLE);
	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_slow_down"));
	gtk_widget_add_accelerator(item, "activate", accel, GDK_F9, 0, GTK_ACCEL_VISIBLE);
	gtk_window_add_accel_group(GTK_WINDOW(ui.win), accel);
}


G_MODULE_EXPORT void	gui_menu_screenshot() {
	GdkPixbuf *pxb;
	GtkClipboard *clipboard;

	clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
	pxb = gdk_pixbuf_copy(ti.lcd.pxb);
	gtk_clipboard_set_image(clipboard, pxb);
	g_object_unref(pxb);		// FIXME: necessary ?
}


G_MODULE_EXPORT void	gui_menu_set_folder() {
	GtkWidget *dlg;
	GtkWidget *box;
	GtkWidget *entry;

	dlg = gtk_dialog_new_with_buttons("Set target folder", GTK_WINDOW(ui.win),
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);

	entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(entry), target_folder);
	box = gtk_dialog_get_content_area(GTK_DIALOG(dlg));
	gtk_box_pack_start(GTK_BOX(box), entry, FALSE, FALSE, 0);

	gtk_window_set_resizable(GTK_WINDOW(dlg), FALSE);
	gtk_widget_show_all(dlg);

	if (gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_OK) {
		char *path = (char *)gtk_entry_get_text(GTK_ENTRY(entry));
		strncpy(target_folder, path, sizeof(target_folder) - 1);
	}
	gtk_widget_destroy(dlg);
}


G_MODULE_EXPORT void	gui_menu_send_doc() {
	GtkWidget *dlg;
	GtkFileFilter *filter;

	dlg = gtk_file_chooser_dialog_new(NULL, NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, "TI-Nspire Documents (*.tns)");
	gtk_file_filter_add_pattern(filter, "*.tns");
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dlg), filter);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, "All Files");
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dlg), filter);

	if (gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_OK) {
		gchar *path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dlg));
		usblink_put_file(path, target_folder);
		g_free(path);
	}
	gtk_widget_destroy(dlg);
}


G_MODULE_EXPORT void	gui_menu_send_ti84_file() {
	GtkWidget *dlg;
	GtkFileFilter *filter;

	dlg = gtk_file_chooser_dialog_new(NULL, NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, "TI-84+ Files (*.8xp, and so on)");
	gtk_file_filter_add_pattern(filter, "*.8x?");
	gtk_file_filter_add_pattern(filter, "*.83?");
	gtk_file_filter_add_pattern(filter, "*.82?");
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dlg), filter);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, "All Files");
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dlg), filter);

	if (gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_OK) {
		gchar *path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dlg));
		send_file(path);
		g_free(path);
	}
	gtk_widget_destroy(dlg);
}


G_MODULE_EXPORT void	gui_menu_speed_up() {
	if (turbo_mode) return;
	gui_calc_timer_off();
	ti.timer.freq += 500;		// FIXME: arbitrary value
	gui_calc_timer_on();
}


G_MODULE_EXPORT void	gui_menu_slow_down() {
	if (turbo_mode || ti.timer.freq <= 500) return;
	gui_calc_timer_off();
	ti.timer.freq -= 500;		// FIXME: arbitrary value
	gui_calc_timer_on();
}


G_MODULE_EXPORT void	gui_menu_reset_cpu() {
	cpu_events |= EVENT_RESET;
}


G_MODULE_EXPORT void	gui_menu_show_speed() {
	show_speed ^= 1;
}


G_MODULE_EXPORT void	gui_menu_turbo_mode() {
	turbo_mode ^= 1;
	if (turbo_mode)
		gui_calc_timer_off();
	else
		gui_calc_timer_on();
}


G_MODULE_EXPORT void	gui_menu_debugger() {
	debugger();
}


G_MODULE_EXPORT void	gui_menu_flash_save() {
	flash_save_changes();
}


G_MODULE_EXPORT void	gui_menu_usb_connect() {
	usblink_connect();
}


G_MODULE_EXPORT void	gui_menu_usb_disconnect() {
	usblink_disconnect();
}


G_MODULE_EXPORT void	gui_on_destroy() {
	gtk_widget_destroy(ui.win);
	gtk_main_quit();
}


G_MODULE_EXPORT gboolean	gui_on_button_press(GtkWidget *widget, GdkEventButton *event, gpointer data) {
	const struct key_desc *p = key_table;
	struct gui_cursor *c;

	if (event->button == 1) {		// left click
		if (n_keypresses < sizeof(keypress) / sizeof(*keypress))
			for (; p < (key_table + (sizeof(key_table) / sizeof(*key_table))); p++) {
				const struct rect *r = &p->hitbox[emulate_ti84_keypad];
				if (event->x >= r->x * ti.zoom && event->x < (r->x + r->w) * ti.zoom &&
						event->y >= r->y * ti.zoom && event->y < (r->y + r->h) * ti.zoom) {
					gui_register_keypress(p, 1);
					break;
				}
			}
	}
	else {
			// Force all key presses to be released before showing the menu
		for (c = keypress; c < keypress + n_keypresses; c++) {
			gui_calc_key_release((&c->u.key.nspire_key)[emulate_ti84_keypad]);
			gui_cursor_hide(c);
		}
		n_keypresses = 0;
		gtk_menu_popup(GTK_MENU(ui.menu), NULL, NULL, NULL, NULL,
			event->button, event->time);
	}
	return FALSE;
}


G_MODULE_EXPORT gboolean	gui_on_button_release(GtkWidget *widget, GdkEventButton *event, gpointer data) {
	struct gui_cursor *end, *c;

	end = keypress + n_keypresses;
	for (c = keypress; c < end; c++)
		if (c->type == 1) {
			gui_calc_key_release((&c->u.key.nspire_key)[emulate_ti84_keypad]);
			gui_cursor_hide(c);
			n_keypresses--;
			end--;
			if (c != end) {
				*c = *end;
				end->pxm = NULL;
			}
		}
	return FALSE;
}


G_MODULE_EXPORT gboolean	gui_on_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data) {
	const struct key_desc *p = key_table;
	const struct gui_cursor *c;

	if (n_keypresses < sizeof(keypress) / sizeof(*keypress))
		for (; p < (key_table + (sizeof(key_table) / sizeof(*key_table))); p++)
			if (p->keycode == event->keyval) {
					// Make sure we don't register an already pressed key
				for (c = keypress; c < keypress + n_keypresses; c++)
					if (c->u.key.keycode == p->keycode)
						goto done;
				gui_register_keypress(p, 0);
				break;
			}
done:
	//printf("state=0x%.4x ", event->state);
	//printf("keyval=0x%.4x\n", event->keyval);
	return FALSE;
}


G_MODULE_EXPORT gboolean	gui_on_key_release(GtkWidget *widget, GdkEventKey *event, gpointer data) {
	struct gui_cursor *end, *c;

	end = keypress + n_keypresses;
	for (c = keypress; c < end; c++)
		if (c->type == 0 && c->u.key.keycode == event->keyval) {
			gui_calc_key_release((&c->u.key.nspire_key)[emulate_ti84_keypad]);
			gui_cursor_hide(c);
			n_keypresses--;
			end--;
			if (c != end) {
				*c = *end;
				end->pxm = NULL;
			}
			break;
		}
	return FALSE;
}


G_MODULE_EXPORT gboolean	gui_on_configure(GtkWidget *widget, GdkEventConfigure *event, gpointer data) {
	GdkGeometry geometry;
	int w, h;

	if (!ti.pxm) {
		gui_calc_resize();
		geometry.min_height = 42;		// FIXME: constant
		geometry.min_aspect = (double)ti.w / ti.h;
		geometry.max_aspect = (double)ti.w / ti.h;
		geometry.min_width = geometry.min_height * geometry.min_aspect;
		gtk_window_set_geometry_hints(GTK_WINDOW(ui.win), NULL, &geometry,
			GDK_HINT_MIN_SIZE | GDK_HINT_ASPECT);
		gdk_drawable_get_size(GDK_DRAWABLE(ti.pxm), &w, &h);
		gtk_window_resize(GTK_WINDOW(ui.win), w, h);
	}
	else {
		gdk_drawable_get_size(GDK_DRAWABLE(ti.pxm), NULL, &h);
		if (event->height != h) {
			ti.zoom = (float)event->height / ti.h;
			gui_calc_resize();
		}
	}
	return FALSE;
}


G_MODULE_EXPORT gboolean	gui_on_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data) {
	const struct gui_cursor *c;

	if (ti.pxm)
		gdk_draw_drawable(GDK_DRAWABLE(event->window),
			widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
			GDK_DRAWABLE(ti.pxm), event->area.x, event->area.y,
			event->area.x, event->area.y, event->area.width, event->area.height);

	for (c = keypress; c < keypress + n_keypresses; c++)
		if (c->pxm)
			gdk_draw_drawable(GDK_DRAWABLE(event->window),
				widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
				GDK_DRAWABLE(c->pxm), 0, 0, c->x, c->y, -1, -1);
	return FALSE;
}


void	gui_initialize() {
	GtkWidget *item;
	GdkColor c;

	gui_ui_new("gui.glade");
	gui_calc_new();

	if (turbo_mode)
		gui_calc_timer_off();

	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_show_speed"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), show_speed);
	item = GTK_WIDGET(gtk_builder_get_object(ui.builder, "item_throttle"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), !turbo_mode);

	gtk_builder_connect_signals(ui.builder, NULL);

	g_timeout_add(1000 / 100, gui_calc_log, NULL);
	g_timeout_add(1000 / 2, gui_show_speed, NULL);

	gui_gdk_color(&c, ti.bg);
	gtk_widget_modify_bg(ui.win, GTK_STATE_NORMAL, &c);		// FIXME

	gtk_widget_show_all(ui.win);
	strcpy(target_folder, "Examples");
}
