/*
 * Decompiled with CFR 0.152.
 */
package elliott803.utils;

import elliott803.machine.Instruction;
import elliott803.machine.Word;
import elliott803.telecode.CharToTelecode;
import elliott803.utils.Args;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

public class Assembler {
    private static final String COMMENT_CHAR = "*";
    private static final String LABEL_CHAR = ":";
    private static final String ADDRESS_CHAR = "=";
    private static final String TRIGGER_CHAR = "@";
    private static final String STRING_CHAR = "'";
    private static final String LABEL_PATTERN = "[A-Za-z]\\w*";
    private static final String NUMBER_PATTERN = "\\d+";
    private static final String CONSTANT_PATTERN = "[\\+-]?\\d+";
    private static final String VALUE_PATTERN = "(([A-Za-z]\\w*)|(\\d+))";
    private static final String LOAD_PATTERN = "=\\d+";
    private static final String TRIGGER_PATTERN = "@(([A-Za-z]\\w*)|(\\d+))";
    private static final String OP_PATTERN = "[01234567]{2}";
    private static final String B_PATTERN = "[:/]";
    private static final String INSTR_PATTERN = "[01234567]{2}\\s+(([A-Za-z]\\w*)|(\\d+))";
    private static final String LABEL_LINE = "[A-Za-z]\\w*\\s*:.*";
    private static final String DIRECTIVE_LINE = "((=\\d+)|(@(([A-Za-z]\\w*)|(\\d+))))";
    private static final String CONSTANT_LINE = "(([\\+-]?\\d+)|([A-Za-z]\\w*))";
    private static final String CONSTANT_SEQ_LINE = "((([\\+-]?\\d+)|([A-Za-z]\\w*)))?(\\s*,\\s*(([\\+-]?\\d+)|([A-Za-z]\\w*)))*";
    private static final String STRING_LINE = "'.*'";
    private static final String CODE_LINE = "[01234567]{2}\\s+(([A-Za-z]\\w*)|(\\d+))(\\s*[:/](\\s*[01234567]{2}\\s+(([A-Za-z]\\w*)|(\\d+)))?)?";
    private LineNumberReader input;
    private OutputStream output;
    private Map<String, Integer> symbols;
    private List<SourceLine> sourceCode;
    private List<Long> objectCode;
    private SourceLine loadDirective;
    private SourceLine triggerDirective;
    private int loadAddress;
    private int triggerAddress;

    public static void main(String[] args) throws Exception {
        Args parms = new Args("Assembler", "inputfile outputtape", args, null);
        File inputFile = parms.getInputFile(1);
        File outputFile = parms.getOutputFile(2);
        if (inputFile == null || outputFile == null) {
            parms.usage();
        }
        LineNumberReader input = new LineNumberReader(new FileReader(inputFile));
        FileOutputStream output = new FileOutputStream(outputFile);
        Assembler assembler = new Assembler(input, output);
        assembler.run();
        input.close();
        output.close();
    }

    Assembler(LineNumberReader in, OutputStream out) {
        this.input = in;
        this.output = out;
    }

    void run() throws IOException {
        this.symbols = new HashMap<String, Integer>();
        this.sourceCode = new ArrayList<SourceLine>();
        this.buildSymbolsAndSource();
        this.loadAddress = -1;
        this.triggerAddress = -1;
        this.setLoadAndEntryAddress();
        this.objectCode = new ArrayList<Long>();
        this.generateObjectCode();
        this.writeOutputTape();
        if (this.triggerAddress != -1) {
            this.writeTrigger();
        }
    }

    private void buildSymbolsAndSource() throws IOException {
        String line = this.input.readLine();
        while (line != null) {
            int i = line.indexOf(COMMENT_CHAR);
            if (i != -1) {
                line = line.substring(0, i);
            }
            if ((line = line.trim()).length() > 0) {
                if (line.matches(LABEL_LINE)) {
                    i = line.indexOf(LABEL_CHAR);
                    String label = line.substring(0, i).trim();
                    if (!this.symbols.containsKey(label)) {
                        this.symbols.put(label, this.sourceCode.size());
                    } else {
                        this.error(0, "Symbol defined more than once: " + label);
                    }
                    line = line.substring(i + 1);
                    continue;
                }
                if (line.matches(DIRECTIVE_LINE)) {
                    if (line.matches(LOAD_PATTERN)) {
                        if (this.loadDirective == null) {
                            this.loadDirective = new SourceLine(line);
                        } else {
                            this.error(0, "Load directive already set to " + this.loadDirective.source);
                        }
                    } else if (line.matches(TRIGGER_PATTERN)) {
                        if (this.triggerDirective == null) {
                            this.triggerDirective = new SourceLine(line);
                        } else {
                            this.error(0, "Entry directive already set to " + this.triggerDirective.source);
                        }
                    } else {
                        this.error(0, "Incorrect directive: " + line);
                    }
                } else if (line.matches(CONSTANT_SEQ_LINE)) {
                    StringTokenizer t = new StringTokenizer(line, ",");
                    while (t.hasMoreTokens()) {
                        this.sourceCode.add(new SourceLine(t.nextToken().trim()));
                    }
                } else if (line.matches(CODE_LINE)) {
                    this.sourceCode.add(new SourceLine(line));
                } else if (line.matches(STRING_LINE)) {
                    String text = line.substring(1);
                    if (text.endsWith(STRING_CHAR)) {
                        text = text.substring(0, text.length() - 1);
                    }
                    CharToTelecode converter = new CharToTelecode();
                    byte[] tc = new byte[text.length() * 2];
                    int len = converter.convert(text.toCharArray(), text.length(), tc);
                    i = 0;
                    while (i < len) {
                        this.sourceCode.add(new SourceLine(Byte.toString(tc[i])));
                        ++i;
                    }
                } else {
                    this.error(0, "Syntax error: " + line);
                }
            }
            line = this.input.readLine();
        }
    }

    private void setLoadAndEntryAddress() {
        this.loadAddress = this.loadDirective != null ? Integer.parseInt(this.loadDirective.source.substring(1)) : 8192 - this.sourceCode.size();
        for (Map.Entry<String, Integer> entry : this.symbols.entrySet()) {
            entry.setValue(entry.getValue() + this.loadAddress);
        }
        if (this.triggerDirective != null) {
            String addr = this.triggerDirective.source.substring(1);
            if (addr.matches(NUMBER_PATTERN)) {
                this.triggerAddress = Integer.parseInt(addr);
            } else if (this.symbols.containsKey(addr)) {
                this.triggerAddress = this.symbols.get(addr);
            } else {
                this.error(this.triggerDirective.lineNo, "Incorrect entry point: " + this.triggerDirective.source);
            }
        }
    }

    private void generateObjectCode() {
        for (SourceLine sourceLine : this.sourceCode) {
            String source = sourceLine.source;
            long value = 0L;
            if (source.matches(CONSTANT_LINE)) {
                if (source.matches(CONSTANT_PATTERN)) {
                    value = Long.parseLong(source.startsWith("+") ? source.substring(1) : source);
                } else if (this.symbols.containsKey(source)) {
                    value = this.symbols.get(source).intValue();
                } else {
                    this.error(sourceLine.lineNo, "Unresolved symbol: " + source);
                }
            } else if (source.matches(CODE_LINE)) {
                int instr1 = 0;
                int b = 0;
                int instr2 = 0;
                StringTokenizer t1 = new StringTokenizer(source, ":/", true);
                instr1 = this.parseInstruction(sourceLine.lineNo, t1.nextToken().trim());
                if (t1.hasMoreTokens()) {
                    int n = b = t1.nextToken().equals(LABEL_CHAR) ? 0 : 1;
                    if (t1.hasMoreTokens()) {
                        instr2 = this.parseInstruction(sourceLine.lineNo, t1.nextToken().trim());
                    }
                }
                value = Word.asInstr(instr1, b, instr2);
            } else {
                this.error(sourceLine.lineNo, "Incorrect source code: " + source);
            }
            this.objectCode.add(value);
        }
    }

    private int parseInstruction(int lineNo, String instruction) {
        int op = 0;
        int addr = 0;
        StringTokenizer t = new StringTokenizer(instruction, " \t");
        op = Integer.parseInt(t.nextToken(), 8);
        String target = t.nextToken();
        if (target.matches(NUMBER_PATTERN)) {
            addr = Integer.parseInt(target);
        } else if (this.symbols.containsKey(target)) {
            addr = this.symbols.get(target);
        } else {
            this.error(lineNo, "Unresolved symbol: " + target);
        }
        return Instruction.asInstr(op, addr);
    }

    private void writeOutputTape() throws IOException {
        this.writeWord(this.loadAddress - 4);
        for (long value : this.objectCode) {
            this.writeWord(value);
        }
    }

    private void writeTrigger() throws IOException {
        int pad = 8192 - (this.loadAddress + this.objectCode.size());
        if (pad > 12) {
            System.out.println("WARNING: will need " + pad + " blanks words for trigger.");
            System.out.println("Suggest loading code at address " + (8192 - this.objectCode.size()));
        }
        int i = 0;
        while (i < pad + 4) {
            this.writeWord(0L);
            ++i;
        }
        this.writeWord(Word.asInstr(0, 0, Instruction.asInstr(18, this.triggerAddress - 4)));
        this.writeWord(0L);
    }

    private void writeWord(long word) throws IOException {
        byte[] bb = new byte[8];
        int i = bb.length;
        while (i > 0) {
            bb[i - 1] = (byte)(word & 0x1FL);
            word >>= 5;
            --i;
        }
        bb[0] = (byte)(bb[0] | 0x10);
        this.output.write(bb);
    }

    private void error(int lineNo, String msg) {
        System.err.println("Line " + (lineNo == 0 ? this.input.getLineNumber() : lineNo) + ": " + msg);
        System.exit(1);
    }

    private class SourceLine {
        int lineNo;
        String source;

        SourceLine(String s) {
            this.lineNo = Assembler.this.input.getLineNumber();
            this.source = s;
        }
    }
}

