const EXSID_USBVID = 0x0403;
const HARDSID_USBVID = 0x6581;
const SIDBLASTER_USBVID = 0x0403;
const SIDBLASTER_USBDESC = "SIDBlaster/USB";
const USBSID_USBVID = 0xcafe;
const USBSID_USBPID = 0x4011;
const USBSID_USBDESC = "USBSID-Pico";

const HardwareFunctions = {
  init: undefined,
  write: undefined,
  delay: undefined,
  pause: undefined,
  continue: undefined,
  next: undefined,
  reset: undefined,
  quit: undefined,
  mapping: undefined,
};

const Chip = {
  NEXT: -1,
  RESET: -2,
  QUIT: -3,
  PAUSE: -4,
  CONTINUE: -5,
};

function Queue() {
  var head, tail;
  return Object.freeze({
    enqueue(value) {
      const link = { value, next: undefined };
      tail = head ? (tail.next = link) : (head = link);
    },
    enqueueAll(queue) {
      if (queue.head) {
        if (head) {
          tail.next = queue.head;
        } else {
          head = queue.head;
        }
        tail = queue.tail;
      }
    },
    dequeue() {
      if (head) {
        var value = head.value;
        head = head.next;
        return value;
      }
      return undefined;
    },
    dequeueAll() {
      var dequeued = {
        head: head,
        tail: tail,
      };
      tail = head = undefined;
      return dequeued;
    },
    peek() {
      return head?.value;
    },
    clear() {
      tail = head = undefined;
    },
    isNotEmpty() {
      return head;
    },
  });
}

var mapping, lastChipModel;
var sidWriteQueue = new Queue();
var usbPlayTimer;
var deviceCount = 0;
var chipCount = 0;
var engine = "EMULATION";

async function init_hardware() {
  var device = undefined;
  let devices = await navigator.usb.getDevices();
  if (devices.length === 0) {
    try {
      device = await navigator.usb.requestDevice({
        filters: [
          {
            vendorId: HARDSID_USBVID,
          },
          {
            vendorId: EXSID_USBVID,
          },
          {
            vendorId: USBSID_USBVID,
            productId: USBSID_USBPID,
          },
        ],
      });
    } catch (e) {
      console.log(e);
    }
  } else {
    for (let i = 0; i < devices.length; i++) {
      let dev = devices[i];
      if (dev.vendorId === HARDSID_USBVID || dev.vendorId === EXSID_USBVID || dev.vendorId === USBSID_USBVID) {
        device = dev;
        break;
      }
    }
  }
  if (!device) {
    console.log("No devices found!");
    return;
  }
  await device.open();
  console.log("Opened:", device.opened);
  console.log(`Product name: ${device.productName}, serial number ${device.serialNumber}`);
  await init_hardware_by_device(device);
}
async function init_hardware_by_device(device) {
  await init_hardware_by_device_and_vendor(device, device.vendorId, device.productName);
}
async function init_hardware_by_vendor(vendorId, productName) {
  await init_hardware_by_device_and_vendor(undefined, vendorId, productName);
}
async function init_hardware_by_device_and_vendor(device, vendorId, productName) {
  if (vendorId == SIDBLASTER_USBVID && productName.startsWith(SIDBLASTER_USBDESC)) {
    HardwareFunctions.init = init_sidblaster;
    HardwareFunctions.reset = reset_sidblaster;
    HardwareFunctions.write = write_sidblaster;
    HardwareFunctions.next = next_sidblaster;
    HardwareFunctions.quit = quit_sidblaster;
    HardwareFunctions.mapping = "sidblaster-mapping/";
    HardwareFunctions.mapped = mapped_sidblaster;
    engine = "SIDBLASTER";
  } else if (vendorId == EXSID_USBVID) {
    HardwareFunctions.init = init_exsid;
    HardwareFunctions.reset = reset_exsid;
    HardwareFunctions.write = write_exsid;
    HardwareFunctions.delay = delay_exsid;
    HardwareFunctions.pause = pause_exsid;
    HardwareFunctions.continue = continue_exsid;
    HardwareFunctions.next = next_exsid;
    HardwareFunctions.quit = quit_exsid;
    HardwareFunctions.mapping = "exsid-mapping/";
    HardwareFunctions.mapped = mapped_exsid;
    engine = "EXSID";
  } else if (vendorId == HARDSID_USBVID) {
    HardwareFunctions.init = init_hardsid;
    HardwareFunctions.reset = reset_hardsid;
    HardwareFunctions.write = write_hardsid;
    HardwareFunctions.delay = delay_hardsid;
    HardwareFunctions.pause = pause_hardsid;
    HardwareFunctions.continue = continue_hardsid;
    HardwareFunctions.next = next_hardsid;
    HardwareFunctions.quit = quit_hardsid;
    HardwareFunctions.mapping = "hardsid-mapping/";
    HardwareFunctions.mapped = mapped_hardsid;
    engine = "HARDSID";
    // } else if (vendorId == USBSID_USBVID || productName.startsWith(USBSID_USBDESC)) {
  } else if (vendorId == USBSID_USBVID) {
    HardwareFunctions.init = init_usbsid;
    HardwareFunctions.reset = reset_usbsid;
    HardwareFunctions.write = write_usbsid;
    HardwareFunctions.delay = delay_usbsid;
    HardwareFunctions.pause = pause_usbsid;
    HardwareFunctions.continue = continue_usbsid;
    HardwareFunctions.next = next_usbsid;
    HardwareFunctions.quit = quit_usbsid;
    HardwareFunctions.mapping = "usbsid-mapping/";
    HardwareFunctions.mapped = mapped_usbsid;
    engine = "USBSID";
  } else {
    engine = "EMULATION";
  }
  deviceCount = 0;
  chipCount = 0;
  await HardwareFunctions.init(device);
  if (typeof usbPlayTimer !== "undefined") {
    clearTimeout(usbPlayTimer);
  }
  reset_UsbQueue();
}

function set_UsbMapping(device_mapping) {
  mapping = device_mapping;
  console.log(mapping);
  HardwareFunctions.mapped();
}

var p = 0;
function reset_UsbQueue() {
  lastChipModel = undefined;
  sidWriteQueue.clear();
  sidWriteQueue.enqueue({
    chip: Chip.RESET,
  });
  p = 0;
}

function pause_UsbQueue() {
  lastChipModel = undefined;
  sidWriteQueue.clear();
  sidWriteQueue.enqueue({
    chip: p == 0 ? Chip.PAUSE : Chip.CONTINUE,
  });
  p = p == 0 ? 1 : 0;
}

async function doPlay() {
  while (sidWriteQueue.isNotEmpty()) {
    write = sidWriteQueue.dequeue();

    if (write.chip == Chip.QUIT) {
      await HardwareFunctions.quit();
      if (ajaxRequest) {
        ajaxRequest.cancel();
      }
      return;
    } else if (write.chip == Chip.RESET) {
      await HardwareFunctions.reset();
      usbPlayTimer = setTimeout(() => this.doPlay(), 250);
      return;
    } else if (write.chip == Chip.PAUSE) {
      HardwareFunctions.pause();
    } else if (write.chip == Chip.CONTINUE) {
      HardwareFunctions.continue();
    } else if (write.chip == Chip.NEXT) {
      if ((await HardwareFunctions.next()) == 0) {
        Vue.nextTick(() => app.setNextPlaylistEntry());
      } else {
        sidWriteQueue.enqueue({
          chip: Chip.NEXT,
        });
        usbPlayTimer = setTimeout(() => this.doPlay(), 250);
        return;
      }
    } else {
      if (write.reg == -1) {
        await HardwareFunctions.delay(write);
      } else {
        await HardwareFunctions.write(write);
      }
    }
  }
  usbPlayTimer = setTimeout(() => this.doPlay());
}

//
// EXSID
//

async function init_exsid(device) {
  deviceCount = 1;
  var ok = await exSID_init(device);
  if (ok != -1) {
    return 0;
  }
  return -1;
}

async function reset_exsid() {
  exSID_reset(0);
}

async function write_exsid(write) {
  write.reg = write.reg & 0x1f;
  if (write.reg <= 0x18) {
    // "Ragga Run.sid" denies to work!

    const chipModel = mapping[write.chip];
    if (lastChipModel !== chipModel) {
      await exSID_chipselect(chipModel === "MOS8580" ? ChipSelect.XS_CS_CHIP1 : ChipSelect.XS_CS_CHIP0);
      lastChipModel = chipModel;
    }
    await exSID_clkdwrite(write.cycles, write.reg, write.value);
  }
}

async function delay_exsid(write) {
  await exSID_delay(write.cycles);
}

async function pause_exsid() {
  xSoutb(XSP_AD_IOCTAM, 1);
}

async function continue_exsid() {
  xSoutb(XSP_AD_IOCTAU, 1);
}

async function quit_exsid() {
  await exSID_exit();
}

async function mapped_exsid() {
  const chipModel = mapping[0];
  const stereo = mapping[-1] === "true";
  const fakeStereo = mapping[-2] === "true";
  const cpuClock = mapping[-3];

  if (fakeStereo) {
    lastChipModel = chipModel;
    exSID_chipselect(ChipSelect.XS_CS_BOTH);
  }
  exSID_audio_op(AudioOp.XS_AU_MUTE);
  exSID_clockselect(cpuClock === "PAL" ? ClockSelect.XS_CL_PAL : ClockSelect.XS_CL_NTSC);
  if (stereo) {
    exSID_audio_op(chipModel === "MOS6581" ? AudioOp.XS_AU_6581_8580 : AudioOp.XS_AU_8580_6581);
  } else {
    exSID_audio_op(chipModel === "MOS6581" ? AudioOp.XS_AU_6581_6581 : AudioOp.XS_AU_8580_8580);
  }
  exSID_audio_op(AudioOp.XS_AU_UNMUTE);
}

async function next_exsid() {
  if (exSID_is_playing()) {
    return -1;
  } else {
    exSID_reset(0);
    return 0;
  }
}

//
// HardSID
//

async function init_hardsid(device) {
  await hardsid_usb_init(device, true, SysMode.SIDPLAY);
  deviceCount = hardsid_usb_getdevcount();
  console.log("Device count: " + deviceCount);
  if (deviceCount > 0) {
    chipCount = hardsid_usb_getsidcount(0);
    console.log("Chip count: " + chipCount);
    return 0;
  }
  return -1;
}

async function reset_hardsid() {
  await hardsid_usb_abortplay(0);
  for (let chipNum = 0; chipNum < chipCount; chipNum++) {
    await hardsid_usb_reset(0, chipNum, 0x00);
  }
}

async function write_hardsid(write) {
  await delay_hardsid(write);
  while ((await hardsid_usb_write(0, (write.chip << 5) | write.reg, write.value)) == WState.BUSY);
}

async function quit_hardsid() {}

async function mapped_hardsid() {}

async function delay_hardsid(write) {
  let Cycles = write.cycles;
  while (Cycles > 65535) {
    while ((await hardsid_usb_delay(0, 65535)) == WState.BUSY);
    Cycles -= 65536;
  }
  if (Cycles > 0) {
    while ((await hardsid_usb_delay(0, Cycles)) == WState.BUSY);
  }
}

async function pause_hardsid() {}

async function continue_hardsid() {}

async function next_hardsid() {
  await hardsid_usb_sync(0);
  while ((await hardsid_usb_flush(0)) == WState.BUSY) {}
  return 0;
}

//
// SIDBlaster
//

async function init_sidblaster() {
  deviceCount = 1;
  var ok = await sidblaster_init();
  if (ok != -1) {
    return 0;
  }
  return -1;
}

async function reset_sidblaster() {
  sidblaster_reset(0);
}

async function write_sidblaster(write) {
  sidblaster_write(write.cycles, write.reg, write.value);
}

async function quit_sidblaster() {
  await sidblaster_exit();
}

async function mapped_sidblaster() {}

async function next_sidblaster() {
  if (sidblaster_is_playing()) {
    return -1;
  } else {
    sidblaster_reset();
    return 0;
  }
}

//
// USBSID-Pico
//
async function init_usbsid(device) {
  deviceCount = 1;
  var ok = await usbsid_init(device, false);
  if (ok != -1) {
    await reset_usbsid();
    return 0;
  }
  return -1;
}

async function delay_usbsid(write) {}

async function pause_usbsid() {
  await usbsid_pause();
}

async function continue_usbsid() {
  await usbsid_continue();
}

async function reset_usbsid() {
  // on: connect, start play, stop play
  await usbsid_reset();
}

async function next_usbsid() {
  if (await usbsid_is_playing()) {
    return -1;
  } else {
    await usbsid_reset();
    return 0;
  }
}

async function write_usbsid(write) {
  await usbsid_writecycled(write.cycles, write.chip, write.reg, write.value);
  /* await usbsid_write(write.chip, write.reg, write.value); */
}

async function quit_usbsid() {
  await usbsid_close();
}

async function mapped_usbsid() {
  const chipModel = mapping[0];
  const stereo = mapping[-1] === "true";
  const fakeStereo = mapping[-2] === "true";
  const cpuClock = mapping[-3];
  const useAwaitForUsbSid = mapping[-4];
  console.log(useAwaitForUsbSid);
}
