importScripts("jsidplay2.js");

// $DEVTOOLS_SECTION_1

function processSamples(lf, ri, le) {
  var left = lf.buffer.slice(lf.byteOffset, lf.byteOffset + (le << 2));
  var right = ri.buffer.slice(ri.byteOffset, ri.byteOffset + (le << 2));
  postMessage(
    {
      eventType: "SAMPLES",
      eventData: {
        left: left,
        right: right,
        length: le,
      },
    },
    [left, right]
  );
}

function processPixels(pi, le) {
  var image = pi.buffer.slice(pi.byteOffset, pi.byteOffset + le);
  createImageBitmap(new ImageData(new Uint8ClampedArray(image), 384, le / 1536)).then((bitmap) =>
    postMessage(
      {
        eventType: "FRAME",
        eventData: {
          image: bitmap,
        },
      },
      [bitmap]
    )
  );
}

function processSidWrite(at, ti, ad, va) {
  postMessage({
    eventType: "SID_WRITE",
    eventData: {
      absTime: at,
      relTime: ti,
      addr: ad,
      value: va,
    },
  });
}

function timerEnd(ed) {
  postMessage({
    eventType: "TIMER_END",
    eventData: {
      end: ed,
    },
  });
}

function processPrinter(op) {
  postMessage({
    eventType: "PRINTER",
    eventData: {
      output: op,
    },
  });
}

function whatsSid(ar, le) {
  var wav = ar.buffer.slice(ar.byteOffset, ar.byteOffset + le);
  postMessage(
    {
      eventType: "WHATSSID",
      eventData: {
        wav: wav,
      },
    },
    [wav]
  );
}

const eventMap = {
  INITIALISE: function (eventData) {
    let args = [];
    eventData && Object.entries(eventData).map(([key, value]) => (args.push("--" + key), args.push("" + value)));
    main(args);

    postMessage({
      eventType: "INITIALISED",
    });
  },
  OPEN: function (eventData) {
    js2open(
      eventData.contents ?? null,
      eventData.tuneName ?? null,
      eventData.startSong,
      eventData.nthFrame,
      eventData.sidWrites,
      eventData.cartContents ?? null,
      eventData.cartName ?? null,
      eventData.command ?? null,
      eventData.songLength || 0
    );

    postMessage({
      eventType: "OPENED",
    });
  },
  CLOCK: function (eventData) {
    js2clock();

    postMessage({
      eventType: "CLOCKED",
    });
  },
  IDLE: function (eventData) {
    if (((eventData && eventData.sleepTime) ?? 0) == 0) {
      postMessage({ eventType: "CLOCKED" });
    } else {
      setTimeout(() => postMessage({ eventType: "CLOCKED" }), eventData.sleepTime);
    }
  },
  //
  // ISidPlay2Section methods
  //
  SET_DEFAULT_PLAY_LENGTH: function (eventData) {
    js2defaultPlayLength(eventData.defaultPlayLength);

    postMessage({
      eventType: "DEFAULT_PLAY_LENGTH_SET",
    });
  },
  SET_LOOP: function (eventData) {
    js2loop(eventData.loop);

    postMessage({
      eventType: "LOOP_SET",
    });
  },
  SET_SINGLE: function (eventData) {
    js2single(eventData.single);

    postMessage({
      eventType: "SINGLE_SET",
    });
  },
  SET_PAL_EMULATION_ENABLE: function (eventData) {
    js2palEmulationEnable(eventData.palEmulationEnable);

    postMessage({
      eventType: "PAL_EMULATION_ENABLE_SET",
    });
  },
  SET_TURBO_TAPE: function (eventData) {
    js2turboTape(eventData.turboTape);

    postMessage({
      eventType: "TURBO_TAPE_SET",
    });
  },
  SET_FADE_TIME: function (eventData) {
    js2fade(eventData.fadeInTime, eventData.fadeOutTime);

    postMessage({
      eventType: "FADE_TIME_SET",
    });
  },
  //
  // IAudioSection methods
  //
  SET_SAMPLING_RATE: function (eventData) {
    js2samplingRate(eventData.samplingRate);

    postMessage({
      eventType: "SAMPLING_RATE_SET",
    });
  },
  SET_SAMPLING: function (eventData) {
    js2sampling(eventData.sampling);

    postMessage({
      eventType: "SAMPLING_SET",
    });
  },
  SET_VOLUME_LEVELS: function (eventData) {
    js2volumeLevels(
      eventData.mainVolume,
      eventData.secondVolume,
      eventData.thirdVolume,
      eventData.mainBalance,
      eventData.secondBalance,
      eventData.thirdBalance,
      eventData.mainDelay,
      eventData.secondDelay,
      eventData.thirdDelay
    );

    postMessage({
      eventType: "VOLUME_LEVELS_SET",
    });
  },
  SET_BUFFER_SIZE: function (eventData) {
    js2bufferSize(eventData.bufferSize);

    postMessage({
      eventType: "BUFFER_SIZE_SET",
    });
  },
  SET_AUDIO_BUFFER_SIZE: function (eventData) {
    js2audioBufferSize(eventData.audioBufferSize);

    postMessage({
      eventType: "AUDIO_BUFFER_SIZE_SET",
    });
  },
  SET_DELAY: function (eventData) {
    js2delay(
      eventData.delayBypass,
      eventData.delay,
      eventData.delayWetLevel,
      eventData.delayDryLevel,
      eventData.delayFeedbackLevel
    );

    postMessage({
      eventType: "DELAY_SET",
    });
  },
  SET_REVERB: function (eventData) {
    js2reverb(
      eventData.reverbBypass,
      eventData.reverbComb1Delay,
      eventData.reverbComb2Delay,
      eventData.reverbComb3Delay,
      eventData.reverbComb4Delay,
      eventData.reverbAllPass1Delay,
      eventData.reverbAllPass2Delay,
      eventData.reverbSustainDelay,
      eventData.reverbDryWetMix
    );

    postMessage({
      eventType: "REVERB_SET",
    });
  },
  //
  // IEmulationSection methods
  //
  SET_ENGINE: function (eventData) {
    js2engine(eventData.engine);

    postMessage({
      eventType: "ENGINE_SET",
    });
  },
  SET_DEFAULT_EMULATION: function (eventData) {
    js2defaultEmulation(eventData.defaultEmulation);

    postMessage({
      eventType: "DEFAULT_EMULATION_SET",
    });
  },
  SET_USER_EMULATION: function (eventData) {
    js2userEmulation(eventData.userEmulation, eventData.stereoEmulation, eventData.thirdEmulation);

    postMessage({
      eventType: "USER_EMULATION_SET",
    });
  },
  SET_DEFAULT_CLOCK_SPEED: function (eventData) {
    js2defaultClockSpeed(eventData.defaultClockSpeed);

    postMessage({
      eventType: "DEFAULT_CLOCK_SPEED_SET",
    });
  },
  SET_USER_CLOCK_SPEED: function (eventData) {
    js2userClockSpeed(eventData.userClockSpeed);

    postMessage({
      eventType: "USER_CLOCK_SPEED_SET",
    });
  },
  SET_DEFAULT_CHIP_MODEL: function (eventData) {
    js2defaultChipModel(eventData.defaultSidModel);

    postMessage({
      eventType: "DEFAULT_CHIP_MODEL_SET",
    });
  },
  SET_USER_CHIP_MODEL: function (eventData) {
    js2userChipModel(eventData.userSidModel, eventData.stereoSidModel, eventData.thirdSIDModel);

    postMessage({
      eventType: "USER_CHIP_MODEL_SET",
    });
  },
  HARDSID_MAPPING: function (eventData) {
    let result = js2hardSidMapping(eventData.chipCount, eventData.hardsid6581, eventData.hardsid8580);

    postMessage({
      eventType: "HARDSID_MAPPED",
      eventData: {
        mapping: result,
      },
    });
  },
  EXSID_MAPPING: function (eventData) {
    let result = js2exSidMapping();

    postMessage({
      eventType: "EXSID_MAPPED",
      eventData: {
        mapping: result,
      },
    });
  },
  SIDBLASTER_MAPPING: function (eventData) {
    let result = js2sidBlasterMapping();

    postMessage({
      eventType: "SIDBLASTER_MAPPED",
      eventData: {
        mapping: result,
      },
    });
  },
  USBSID_MAPPING: function (eventData) {
    let result = js2usbSidMapping();

    postMessage({
      eventType: "USBSID_MAPPED",
      eventData: {
        mapping: result,
      },
    });
  },
  SET_FILTER_ENABLE: function (eventData) {
    js2filterEnable(eventData.sidNum, eventData.filterEnable);

    postMessage({
      eventType: "FILTER_ENABLE_SET",
    });
  },
  SET_FILTER_NAME: function (eventData) {
    js2filterName(eventData.emulation, eventData.chipModel, eventData.sidNum, eventData.filterName);

    postMessage({
      eventType: "FILTER_NAME_SET",
    });
  },
  SET_DIGI_BOOSTED_8580: function (eventData) {
    js2digiBoosted8580(eventData.digiBoosted8580);

    postMessage({
      eventType: "DIGI_BOOSTED_8580_SET",
    });
  },
  SET_STEREO: function (eventData) {
    js2stereo(
      eventData.stereoMode,
      eventData.dualSidBase,
      eventData.thirdSIDBase,
      eventData.fakeStereo,
      eventData.sidToRead
    );

    postMessage({
      eventType: "STEREO_SET",
    });
  },
  SET_MUTE: function (eventData) {
    js2mute(eventData.sidNum, eventData.voice, eventData.value);

    postMessage({
      eventType: "MUTE_SET",
    });
  },
  SET_DETECT_PSID64_CHIP_MODEL: function (eventData) {
    js2detectPSID64ChipModel(eventData.detectPSID64ChipModel);

    postMessage({
      eventType: "DETECT_PSID64_CHIP_MODEL_SET",
    });
  },
  //
  // IC1541Section methods
  //
  TURN_DRIVE_ON: function (eventData) {
    js2turnDriveOn(eventData.driveOn);

    postMessage({
      eventType: "DRIVE_TURNED_ON",
    });
  },
  SET_PARALLEL_CABLE: function (eventData) {
    js2parallelCable(eventData.parallelCable);

    postMessage({
      eventType: "PARALLEL_CABLE_SET",
    });
  },
  SET_JIFFY_DOS_INSTALLED: function (eventData) {
    js2jiffyDosInstalled(eventData.jiffyDosInstalled);

    postMessage({
      eventType: "JIFFY_DOS_INSTALLED_SET",
    });
  },
  SET_RAM_EXPANSION: function (eventData) {
    js2ramExpansion(
      eventData.ramExpansion0,
      eventData.ramExpansion1,
      eventData.ramExpansion2,
      eventData.ramExpansion3,
      eventData.ramExpansion4
    );

    postMessage({
      eventType: "RAM_EXPANSION_SET",
    });
  },
  SET_FLOPPY_TYPE: function (eventData) {
    js2floppyType(eventData.floppyType);

    postMessage({
      eventType: "FLOPPY_TYPE_SET",
    });
  },
  //
  // IPrinterSection methods
  //
  TURN_PRINTER_ON: function (eventData) {
    js2printerOn(eventData.printerOn);

    postMessage({
      eventType: "PRINTER_TURNED_ON",
    });
  },
  //
  // IWhatsSidSection methods
  //
  SET_WHATSSID: function (eventData) {
    js2whatsSID(
      eventData.enable,
      eventData.captureTime,
      eventData.matchStartTime,
      eventData.matchRetryTime,
      eventData.minimumRelativeConfidence
    );

    postMessage({
      eventType: "WHATSSID_SET",
    });
  },
  //
  // Business methods
  //
  SET_COMMAND: function (eventData) {
    js2typeInCommand(eventData.command ?? null);

    postMessage({
      eventType: "COMMAND_SET",
    });
  },
  TYPE_KEY: function (eventData) {
    js2typeKey(eventData.key ?? null);

    postMessage({
      eventType: "KEY_TYPED",
    });
  },
  PRESS_KEY: function (eventData) {
    js2pressKey(eventData.key ?? null);

    postMessage({
      eventType: "KEY_PRESSED",
    });
  },
  RELEASE_KEY: function (eventData) {
    js2releaseKey(eventData.key ?? null);

    postMessage({
      eventType: "KEY_RELEASED",
    });
  },
  PRESS_JOYSTICK: function (eventData) {
    js2joystick(eventData.number, eventData.value);

    postMessage({
      eventType: "JOYSTICK_PRESSED",
    });
  },
  FAST_FORWARD: function (eventData) {
    js2fastForward();

    postMessage({
      eventType: "FAST_FORWARD_SET",
    });
  },
  NORMAL_SPEED: function (eventData) {
    js2normalSpeed();

    postMessage({
      eventType: "NORMAL_SPEED_SET",
    });
  },
  GET_TUNE_INFO: function (eventData) {
    let result = js2tuneInfo();

    postMessage({
      eventType: "GOT_TUNE_INFO",
      eventData: {
        tuneInfo: result,
      },
    });
  },
  GET_PLAYLIST: function (eventData) {
    let result = js2playList();

    postMessage({
      eventType: "GOT_PLAYLIST",
      eventData: {
        playList: result,
      },
    });
  },
  GET_STATUS: function (eventData) {
    let result = js2status();

    postMessage({
      eventType: "GOT_STATUS",
      eventData: {
        status: result,
      },
    });
  },
  INSERT_DISK: function (eventData) {
    js2insertDisk(eventData.contents ?? null, eventData.diskName ?? null);

    postMessage({
      eventType: "DISK_INSERTED",
    });
  },
  EJECT_DISK: function (eventData) {
    js2ejectDisk();

    postMessage({
      eventType: "DISK_EJECTED",
    });
  },
  INSERT_TAPE: function (eventData) {
    js2insertTape(eventData.contents ?? null, eventData.tapeName ?? null);

    postMessage({
      eventType: "TAPE_INSERTED",
    });
  },
  EJECT_TAPE: function (eventData) {
    js2ejectTape();

    postMessage({
      eventType: "TAPE_EJECTED",
    });
  },
  CONTROL_DATASETTE: function (eventData) {
    js2controlDatasette(eventData.control ?? null);

    postMessage({
      eventType: "DATASETTE_CONTROLLED",
    });
  },
  INSERT_REU_FILE: function (eventData) {
    js2insertREUfile(eventData.contents ?? null, eventData.reuName ?? null);

    postMessage({
      eventType: "REU_FILE_INSERTED",
    });
  },
  INSERT_REU: function (eventData) {
    js2insertREU(eventData.sizeKb);

    postMessage({
      eventType: "REU_INSERTED",
    });
  },
  FREEZE_CARTRIDGE: function (eventData) {
    js2freezeCartridge();

    postMessage({
      eventType: "CARTRIDGE_FREEZED",
    });
  },
};

// $DEVTOOLS_SECTION_2

// Handle incoming messages
addEventListener("message", (event) => eventMap[event.data.eventType](event.data.eventData), false);
