/***************************************************************************
 *   Copyright (C) 2007 by Sindre Aamås                                    *
 *   aamas@stud.ntnu.no                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2 as     *
 *   published by the Free Software Foundation.                            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License version 2 for more details.                *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   version 2 along with this program; if not, write to the               *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "gambatte.h"
#include "cpu.h"
#include "savestate.h"
#include "statesaver.h"
#include "initstate.h"
#include "state_osd_elements.h"
#include <sstream>
#include <cstring>

//
// Modified 2012-07-10 to 2012-07-14 by H. Ilari Liusvaara
//	- Make it rerecording-friendly.

static const std::string itos(const int i) {
	std::stringstream ss;
	ss << i;
	return ss.str();
}

static const std::string statePath(const std::string &basePath, const int stateNo) {
	return basePath + "_" + itos(stateNo) + ".gqs";
}

namespace
{
	time_t default_walltime()
	{
		return time(0);
	}
}

namespace gambatte {
struct GB::Priv {
	CPU cpu;
	int stateNo;
	bool gbaCgbMode;
	
	Priv(time_t (**_getCurrentTime)()) : stateNo(1), gbaCgbMode(false), cpu(_getCurrentTime) {}
};
	
GB::GB() : p_(new Priv(&walltime)), walltime(default_walltime) {}

GB::~GB() {
	if (p_->cpu.loaded())
		p_->cpu.saveSavedata();
	
	delete p_;
}

signed GB::runFor(gambatte::uint_least32_t *const videoBuf, const int pitch,
			gambatte::uint_least32_t *const soundBuf, unsigned &samples) {
	if (!p_->cpu.loaded()) {
		samples = 0;
		return -1;
	}
	
	p_->cpu.setVideoBuffer(videoBuf, pitch);
	p_->cpu.setSoundBuffer(soundBuf);
	const signed cyclesSinceBlit = p_->cpu.runFor(samples * 2);
	samples = p_->cpu.fillSoundBuffer();
	
	return cyclesSinceBlit < 0 ? cyclesSinceBlit : static_cast<signed>(samples) - (cyclesSinceBlit >> 1);
}

void GB::reset() {
	if (p_->cpu.loaded()) {
		p_->cpu.saveSavedata();
		
		SaveState state;
		p_->cpu.setStatePtrs(state);
		setInitState(state, p_->cpu.isCgb(), p_->gbaCgbMode, walltime());
		p_->cpu.loadState(state);
		p_->cpu.loadSavedata();
	}
}

void GB::setInputGetter(InputGetter *getInput) {
	p_->cpu.setInputGetter(getInput);
}

void GB::setSaveDir(const std::string &sdir) {
	p_->cpu.setSaveDir(sdir);
}

void GB::preload_common()
{
	if (p_->cpu.loaded())
		p_->cpu.saveSavedata();
}

void GB::postload_common(const unsigned flags)
{
	SaveState state;
	p_->cpu.setStatePtrs(state);
	setInitState(state, p_->cpu.isCgb(), p_->gbaCgbMode = flags & GBA_CGB, walltime());
	p_->cpu.loadState(state);
	p_->cpu.loadSavedata();

	p_->stateNo = 1;
	p_->cpu.setOsdElement(std::auto_ptr<OsdElement>());
}

int GB::load(const std::string &romfile, const unsigned flags) {
	preload_common();

	const int failed = p_->cpu.load(romfile, flags & FORCE_DMG, flags & MULTICART_COMPAT);

	if (!failed)
		postload_common(flags);

	return failed;
}

int GB::load(const unsigned char* image, size_t isize, unsigned flags) {
	preload_common();

	const int failed = p_->cpu.load(image, isize, flags & FORCE_DMG, flags & MULTICART_COMPAT);

	if (!failed)
		postload_common(flags);

	return failed;
}

bool GB::isCgb() const {
	return p_->cpu.isCgb();
}

bool GB::isLoaded() const {
	return p_->cpu.loaded();
}

void GB::setDmgPaletteColor(unsigned palNum, unsigned colorNum, unsigned rgb32) {
	p_->cpu.setDmgPaletteColor(palNum, colorNum, rgb32);
}

void GB::loadState(const std::string &filepath, const bool osdMessage) {
	if (p_->cpu.loaded()) {
		p_->cpu.saveSavedata();
		
		SaveState state;
		p_->cpu.setStatePtrs(state);
		
		if (StateSaver::loadState(state, filepath)) {
			p_->cpu.loadState(state);
			
			if (osdMessage)
				p_->cpu.setOsdElement(newStateLoadedOsdElement(p_->stateNo));
		}
	}
}

void GB::saveState(const gambatte::uint_least32_t *const videoBuf, const int pitch) {
	if (p_->cpu.loaded()) {
		saveState(videoBuf, pitch, statePath(p_->cpu.saveBasePath(), p_->stateNo));
		p_->cpu.setOsdElement(newStateSavedOsdElement(p_->stateNo));
	}
}

void GB::loadState() {
	loadState(statePath(p_->cpu.saveBasePath(), p_->stateNo), true);
}

void GB::saveState(const gambatte::uint_least32_t *const videoBuf, const int pitch, const std::string &filepath) {
	if (p_->cpu.loaded()) {
		SaveState state;
		p_->cpu.setStatePtrs(state);
		p_->cpu.saveState(state);
		StateSaver::saveState(state, videoBuf, pitch, filepath);
	}
}

void GB::loadState(const std::string &filepath) {
	loadState(filepath, false);
}

void GB::saveState(std::vector<char>& data, const std::vector<char>& cmpdata) {
	if (p_->cpu.loaded()) {
		loadsave_save l(cmpdata);
		p_->cpu.loadOrSave(l);
		data = l.get();
	}
}

void GB::saveState(std::vector<char>& data) {
	if (p_->cpu.loaded()) {
		loadsave_save l;
		p_->cpu.loadOrSave(l);
		data = l.get();
	}
}

void GB::loadState(const std::vector<char>& data) {
	if (p_->cpu.loaded()) {
		loadsave_load l(data);
		p_->cpu.loadOrSave(l);
	}
}

void GB::selectState(int n) {
	n -= (n / 10) * 10;
	p_->stateNo = n < 0 ? n + 10 : n;
	
	if (p_->cpu.loaded())
		p_->cpu.setOsdElement(newSaveStateOsdElement(statePath(p_->cpu.saveBasePath(), p_->stateNo), p_->stateNo));
}

int GB::currentState() const { return p_->stateNo; }

const std::string GB::romTitle() const {
	if (p_->cpu.loaded()) {
		char title[0x11];
		std::memcpy(title, p_->cpu.romTitle(), 0x10);
		title[(title[0xF] & 0x80) ? 0xF : 0x10] = '\0';
		return std::string(title);
	}
	
	return std::string();
}

void GB::setGameGenie(const std::string &codes) {
	p_->cpu.setGameGenie(codes);
}

void GB::setGameShark(const std::string &codes) {
	p_->cpu.setGameShark(codes);
}

void GB::setRtcBase(time_t time) {
	p_->cpu.setRtcBase(time);
}

time_t GB::getRtcBase() {
	return p_->cpu.getRtcBase();
}

std::pair<unsigned char*, size_t> GB::getWorkRam() {
	return p_->cpu.getWorkRam();
}

std::pair<unsigned char*, size_t> GB::getSaveRam() {
	return p_->cpu.getSaveRam();
}

std::pair<unsigned char*, size_t> GB::getIoRam() {
	return p_->cpu.getIoRam();
}

std::pair<unsigned char*, size_t> GB::getVideoRam() {
	return p_->cpu.getVideoRam();
}

void GB::set_walltime_fn(time_t (*_walltime)())
{
	walltime = _walltime;
}

std::string GB::version()
{
	return "SVN320";
}

}
