// *********************************************************
// ***                 achtbit rom                       ***
// ***       (c) 2012 by Christian Wahlmann              ***
// *********************************************************


// ******************* system adresses *********************
// start of the input buffer, an alist (array list)
D200.		sys_input_buffer:
// start of the charset (256*10 byte)
D600.		sys_charset:
// start of the screen ram
E000.		sys_screen:
// start of the text screen color table
E400.		sys_screen_color:
// start of the palette definition (16*3 bytes rgb)
FF40.		sys_palette:
// the screen mode register
FF70.		sys_screen_mode:
// the monocrome graphics mode color register
FF71.		sys_gfx_color:
// set to 1 if the screen is refreshed (50 Herz)
FF72.		sys_screen_refresh:
// when the screen interrupt is started, it calls sys_screen_interrupt:
// on startup it contains the command 'RETI' (0x03)
// to install your own routine, change it to 'JP myRoutine' (0x40 n n)
FF73.		sys_screen_interrupt:
// the text cursor position
FF76.		sys_cursor_x:
FF77.		sys_cursor_y:
// the text color (foreground and background)
FF78.		sys_text_color:
// when 1, a cursor is displayed
FF79.		sys_enable_cursor:
// stores the position, where to insert chars into the buffer
FF7A.		sys_input_buffer_pointer:
// speed of the cursor blinking in 50/n Herz
FF7C.		sys_cursor_speed:
// stores the system time
FF7D.		sys_time_sec:
FF7E.		sys_time_min:
FF7F.		sys_time_hour:
FF80.		sys_date_day:
FF81.		sys_date_month:
FF82.		sys_date_year:
// vars for the sound system
FF84.		sys_sound_speed:  // beats per minute (not yet implemented)
FF85.		sys_sound_key: // 0..11
FF86.		sys_sound_pitch: // -128..127
FF87.		sys_sound_length: // length-unit (f.e. 4 = 1/4)
FF88.		sys_sound_rythm: // type of broken rythm, stored for the second note
					            // 0=none, 1=1/2, 3=3/2
FF89.		sys_sound_beat: // length, unit (f.e. 6,4 = 6/4)
FF8B.		sys_sound_keytable: // translates the notes [0..6] to half tones [0..11]
FF92.		sys_sound_repeat: // pointer to the begin of a repeat (0 if none)
FF94.		sys_sound_triole_type: // 0=none, 3=triole (3 in 2), 4= (4 in 3)
FF95.		sys_sound_triole_count: // number of notes remaining in the triole

// FF96...

.run 0x0000
.org 0x0000
	JP init:

// **************** print char ********************
// prints a char at the cursor position
//
// in: A = char
// out:
// used: HL, DE, BC

print_char:
	CMP A,0x0A
	JR Z zeilenumbruch:
	CALL cursor_adr:
	ADD HL,sys_screen_color:
	LD D,(sys_text_color:)
	LD (HL),D
	SUB HL,0x0400
	CMP A,0x08
	JR NZ no_backspace:
	CMP HL,sys_screen:
	RET Z
	DEC HL
	JR cursor_left:
no_backspace:
	LD (HL),A
	JR cursor_right:

// ***************** zeilenumbruch ****************
// move the cursor to the next line
//
// in:
// out:
// used: HL, DE, BC

zeilenumbruch:
	XOR L,L
	LD (sys_cursor_x:),L
	JR cursor_down:
	
// ***************** cursor left ****************
// move the cursor to the left
//
// in:
// out:
// used: HL

cursor_left:
	LD L,(sys_cursor_x:)
	OR L,L
	JR Z cursor_next1:
	DEC L
	LD (sys_cursor_x:),L
	RET
cursor_next1:
	LD L,39
	LD (sys_cursor_x:),L

// ***************** cursor up ****************
// move the cursor up
//
// in:
// out:
// used: HL

cursor_up:
	LD H,(sys_cursor_y:)
	OR H,H
	RET Z
	DEC H
	LD (sys_cursor_y:),H
	RET
	
// ***************** cursor right ****************
// move the cursor to the right
//
// in:
// out:
// used: HL, DE, BC

cursor_right:
	LD L,(sys_cursor_x:)
	CMP L,39
	JR Z cursor_next2:
	INC L
	LD (sys_cursor_x:),L
	RET
cursor_next2:
	LD L,0
	LD (sys_cursor_x:),L

// ***************** cursor down ****************
// move the cursor to the down
//
// in:
// out:
// used: HL, DE, BC

cursor_down:
	LD H,(sys_cursor_y:)
	CMP H,19
	JR Z scroll_up:
	INC H
	LD (sys_cursor_y:),H
	RET	

// ***************** scroll up ****************
// scroll the text screen up
//
// in:
// out:
// used: HL, DE, BC

scroll_up:
	PUSH AF
	LD DE,sys_screen:
	LD BC,DE
	ADD BC,0x0028
	LD HL,0x02f8
scroll_loop1:
	LD A,(BC)
	LD (DE),A
	INC DE
	INC BC
	DEC HL
	JR NZ scroll_loop1:
	LD l,0x28	
scroll_loop2:
	LD (DE),0x20
	INC DE
	DEC L
	JR NZ scroll_loop2:

	LD DE,sys_screen_color:
	LD BC,DE
	ADD BC,0x0028
	LD HL,0x02f8
scroll_loop3:
	LD A,(BC)
	LD (DE),A
	INC DE
	INC BC
	DEC HL
	JR NZ scroll_loop3:
	LD l,0x28
	LD A,(sys_text_color:)	
scroll_loop4:
	LD (DE),A
	INC DE
	DEC L
	JR NZ scroll_loop4:
	POP AF
	RET

// ***************** print str ****************
// print a null terminated string
//
// in: HL (String adress)
// out: HL (pointing to the end of the string)
// used: AF

print_str:
	LD A,(HL)
	OR A,A
	RET Z
	PUSH HL
	CALL print_char:
	POP HL
	INC HL
	JR print_str:

// ***************** clear screen ****************
// clear the screen
//
// in:
// out:
// used: HL, DE, BC, F


clear_scr:
	LD C,0x20

// ***************** fill screen ****************
// fill the screen with a char
//
// in: C (char)
// out:
// used: HL, DE, B, F

fill_scr:
	LD B,(sys_text_color:)
	LD HL,sys_screen:
	LD DE,sys_screen_color:
fill_scr_loop1:
	LD (DE),B
	LD (HL),C
	INC HL
	INC DE
	CMP HL,0xE320
	JR NZ fill_scr_loop1:
	RET

// ******************* alist - Array Byte List ***********************
// HL - points to the list
// BC - the index
// A  - the value
// HL(0): max. length
// HL(2): length
// HL(4): values

// ****************** alist clear *****************************
// clear a list
// in: HL (pointer to list)
// out:
// used:

alist_clear:
	PUSH HL
	INC HL
	INC HL
	LD (HL),0x00 // set the length to 0
	INC HL
	LD (HL),0x00
	POP HL
	RET

// **************** alist append **********************
// append a value
// in: HL (pointer to list)
//       A (value to append)
// out: Z-Flag, when list was full
// used: DE, BC

alist_append:
	PUSH HL
	LD DE,(HL) // get max. length
	INC HL
	INC HL
alist_append1:
	LD BC,(HL) // get length
	CMP DE,BC  // max. length==length?
	JR Z alist_next1:  // list is full; return
	LD DE,HL
	INC DE
	INC DE // points now to the first entry
	ADD DE,BC // find the next free cell
	LD (DE),A // store A
	INC BC	// length+=1
	LD (HL),BC
alist_next1:
	POP HL
	RET

// **************** alist append **********************
// get last value
// in: HL (pointer to list)
// out: Z-Flag, when list was empty
//       A (value)
// used: DE, BC

alist_get_last:
	PUSH HL
	INC HL
	INC HL
	LD BC,(HL)
	XOR A,A
	OR BC,BC 	// is empty?
	JR Z alist_next2:
	DEC BC
	LD (HL),BC
	INC HL
	INC HL
	ADD HL,BC
	LD A,(HL)
alist_next2:
	POP HL
	RET

// **************** alist insert **********************
// insert value at given pos
// in: HL (pointer to list)
//       BC (position)
//       A (value)
// out: Z-Flag, when list was empty
// used: DE, BC

alist_insert:
	PUSH HL
	LD DE,(HL)
	INC HL
	INC HL
	DEC DE
	CMP DE,(HL)
	JR C alist_next3: // list is full
	CMP BC,(HL)
	JR NC alist_append1: // index is out of bounds
	// shift values
	LD DE,(HL)
	INC DE
	LD (HL),DE // length+=1
	INC HL
	INC HL
	ADD DE,HL
	ADD BC,HL
	PUSH AF
alist_loop1:
	DEC DE
	LD A,(DE)
	INC DE
	LD (DE),A
	DEC DE
	CMP DE,BC
	JR NZ alist_loop1:
	
	POP AF
	LD (DE),A
alist_next3:
	POP HL
	RET

// **************** alist get at **********************
// get value at given pos
// in: HL (pointer to list)
//       BC (position)
// out: C-Flag, when BC is out of bounds
//       A (value)
// used: DE, BC

alist_get_at:
	PUSH HL
	INC HL
	INC HL
	LD DE,(HL) // length
	CMP DE,BC
	JR C  alist_next4: // index is out of bounds
	INC HL
	INC HL
	ADD HL,BC
	LD A,(HL)
alist_next4:
	POP HL
	RET

// **************** alist get at **********************
// remove value at given pos
// in: HL (pointer to list)
//       BC (position)
// out: C-Flag, when BC is out of bounds
// used: DE, BC

alist_remove_at:
	PUSH HL
	INC HL
	INC HL
	LD DE,(HL) // length
	DEC DE
	CMP DE,BC
	JR C  alist_next6: // index is out of bounds
	LD (HL),DE
	INC HL
	INC HL
	ADD DE,HL
	ADD BC,HL
	LD A,(BC)
	PUSH A
alist_loop2:
	CMP BC,DE
	JR Z alist_next5:
	INC BC
	LD A,(BC)
	DEC BC
	LD (BC),A
	INC BC
	JR alist_loop2:
alist_next5:
	POP A
alist_next6:
	POP HL
	RET

// **************** alist test *******************
test_alist:
	LD HL,0xD200
	LD DE,5
	LD (HL),DE
	CALL alist_clear:
	HLT
	LD A,0x7f
	CALL alist_append:
	LD A,0x30
	CALL alist_append:
	LD A,0x31
	CALL alist_append:
	HLT
	LD A,0x42
	LD BC,0x001
	CALL alist_insert:
	HLT
	LD BC,3
	CALL alist_remove_at:
	HLT

// **************** clear buf *******************
// clear the input buffer

clear_buf:
	LD HL,sys_input_buffer:
	LD BC,0x03fc
	LD (HL),BC
	CALL alist_clear:
	LD HL,0
	LD (sys_input_buffer_pointer:),HL
	RET

// **************** buf insert char *******************
// insert a char into the input buffer

// in: A (char)
// out:
// used: HL, BC, DE

buf_ins_char:
	LD HL,sys_input_buffer:
	LD BC,(sys_input_buffer_pointer:)
	PUSH BC
	CALL alist_insert:
	POP BC
	RET C // list was full
	INC BC
	LD (sys_input_buffer_pointer:),BC
	RET

// **************** buf insert char *******************
// delete a char from the right in the input buffer

// in:
// out:
// used: HL, BC, DE

buf_del_right_char:
	LD HL,sys_input_buffer:
	LD BC,(sys_input_buffer_pointer:)
	CALL alist_remove_at:
	RET

// **************** buf insert char *******************
// delete a char from the left in the input buffer

// in:
// out:
// used: HL, BC, DE

buf_del_left_char:
	LD HL,sys_input_buffer:
	LD BC,(sys_input_buffer_pointer:)
	CMP BC,1
	RET C // already most left pos
	DEC BC
	LD (sys_input_buffer_pointer:),BC
	CALL alist_remove_at:
	RET
	
// **************** input *******************
// input a line of text from keyboard into the input buffer
//
// in:
// out:
// used: AF, HL, BC, DE

input:
	CALL clear_buf:
input_loop1: 
	IN A,(00)
	OR A,A
	JR Z input_loop1:

	CMP A,0x9B // --- insert
	JR NZ input_next1n:
	LD a,0x20
	CALL input_insert_char:
	LD BC,(sys_input_buffer_pointer:)
	OR BC,BC
	JR Z input_loop1:
	DEC BC
	LD (sys_input_buffer_pointer:),BC
	CALL cursor_left:	
	JR input_loop1:

input_next1n:
	CMP A,0x7F // --- delete
	JR NZ input_next0:
	CALL buf_del_right_char:
	JR C input_loop1:
input_next01:
	LD HL,sys_input_buffer:
	INC HL
	INC HL
	LD DE,(HL)
	PUSH DE
	CALL cursor_adr:
	ADD HL,sys_screen:
	POP DE
	LD BC,(sys_input_buffer_pointer:)
	SUB DE,BC
	JR Z input_next00:
input_loop4:
	INC HL
	LD A,(HL)
	DEC HL
	LD (HL),A
	INC HL
	DEC DE
	JR NZ input_loop4:
input_next00:
	LD (HL),0x20
	JR input_loop1:
input_next0:
	CMP A,0x08 // --- Backspace
	JR NZ input_next1:
	CALL buf_del_left_char:
	JR C input_loop1:
	CALL cursor_left:
	JR input_next01:
input_next1:
	CMP A,0x15 // left
	JR NZ input_next1a:
	LD BC,(sys_input_buffer_pointer:)
	OR BC,BC
	JR z input_loop1: // allready pos1
	DEC BC
	LD (sys_input_buffer_pointer:),BC
	CALL cursor_left:
	jr input_loop1:
input_next1a:
	CMP A,0x17 // right
	JR NZ input_next1b:
	LD BC,(sys_input_buffer_pointer:)
	LD HL,sys_input_buffer:
	INC HL
	INC HL
	CMP BC,(HL)
	JP z input_loop1: // allready end pos
	INC BC
	LD (sys_input_buffer_pointer:),BC
	CALL cursor_right:
	JP input_loop1:
input_next1b:
	CMP A,0x13 // End
	JR NZ input_next1c:
	LD HL,sys_input_buffer:
	INC HL
	INC HL
	LD DE,(HL)
	LD BC,(sys_input_buffer_pointer:)
	LD (sys_input_buffer_pointer:),DE
	SUB DE,BC
input_loop1c:
	CALL cursor_right:
	DEC DE
	JR NZ input_loop1c:
	JP input_loop1:
input_next1c:
	CMP A,0x14 // pos1
	JR NZ input_next1d:
	LD BC,(sys_input_buffer_pointer:)
input_loop1d:
	CALL cursor_left:
	DEC BC
	JR NZ input_loop1d:
	LD (sys_input_buffer_pointer:),BC
	JP input_loop1:
input_next1d:
	CMP A,0x0A // enter
	RET Z
	CMP A,0x20 // any other function key?
	JP C input_loop1:
	OR A,A
	CALL NZ input_insert_char:
	JP input_loop1:
	
input_insert_char:
	CALL buf_ins_char:
	LD HL,sys_input_buffer:
	INC HL
	INC HL
	LD BC,(sys_input_buffer_pointer:)
	CMP BC,(HL)
	JR z input_next3:
	PUSH A
	LD DE,(HL)
	SUB DE,BC
	PUSH DE
	CALL cursor_adr:
	ADD HL,sys_screen:
	POP DE
	ADD HL,DE
input_loop2:
	DEC HL
	LD A,(HL)
	INC HL
	LD (HL),A
	DEC HL
	DEC DE
	JR NZ input_loop2:
	POP A
input_next3:
	CALL print_char:
	RET

// **************** byte_hex *******************
// print a hexadecimal digit

// in: A (integer)
// out:
// used: HL, BC, DE

byte_hex:
	CMP A,0x0A
	JR NC byte_hex_next1:
	ADD A,0x30
	JR byte_hex_next2:
byte_hex_next1:
	ADD A,0x37
byte_hex_next2:
	CALL print_char:
	RET

// **************** word hex *******************
// print a hexadecimal byte

// in: A (integer)
// out:
// used: HL, BC, DE

word_hex:
	PUSH AF
	SHR A
	SHR A
	SHR A
	SHR A
	CALL byte_hex:
	POP AF
	AND A,0x0F
	CALL byte_hex:
	RET

// **************** cursor adr ****************
// calculate the memory adress of the cursor
// screen or screen_color must still be added
// in:
// out: HL (cursor adress)
// used: DE

cursor_adr:
	LD L,(sys_cursor_y:)
	LD H,0x00
	MUL HL,0x0028
	LD E,(sys_cursor_x:)
	LD D,0x00
	ADD HL,DE
	RET

// **************** word dec *******************
// print a decimal word

// in: HL (integer)
//       B (number of digits)
// out: 
// used: HL, BC, DE

word_dec:
	LD E,0
	PUSH E
word_dec_loop1:
	LD DE,HL
	MOD DE,0x000A
	ADD E,0x30
	PUSH E
	DIV HL,0x000A
	DEC B
	JR NZ word_dec_loop1:
word_dec_loop2:
	POP A
	OR A,A
	RET Z
	CALL print_char:
	JR word_dec_loop2:

// **************** date time *******************
// print data and time in the right upper corner of the screen

// in:
// out: 
// used: AF, HL, BC, DE

date_time:
	LD L,(sys_date_day:)
	LD H,0x00
	LD B,0x02
	CALL word_dec:
	LD A,'.'
	CALL print_char:
	
	LD L,(sys_date_month:)
	LD H,0x00
	LD B,0x02
	CALL word_dec:
	LD A,'.'
	CALL print_char:

	LD HL,(sys_date_year:)
	LD B,0x04
	CALL word_dec:
	LD A,0x20
	CALL print_char:

	LD L,(sys_time_hour:)
	LD H,0x00
	LD B,0x02
	CALL word_dec:
	LD A,':'
	CALL print_char:

	LD L,(sys_time_min:)
	LD H,0x00
	LD B,0x02
	CALL word_dec:
	LD A,'.'
	CALL print_char:

	LD L,(sys_time_sec:)
	LD H,0x00
	LD B,0x02
	CALL word_dec:

	RET


// **************** random *******************
// print a decimal word

// in: A = max
// out: A = random[0..max-1]
// used: 

random:
	PUSH HL
	PUSH DE
	IN L,(1)
	LD H,0
	LD E,A
	LD D,0
	MUL HL,DE
	LD A,H
	POP DE
	POP HL
	RET

// ***************** init **********************
// init the system

init: 
	LD HL,int_func:
	LD (0xFF74),HL
	LD A,0x40
	LD (0xFF73),A
	JR test_input:

// *************** int func ********************
// displays and updates date and time

int_func:
	PUSH HL
	PUSH DE
	PUSH BC
	PUSH AF
	LD HL,(sys_cursor_x:)
	PUSH HL
	LD A,(sys_text_color:)
	PUSH A
	LD HL,0x0015
	LD (sys_cursor_x:),HL
	LD A,0xBA
	LD (sys_text_color:),A
	CALL date_time:
	POP A
	LD (sys_text_color:),A
	POP HL
	LD (sys_cursor_x:),HL
	POP AF
	POP BC
	POP DE
	POP HL
	RETI

// ************** test input ***************
// tests the input line routine

test_input:
	call clear_scr:
	LD HL,hello_str:
	CALL print_str:
	CALL sound_test:
test_input_lp1:
	LD A,0x0A
	CALL print_char:
	CALL print_char:
	LD HL,input_msg:
	CALL print_str:
	CALL input:
	LD HL,output_msg:
	CALL print_str:
	LD HL,sys_input_buffer:
	INC HL
	INC HL
	LD BC,(HL)
	OR BC,BC
	JR Z test_input_lp1:
	INC HL
	INC HL
test_input_loop2:
	LD A,(HL)
	PUSH HL
	PUSH BC
	CALL print_char:
	POP BC
	POP HL
	INC HL
	DEC BC
	JR NZ test_input_loop2:
	LD HL,output_msg_end:
	CALL print_str:
	JR test_input_lp1:

hello_str:
	db 0x0a
	db "    *******************************" 0x0a
	db "    *    H E L L O                *" 0x0a
	db "    *         W O R L D           *" 0x0a
	db "    * (c) 2012 Christian Wahlmann *" 0x0a
	db "    *******************************" 0x0a
	db 0x0a
	db 0
input_msg:
	db "8bit> " 0
output_msg:
	db 0x0a '"' 0
output_msg_end:
	db '"' 0

// *********** zeichen ********************
// reads keys and prints the hex code

zeichen:
	IN A,(0)
	OR A,A
	JR Z zeichen:
	CALL word_hex:
	ld a,10
	Call print_char:
	JR zeichen:

// ************** sound **********************
// ************** sound calc keytable *************
// IN:
//   (sys_sound_key:) = the key [0..11]
// OUT:
//   (sys_sound_keytable:)[0..6] = the halftones corresponding to the key

sound_basic_keytable:
	db 0 // C
	db 2 // D
	db 4 // E
	db 5 // F
	db 7 // G
	db 9 // A
	db 11 // B

sound_calc_keytable:
	LD C,(sys_sound_key:)
	LD HL,sound_basic_keytable:
	LD DE,sys_sound_keytable:
	INC C // +1
	SHR C // :2
	XOR B,B
	ADD DE,BC
	LD B,(sys_sound_key:)
	LD C,7
sound_calc_keytable_loop1:
	LD A,(HL)
	ADD A,B
	// overflow?
	CMP A,12 // no. half tones
	JR NC sound_calc_keytable_next1: // yes. jump to the routine, that subs 12
	LD (DE),A
	INC HL
	INC DE
	DEC C
	JR NZ sound_calc_keytable_loop1:
	RET
	
sound_calc_keytable_next1:
	LD DE,sys_sound_keytable: // jump to the beginning of the table
	JR sound_calc_keytable_next2:

sound_calc_keytable_loop2: // in this loop A+B always result in a overflow
	LD A,(HL)
	ADD A,B
sound_calc_keytable_next2:
	SUB A,12		// so cool it down an octave
	LD (DE),A
	INC HL
	INC DE
	DEC C
	JR NZ sound_calc_keytable_loop2:
	RET

// ************** sound parser whitespace *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   just ignores any whitespace: space and newline
//
sound_parser_whitespace:
	CMP A,0x20
	JR Z sound_parser_whitespace_next1:
	CMP A,0x0a
	RET NZ
sound_parser_whitespace_next1:
	LD A,(HL)
	INC HL
	JR sound_parser_whitespace:
	
// ************** sound parser rawnote *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   C = note
//   B = oktave (0,1)
sound_parser_rawnote:
	LD B,0 // octave 0
	LD C,A
	LD A,(HL)
	INC HL
	CMP C,'a'
	JR C sound_parser_rawnote_next0:
	CMP C,'h'
	JR NC sound_parser_rawnote_next0:
	
	LD B,1 // octace 1
	SUB C,'a'
	ADD C,'A'
	JR sound_parser_rawnote_next3:
	
sound_parser_rawnote_next0:
	CMP C,'A' // <'A'?
	JR C sound_parser_rawnote_error:
	CMP C,'H' // >='H'?
	JR NC sound_parser_rawnote_error:

sound_parser_rawnote_next3:
	CMP C,'C' // >='C'?
	JR NC sound_parser_rawnote_next1:
	SUB C,'A'
	ADD C,5
	RET
	
sound_parser_rawnote_next1:
	SUB C,'C'
	RET
	
sound_parser_rawnote_error:
	LD C,0xff
	RET

// ************** sound parser key *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets the key and recalculates the sys_sound_keytable:
//
sound_parser_key:
	// read ahead
	LD A,(HL)
	INC HL
	OR A,A // is it null?
	RET Z
	CMP A,':'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_whitespace:
	CALL sound_parser_rawnote: // (c=0..6)
	CMP C,0xff // error code
	RET Z
	LD B,0
	ADD BC,sound_basic_keytable:
	LD C,(BC)
	// b?
	CMP A,'b'
	JR NZ sound_parser_key_next2:
	LD A,(HL)
	INC HL
	ADD C,11
	JR sound_parser_key_next3:
sound_parser_key_next2:
	CMP A,'#'
	JR NZ sound_parser_key_next3:
	LD A,(HL)
	INC HL
	INC C
sound_parser_key_next3:
	CMP A,'m' // moll? just add 3 halfnotes (A-moll = C-dur)
	JR NZ sound_parser_key_next1:
	LD A,(HL)
	INC HL
	ADD C,3
sound_parser_key_next1:
	MOD C,12
	LD (sys_sound_key:),C
	PUSH HL
	PUSH A
	CALL sound_calc_keytable:
	POP A
	POP HL
	RET

// ************** sound parser number *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   D = result
//
sound_parser_number:
	LD D,0 // result
sound_parser_number_loop1:
	CMP A,0x30
	RET C // <'0': ready
	CMP A,0x3a
	RET NC // >='9'+1
	SUB A,0x30
	MUL D,10
	ADD D,A
	LD A,(HL)
	INC HL
	JR sound_parser_number_loop1:
	
// ************** sound parser meter *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets sys_sound_beat:
//
sound_parser_meter:
	// read ahead
	LD A,(HL)
	INC HL
	CMP A,':'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_whitespace:
	CALL sound_parser_number:
	LD C,D
	CMP A,'/'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_number:
	LD B,D
	LD (sys_sound_beat:),BC
	RET

// ************** sound parser length *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets sys_sound_length:
//
sound_parser_length:
	// read ahead
	LD A,(HL)
	INC HL
	CMP A,':'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_whitespace:
	CMP A,'1'
	RET NZ
	LD A,(HL)
	INC HL
	CMP A,'/'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_number:
	LD (sys_sound_length:),D
	RET

// ************** sound parser tempo *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets sys_sound_speed:
//
sound_parser_tempo:
	// read ahead
	LD A,(HL)
	INC HL
	CMP A,':'
	RET NZ
	LD A,(HL)
	INC HL
	CALL sound_parser_whitespace:
	CALL sound_parser_number:
	LD (sys_sound_speed:),D
	OUT (6),D
	RET

// ************** sound parser triole *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets sys_sound_triole_type: and sys_sound_triole_count:
//
sound_parser_triole:
	// read ahead
	LD A,(HL)
	INC HL
	CMP A,'3'
	JR Z sound_parser_triole_next1:
	CMP A,'4'
	RET NZ
sound_parser_triole_next1:
	SUB A,'0'
	LD (sys_sound_triole_type:),A
	LD (sys_sound_triole_count:),A
	LD A,(HL)
	INC HL
	RET

// ************** sound parser bar *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   sets sound_parser_repeat:
//
sound_parser_bar:
	CMP A,'|'
	JR NZ sound_parser_bar_next1:
	LD A,(HL)
	INC HL
	CMP A,'|'
	JR Z sound_parser_bar_repeatOrNew: // "||"
	CMP A,']'
	JR Z sound_parser_bar_repeat: // "|]"
	CMP A,':'
	JR Z sound_parser_bar_new: // "|:"
	RET // "|" does nothing!
	
sound_parser_bar_next1:
	CMP A,':'
	RET NZ
	LD A,(HL)
	INC HL
	CMP A,':'
	JR Z sound_parser_bar_repeatOrNew: // "::"
	CMP A,'|'
	JR Z sound_parser_bar_next2:
	DEC HL
	JR sound_parser_bar_repeat: // ":"
sound_parser_bar_next2:
	LD A,(HL)
	INC HL
	CMP A,':'
	JR Z sound_parser_bar_repeatOrNew: // ":|:"
	CMP A,'|'
	JR Z sound_parser_bar_next3:
	DEC HL
	JR sound_parser_bar_repeat: // ":|"
sound_parser_bar_next3:
	LD A,(HL)
	INC HL
	LD A,(HL)
	CMP A,':'
	JR Z sound_parser_bar_repeatOrNew: // ":||:"
	DEC HL
	JR sound_parser_bar_repeatOrNew: // ":||"

sound_parser_bar_repeatOrNew:
	LD DE,(sys_sound_repeat:)
	CMP DE,0
	JR Z sound_parser_bar_new:
	JR sound_parser_bar_repeat:

sound_parser_bar_repeat:
	LD HL,(sys_sound_repeat:)
	LD DE,0
	LD (sys_sound_repeat:),DE
	LD A,(HL)
	INC HL
	RET

sound_parser_bar_new:
	LD (sys_sound_repeat:),HL
	LD A,(HL)
	INC HL
	RET


// ************** sound parser note *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   plays the given note
//

sound_parser_note:
	CALL sound_parser_rawnote: // (c=0..6)
	LD E,12
	MUL E,B // octave offset
	ADD E,60
	LD B,0
	ADD BC,sys_sound_keytable:
	LD C,(BC)
	ADD C,E

	// octave modifications?
	CMP A,0x27 // " ' "
	JR NZ sound_parser_note_next6:
sound_parser_note_loop1:
	CMP C,243
	JR NC sound_parser_note_next7:
	ADD C,12
	LD A,(HL)
	INC HL
	CMP A,0x27
	JR Z sound_parser_note_loop1:
	JR sound_parser_note_next7:

sound_parser_note_next6:
	CMP A,0x2c // " , "
	JR NZ sound_parser_note_next7:
sound_parser_note_loop2:
	CMP C,12
	JR C sound_parser_note_next7:
	SUB C,12
	LD A,(HL)
	INC HL
	CMP A,0x2c
	JR Z sound_parser_note_loop2:
sound_parser_note_next7:

sound_parser_note_length:
	// default length
	LD D,1 // length
	LD E,(sys_sound_length:)
	// any length given?
	CMP A,0x31
	JR C sound_parser_note_next1:
	CMP A,0x3a
	JR NC sound_parser_note_next1:
	CALL sound_parser_number:
sound_parser_note_next1:
	CMP A,'/'
	JR NZ sound_parser_note_next2:
	LD A,(HL)
	INC HL
	// any length_unit given?
	CMP A,0x31
	JR C sound_parser_note_next2:
	CMP A,0x3a
	JR NC sound_parser_note_next2:
	PUSH D
	CALL sound_parser_number:
	MUL E,D
	POP D
sound_parser_note_next2:
	// check for broken rythm
	LD B,(sys_sound_rythm:)
	OR B,B
	JR Z sound_parser_note_next0:
	MUL D,B
	MUL E,2
	LD B,0
	LD (sys_sound_rythm:),B
	JR sound_parser_note_next4:
sound_parser_note_next0:	
	CMP A,'<'
	JR NZ sound_parser_note_next3:
	MUL E,2
	LD A,3
	LD (sys_sound_rythm:),A
	LD A,(HL)
	INC HL
	JR sound_parser_note_next4:
sound_parser_note_next3:
	CMP A,'>'
	JR NZ sound_parser_note_next4:
	MUL D,3
	MUL E,2
	LD A,1
	LD (sys_sound_rythm:),A
	LD A,(HL)
	INC HL
sound_parser_note_next4:
	// check for triole
	LD B,(sys_sound_triole_type:)
	CMP B,3
	JR Z sound_parser_note_triole:
	CMP B,4
	JR NZ sound_parser_note_next5:
sound_parser_note_triole:
	MUL E,B
	DEC B
	MUL D,B
	LD B,(sys_sound_triole_count:)
	DEC B
	LD (sys_sound_triole_count:),B
	OR B,B
	JR NZ sound_parser_note_next5:
	LD (sys_sound_triole_type:),A
	
sound_parser_note_next5:
	// add pitch
	LD B,(sys_sound_pitch:)
	ADD C,B
	// enqueue sound (note C, length D, unit E)
	OUT (3),D
	OUT (4),E
	OUT (5),C
	RET

sound_parser_note_sharp:
	LD E,1
	LD A,(HL)
	INC HL
	JR sound_parser_note_no_key:

sound_parser_note_flat:
	LD E,0xff
	LD A,(HL)
	INC HL
	JR sound_parser_note_no_key:

sound_parser_note_natural:
	LD E,0
	LD A,(HL)
	INC HL

sound_parser_note_no_key:
	CALL sound_parser_rawnote: // (c=0..6)
	MUL B,12 // octave offset
	ADD E,B // offset (sharp, natural, flat)
	ADD E,60 // base C
	LD B,0
	ADD BC,sound_basic_keytable:
	LD C,(BC)
	ADD C,E
	JP sound_parser_note_length:

// ************** sound parser pause *************
// IN:
//   HL -> null-terminated string
//   A = char
// OUT:
//   HL -> null-terminated string
//   A = char
//   plays the given pause
//
sound_parser_pause:
	LD C,1
	LD A,(HL)
	INC HL
	JP sound_parser_note_length:

// ************** sound parser abc*************
// IN:
//   HL -> null-terminated string
// OUT:
//   plays the song
//
sound_parser_abc:
	LD A,100
	LD (sys_sound_speed:),A
	OUT (6),A
	LD A,0
	LD (sys_sound_key:),A
	LD (sys_sound_pitch:),A
	LD (sys_sound_triole_type:),A
	LD (sys_sound_rythm:),A
	LD BC,0
	LD (sys_sound_repeat:),BC
	LD BC,0x0404
	LD (sys_sound_beat:),BC
	LD A,8
	LD (sys_sound_length:),A

	PUSH HL
	CALL sound_calc_keytable:
	POP HL

	LD A,(HL)
	INC HL

sound_parser_loop:
	CALL sound_parser_whitespace:

	OR A,A // end of string?
	RET Z
		
	LD BC, sound_parser_loop: 
	PUSH BC // simulates a call with return adress sound_parser_loop:

	CMP A,'K'
	JP Z sound_parser_key:
	
	CMP A,'M'
	JP Z sound_parser_meter:
	
	CMP A,'L'
	JP Z sound_parser_length:
	
	CMP A,'Q'
	JP Z sound_parser_tempo:
	
	CMP A,'('
	JP Z sound_parser_triole:
	
	CMP A,'|'
	JP Z sound_parser_bar:
	CMP A,':'
	JP Z sound_parser_bar:
	
	CMP A,'^'
	JP Z sound_parser_note_sharp:
	CMP A,'='
	JP Z sound_parser_note_natural:
	CMP A,'_'
	JP Z sound_parser_note_flat:
	
	CMP A,'z'
	JP Z sound_parser_pause:
	CMP A,'x'
	JP Z sound_parser_pause:

	CMP A,'Z'
//	JP Z sound_parser_meter_pause:
	CMP A,'X'
//	JP Z sound_parser_meter_pause:
	
	CMP A,'A'
	
	JR C sound_parser_abc_next1:
	CMP A,'H'
	JP C sound_parser_note:

sound_parser_abc_next1:
	CMP A,'a'
	JR C sound_parser_abc_next2:
	CMP A,'h'
	JP C sound_parser_note:

sound_parser_abc_next2:
	POP BC // that was ->sound_parser_loop:, to simulate a call
	// wrong char. exit to prevent the sound system interpreting the whole 64k mem.
	RET

// ************** sound test *******************
//

//.include ../irishTunes
sound_test:
//	LD HL,tunes_stivel_abc:
//	CALL sound_parser_abc:
	RET
