/* Standard C and Windows API includes */
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <commdlg.h>
#include <shellapi.h>
#include <shlwapi.h>
#include <uxtheme.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* M1 includes */
#include "m1ui.h"
#include "resource.h"

/* Interface colors */
enum
{
	COLOR_INIT = 0,		// blue
	COLOR_ERROR,		// dark red
	COLOR_PLAY,			// green
	COLOR_RECORD,		// red
	COLOR_PAUSE,		// orange
	COLOR_STOP			// brown
};

/* Highly useful macro for compile-time knowledge of an array size */
#define ARRAY_LENGTH(x)     (sizeof(x) / sizeof(x[0]))

/* RGB colors defines */
#define COLOR_BROWN			RGB(185, 122, 87)
#define COLOR_DARKRED		RGB(136, 00, 21)
#define COLOR_GREEN			RGB(34, 177, 76)
#define COLOR_RED			RGB(237, 28, 36)
#define COLOR_ORANGE		RGB(255, 127, 39)
#define COLOR_BLUE			RGB(63, 72, 204)
#define COLOR_WHITE			RGB(255, 255, 255)
#define COLOR_BLACK			RGB(0, 0, 0)
#define COLOR_PURPLE		RGB(163, 73, 164)
#define COLOR_CYAN			RGB(112, 146, 190)
#define COLOR_GREY			RGB(239, 239, 239)
#define COLOR_DARKGREY		RGB(95, 95, 95)
#define COLOR_BARPLAY		RGB(85, 191, 132)
#define COLOR_BARREC		RGB(251, 0, 52)

/* Global variables */
static int start = 0;
static int temp = 0;
static int minute = 0;
static int second = 0;
static int left = 0;
static int right = 0;
static int curgame = 0;
static int cursong = 0;
static int max_games = 0;
static int volume = 0;
static int color = 0;
static int curtime = 0;
static BOOL record = FALSE;
static BOOL pause = FALSE;
static BOOL gotone = FALSE;
static int parts[3] = {200, 280, -1};
static char buffer[512];
static char linebuf[512];
static char curpath[MAX_PATH];
static char rompath[MAX_PATH];
static char wavefile[MAX_PATH];
static char gamename[MAX_PATH];
static char *gametitle = NULL;
static char *gamemaker = NULL;
static char *gameyear = NULL;
static char *gamehardware = NULL;
static char *gamedriver = NULL;
static char *version = NULL;
static char *m1fxversion = "M1FX 12.0";

/* Global Windows handlers */
static NOTIFYICONDATA tray_icon;
static MSG uMsg;
static HWND hMain = NULL;
static HWND hBarRight = NULL;
static HWND hBarLeft = NULL;
static HWND hStatusBar = NULL;
static HINSTANCE hInst = NULL;
static HMENU hMenu = NULL;
static HICON hIcon = NULL;
static HICON hIconStatus = NULL;
static HBRUSH hBrush = NULL;
static HBRUSH hBrushAbout = NULL;
static HDC hDC = NULL;
static HDC hDCAbout = NULL;
static HFONT hFont = NULL;
static HFONT hFontStatus = NULL;
static HFONT hFontAbout = NULL;
static HBITMAP hAbout = NULL;
static HBITMAP hExit = NULL;
static HBITMAP hForum = NULL;
static HBITMAP hPause = NULL;
static HBITMAP hPlay = NULL;
static HBITMAP hPrev = NULL;
static HBITMAP hNext = NULL;
static HBITMAP hSelect = NULL;
static HBITMAP hStop = NULL;
static HBITMAP hRecord = NULL;
static HACCEL hAccel = NULL;

/* M1 core messages to the interface */
static int m1ui_message(void *none, int message, char *txt, int iparm);

/* Parse m1.ini to store the ROMs path */
static void ReadConfig(FILE *cnf)
{
	int i = 0;

	fgets(linebuf, 512, cnf);
	char *p = linebuf;

	while (*p != ';')
	{
		curpath[i++] = *p++;
	}

	curpath[i] = '\0';
	strcpy(rompath, curpath);
}

/* Search for the selected game ROMset */
static int FindRom(char *fn)
{
    char *p = rompath;

    while (*p)
    {
		int i = 0;		
	
        while (*p)
        {
            curpath[i++] = *p++;
        }

        curpath[i] = '\0';
        strcat(curpath, "\\");
        strcat(curpath, fn);
		FILE *f = fopen(curpath, "r");

        if (f)
        {
            fclose(f);
            strcpy(fn, curpath);
            m1snd_set_info_str(M1_SINF_SET_ROMPATH, fn, 0, 0, 0);
            return TRUE;
        }
    }

    return FALSE;
}

/* Search for m1.ini */
static void SearchConfigFile(void)
{
	FILE *cnf = fopen(".\\m1\\m1.ini", "r");
	
	if (!cnf)
		strcpy(rompath, "roms");		// default ROMs path
	else
	{	
		ReadConfig(cnf);				// store the new ROMs path
		fclose(cnf);
	}
}

/* Current value of the equalizer */
static void WaveData(char *txt, int iparm)
{
    int lpeak = 0;
	int rpeak = 0;
    short *pointer = (short*)txt;

    for (int i = 0; i < iparm; i += 2)
    {
        if (lpeak < abs(pointer[i]))
           	lpeak = abs(pointer[i]);
		
       	if (rpeak < abs(pointer[i + 1]))
           	rpeak = abs(pointer[i + 1]);
    }

    left = (100 * log10((double)lpeak / 32768)) + 200;
    right = (100 * log10((double)rpeak / 32768)) + 200;
}

/* Callbacks from the core of interest to the user interface */
static int m1ui_message(void *none, int message, char *txt, int iparm)
{
   	switch (message)
   	{
       	// called when switching to a new game
   		case M1_MSG_SWITCHING:
       		break;

       	// called to show the current game's name
   		case M1_MSG_GAMENAME:
			gametitle = txt;
       		break;

       	// called to show the driver's name
   		case M1_MSG_DRIVERNAME:
			gamedriver = txt;
       		break;

       	// called to show the hardware description
    	case M1_MSG_HARDWAREDESC:
			gamehardware = txt;
        	break;

        // called when ROM loading fails for a game
    	case M1_MSG_ROMLOADERR:
        	break;

        // called when a song is (re)triggered
    	case M1_MSG_STARTINGSONG:
        	break;

        // called if a hardware error occurs
    	case M1_MSG_HARDWAREERROR:
        	break;

        // called when the hardware begins booting
    	case M1_MSG_BOOTING:
        	break;

        // called when the hardware is finished booting and is ready to play
    	case M1_MSG_BOOTFINISHED:
        	break;

        // called when there's been at least 2 seconds of silence
    	case M1_MSG_SILENCE:
			break;

        // called to let the UI do vu meters/spectrum analyzers/etc
        // txt = pointer to the frame's interleaved 16-bit stereo wave data
        // iparm = number of samples (bytes / 4)
    	case M1_MSG_WAVEDATA:
        	WaveData(txt, iparm);
        	break;

		// called when core needs to find the ROMset
    	case M1_MSG_MATCHPATH:
        	return FindRom(txt);
        	break;

		// called when starting to record WAV output
		case M1_MSG_GETWAVPATH:
			snprintf(wavefile, ARRAY_LENGTH(wavefile), ".\\audio\\%s_song_#%03d.wav", gamename, cursong);
			strcpy(txt, wavefile);
			break;
		
		// nothing to do
		case M1_MSG_ERROR:
        	break;
    }

    return FALSE;
}

/* Setup text for current status of Record menu voice */
static void SetRecordMenuItem(void)
{
	MENUITEMINFO mmi;
	
	memset(&mmi, 0, sizeof(MENUITEMINFO));
	mmi.cbSize = sizeof(MENUITEMINFO);
	mmi.fMask = MIIM_TYPE;
	mmi.fType = MFT_STRING;

	if (record)
		mmi.dwTypeData = "Stop recording WAV output\tAlt+R";
	else
		mmi.dwTypeData = "Start recording WAV output\tAlt+R";
		
	mmi.cch = strlen(mmi.dwTypeData);
	SetMenuItemInfo(hMenu, ID_RECORD, FALSE, &mmi);
}

/* Initialize M1 core, set some useful options and search for selected game */
static void InitializeM1(HWND hDlg)
{
	gotone = FALSE;
	pause = FALSE;
	record = FALSE;
	volume = 50;
	left = 0;
	right = 0;
	curtime = 0;
	color = COLOR_INIT;    
	snprintf(buffer, ARRAY_LENGTH(buffer), "Initializing... Please wait...");
	SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
	SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
	SetWindowText(GetDlgItem(hDlg, IDC_TIME), "00:00");
	SetWindowText(GetDlgItem(hDlg, IDC_RECORD), "Start recording WAV output");
	SetWindowText(GetDlgItem(hDlg, IDC_GAME), "");
    SetWindowText(GetDlgItem(hDlg, IDC_YEAR), "");
    SetWindowText(GetDlgItem(hDlg, IDC_HARDWARE), "");
    SetWindowText(GetDlgItem(hDlg, IDC_DRIVER), "");
	SetRecordMenuItem();
	snprintf(buffer, ARRAY_LENGTH(buffer), "%03d", volume);
	SetWindowText(GetDlgItem(hDlg, IDC_VOLTEXT), buffer);
	SendMessage(GetDlgItem(hDlg, IDC_VOLUME), TBM_SETPOS, TRUE, volume);	
	SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
	SendMessage(hBarRight, PBM_SETPOS, 0, 0);
	SendMessage(hBarLeft, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
	SendMessage(hBarRight, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
	m1snd_init(NULL, m1ui_message);
    m1snd_setoption(M1_OPT_NORMALIZE, 0);
    m1snd_setoption(M1_OPT_RESETNORMALIZE, 0);
    m1snd_setoption(M1_OPT_STEREOMIX, 0);
	m1snd_setoption(M1_OPT_WAVELOG, 0);
    max_games = m1snd_get_info_int(M1_IINF_TOTALGAMES, 0);
    snprintf(buffer, ARRAY_LENGTH(buffer), "%d games", max_games);
	SendMessage(hStatusBar, SB_SETTEXT, 1, (LPARAM)buffer);

    for (int i = 0; i < max_games; i++)
    {
        if (!strcmp(gamename, m1snd_get_info_str(M1_SINF_ROMNAME, i)))
        {
			snprintf(buffer, ARRAY_LENGTH(buffer), "Booting hardware...");
			SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
			SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
            m1snd_run(M1_CMD_GAMEJMP, i);
            gotone = TRUE;
            break;
        }
    }

	EnableWindow(GetDlgItem(hDlg, IDC_GAME), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_YEAR), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_HARDWARE), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_DRIVER), gotone);
	EnableWindow(GetDlgItem(hDlg, IDC_PLAY), gotone);
	EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_PREV), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_NEXT), gotone);
    EnableWindow(GetDlgItem(hDlg, IDC_STOP), gotone);
	EnableWindow(GetDlgItem(hDlg, IDC_VOLUME), gotone);
	EnableWindow(GetDlgItem(hDlg, IDC_RECORD), gotone);
	EnableMenuItem(hMenu, ID_PLAY, (gotone ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(hMenu, ID_PAUSE, (gotone ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(hMenu, ID_RECORD, (gotone ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(hMenu, ID_NEXT, (gotone ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(hMenu, ID_PREV, (gotone ? MF_ENABLED : MF_GRAYED));
	EnableMenuItem(hMenu, ID_STOP, (gotone ? MF_ENABLED : MF_GRAYED));
		
    if (!gotone)
    {
		color = COLOR_ERROR;
		snprintf(buffer, ARRAY_LENGTH(buffer), "Unknown or unsupported game %s", gamename);
		SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
		SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
   	}
    else
    {
		color = COLOR_PLAY;
		curgame = m1snd_get_info_int(M1_IINF_CURGAME, 0);
       	snprintf(buffer, ARRAY_LENGTH(buffer), "%s", gametitle);
       	SetWindowText(GetDlgItem(hDlg, IDC_GAME), buffer);
       	gameyear = m1snd_get_info_str(M1_SINF_YEAR, curgame);
       	gamemaker = m1snd_get_info_str(M1_SINF_MAKER, curgame);
       	snprintf(buffer, ARRAY_LENGTH(buffer), "%s %s", gameyear, gamemaker);
       	SetWindowText(GetDlgItem(hDlg, IDC_YEAR), buffer);
       	snprintf(buffer, ARRAY_LENGTH(buffer), "Hardware: %s", gamehardware);
       	SetWindowText(GetDlgItem(hDlg, IDC_HARDWARE), buffer);
       	snprintf(buffer, ARRAY_LENGTH(buffer), "Hardware type: %s", gamedriver);
       	SetWindowText(GetDlgItem(hDlg, IDC_DRIVER), buffer);
        cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
      	snprintf(buffer, ARRAY_LENGTH(buffer), "Playing song #%d", cursong);
        SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
		SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
		EnableWindow(GetDlgItem(hDlg, IDC_PLAY), FALSE);
		EnableMenuItem(hMenu, ID_PLAY, MF_GRAYED);
		SetTimer(hMain, IDT_M1PLAY, 60, NULL);
		m1snd_run(M1_CMD_SONGJMP, cursong);
		m1snd_setoption(M1_OPT_FIXEDVOLUME, volume);
		m1snd_run(M1_CMD_UNPAUSE, 0);
	}
}

/* Simple function to center a window in the screen */
static void CenterWindow(HWND hWnd)
{
	RECT rcCenter;
	RECT rcWnd;

	GetWindowRect(hWnd, &rcWnd);
	int iWndWidth  = rcWnd.right - rcWnd.left;
	int iWndHeight = rcWnd.bottom - rcWnd.top;
	rcCenter.left = 0;
	rcCenter.top = 0;
	rcCenter.right = GetSystemMetrics(SM_CXFULLSCREEN);
	rcCenter.bottom = GetSystemMetrics(SM_CYFULLSCREEN);
	int iScrWidth  = rcCenter.right - rcCenter.left;
	int iScrHeight = rcCenter.bottom - rcCenter.top;
	int xLeft = rcCenter.left;
	int yTop = rcCenter.top;

	if (iScrWidth > iWndWidth)
		xLeft += ((iScrWidth - iWndWidth) / 2);
	
	if (iScrHeight > iWndHeight)
		yTop += ((iScrHeight - iWndHeight) / 2);

	// map screen coordinates to child coordinates
	SetWindowPos(hWnd, HWND_TOP, xLeft, yTop, -1, -1, SWP_NOSIZE);
}

/* Open file hook procedure to initialize some things */
static UINT_PTR CALLBACK OFNHookProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			//CenterWindow(GetParent(hDlg));
        	hIcon = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_M1FX_ICON));
        	SendMessage(hDlg, WM_SETICON, ICON_SMALL, (LPARAM)hIcon);
			break;
	}
	
	return FALSE;
}

/* Select new game procedure */
static void SelectNewGame(void)
{
	OPENFILENAME ofn;
	char file[MAX_PATH];
	char curdir[MAX_PATH];

	file[0] = '\0';
	
	if (GetCurrentDirectory(MAX_PATH, curdir) > MAX_PATH)
		curdir[0] = '\0';
	
	memset(&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hMain;
	ofn.hInstance = NULL;
	ofn.lpstrFilter = "romsets (*.zip)\0*.zip\0";
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter = 0;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile = file;
	ofn.nMaxFile = sizeof(file);
	ofn.lpstrFileTitle = NULL;
	ofn.nMaxFileTitle = 0;
	ofn.lpstrInitialDir = rompath;
	ofn.lpstrTitle = "Select a game";
	ofn.Flags = OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_ENABLEHOOK;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;
	ofn.lpstrDefExt = "zip";
	ofn.lCustData = 0;
	ofn.lpfnHook = &OFNHookProc;
	ofn.lpTemplateName = NULL;
	
	if (GetOpenFileName(&ofn))
	{
		char *tempname = PathFindFileName(file);
		PathRemoveExtension(tempname);		// deprecated but still useful
		strcpy(gamename, tempname);
		GetCurrentDirectory(MAX_PATH, rompath);
		SetCurrentDirectory(curdir);
	}
	else
		gamename[0] = '\0';
}

/* Easy function to have transparent menu icons */
static HBITMAP CreateBitmapTransparent(HBITMAP hSource)
{
	BITMAP bm;
	
	HDC hSrc = CreateCompatibleDC(NULL);
	HDC hDst = CreateCompatibleDC(NULL);
	GetObject(hSource, sizeof(bm), &bm);
	SelectObject(hSrc, hSource);
	HBITMAP hNew = CreateBitmap(bm.bmWidth, bm.bmHeight, bm.bmPlanes, bm.bmBitsPixel, NULL);
	SelectObject(hDst, hNew);
    BitBlt(hDst, 0, 0, bm.bmWidth, bm.bmHeight, hSrc, 0, 0, SRCCOPY);
    COLORREF clrTP = RGB(128, 128, 128);
    COLORREF clrBK = GetSysColor(COLOR_MENU);
	
	for (int nRow = 0; nRow < bm.bmHeight; nRow++)
	{
        for (int nCol = 0; nCol < bm.bmWidth; nCol++)
		{
            if (GetPixel(hSrc, nCol, nRow) == clrTP)
                SetPixel(hDst, nCol, nRow, clrBK);
		}
	}
	
    DeleteDC(hDst);
	DeleteDC(hSrc);
    return hNew;
}

/* Initialize window menu */
static void InitMenuIcons()
{
	HBITMAP hTemp = NULL;
	MENUINFO mi;

	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_ABOUT));
	hAbout = CreateBitmapTransparent(hTemp);	
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_EXIT));
	hExit = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_FORUM));
	hForum = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_NEXT));
	hNext = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_PAUSE));
	hPause = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_PLAY));
	hPlay = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_PREV));
	hPrev = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_SELECT));
	hSelect = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_STOP));
	hStop = CreateBitmapTransparent(hTemp);
	hTemp = LoadBitmap(GetModuleHandle(NULL), MAKEINTRESOURCE(IDB_RECORD));
	hRecord = CreateBitmapTransparent(hTemp);
	
	memset(&mi, 0, sizeof(MENUINFO));
	mi.cbSize = sizeof(MENUINFO);
	mi.fMask = MIM_BACKGROUND | MIM_STYLE;
	mi.dwStyle = MNS_CHECKORBMP;
	mi.hbrBack = GetSysColorBrush(COLOR_WINDOW);

	HMENU hFile = GetSubMenu(hMenu, 0);
	HMENU hHelp = GetSubMenu(hMenu, 1);	
	SetMenuInfo(hFile, &mi);
	SetMenuInfo(hHelp, &mi);
	SetMenuItemBitmaps(hMenu, ID_ABOUT, MF_BYCOMMAND, hAbout, hAbout);
	SetMenuItemBitmaps(hMenu, ID_EXIT, MF_BYCOMMAND, hExit, hExit);
	SetMenuItemBitmaps(hMenu, ID_FORUM, MF_BYCOMMAND, hForum, hForum);
	SetMenuItemBitmaps(hMenu, ID_PAUSE, MF_BYCOMMAND, hPause, hPause);
	SetMenuItemBitmaps(hMenu, ID_PLAY, MF_BYCOMMAND, hPlay, hPlay);
	SetMenuItemBitmaps(hMenu, ID_PREV, MF_BYCOMMAND, hPrev, hPrev);
	SetMenuItemBitmaps(hMenu, ID_NEXT, MF_BYCOMMAND, hNext, hNext);
	SetMenuItemBitmaps(hMenu, ID_RECORD, MF_BYCOMMAND, hRecord, hRecord);
	SetMenuItemBitmaps(hMenu, ID_SELECT, MF_BYCOMMAND, hSelect, hSelect);
	SetMenuItemBitmaps(hMenu, ID_STOP, MF_BYCOMMAND, hStop, hStop);
}

/* About M1FX dialog window procedures and messages handling */
static INT_PTR CALLBACK AboutDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			start = 220;
			CenterWindow(hDlg);
			hFontAbout = CreateFont(-27, 0, 0, 0, 700, 0, 0, 0, 0, 3, 2, 1, 34, "Trebuchet MS");
			SetWindowFont(GetDlgItem(hDlg, IDC_M1TITLE), hFontAbout, TRUE);
			hBrushAbout = CreateSolidBrush(COLOR_GREY);
			SetWindowText(GetDlgItem(hDlg, IDC_M1TITLE), m1fxversion);
			version = m1snd_get_info_str(M1_SINF_COREVERSION, 0);
			snprintf(buffer, ARRAY_LENGTH(buffer), "M1 Core version: %s", version);
			SetWindowText(GetDlgItem(hDlg, IDC_M1CORE), buffer);
			SetWindowText(GetDlgItem(hDlg, IDC_M1VERSION), "Build time: " __DATE__" - " __TIME__"");
			snprintf(buffer, ARRAY_LENGTH(buffer), "Compiled with MinGW %d.%d - GCC " __VERSION__"", __MINGW32_MAJOR_VERSION, __MINGW32_MINOR_VERSION);
			SetWindowText(GetDlgItem(hDlg, IDC_GCC), buffer);
			MoveWindow(GetDlgItem(hDlg, IDC_M1FX), start, 101, 240, 16, TRUE);
			SetTimer(hDlg, IDT_SCROLL, 60, NULL);
			return TRUE;

		case WM_CTLCOLORDLG:
			return (LRESULT)hBrushAbout;

		case WM_CTLCOLORSTATIC:
		case WM_CTLCOLORBTN:
			hDCAbout = (HDC)wParam;
			SetBkMode(hDCAbout, TRANSPARENT);
			SetTextColor(hDCAbout, COLOR_DARKGREY);

			if ((HWND)lParam == GetDlgItem(hDlg, IDC_M1TITLE))
				SetTextColor(hDCAbout, COLOR_BLACK);

			if ((HWND)lParam == GetDlgItem(hDlg, IDC_M1FX))
				SetTextColor(hDCAbout, COLOR_BLUE);
	
			return (LRESULT)hBrushAbout;
		
		case WM_NOTIFY:
			switch (((NMHDR*)lParam)->code)
			{
				case NM_CLICK:
					if (((NMHDR*)lParam)->hwndFrom == GetDlgItem(hDlg, IDC_LINK))
						ShellExecute(hDlg, NULL, "http://mame32fx.altervista.org/", NULL, NULL, SW_SHOWNORMAL);
				
					break;
			}
			break;
		
		case WM_TIMER:
			switch (LOWORD(wParam))
			{
				case IDT_SCROLL:
					MoveWindow(GetDlgItem(hDlg, IDC_M1FX), start--, 101, 240, 16, TRUE);
				
					if (start == -240)
						start = 240;
				
					break;
			}
			break;
		
		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDOK:
				case IDCANCEL:
					KillTimer(hDlg, IDT_SCROLL);
					DeleteObject(hBrushAbout);
					DeleteFont(hFontAbout);
					EndDialog(hDlg, 0);
					return TRUE;
			}
			break;
	}

	return FALSE;
}

/* Main window dialog procedures and messages handling */
static INT_PTR CALLBACK DialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    	case WM_INITDIALOG:
			CenterWindow(hDlg);
			hFont = CreateFont(-12, 0, 0, 0, 400, 0, 0, 0, 0, 3, 2, 1, 34, "Verdana");
			hFontStatus = CreateFont(-11, 0, 0, 0, 400, 0, 0, 0, 0, 3, 2, 1, 34, "Tahoma");
			SetWindowFont(GetDlgItem(hDlg, IDC_TITLE), hFont, TRUE);
			SetWindowFont(GetDlgItem(hDlg, IDC_COPY), hFont, TRUE);
			SetWindowFont(GetDlgItem(hDlg, IDC_SONG), hFont, TRUE);
			hBrush = CreateSolidBrush(COLOR_WHITE);
        	hIcon = LoadIcon(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_M1FX_ICON));
			hIconStatus = (HICON)LoadImage(GetModuleHandle(NULL), MAKEINTRESOURCE(IDI_M1FX_ICON), IMAGE_ICON, 16, 16, LR_SHARED);
        	SendMessage(hDlg, WM_SETICON, ICON_SMALL, (LPARAM)hIcon);
        	hBarLeft = GetDlgItem(hDlg, IDC_PROGLEFT);
        	hBarRight = GetDlgItem(hDlg, IDC_PROGRIGHT);
        	hStatusBar = GetDlgItem(hDlg, IDC_STATUSBAR);
			SetWindowFont(hStatusBar, hFontStatus, TRUE);
			SetWindowTheme(hBarLeft, L" ", L" ");
			SetWindowTheme(hBarRight, L" ", L" ");
        	SendMessage(hBarLeft, PBM_SETRANGE, 0, MAKELPARAM(0, 128));
        	SendMessage(hBarRight, PBM_SETRANGE, 0, MAKELPARAM(0, 128));
			SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
			SendMessage(hBarRight, PBM_SETPOS, 0, 0);
			SendMessage(hBarLeft, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
			SendMessage(hBarRight, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
			SendMessage(hBarLeft, PBM_SETBKCOLOR, 0, COLOR_WHITE);
			SendMessage(hBarRight, PBM_SETBKCOLOR, 0, COLOR_WHITE);
			SendMessage(hStatusBar, SB_SETPARTS, 3, (LPARAM)parts);
			SendMessage(hStatusBar, SB_SETTEXT, 2, (LPARAM)"M1FX");
			SendMessage(hStatusBar, SB_SETICON, 2, (LPARAM)hIconStatus);
			SendMessage(GetDlgItem(hDlg, IDC_VOLUME), TBM_SETRANGE, TRUE, MAKELPARAM(0, 100));
			SendMessage(GetDlgItem(hDlg, IDC_VOLUME), TBM_SETTICFREQ, 10, 0);
			snprintf(buffer, ARRAY_LENGTH(buffer), "%s FINAL", m1fxversion);
			SetWindowText(hDlg, buffer);
			version = m1snd_get_info_str(M1_SINF_COREVERSION, 0);
			snprintf(buffer, ARRAY_LENGTH(buffer), "M1 %s Arcade Music Player by R.Belmont", version);
        	SetWindowText(GetDlgItem(hDlg, IDC_TITLE), buffer);
        	SetWindowText(GetDlgItem(hDlg, IDC_COPY), "Stand-Alone Graphic Interface by Mamesick");
        	break;
  
		case WM_CTLCOLORDLG:
			return (LRESULT) hBrush;

		case WM_CTLCOLORSTATIC:
		case WM_CTLCOLORBTN:
			hDC = (HDC)wParam;
			SetBkMode(hDC, TRANSPARENT);
			SetTextColor(hDC, COLOR_BLACK);

			if ((HWND)lParam == GetDlgItem(hDlg, IDC_TITLE))
				SetTextColor(hDC, COLOR_PURPLE);

			if ((HWND)lParam == GetDlgItem(hDlg, IDC_COPY))
				SetTextColor(hDC, COLOR_CYAN);

			if ((HWND)lParam == GetDlgItem(hDlg, IDC_SONG))
			{
				switch(color)
				{
					case COLOR_INIT:
						SetTextColor(hDC, COLOR_BLUE);
						break;

						case COLOR_STOP:
						SetTextColor(hDC, COLOR_BROWN);
						break;

					case COLOR_ERROR:
						SetTextColor(hDC, COLOR_DARKRED);
						break;

						case COLOR_RECORD:
						SetTextColor(hDC, COLOR_RED);
						break;

					case COLOR_PLAY:
						SetTextColor(hDC, COLOR_GREEN);
						break;

					case COLOR_PAUSE:
						SetTextColor(hDC, COLOR_ORANGE);
						break;
				}
			}

			return (LRESULT) hBrush;
	
		case WM_HSCROLL:
			volume = SendMessage(GetDlgItem(hDlg, IDC_VOLUME), TBM_GETPOS, 0, 0);
        	snprintf(buffer, ARRAY_LENGTH(buffer), "%03d", volume);
        	SetWindowText(GetDlgItem(hDlg, IDC_VOLTEXT), buffer);
			m1snd_setoption(M1_OPT_FIXEDVOLUME, volume);
			break;

		case WM_COMMAND:
        	switch (LOWORD(wParam))
        	{
				case ID_EXIT:
					PostMessage(hMain, WM_CLOSE, 0, 0);
					break;
				
				case ID_FORUM:
					ShellExecute(hMain, NULL, "http://mame32fx.altervista.org/forum/", NULL, NULL, SW_SHOWNORMAL);
					break;
				
				case ID_ABOUT:
					DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ABOUT), hMain, AboutDialogProc);
					SetFocus(hMain);
					break;
					
				case ID_SELECT:
				case IDC_SELECT:
					m1snd_run(M1_CMD_STOP, 0);
					m1snd_shutdown();
					KillTimer(hMain, IDT_M1PLAY);
					SelectNewGame();
					UpdateWindow(hMain);
					SetActiveWindow(hMain);
					SetForegroundWindow(hMain);					
					InitializeM1(hMain);
					break;

				case ID_PAUSE:
        		case IDC_PAUSE:
            		pause = TRUE;
					color = COLOR_PAUSE;
                  	m1snd_run(M1_CMD_PAUSE, 0);
                  	SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
                	SendMessage(hBarRight, PBM_SETPOS, 0, 0);
            		cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
                	snprintf(buffer, ARRAY_LENGTH(buffer), "Paused song #%d", cursong);
                	SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
					SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
					EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PLAY), TRUE);
					EnableMenuItem(hMenu, ID_PAUSE, MF_GRAYED);
					EnableMenuItem(hMenu, ID_PLAY, MF_ENABLED);
					break;

				case ID_PLAY:
        		case IDC_PLAY:
					pause = FALSE;
					color = COLOR_PLAY;
                 	m1snd_run(M1_CMD_UNPAUSE, 0);
               		cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
                	snprintf(buffer, ARRAY_LENGTH(buffer), "Playing song #%d", cursong);
                	SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
					SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
					EnableWindow(GetDlgItem(hDlg, IDC_PREV), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEXT), TRUE);					
					EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_STOP), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDC_PLAY), FALSE);
					EnableMenuItem(hMenu, ID_PAUSE, MF_ENABLED);
					EnableMenuItem(hMenu, ID_PLAY, MF_GRAYED);
					EnableMenuItem(hMenu, ID_PREV, MF_ENABLED);
					EnableMenuItem(hMenu, ID_NEXT, MF_ENABLED);
					EnableMenuItem(hMenu, ID_STOP, MF_ENABLED);
            		break;
				
				case ID_NEXT:
        		case IDC_NEXT:
            		cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);

            		if (cursong < m1snd_get_info_int(M1_IINF_MAXSONG, 0))
            		{
                 		m1snd_run(M1_CMD_SONGJMP, cursong + 1);

						if (pause)
							snprintf(buffer, ARRAY_LENGTH(buffer), "Selected song #%d", cursong + 1);
						else
							snprintf(buffer, ARRAY_LENGTH(buffer), "Playing song #%d", cursong + 1);
							
                		SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
						SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
					}
            		break;
				
				case ID_PREV:
				case IDC_PREV:
            		cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);

            		if (cursong > m1snd_get_info_int(M1_IINF_MINSONG, 0))
            		{
                 		m1snd_run(M1_CMD_SONGJMP, cursong - 1);

						if (pause)
							snprintf(buffer, ARRAY_LENGTH(buffer), "Selected song #%d", cursong - 1);
						else
							snprintf(buffer, ARRAY_LENGTH(buffer), "Playing song #%d", cursong - 1);

            			SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
						SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
					}
           			break;

				case ID_STOP:
				case IDC_STOP:
					color = COLOR_STOP;
					volume = 50;
 					pause = TRUE;
                	m1snd_run(M1_CMD_STOP, 0);
                	SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
                	SendMessage(hBarRight, PBM_SETPOS, 0, 0);
					snprintf(buffer, ARRAY_LENGTH(buffer), "Stopped - Start again from first song");
                	SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
					SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
         			SetWindowText(GetDlgItem(hDlg, IDC_TIME), "00:00");
					EnableWindow(GetDlgItem(hDlg, IDC_PREV), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_NEXT), FALSE);					
					EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), FALSE);					
					EnableWindow(GetDlgItem(hDlg, IDC_STOP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PLAY), TRUE);
					EnableMenuItem(hMenu, ID_PREV, MF_GRAYED);
					EnableMenuItem(hMenu, ID_NEXT, MF_GRAYED);
					EnableMenuItem(hMenu, ID_PAUSE, MF_GRAYED);
					EnableMenuItem(hMenu, ID_STOP, MF_GRAYED);
					EnableMenuItem(hMenu, ID_PLAY, MF_ENABLED);
					snprintf(buffer, ARRAY_LENGTH(buffer), "%03d", volume);
					SetWindowText(GetDlgItem(hDlg, IDC_VOLTEXT), buffer);          			
					SendMessage(GetDlgItem(hDlg, IDC_VOLUME), TBM_SETPOS, TRUE, volume);
					m1snd_run(M1_CMD_GAMEJMP, curgame);
            		cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
         			m1snd_run(M1_CMD_SONGJMP, cursong);
					m1snd_setoption(M1_OPT_FIXEDVOLUME, volume);
                  	m1snd_run(M1_CMD_PAUSE, 0);
           			break;

				case ID_RECORD:
        		case IDC_RECORD:
            		record = !record;

            		if (record)
            		{
						color = COLOR_RECORD;
						pause = FALSE;
						m1snd_run(M1_CMD_PAUSE, 0);
						SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
						SendMessage(hBarRight, PBM_SETPOS, 0, 0);
						SendMessage(hBarLeft, PBM_SETBARCOLOR, 0, COLOR_BARREC);
						SendMessage(hBarRight, PBM_SETBARCOLOR, 0, COLOR_BARREC);
						EnableWindow(GetDlgItem(hDlg, IDC_PREV), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEXT), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_PLAY), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_STOP), FALSE);
						EnableMenuItem(hMenu, ID_PLAY, MF_GRAYED);
						EnableMenuItem(hMenu, ID_PAUSE, MF_GRAYED);
						EnableMenuItem(hMenu, ID_PREV, MF_GRAYED);
						EnableMenuItem(hMenu, ID_NEXT, MF_GRAYED);
						EnableMenuItem(hMenu, ID_STOP, MF_GRAYED);
						SetWindowText(GetDlgItem(hDlg, IDC_RECORD), "Stop recording WAV output");
						SetRecordMenuItem();
						cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
                		snprintf(buffer, ARRAY_LENGTH(buffer), "Recording song #%d", cursong);
						SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
						SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
						m1snd_setoption(M1_OPT_WAVELOG, 1);
						m1snd_run(M1_CMD_SONGJMP, cursong);
						m1snd_run(M1_CMD_UNPAUSE, 0);
            		}
            		else
            		{
						color = COLOR_PLAY;
						pause = FALSE;
						m1snd_run(M1_CMD_PAUSE, 0);
						SendMessage(hBarLeft, PBM_SETPOS, 0, 0);
						SendMessage(hBarRight, PBM_SETPOS, 0, 0);
						SendMessage(hBarLeft, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
						SendMessage(hBarRight, PBM_SETBARCOLOR, 0, COLOR_BARPLAY);
						EnableWindow(GetDlgItem(hDlg, IDC_PREV), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_NEXT), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_PAUSE), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_PLAY), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_STOP), TRUE);
						EnableMenuItem(hMenu, ID_PLAY, MF_GRAYED);
						EnableMenuItem(hMenu, ID_PAUSE, MF_ENABLED);
						EnableMenuItem(hMenu, ID_PREV, MF_ENABLED);
						EnableMenuItem(hMenu, ID_NEXT, MF_ENABLED);
						EnableMenuItem(hMenu, ID_STOP, MF_ENABLED);
                		SetWindowText(GetDlgItem(hDlg, IDC_RECORD), "Start recording WAV output");
						SetRecordMenuItem();
               			cursong = m1snd_get_info_int(M1_IINF_CURSONG, 0);
                		snprintf(buffer, ARRAY_LENGTH(buffer), "Playing song #%d", cursong);
						SetWindowText(GetDlgItem(hDlg, IDC_SONG), buffer);
						SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)buffer);
						m1snd_setoption(M1_OPT_WAVELOG, 0);
						m1snd_run(M1_CMD_SONGJMP, cursong);
						m1snd_run(M1_CMD_UNPAUSE, 0);
					}
            		break;
       		}
        	break;
			
    	case WM_TIMER:
        	switch (LOWORD(wParam))
        	{
        		case IDT_M1PLAY:
            		if (gotone)
            		{
                		m1snd_run(M1_CMD_IDLE, 0);

                		if (!pause)
                		{
                    		SendMessage(hBarLeft, PBM_SETPOS, left, 0);
                    		SendMessage(hBarRight, PBM_SETPOS, right, 0);
         					curtime = m1snd_get_info_int(M1_IINF_CURTIME, 0);
							temp = curtime / 60;
							minute = temp / 60;
							second = temp - 60 * minute;
							snprintf(buffer, ARRAY_LENGTH(buffer), "%02d:%02d", minute, second);
        					SetWindowText(GetDlgItem(hDlg, IDC_TIME), buffer);
           				}
           			}
            		break;
        	}
        	break;
			
		case WM_SIZE:
			SendMessage(hStatusBar, WM_SIZE, 0, 0);
			break;

    	case WM_CLOSE:
        	m1snd_shutdown();
          	KillTimer(hMain, IDT_M1PLAY);
			Shell_NotifyIcon(NIM_DELETE, &tray_icon);
			DeleteBitmap(hAbout);
			DeleteBitmap(hExit);
			DeleteBitmap(hForum);
			DeleteBitmap(hPause);
			DeleteBitmap(hPlay);
			DeleteBitmap(hPrev);
			DeleteBitmap(hNext);
			DeleteBitmap(hRecord);
			DeleteBitmap(hSelect);
			DeleteBitmap(hStop);
			DeleteFont(hFont);
			DeleteFont(hFontStatus);
			DeleteObject(hBrush);
        	DestroyIcon(hIcon);
			DestroyMenu(hMenu);
			DestroyAcceleratorTable(hAccel);
        	DestroyWindow(hMain);
        	break;

    	case WM_DESTROY:
        	PostQuitMessage(0);
        	break;

		case WM_SYSCOMMAND:
			if (wParam == SC_MINIMIZE)
			{
				ShowWindow(hMain, SW_MINIMIZE);
				ShowWindow(hMain, SW_HIDE);
				Shell_NotifyIcon(NIM_ADD, &tray_icon);
				Shell_NotifyIcon(NIM_SETVERSION, &tray_icon);			
			}
			break;

		case WM_USER:
			if (lParam == WM_LBUTTONDBLCLK)
			{
				Shell_NotifyIcon(NIM_DELETE, &tray_icon);
				ShowWindow(hMain, SW_RESTORE);
				SetActiveWindow(hMain);
				SetForegroundWindow(hMain);
			}
			break;
    }

    return FALSE;
}

/* Startup */
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hInstPrev, LPSTR lpCmdLine, int nCmdShow)
{
	char *command = GetCommandLine();
	char *game = strrchr(command, ' ');
	strcpy(gamename, game + 1);
	
	hInst = hInstance;
    hMain = CreateDialog(hInst, MAKEINTRESOURCE(IDD_MAINDIALOG), 0, DialogProc);
	hMenu = LoadMenu(hInst, MAKEINTRESOURCE(IDR_MENU));
	SetMenu(hMain, hMenu);
	InitMenuIcons();
	hAccel = LoadAccelerators(hInst, MAKEINTRESOURCE(IDA_KEYS));
	
    memset(&tray_icon, 0, sizeof(NOTIFYICONDATA));
	tray_icon.cbSize = sizeof(NOTIFYICONDATA);
	tray_icon.hWnd = hMain;
	tray_icon.hIcon	= LoadIcon(hInst, MAKEINTRESOURCE(IDI_M1FX_ICON));
	tray_icon.uID = 0;
	tray_icon.uCallbackMessage = WM_USER;
	tray_icon.uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP | NIF_INFO;
	tray_icon.dwInfoFlags = NIIF_INFO;
	tray_icon.uVersion = NOTIFYICON_VERSION;

	strcpy(tray_icon.szInfoTitle, "M1FX");
	strcpy(tray_icon.szInfo, "Still running....");
	strcpy(tray_icon.szTip, "M1FX");

    ShowWindow(hMain, SW_SHOWNORMAL);
	SetActiveWindow(hMain);
	SetForegroundWindow(hMain);
	CreateDirectory(".\\audio", NULL);
	SearchConfigFile();
	InitializeM1(hMain);

 	while(GetMessage(&uMsg, NULL, 0, 0))
    {
		if (IsWindow(hMain))
		{
			if (!TranslateAccelerator(hMain, hAccel, &uMsg))
			{
				if (!IsDialogMessage(hMain, &uMsg))
				{
					TranslateMessage(&uMsg);
					DispatchMessage(&uMsg);
				}
			}
		}
    }
	
    return uMsg.wParam;
}
