/*===============================================================================*\
|#######################   [HLSL PCSX2 FX SHADER V1.11]   ########################|
|########################	    [BY ASMODEAN@PCSX2]	     #########################|
||																				 ||
||		  This program is free software; you can redistribute it and/or			 ||
||		  modify it under the terms of the GNU General Public License			 ||
||		  as published by the Free Software Foundation; either version 2		 ||
||		  of the License, or (at your option) any later version.				 ||
||																				 ||
||		  This program is distributed in the hope that it will be useful,		 ||
||		  but WITHOUT ANY WARRANTY; without even the implied warranty of		 ||
||		  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			 ||
||		  GNU General Public License for more details.							 ||
||																				 ||
|#################################################################################|
\*===============================================================================*/

#ifdef SHADER_MODEL
/*------------------------------------------------------------------------------
						[DEFINITIONS & ON/OFF OPTIONS]
------------------------------------------------------------------------------*/

//---------------------------#[CHOOSE EFFECTS]#--------------------------------\\

#define FXAA_PASS					 1		//#HQ Fast Approximate Anti Aliasing
#define LUMA_SHARPEN				 1		//#HQ Luma-Based Sharpen, looks similar to a negative mip LOD Bias, enhances texture fidelity.
#define COLOR_CORRECTION			 1		//#Scene Color Correction & Tonemapping pass of the HDR.
#define BLOOM_PASS					 0		//#Bloom high pass of the HDR.
#define HDR_PASS					 1		//#Defog & Range Pass of the HDR
#define INTELLI_VIBRANCE			 1		//#Dynamic per pixel vibrance. It can also be used as color offset calibration, due to the HDR skewing the gamut.
#define DYNAMIC_CONTRAST			 1		//#S-Curve Dynamic Scene Contrast Enhancement.
#define PERPIXEL_LIGHTING			 1		//#Light Attenuation, (lighting enhancement WIP) doesn't work for all games, is why it's optional.
#define DITHERING					 1		//#Dithering to simulate more colors than your monitor can display. This lessens color banding.

/*------------------------------------------------------------------------------
					[SHADER FX CONFIG OPTIONS & LOCAL VARS]
------------------------------------------------------------------------------*/

#if (FXAA_PASS == 1)
	#define FXAA_PC 1
	#define FXAA_HLSL_5 1
	#define FXAA_STRONG 1			//[0 or 1] 0 = subtle 1= more jaggie removal
	#define FXAA_QUALITY__SUBPIX (0)
	#define FXAA_QUALITY__EDGE_THRESHOLD (1.0/16.0)
	#define FXAA_QUALITY__EDGE_THRESHOLD_MIN (1.0/16.0)
	#define FXAA_QUALITY__PRESET 39
	#define FXAA_GATHER4_ALPHA 1
#endif

#if (COLOR_CORRECTION == 1)
	#define Saturation 0.00  		//[-1.000 to 1.000] Adjust saturation
	#define Bleach 0.00      		//[0.000 to 1.000] Brightens the shadows and fades the colors
	#define Luma 100.00f			//Luma Percent Avg Correction(luma correction).
	#define Exposure 100.00f		//White Percent Avg Correction (brightness correction)
	#define WhitePoint 0.99f		//Whitepoint Avg (fog correction)
	#define BlackLevel 0.03f		//black levels for (Shadow correction)
#endif

#if (BLOOM_PASS == 1)
	#define OverlayGlow 0			//[0 or 1] Overlay glow for the bloom, Needs some work, but it's ok for now.
	#define GlowFactor 0.50			//[0.00 to 0.90] Amount of blur for the glow
	#define BloomThreshold 21.50f 	//[0.00 to 50.00] Threshold for what is a bright light (that causes bloom) and what isn't.
	#define BloomPower 5.200     	//[0.000 to 8.000] Strength of the bloom
	#define BloomWidth 0.01425    	//[0.0000 to 1.0000] Width of the bloom
#endif

#if (HDR_PASS == 1)
	#define Gamma 1.00f  			//[0.00 to 8.00] Lower Values = more Gamma toning(darker), higher Values = brighter
	#define HdrRadius 0.80f  		//[0.00 to 8.00] Raise this to make the effect stronger and also brighter
#endif

#if (LUMA_SHARPEN == 1)
	#define sharp_strength 0.50f   	//[0.10 to 3.00] Strength of the sharpening
	#define sharp_clamp    0.035f  	//[0.000 to 1.000] Limits maximum amount of sharpening a pixel recieves - Default is 0.035

	#define pattern 9        		//[1|2|3|4|8|9] Choose a sample pattern. 1 = Fast, 2 = Normal, 3 = Wider, 4 = Pyramid shaped.
	#define offset_bias 6.0  		//[0.0 to 6.0] Offset bias adjusts the radius of the sampling pattern.

	#define show_sharpen 0   		//[0 or 1] Visualize the strength of the sharpen (multiplied by 4 to see it better)
	
	#define CoefLuma float3(0.212656f, 0.715158f, 0.072186f)
	#define sharp_strength_luma (CoefLuma * sharp_strength)
#endif

#if (INTELLI_VIBRANCE == 1)
	#define Vibrance 0.05f 			//[-1.00 to 1.00] Intelligently saturates (or desaturates) the pixels depending on their original.
#endif

#if (DYNAMIC_CONTRAST == 1)
	#define Curves_mode 0				//[0|1|2] Choose what to apply contrast to. 0 = Luma, 1 = Chroma, 2 = both Luma and Chroma. Default is 0 (Luma)
	#define Curves_contrast 0.35f		//[-1.00 to 1.00] The amount of contrast you want
	#define Curves_formula 10			//[1|2|3|4|5|6|7|8|9|10] The contrast s-curve you want to use.
#endif

#if (DITHERING == 1)
	#define dither_method 2  			//[1 or 2] 1 = Ordering dithering (good and very fast), 2 = Random dithering (even better dithering but not as fast)
#endif

/*------------------------------------------------------------------------------
							[GLOBAL VARIABLES]
------------------------------------------------------------------------------*/

Texture2D Texture;
SamplerState colorSampler : register(s0)
{
    Texture   = <Texture>;
	MinFilter = ANISOTROPIC;
	MagFilter = ANISOTROPIC;
	MipFilter = ANISOTROPIC;
	MaxAnisotropy = 16;
	AddressU = Clamp;
	AddressV = Clamp;
	SRGBTexture = true;
};

Texture2D phongTexture;
SamplerState phongSampler : register(s1)
{
    Texture   = <phongTexture>;
	MinFilter = ANISOTROPIC;
	MagFilter = ANISOTROPIC;
	MipFilter = ANISOTROPIC;
	MaxAnisotropy = 16;
	AddressU = Clamp;
	AddressV = Clamp;
};

Texture2D phongWhiteTexture;
SamplerState phongWhiteSampler : register(s2)
{
    Texture   = <phongWhiteTexture>;
	MinFilter = ANISOTROPIC;
	MagFilter = ANISOTROPIC;
	MipFilter = ANISOTROPIC;
	MaxAnisotropy = 16;
	AddressU = Clamp;
	AddressV = Clamp;
};

Buffer<float4>g_Buffer;

cbuffer ViewportBuffer
{
	float4 _rcpFrame : register(c0);
	float4 _rcpFrameOpt : register(c1);
};

cbuffer MatrixBuffer
{
    matrix worldMatrix;
    matrix viewMatrix;
    matrix projectionMatrix;
};

RasterizerState rsSolid
{
	  DepthBias = 0;
	  MultisampleEnable = true;
	  AntialiasedLineEnable = true;
};

BlendState Blend
{
	BlendEnable = true;
};

struct DirectionalLight
{
	float4 color;
	float3 dir;
};

struct Material
{
	float Ka, Kd, Ks, A;
};

struct VS_OUTPUT
{
    float4 p : SV_Position;
	float2 t : TEXCOORD;
};

struct VS_INPUT
{
    float4 p : POSITION;
	float2 t : TEXCOORD;
	float3 n : NORMAL;
	float4 d : SV_Depth;
};

struct PS_INPUT_PV
{
	float4 p : SV_Position;
	float2 t : TEXCOORD;
	float4 i : COLOR;
};

struct PS_INPUT_PP_PHONG
{
	float4 p : SV_Position;
	float4 wp : POSITION;
	float2 t : TEXCOORD;
	float3 n : TEXCOORD1;
};

struct PS_INPUT_PP_BLINNPHONG
{
	float4 p : SV_Position;
	float2 t : TEXCOORD;
	float3 n : TEXCOORD1;
	float3 h : TEXCOORD2;
};


#if SHADER_MODEL >= 0x400
	#define TexSample(s, t) Texture.Sample(s, t)
#else
	#define TexSample(s, t) tex2D(s, t)
#endif

#define BUFFER_RCP_WIDTH (1.0/4096)
#define BUFFER_RCP_HEIGHT (1.0/2048)
#define BUFFER_WIDTH _rcpFrame.x
#define BUFFER_HEIGHT _rcpFrame.y
#define screen_size float2(BUFFER_WIDTH,BUFFER_HEIGHT)
#define px BUFFER_RCP_WIDTH
#define py BUFFER_RCP_HEIGHT
#define pixel float2(px,py)

/*------------------------------------------------------------------------------
							[VERTEX CODE SECTION] - not used
------------------------------------------------------------------------------*/

VS_OUTPUT vs_main(VS_INPUT IN)
{	
	VS_OUTPUT OUT;
	
	OUT.p = mul(IN.p, worldMatrix);
    OUT.p = mul(IN.p, viewMatrix);
    OUT.p = mul(IN.p, projectionMatrix);
	OUT.p=float4(IN.p.x,IN.p.y,IN.p.z,1.0);
	
	return OUT;
}

/*------------------------------------------------------------------------------
                       [TEXTURE FILTER FUNCTIONS]
------------------------------------------------------------------------------*/

float4 tex2DBiLinear(SamplerState textureSampler_i, float2 texCoord_i)
{	
	float fWidth = (1.0/_rcpFrame.x);
	float fHeight = (1.0/_rcpFrame.y);	
	
	float texelSizeX = (1.0 / fWidth); //size of one texel 
	float texelSizeY = (1.0 / fHeight); //size of one texel 

	int nX = int( texCoord_i.x * fWidth );
	int nY = int( texCoord_i.y * fHeight );
	
	float2 texCoord_New = float2( ( float( nX ) + 0.5 ) / fWidth,
							  ( float( nY ) + 0.5 ) / fHeight );
	// Take nearest two data in current row.
    float4 p0q0 = TexSample(textureSampler_i, texCoord_New);
    float4 p1q0 = TexSample(textureSampler_i, texCoord_New + float2(texelSizeX, 0));

	// Take nearest two data in bottom row.
    float4 p0q1 = TexSample(textureSampler_i, texCoord_New + float2(0, texelSizeY));
    float4 p1q1 = TexSample(textureSampler_i, texCoord_New + float2(texelSizeX , texelSizeY));

    float a = frac(texCoord_i.x * fWidth); // Get Interpolation factor for X direction.
											 // Fraction near to valid data.
	// Interpolation in X direction.
    float4 pInterp_q0 = lerp( p0q0, p1q0, a ); // Interpolates top row in X direction.
    float4 pInterp_q1 = lerp( p0q1, p1q1, a ); // Interpolates bottom row in X direction.

    float b = frac(texCoord_i.y * fHeight); // Get Interpolation factor for Y direction.
	
    return lerp(pInterp_q0, pInterp_q1, b); // Interpolate in Y direction.
}

float4 BlurFunction7x7(float4 blurColor : COLOR, float2 t : TEXCOORD) : COLOR0
{
	float fWidth = (1.0/_rcpFrame.x);
	float fHeight = (1.0/_rcpFrame.y);

	return (
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  -3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  -3.0f / fHeight)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  -2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  -2.0f / fHeight)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  -1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  -1.0f / fHeight)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, 0)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, 0)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, 0)) +
    Texture.Sample(colorSampler, t + float2(0,              0)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  0)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  0)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  0)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, 1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, 1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, 1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  1.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  1.0f / fHeight)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, 2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, 2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, 2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  2.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  2.0f / fHeight)) +
 
    Texture.Sample(colorSampler, t + float2(-3.0f / fWidth, 3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-2.0f / fWidth, 3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(-1.0f / fWidth, 3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(0,              3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(1.0f / fWidth,  3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(2.0f / fWidth,  3.0f / fHeight)) +
    Texture.Sample(colorSampler, t + float2(3.0f / fWidth,  3.0f / fHeight))
  ) / 49;
  
  //return blurColor;
}

/*------------------------------------------------------------------------------
                             [FXAA CODE SECTION]
------------------------------------------------------------------------------*/

#if (FXAA_PASS == 1)
#ifndef FXAA_PC
    //
    // FXAA Quality
    // The high quality PC algorithm.
    //
    #define FXAA_PC 1
#endif
/*--------------------------------------------------------------------------*/
#ifndef FXAA_HLSL_5
    #define FXAA_HLSL_5 1
#endif

/*--------------------------------------------------------------------------*/
#ifndef FXAA_GATHER4_ALPHA
    #if (FXAA_HLSL_5 == 1)
        #define FXAA_GATHER4_ALPHA 1
    #endif
#endif

/*============================================================================
                        FXAA QUALITY - TUNING KNOBS
============================================================================*/
#ifndef FXAA_QUALITY__EDGE_THRESHOLD
    #define FXAA_QUALITY__EDGE_THRESHOLD (1.0/16.0)
#endif
/*--------------------------------------------------------------------------*/
#ifndef FXAA_QUALITY__EDGE_THRESHOLD_MIN
    #define FXAA_QUALITY__EDGE_THRESHOLD_MIN (1.0/16.0)
#endif
/*--------------------------------------------------------------------------*/
#ifndef FXAA_QUALITY__SUBPIX
    #define FXAA_QUALITY__SUBPIX (0)
#endif
/*--------------------------------------------------------------------------*/
#ifndef FXAA_QUALITY__PRESET
    #define FXAA_QUALITY__PRESET 39
#endif

/*============================================================================
                     FXAA QUALITY - EXTREME QUALITY
============================================================================*/
#if (FXAA_QUALITY__PRESET == 39)
#if (FXAA_STRONG == 1)
	#define FXAA_QUALITY__PS 12
    #define FXAA_QUALITY__P0 0.5
    #define FXAA_QUALITY__P1 0.5
    #define FXAA_QUALITY__P2 0.5
    #define FXAA_QUALITY__P3 0.5
    #define FXAA_QUALITY__P4 0.5
    #define FXAA_QUALITY__P5 1.5
    #define FXAA_QUALITY__P6 2.0
    #define FXAA_QUALITY__P7 2.0
    #define FXAA_QUALITY__P8 2.0
    #define FXAA_QUALITY__P9 2.0
    #define FXAA_QUALITY__P10 4.0
    #define FXAA_QUALITY__P11 8.0
	#define FXAA_QUALITY__P12 8.0
#else
    #define FXAA_QUALITY__PS 12
    #define FXAA_QUALITY__P0 1.0
    #define FXAA_QUALITY__P1 1.0
    #define FXAA_QUALITY__P2 1.0
    #define FXAA_QUALITY__P3 1.0
    #define FXAA_QUALITY__P4 1.0
    #define FXAA_QUALITY__P5 1.5
    #define FXAA_QUALITY__P6 2.0
    #define FXAA_QUALITY__P7 2.0
    #define FXAA_QUALITY__P8 2.0
    #define FXAA_QUALITY__P9 2.0
    #define FXAA_QUALITY__P10 4.0
    #define FXAA_QUALITY__P11 8.0
#endif
#endif

/*============================================================================

                                API PORTING

============================================================================*/

/*--------------------------------------------------------------------------*/
#if FXAA_HLSL_5
    #define FxaaInt2 int2
    #define FxaaFloat2 float2
    #define FxaaFloat3 float3
    #define FxaaFloat4 float4
    #define FxaaDiscard clip(-1)
    #define FxaaDot3(a, b) dot(a, b)
    #define FxaaSat(x) saturate(x)
    #define FxaaLerp(x,y,s) lerp(x,y,s)
    struct FxaaTex { SamplerState smpl; Texture2D tex; };
    #define FxaaTexTop(t, p) t.tex.SampleLevel(t.smpl, p, 0.0)
    #define FxaaTexOff(t, p, o, r) t.tex.SampleLevel(t.smpl, p, 0.0, o)
    #define FxaaTexAlpha4(t, p, r) t.tex.GatherAlpha(t.smpl, p)
    #define FxaaTexOffAlpha4(t, p, o, r) t.tex.GatherAlpha(t.smpl, p, o)
#endif

/*============================================================================

                             FXAA3 QUALITY - PC

============================================================================*/
#if (FXAA_PC == 1)
/*--------------------------------------------------------------------------*/
float4 FxaaPixelShader(
    // {xy} = center of pixel
    float2 pos,
    // {xyzw} = not used on FXAA3 Quality
    float4 posPos,
    // {rgb_} = color in linear or perceptual color space
    // {___a} = luma in perceptual color space (not linear)
    FxaaTex tex,
    // This must be from a constant/uniform.
    // {x_} = 1.0/screenWidthInPixels
    // {_y} = 1.0/screenHeightInPixels
    float2 rcpFrame,
    // {xyzw} = not used on FXAA3 Quality
    float4 rcpFrameOpt
) {
/*--------------------------------------------------------------------------*/
    float2 posM;
    posM.x = pos.x;
    posM.y = pos.y;
    #if (FXAA_GATHER4_ALPHA == 1)
        #if (FXAA_DISCARD == 0)
            float4 rgbyM = FxaaTexTop(tex, posM);
            #define lumaM rgbyM.w
        #endif
        float4 luma4A = FxaaTexAlpha4(tex, posM, rcpFrame.xy);
        float4 luma4B = FxaaTexOffAlpha4(tex, posM, FxaaInt2(-1, -1), rcpFrame.xy);
        #if (FXAA_DISCARD == 1)
            #define lumaM luma4A.w
        #endif
        #define lumaE luma4A.z
        #define lumaS luma4A.x
        #define lumaSE luma4A.y
        #define lumaNW luma4B.w
        #define lumaN luma4B.z
        #define lumaW luma4B.x
    #else
        float4 rgbyM = FxaaTexTop(tex, posM);
        #define lumaM rgbyM.w
        float lumaS = FxaaTexOff(tex, posM, FxaaInt2( 0, 1), rcpFrame.xy).w;
        float lumaE = FxaaTexOff(tex, posM, FxaaInt2( 1, 0), rcpFrame.xy).w;
        float lumaN = FxaaTexOff(tex, posM, FxaaInt2( 0,-1), rcpFrame.xy).w;
        float lumaW = FxaaTexOff(tex, posM, FxaaInt2(-1, 0), rcpFrame.xy).w;
    #endif
/*--------------------------------------------------------------------------*/
    float maxSM = max(lumaS, lumaM);
    float minSM = min(lumaS, lumaM);
    float maxESM = max(lumaE, maxSM);
    float minESM = min(lumaE, minSM);
    float maxWN = max(lumaN, lumaW);
    float minWN = min(lumaN, lumaW);
    float rangeMax = max(maxWN, maxESM);
    float rangeMin = min(minWN, minESM);
    float rangeMaxScaled = rangeMax * FXAA_QUALITY__EDGE_THRESHOLD;
    float range = rangeMax - rangeMin;
    float rangeMaxClamped = max(FXAA_QUALITY__EDGE_THRESHOLD_MIN, rangeMaxScaled);
    bool earlyExit = range < rangeMaxClamped;
/*--------------------------------------------------------------------------*/
    if(earlyExit)
        #if (FXAA_DISCARD == 1)
            FxaaDiscard;
        #else
            return rgbyM;
        #endif
/*--------------------------------------------------------------------------*/
    #if (FXAA_GATHER4_ALPHA == 0)
        float lumaNW = FxaaTexOff(tex, posM, FxaaInt2(-1,-1), rcpFrame.xy).w;
        float lumaSE = FxaaTexOff(tex, posM, FxaaInt2( 1, 1), rcpFrame.xy).w;
        float lumaNE = FxaaTexOff(tex, posM, FxaaInt2( 1,-1), rcpFrame.xy).w;
        float lumaSW = FxaaTexOff(tex, posM, FxaaInt2(-1, 1), rcpFrame.xy).w;
    #else
        float lumaNE = FxaaTexOff(tex, posM, FxaaInt2(1, -1), rcpFrame.xy).w;
        float lumaSW = FxaaTexOff(tex, posM, FxaaInt2(-1, 1), rcpFrame.xy).w;
    #endif
/*--------------------------------------------------------------------------*/
    float lumaNS = lumaN + lumaS;
    float lumaWE = lumaW + lumaE;
    float subpixRcpRange = 1.0/range;
    float subpixNSWE = lumaNS + lumaWE;
    float edgeHorz1 = (-2.0 * lumaM) + lumaNS;
    float edgeVert1 = (-2.0 * lumaM) + lumaWE;
/*--------------------------------------------------------------------------*/
    float lumaNESE = lumaNE + lumaSE;
    float lumaNWNE = lumaNW + lumaNE;
    float edgeHorz2 = (-2.0 * lumaE) + lumaNESE;
    float edgeVert2 = (-2.0 * lumaN) + lumaNWNE;
/*--------------------------------------------------------------------------*/
    float lumaNWSW = lumaNW + lumaSW;
    float lumaSWSE = lumaSW + lumaSE;
    float edgeHorz4 = (abs(edgeHorz1) * 2.0) + abs(edgeHorz2);
    float edgeVert4 = (abs(edgeVert1) * 2.0) + abs(edgeVert2);
    float edgeHorz3 = (-2.0 * lumaW) + lumaNWSW;
    float edgeVert3 = (-2.0 * lumaS) + lumaSWSE;
    float edgeHorz = abs(edgeHorz3) + edgeHorz4;
    float edgeVert = abs(edgeVert3) + edgeVert4;
/*--------------------------------------------------------------------------*/
    float subpixNWSWNESE = lumaNWSW + lumaNESE;
    float lengthSign = rcpFrame.x;
    bool horzSpan = edgeHorz >= edgeVert;
    float subpixA = subpixNSWE * 2.0 + subpixNWSWNESE;
/*--------------------------------------------------------------------------*/
    if(!horzSpan) lumaN = lumaW;
    if(!horzSpan) lumaS = lumaE;
    if(horzSpan) lengthSign = rcpFrame.y;
    float subpixB = (subpixA * (1.0/12.0)) - lumaM;
/*--------------------------------------------------------------------------*/
    float gradientN = lumaN - lumaM;
    float gradientS = lumaS - lumaM;
    float lumaNN = lumaN + lumaM;
    float lumaSS = lumaS + lumaM;
    bool pairN = abs(gradientN) >= abs(gradientS);
    float gradient = max(abs(gradientN), abs(gradientS));
    if(pairN) lengthSign = -lengthSign;
    float subpixC = FxaaSat(abs(subpixB) * subpixRcpRange);
/*--------------------------------------------------------------------------*/
    float2 posB;
    posB.x = posM.x;
    posB.y = posM.y;
    float2 offNP;
    offNP.x = (!horzSpan) ? 0.0 : rcpFrame.x;
    offNP.y = ( horzSpan) ? 0.0 : rcpFrame.y;
    if(!horzSpan) posB.x += lengthSign * 0.5;
    if( horzSpan) posB.y += lengthSign * 0.5;
/*--------------------------------------------------------------------------*/
    float2 posN;
    posN.x = posB.x - offNP.x * FXAA_QUALITY__P0;
    posN.y = posB.y - offNP.y * FXAA_QUALITY__P0;
    float2 posP;
    posP.x = posB.x + offNP.x * FXAA_QUALITY__P0;
    posP.y = posB.y + offNP.y * FXAA_QUALITY__P0;
    float subpixD = ((-2.0)*subpixC) + 3.0;
    float lumaEndN = FxaaTexTop(tex, posN).w;
    float subpixE = subpixC * subpixC;
    float lumaEndP = FxaaTexTop(tex, posP).w;
/*--------------------------------------------------------------------------*/
    if(!pairN) lumaNN = lumaSS;
    float gradientScaled = gradient * 1.0/4.0;
    float lumaMM = lumaM - lumaNN * 0.5;
    float subpixF = subpixD * subpixE;
    bool lumaMLTZero = lumaMM < 0.0;
/*--------------------------------------------------------------------------*/
    lumaEndN -= lumaNN * 0.5;
    lumaEndP -= lumaNN * 0.5;
    bool doneN = abs(lumaEndN) >= gradientScaled;
    bool doneP = abs(lumaEndP) >= gradientScaled;
    if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P1;
    if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P1;
    bool doneNP = (!doneN) || (!doneP);
    if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P1;
    if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P1;
/*--------------------------------------------------------------------------*/
    if(doneNP) {
        if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
        if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
        if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
        if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
        doneN = abs(lumaEndN) >= gradientScaled;
        doneP = abs(lumaEndP) >= gradientScaled;
        if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P2;
        if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P2;
        doneNP = (!doneN) || (!doneP);
        if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P2;
        if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P2;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 3)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P3;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P3;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P3;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P3;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 4)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P4;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P4;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P4;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P4;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 5)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P5;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P5;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P5;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P5;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 6)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P6;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P6;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P6;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P6;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 7)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P7;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P7;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P7;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P7;
/*--------------------------------------------------------------------------*/
    #if (FXAA_QUALITY__PS > 8)
    if(doneNP) {
        if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
        if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
        if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
        if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
        doneN = abs(lumaEndN) >= gradientScaled;
        doneP = abs(lumaEndP) >= gradientScaled;
        if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P8;
        if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P8;
        doneNP = (!doneN) || (!doneP);
        if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P8;
        if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P8;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 9)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P9;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P9;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P9;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P9;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 10)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P10;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P10;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P10;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P10;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 11)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P11;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P11;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P11;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P11;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 12)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaTexTop(tex, posN.xy).w;
		if(!doneP) lumaEndP = FxaaTexTop(tex, posP.xy).w;
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P12;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P12;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P12;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P12;
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
	#endif
/*--------------------------------------------------------------------------*/
	}
/*--------------------------------------------------------------------------*/
    float dstN = posM.x - posN.x;
    float dstP = posP.x - posM.x;
    if(!horzSpan) dstN = posM.y - posN.y;
    if(!horzSpan) dstP = posP.y - posM.y;
/*--------------------------------------------------------------------------*/
    bool goodSpanN = (lumaEndN < 0.0) != lumaMLTZero;
    float spanLength = (dstP + dstN);
    bool goodSpanP = (lumaEndP < 0.0) != lumaMLTZero;
    float spanLengthRcp = 1.0/spanLength;
/*--------------------------------------------------------------------------*/
    bool directionN = dstN < dstP;
    float dst = min(dstN, dstP);
    bool goodSpan = directionN ? goodSpanN : goodSpanP;
    float subpixG = subpixF * subpixF;
    float pixelOffset = (dst * (-spanLengthRcp)) + 0.5;
    float subpixH = subpixG * FXAA_QUALITY__SUBPIX;
/*--------------------------------------------------------------------------*/
    float pixelOffsetGood = goodSpan ? pixelOffset : 0.0;
    float pixelOffsetSubpix = max(pixelOffsetGood, subpixH);
    if(!horzSpan) posM.x += pixelOffsetSubpix * lengthSign;
    if( horzSpan) posM.y += pixelOffsetSubpix * lengthSign;
    #if (FXAA_DISCARD == 1)
        return FxaaTexTop(tex, posM);
    #else
        return float4(FxaaTexTop(tex, posM).xyz, lumaM);
    #endif
}
/*==========================================================================*/
#endif
float4 FXAAPass(float4 posPos : COLOR, float2 t : TEXCOORD) : SV_Target
{
	FxaaTex tex;
	float2 pos = t;

	#if SHADER_MODEL >= 0x400
		tex.tex = Texture;
		tex.smpl = colorSampler;
	#else
		tex = colorSampler;
	#endif

	posPos = FxaaPixelShader(pos, posPos, tex, _rcpFrame.xy, _rcpFrameOpt);

	return saturate(posPos);
}
#endif
/*------------------------------------------------------------------------------
                        [HDR BLOOM PASS CODE SECTION]
------------------------------------------------------------------------------*/

#if (BLOOM_PASS == 1)
float4 BloomPass(float4 ColorInput2 : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float3 BlurColor2 = 0;
	float3 Blurtemp = 0;

	float MaxDistance = sqrt(8*BloomWidth);
	float CurDistance = 0;
	
	#if (OverlayGlow == 1)
		ColorInput2 = lerp(ColorInput2, BlurFunction7x7(ColorInput2, t), GlowFactor)*1.2;
	#endif
	
	float Samplecount = 16.0;
	
	float2 blurtempvalue = t * pixel * BloomWidth;
	
	float2 BloomSample = Texture.SampleLevel(colorSampler, t, 0, float2(2.5,-2.5));
	float2 BloomSampleValue;
	
	for(BloomSample.x = (2.5); BloomSample.x > -2.0; BloomSample.x = BloomSample.x - 1.0) // runs 5 times
	{
        BloomSampleValue.x = BloomSample.x * blurtempvalue.x;
        float2 distancetemp = BloomSample.x * BloomSample.x * BloomWidth;
        
		for(BloomSample.y = (- 2.5); BloomSample.y < 2.0; BloomSample.y = BloomSample.y + 1.0) // runs 5 ( * 5) times
		{
            distancetemp.y = BloomSample.y * BloomSample.y;
			CurDistance = sqrt( (distancetemp.y * BloomWidth) + distancetemp.x);
			
			BloomSampleValue.y = BloomSample.y * blurtempvalue.y;
			Blurtemp.rgb = Texture.Sample(colorSampler, float2(t + BloomSampleValue)).rgb;
			
			BlurColor2.rgb += lerp(Blurtemp.rgb,ColorInput2.rgb, 1.0 - ((MaxDistance - CurDistance) / MaxDistance));
		}
	}
	
	BlurColor2.rgb = (BlurColor2.rgb / (Samplecount - (BloomPower - BloomThreshold*5)));
	float Bloomamount = (dot(ColorInput2.rgb, float3(0.212656f, 0.715158f, 0.072186f)));
	float3 BlurColor = BlurColor2.rgb * BloomPower;

	ColorInput2.rgb = lerp(ColorInput2.rgb, BlurColor.rgb, Bloomamount);	

	return saturate(ColorInput2);
}
#endif

/*------------------------------------------------------------------------------
							[HDR PASS CODE SECTION]
------------------------------------------------------------------------------*/

#if (HDR_PASS == 1)
float4 HDRPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float3 c_center = Texture.SampleLevel(colorSampler, t, 0).rgb;

	float radius1 = 0.793f;
	float3 bloom_sum1 = Texture.SampleLevel(colorSampler, t, 0, float2(1.5, -1.5) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(-1.5, -1.5) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(1.5, 1.5) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(-1.5, 1.5) * radius1).rgb;

	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(0, -2.5) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(0, 2.5) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(-2.5, 0) * radius1).rgb;
	bloom_sum1 += Texture.SampleLevel(colorSampler, t, 0, float2(2.5, 0) * radius1).rgb;

	bloom_sum1 *= 0.010f;

	float3 bloom_sum2 = Texture.SampleLevel(colorSampler, t, 0, float2(1.5, -1.5) * HdrRadius).rgb;
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(-1.5, -1.5) * HdrRadius).rgb;
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(1.5, 1.5) * HdrRadius).rgb;
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(-1.5, 1.5) * HdrRadius).rgb;


	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(0, -2.5) * HdrRadius).rgb;	
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(0, 2.5) * HdrRadius).rgb;
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(-2.5, 0) * HdrRadius).rgb;
	bloom_sum2 += Texture.SampleLevel(colorSampler, t, 0, float2(2.5, 0) * HdrRadius).rgb;

	bloom_sum2 *= 0.020f;

	float dist = HdrRadius - radius1;

	float3 HDR = (c_center + (bloom_sum2 - bloom_sum1)) * dist;
	float3 blend = HDR + colorInput.rgb;
	colorInput.rgb = HDR + pow(blend, (1.0/Gamma)); // pow - don't use fractions for HDRpower

	return saturate(colorInput);
}
#endif

/*------------------------------------------------------------------------------
                          [LUMA SHARPEN CODE SECTION]
						[Adapted from CeeJay's SweetFX]
	[http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (LUMA_SHARPEN == 1)
float4 LumaSharpenPass(float4 inputcolor : COLOR, float2 t : TEXCOORD) : SV_Target 
{

  // -- Get the original pixel --
  float3 ori = Texture.SampleLevel(colorSampler, t, 0).rgb;       // ori = original pixel

  // -- Pattern 1 -- A (fast) 7 tap gaussian using only 2+1 texture fetches.
  #if (pattern == 1)
    float3 blur_ori = Texture.SampleLevel(colorSampler, t + (float2(px,py) / 3) * offset_bias, 0).rgb;  // North West
    blur_ori += Texture.SampleLevel(colorSampler, t + (float2(-px,-py) / 3) * offset_bias, 0).rgb; // South East

    blur_ori /= 2;  //Divide by the number of texture fetches
    
    sharp_strength_luma *= 1.5; // Adjust strength to aproximate the strength of pattern 2
  #endif
  
  // -- Pattern 2 -- A 9 tap gaussian using 4+1 texture fetches.
  #if (pattern == 2)
    float3 blur_ori = Texture.SampleLevel(colorSampler, t + float2(px,-py) * 0.5 * offset_bias, 0).rgb; // South East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(-px,-py) * 0.5 * offset_bias, 0).rgb;  // South West    
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(px,py) * 0.5 * offset_bias, 0).rgb; // North East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(-px,py) * 0.5 * offset_bias, 0).rgb; // North West

    blur_ori *= 0.25;  // ( /= 4) Divide by the number of texture fetches
  #endif 

  // -- Pattern 3 -- An experimental 17 tap gaussian using 4+1 texture fetches.
  #if (pattern == 3)
    float3 blur_ori = Texture.SampleLevel(colorSampler, t + float2(0.4*px,-1.2*py)* offset_bias, 0).rgb;  // South South East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(-1.2*px,-0.4*py) * offset_bias, 0).rgb; // West South West
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(1.2*px,0.4*py) * offset_bias, 0).rgb; // East North East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(-0.4*px,1.2*py) * offset_bias, 0).rgb; // North North West

    blur_ori *= 0.25;  // ( /= 4) Divide by the number of texture fetches
    
    sharp_strength_luma *= 0.51;
  #endif

  // -- Pattern 4 -- A 9 tap high pass (pyramid filter) using 4+1 texture fetches.
  #if (pattern == 4)
    float3 blur_ori = Texture.SampleLevel(colorSampler, t + float2(0.5 * px,-py * offset_bias), 0).rgb;  // South South East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(offset_bias * -px,0.5 * -py), 0).rgb; // West South West
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(offset_bias * px,0.5 * py), 0).rgb; // East North East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(0.5 * -px,py * offset_bias), 0).rgb; // North North West

    //blur_ori += (2 * ori); // Probably not needed. Only serves to lessen the effect.
	
    blur_ori /= 4.0;  //Divide by the number of texture fetches

    sharp_strength_luma *= 0.666; // Adjust strength to aproximate the strength of pattern 2
  #endif

  // -- Pattern 8 -- A (slower) 9 tap gaussian using 9 texture fetches.
  #if (pattern == 8)
    half3 blur_ori = Texture.SampleLevel(colorSampler, t + float2(-px,py) * offset_bias, 0).rgb; // North West
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(px,-py) * offset_bias, 0).rgb;     // South East
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(-px,-py)  * offset_bias, 0).rgb;  // South West
    blur_ori += Texture.SampleLevel(colorSampler, t + float2(px,py) * offset_bias, 0).rgb;    // North East
    
    half3 blur_ori2 = Texture.SampleLevel(colorSampler, t + float2(0,py) * offset_bias, 0).rgb; // North
    blur_ori2 += Texture.SampleLevel(colorSampler, t + float2(0,-py) * offset_bias, 0).rgb;    // South
    blur_ori2 += Texture.SampleLevel(colorSampler, t + float2(-px,0) * offset_bias, 0).rgb;   // West
    blur_ori2 += Texture.SampleLevel(colorSampler, t + float2(px,0) * offset_bias, 0).rgb;   // East
    blur_ori2 *= 2.0;

    blur_ori += blur_ori2;
    blur_ori += (ori * 4);

    blur_ori /= 16.0;  //Divide by the number of texture fetches
  #endif

  // -- Pattern 9 -- A (slower) 9 tap high pass using 9 texture fetches.
  #if (pattern == 9)
    float3 blur_ori = Texture.SampleLevel(colorSampler, (t + float2(-px,py) * offset_bias), 0).rgb; // North West
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(px,-py) * offset_bias), 0).rgb;     // South East
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(-px,-py)  * offset_bias), 0).rgb;  // South West
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(px,py) * offset_bias), 0).rgb;    // North East
    
    blur_ori += ori.rgb; // Probably not needed. Only serves to lessen the effect.
    
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(0,py) * offset_bias), 0).rgb;    // North
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(0,-py) * offset_bias), 0).rgb;  // South
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(-px,0) * offset_bias), 0).rgb; // West
    blur_ori += Texture.SampleLevel(colorSampler, (t + float2(px,0) * offset_bias), 0).rgb; // East

    blur_ori /= 9;  //Divide by the number of texture fetches
  #endif
  
  // -- Calculate the sharpening --  
  float3 sharp = ori - blur_ori;  //Subtracting the blurred image from the original image
  
  // -- Adjust strength of the sharpening --
  float sharp_luma = dot(sharp, sharp_strength_luma); //Calculate the luma and adjust the strength

  // -- Clamping the maximum amount of sharpening to prevent halo artifacts --
  sharp_luma = clamp(sharp_luma, -sharp_clamp, sharp_clamp);  //TODO Try a curve function instead of a clamp

  float4 done = inputcolor + sharp_luma;    // Add the sharpening to the input color.

  #if (show_sharpen == 1)
    done = saturate(0.5f + (sharp_luma * 4)).rrrr;
  #endif

  return saturate(done);
}
#endif

/*------------------------------------------------------------------------------
						  [COLOR CORRECTION CODE SECTION]
------------------------------------------------------------------------------*/

#if (COLOR_CORRECTION == 1)
float4 TonemapPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{

	// Average scene luminance is stored in R channel
	float avgLuminance = Luma;
	float exposure = Exposure;
	float whitePoint = WhitePoint;
	
	float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
	float lum = dot(lumCoeff, colorInput.rgb);
	
	float3 blend = lum.rrr;
	
	float L = saturate( 10.0f * (lum - 0.45f) );
  	
	float3 result1 = 2.0f * colorInput.rgb * blend;
	float3 result2 = 1.0f - 2.0f * (1.0f - blend) * (1.0f - colorInput.rgb);
	
	float3 newColor = lerp(result1, result2, L);
	float3 A2 = Bleach * colorInput.rgb;
	float3 mixRGB = A2 * newColor;
	
	colorInput.rgb += ((1.0f - A2) * mixRGB);
	
	float3 middlegray = dot(colorInput,(1.0f/3.0f));
	
	float3 diffcolor = colorInput.rgb - middlegray;
	colorInput.rgb = (colorInput.rgb + diffcolor * Saturation)/(1+(diffcolor*Saturation)); //saturation
	
	float2 pos_color_black = float2(0,0);
	float4 color_black = phongTexture.SampleLevel(phongSampler, t , 0);

	if(color_black.r == 0.0 && color_black.g == 0.0 && color_black.b == 0.0)
	{
		pos_color_black = t;
		colorInput.r -= BlackLevel;
		colorInput.g -= BlackLevel;
		colorInput.b -= BlackLevel;
	}
	
	float2 pos_color_white = float2(1,1);
	float4 color_white = phongWhiteTexture.SampleLevel(phongWhiteSampler, t, 0);

	if(color_white.r == 1.0 && color_white.g == 1.0 && color_white.b == 1.0)
	{
		pos_color_white = t;
		colorInput.r += BlackLevel;
		colorInput.g += BlackLevel;
		colorInput.b += BlackLevel;
	}

	// RGB -> XYZ conversion
	const float3x3 RGB2XYZ = {0.5141364, 0.3238786,  0.16036376,
							  0.265068,  0.67023428, 0.06409157,
							  0.0241188, 0.1228178,  0.84442666};				                    
	float3 XYZ = mul(RGB2XYZ, colorInput.rgb);
  
	// XYZ -> Yxy conversion
	float3 Yxy;
	Yxy.r = XYZ.g;                            // copy luminance Y
	Yxy.g = XYZ.r / (XYZ.r + XYZ.g + XYZ.b ); // x = X / (X + Y + Z)
	Yxy.b = XYZ.g / (XYZ.r + XYZ.g + XYZ.b ); // y = Y / (X + Y + Z)
    
	// (Lp) Map average luminance to the middlegrey zone by scaling pixel luminance
	float Lp = Yxy.r * exposure / avgLuminance;                       
	// (Ld) Scale all luminance within a displayable range of 0 to 1
	Yxy.r = (Lp * (1.0f + Lp/(whitePoint * whitePoint)))/(1.0f + Lp);
  
	// Yxy -> XYZ conversion
	XYZ.r = Yxy.r * Yxy.g / Yxy. b;               // X = Y * x / y
	XYZ.g = Yxy.r;                                // copy luminance Y
	XYZ.b = Yxy.r * (1 - Yxy.g - Yxy.b) / Yxy.b;  // Z = Y * (1-x-y) / y
    
	// XYZ -> RGB conversion
	const float3x3 XYZ2RGB  = { 2.5651,-1.1665,-0.3986,
							   -1.0217, 1.9777, 0.0439, 
							    0.0753, -0.2543, 1.1892};
	colorInput.rgb = mul(XYZ2RGB, XYZ);
	colorInput = clamp(colorInput, 0.0 , 1.0);
	
	return saturate(colorInput);
}
#endif

/*------------------------------------------------------------------------------
						[DYNAMIC VIBRANCE CODE SECTION]
						[Adapted from CeeJay's SweetFX]
	[http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (INTELLI_VIBRANCE == 1)
float4 VibrancePass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float4 color = colorInput;
    float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);

	float luma = dot(lumCoeff, color.rgb);
	
	float max_color = max(colorInput.r, max(colorInput.g,colorInput.b)); //Find the strongest color
	float min_color = min(colorInput.r, min(colorInput.g,colorInput.b)); //Find the weakest color
	
    float color_saturation = max_color - min_color; //The difference between the two is the saturation
  
    color.rgb = lerp(luma, color.rgb, (1.0f + (Vibrance * (1.0f - (sign(Vibrance) * color_saturation)))));
  
	return color; //return the result
}
#endif

/*------------------------------------------------------------------------------
						[DYNAMIC CONTRAST CODE SECTION]
						[Adapted from CeeJay's SweetFX]
    [http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (DYNAMIC_CONTRAST == 1)
float4 CurvesPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
	float Curves_contrast_blend = Curves_contrast;
	float PI = acos(-1); //3.1415926589

	#if (Curves_mode != 2)
		float luma = dot(lumCoeff, colorInput.rgb);
		float3 chroma = colorInput.rgb - luma;
	#endif

	#if (Curves_mode == 2)
		float3 x = colorInput.rgb;
	#elif (Curves_mode == 1)
		float3 x = chroma;
		x = x * 0.5 + 0.5;
	#else
		float x = luma;
	#endif
		
	// -- Curve 1 -- Sin
	#if (Curves_formula == 1)
		x = sin(PI * 0.5 * x); 
		x *= x;  
	#endif

	// -- Curve 2 -- ABS
	#if (Curves_formula == 2)
		x = ( (x - 0.5) / (0.5 + abs(x-0.5)) ) + 0.5;
	#endif

	// -- Curve 3 -- Smoothstep
	#if (Curves_formula == 3)
		x = x*x*(3.0-2.0*x);
	#endif

	// -- Curve 4 -- EXP formula
	#if (Curves_formula == 4)
		x = 1.1048 / (1.0 + exp(-3.0 * (x * 2.0 - 1.0))) - (0.1048 / 2.0);
	#endif

	// -- Curve 5 -- Horner form
	#if (Curves_formula == 5)
		x = x * (x * (1.5-x) + 0.5);
		Curves_contrast_blend = Curves_contrast * 2.0; //I multiply by two to give it a strength closer to the other curves.
	#endif

	// -- Curve 6 -- Perlins smootherstep
	#if (Curves_formula == 6)
		x = x*x*x*(x*(x*6.0 - 15.0) + 10.0);
	#endif

	// -- Curve 7 -- ABS II
	#if (Curves_formula == 7)
		x = ((x-0.5) / ((0.5/(4.0/3.0)) + abs((x-0.5)*1.25))) + 0.5;
	#endif

	// -- Curve 8 -- Techicolor Cinestyle
	#if (Curves_formula == 8)
		x =  x * x *(2 * x-3)  *  (2 * x * x *x - 3 * x * x +3)  *  (4 * x * x * x -6 * x * x - 3) * 0.10;
	#endif

	// -- Curve 9 -- Parabola
	#if (Curves_formula == 9)
		x = 0.5 -(abs(x*2-1)-2) * (x*2-1) * 0.5;
	#endif

	// -- Curve 10 -- Cubic catmull
	#if (Curves_formula == 10)
		float a = 1.00f; //control point 1
		float b = 0.00f; //start point
		float c = 1.00f; //endpoint
		float d = 0.20f; //control point 2
		x = 0.5f * ((-a + 3*b -3*c + d)*x*x*x + (2*a -5*b + 4*c - d)*x*x + (-a+c)*x + 2*b); //A customizable cubic catmull-rom spline
	#endif

	// -- Curve 11 -- Cubic Bezier spline
	#if (Curves_formula == 11)
		float a = 0.00; //start point
		float b = 0.00; //control point 1
		float c = 1.00; //control point 2
		float d = 1.00; //endpoint

		float r  = (1-x);
		float r2 = r*r;
		float r3 = r2 * r;
		float x2 = x*x;
		float x3 = x2*x;

		x = a*(1-x)*(1-x)*(1-x) + 3*b*(1-x)*(1-x)*x + 3*c*(1-x)*x*x + d*x*x*x;
	#endif

	// -- Curve 12 -- Cubic Bezier spline II
	#if (Curves_formula == 12)
		float3 a = float3(0.00,0.00,0.00); //start point
		float3 b = float3(0.25,0.15,0.85); //control point 1
		float3 c = float3(0.75,0.85,0.15); //control point 2
		float3 d = float3(1.00,1.00,1.00); //endpoint

		float3 ab = lerp(a,b,x);           // point between a and b (green)
		float3 bc = lerp(b,c,x);           // point between b and c (green)
		float3 cd = lerp(c,d,x);           // point between c and d (green)
		float3 abbc = lerp(ab,bc,x);       // point between ab and bc (blue)
		float3 bccd = lerp(bc,cd,x);       // point between bc and cd (blue)
		float3 dest = lerp(abbc,bccd,x);   // point on the bezier-curve (black)
		x = dest;
	#endif
	
  #if (Curves_mode == 2) //Both Luma and Chroma
		float3 color = x; 
		colorInput.rgb = lerp(colorInput.rgb, color, Curves_contrast_blend);
  #elif (Curves_mode == 1) //Only Chroma
		x = x * 2 - 1;
		float3 color = luma + x; //Luma + Chroma
		colorInput.rgb = lerp(colorInput.rgb, color, Curves_contrast_blend);
  #else // Curves_mode == 0 //Only Luma
		x = lerp(luma, x, Curves_contrast_blend);
		colorInput.rgb = x + chroma; //Luma + Chroma 
  #endif

  return colorInput;
}
#endif

/*------------------------------------------------------------------------------
							[DITHERING CODE SECTION]
------------------------------------------------------------------------------*/

#if (DITHERING == 1)
float4 DitherPass(float4 colorInput : COLOR, float2 tex : TEXCOORD)
{
   float3 color = colorInput.rgb;
   
   float dither_size = 2.0;  //Size of the dithering grid - I'm using a 2x2 grid here.
   float dither_bit  = 10.0;  //Number of bits per channel. Should be 8 for most monitors.

   #if dither_method == 2 //random dithering

     //make some noise
     float noise = frac(sin(dot(float4(tex,-tex.yx), float4(float2(12.9898,78.233),float2(12.9898,78.233)* acos(-1)))) * 43758.5453); //pseudo random number generator
     noise -= 0.5;
     
     //Calculate how big the shift should be
     float dither_shift = (noise) * (1.0 / (pow(2,dither_bit) - 1.0)); // using noise to determine shift. The noise ahould vary between +- 0.5.
                                                                       
     //Shift the individual colors differently, thus making it even harder to see the dithering pattern
     float3 dither_shift_RGB = float3(-dither_shift, dither_shift, -dither_shift); //subpixel dithering
     
     color.rgb += dither_shift_RGB;
     
   #else //dither_method == 1 , Ordered dithering
     //Calculate grid position
     float grid_position = frac(dot(tex,(screen_size / dither_size)) + (0.5 / dither_size)); //returns 0.25 and 0.75
   
     //Calculate how big the shift should be
     float dither_shift = (0.25) * (1.0 / (pow(2,dither_bit) - 1.0)); // 0.25 seems good both when using math and when eyeballing it. So does 0.75 btw.

     //Shift the individual colors differently, thus making it even harder to see the dithering pattern
     float3 dither_shift_RGB = float3(dither_shift, -dither_shift, dither_shift); //subpixel dithering

     //modify shift acording to grid position.
     dither_shift_RGB = lerp(2.0 * dither_shift_RGB, -2.0 * dither_shift_RGB, grid_position); //shift acording to grid position.
     
     //shift the color by dither_shift
     //color.rgb += lerp(2.0 * dither_shift_RGB, -2.0 * dither_shift_RGB, grid_position); //shift acording to grid position.
     color.rgb += dither_shift_RGB;
   #endif

   //colorInput.rgb = (dither_shift_RGB * 2.0 * (pow(2,dither_bit) - 1.0) ) + 0.5; //visualize the shift
   //colorInput.rgb = grid_position; //visualize the grid
   //colorInput.rgb = noise; //visualize the noise
   
   colorInput.rgb = color.rgb;
   
   return colorInput;
}
#endif

/*------------------------------------------------------------------------------
							[LIGHTING CODE SECTION]
------------------------------------------------------------------------------*/

#if (PERPIXEL_LIGHTING == 1)
float3 eye;
float4 ambientLight;
DirectionalLight light;
Material material;

//--------------------------------------------------------------------------------------
// Phong Lighting Reflection Model
//--------------------------------------------------------------------------------------
float4 calcPhongLighting( Material M, float4 LColor, float3 N, float3 L, float3 V, float3 R )
{	
	float4 Ia = M.Ka * ambientLight;
	float4 Id = M.Kd * saturate( dot(N,L) );
	float4 Is = M.Ks * pow( saturate(dot(R,V)), M.A );
	
	return Ia + (Id + Is) * LColor;
} 
//--------------------------------------------------------------------------------------
// Blinn-Phong Lighting Reflection Model
//--------------------------------------------------------------------------------------
float4 calcBlinnPhongLighting( Material M, float4 LColor, float3 N, float3 L, float3 H )
{	
	float4 Ia = M.Ka * ambientLight;
	float4 Id = M.Kd * saturate( dot(N,L) );
	float4 Is = M.Ks * pow( saturate(dot(N,H)), M.A );
	
	return Ia + (Id + Is) * LColor;
}

//--------------------------------------------------------------------------------------
// PER VERTEX LIGHTING - PHONG
//--------------------------------------------------------------------------------------
PS_INPUT_PV VS_VERTEX_LIGHTING_PHONG( VS_INPUT input )
{
	PS_INPUT_PV output;
	
	//transform position to clip space
	input.p = mul( input.p, worldMatrix );
	output.p = mul( input.p, viewMatrix );    
	output.p = mul( output.p, projectionMatrix );	
	
	//set texture coords
	output.t = input.t;		
	
	//calculate lighting vectors
	float3 N = normalize( mul( input.n, (float3x3) worldMatrix) );
	float3 V = normalize( eye - (float3) input.p );	
	//DONOT USE -light.dir since the reflection returns a ray from the surface	
	float3 R = reflect( light.dir, N); 
	
	//calculate per vertex lighting intensity and interpolate it like a color
	output.i = calcPhongLighting( material, light.color, N, -light.dir, V, R);
    
	return output;  
}

float4 PS_VERTEX_LIGHTING_PHONG( PS_INPUT_PV input ) : SV_Target
{    
	//with texturing
	//return input.i * Texture.Sample(colorSampler, input.t);
	
	//no texturing pure lighting
	return input.i;      
}

//--------------------------------------------------------------------------------------
// PER VERTEX LIGHTING - BLINN-PHONG
//--------------------------------------------------------------------------------------
PS_INPUT_PV VS_VERTEX_LIGHTING_BLINNPHONG( VS_INPUT input )
{
	PS_INPUT_PV output;
	
	//transform position to clip space
	input.p = mul( input.p, worldMatrix );
    output.p = mul( input.p, viewMatrix );    
    output.p = mul( output.p, projectionMatrix );	
	
	//set texture coords
	output.t = input.t;		
	
	//calculate lighting
	float3 N = normalize( mul( input.n, (float3x3) worldMatrix) );
	float3 V = normalize( eye - (float3) input.p );
	float3 H = normalize( -light.dir + V );
	
	//calculate per vertex lighting intensity and interpolate it like a color
	output.i = calcBlinnPhongLighting( material, light.color, N, -light.dir, H);
    
	return output;  
}

float4 PS_VERTEX_LIGHTING_BLINNPHONG( PS_INPUT_PV input ) : SV_Target
{    
	//with texturing
	//return input.i * Texture.Sample(colorSampler, input.t);
	
	//no texturing pure lighting
	return input.i;      
}
//--------------------------------------------------------------------------------------
// PER PIXEL LIGHTING 
//--------------------------------------------------------------------------------------
PS_INPUT_PP_PHONG VS_PIXEL_LIGHTING_PHONG( VS_INPUT input )
{
	PS_INPUT_PP_PHONG output;
	
	//transform position to clip space - keep worldspace position
	output.wp = mul( input.p, worldMatrix );
	output.p = mul( output.wp, viewMatrix );    
	output.p = mul( output.p, projectionMatrix );	
	
	//set texture coords
	output.t = input.t;			
	
	//set required lighting vectors for interpolation
	output.n = normalize( mul(input.n, (float3x3)worldMatrix) );
   
    return output;  
}

float4 PS_PIXEL_LIGHTING_PHONG( PS_INPUT_PP_PHONG input ) : SV_Target
{     	
	//calculate lighting vectors - renormalize vectors
	input.n = normalize( input.n );		
	float3 V = normalize( eye - (float3) input.wp );
	//DONOT USE -light.dir since the reflection returns a ray from the surface
	float3 R = reflect( light.dir, input.n);
	
	//calculate lighting		
	float4 I = calcPhongLighting( material, light.color, input.n, -light.dir, V, R );
    
	//with texturing
	//return I * Texture.Sample(colorSampler, input.t);
	
	//no texturing pure lighting
	return I;    
}
//--------------------------------------------------------------------------------------
// PER PIXEL LIGHTING 
//--------------------------------------------------------------------------------------
PS_INPUT_PP_BLINNPHONG VS_PIXEL_LIGHTING_BLINNPHONG( VS_INPUT input )
{
	PS_INPUT_PP_BLINNPHONG output;
	
	//set position into clip space
	input.p = mul( input.p, worldMatrix );
	output.p = mul( input.p, viewMatrix );    
	output.p = mul( output.p, projectionMatrix );	
	
	//set texture coords
	output.t = input.t;			
	
	//set required lighting vectors for interpolation
	float3 V = normalize( eye - (float3) input.p );
	output.n = normalize( mul(input.n, (float3x3)worldMatrix) );	
	output.h = normalize( -light.dir + V );		  
    
	return output;  
}

float4 PS_PIXEL_LIGHTING_BLINNPHONG( PS_INPUT_PP_BLINNPHONG input ) : SV_Target
{     	
	//renormalize interpolated vectors
	input.n = normalize( input.n );		
	input.h = normalize( input.h );
	
	//calculate lighting	
	float4 I = calcBlinnPhongLighting( material, light.color, input.n, -light.dir, input.h );
	
	//with texturing
	//return I * Texture.Sample(colorSampler, input.t);
	
	//no texturing pure lighting
	return I;    
}
#endif

/*------------------------------------------------------------------------------
                      [MAIN() & COMBINE PASS CODE SECTION]
------------------------------------------------------------------------------*/

float4 ps_main(VS_OUTPUT In) : SV_Target
{
	float4 baseColor;
	float4 TCBuffer = g_Buffer.Load(In.t);
	
	#if (FXAA_PASS == 1)
		baseColor = FXAAPass(baseColor, In.t);
	#else
		baseColor = Texture.Sample(colorSampler, In.t);
	#endif
	
	#if (LUMA_SHARPEN == 1)
		baseColor = LumaSharpenPass(baseColor, In.t);
	#endif
	
	#if (BLOOM_PASS == 1)
		baseColor = BloomPass(baseColor, In.t);
	#endif
	
	#if (HDR_PASS == 1)
		baseColor = HDRPass(baseColor, In.t);
	#endif
	
	#if (INTELLI_VIBRANCE == 1)
		baseColor = VibrancePass(baseColor, In.t);
	#endif
	
	#if (DYNAMIC_CONTRAST == 1)
		baseColor = CurvesPass(baseColor, In.t);
	#endif
	
	#if (COLOR_CORRECTION == 1)
		baseColor = TonemapPass(baseColor, In.t);
	#endif
	
	#if (DITHERING == 1)
		baseColor = DitherPass(baseColor, In.t);
	#endif
	
	return saturate(baseColor);
}

/*------------------------------------------------------------------------------
						[TECHNIQUES - NOT USED AT PRESENT]
------------------------------------------------------------------------------*/
#if (PERPIXEL_LIGHTING == 1)
technique10 RENDER
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS_PIXEL_LIGHTING_PHONG() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS_PIXEL_LIGHTING_PHONG() ) );
        SetRasterizerState( rsSolid );
    }    
}

technique10 RENDER_VL_PHONG
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS_VERTEX_LIGHTING_PHONG() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS_VERTEX_LIGHTING_PHONG() ) );
        SetRasterizerState( rsSolid );
    }    
}

technique10 RENDER_VL_BLINNPHONG
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS_VERTEX_LIGHTING_BLINNPHONG() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS_VERTEX_LIGHTING_BLINNPHONG() ) );
        SetRasterizerState( rsSolid );
    }    
}

technique10 RENDER_PL_PHONG
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS_PIXEL_LIGHTING_PHONG() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS_PIXEL_LIGHTING_PHONG() ) );
        SetRasterizerState( rsSolid );
    }    
}

technique10 RENDER_PL_BLINNPHONG
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, VS_PIXEL_LIGHTING_BLINNPHONG() ) );
        SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS_PIXEL_LIGHTING_BLINNPHONG() ) );
        SetRasterizerState( rsSolid );
    }    
}
#endif
technique10 MAIN
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, vs_main() ) );
        SetPixelShader( CompileShader( ps_5_0, ps_main() ) );
    }    
}


#endif
