/*===============================================================================*\
|#######################   [HLSL PCSX2 FX SHADER V2.00]   ########################|
|########################	    [BY ASMODEAN@PCSX2]	     #########################|
||																				 ||
||		  This program is free software; you can redistribute it and/or			 ||
||		  modify it under the terms of the GNU General Public License			 ||
||		  as published by the Free Software Foundation; either version 2		 ||
||		  of the License, or (at your option) any later version.				 ||
||																				 ||
||		  This program is distributed in the hope that it will be useful,		 ||
||		  but WITHOUT ANY WARRANTY; without even the implied warranty of		 ||
||		  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			 ||
||		  GNU General Public License for more details.							 ||
||																				 ||
|#################################################################################|
\*===============================================================================*/

#if defined(SHADER_MODEL)  // make safe to include in resource file to enforce dependency

#include "PCSX2Fx_Settings.txt"

/*------------------------------------------------------------------------------
					 [BUFFERS, STRUCTS & GLOBAL VARIABLES]
------------------------------------------------------------------------------*/

#if (SHADER_MODEL == 0x500)
	#define VS_VERSION vs_5_0
	#define PS_VERSION ps_5_0
#else
	#define VS_VERSION vs_4_0
	#define PS_VERSION ps_4_0
#endif

Texture2D<float4> Texture : TEXTURE : register(PS_VERSION, t0);
SamplerState TextureSampler : register(PS_VERSION, s0)
{
    Texture   = <Texture>;
	Filter = MIN_MAG_MIP_LINEAR;
	MaxAnisotropy = 16;
	AddressU = Clamp;
	AddressV = Clamp;
	AddressW = Clamp;
	SRGBTexture = true;
	MaxLOD = 0;
	MinLOD = 0;
	MipLODBias = 0;
};

Texture2D<float4> TexturePhong : TEXTURE : register(PS_VERSION, t1);
SamplerState phongSampler : register(PS_VERSION, s1)
{
    Texture = <TexturePhong>;
	Filter = MIN_MAG_MIP_POINT;
	MaxAnisotropy = 16;
	AddressU = Wrap;
	AddressV = Wrap;
	AddressW = Wrap;
	SRGBTexture = true;
	MaxLOD = 0;
	MinLOD = 0;
	MipLODBias = 0;
};

float LightDistanceSquared = 1000.0f;
float SpecularPower = 100.0f;
float specularIntensity = 100.0f;

float3 CameraPos : POSITION;
float3 LightPosition : DIRECTION;
float4 LightDiffuseColor : LIGHTCOLOR;
float4 LightSpecularColor : LIGHTCOLOR;
float4 AmbientLightColor : LIGHTCOLOR;
float4 DiffuseColor : DIFFUSEMAP;
float4 EmissiveColor : LIGHTCOLOR;
float4 SpecularColor : SPECULARMAP;

#define BUFFER_RCP_WIDTH (1.0f/3584)
#define BUFFER_RCP_HEIGHT (1.0f/3072)
#define BUFFER_WIDTH _rcpFrame.x
#define BUFFER_HEIGHT _rcpFrame.y
#define SCREEN_SIZE float2(BUFFER_WIDTH,BUFFER_HEIGHT)
#define CRT_SCREEN_SIZE float2(1920,1080)
#define px BUFFER_RCP_WIDTH
#define py BUFFER_RCP_HEIGHT
#define PIXEL float2(px,py)

Buffer<float4>g_Buffer : BUFFER;

cbuffer ColorBuffer
{
	float4 outColor : SV_Target0 : register(PS_VERSION, b0);
};

cbuffer FrameBuffer
{
	float4 _rcpFrame : VIEWPORT : register(PS_VERSION, b1);
	float4 _rcpFrameOpt : register(PS_VERSION, b2);
};

cbuffer MatrixBuffer
{
	matrix<float, 4, 4>worldMatrix : WORLDMATRIX;
	matrix<float, 4, 4>viewMatrix : VIEWMATRIX;
	matrix<float, 4, 4>projectionMatrix : PROJECTIONMATRIX;
};

struct PS_INPUT
{
	float4 p : SV_Position;
	float2 t : TEXCOORD0;
	float3 n : TEXCOORD1;
	float3 wp : TEXCOORD2;	
};

struct PS_OUTPUT
{
	float4 c : SV_Target0;	
};


struct VS_INPUT
{
	float4 p : POSITION;
	float4 c : COLOR0;
	float2 t : TEXCOORD0;
	float3 wp : TEXCOORD1;
};

struct VS_OUTPUT
{
	float4 p : SV_Position;
	float4 c : COLOR0;
	float2 t : TEXCOORD0;
	float3 wp : TEXCOORD1;
};

/*------------------------------------------------------------------------------
							[VERTEX CODE SECTION]
------------------------------------------------------------------------------*/

VS_OUTPUT vs_main(VS_INPUT input)
{	
	VS_OUTPUT output;  

	float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
	
	output.p = mul(input.p, worldMatrix);
	output.p = mul(input.p, viewMatrix);
	output.p = mul(input.p, projectionMatrix);
	output.p = input.p;
	output.t = input.t;

	float4 worldPosition =  mul(input.p, worldMatrix);
	output.wp = worldPosition / worldPosition.w;
	output.c = input.c;
	output.c.a = dot(input.c.rgb, lumCoeff);

	return output;
}

/*------------------------------------------------------------------------------
                          [TEXTURE FILTER FUNCTIONS]
------------------------------------------------------------------------------*/

static const float sampleOffsets[5] = { 0.0, 1.4347826, 3.3478260, 5.2608695, 7.1739130 };
static const float sampleWeights[5] = { 0.16818994, 0.27276957, 0.11690125, 0.024067905, 0.0021112196 };

float4 GaussFunction(float4 blurColor : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float4 blur;
	for(int i = 1; i < 5; ++i)
	{
		blur += Texture.Sample(TextureSampler, t + float2(sampleOffsets[i] * SCREEN_SIZE.x, 0.0)) * sampleWeights[i];
		blur += Texture.Sample(TextureSampler, t - float2(sampleOffsets[i] * SCREEN_SIZE.x, 0.0)) * sampleWeights[i];
		blur += Texture.Sample(TextureSampler, t + float2(0.0, sampleOffsets[i] * SCREEN_SIZE.y)) * sampleWeights[i];
		blur += Texture.Sample(TextureSampler, t - float2(0.0, sampleOffsets[i] * SCREEN_SIZE.y)) * sampleWeights[i];
	}
	
	return blur;
}

/*------------------------------------------------------------------------------
                         [COLOR CORRECTION FUNCTIONS]
------------------------------------------------------------------------------*/

float BlendScreen(float base, float blend)
{
	return (1.0f - ((1.0f - base) * (1.0f - blend)));
}

float RGBLuminance(float4 color : COLOR) : SV_Target
{
   float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
   color.a = dot(color.rgb, lumCoeff);
   return color.a;
}

float TrueRGBLuminance(float4 color)
{
   float r = color.x;
   float g = color.y;
   float b = color.z;
   float maxRGB;
   float minRGB;
   if (r >= g) { maxRGB = r; }
   if (r >= b) { maxRGB = r; }
   if (g >= r) { maxRGB = g; }
   if (g >= b) { maxRGB = g; }
   if (b >= r) { maxRGB = b; }
   if (b >= g) { maxRGB = b; }

   if (r <= g) { minRGB = r; }
   if (r <= b) { minRGB = r; }
   if (g <= r) { minRGB = g; }
   if (g <= b) { minRGB = g; }
   if (b <= r) { minRGB = b; }
   if (b <= g) { minRGB = b; }

   float lumin = ((maxRGB + minRGB)/2);
   return lumin;
}

static const float sqrt2_05=sqrt(2.f)*0.5f;
static const float smi=1e-4;

#define floatn float

floatn circular_sCurve(floatn x, float tau)
{
	x-=smi;
	
	float st=sign(sign(-tau)+1);
	floatn p2=step(0.5,x);
	floatn p3=abs(st-p2);
	
	x=fmod(x*2,1);
	x=abs(p3-x);
	
	tau=(1-abs(tau))*sqrt2_05;
	tau=min(tau,sqrt2_05-smi);
	
	float tau2=tau*tau;
	
	float remap0=tau-sqrt(1-2*tau2+tau2);
	float remap1=1-sqrt(1-tau2);
	
	x=-x*remap0+tau;
	
	floatn X=1-sqrt(1-x*x);
	X=(X-remap1)/((1-tau)-remap1);
	
	return (abs(p3-saturate(X))+p2)*0.5;		
}

float4 Desaturate(float3 color, float Desaturation)
{
	float3 grayXfer = float3(0.212656f, 0.715158f, 0.072186f);
	float grayf = dot(grayXfer, color);
	float3 gray = float3(grayf, grayf, grayf);
	return float4(lerp(color, gray, Desaturation), 1.0);
}

float3 RGBToHSL(float3 color)
{
	float3 hsl; // init to 0 to avoid warnings ? (and reverse if + remove first part)
	
	float fmin = min(min(color.r, color.g), color.b);    //Min. value of RGB
	float fmax = max(max(color.r, color.g), color.b);    //Max. value of RGB
	float delta = fmax - fmin;             //Delta RGB value

	hsl.z = (fmax + fmin) / 2.0; // Luminance

	if (delta == 0.0)		//This is a gray, no chroma...
	{
		hsl.x = 0.0;	// Hue
		hsl.y = 0.0;	// Saturation
	}
	else                                    //Chromatic data...
	{
		if (hsl.z < 0.5)
			hsl.y = delta / (fmax + fmin); // Saturation
		else
			hsl.y = delta / (2.0 - fmax - fmin); // Saturation
		
		float deltaR = (((fmax - color.r) / 6.0) + (delta / 2.0)) / delta;
		float deltaG = (((fmax - color.g) / 6.0) + (delta / 2.0)) / delta;
		float deltaB = (((fmax - color.b) / 6.0) + (delta / 2.0)) / delta;

		if (color.r == fmax )
			hsl.x = deltaB - deltaG; // Hue
		else if (color.g == fmax)
			hsl.x = (1.0 / 3.0) + deltaR - deltaB; // Hue
		else if (color.b == fmax)
			hsl.x = (2.0 / 3.0) + deltaG - deltaR; // Hue

		if (hsl.x < 0.0)
			hsl.x += 1.0; // Hue
		else if (hsl.x > 1.0)
			hsl.x -= 1.0; // Hue
	}

	return hsl;
}

float HueToRGB(float f1, float f2, float hue)
{
	if (hue < 0.0)
		hue += 1.0;
	else if (hue > 1.0)
		hue -= 1.0;
	float res;
	if ((6.0 * hue) < 1.0)
		res = f1 + (f2 - f1) * 6.0 * hue;
	else if ((2.0 * hue) < 1.0)
		res = f2;
	else if ((3.0 * hue) < 2.0)
		res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
	else
		res = f1;
	return res;
}

float3 HSLToRGB(float3 hsl)
{
	float3 rgb;
	
	if (hsl.y == 0.0)
		rgb = float3(hsl.z, hsl.z, hsl.z); // Luminance
	else
	{
		float f2;
		
		if (hsl.z < 0.5)
			f2 = hsl.z * (1.0 + hsl.y);
		else
			f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);
			
		float f1 = 2.0 * hsl.z - f2;
		
		rgb.r = HueToRGB(f1, f2, hsl.x + (1.0/3.0));
		rgb.g = HueToRGB(f1, f2, hsl.x);
		rgb.b= HueToRGB(f1, f2, hsl.x - (1.0/3.0));
	}
	
	return rgb;
}

// For all settings: 1.0 = 100% 0.5=50% 1.5 = 150%
float3 ContrastSaturationBrightness(float3 color, float brt, float sat, float con)
{
	// Increase or decrease theese values to adjust r, g and b color channels seperately
	const float AvgLumR = 0.5;
	const float AvgLumG = 0.5;
	const float AvgLumB = 0.5;
	
	const float3 LumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
	
	float3 AvgLumin = float3(AvgLumR, AvgLumG, AvgLumB);
	float3 brtColor = color * brt;
	float intensityf = dot(brtColor, LumCoeff);
	float3 intensity = float3(intensityf, intensityf, intensityf);
	float3 satColor = lerp(intensity, brtColor, sat);
	float3 conColor = lerp(AvgLumin, satColor, con);
	return conColor;
}

#define GammaCorrection(color, gamma) pow(color, gamma)
#define InverseGammaCorrection(color, gamma) pow(color, 1.0f/gamma)

#define BlendLinearDodgef 				BlendAddf
#define BlendLinearBurnf 				BlendSubstractf
#define BlendAddf(base, blend) 			min(base + blend, 1.0)
#define BlendSubstractf(base, blend) 	max(base + blend - 1.0, 0.0)
#define BlendLightenf(base, blend) 		max(blend, base)
#define BlendDarkenf(base, blend) 		min(blend, base)
#define BlendLinearLightf(base, blend) 	(blend < 0.5 ? BlendLinearBurnf(base, (2.0 * blend)) : BlendLinearDodgef(base, (2.0 * (blend - 0.5))))
#define BlendScreenDef(base, blend) 	(1.0 - ((1.0 - base) * (1.0 - blend)))
#define BlendOverlay(base, blend) 		(base < 0.5 ? (2.0 * base * blend) : (1.0 - 2.0 * (1.0 - base) * (1.0 - blend)))
#define BlendSoftLight(base, blend) 	((blend < 0.5) ? (2.0 * base * blend + base * base * (1.0 - 2.0 * blend)) : (sqrt(base) * (2.0 * blend - 1.0) + 2.0 * base * (1.0 - blend)))
#define BlendColorDodgef(base, blend) 	((blend == 1.0) ? blend : min(base / (1.0 - blend), 1.0))
#define BlendColorBurnf(base, blend) 	((blend == 0.0) ? blend : max((1.0 - ((1.0 - base) / blend)), 0.0))
#define BlendVividLightf(base, blend) 	((blend < 0.5) ? BlendColorBurnf(base, (2.0 * blend)) : BlendColorDodgef(base, (2.0 * (blend - 0.5))))
#define BlendPinLightf(base, blend) 	((blend < 0.5) ? BlendDarkenf(base, (2.0 * blend)) : BlendLightenf(base, (2.0 *(blend - 0.5))))
#define BlendHardMixf(base, blend) 		((BlendVividLightf(base, blend) < 0.5) ? 0.0 : 1.0)
#define BlendReflectf(base, blend) 		((blend == 1.0) ? blend : min(base * base / (1.0 - blend), 1.0))

// Hue Blend mode creates the result color by combining the luminance and saturation of the base color with the hue of the blend color.
float3 BlendHue(float3 base, float3 blend)
{
	float3 baseHSL = RGBToHSL(base);
	return HSLToRGB(float3(RGBToHSL(blend).r, baseHSL.g, baseHSL.b));
}

// Saturation Blend mode creates the result color by combining the luminance and hue of the base color with the saturation of the blend color.
float3 BlendSaturation(float3 base, float3 blend)
{
	float3 baseHSL = RGBToHSL(base);
	return HSLToRGB(float3(baseHSL.r, RGBToHSL(blend).g, baseHSL.b));
}

// Color Mode keeps the brightness of the base color and applies both the hue and saturation of the blend color.
float3 BlendColor(float3 base, float3 blend)
{
	float3 blendHSL = RGBToHSL(blend);
	return HSLToRGB(float3(blendHSL.r, blendHSL.g, RGBToHSL(base).b));
}

// Luminosity Blend mode creates the result color by combining the hue and saturation of the base color with the luminance of the blend color.
float3 BlendLuminosity(float3 base, float3 blend)
{
	float3 baseHSL = RGBToHSL(base);
	return HSLToRGB(float3(baseHSL.r, baseHSL.g, RGBToHSL(blend).b));
}

/*------------------------------------------------------------------------------
                             [FXAA CODE SECTION]
------------------------------------------------------------------------------*/
#if (FXAA_PASS == 1)
#define FXAA_DISCARD 0
#define FXAA_GREEN_AS_LUMA 0

#if (SHADER_MODEL == 0x500)
	#define FXAA_HLSL_5 1
	#define FXAA_GATHER4_ALPHA 1
	RWBuffer<float> g_vertexBuffer : register(u0);
#else
	#define FXAA_HLSL_4 1
	#define FXAA_GATHER4_ALPHA 0
#endif

#if (FXAA_Quality == 4)
	#define FXAA_EDGE_THRESHOLD (0.063)
	#define FXAA_EDGE_THRESHOLD_MIN (0.0312)
	#define FXAA_QUALITY__PS 15
    #define FXAA_QUALITY__P0 1.0
    #define FXAA_QUALITY__P1 1.0
    #define FXAA_QUALITY__P2 1.0
    #define FXAA_QUALITY__P3 1.0
    #define FXAA_QUALITY__P4 1.0
    #define FXAA_QUALITY__P5 1.0
    #define FXAA_QUALITY__P6 1.0
    #define FXAA_QUALITY__P7 1.0
    #define FXAA_QUALITY__P8 1.0
    #define FXAA_QUALITY__P9 1.0
    #define FXAA_QUALITY__P10 1.0
    #define FXAA_QUALITY__P11 1.0
	#define FXAA_QUALITY__P12 1.0
#endif
#if (FXAA_Quality == 3)
	#define FXAA_EDGE_THRESHOLD (0.125)
	#define FXAA_EDGE_THRESHOLD_MIN (0.0312)
    #define FXAA_QUALITY__PS 12
    #define FXAA_QUALITY__P0 1.0
    #define FXAA_QUALITY__P1 1.0
    #define FXAA_QUALITY__P2 1.0
    #define FXAA_QUALITY__P3 1.0
    #define FXAA_QUALITY__P4 1.0
    #define FXAA_QUALITY__P5 1.5
    #define FXAA_QUALITY__P6 2.0
    #define FXAA_QUALITY__P7 2.0
    #define FXAA_QUALITY__P8 2.0
    #define FXAA_QUALITY__P9 2.0
    #define FXAA_QUALITY__P10 4.0
    #define FXAA_QUALITY__P11 8.0
#endif
#if (FXAA_Quality == 2)
	#define FXAA_EDGE_THRESHOLD (0.166)
	#define FXAA_EDGE_THRESHOLD_MIN (0.0625)
	#define FXAA_QUALITY__PS 12
    #define FXAA_QUALITY__P0 1.0
    #define FXAA_QUALITY__P1 1.5
    #define FXAA_QUALITY__P2 2.0
    #define FXAA_QUALITY__P3 2.0
    #define FXAA_QUALITY__P4 2.0
    #define FXAA_QUALITY__P5 2.0
    #define FXAA_QUALITY__P6 2.0
    #define FXAA_QUALITY__P7 2.0
    #define FXAA_QUALITY__P8 2.0
    #define FXAA_QUALITY__P9 2.0
    #define FXAA_QUALITY__P10 4.0
    #define FXAA_QUALITY__P11 8.0
#endif
#if (FXAA_Quality == 1)
	#define FXAA_EDGE_THRESHOLD (0.250)
	#define FXAA_EDGE_THRESHOLD_MIN (0.0833)
	#define FXAA_QUALITY__PS 8
    #define FXAA_QUALITY__P0 1.0
    #define FXAA_QUALITY__P1 1.5
    #define FXAA_QUALITY__P2 2.0
    #define FXAA_QUALITY__P3 2.0
    #define FXAA_QUALITY__P4 2.0
    #define FXAA_QUALITY__P5 2.0
    #define FXAA_QUALITY__P6 4.0
    #define FXAA_QUALITY__P7 12.0
#endif

#if (FXAA_HLSL_5 == 1)
    #define FxaaInt2 int2
    struct FxaaTex { SamplerState smpl; Texture2D tex; };
    #define FxaaTexTop(t, p) t.tex.SampleLevel(t.smpl, p, 0.0)
    #define FxaaTexOff(t, p, o, r) t.tex.SampleLevel(t.smpl, p, 0.0, o)
    #define FxaaTexAlpha4(t, p) t.tex.GatherAlpha(t.smpl, p)
    #define FxaaTexOffAlpha4(t, p, o) t.tex.GatherAlpha(t.smpl, p, o)
    #define FxaaTexGreen4(t, p) t.tex.GatherGreen(t.smpl, p)
    #define FxaaTexOffGreen4(t, p, o) t.tex.GatherGreen(t.smpl, p, o)
	#define FxaaBool bool
	#define FxaaDiscard clip(-1)
	#define FxaaFloat float
	#define FxaaFloat2 float2
	#define FxaaFloat3 float3
	#define FxaaFloat4 float4
	#define FxaaHalf half
	#define FxaaHalf2 half2
	#define FxaaHalf3 half3
	#define FxaaHalf4 half4
	#define FxaaSat(x) saturate(x)
#endif

#if (FXAA_HLSL_4 == 1)
    #define FxaaInt2 int2
    struct FxaaTex { SamplerState smpl; Texture2D tex; };
    #define FxaaTexTop(t, p) t.tex.SampleLevel(t.smpl, p, 0.0)
    #define FxaaTexOff(t, p, o, r) t.tex.SampleLevel(t.smpl, p, 0.0, o)
	#define FxaaBool bool
	#define FxaaDiscard clip(-1)
	#define FxaaFloat float
	#define FxaaFloat2 float2
	#define FxaaFloat3 float3
	#define FxaaFloat4 float4
	#define FxaaHalf half
	#define FxaaHalf2 half2
	#define FxaaHalf3 half3
	#define FxaaHalf4 half4
	#define FxaaSat(x) saturate(x)
#endif

#if (FXAA_GREEN_AS_LUMA == 0)
    FxaaFloat FxaaLuma(FxaaFloat4 rgba) { return rgba.w; }
#else
    FxaaFloat FxaaLuma(FxaaFloat4 rgba) { return rgba.y; }
#endif 

FxaaFloat4 FxaaPixelShader(FxaaFloat2 pos, FxaaTex tex, FxaaFloat2 fxaaQualityRcpFrame, FxaaFloat fxaaQualitySubpix, FxaaFloat fxaaQualityEdgeThreshold, FxaaFloat fxaaQualityEdgeThresholdMin)
{
    FxaaFloat2 posM;
    posM.x = pos.x;
    posM.y = pos.y;

    #if (FXAA_GATHER4_ALPHA == 1)
        #if (FXAA_DISCARD == 0)
            FxaaFloat4 rgbyM = FxaaTexTop(tex, posM);
            #if (FXAA_GREEN_AS_LUMA == 0)
                #define lumaM rgbyM.w
            #else
                #define lumaM rgbyM.y
            #endif
        #endif
        #if (FXAA_GREEN_AS_LUMA == 0)
            FxaaFloat4 luma4A = FxaaTexAlpha4(tex, posM);
            FxaaFloat4 luma4B = FxaaTexOffAlpha4(tex, posM, FxaaInt2(-1, -1));
        #else
            FxaaFloat4 luma4A = FxaaTexGreen4(tex, posM);
            FxaaFloat4 luma4B = FxaaTexOffGreen4(tex, posM, FxaaInt2(-1, -1));
        #endif
        #if (FXAA_DISCARD == 1)
            #define lumaM luma4A.w
        #endif
        #define lumaE luma4A.z
        #define lumaS luma4A.x
        #define lumaSE luma4A.y
        #define lumaNW luma4B.w
        #define lumaN luma4B.z
        #define lumaW luma4B.x
    #else
        FxaaFloat4 rgbyM = FxaaTexTop(tex, posM);
        #if (FXAA_GREEN_AS_LUMA == 0)
            #define lumaM rgbyM.w
        #else
            #define lumaM rgbyM.y
        #endif
        FxaaFloat lumaS = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2( 0, 1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaE = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2( 1, 0), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaN = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2( 0,-1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaW = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2(-1, 0), fxaaQualityRcpFrame.xy));
    #endif
/*--------------------------------------------------------------------------*/
    FxaaFloat maxSM = max(lumaS, lumaM);
    FxaaFloat minSM = min(lumaS, lumaM);
    FxaaFloat maxESM = max(lumaE, maxSM);
    FxaaFloat minESM = min(lumaE, minSM);
    FxaaFloat maxWN = max(lumaN, lumaW);
    FxaaFloat minWN = min(lumaN, lumaW);
    FxaaFloat rangeMax = max(maxWN, maxESM);
    FxaaFloat rangeMin = min(minWN, minESM);
    FxaaFloat rangeMaxScaled = rangeMax * fxaaQualityEdgeThreshold;
    FxaaFloat range = rangeMax - rangeMin;
    FxaaFloat rangeMaxClamped = max(fxaaQualityEdgeThresholdMin, rangeMaxScaled);
    FxaaBool earlyExit = range < rangeMaxClamped;
/*--------------------------------------------------------------------------*/
    if(earlyExit)
        #if (FXAA_DISCARD == 1)
            FxaaDiscard;
        #else
            return rgbyM;
        #endif
/*--------------------------------------------------------------------------*/
    #if (FXAA_GATHER4_ALPHA == 0)
        FxaaFloat lumaNW = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2(-1,-1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaSE = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2( 1, 1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaNE = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2( 1,-1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaSW = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2(-1, 1), fxaaQualityRcpFrame.xy));
    #else
        FxaaFloat lumaNE = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2(1, -1), fxaaQualityRcpFrame.xy));
        FxaaFloat lumaSW = FxaaLuma(FxaaTexOff(tex, posM, FxaaInt2(-1, 1), fxaaQualityRcpFrame.xy));
    #endif
/*--------------------------------------------------------------------------*/
    FxaaFloat lumaNS = lumaN + lumaS;
    FxaaFloat lumaWE = lumaW + lumaE;
    FxaaFloat subpixRcpRange = 1.0/range;
    FxaaFloat subpixNSWE = lumaNS + lumaWE;
    FxaaFloat edgeHorz1 = (-2.0 * lumaM) + lumaNS;
    FxaaFloat edgeVert1 = (-2.0 * lumaM) + lumaWE;
/*--------------------------------------------------------------------------*/
    FxaaFloat lumaNESE = lumaNE + lumaSE;
    FxaaFloat lumaNWNE = lumaNW + lumaNE;
    FxaaFloat edgeHorz2 = (-2.0 * lumaE) + lumaNESE;
    FxaaFloat edgeVert2 = (-2.0 * lumaN) + lumaNWNE;
/*--------------------------------------------------------------------------*/
    FxaaFloat lumaNWSW = lumaNW + lumaSW;
    FxaaFloat lumaSWSE = lumaSW + lumaSE;
    FxaaFloat edgeHorz4 = (abs(edgeHorz1) * 2.0) + abs(edgeHorz2);
    FxaaFloat edgeVert4 = (abs(edgeVert1) * 2.0) + abs(edgeVert2);
    FxaaFloat edgeHorz3 = (-2.0 * lumaW) + lumaNWSW;
    FxaaFloat edgeVert3 = (-2.0 * lumaS) + lumaSWSE;
    FxaaFloat edgeHorz = abs(edgeHorz3) + edgeHorz4;
    FxaaFloat edgeVert = abs(edgeVert3) + edgeVert4;
/*--------------------------------------------------------------------------*/
    FxaaFloat subpixNWSWNESE = lumaNWSW + lumaNESE;
    FxaaFloat lengthSign = fxaaQualityRcpFrame.x;
    FxaaBool horzSpan = edgeHorz >= edgeVert;
    FxaaFloat subpixA = subpixNSWE * 2.0 + subpixNWSWNESE;
/*--------------------------------------------------------------------------*/
    if(!horzSpan) lumaN = lumaW;
    if(!horzSpan) lumaS = lumaE;
    if(horzSpan) lengthSign = fxaaQualityRcpFrame.y;
    FxaaFloat subpixB = (subpixA * (1.0/12.0)) - lumaM;
/*--------------------------------------------------------------------------*/
    FxaaFloat gradientN = lumaN - lumaM;
    FxaaFloat gradientS = lumaS - lumaM;
    FxaaFloat lumaNN = lumaN + lumaM;
    FxaaFloat lumaSS = lumaS + lumaM;
    FxaaBool pairN = abs(gradientN) >= abs(gradientS);
    FxaaFloat gradient = max(abs(gradientN), abs(gradientS));
    if(pairN) lengthSign = -lengthSign;
    FxaaFloat subpixC = FxaaSat(abs(subpixB) * subpixRcpRange);
/*--------------------------------------------------------------------------*/
    FxaaFloat2 posB;
    posB.x = posM.x;
    posB.y = posM.y;
    FxaaFloat2 offNP;
    offNP.x = (!horzSpan) ? 0.0 : fxaaQualityRcpFrame.x;
    offNP.y = ( horzSpan) ? 0.0 : fxaaQualityRcpFrame.y;
    if(!horzSpan) posB.x += lengthSign * 0.5;
    if( horzSpan) posB.y += lengthSign * 0.5;
/*--------------------------------------------------------------------------*/
    FxaaFloat2 posN;
    posN.x = posB.x - offNP.x * FXAA_QUALITY__P0;
    posN.y = posB.y - offNP.y * FXAA_QUALITY__P0;
    FxaaFloat2 posP;
    posP.x = posB.x + offNP.x * FXAA_QUALITY__P0;
    posP.y = posB.y + offNP.y * FXAA_QUALITY__P0;
    FxaaFloat subpixD = ((-2.0)*subpixC) + 3.0;
    FxaaFloat lumaEndN = FxaaLuma(FxaaTexTop(tex, posN));
    FxaaFloat subpixE = subpixC * subpixC;
    FxaaFloat lumaEndP = FxaaLuma(FxaaTexTop(tex, posP));
/*--------------------------------------------------------------------------*/
    if(!pairN) lumaNN = lumaSS;
    FxaaFloat gradientScaled = gradient * 1.0/4.0;
    FxaaFloat lumaMM = lumaM - lumaNN * 0.5;
    FxaaFloat subpixF = subpixD * subpixE;
    FxaaBool lumaMLTZero = lumaMM < 0.0;
/*--------------------------------------------------------------------------*/
    lumaEndN -= lumaNN * 0.5;
    lumaEndP -= lumaNN * 0.5;
    FxaaBool doneN = abs(lumaEndN) >= gradientScaled;
    FxaaBool doneP = abs(lumaEndP) >= gradientScaled;
    if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P1;
    if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P1;
    FxaaBool doneNP = (!doneN) || (!doneP);
    if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P1;
    if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P1;
/*--------------------------------------------------------------------------*/
    if(doneNP) {
        if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
        if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
        if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
        if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
        doneN = abs(lumaEndN) >= gradientScaled;
        doneP = abs(lumaEndP) >= gradientScaled;
        if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P2;
        if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P2;
        doneNP = (!doneN) || (!doneP);
        if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P2;
        if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P2;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 3)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P3;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P3;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P3;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P3;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 4)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P4;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P4;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P4;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P4;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 5)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P5;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P5;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P5;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P5;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 6)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P6;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P6;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P6;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P6;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 7)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P7;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P7;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P7;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P7;
/*--------------------------------------------------------------------------*/
    #if (FXAA_QUALITY__PS > 8)
    if(doneNP) {
        if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
        if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
        if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
        if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
        doneN = abs(lumaEndN) >= gradientScaled;
        doneP = abs(lumaEndP) >= gradientScaled;
        if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P8;
        if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P8;
        doneNP = (!doneN) || (!doneP);
        if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P8;
        if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P8;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 9)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P9;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P9;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P9;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P9;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 10)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P10;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P10;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P10;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P10;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 11)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P11;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P11;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P11;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P11;
/*--------------------------------------------------------------------------*/
	#if (FXAA_QUALITY__PS > 12)
	if(doneNP) {
		if(!doneN) lumaEndN = FxaaLuma(FxaaTexTop(tex, posN.xy));
		if(!doneP) lumaEndP = FxaaLuma(FxaaTexTop(tex, posP.xy));
		if(!doneN) lumaEndN = lumaEndN - lumaNN * 0.5;
		if(!doneP) lumaEndP = lumaEndP - lumaNN * 0.5;
		doneN = abs(lumaEndN) >= gradientScaled;
		doneP = abs(lumaEndP) >= gradientScaled;
		if(!doneN) posN.x -= offNP.x * FXAA_QUALITY__P12;
		if(!doneN) posN.y -= offNP.y * FXAA_QUALITY__P12;
		doneNP = (!doneN) || (!doneP);
		if(!doneP) posP.x += offNP.x * FXAA_QUALITY__P12;
		if(!doneP) posP.y += offNP.y * FXAA_QUALITY__P12;
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
#endif
/*--------------------------------------------------------------------------*/
}
/*--------------------------------------------------------------------------*/
    FxaaFloat dstN = posM.x - posN.x;
    FxaaFloat dstP = posP.x - posM.x;
    if(!horzSpan) dstN = posM.y - posN.y;
    if(!horzSpan) dstP = posP.y - posM.y;
/*--------------------------------------------------------------------------*/
    FxaaBool goodSpanN = (lumaEndN < 0.0) != lumaMLTZero;
    FxaaFloat spanLength = (dstP + dstN);
    FxaaBool goodSpanP = (lumaEndP < 0.0) != lumaMLTZero;
    FxaaFloat spanLengthRcp = 1.0/spanLength;
/*--------------------------------------------------------------------------*/
    FxaaBool directionN = dstN < dstP;
    FxaaFloat dst = min(dstN, dstP);
    FxaaBool goodSpan = directionN ? goodSpanN : goodSpanP;
    FxaaFloat subpixG = subpixF * subpixF;
    FxaaFloat pixelOffset = (dst * (-spanLengthRcp)) + 0.5;
    FxaaFloat subpixH = subpixG * fxaaQualitySubpix;
/*--------------------------------------------------------------------------*/
    FxaaFloat pixelOffsetGood = goodSpan ? pixelOffset : 0.0;
    FxaaFloat pixelOffsetSubpix = max(pixelOffsetGood, subpixH);
    if(!horzSpan) posM.x += pixelOffsetSubpix * lengthSign;
    if( horzSpan) posM.y += pixelOffsetSubpix * lengthSign;
    #if (FXAA_DISCARD == 1)
        return FxaaTexTop(tex, posM);
    #else
        return FxaaFloat4(FxaaTexTop(tex, posM).xyz, lumaM);
    #endif
}
/*==========================================================================*/

float4 FXAAPass(float4 FxaaColor : COLOR, float2 t : TEXCOORD) : SV_Target
{
	FxaaTex tex;
	FxaaFloat2 pos;
	FxaaFloat2 FxaaFrame;

	tex.tex = Texture;
	tex.smpl = TextureSampler;
	pos = t;
	
	Texture.GetDimensions(FxaaFrame.x, FxaaFrame.y);
	
	FxaaFloat fxaaSubpix = FXAA_Subpix;
    FxaaFloat fxaaEdgeThreshold = FXAA_EDGE_THRESHOLD;
    FxaaFloat fxaaEdgeThresholdMin = FXAA_EDGE_THRESHOLD_MIN;

	FxaaColor = FxaaPixelShader(pos, tex, 1.0/FxaaFrame, fxaaSubpix, fxaaEdgeThreshold, fxaaEdgeThresholdMin);

	return saturate(FxaaColor);
}
#endif

/*------------------------------------------------------------------------------
                          [LUMA SHARPEN CODE SECTION]
						[Adapted from CeeJay's SweetFX]
	[http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (LUMA_SHARPEN == 1)
#define CoefLuma float3(0.212656f, 0.715158f, 0.072186f)
#define LumaSharpeningStrength (CoefLuma * SharpenStrength)
	
float4 LumaSharpenPass(float4 inputcolor : COLOR, float2 t : TEXCOORD) : SV_Target 
{

	// -- Get the original PIXEL --
	float3 ori = Texture.SampleLevel(TextureSampler, t, 0).rgb;

	// -- Pattern 1 -- A (fast) 7 tap gaussian using only 2+1 texture fetches.
	#if (Pattern == 1)
	float3 blur_ori = Texture.SampleLevel(TextureSampler, t + (float2(px,py) / 3) * OffsetBias, 0).rgb;  // North West
	blur_ori += Texture.SampleLevel(TextureSampler, t + (float2(-px,-py) / 3) * OffsetBias, 0).rgb; // South East

	blur_ori /= 2;  //Divide by the number of texture fetches

	LumaSharpeningStrength *= 1.5; // Adjust strength to aproximate the strength of Pattern 2
	#endif

	// -- Pattern 2 -- A 9 tap gaussian using 4+1 texture fetches.
	#if (Pattern == 2)
	float3 blur_ori = Texture.SampleLevel(TextureSampler, t + float2(px,-py) * 0.5 * OffsetBias, 0).rgb; // South East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(-px,-py) * 0.5 * OffsetBias, 0).rgb;  // South West    
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(px,py) * 0.5 * OffsetBias, 0).rgb; // North East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(-px,py) * 0.5 * OffsetBias, 0).rgb; // North West

	blur_ori *= 0.25;  // ( /= 4) Divide by the number of texture fetches
	#endif 

	// -- Pattern 3 -- An experimental 17 tap gaussian using 4+1 texture fetches.
	#if (Pattern == 3)
	float3 blur_ori = Texture.SampleLevel(TextureSampler, t + float2(0.4*px,-1.2*py)* OffsetBias, 0).rgb;  // South South East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(-1.2*px,-0.4*py) * OffsetBias, 0).rgb; // West South West
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(1.2*px,0.4*py) * OffsetBias, 0).rgb; // East North East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(-0.4*px,1.2*py) * OffsetBias, 0).rgb; // North North West

	blur_ori *= 0.25;  // ( /= 4) Divide by the number of texture fetches

	LumaSharpeningStrength *= 0.51;
	#endif

	// -- Pattern 4 -- A 9 tap high pass (pyramid filter) using 4+1 texture fetches.
	#if (Pattern == 4)
	float3 blur_ori = Texture.SampleLevel(TextureSampler, t + float2(0.5 * px,-py * OffsetBias), 0).rgb;  // South South East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(OffsetBias * -px,0.5 * -py), 0).rgb; // West South West
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(OffsetBias * px,0.5 * py), 0).rgb; // East North East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(0.5 * -px,py * OffsetBias), 0).rgb; // North North West

	//blur_ori += (2 * ori); // Probably not needed. Only serves to lessen the effect.

	blur_ori /= 4.0;  //Divide by the number of texture fetches

	LumaSharpeningStrength *= 0.666; // Adjust strength to aproximate the strength of Pattern 2
	#endif

	// -- Pattern 8 -- A (slower) 9 tap gaussian using 9 texture fetches.
	#if (Pattern == 8)
	half3 blur_ori = Texture.SampleLevel(TextureSampler, t + float2(-px,py) * OffsetBias, 0).rgb; // North West
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(px,-py) * OffsetBias, 0).rgb;     // South East
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(-px,-py)  * OffsetBias, 0).rgb;  // South West
	blur_ori += Texture.SampleLevel(TextureSampler, t + float2(px,py) * OffsetBias, 0).rgb;    // North East

	half3 blur_ori2 = Texture.SampleLevel(TextureSampler, t + float2(0,py) * OffsetBias, 0).rgb; // North
	blur_ori2 += Texture.SampleLevel(TextureSampler, t + float2(0,-py) * OffsetBias, 0).rgb;    // South
	blur_ori2 += Texture.SampleLevel(TextureSampler, t + float2(-px,0) * OffsetBias, 0).rgb;   // West
	blur_ori2 += Texture.SampleLevel(TextureSampler, t + float2(px,0) * OffsetBias, 0).rgb;   // East
	blur_ori2 *= 2.0;

	blur_ori += blur_ori2;
	blur_ori += (ori * 4);

	blur_ori /= 16.0;  //Divide by the number of texture fetches
	#endif

	// -- Pattern 9 -- A (slower) 9 tap high pass using 9 texture fetches.
	#if (Pattern == 9)
	float3 blur_ori = Texture.Sample(TextureSampler, (t + float2(-px,py) * OffsetBias)).rgb; // North West
	blur_ori += Texture.Sample(TextureSampler, (t + float2(px,-py) * OffsetBias)).rgb;     // South East
	blur_ori += Texture.Sample(TextureSampler, (t + float2(-px,-py)  * OffsetBias)).rgb;  // South West
	blur_ori += Texture.Sample(TextureSampler, (t + float2(px,py) * OffsetBias)).rgb;    // North East

	blur_ori += ori.rgb; // Probably not needed. Only serves to lessen the effect.

	blur_ori += Texture.Sample(TextureSampler, (t + float2(0,py) * OffsetBias)).rgb;    // North
	blur_ori += Texture.Sample(TextureSampler, (t + float2(0,-py) * OffsetBias)).rgb;  // South
	blur_ori += Texture.Sample(TextureSampler, (t + float2(-px,0) * OffsetBias)).rgb; // West
	blur_ori += Texture.Sample(TextureSampler, (t + float2(px,0) * OffsetBias)).rgb; // East

	blur_ori /= 9;  //Divide by the number of texture fetches
	#endif

	// -- Calculate the sharpening --  
	float3 sharp = ori - blur_ori;  //Subtracting the blurred image from the original image

	#if 0
	// -- Adjust strength of the sharpening --
	float sharp_luma = dot(sharp, LumaSharpeningStrength); //Calculate the luma and adjust the strength
	sharp_luma = (abs(sharp_luma)*8.0) * exp(1.0-(abs(sharp_luma)*8.0)) * sign(sharp_luma) / 16.0;

	#elif 0
	// -- Clamping the maximum amount of sharpening to prevent halo artifacts --
	float sharp_luma = dot(sharp, LumaSharpeningStrength); //Calculate the luma and adjust the strength
	sharp_luma = clamp(sharp_luma, -SharpenClamp, SharpenClamp);  //TODO Try a curve function instead of a clamp

	#else
    // -- Adjust strength of the sharpening and clamp it--
    float4 sharp_strength_luma_clamp = float4(LumaSharpeningStrength * (0.5 / SharpenClamp),0.5); //Roll part of the clamp into the dot
    float sharp_luma = saturate(dot(float4(sharp,1.0), sharp_strength_luma_clamp)); //Calculate the luma, adjust the strength, scale up and clamp
    sharp_luma = (SharpenClamp * 2.0) * sharp_luma - SharpenClamp; //scale down
	#endif
	
	inputcolor.rgb = inputcolor.rgb + sharp_luma;    // Add the sharpening to the input color.
	inputcolor.a = RGBLuminance(inputcolor);

	#if (ShowSharpen == 1)
	inputcolor.rgb = saturate(0.5f + (sharp_luma * 4)).rrrr;
	#endif

	return saturate(inputcolor);
}
#endif

/*------------------------------------------------------------------------------
                          [BLOOM PASS CODE SECTION]
					 [Adapted from KrossX's Screen Bloom. TY]
------------------------------------------------------------------------------*/

#if (BLOOM_PASS == 1)
static const float blurspread = BlurSpread;

float4 BloomPass(float4 color : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float bloomPower = BloomPower;
	float4 blur = color * 0.22520613262190495;
	color.a = RGBLuminance(color);

	const float2 dx = float2(_rcpFrame.x * blurspread, 0.0);
	const float2 dy = float2(0.0, _rcpFrame.y * 2.0f * blurspread);
	const float2 dx2 = 2 * dx;
	const float2 dy2 = 2 * dy;
	
	float4 colorU = Texture.Sample(TextureSampler, t + dy);
	
	blur += 0.002589001911021066 * Texture.Sample(TextureSampler, t -dx2 +dy2); 
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t -dx +dy2); 
	blur += 0.024146616900339800 * Texture.Sample(TextureSampler, t +dy2); 
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t +dx +dy2); 
	blur += 0.002589001911021066 * Texture.Sample(TextureSampler, t +dx2 +dy2); 

	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t -dx2 +dy);
	blur += 0.044875475183061630 * Texture.Sample(TextureSampler, t -dx +dy);
	blur += 0.100529757860782610 * Texture.Sample(TextureSampler, t +dy);
	blur += 0.044875475183061630 * Texture.Sample(TextureSampler, t +dx +dy);
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t +dx2 +dy);

	blur += 0.024146616900339800 * Texture.Sample(TextureSampler, t -dx2);
	blur += 0.100529757860782610 * Texture.Sample(TextureSampler, t -dx);	
	blur += 0.100529757860782610 * Texture.Sample(TextureSampler, t +dx);
	blur += 0.024146616900339800 * Texture.Sample(TextureSampler, t +dx2);

	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t -dx2 -dy);
	blur += 0.044875475183061630 * Texture.Sample(TextureSampler, t -dx -dy);
	blur += 0.100529757860782610 * Texture.Sample(TextureSampler, t -dy);
	blur += 0.044875475183061630 * Texture.Sample(TextureSampler, t +dx -dy);
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t +dx2 -dy);

	blur += 0.002589001911021066 * Texture.Sample(TextureSampler, t -dx2 -dy2);
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t -dx -dy2);
	blur += 0.024146616900339800 * Texture.Sample(TextureSampler, t -dy2);
	blur += 0.010778807494659370 * Texture.Sample(TextureSampler, t +dx -dy2);
	blur += 0.002589001911021066 * Texture.Sample(TextureSampler, t +dx2 -dy2);				
	
	float4 newcolor = float4(BloomType(color.r, blur.r),
							 BloomType(color.g, blur.g),
							 BloomType(color.b, blur.b),
							 RGBLuminance(color));
	
	color = lerp(color, newcolor, BloomPower);
	
	return saturate(color);
}
#endif

/*------------------------------------------------------------------------------
					 [GAMMA CORRECTION PASS CODE SECTION]
------------------------------------------------------------------------------*/

#if (GAMMA_CORRECTION == 1)
float4 GammaPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{

	float gammaVal = Gamma;
	float3 correctedGamma;
	float3 InverseCorrectedGamma;
	
	correctedGamma = GammaCorrection(colorInput.rgb, (gammaVal/2.2f));
	InverseCorrectedGamma = InverseGammaCorrection(colorInput.rgb, (gammaVal/2.2f));
	
	colorInput.rgb = ((correctedGamma+InverseCorrectedGamma)/2);
	colorInput.a = RGBLuminance(colorInput);

	return saturate(colorInput);
}
#endif


/*------------------------------------------------------------------------------
						[COLOR CORRECTION CODE SECTION]
------------------------------------------------------------------------------*/

#if (COLOR_CORRECTION == 1)
float4 ColorPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	
	// Average scene luminance is stored in R channel
	float avgLuminance = Luma;
	float exposure = Exposure;
	float whitePoint = WhitePoint;
	float blackLevel = BlackLevel;

	//TEXCOORD mapping swap
	float2 colorTexel = float2(0,0);
	
	#if(TonemappingType == 1)
	// Local tone mapping
		float4 ColorTone = Texture.Sample(phongSampler, t);
	#else
	// Scene tone mapping
		float4 ColorTone = TexturePhong.Sample(phongSampler, t);
	#endif

	float3 lumCoeff = float3(0.212656f, 0.715158f, 0.072186f);
	
	// Detect dark areas, depending on mapping type
	if(ColorTone.r < 0.225f && ColorTone.g < 0.225f && ColorTone.b < 0.225f)
	{
		colorTexel = t;
		ColorTone = Texture.Sample(phongSampler, colorTexel);
		colorInput.rgb -= blackLevel;
	}
	
	// RGB -> XYZ conversion
	const float3x3 RGB2XYZ = {0.4124564f,  0.3575761f,  0.1804375f,
							  0.2126729f,  0.7151522f,  0.0721750f,
							  0.0193339f,  0.1191920f,  0.9503041f};
							  
	float3 XYZ = mul(RGB2XYZ, colorInput.rgb);
  
	// XYZ -> Yxy conversion
	float3 Yxy = lumCoeff;
	Yxy.r = XYZ.g;                            // copy luminance Y
	Yxy.g = XYZ.r / (XYZ.r + XYZ.g + XYZ.b ); // x = X / (X + Y + Z)
	Yxy.b = XYZ.g / (XYZ.r + XYZ.g + XYZ.b ); // y = Y / (X + Y + Z)
    
	// (Lp) Map average luminance to the middlegrey zone by scaling pixel luminance
	float Lp = Yxy.r * exposure / avgLuminance;                       
	// (Ld) Scale all luminance within a displayable range of 0 to 1
	Yxy.r = (Lp * (1.0f + Lp/(whitePoint * whitePoint)))/(1.0f + Lp);
  
	// Yxy -> XYZ conversion
	XYZ.r = Yxy.r * Yxy.g / Yxy. b;               // X = Y * x / y
	XYZ.g = Yxy.r;                                // copy luminance Y
	XYZ.b = Yxy.r * (1 - Yxy.g - Yxy.b) / Yxy.b;  // Z = Y * (1-x-y) / y
    
	// XYZ -> RGB conversion
	const float3x3 XYZ2RGB  = {3.2404542f, -1.5371385f, -0.4985314f,
							  -0.9692660f,  1.8760108f,  0.0415560f, 
							   0.0556434f, -0.2040259f,  1.0572252f};
							   
	colorInput.rgb = mul(XYZ2RGB, XYZ);
	colorInput.a = RGBLuminance(colorInput);
	
	return saturate(colorInput);
}
#endif

/*------------------------------------------------------------------------------
						[DYNAMIC VIBRANCE CODE SECTION]
						[Adapted from CeeJay's SweetFX]
	[http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (INTELLI_VIBRANCE == 1)
float4 VibrancePass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float4 color = colorInput;
	float vibrance = Vibrance;
	float luma = RGBLuminance(color);
	
	float max_color = max(colorInput.r, max(colorInput.g,colorInput.b)); //Find the strongest color
	float min_color = min(colorInput.r, min(colorInput.g,colorInput.b)); //Find the weakest color
	
    float color_saturation = (max_color - min_color); //The difference between the two is the saturation
  
    color.rgb = lerp(luma, color.rgb, (1.0f + (Vibrance * (1.0f - (sign(Vibrance) * color_saturation)))));
	color.a = RGBLuminance(color);
  
	return saturate(color); //return the result
}
#endif

/*------------------------------------------------------------------------------
						[DYNAMIC CONTRAST CODE SECTION]
						[Adapted from CeeJay's SweetFX]
    [http://www.guru3d.com/files_details/sweetfx_shader_suite_download.html]
------------------------------------------------------------------------------*/

#if (S_CURVE_CONTRAST == 1)
float4 CurvesPass(float4 colorInput : COLOR, float2 t : TEXCOORD) : SV_Target
{
	float Curves_contrast_blend = CurvesContrast;
	float PI = acos(-1); //3.1415926589

	#if (CurveType != 2)
		float luma = RGBLuminance(colorInput);
		float3 chroma = colorInput.rgb - luma;
	#endif

	#if (CurveType == 2)
		float3 x = colorInput.rgb;
	#elif (CurveType == 1)
		float3 x = chroma;
		x = x * 0.5 + 0.5;
	#else
		float x = luma;
	#endif
		
	// -- Curve 1 -- Sin
	#if (CurvesFormula == 1)
		x = sin(PI * 0.5 * x); 
		x *= x;  
	#endif

	// -- Curve 2 -- ABS
	#if (CurvesFormula == 2)
		x = ( (x - 0.5) / (0.5 + abs(x-0.5)) ) + 0.5;
	#endif

	// -- Curve 3 -- Smoothstep
	#if (CurvesFormula == 3)
		x = x*x*(3.0-2.0*x);
	#endif

	// -- Curve 4 -- EXP formula
	#if (CurvesFormula == 4)
		x = 1.1048 / (1.0 + exp(-3.0 * (x * 2.0 - 1.0))) - (0.1048 / 2.0);
	#endif

	// -- Curve 5 -- Horner form
	#if (CurvesFormula == 5)
		x = x * (x * (1.5-x) + 0.5);
		Curves_contrast_blend = CurvesContrast * 2.0; //I multiply by two to give it a strength closer to the other curves.
	#endif

	// -- Curve 6 -- Perlins smootherstep
	#if (CurvesFormula == 6)
		x = x*x*x*(x*(x*6.0 - 15.0) + 10.0);
	#endif

	// -- Curve 7 -- ABS II
	#if (CurvesFormula == 7)
		x = ((x-0.5) / ((0.5/(4.0/3.0)) + abs((x-0.5)*1.25))) + 0.5;
	#endif

	// -- Curve 8 -- Techicolor Cinestyle
	#if (CurvesFormula == 8)
		x =  x * x *(2 * x-3)  *  (2 * x * x *x - 3 * x * x +3)  *  (4 * x * x * x -6 * x * x - 3) * 0.10;
	#endif

	// -- Curve 9 -- Parabola
	#if (CurvesFormula == 9)
		x = 0.5 -(abs(x*2-1)-2) * (x*2-1) * 0.5;
	#endif

	// -- Curve 10 -- Cubic catmull
	#if (CurvesFormula == 10)
		float a = 1.00f; //control point 1
		float b = 0.00f; //start point
		float c = 1.00f; //endpoint
		float d = 0.20f; //control point 2
		x = 0.5f * ((-a + 3*b -3*c + d)*x*x*x + (2*a -5*b + 4*c - d)*x*x + (-a+c)*x + 2*b); //A customizable cubic catmull-rom spline
	#endif

	// -- Curve 11 -- Cubic Bezier spline
	#if (CurvesFormula == 11)
		float a = 0.00; //start point
		float b = 0.00; //control point 1
		float c = 1.00; //control point 2
		float d = 1.00; //endpoint

		float r  = (1-x);
		float r2 = r*r;
		float r3 = r2 * r;
		float x2 = x*x;
		float x3 = x2*x;

		x = a*(1-x)*(1-x)*(1-x) + 3*b*(1-x)*(1-x)*x + 3*c*(1-x)*x*x + d*x*x*x;
	#endif

	// -- Curve 12 -- Cubic Bezier spline II
	#if (CurvesFormula == 12)
		float3 a = float3(0.00,0.00,0.00); //start point
		float3 b = float3(0.25,0.15,0.85); //control point 1
		float3 c = float3(0.75,0.85,0.15); //control point 2
		float3 d = float3(1.00,1.00,1.00); //endpoint

		float3 ab = lerp(a,b,x);           // point between a and b (green)
		float3 bc = lerp(b,c,x);           // point between b and c (green)
		float3 cd = lerp(c,d,x);           // point between c and d (green)
		float3 abbc = lerp(ab,bc,x);       // point between ab and bc (blue)
		float3 bccd = lerp(bc,cd,x);       // point between bc and cd (blue)
		float3 dest = lerp(abbc,bccd,x);   // point on the bezier-curve (black)
		x = dest;
	#endif

	#if (CurveType == 2) //Both Luma and Chroma
		float3 color = x; 
		colorInput.rgb = lerp(colorInput.rgb, color, Curves_contrast_blend);
	#elif (CurveType == 1) //Only Chroma
		x = x * 2 - 1;
		float3 color = luma + x; //Luma + Chroma
		colorInput.rgb = lerp(colorInput.rgb, color, Curves_contrast_blend);
	#else // CurveType == 0 //Only Luma
		x = lerp(luma, x, Curves_contrast_blend);
		colorInput.rgb = x + chroma; //Luma + Chroma 
	#endif
	
	colorInput.a = RGBLuminance(colorInput);
	
	return saturate(colorInput);
	}
#endif

/*------------------------------------------------------------------------------
					[PER-PIXEL LIGHTING CODE SECTION] - WIP
------------------------------------------------------------------------------*/

#if (PERPIXEL_LIGHTING == 1)
float4 LightPass(float4 color : COLOR, float4 p : POSITION0, float2 t : TEXCOORD0, float3 n : TEXCOORD1, float3 wp : TEXCOORD2)
{
	
	float2 light_pos = float2(0,0);
	float4 light = Texture.Sample(phongSampler, t);
	
	if(light.r > 0.225f && light.g > 0.225f && light.b > 0.225f)
	{
		light_pos = t;
		light = Texture.Sample(phongSampler, light_pos);
		color.rgb += (light*0.0333f);
	}
	
	float3 lightDir = normalize(wp - LightPosition); // per pixel diffuse lighting
	float diffuseLighting = saturate(dot(n, -lightDir));
	
	diffuseLighting *= (LightDistanceSquared / dot(LightPosition - wp, LightPosition - wp));
	
	float3 h = normalize(normalize(CameraPos - wp) - lightDir);
	float specLighting = pow(saturate(dot(h, n)), SpecularPower);
	
	float4 texel = Texture.Sample(TextureSampler, t);
	
	color += float4(saturate(AmbientLightColor + (texel.xyz * DiffuseColor * LightDiffuseColor * diffuseLighting * 0.6) + (SpecularColor * LightSpecularColor * specLighting * 0.5)), texel.w);
	color.a = RGBLuminance(color);
	
	return saturate(color);
}
#endif

/*------------------------------------------------------------------------------
						[SUBPIXEL DITHERING CODE SECTION]
------------------------------------------------------------------------------*/

#if (SUBPIXEL_DITHERING == 1)
float4 DitherPass(float4 colorInput : COLOR, float2 tex : TEXCOORD)
{
   float3 color = colorInput.rgb;
   
   float dither_size = 2.0;  //Size of the dithering grid - I'm using a 2x2 grid here.
   float dither_bit  = 8.0;  //Number of bits per channel. Should be 8 for most monitors.

   #if DitherMethod == 2 //random dithering

     //make some noise
     float noise = frac(sin(dot(float4(tex,-tex.yx), float4(float2(12.9898,78.233),float2(12.9898,78.233)* acos(-1)))) * 43758.5453); //pseudo random number generator
	 
     noise -= 0.5;

     float dither_shift = (noise) * (1.0 / (pow(2,dither_bit) - 1.0)); // using noise to determine shift. The noise ahould vary between +- 0.5.
                                                                       
     float3 dither_shift_RGB = float3(-dither_shift, dither_shift, -dither_shift); //subpixel dithering
     
     color.rgb += dither_shift_RGB;
     
   #else //DitherMethod == 1 , Ordered dithering
     //Calculate grid position
     float grid_position = frac(dot(tex,(SCREEN_SIZE / dither_size)) + (0.5 / dither_size)); //returns 0.25 and 0.75
   
     //Calculate how big the shift should be
     float dither_shift = (0.75) * (1.0 / (pow(2,dither_bit) - 1.0)); // 0.25 seems good both when using math and when eyeballing it. So does 0.75 btw.

     //Shift the individual colors differently, thus making it even harder to see the dithering Pattern
     float3 dither_shift_RGB = float3(dither_shift, -dither_shift, dither_shift); //subpixel dithering

     //modify shift acording to grid position.
     dither_shift_RGB = lerp(2.0 * dither_shift_RGB, -2.0 * dither_shift_RGB, grid_position); //shift acording to grid position.
     
     color.rgb += dither_shift_RGB;
   #endif
   
   colorInput.rgb = color.rgb;
   
   return colorInput;
}
#endif

#if (CRT_EMULATION == 1)
#define Smooth
#define Illuminate
#define Scanlines
#define fCurviness 0.05

#define ResX 512.0
#define ResY 380.0
#define PixelX 1.0/ResX
#define PixelY 1.0/ResY


#ifdef Curved
float2 radialDistortion(float2 coord)
{
	float2 cc = coord - 0.5;
	float dist = dot(cc, cc) * fCurviness;				
	return (coord + cc * (1.0 + dist) * dist);
}
#endif

float4 CRTPass(float4 color : COLOR, float2 coord : TEXCOORD0)
{
	float2 vPixelPos;
	vPixelPos.x = (coord.x * ResX) + 0.5;
	vPixelPos.y = (coord.y * ResY) + 0.5;
	
	float2 vPixel = round(vPixelPos);
	float2 vPixelDist = vPixelPos - vPixel;

	#ifdef Scanlines
	vPixelDist.x = 0.0;
	#endif

	#ifdef Wasteful
	coord.x = vPixel.x / ResX;
	coord.y = vPixel.y / ResY;
	#endif
	
	#ifdef Smooth
	float fXDiff = PixelX * FilterStrength;
	float fYDiff = PixelY * FilterStrength;
	coord.x -= fXDiff;
	coord.y -= fYDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.x += fXDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.x += fXDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.y += fYDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.y += fYDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.x -= fXDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.x -= fXDiff;
	color += Texture.Sample(TextureSampler, coord);
	coord.y -= fYDiff;
	color += Texture.Sample(TextureSampler, coord);
	color /= 9;
	#endif

	#ifdef Vignette
	color *= saturate(1.3 - (pow(coord.x-0.5,2) + pow(coord.y-0.5,2)) );
	#endif

	// Shame this doesn't work, not entirely sure why either. Should split the RGB slightly.
	#if SplitRGB
	float fTexelPos = (vPixelDist.x + 1.0) * 0.7854;
	color.x *= 1.0 - sin(fTexelPos+0.00) * 0.25;
	color.y *= 1.0 - sin(fTexelPos+0.52) * 0.25;
	color.z *= 1.0 - sin(fTexelPos+1.04) * 0.25;
	color.xyz *= 1.5;
	#endif

	float fSize = Strength;

	#ifdef Illuminate
	float fDarkness = 1.0 - ((color.x + color.y + color.z) / 3.0);
	fSize *= (fDarkness * DimmingStrength) + 0.2;
	#endif

	vPixelDist *= fSize;
	float fPixelDist = saturate( 1.0 - sqrt( pow( abs(vPixelDist.x),2 ) + pow( abs(vPixelDist.y),2 ) ) * 0.8 );

	color.xyz *= fPixelDist;
	
	return color;
}
#endif

#if (VIGNETTE == 1)
#define VignetteCenter float2(0.500, 0.500)
#ifndef VignetteRatio
  #define VignetteRatio 1.0
#endif

#ifndef VignetteType
  #define VignetteType 1
#endif

float XOR( float xor_A, float xor_B )
{
  return saturate( dot(float4(-xor_A ,-xor_A ,xor_A , xor_B) , float4(xor_B, xor_B ,1.0 ,1.0 ) ) ); // -2 * A * B + A + B
}

float4 VignettePass( float4 colorInput, float2 tex )
{
	#if VignetteType == 1
		float2 tc = tex - VignetteCenter;
		tc *= float2((PIXEL.y / PIXEL.x),VignetteRatio);
		tc /= VignetteRadius;
		float v = dot(tc,tc);	
		colorInput.rgb *= (1.0 + pow(v, VignetteSlope * 0.5) * VignetteAmount); //pow - multiply
	#endif

	#if VignetteType == 2 // round method.

		tex = -tex * tex + tex;
		colorInput.rgb = saturate(( (PIXEL.y / PIXEL.x)*(PIXEL.y / PIXEL.x) * VignetteRatio * tex.x + tex.y) * 4.0) * colorInput.rgb;
	#endif

	#if VignetteType == 3 // TV style method.

		tex = -tex * tex + tex;
		colorInput.rgb = saturate(tex.x * tex.y * 100.0) * colorInput.rgb;
	#endif
		
	#if VignetteType == 4
		tex = abs(tex - 0.5);
		float tc = dot(float4(-tex.x ,-tex.x ,tex.x , tex.y) , float4(tex.y, tex.y ,1.0 ,1.0 ) ); //XOR

		tc = saturate(tc -0.495);
		colorInput.rgb *= (pow((1.0 - tc * 200),4)+0.25);
	#endif

	#if VignetteType == 5
		tex = abs(tex - 0.5);
		float tc = dot(float4(-tex.x ,-tex.x ,tex.x , tex.y) , float4(tex.y, tex.y ,1.0 ,1.0 ) ); //XOR

		tc = saturate(tc -0.495)-0.0002;
		colorInput.rgb *= (pow((1.0 - tc * 200),4)+0.0);
	#endif

	#if VignetteType == 6 //MAD version of 2
		tex = abs(tex - 0.5);
		float tc = tex.x * (-2.0 * tex.y + 1.0) + tex.y; //XOR

		tc = saturate(tc -0.495);
		colorInput.rgb *= (pow((-tc * 200 + 1.0),4)+0.25);
	#endif

	#if VignetteType == 7 // New round (-x*x+x) * (-y*y+y) method.
		float tex_xy = dot( float4(tex,tex) , float4(-tex,1.0,1.0) );
		colorInput.rgb = saturate(tex_xy * 4.0) * colorInput.rgb;
	#endif

	return colorInput;
}
#endif

/*------------------------------------------------------------------------------
                      [MAIN() & COMBINE PASS CODE SECTION]
------------------------------------------------------------------------------*/

float4 ps_main(PS_INPUT IN) : SV_Target
{
	int colorBuffer = 2048;
	float4 bufferData = g_Buffer.Load(colorBuffer);

	float4 color = Texture.SampleLevel(TextureSampler, IN.t, 0);
	color.a = RGBLuminance(color);

	#if (FXAA_PASS == 1)
		color = FXAAPass(color, IN.t);
	#endif
	
	#if (CRT_EMULATION == 1)
		color = CRTPass(color, IN.t);
	#endif
	
	#if (LUMA_SHARPEN == 1)
		color = LumaSharpenPass(color, IN.t);
	#endif
	
	#if (INTELLI_VIBRANCE == 1)
		color = VibrancePass(color, IN.t);
	#endif
	
	#if (BLOOM_PASS == 1)
		color = BloomPass(color, IN.t);
	#endif
	
	#if (GAMMA_CORRECTION == 1)
		color = GammaPass(color, IN.t);
	#endif
	
	#if (COLOR_CORRECTION == 1)
		color = ColorPass(color, IN.t);
	#endif
	
	#if (S_CURVE_CONTRAST == 1)
		color = CurvesPass(color, IN.t);
	#endif
	
	#if (PERPIXEL_LIGHTING == 1)
		color = LightPass(color, IN.p, IN.t, IN.n, IN.wp);
	#endif
	
	#if (SUBPIXEL_DITHERING == 1)
		color = DitherPass(color, IN.t);
	#endif
	
	#if (VIGNETTE == 1)
		color = VignettePass(color, IN.t);
	#endif
	
	return saturate(color);
}
#endif
