
'======================================================================================
' This code may be freely distributed, but is not to be included with any profit-making
' software or product without the express permission from the scripting author, Steely.
'======================================================================================
'	 /			  /				 /				/			   /			  /		  '
'	/			<<<-->>>		/			   /			  /				 /		  '
'  /		 <<< ------- >@>   /			  /				 /				/		  '
' /		   <</	  Ball --- >@@/				 /				/			   /		  '
'/		  </	   - to --- >@@				/			   /			  /			  '
'		 </			- Ball - >@@		   /			  /				 /			  '
'		 </		   Collision >@@>		  /				 /				/			  '
'		 </		   - for ---- @@@		 /				/			   /			  '
'		 <</	 Visual ----- @@@		/			   /			  /				 /'
'		/<<<   Pinball ----- >@@>	   /			  /				 /				/ '
'	   /  <<< -------------- @@@> A	  /				 /				/			   /  '
'	  /	   <<<< ---------- >@@@> PinballKen			/			   /			  /	  '
'	 /		 <<<<< ----- >@@@> & Steely			   /			  /				 /	  '
'	/			<<<<->>@@> Production			  /				 /	Ver 1.1		/	  '
'  /			   /			  /				 /				/			   /	  '
'======================================================================================
' Many thanks go to Randy Davis and to all the multitudes of people who have
' contributed to VP over the years, keeping it alive!!! Enjoy, Steely & PK
'
' v1.1 Updated with code changes/additions by JimmyFingers and Koadic (file modified by Koadic)
'======================================================================================
'

' To add B2B collision sounds to a VP9 table follow these few steps...

' 1) Create a new timer in the VP playfield editor and name it ==> XYdata
'	 DO NOT ENABLE this timer, it automatically turns on/off when needed.
' 2) Import the ten provided collide*.wav files (0-9) with the sound manager.
'	 If you choose to use your own sound files, they must be named collide0.wav
'	 thru whatever number. Example: collide0.wav to collide7.wav (8 files)
' 3) Copy/paste the next few lines into your table script.
'	 NOTE: these five lines should be placed before any other table script but after the LoadVPM call.
'	 Set the quantites for tnopb and nosf.
'
' This "B2B.vbs" file must be placed in the same folder as the table.
'
'	Dim tnopb, nosf, B2BOn
'	tnopb = 4	' <<<<< SET to the "Total Number Of Possible Balls" in play at any one time
'	nosf = 10	' <<<<< SET to the "Number Of Sound Files" used / B2B collision volume levels
'	B2BOn = 2	' <<<<< 0=Off, 1=On, 2=AutoDetect
'
'	ExecuteGlobal GetTextFile("B2B.vbs")
'
'
' 4) For each and every kicker that creates and/or destroys balls you must add a command
'	 for ball identification... This is very important when adapting a table. If you get
'	 an error while playing during a multiball sequence, you may have missed adding code
'	 to a kicker's Hit or unHit event or the command code was not placed in the proper
'	 location. Also note that not all kickers create and destroy balls.
'
'	 First: Determine which kickers destroy balls and insert "ClearBallid" into the kicker's
'	 Hit sub. This must be placed before a ball is destroyed to clear the ball object ID.
'
'	 Example below:
'
'		Sub Kicker1_Hit() ClearBallid : AddBall me : End Sub		'AddBall - simulating a VPM command
'		Sub Kicker2_Hit() ClearBallid : Kicker2.DestroyBall : End Sub
'		Sub Drain_Hit()
'			PlaySound "drain"
'			ClearBallid
'			Drain.DestroyBall
'		End Sub
'
'	Second: For each kicker that creates balls, you need to replace Kicker.createball with "CreateBallid(Kicker)".
'	Alternatively, for VPM tables, you can instead call vpmCreateBall(Kicker) as the standard vpmCreateBall call is
'	replaced via this script. Balls generated by the core.vbs for troughs and the like will be created with an ID
'	automatically. If you want to create a custom sized ball with an ID, you can use CreateBallID2(Kicker, BallSize)
'
'______________________________________________________________________________________
'
'		================================================
'		======== Ball to Ball Collision Effects ========
'		================================================
'			A brief description of operation
'
' Features...
'
' * This code senses ball-to-ball proximity to determine a collision and then calculates
' the collision force (cForce) to select a varying sound level or volume.
'
' * Any number of sound levels/files can be used and set by the table author.
'
' * Any number of balls can be set by the author, however the quantity is limited by what
' the individual computer can handle. The majority of tables use only a few balls at
' one time, so hopefully this shouldn't be an issue for most people.
'
' To combat this, the XYdata_Timer interval is self regulating to help retain performance.
' There is also an auto-shutoff, "coff" collision off variable*, which is triggered by
' the timer interval if it should go too high (*changed to B2BOn).
'
' The main elements or commands used for this B2B collision sound effects are...
' 1) PlaySound("collide" & cForce)
'	This line selects and plays the proper wav file, combining "collide" with the
'	collision force for a variable sound level or volume. So the files are named...
'		"collide0", "collide1"... thru "collide9".
'
' 2) cForce = Cint((abs(TotalVel(cb1,id3)*Cos(cAngle-bAngle1))+abs(TotalVel(cb2,id3)*Cos(cAngle-bAngle2))))
'	There are more lines for the cForce calculations, but this main equation works by
'	taking the total velocity of each ball and multiplying it by the difference between
'	the collision angle and the angle of the traveling ball. In other words, a head on
'	collision and/or higher velocity produces a louder sound than a glancing blow.
'
' 3) XYdata_Timer...
'	This timer collects the balls' coordinates and velocities. It then determines
'	if two balls are close enough for a collision to occur.
'
' 4) Identifying the balls for reading the ball coordinates and velocities...
'	Unfortunately balls don't have scriptable hit events, so they must be identified
'	and set as an object whenever they are created. Balls then need to be marked as
'	inactive when one is destroyed. This is taken care of by the "CreateBallID" (and
'	vpmCreateBall via the core.vbs) and "ClearBallid" subs.
'
'=======================================================
' Detailed descriptions are given below within the code
'=======================================================

Option Explicit

ReDim CurrentBall(tnopb), ballStatus(tnopb)
XYdata.interval = 10			' Timer interval starts at 1 for the highest ball data sample rate

Dim cnt
For cnt = 0 to ubound(ballStatus)	' Initialize/clear all ball stats, 1 = active, 0 = non-existant
	ballStatus(cnt) = 0
Next

CheckB2B

'****************************************
' B2B AutoDisable for XP x64 Added by Koadic
'****************************************

Sub CheckB2B			' Added by Koadic for XP x64 handling
  Dim osver, cpuver, check
  On Error Resume Next
	For x = 0 to 1 : If B2BOn = x Then Exit Sub : End If : Next 'If B2BOn is set manually, then end routine
	Set check = CreateObject("WScript.Shell")
	osver = check.RegRead ("HKLM\Software\Microsoft\Windows NT\CurrentVersion\CurrentVersion")
	cpuver = check.RegRead ("HKLM\SYSTEM\ControlSet001\Control\Session Manager\Environment\Processor_Architecture")
	If osver < 6 and cpuver = "AMD64" Then B2BOn = 0 Else B2BOn = 1 'If OS is XP and 64bit, then disable B2B
	If Err Then B2BOn = 1 'If there is an error in detecting either OS or x32/x64, then default to On
  On Error Goto 0
End Sub

 '======================================================
' <<<<<<<<<<<<<< Ball Identification >>>>>>>>>>>>>>
'======================================================
'******************************
' Destruk's alternative vpmCreateBall for use with B2B Enabled tables
' Core.vbs calls vpmCreateBall when a ball is created from a ball stack
'******************************
If IsEmpty(Eval("vpmCreateBall"))=false Then Set vpmCreateBall = GetRef("B2BvpmCreateBall")	' Override the core.vbs and redefine vpmCreateBall

Function B2BvpmCreateBall(aKicker)
	Dim bsize2:If IsEmpty(Eval("ballsize"))=true Then bsize2 = 25 Else bsize2 = ballsize/2
	For cnt = 1 to ubound(ballStatus)				' Loop through all possible ball IDs
		If ballStatus(cnt) = 0 Then					' If ball ID is available...
			If Not IsEmpty(vpmBallImage) Then		' Set ball object with the first available ID
				Set CurrentBall(cnt) = aKicker.Createsizedball(bsize2).Image
			Else
				Set CurrentBall(cnt) = aKicker.Createsizedball(bsize2)
			End If
			Set B2BvpmCreateBall = aKicker
			CurrentBall(cnt).uservalue = cnt		' Assign the ball's uservalue to it's new ID
			ballStatus(cnt) = 1						' Mark this ball status active
			ballStatus(0) = ballStatus(0)+1			' Increment ballStatus(0), the number of active balls
			If B2BOn > 0 Then						' If B2BOn is 0, it overrides auto-turn on collision detection
													' If more than one ball active, start collision detection process
				If ballStatus(0) > 1 and XYdata.enabled = False Then XYdata.enabled = True
			End If
			Exit For								' New ball ID assigned, exit loop
		End If
	Next
End Function

' Use CreateBallID(kickername) to manually create a ball with a BallID
' Can also be used on nonVPM tables (EM or Custom)

Sub CreateBallID(aKicker)
	Dim bsize2:If IsEmpty(Eval("ballsize"))=true Then bsize2 = 25 Else bsize2 = ballsize/2
	For cnt = 1 to ubound(ballStatus)				' Loop through all possible ball IDs
		If ballStatus(cnt) = 0 Then					' If ball ID is available...
			Set CurrentBall(cnt) = aKicker.Createsizedball(bsize2)		' Set ball object with the first available ID
			CurrentBall(cnt).uservalue = cnt		' Assign the ball's uservalue to it's new ID
			ballStatus(cnt) = 1						' Mark this ball status active
			ballStatus(0) = ballStatus(0)+1			' Increment ballStatus(0), the number of active balls
			If B2BOn > 0 Then						' If B2BOn is 0, it overrides auto-turn on collision detection
													' If more than one ball active, start collision detection process
				If ballStatus(0) > 1 and XYdata.enabled = False Then XYdata.enabled = True
			End If
			Exit For								' New ball ID assigned, exit loop
		End If
	Next
End Sub

' Use CreateBallID2(kickername, ballsize) to manually create a custom sized ball with a BallID
' Can also be used on nonVPM tables (EM or Custom)

Sub CreateBallID2(aKicker, bsize2)					' Use to manually create a ball with a BallID with a custom size
	For cnt = 1 to ubound(ballStatus)				' Loop through all possible ball IDs
		If ballStatus(cnt) = 0 Then					' If ball ID is available...
			Set CurrentBall(cnt) = aKicker.Createsizedball(bsize2/2)		' Set ball object with the first available ID
			CurrentBall(cnt).uservalue = cnt		' Assign the ball's uservalue to it's new ID
			ballStatus(cnt) = 1						' Mark this ball status active
			ballStatus(0) = ballStatus(0)+1			' Increment ballStatus(0), the number of active balls
			If B2BOn > 0 Then						' If B2BOn is 0, it overrides auto-turn on collision detection
													' If more than one ball active, start collision detection process
				If ballStatus(0) > 1 and XYdata.enabled = False Then XYdata.enabled = True
			End If
			Exit For								' New ball ID assigned, exit loop
		End If
	Next
End Sub

'Call this sub from every kicker that destroys a ball, before the ball is destroyed.

Sub ClearBallid
	Dim iball
	On Error Resume Next				' Error handling for debugging purposes
	iball = ActiveBall.uservalue		' Get the ball ID to be cleared
	If Err Then Msgbox Err.description & vbCrLf & iball
		ballStatus(iBall) = 0			' Clear the ball status
	ballStatus(0) = ballStatus(0)-1		' Subtract 1 ball from the # of balls in play
	On Error Goto 0
End Sub


'=====================================================
' <<<<<<<<<<<<<<<<< XYdata_Timer >>>>>>>>>>>>>>>>>
'=====================================================
' Ball data collection and B2B Collision detection.

Dim cFactor

Sub XYdata_Timer()
	Dim bDistance, xyTime, id, id2, id3, B1, B2
	ReDim baX(tnopb,4), baY(tnopb,4), bVx(tnopb,4), bVy(tnopb,4), TotalVel(tnopb,4)
	' xyTime... Timers will not loop or start over 'til it's code is finished executing. To maximize
	' performance, at the end of this timer, if the timer's interval is shorter than the individual
	' computer can handle this timer's interval will increment by 1 millisecond.
	xyTime = Timer+(XYdata.interval*.001)	' xyTime is the system timer plus the current interval time
	' Ball Data... When a collision occurs a ball's velocity is often less than it's velocity before the
	' collision, if not zero. So the ball data is sampled and saved for four timer cycles.
	If id2 >= 4 Then id2 = 0						' Loop four times and start over
	id2 = id2+1								' Increment the ball sampler ID
	For id = 1 to ubound(ballStatus)					' Loop once for each possible ball
		If ballStatus(id) = 1 Then								' If ball is active...
			baX(id,id2) = round(CurrentBall(id).x,2)			' Sample x-coord
			baY(id,id2) = round(CurrentBall(id).y,2)			' Sample y-coord
			baZ(id,id2) = round(CurrentBall(id).z,2)			' Sample z-coord
			bVx(id,id2) = round(CurrentBall(id).velx,2)			' Sample x-velocity
			bVy(id,id2) = round(CurrentBall(id).vely,2)			' Sample y-velocity
			TotalVel(id,id2) = (bVx(id,id2)^2 + bVy(id,id2)^2)	' Calculate total velocity
			If TotalVel(id,id2) > TotalVel(0,0) Then TotalVel(0,0) = int(TotalVel(id,id2))
		End If
	Next
	' Collision Detection Loop - check all possible ball combinations for a collision.
	' bDistance automatically sets the distance between two colliding balls. Zero milimeters between
	' balls would be perfect, but because of timing issues with ball velocity, fast-traveling balls
	' prevent a low setting from always working, so bDistance becomes more of a sensitivity setting,
	' which is automated with calculations using the balls' velocities.
	' Ball x/y-coords plus the bDistance determines B2B proximity and triggers a collision.
	id3 = id2 : B2 = 2 : B1 = 1						' Set up the counters for looping
	Do
		If ballStatus(B1) = 1 and ballStatus(B2) = 1 Then		' If both balls are active...
			bDistance = int((TotalVel(B1,id3)+TotalVel(B2,id3))^(1.04 * (CurrentBall(B1).radius + CurrentBall(B2).radius)/50))
			If ((baX(B1,id3) - baX(B2,id3))^2 + (baY(B1,id3) - baY(B2,id3))^2) < (2800 * ((CurrentBall(B1).radius + CurrentBall(B2).radius)/50)^2) + bDistance Then
				If ABS(baZ(B1,id3) - baZ(B2,id3)) < (CurrentBall(B1).radius + CurrentBall(B2).radius) Then collide B1,B2 : Exit Sub 'added z axis collision detection here
			End If
		End If
		B1 = B1+1							' Increment ball1
		If B1 >= ubound(ballStatus) Then Exit Do		' Exit loop if all ball combinations checked
		If B1 >= B2 then B1 = 1:B2 = B2+1				' If ball1 >= reset ball1 and increment ball2
	Loop

	If ballStatus(0) <= 1 Then XYdata.enabled = False	' Turn off timer if one ball or less

	If XYdata.interval >= 40 Then B2BOn = False : XYdata.enabled = False	' Auto-shut off
	If Timer > xyTime * 3 Then B2BOn = False : XYdata.enabled = False		' Auto-shut off
	If Timer > xyTime Then XYdata.interval = XYdata.interval+1				' Increment interval if needed
End Sub

'=========================================================
' <<<<<<<<<<< Collide(ball id1, ball id2) >>>>>>>>>>>
'=========================================================
'Calculate the collision force and play sound accordingly.
Dim cTime

Sub Collide(cb1,cb2)
	Dim avgBallx, cAngle, bAngle1, bAngle2, cForce
' The Collision Factor(cFactor) uses the maximum total ball velocity and automates the cForce calculation, maximizing the
' use of all sound files/volume levels. So all the available B2B sound levels are automatically used by adjusting to a
' player's style and the table's characteristics.
	If TotalVel(0,0)/1.8 > cFactor Then cFactor = int(TotalVel(0,0)/1.8)
' The following six lines limit repeated collisions if the balls are close together for any period of time
	avgBallx = (bvX(cb2,1)+bvX(cb2,2)+bvX(cb2,3)+bvX(cb2,4))/4
	If avgBallx < bvX(cb2,id2)+.1 and avgBallx > bvX(cb2,id2)-.1 Then
	If ABS(TotalVel(cb1,id2)-TotalVel(cb2,id2)) < .000005 Then Exit Sub
	End If
	If Timer < cTime Then Exit Sub
	cTime = Timer+.1				' Limits collisions to .1 seconds apart
' GetAngle(x-value, y-value, the angle name) calculates any x/y-coords or x/y-velocities and returns named angle in radians
	GetAngle baX(cb1,id3)-baX(cb2,id3), baY(cb1,id3)-baY(cb2,id3),cAngle	' Collision angle via x/y-coordinates
	id3 = id3 - 1 : If id3 = 0 Then id3 = 4	' Step back one xyData sampling for a good velocity reading
	GetAngle bVx(cb1,id3), bVy(cb1,id3), bAngle1	' ball 1 travel direction, via velocity
	GetAngle bVx(cb2,id3), bVy(cb2,id3), bAngle2	' ball 2 travel direction, via velocity
' The main cForce formula, calculating the strength of a collision
	cForce = Cint((abs(TotalVel(cb1,id3)*Cos(cAngle-bAngle1))+abs(TotalVel(cb2,id3)*Cos(cAngle-bAngle2))))
	If cForce < 4 Then Exit Sub				' Another collision limiter
	cForce = Cint(cForce/(cFactor/nosf))	' Divides up cForce for the proper sound selection.
	If cForce > nosf-1 Then cForce = nosf-1	' First sound file 0(zero) minus one from number of sound files
	PlaySound("collide" & cForce)			' Combines "collide" with the calculated sound level and play sound
End Sub

'=================================================
' <<<<<<<< GetAngle(X, Y, Anglename) >>>>>>>>
'=================================================
' A repeated function which takes any set of coordinates or velocities and calculates an angle in radians.

Sub GetAngle(Xin, Yin, wAngle)
	Dim rAngle,Radit
	Const Pi = 3.14159265358979
	If Sgn(Xin) = 0 Then
		If Sgn(Yin) = 1 Then rAngle = 3 * Pi/2 Else rAngle = Pi/2
		If Sgn(Yin) = 0 Then rAngle = 0
	Else
		rAngle = atn(-Yin/Xin)			' Calculates angle in radians before quadrant data
	End If
	If sgn(Xin) = -1 Then Radit = Pi Else Radit = 0
	If sgn(Xin) = 1 and sgn(Yin) = 1 Then Radit = 2 * Pi
	wAngle = round((Radit + rAngle),4)		' Calculates angle in radians with quadrant data
	'"wAngle = round((180/Pi) * (Radit + rAngle),4)" ' Will convert radian measurements to degrees - to be used in future
End Sub
