/*
 * portable, cycle exact 68000 emulator
 *
 */
#ifndef CORE_68k_h
#define CORE_68k_h

#include <dataTypes.h>

class Core_68k {

public:
    Core_68k() { build_optable(); }

    enum Interrupt { USER_VECTOR/*Amiga*/, AUTO_VECTOR/*Genesis*/, SPURIOUS, UNINITIALIZED };

//public interface
    void power(); //hard reset, includes reset exception
    void reset(); //performs reset exception
    void process(); //emulates next opcode
    void setInterrupt( u8 level ); //generated by external device (level 0 - 7), 0 -> DISABLE, 7 -> NMI
    void setInterruptType( Interrupt type ) { interrupt = type; } //default is AUTO_VECTOR
    u16 openBus() { return reg_irc; } //last readed program counter value

    /** checks if read-modify-write cycle of TAS instruction is active
        address strobe (AS pin) remains asserted after read cycle, so the read and the following write
        are indivisible, just like one single bus cycle.
        by emulating concurrent cpu instances you should assure second cpu doesn't get bus during this write cycle
        to prevent deadlocks (if used for semaphore operations and second cpu reads same data like first cpu => DEADLOCK)

        it's unsupported for Genesis (I/O controller of Genesis 1 and 2 prevent write, so check it)
        it's unsupported for Amiga (danger: TAS is not atomic if Amiga uses DMA)

        testing only makes sense during bus access callbacks, it's useless at opcode edge
    **/
    bool isRMWCycle() { return rmwCycle; }

    //bus error is generated by an external device (available in Atari ST, not in Amiga or Genesis)
    //call it as a result of an unwanted memory access
    void raiseBusError(u32 addr) {
        fault_address = addr;
        group0exception( BUS_ERROR );
    }

protected:
    //callbacks you have to implement

    /** reads / writes in your memory map **/
    virtual u8 memRead(u32 addr) = 0;
    virtual u16 memWordRead(u32 addr) = 0;
    virtual void memWrite(u32 addr, u8 data) = 0;
    virtual void memWordWrite(u32 addr, u16 data) = 0;

    /** in comparison to other emulators "sync" doesn't gives you the complete instruction cycle count
        but the consumed cycles more times per instruction. in other words by each memory access
        during the instruction you have knowledge about how many cycles have passed so far.
        you have to sum up the cycles by yourself !
        so you have the chance to exactly sync up your other concurrent tasks before accessing system bus **/
    virtual void sync(u8 cycles) = 0;

    //other callbacks ( if needed )

    virtual void cpuHalted() {} //cpu informs about halted state, needs a reset to recover
    virtual void resetInstruction() {} //resets external devices (e.g. Amiga), unsupported for (e.g. Genesis)
    //cpu samples state of ipl lines (interrupt level) 2 cycles before opcode edge, maybe you want to sync up your concurrent tasks before
    virtual void sampleIrq();
    //interrupt acknowledge cycle fetches vector from external device (e.g. Amiga), not needed for AUTO_VECTOR (e.g. Genesis)
    virtual unsigned getUserVector( u8 level ) { return 0; }

//end of public interface

    enum Size { SizeByte = 0, SizeWord = 1, SizeLong = 2 };
	enum { flag_logical, flag_cmp, flag_add, flag_sub, flag_addx, flag_subx, flag_zn };
    enum ADM {	DR_DIRECT = 0, AR_DIRECT = 8, AR_INDIRECT = 16, AR_INDIRECT_INC = 24
            ,AR_INDIRECT_DEC = 32, AR_INDIRECT_D16 = 40, AR_INDIRECT_D8 = 48
            ,ABS_SHORT = 56, ABS_LONG = 57, PC_INDIRECT_D16 = 58
            ,PC_INDIRECT_D8 = 59, IMMEDIATE = 60, UNSELECT = 0xff};

    enum { BUS_ERROR, ADDRESS_ERROR };
    enum { ILLEGAL_OPCODE, PRIVILEGE, LINE_A, LINE_F, NONE };

    //for debugging purposes
    void group1exceptions();
    virtual void op_illegal(u16 opcode);
    virtual void group0exception(u8 type);
    virtual void trapException(u8 vector);
    virtual void setPrivilegeException();
    virtual void logInstruction(u16 _word, bool isNextOpcode) {}

    u32 getRegA(unsigned reg) {
        return reg_a[reg & 7];
    }
    u32 getRegD(unsigned reg) {
        return reg_d[reg & 7];
    }
    void setRegA(u8 reg, u32 value) {
        reg_a[reg & 7] = value;
    }
    void setRegD(u8 reg, u32 value) {
        reg_d[reg & 7] = value;
    }
    u32 getSR() {
        return reg_s;
    }
    void setSR(u16 val) {
        reg_s = val;
    }
    u16 getRegIrd() {
        return reg_ird;
    }
    void setSSP(u32 val) {
        reg_ssp = val;
    }
    void setCCR(u8 val) {
        reg_s.l = val;
    }

private:
    #include "regs.h"
    #include "mem.h"

    //needed for bus/address error stackframe
    struct STATUS_CODE {
        bool _program;      //program or data
        bool _read;         //read or write
        bool _instruction;  //instruction or not

        void reset() {
            _program = false;
            _read = true;
            _instruction = true;
        }
    } status_code;

    class CpuException {
    public:
        CpuException() {}
    };

    void (Core_68k::*opcodes[0x10000])(u16 opcode);

    unsigned getInterruptVector(u8 level);

    Interrupt interrupt;
    u8 irqPendingLevel;
    u8 irqSamplingLevel;

    u32 fault_address;

    bool doubleFault; //double fault, needs reset to recover
    bool trace;
    bool stop;
    bool rmwCycle;
    u8 illegalMode;

	u32 eaAddr; //calced ea
	reg_32* eaReg; //used ea reg
    ADM adm; //adress mode

    void traceException();
    void illegalException(u8 iType);
    void interruptException(u8 level);

    void executeAt(u8 vector);
    void switchToSupervisor();
    void switchToUser();

	void build_optable();

	void Arithmetic(u16 opcode);
	void ArithmeticI(u16 opcode);
	void ArithmeticQ(u16 opcode);

    template<u8 size, bool writeEa> void op_and(u16 opcode);
    template<u8 size> void op_andi(u16 opcode);
    template<u8 size> void op_eori(u16 opcode);
    template<u8 size> void op_addi(u16 opcode);
    template<u8 size> void op_subi(u16 opcode);
    template<u8 size> void op_cmpi(u16 opcode);
    template<u8 size> void op_cmpm(u16 opcode);
    template<u8 size> void op_addq(u16 opcode);
    template<u8 size> void op_subq(u16 opcode);
    template<u8 size, bool memTomem> void op_addx(u16 opcode);
    template<u8 size, bool memTomem> void op_subx(u16 opcode);
	void op_andiccr(u16 opcode);
	void op_andisr(u16 opcode);
    template<u8 size, bool writeEa> void op_or(u16 opcode);
    template<u8 size> void op_ori(u16 opcode);
	void op_orisr(u16 opcode);
	void op_oriccr(u16 opcode);
    template<u8 size> void op_eor(u16 opcode);
	void op_eorisr(u16 opcode);
	void op_eoriccr(u16 opcode);
    template<u8 size> void op_cmp(u16 opcode);
    template<u8 size, bool writeEa> void op_add(u16 opcode);
    template<u8 size, bool writeEa> void op_sub(u16 opcode);
    template<u8 size> void op_cmpa(u16 opcode);
    template<u8 size> void op_adda(u16 opcode);
    template<u8 size> void op_suba(u16 opcode);
    template<bool memTomem> void op_abcd(u16 opcode);
    template<bool memTomem> void op_sbcd(u16 opcode);
	void op_swap(u16 opcode);

	void op_mulu(u16 opcode);
	void op_muls(u16 opcode);
	void op_divu(u16 opcode);
	void op_divs(u16 opcode);
	void op_unlk(u16 opcode);
	void op_link(u16 opcode);
	void op_bcc(u16 opcode);
	void op_bset_reg(u16 opcode);
	void op_bset_mem(u16 opcode);
	void op_btst_reg(u16 opcode);
	void op_btst_mem(u16 opcode);
	void op_bra(u16 opcode);
	void op_bsr(u16 opcode);
	void op_chk(u16 opcode);
	void op_dbcc(u16 opcode);
    template<u8 opmode> void op_exg(u16 opcode);
    template<bool _longSize> void op_ext(u16 opcode);
	void op_jmp(u16 opcode);
	void op_jsr(u16 opcode);
	void op_lea(u16 opcode);
	void op_pea(u16 opcode);
	void op_movefromsr(u16 opcode);
	void op_movetoccr(u16 opcode);
	void op_movetosr(u16 opcode);
    template<bool dr> void op_moveusp(u16 opcode);
	void op_nop(u16 opcode);
	void op_reset(u16 opcode);
	void op_rte(u16 opcode);
	void op_rtr(u16 opcode);
	void op_rts(u16 opcode);
	void op_stop(u16 opcode);
	void op_trap(u16 opcode);
	void op_trapv(u16 opcode);

	template<bool left, bool memory, bool arithmetic> void op_xsx(u16 opcode);
	template<bool left, bool memory, bool extend> void op_rox(u16 opcode);
	template<bool dynamic> void op_bchg(u16 opcode);
	template<bool dynamic> void op_bclr(u16 opcode);
	template<bool dynamic> void op_bset(u16 opcode);
	template<bool dynamic> void op_btst(u16 opcode);
    template<u8 size> void op_clr(u16 opcode);
	void op_nbcd(u16 opcode);
    template<bool negx, u8 size> void op_neg(u16 opcode);
    template<u8 size> void op_not(u16 opcode);
	void op_scc(u16 opcode);
	void op_tas(u16 opcode);
    template<u8 size> void op_tst(u16 opcode);
    template<u8 size> void op_move(u16 opcode);
    template<u8 size> void op_movea(u16 opcode);
	void op_moveq(u16 opcode);
    template<u8 size, bool memToReg> void op_movem(u16 opcode);
    template<u8 opmode> void op_movep(u16 opcode);

	int CheckEA(u8 ea, char* valid);
	bool conditionalTest(u8 code);
    void setFlags(u8 type, u8 size, u64 result, u32 src = 0, u32 dest = 0);

    u32 LoadEA(u8 size, u8 ea, bool noReadFromEA = false, bool fetchLastExtension = true, bool noSyncDec = false);
    void writeEA(u8 size, u32 value, bool lastBusCycle = false);
    void updateRegAForIndirectAddressing(u8 size, u8 regPos);

	int getDivs68kCycles (i32 dividend, i16 divisor);
	int getDivu68kCycles (u32 dividend, u16 divisor);

    template<bool arithmetic, u8 size> void shift_left(u8 shiftCount, u32& data);
    template<bool arithmetic, u8 size> void shift_right(u8 shiftCount, u32& data);
    template<bool extend, u8 size> void rotate_left(u8 shiftCount, u32& data);
    template<bool extend, u8 size> void rotate_right(u8 shiftCount, u32& data);

	u8 bits_(u8 size) {
		switch(size) {
			case SizeByte: return 8;
			case SizeWord: return 16;
			case SizeLong: return 32;
			default: return 0;
		}
	}

	u32 msb_(u8 size) {
		switch(size) {
			case SizeByte: return 0x80;
			case SizeWord: return 0x8000;
			case SizeLong: return 0x80000000;
			default: return 0;
		}
	}

    u32 maskVal_(u32 value, u8 size) {
        return value & mask_(size);
    }

    u32 mask_(u8 size) {
		switch(size) {
            case SizeByte: return 0xFF;
            case SizeWord: return 0xFFFF;
            case SizeLong:
            default:
                return 0xFFFFFFFF;
        }
	}

    bool isMemoryOperand() {
        if (adm != DR_DIRECT && adm != AR_DIRECT && adm != IMMEDIATE) {
            return true;
        }
        return false;
    }

    bool isRegisterMode() {
        return eaReg ? true : false;
    }

    void incrementPc() {
        reg_pc += 2;
    }

};

#endif
