;
;       ZX81 1K CHESS
;
;       Format: Z80
;
;       Z80 Assembler by Gavin Turner
;
;       Based on ZX81 1K CHESS by David R Horne 1983
;
;
;  DRIVER
;       |->KYBD -> TPK
;       |  PIECE -> STR1 -> PAWN
;       |            \ALIST
;       -- TESTLIST
;          PMOVE
;              |
;              MPSCAN
;                   |-> STR -> PAWN
;                   |   PIECE -> STR1 -> PAWN
;                   |             \ALIST
;                   | |->TESTLIST
;                   | |  CHK
;                   | |  SCORE -> PIECE
;                   | |            \INC -> SQATTACK
;                   |-|--CHGSQ -> CHGMV
.Z80
NOLOG
		inputbuffer 	   EQU $839F
		chess_board 	   EQU $833E
		whos_turn 	   EQU $8337
		piece_score_table  EQU $80F2
		table_data 	   EQU $80E7
		legal_moves_list   EQU $80E4
		attack_register    EQU $8080
		move_buffer_alt    EQU $8063
		move_buffer 	   EQU $8046
		buffer_index 	   EQU $8046
		move_score 	   EQU $8041
		move_from 	   EQU $8007

		high_addr 	   EQU $83

		ORG $8082-7

NEWGAME:
		LD HL,DRIVER	  ;DRIVER IS 1K CHESS ENTRY POINT
		PUSH HL 	  ;
		JP INITIALIZE	  ;Emulator Preparation Subroutine

;<----------------------------  THE ORIGINAL ZX81 1K CHESS SOURCE CODE STARTS HERE AT $4082  ------------------------------->



;  TKP  $4082  -  Test Key Press waits for input from human player, Returns only when valid keys are pressed
;
;  Description: This 'key-scan' subroutine waits in an infinite loop until valid keys are pressed 1 - 8 or A - H
;               on entry HL should point to the input buffer position and result is stored in the input buffer pointer HL
;
;  Registers Modified:  DE , A

TKP:
		PUSH HL 		    ;Save Registers BC,HL
					    ;
TKP01:  	PUSH BC 		    ;
TKP02:  	CALL ROM02BB		    ;Keyboard Scan, Returns 255 if no key down
		LD B,H			    ;
		LD C,L			    ;
		LD D,L			    ;
		INC D			    ;Test if D = 255 (No Key Pressed)
		JR Z,TKP02		    ;Loop indefinitely until a key is pressed
	       ;CALL ROM07BD_DECODE         ;ZX81 ROM Subroutine - DECODE KEYBOARD SCAN
	       ;LD A,(HL)                   ;
		POP BC			    ;Restore C (Numbers or Letters Range) and Loop iterator B = 8
		PUSH BC 		    ;
TKP03:  	CP C			    ;Compare Scan Code in A against valid entries that range from C to C + 7
		JR Z,TKP04		    ;
		INC C			    ;
		DJNZ TKP03		    ;Repeat for the rest of allowed character range
		POP BC			    ;
		JR TKP01		    ;Key press invalid try again
TKP04:  				    ;
		POP BC			    ;
		POP HL			    ;   Valid Results-;'1'=$1D '2'=$1E '3'=$1F '4'=$20 '5'=$21 '6'=$22 '7'=$23 '8'=$24
		LD (HL),A		    ;Save Scan Code   ;'A'=$26 'B'=$27 'C'=$28 'D'=$29 'E'=$2A 'F'=$2B 'G'=$2C 'H'=$2D
;------------------------------------------------------------------------------
 CALL PRINT				    ;THIS CALL OUTPUTS THE KEY INPUT TO THE DISPLAY
 NOP 					    ;NOP TO KEEP CODE ALIGNED TO ORIGINAL PROGRAM
;------------------------------------------------------------------------------
		RET 			    ;



;  KYBD  $40A0  -  Uses TKP to get move input of chess board grid coordinates, the memory address of the square is returned in HL
;
;  Description:  Using C to indicate rows (numbers) and columns (letters) calls TKP which doesnt return until a valid key press has occurred
;                in the range of C to C + 7 the ZX81 chess board is 11 bytes wide in screen RAM a simple math routine returns a 16 bit board square location
;
;  Registers Modified:  HL , BC , DE , A

KYBD:
		LD BC,$081D		    ;Loop iterator = 8 ; C = 29 ; Keyboard Scan Code range 29 to 37
		CALL TKP		    ;Wait for valid key press within range
		DEC HL			    ;
		LD C,$26		    ;C = 38 ; Keyboard Scan Code range 38 to 46
		CALL TKP		    ;Wait for valid key press within range
		INC HL			    ;-----  Convert X Y grid coords to a chess board memory address ------
		LD A,(HL)		    ; A = Column
		SUB $1C 		    ; Subract 28 to get Row (Range 1 - 8)
		LD B,A			    ; B = Row
		LD C,$0B		    ; Row multiplier 11 (size of each row in ZX81 screen memory)
		XOR A			    ;
					    ;
GKI01:  	ADD A,C 		    ; 11 times Row
		DJNZ GKI01		    ;
					    ;
		ADD A,$61		    ; Add 97 to result
		DEC HL			    ; Point 'HL' to column entered by Mover
		SUB (HL)		    ; (HL) =-($61+A) ; Adjust value in HL
		;flows onto STR


;  STR  $40BB  -  This routine takes the board address and determines whether the contents are:
;
;                       '0' Different from the current movers colour
;                       '1' Emtpy
;                       '2' Outside the chess board or
;                       '3' The same colour as the current mover
;
;  Description:  Useful Analyze square subroutine , Called from MOVE PAWN SQATTACK and MPSCAN subroutines for deciding possible moves
;
;  Registers Modified:  HL , BC , DE , A

STR1:
		LD C,A			    ;
		LD L,C			    ;Index points to square
		LD H,high_addr		    ;Point 'HL' to board address
STR: 					    ;
		LD A,(HL)		    ;Get square data from Chess Board at address HL
		LD B,1			    ;
		AND $7F 		    ;Mask Out Colour Bit
		CP 0			    ;
		JR Z,STR01		    ;If A < $80 Then return A='1' Empty Square
		INC B			 ;-------;
		CP $76			 ;       ;
		JR Z,STR01		 ;       ;If A = $76 or A > $27 Then return A='2' Move lands on 'Wall'
		CP $27			 ;       ;
		JR C,STR01		 ;-------;
		LD A,(HL)		    ;Refetch Board Square data, Square has a piece there
		INC B			    ;B = 3
		LD L,$37		    ;
		ADD A,(HL)		    ;ADD A,(whos_turn) $00 OR $80
		BIT 7,A 		    ;
		JR Z,STR01		    ;Bit 7 is not set so square has same colour piece, return a result of '3' Same Colour
		LD B,0			    ;B = 0 'Different' Colour Piece on square
STR01:  	LD A,B			    ;
		LD L,C			    ;L HOLDS BOARD OFFSET
		RET
      ;Result in A '0' DIFF  '1' EMPTY  '2' WALL  '3' SAME


;Possible chess piece moves data
		DB $01,$0B,$FF,$F5,$F6,$F4,$0C,$0A
		DB $0D,$F3,$15,$EB,$17,$E9,$F7,$09
		DB $0B,$0A,$0C
piece_score_data:
		;Q(5),R(4),B(3),N(2),P(1)
		DB $36,$37,$27,$33,$35



;  PIECE  $40F7  -  This sets up pointers to possible move tables and number of possible move directions
;
;  Description:  PIECE is a preparation routine for MOVE (MOVE is not called from anywhere in the code)
;
;                PIECE is called by DRIVER to generate valid moves from 'Move From' for testing 'Move To' input
;                also used by SQATTCK and MPSCAN when the computer 'thinks out its' move
;
;  Registers Modified:  HL , BC , DE , A

PIECE:
		XOR A			    ;Reset Move List index
		LD (buffer_index),A	    ;
		LD A,(HL)		    ;Get piece code
		AND $7F 		    ;Mask out colour bit
		CP $35	     ;$35='Pawn'    ;
		JR Z,PAWN		    ;Handle the Pawns one or two square moves
		LD C,1			    ;
		LD B,8			    ;Can move in any of 8 directions indicated in register B
		LD HL,table_data	    ; HL =  $40E7
		CP $33	     ;$33 = 'Knight';
		JR Z,MOVE		    ;Knights can move in 8 directions
		LD L,$DF		    ; HL =  $40DF
		CP $30	     ;$30 = 'King'  ;
		JR Z,MOVE		    ;
		LD C,B			    ;
		CP $36	     ;$36 = 'Queen' ;
		JR Z,MOVE		    ;
		LD B,4			    ;The rest of the pieces can move in 4 directions
		CP $37	     ;$37 = 'Rook'  ;
		JR Z,MOVE		    ;
		LD L,$E3		    ; HL =  $40E3
		CP $27			    ;
		RET NZ			    ;
		;flows onto MOVE


;  MOVE  $4123  -  Produces a list of legal moves avaliable to the piece under consideration
;
;  Description:  MOVE does not get called from anywhere in program, PAWN flows through to MOVE instead
;
;  Registers Modified:  HL , BC , DE , A

MOVE:
		LD A,E			    ;
					    ;
MV01: 		ADD A,(HL)		    ;
		PUSH AF 		    ;Save registers
		PUSH HL 		    ;
		PUSH BC 		    ;
		CP $3F			    ;
		JR C,MV02		    ;
		CP $94			    ;
		JR NC,MV02		    ;
		CALL STR1		    ;Determine contents of square
		CP 2			    ;
		JR NC,MV02		    ;Branch if A<2
		PUSH AF 		    ;
		CALL ALIST		    ;Add Move to the current legal move list
		POP AF			    ;
		CP 0			    ;
		JR Z,MV02		    ;
		POP BC			    ;
		POP HL			    ;
		LD A,C			    ;
		CP 1			    ;
		JR Z,MV04		    ;
		POP AF			    ;Restore A
		JR MV01 		    ;
					    ;
MV02: 		POP BC			    ;
		POP HL			    ;
MV04: 		POP AF			    ;
MV03: 		INC HL			    ;
		DJNZ MOVE		    ;
		RET 			    ;


;  PAWN  $4151  -  Produces a list of all possible legal moves including initial double moves
;
;  Description:  PAWN is not a subroutine PIECE conditionally takes a relative branch to PAWN
;
;  Registers Modified:  HL , BC , DE , A

PAWN:
		LD A,(HL)		    ;Get Piece data from square
		AND $80 		    ;Mask out piece colour
		LD HL,legal_moves_list	    ;
		JR NZ,PW01		    ;
		LD L,$F1		    ;Move pointer for other colour pieces moving in opposite direction
PW01: 		LD D,3			    ;
					    ;
PW02: 		LD A,E			    ;
PW03: 		ADD A,(HL)		    ;Get Piece data
		PUSH HL 		    ;
		PUSH AF 		    ;
		CP $3F			    ;
		JR C,PW04		    ;If Less than $3F skip
		CP $94			    ;
		JR NC,PW04		    ;if More than $94 skip
		CALL STR1		    ;Determine contents of square
		CP 0			    ;
		JR Z,PW05		    ;
		CP 1			    ;
		JR NZ,PW04		    ;
		LD A,D			    ;
		CP 1			    ;
		JR NZ,PW04		    ;
		CALL ALIST		    ;Add Move to the current legal move list
		LD A,E			    ;
		CP $52			    ;
		JR C,PW06		    ;
		CP $7E			    ;
		JR NC,PW06		    ;
PW04: 		POP AF			    ;
		POP HL			    ;
		DEC HL			    ;
		DEC D			    ;
		JR NZ,PW02		    ;Repeat 3 times
		RET 			    ;
					    ;
PW05: 		LD A,D			    ;
		CP 1			    ;
		CALL NZ,ALIST		    ;Add Move to the current legal move list
		JR PW04 		    ;
PW06: 		POP AF			    ;
PW07: 		POP HL			    ;
		LD E,A			    ;
		JR PW03 		    ;



;  SCORE  $4199  -  Provides a move score based on the following
;
;                       1. The move takes a piece
;                       2. The pieces position is attacked
;                       3. The move position is attacked
;                       4. The move results in a check
;                       5. The move "from" position is defended
;
;  Description: Used by MPSCAN (computers move logic) the current move score is the compared with the previous best
;               and if this is superior the move is saved as the best move so far
;
;  Registers Modified:  HL , BC , DE , A

SCORE:
		PUSH HL 		    ;Save registers
		PUSH BC 		    ;
		PUSH DE 		    ;
		PUSH HL 		    ;
		PUSH BC 		    ;
		LD D,L			    ;Board square index indicates piece to score
		LD HL,move_score-1	    ;
		CALL ROM0724		    ;Call enters near end of a ROM routine to save space
		CALL PSC		    ;Get piece Score Q(5),R(4),B(3),N(2),P(1)
		LD A,B			    ;
		ADD A,H 		    ;
		LD C,A			    ;
		POP AF			    ;
		CALL PSC		    ;Get piece Score
		POP HL			    ;
		CALL INC		    ;Calls SQATTACK to determine whether a square is being attacked
		JR NC,SC01		    ;
		ADD A,B 		    ;
					    ;
SC01: 		LD C,A			    ;
		POP HL			    ;
		POP DE			    ;
		LD E,(HL)		    ;
		LD (HL),D		    ;move position for consideration
		PUSH HL 		    ;
		PUSH DE 		    ;
		CALL INC		    ;Determine whether a square is being attacked
		JR NC,SC02		    ;No square cannot be attacked by opposition
		SUB B			    ;
SC02: 		PUSH AF 		    ;
		CALL CHGMV		    ;Toggle variable used to indicate current players colour
		CALL CHK		    ;Locate players King and store in attack register
		POP BC			    ;
		JR NC,SC03		    ;
		INC B			    ;
		INC B			    ;
SC03: 		POP DE			    ;
		POP HL			    ;
		LD (HL),E		    ;
		POP HL			    ;
		CALL CHG		    ;Change current players colour
		CALL INC		    ;Determine whether a square is being attacked
		JR NC,SC04		    ;
		DEC B			    ;
SC04: 		CALL CHG		    ;Change current players colour
		CALL CHGMV		    ;Toggle variable used to indicate current players colour
		LD A,B			    ;
		LD HL,move_score-5	    ;
		LD (HL),A		    ;
		EX DE,HL		    ;
		LD HL,move_score	    ;
		CP (HL) 		    ;Compare Scores
		RET C			    ;
		LD BC,5 		    ;
		JR SH01 		    ;Continue through 5 pieces



;  SHIFT  $41F2  -  Relocates the move list to a safe location whilst check is being evaluated, then
;     restores the list also used to shift the best move so far up into the move list
;
;  Description: Used by MPSCAN (computers move logic) swaps two blocks of memory this routines saves having a third area of memory for
;                the swap by using the carry flag the subroutine has to be called with Carry Set then Carry Clear to swap back
;
;  Registers Modified:  HL , BC , DE

SHIFT:
		LD HL,move_buffer_alt	    ;Swap buffer
		LD DE,move_buffer	    ;Buffer
		LD BC,$1C		    ;Size of move buffer
		JR C,SH02		    ;Carry flag is used to select which buffer is current
SH01: 		EX DE,HL		    ;Swap buffer pointers
SH02: 		LDIR 			    ;Block memory copy
		RET 			    ;



;  CHK  $4201  -  Locates current movers King and stores the position in the attack register
;
;  Description:  Used by SCORE DRIVER and MPSCAN , Makes use of the Z80's CPIR instruction to search a block of memory for the King
;
;  Registers Modified:  HL , BC , DE , A , B

CHK:
		LD A,(whos_turn)	    ;Start with Movers Colour
		ADD A,$30		    ;Add $30 King Piece with correct colour
		LD HL,chess_board	    ;
		LD B,A			    ;
		CPIR 			    ;Search board for King Piece
		DEC HL			    ;
		LD (attack_register),HL     ;Store position of King in attack register
		;flows onto SQATTACK


;  SQATTACK  $4210  -  Determines whether the opposition can attack the square in the attack register
;
;  Description:  Also Called from INC , Uses the Carry Flag to return a Set 'True' or Clear 'False' result
;
;  Registers Modified:  HL , B , A

SQATTACK: 	LD B,$56		    ;Size of chess board data
		LD HL,chess_board	    ;
					    ;
CK00: 		INC HL			    ;Point to next square on chess board
		PUSH HL 		    ;Save registers BC,HL
		PUSH BC 		    ;
		LD E,L			    ;Use E to hold contents of L
		CALL STR		    ;Determine contents of square A '0' DIFF  '1' EMPTY  '2' WALL  '3' SAME
		CP 0			    ;
		JR NZ,CK02		    ;Square is not a '0' DIFF
		CALL CHGMV		    ;Change to oppositions Colour
		LD L,E			    ;Restore L
		CALL PIECE		    ;
		CALL CHGMV		    ;Restore movers colour
					    ;
CK01: 		CALL TESTLIST		    ;Any moves left in move list
		JR Z,CK02		    ;if ZF = true Then End of list
		LD HL,(attack_register)     ;
		CP L			    ;is it on list ?
		JR NZ,CK01		    ;Continue to test attack register until end of list
					    ;
		POP BC			    ;
		POP HL			    ;
		SCF 			    ;Yes the opposition can attack the square
		RET 			    ;
CK02: 		POP BC			    ;End of Attack Register Search
		POP HL			    ;
		DJNZ CK00		    ;
					    ;
		AND A			    ;No the opposition cannot attack the square
		RET 			    ;



; DRIVER  $423F  -  Main Loop for chess game, uses all the subroutines to provide program control
;
;  Description:  This is the entry point on the original ZX81 game
;
;  Registers Modified:  HL , BC , DE , A
;
;
;  DRIVER
;       |->KYBD -> TPK
;       |  PIECE -> STR1 -> PAWN
;       |            \ALIST
;       -- TESTLIST
;          PMOVE

DRIVER:
		LD B,5			    ;Entry Point
		LD A,8			    ;
;----------------------------------------------------------------------------------------------------------------------------------
	       ;LD HL,inputbuffer           ;THIS COMMENTED LINE HAS BEEN MOVED TO END OF 'HOME' TO KEEP CODE ALIGNED WITH ORIGINAL
 CALL HOME				    ;THIS CALL REMOVES THE ENTERED MOVE FROM THE DISPLAY AND RESETS THE CURSOR POSITION
;----------------------------------------------------------------------------------------------------------------------------------
DR01: 		INC HL			    ;
		LD (HL),A		    ;Clear move buffer
		DJNZ DR01		    ;
;-------------------------------------------;
DR02: 		CALL KYBD		    ;Wait for Player Coords input
		CP 3			    ;Returns A=3 Same, A=2 Outside, A=1 Empty, A=0 Diff, HL is Board Square memory address
		JR NZ,DRIVER		    ;if Entered Square is not '3' Same Colour, Then go back
		LD (move_from),HL	    ;Save Move
		LD E,L			    ;Store offset in E
		CALL PIECE		    ;Scan board and make a list of all avaliable moves for piece
		LD HL,inputbuffer+2	    ;
					    ;
		CALL KYBD		    ;Get 'Move To' position
		CP 2			    ;
		EX DE,HL		    ;Put 'Move To' position in DE
		JR NC,DRIVER		    ;IF A NOT '0' DIFFERENTCOLOUR OR '1' EMPTY THEN DRIVER
					    ;
DR03: 		CALL TESTLIST		    ;Check if move is in the list of possible valid moves
		JR Z,DRIVER		    ;No 'Move' is not in the list
		CP C			    ;
		JR NZ,DR03		    ;Check move list until
		CALL UPDATEANDPMOVE	    ;Change board, 'move_from'(16391) holds start, DE holds 'Move To' position on board
;-------------------------------------------;
DEMOMODE: 				    ;
		EXX 			    ;
		CALL CHK		    ;Locate players King and store in attack register
		EXX 			    ;
		JR C,DR04		    ;Take Players King
		CALL CHGSQ		    ;Clears chess board square and toggles current mover variable
		;COMPUTERS MOVE LOGIC       ;
		CALL MPSCAN		    ;Computer takes turn
DR05: 		JR DRIVER		    ;Computer has moved now branch back to start (Humans turn)
DR04: 					    ;
		LD (HL),B		    ;Remove piece to be moved
		LD A,C			    ;
		LD (DE),A		    ;Put piece on board
		JR DR05 		    ;



;  TESTLIST  $4282  -  Tests to see if there are any moves in the move list
;
;  Description:  Called by DRIVER MPSCAN and SQATTACK decrements List Index variable
;                  then returns result in A , if the list is empty the ZERO flag is set
;
;  Registers Modified:  HL , A

TESTLIST:
		LD HL,buffer_index	    ;First byte of buffer is 0-255 index
		DEC (HL)		    ;Decrement Index
		LD A,(HL)		    ;
		INC A			    ;
		RET Z			    ;Exit subroutine if List is empty
		ADD A,L 		    ;Add offset to HL
		LD L,A			    ;
		LD A,(HL)		    ;Get result in A from list
		RET 			    ;



;  ADDLIST  $428D  -  Adds another entry to the current legal move list
;
;  Description:  Called from MOVE and PAWN on entry C holds the Piece to Add to the 'move list' first
;                 the index is incremented then C is stored top of the list
;
;  Registers Modified:  HL , A

ALIST:
		LD HL,buffer_index	    ;First byte of buffer is 0-255 Index
		INC (HL)		    ;Increment Index
		LD A,(HL)		    ;
		ADD A,L 		    ;Add offset to HL
		LD L,A			    ;
		LD (HL),C		    ;Store C on top of list
		RET 			    ;



;  MPSCAN  $4296  -  Scans the board for computer pieces, using MOVE and SCORE determines best moves
;
;  Description:  Computers move logic Scans every square on chess board for computers pieces and scores each piece in a 'Nested Loop' also
;                 takes care of toggling current player colour to compliment the call from DRIVER before returning back to DRIVER (the main loop)
;
;               Subroutines -  STR ,PIECE ,TESTLIST ,PMOVE ,SHIFT ,CHK ,SCORE and SHIFT
;
;  Registers Modified:  HL , DE , A , B
;
;              MPSCAN
;                   |-> STR -> PAWN
;                   |   PIECE -> STR1 -> PAWN
;                   |             ALIST
;                   | |->TESTLIST
;                   | |  CHK
;                   | |  SCORE -> PIECE
;                   | |            INC -> SQATTACK
;                   |-|--CHGSQ -> CHGMV

MPSCAN:
		XOR A			    ;Reset move score
		LD (move_score),A	    ;
		LD B,$56		    ;Size of Chess board data
		LD HL,chess_board	    ;
					    ;
MS01: 		INC HL			    ;Point to next square on board
		PUSH HL 		    ;
		PUSH BC 		    ;
		LD E,L			    ;
		CALL STR		    ;Check for 'same colour' pieces
		CP 3			    ;
		JR NZ,MS04		    ;If square is 'same colour' Then goto MS04
		LD L,E			    ;
		LD (move_from),HL	    ;
		CALL PIECE		    ;Make a list of possible moves for piece
;-------------------------------------------;
MS02: 		 CALL TESTLIST		    ;Any moves left in move list
		 JR Z,MS04		    ;
		 LD E,A 		    ;
		 LD D,high_addr 	    ;
		 CALL PMOVE		    ;Updates chess board display, Moves piece from HL to DE
		 EXX 			    ;
		 AND A			    ;Clear Carry Flag
		 CALL SHIFT		    ;Save List
		 CALL CHK		    ;Will move take opponents King ? This move takes highest priority
		 EXX 			    ;
		 LD (HL),B		    ;Erase Piece
		 LD A,C 		    ;
		 LD (DE),A		    ;Put Piece on Board at new location
		 JR C,MS03		    ;Take opponents King
		 CALL SCORE		    ;Provide a Move Score
MS03: 		 SCF 			    ;Set Carry Flag for SHIFT
		 CALL SHIFT		    ;Restore Move list
		 JR MS02		    ;
;-------------------------------------------;
MS04: 		POP BC			    ;
		POP HL			    ;
		DJNZ MS01		    ;

MS06: 		LD A,(move_score)	    ;
		CP 0			    ;
					    ;
MS07: 		JR Z,MS07		    ;CHECK MATE END OF GAME (Infinite Loop)
					    ;
		LD HL,move_buffer-1	    ;
		LD A,(HL)		    ;Get piece
		DEC HL			    ;
		DEC HL			    ;
		LD E,(HL)		    ;Index square on board
		LD D,high_addr		    ;Point to chess board
		LD (DE),A		    ;Put piece on Chess board
		DEC HL			    ;
		LD L,(HL)		    ;Get data of moved piece from board
		LD H,D			    ;
		;Flows onto CHGSQ

;  CHGSQ  -  Move pieces on Chess Board and calculate correct colour of newly empty square to clear square
;
;  Description:  Also called from DRIVER for moving other player, Clears chess board square and toggles current mover variable
;
;  Registers Modified:  HL , A

CHGSQ:
		BIT 0,L 		    ;Test for odd or even square obviously the uneven 11 byte wide Chess board on the ZX81 is used to effect here
		LD (HL),$80		    ;Erase with a White square
		JR Z,MS05		    ;
		LD (HL),$0		    ;its a black square
MS05: 		CALL CHGMV		    ;Toggle player colour variable
		RET 			    ;



;  CHGMV  $42F7  -  Toggles the bit used to determine if its black or white players turn to move
;
;  Description:  Called by CHGSQ ,SCORE and SQATTACK Simply toggles bit 7 of a variable to indicate current movers colour
;
;  Registers Modified:  HL , A

CHGMV:
		LD HL,whos_turn 	    ;Point HL to current mover colour variable
CHG: 					    ;
		LD A,(HL)		    ;
		ADD A,$80		    ;Toggle between $0 and $80
		LD (HL),A		    ;
		RET 			    ;



;  PMOVE  $42FF  -  Updates chess board display, Moves piece from HL to DE
;
;  Description:  Used by MPSCAN and DRIVER does the updating of Display RAM , the ZX81 will then redraw the chess board via display handling interrupts
;
;  Registers Modified:  HL , BC , A

PMOVE:
		LD HL,(move_from)	    ;Get 'Move From'
		LD A,(DE)		    ;
		LD C,A			    ;Save Piece at 'Move to' position
		LD A,(HL)		    ;Get Piece code
		LD (HL),0		    ;Remove piece at 'From' square
		LD (DE),A		    ;Store piece at new square
		LD B,A			    ;
		RET 			    ;Returns with Moved Piece in 'A', if a piece is taken it is in 'C'



;  PSC  $430A  -  Gives a score to a chess piece Q(5),R(4),B(3),N(2),P(1)
;
;  Description:  PSC is used by SCORE simply searchs through data consisting of five piece codes Queen, Rook, Bishop, Knight, Pawn
;
;  Registers Modified:  HL , B , A

PSC:
		AND $7F 		    ;Mask out Piece Colour bit
		LD HL, piece_score_data     ;Point HL to table of 5 character codes
		LD B,5			    ;
					    ;
PS01: 		CP (HL) 		    ;Compare against the ZX81 screen display codes
		RET Z			    ;Return with result in 'B' (1 to 5 range) if a Match found
		INC HL			    ;
		DJNZ PS01		    ;Continue searching data
					    ;
		LD A,B			    ;A = 0
		RET 			    ;



;  INC  $4318  -  Determines whether a square is being attacked
;
;  Description:  Switchs to alternative set of registers and calls SQATTACK
;
;  Registers Modified:  A

INC:
		LD A,L			    ;L = board index
		EXX 			    ;Use alternate set of registers for SQATTACK
		LD (attack_register),A	    ;
		CALL SQATTACK		    ;Determine if in attack register
		EXX 			    ;
		LD A,C			    ;Return result A
		RET 			    ;



;  CHESS BOARD  $4318  -  Chess board data
;
;  Description:  A Copy of the ZX81 screen RAM contents
;

chess_board_data:
		DB $76,$00,$02,$02
		DB $00,$E4,$76,$00,$03,$04,$00,$F9,$D4,$3D,$76
		DB $76,$76,$76,$76,$76,$80,$08,$A9,$B7,$AD,$76
		DB $1D,$08,$37,$33,$27,$30,$36,$27,$33,$37,$76
		DB $1E,$08,$35,$35,$35,$35,$00,$35,$35,$35,$76
		DB $1F,$08,$00,$80,$00,$80,$35,$80,$00,$80,$76
		DB $20,$08,$80,$00,$80,$00,$80,$00,$80,$00,$76
		DB $21,$08,$00,$80,$00,$80,$00,$80,$00,$80,$76
		DB $22,$08,$80,$00,$80,$00,$80,$00,$80,$00,$76
		DB $23,$08,$B5,$B5,$B5,$B5,$B5,$B5,$B5,$B5,$76
		DB $24,$08,$B7,$B3,$A7,$B0,$B6,$A7,$B3,$B7,$76
		DB $08,$08,$2D,$2C,$2B,$2A,$29,$28,$27,$26,$76
		DB $08,$08,$08,$08,$08,$76,$76,$76,$76,$76,$76
		DB $76,$76,$76,$76,$7D,$8F,$04,$7E,$00,$00,$80
		DB $00

;<--------------------------------    THE ORIGINAL ZX81 1K CHESS SOURCE CODE ENDS HERE ---------------------------------->

include "emulator_wrapper.asm"

