/*
 * Decompiled with CFR 0.152.
 */
package org.free.j64.cpu;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerArray;
import org.free.j64.app.C64;
import org.free.j64.cpu.Debugger;
import org.free.j64.cpu.Ops;
import org.free.j64.etc.Utils;
import org.free.j64.io.Screen;
import org.free.j64.io.TimeEvent;

public enum CPU {
    CPU;

    public static final int IO_OFFSET = 12288;
    public static final int CHAR_ROM2 = 118784;
    private static final int BASIC_ROM2 = 106496;
    private static final int KERNAL_ROM2 = 122880;
    private static final int NMI_DELAY = 2;
    private static final int IRQ_DELAY = 2;
    public final AtomicIntegerArray memory = new AtomicIntegerArray(131072);
    private final Cpu cpu = new Cpu();
    private Future<?> cpuTask;
    private Future<?> debugTask;
    private final Queue<Object[]> EDQ = new LinkedBlockingQueue<Object[]>();
    private volatile boolean debug;
    private volatile boolean debugging;
    private volatile boolean pause;
    private volatile long baLowUntil;

    public void init() {
        this.loadROM(Utils.getResourceStream(CPU.class, "/META-INF/kernal.c64"), 122880, 8192);
        this.loadROM(Utils.getResourceStream(CPU.class, "/META-INF/basic.c64"), 106496, 8192);
        this.loadROM(Utils.getResourceStream(CPU.class, "/META-INF/chargen.c64"), 118784, 4096);
    }

    private void debug() {
        this.debugging = true;
        final int id = C64.C64.getTaskId();
        this.debugTask = C64.C64.addTask(id, new Callable<Void>(){

            @Override
            public Void call() {
                CPU.this.EDQ.offer(new Object[]{"debugStart"});
                while (!Thread.currentThread().isInterrupted()) {
                    if (CPU.this.pause) continue;
                    CPU.this.EDQ.offer(new Object[]{"debug"});
                    try {
                        TimeUnit.MILLISECONDS.sleep(10L);
                    }
                    catch (InterruptedException ex) {
                        Thread.currentThread().interrupt();
                    }
                }
                C64.C64.removeTask(id);
                CPU.this.debugging = false;
                return null;
            }
        });
    }

    public void start() {
        this.reset();
        int id = C64.C64.getTaskId();
        this.cpu.taskId = id;
        this.cpuTask = C64.C64.addTask(id, this.cpu);
    }

    public void pause() {
        this.pause = !this.pause;
    }

    public void stop() {
        this.cpuTask.cancel(true);
    }

    public void hardReset() {
        this.EDQ.offer(new Object[]{"hardReset"});
    }

    public void reset() {
        this.EDQ.offer(new Object[]{"reset"});
    }

    public void text(String txt) {
        int pos = 0;
        for (int n : txt.toUpperCase().toCharArray()) {
            if (n == 126) {
                n = 13;
            }
            this.memory.set(631 + pos++, n);
            if (pos != 5) continue;
            this.clearMsg(pos);
            pos = 0;
        }
        this.clearMsg(pos);
    }

    public void runBasic() {
        this.memory.set(631, 82);
        this.memory.set(632, 85);
        this.memory.set(633, 78);
        this.memory.set(634, 13);
        this.memory.set(198, 4);
    }

    public void addEvent(TimeEvent t) {
        EventQueue.EQ.add(t);
    }

    public boolean isDebug() {
        return this.debug;
    }

    public void toggleDebug() {
        boolean bl = this.debug = !this.debug;
        if (this.debug) {
            if (!this.debugging) {
                this.debug();
            }
        } else if (this.debugging) {
            this.debugTask.cancel(true);
        }
    }

    public long getCycles() {
        return this.cpu.cycles;
    }

    public int getLastReadOP() {
        return this.cpu.lastReadOP;
    }

    int getSP() {
        return this.cpu.stackp;
    }

    public boolean isBasicROM() {
        return this.cpu.basicROM;
    }

    public boolean isCharROM() {
        return this.cpu.charROM;
    }

    public boolean isIoON() {
        return this.cpu.ioON;
    }

    public boolean isKernalROM() {
        return this.cpu.kernalROM;
    }

    public boolean isIRQLow() {
        return this.cpu.IRQLow;
    }

    public boolean isNMILow() {
        return this.cpu.NMILow;
    }

    public void removeEvent(TimeEvent t) {
        EventQueue.EQ.remove(t);
    }

    public void setBaLowUntil(long b) {
        this.baLowUntil = b;
    }

    public void setIRQLow(boolean low) {
        this.EDQ.offer(new Object[]{"setIRQLow", low});
    }

    public void setNMILow(boolean low) {
        this.EDQ.offer(new Object[]{"setNMILow", low});
    }

    private void clearMsg(int pos) {
        this.memory.set(198, pos);
        int tries = 6;
        while (--tries > 0 && this.memory.get(198) > 0) {
            try {
                TimeUnit.MILLISECONDS.sleep(15L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    private void loadROM(InputStream is, int start, int len) {
        try (BufferedInputStream bis = new BufferedInputStream(is);){
            int t;
            byte[] charBuf = new byte[len];
            int pos = 0;
            while ((t = ((InputStream)bis).read(charBuf, pos, len - pos)) > 0) {
                pos += t;
            }
            int n = charBuf.length;
            for (int i = 0; i < n; ++i) {
                this.memory.set(i + start, charBuf[i] & 0xFF);
            }
        }
        catch (Exception e) {
            System.err.println("Error loading resource" + e);
        }
    }

    private final class Cpu
    implements Callable<Void> {
        private int rindex;
        private int acc;
        private int x;
        private int y;
        private int romFlag = 40960;
        private int interruptInExec;
        private int lastInterrupt;
        private int oldpc;
        private int pc = 64738;
        private int stackp = 255;
        private int lastReadOP;
        private long cycles;
        private long nmiCycleStart;
        private long irqCycleStart;
        private boolean sign;
        private boolean zero;
        private boolean carry;
        private boolean decimal;
        private boolean brk;
        private boolean resetFlag;
        private boolean disableInterupt;
        private boolean overflow;
        private boolean NMILastLow;
        private boolean basicROM;
        private boolean charROM;
        private boolean ioON = true;
        private boolean kernalROM;
        private boolean IRQLow;
        private boolean NMILow;
        private int taskId;

        private Cpu() {
        }

        @Override
        public Void call() {
            try {
                while (!Thread.currentThread().isInterrupted()) {
                    if (!CPU.this.pause) {
                        this.decodeOp();
                    }
                    if (CPU.this.EDQ.isEmpty()) continue;
                    this.dispatch((Object[])CPU.this.EDQ.poll());
                }
            }
            catch (Exception e) {
                System.err.println(Debugger.DEBUGGER.disAssemble(this.rindex, this.acc, this.x, this.y, (byte)this.getStatusByte(), this.interruptInExec, this.lastInterrupt));
            }
            C64.C64.removeTask(this.taskId);
            return null;
        }

        private void dispatch(Object[] m) {
            switch ((String)m[0]) {
                case "debug": {
                    this.debug();
                    break;
                }
                case "debugStart": {
                    this.oldpc = this.pc;
                    break;
                }
                case "hardReset": {
                    this.hardReset();
                    break;
                }
                case "reset": {
                    this.reset();
                    break;
                }
                case "setIRQLow": {
                    this.setIRQLow((Boolean)m[1]);
                    break;
                }
                case "setNMILow": {
                    this.setNMILow((Boolean)m[1]);
                    break;
                }
            }
        }

        private void debug() {
            if (CPU.this.baLowUntil <= this.cycles && this.oldpc != this.pc) {
                this.fixRindex(this.pc);
                System.out.println(Debugger.DEBUGGER.disAssemble(this.rindex, this.acc, this.x, this.y, (byte)this.getStatusByte(), this.interruptInExec, this.lastInterrupt));
            }
            this.oldpc = this.pc;
        }

        private void hardReset() {
            for (int i = 0; i < 65536; ++i) {
                CPU.this.memory.set(i, 0);
            }
            this.reset();
        }

        private void reset() {
            this.writeByte(1, 7);
            this.IRQLow = false;
            this.brk = false;
            this.NMILow = false;
            this.resetFlag = true;
        }

        private void setIRQLow(boolean low) {
            if (!this.IRQLow && low) {
                this.irqCycleStart = this.cycles + 2L;
            }
            this.IRQLow = low;
        }

        private void setNMILow(boolean low) {
            if (!this.NMILow && low) {
                this.nmiCycleStart = this.cycles + 2L;
            }
            this.NMILow = low;
            if (!low) {
                this.NMILastLow = low;
            }
        }

        private void fixRindex(int address) {
            if (this.basicROM && (address & 0xE000) == 40960 || this.kernalROM && (address & 0xE000) == 57344 || this.charROM && (address & 0xF000) == 53248) {
                address |= 0x10000;
            }
            this.rindex = address;
        }

        private void setStatusByte(int status) {
            this.carry = (status & 1) != 0;
            this.zero = (status & 2) != 0;
            this.disableInterupt = (status & 4) != 0;
            this.decimal = (status & 8) != 0;
            this.brk = (status & 0x10) != 0;
            this.overflow = (status & 0x40) != 0;
            this.sign = (status & 0x80) != 0;
        }

        private void setZS(int data) {
            this.zero = data == 0;
            this.sign = data > 127;
        }

        private void setCarry(int data) {
            this.carry = data > 127;
        }

        private void branch(boolean branch, int addr, int cycDiff) {
            if (!branch) {
                return;
            }
            int oldPC = this.pc;
            this.pc = addr;
            if (cycDiff == 1) {
                this.readByte(this.pc);
            } else {
                if (this.pc < oldPC) {
                    this.readByte(this.pc + 256);
                } else {
                    this.readByte(this.pc - 256);
                }
                this.readByte(this.pc);
            }
        }

        private boolean checkInterrupt() {
            if (this.NMILow && !this.NMILastLow && this.cycles >= this.nmiCycleStart) {
                this.lastInterrupt = 1;
                this.doInterrupt(65530, this.getStatusByte() & 0xEF);
                this.disableInterupt = true;
                this.NMILastLow = this.NMILow;
                return true;
            }
            if (this.IRQLow && this.cycles >= this.irqCycleStart || this.brk) {
                if (!this.disableInterupt) {
                    this.lastInterrupt = 2;
                    int status = this.getStatusByte();
                    if (this.brk) {
                        status |= 0x10;
                        ++this.pc;
                    } else {
                        status &= 0xEF;
                    }
                    this.doInterrupt(65534, status);
                    this.disableInterupt = true;
                    this.brk = false;
                    return true;
                }
                this.brk = false;
            } else if (this.resetFlag) {
                this.doReset();
            }
            return false;
        }

        private void doInterrupt(int addr, int status) {
            this.readByte(this.pc);
            this.readByte(this.pc + 1);
            this.push((this.pc & 0xFF00) >> 8);
            this.push(this.pc & 0xFF);
            this.push(status);
            ++this.interruptInExec;
            this.pc = (this.readByte(addr + 1) << 8) + this.readByte(addr);
        }

        private void decodeOp() {
            if (this.checkInterrupt()) {
                return;
            }
            int data = Ops.INSTRUCTION_SET[this.readByte(this.pc++)];
            int op = data & Ops.Op.OP_MASK.getCode();
            int addrMode = data & Ops.Op.ADDRESSING_MASK.getCode();
            int address = 0;
            int tmp = 0;
            boolean read = (data & Ops.Op.READ.getCode()) != 0;
            boolean write = (data & Ops.Op.WRITE.getCode()) != 0;
            this.lastReadOP = this.rindex;
            int p1 = this.readByte(this.pc++);
            switch (Ops.getOp(addrMode)) {
                case IMMEDIATE: {
                    data = p1;
                    break;
                }
                case ABSOLUTE: {
                    address = (this.readByte(this.pc++) << 8) + p1;
                    if (!read) break;
                    data = this.readByte(address);
                    break;
                }
                case ZERO: {
                    address = p1;
                    if (!read) break;
                    data = this.readByte(address);
                    break;
                }
                case ZERO_X: 
                case ZERO_Y: {
                    this.readByte(p1);
                    address = addrMode == Ops.Op.ZERO_X.getCode() ? p1 + this.x & 0xFF : p1 + this.y & 0xFF;
                    if (!read) break;
                    data = this.readByte(address);
                    break;
                }
                case ABSOLUTE_X: 
                case ABSOLUTE_Y: {
                    address = this.readByte(this.pc++) << 8;
                    p1 = addrMode == Ops.Op.ABSOLUTE_X.getCode() ? (p1 += this.x) : (p1 += this.y);
                    data = this.readByte(address + (p1 & 0xFF));
                    address += p1;
                    if (!read || p1 <= 255 && !write) break;
                    data = this.readByte(address);
                    break;
                }
                case RELATIVE: {
                    address = this.pc + (byte)p1;
                    if (((address ^ this.pc) & 0xFF00) > 0) {
                        tmp = 2;
                        break;
                    }
                    tmp = 1;
                    break;
                }
                case ACCUMULATOR: {
                    --this.pc;
                    data = this.acc;
                    write = false;
                    break;
                }
                case INDIRECT_X: {
                    this.readByte(p1);
                    tmp = p1 + this.x & 0xFF;
                    address = this.readByte(tmp + 1) << 8;
                    address |= this.readByte(tmp);
                    if (!read) break;
                    data = this.readByte(address);
                    break;
                }
                case INDIRECT_Y: {
                    address = this.readByte(p1 + 1) << 8;
                    p1 = this.readByte(p1) + this.y;
                    data = this.readByte(address + (p1 & 0xFF));
                    address += p1;
                    if (!read || p1 <= 255 && !write) break;
                    data = this.readByte(address);
                    break;
                }
                case INDIRECT: {
                    address = (this.readByte(this.pc) << 8) + p1;
                    tmp = address & 0xFFF00 | address + 1 & 0xFF;
                    address = this.readByte(address) + (this.readByte(tmp) << 8);
                    break;
                }
                default: {
                    --this.pc;
                }
            }
            if (read && write) {
                this.writeByte(address, data);
            }
            switch (Ops.getOp(op)) {
                case BRK: {
                    this.brk = true;
                    break;
                }
                case AND: {
                    this.acc &= data;
                    this.setZS(this.acc);
                    break;
                }
                case ADC: {
                    this.opADCimp(data);
                    break;
                }
                case SBC: {
                    this.opSBCimp(data);
                    break;
                }
                case ORA: {
                    this.acc |= data;
                    this.setZS(this.acc);
                    break;
                }
                case EOR: {
                    this.acc ^= data;
                    this.setZS(this.acc);
                    break;
                }
                case BIT: {
                    this.sign = data > 127;
                    this.overflow = (data & 0x40) > 0;
                    this.zero = (this.acc & data) == 0;
                    break;
                }
                case LSR: {
                    this.carry = (data & 1) != 0;
                    this.zero = (data >>= 1) == 0;
                    this.sign = false;
                    break;
                }
                case ROL: {
                    data = (data << 1) + (this.carry ? 1 : 0);
                    this.carry = (data & 0x100) != 0;
                    this.setZS(data &= 0xFF);
                    break;
                }
                case ROR: {
                    boolean nxtcarry = (data & 1) != 0;
                    data = (data >> 1) + (this.carry ? 128 : 0);
                    this.carry = nxtcarry;
                    this.setZS(data);
                    break;
                }
                case TXA: {
                    this.acc = this.x;
                    this.setZS(this.acc);
                    break;
                }
                case TAX: {
                    this.x = this.acc;
                    this.setZS(this.x);
                    break;
                }
                case TYA: {
                    this.acc = this.y;
                    this.setZS(this.acc);
                    break;
                }
                case TAY: {
                    this.y = this.acc;
                    this.setZS(this.y);
                    break;
                }
                case TSX: {
                    this.x = this.stackp;
                    this.setZS(this.x);
                    break;
                }
                case TXS: {
                    this.stackp = this.x & 0xFF;
                    break;
                }
                case DEC: {
                    data = data - 1 & 0xFF;
                    this.setZS(data);
                    break;
                }
                case INC: {
                    data = data + 1 & 0xFF;
                    this.setZS(data);
                    break;
                }
                case INX: {
                    this.x = this.x + 1 & 0xFF;
                    this.setZS(this.x);
                    break;
                }
                case DEX: {
                    this.x = this.x - 1 & 0xFF;
                    this.setZS(this.x);
                    break;
                }
                case INY: {
                    this.y = this.y + 1 & 0xFF;
                    this.setZS(this.y);
                    break;
                }
                case DEY: {
                    this.y = this.y - 1 & 0xFF;
                    this.setZS(this.y);
                    break;
                }
                case JSR: {
                    ++this.pc;
                    address = (this.readByte(this.pc) << 8) + p1;
                    this.readByte(this.stackp | 0x100);
                    this.push((this.pc & 0xFF00) >> 8);
                    this.push(this.pc & 0xFF);
                    this.pc = address;
                    break;
                }
                case JMP: {
                    this.pc = address;
                    break;
                }
                case RTS: {
                    this.readByte(this.stackp | 0x100);
                    this.pc = this.pop() + (this.pop() << 8);
                    ++this.pc;
                    this.readByte(this.pc);
                    break;
                }
                case RTI: {
                    this.readByte(this.stackp | 0x100);
                    tmp = this.pop();
                    this.setStatusByte(tmp);
                    this.pc = this.pop() + (this.pop() << 8);
                    this.brk = false;
                    --this.interruptInExec;
                    break;
                }
                case TRP: {
                    break;
                }
                case NOP: {
                    break;
                }
                case ASL: {
                    this.setCarry(data);
                    data = data << 1 & 0xFF;
                    this.setZS(data);
                    break;
                }
                case PHA: {
                    this.push(this.acc);
                    break;
                }
                case PLA: {
                    this.readByte(this.stackp | 0x100);
                    this.acc = this.pop();
                    this.setZS(this.acc);
                    break;
                }
                case PHP: {
                    this.brk = true;
                    this.push(this.getStatusByte());
                    this.brk = false;
                    break;
                }
                case PLP: {
                    tmp = this.pop();
                    this.setStatusByte(tmp);
                    this.brk = false;
                    break;
                }
                case ANC: {
                    this.acc &= data;
                    this.setZS(this.acc);
                    this.carry = (this.acc & 0x80) != 0;
                    break;
                }
                case CMP: {
                    data = this.acc - data;
                    this.carry = data >= 0;
                    this.setZS(data & 0xFF);
                    break;
                }
                case CPX: {
                    data = this.x - data;
                    this.carry = data >= 0;
                    this.setZS(data & 0xFF);
                    break;
                }
                case CPY: {
                    data = this.y - data;
                    this.carry = data >= 0;
                    this.setZS(data & 0xFF);
                    break;
                }
                case BCC: {
                    this.branch(!this.carry, address, tmp);
                    break;
                }
                case BCS: {
                    this.branch(this.carry, address, tmp);
                    break;
                }
                case BEQ: {
                    this.branch(this.zero, address, tmp);
                    break;
                }
                case BNE: {
                    this.branch(!this.zero, address, tmp);
                    break;
                }
                case BVC: {
                    this.branch(!this.overflow, address, tmp);
                    break;
                }
                case BVS: {
                    this.branch(this.overflow, address, tmp);
                    break;
                }
                case BPL: {
                    this.branch(!this.sign, address, tmp);
                    break;
                }
                case BMI: {
                    this.branch(this.sign, address, tmp);
                    break;
                }
                case CLC: {
                    this.carry = false;
                    break;
                }
                case SEC: {
                    this.carry = true;
                    break;
                }
                case CLD: {
                    this.decimal = false;
                    break;
                }
                case SED: {
                    this.decimal = true;
                    break;
                }
                case CLV: {
                    this.overflow = false;
                    break;
                }
                case SEI: {
                    this.disableInterupt = true;
                    break;
                }
                case CLI: {
                    this.disableInterupt = false;
                    break;
                }
                case LDA: {
                    this.acc = data;
                    this.setZS(data);
                    break;
                }
                case LDX: {
                    this.x = data;
                    this.setZS(data);
                    break;
                }
                case LDY: {
                    this.y = data;
                    this.setZS(data);
                    break;
                }
                case STA: {
                    data = this.acc;
                    break;
                }
                case STX: {
                    data = this.x;
                    break;
                }
                case STY: {
                    data = this.y;
                    break;
                }
                case ANE: {
                    this.acc = p1 & this.x & (this.acc | 0xEE);
                    this.setZS(this.acc);
                    break;
                }
                case ARR: {
                    tmp = p1 & this.acc;
                    int n = this.acc = this.carry ? tmp >> 1 | 0x80 : tmp >> 1;
                    if (!this.decimal) {
                        this.setZS(this.acc);
                        this.carry = (this.acc & 0x40) != 0;
                        this.overflow = (this.acc & 0x40 ^ (this.acc & 0x20) << 1) != 0;
                        break;
                    }
                    this.sign = this.carry;
                    this.zero = this.acc == 0;
                    boolean bl = this.overflow = ((tmp ^ this.acc) & 0x40) != 0;
                    if ((tmp & 0xF) + (tmp & 1) > 5) {
                        this.acc = this.acc & 0xF0 | this.acc + 6 & 0xF;
                    }
                    if (!(this.carry = (tmp + (tmp & 0x10) & 0x1F0) > 80)) break;
                    this.acc += 96;
                    break;
                }
                case ASR: {
                    this.acc &= data;
                    boolean nxtcarry = (this.acc & 1) != 0;
                    this.acc >>= 1;
                    this.carry = nxtcarry;
                    this.setZS(this.acc);
                    break;
                }
                case DCP: {
                    data = data - 1 & 0xFF;
                    this.setZS(data);
                    tmp = this.acc - data;
                    this.carry = tmp >= 0;
                    this.setZS(tmp & 0xFF);
                    break;
                }
                case ISB: {
                    data = data + 1 & 0xFF;
                    this.opSBCimp(data);
                    break;
                }
                case LAX: {
                    this.acc = this.x = data;
                    this.setZS(this.acc);
                    break;
                }
                case LAS: {
                    this.x = this.stackp = data & this.stackp;
                    this.acc = this.stackp;
                    this.setZS(this.acc);
                    break;
                }
                case LXA: {
                    this.x = this.acc = (this.acc | 0xEE) & p1;
                    this.setZS(this.acc);
                    break;
                }
                case RLA: {
                    data = (data << 1) + (this.carry ? 1 : 0);
                    this.carry = (data & 0x100) != 0;
                    this.acc &= (data &= 0xFF);
                    this.zero = this.acc == 0;
                    this.sign = this.acc > 127;
                    break;
                }
                case RRA: {
                    boolean nxtcarry = (data & 1) != 0;
                    data = (data >> 1) + (this.carry ? 128 : 0);
                    this.carry = nxtcarry;
                    this.opADCimp(data);
                    break;
                }
                case SBX: {
                    this.x = (this.acc & this.x) - p1;
                    this.carry = this.x >= 0;
                    this.x &= 0xFF;
                    this.setZS(this.x);
                    break;
                }
                case SHA: {
                    data = this.acc & this.x & (address >> 8) + 1;
                    break;
                }
                case SHS: {
                    data = this.acc & this.x & (address >> 8) + 1;
                    this.stackp = this.acc & this.x;
                    break;
                }
                case SHX: {
                    data = this.x & (address >> 8) + 1;
                    break;
                }
                case SHY: {
                    data = this.y & (address >> 8) + 1;
                    break;
                }
                case SAX: {
                    data = this.acc & this.x;
                    break;
                }
                case SRE: {
                    this.carry = (data & 1) != 0;
                    this.acc ^= (data >>= 1);
                    this.setZS(this.acc);
                    break;
                }
                case SLO: {
                    this.setCarry(data);
                    data = data << 1 & 0xFF;
                    this.acc |= data;
                    this.setZS(this.acc);
                    break;
                }
                default: {
                    System.err.format("Unknown instruction: %d%n", op);
                }
            }
            if (write) {
                this.writeByte(address, data);
            } else if (addrMode == Ops.Op.ACCUMULATOR.getCode()) {
                this.acc = data;
            }
        }

        private void doReset() {
            this.resetFlag = false;
            this.IRQLow = false;
            this.NMILastLow = false;
            this.NMILow = false;
            this.disableInterupt = false;
            this.brk = false;
            this.decimal = false;
            this.carry = false;
            this.overflow = false;
            this.zero = false;
            this.sign = false;
            this.rindex = 0;
            this.interruptInExec = 0;
            EventQueue.EQ.clear();
            Screen.Renderer.RENDERER.getRenderer().reset();
            this.pc = this.readByte(65532) + (this.readByte(65533) << 8);
        }

        private int pop() {
            this.stackp = this.stackp + 1 & 0xFF;
            return this.readByte(this.stackp | 0x100);
        }

        private void push(int data) {
            this.writeByte(this.stackp & 0xFF | 0x100, data);
            this.stackp = this.stackp - 1 & 0xFF;
        }

        private void schedule(long cycles) {
            Screen.Renderer.RENDERER.getRenderer().clock(cycles);
            while (cycles >= EventQueue.EQ.getNextTime()) {
                TimeEvent t = EventQueue.EQ.poll();
                if (t != null) {
                    t.execute(cycles);
                    continue;
                }
                return;
            }
        }

        private int readByte(int addr) {
            this.schedule(++this.cycles);
            while (CPU.this.baLowUntil > this.cycles) {
                this.schedule(++this.cycles);
            }
            this.rindex = addr;
            if ((this.romFlag & addr) == this.romFlag) {
                this.rindex |= 0x10000;
                return CPU.this.memory.get(this.rindex);
            }
            if ((addr & 0xF000) == 53248) {
                if (this.ioON) {
                    return Screen.Renderer.RENDERER.getRenderer().performRead(this.rindex, this.cycles);
                }
                if (this.charROM) {
                    this.rindex |= 0x10000;
                    return CPU.this.memory.get(this.rindex);
                }
            }
            return CPU.this.memory.get(this.rindex);
        }

        private void writeByte(int addr, int data) {
            this.schedule(++this.cycles);
            if (addr <= 1) {
                CPU.this.memory.set(addr, data);
                int p = CPU.this.memory.get(0) ^ 0xFF | CPU.this.memory.get(1);
                this.kernalROM = (p & 2) == 2;
                this.basicROM = (p & 3) == 3;
                this.charROM = (p & 3) != 0 && (p & 4) == 0;
                boolean bl = this.ioON = (p & 3) != 0 && (p & 4) != 0;
                this.romFlag = this.basicROM ? 40960 : (this.kernalROM ? 57344 : 65536);
            }
            if (this.ioON && ((addr &= 0xFFFF) & 0xF000) == 53248) {
                Screen.Renderer.RENDERER.getRenderer().performWrite(addr, data, this.cycles);
            } else {
                CPU.this.memory.set(addr, data);
            }
        }

        private int getStatusByte() {
            return (this.carry ? 1 : 0) + (this.zero ? 2 : 0) + (this.disableInterupt ? 4 : 0) + (this.decimal ? 8 : 0) + (this.brk ? 16 : 0) + 32 + (this.overflow ? 64 : 0) + (this.sign ? 128 : 0);
        }

        private void opADCimp(int data) {
            int tmp = data + this.acc + (this.carry ? 1 : 0);
            boolean bl = this.zero = (tmp & 0xFF) == 0;
            if (this.decimal) {
                tmp = (this.acc & 0xF) + (data & 0xF) + (this.carry ? 1 : 0);
                if (tmp > 9) {
                    tmp += 6;
                }
                if ((tmp = (tmp & 0xF) + (this.acc & 0xF0) + (data & 0xF0)) > 15) {
                    tmp += 16;
                }
                this.overflow = ((this.acc ^ data) & 0x80) == 0 && ((this.acc ^ tmp) & 0x80) != 0;
                boolean bl2 = this.sign = (tmp & 0x80) > 0;
                if ((tmp & 0x1F0) > 144) {
                    tmp += 96;
                }
                this.carry = tmp > 153;
            } else {
                this.overflow = ((this.acc ^ data) & 0x80) == 0 && ((this.acc ^ tmp) & 0x80) != 0;
                this.carry = tmp > 255;
                this.sign = (tmp & 0x80) > 0;
            }
            this.acc = tmp & 0xFF;
        }

        private void opSBCimp(int data) {
            int tmp = this.acc - data - (this.carry ? 0 : 1);
            boolean nxtcarry = tmp >= 0;
            this.sign = ((tmp &= 0x1FF) & 0x80) == 128;
            this.zero = (tmp & 0xFF) == 0;
            boolean bl = this.overflow = ((this.acc ^ tmp) & 0x80) != 0 && ((this.acc ^ data) & 0x80) != 0;
            if (this.decimal) {
                tmp = (this.acc & 0xF) - (data & 0xF) - (this.carry ? 0 : 1);
                if (((tmp = (tmp & 0x10) > 0 ? tmp - 6 & 0xF | (this.acc & 0xF0) - (data & 0xF0) - 16 : tmp & 0xF | (this.acc & 0xF0) - (data & 0xF0)) & 0x100) > 0) {
                    tmp -= 96;
                }
            }
            this.acc = tmp & 0xFF;
            this.carry = nxtcarry;
        }
    }

    private static enum EventQueue {
        EQ;

        private final Queue<TimeEvent> queue = new PriorityBlockingQueue<TimeEvent>();

        private void add(TimeEvent event) {
            if (event == null) {
                return;
            }
            if (event.isScheduled()) {
                this.queue.remove(event);
            }
            event.setScheduled(true);
            this.queue.add(event);
        }

        private void remove(TimeEvent event) {
            if (event == null) {
                return;
            }
            this.queue.remove(event);
        }

        private TimeEvent poll() {
            TimeEvent e = this.queue.poll();
            if (e != null) {
                e.setScheduled(false);
            }
            return e;
        }

        private void clear() {
            for (TimeEvent e : this.queue) {
                e.setScheduled(false);
            }
            this.queue.clear();
        }

        private long getNextTime() {
            if (!this.queue.isEmpty()) {
                return this.queue.peek().getTime();
            }
            return 0L;
        }
    }
}

