#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include "host.h"
#include "shinobi.h"
#include "main.h"
#include "mz80.h"

int SAMPLERATE=22050;
#define SIZEBUFFER 8192
#define MAX_SLICE SAMPLERATE/30
char samples_path[80] = "";
extern zsmp_path[];

INT8 sound_card = 0;

extern UINT8 pf_state;
extern UINT8 sound_type, sampled_sound_support;
extern UINT8 type_sound_subboard;
UINT8 bgm_start = 0x00, bgm_end = 0x00;
UINT8 sfx_start = 0x00, sfx_end = 0x00;
UINT8 vce_start = 0x00, vce_end = 0x00;
UINT8 current_bgm;
UINT8 bgm_flags[256];

#define FM0_CHANNEL 0
#define FM1_CHANNEL 1
#define FM2_CHANNEL 2
#define FM3_CHANNEL 3
#define VOICE_CHANNEL 4
#define MIN_SFX_CHANNEL 5
#define MAX_SFX_CHANNEL 9
#define MIN_PCM_CHANNEL 10
#define MAX_PCM_CHANNEL 25
#define NUMBER_OF_CHANNELS 26

static UINT8 sample_ok[256];
extern UINT8 sample_loop[256];

#include "dosaudio.c"

static WaveBuffer *Samples[256];
static AudioChannel Channels[NUMBER_OF_CHANNELS];

#define SIZE_VOICE_QUEUE 50
static int current_voice_queue_write_index, current_voice_queue_read_index;
static int voice_sample_to_play_queue[SIZE_VOICE_QUEUE];
static int current_sfx_channel;
static int sound_effect_sample_to_play[MAX_SFX_CHANNEL-MIN_SFX_CHANNEL+1];

static WaveBuffer wave_stream[4];

void (*WriteFMReg)(int,int,int);
static void (*UpdateOPM)(int,int);
static void (*poll_fm_emu)(void);
static void (*poll_pcm_emu)(void);

extern FILE *LOG;
extern UINT8 LOG_ACTIVE;

static int indices[16] = {
   -1,-1,-1,-1,2,4,6,8,-1,-1,-1,-1,2,4,6,8
};

static int step_size[32] = {
   1166,1282,1411,1552,1707,1878,2066,2272,2499,2749,3024,3327,
   3660,4026,4428,4871,5358,5894,6484,7132,7845,8630,9493,10442,
   11487,12635,13899,15289,16818,18500,20350,22385
};

static int samples_adr[14] =  {
   0,0x1570,0x1D50,0x25D0,0x2B00,0x2FA0,0x3460,0x4000,0x4670,0x49A0,0x4CDE,
   0x57A0,0x5BD0,0x7418
};

static int step=0,salida=1,ssndx=0;
static unsigned char delta=0,signo=0;

short int unpack_adpcm(UINT8 ax)
{
   int ebx;
   ax&=0x0F;
   ssndx+=indices[ax];
   if (ssndx>=31) ssndx=31;
   else if (ssndx<=0) ssndx=0;
   delta=signo=ax;
   signo&=0x8;
   delta&=0x7;
   ebx=0;
   if (delta&0x4) ebx+=(step<<2);
   if (delta&0x2) ebx+=(step<<1);
   if (delta&0x1) ebx+=step;
   ebx=ebx>>2;
   if (signo) salida-=ebx; else salida+=ebx;
   if (salida>32767) salida=32767;
   else if (salida<-32768) salida=-32768;
   step=step_size[ssndx];
   return(salida&0xFFFF);
}

int decoder(UINT8 *buf_in, INT16 *buf_out, int start, int end)
{
   unsigned char ah, al;
   int iw=0;
   signo=delta=step=salida=0;
   ssndx=1;
   step=step_size[0];
   if (buf_in[start]==0xFF) start++;
   do {
      al=buf_in[start++];
      ah=al&0xF;
      al=al>>4;
      buf_out[iw++]=unpack_adpcm(al);
      buf_out[iw++]=unpack_adpcm(ah);
   } while (start<end);
   return(iw*2);
}

extern UINT8 Z80_active, ym2151_active, Jarek2151, ym2203_active, ym3438_active;
extern UINT8 uPD7759_active, uPD7751_active;

void _YM2203WriteReg(int b, int r, int v)
{
  if (r<0x10) {
    AYWriteReg(b, r, v);
  } else {
    if (r<=0x20) return;
    if (r<0x30) {
      if (r==0x2d || r==0x2e || r==0x2f) { // set pre-scaler
        static UINT8 HadSetPreScaler=0;
        if (!HadSetPreScaler) HadSetPreScaler=r;
        else {
          return; // never set pre-scaler twice
        }
      }
    }
    YM2203Write(b, r, v);
  }
}

static char fm0_buffer[SIZEBUFFER];
static char fm1_buffer[SIZEBUFFER];
static char fm2_buffer[SIZEBUFFER];
static char fm3_buffer[SIZEBUFFER];
static int init_ym2151()
{
  char *buffer[3];
  int i;
  for (i=0;i!=3;i++) buffer[i] = fm0_buffer;
  if (Jarek2151) {
    if (YMInit(1, SAMPLERATE, SIZEBUFFER, buffer) == 0) {
      for (i = 0; i < 1; i++) YMSetIrqHandler(i, NULL);
    }
  } else {
    if (OPMInit(1, 4050000/*3582071*/, SAMPLERATE, 8, sizeof(fm0_buffer), buffer) == 0) {
//      for (i = 0; i < 1; i++) OPMSetIrqHandler(i, NULL);
    }
  }
  if (init_wave_stream(&wave_stream[0], 8) != 0) return -1;
  init_wavestream_pointers();
  
  return 0;
}

static int init_ym2203()
{
  char *buffer[3];
  char *buffer2[3];
  int i;
  for (i=0;i!=3;i++) buffer[i] = fm0_buffer;
  for (i=0;i!=3;i++) buffer2[i] = fm1_buffer;
  if (YM2203Init(1, 4000000/*3582071*/, SAMPLERATE, 8, sizeof(fm0_buffer), buffer) == 0) {
//    for (i = 0; i < 1; i++) OPNSetIrqHandler(i, NULL);
    if (init_wave_stream(&wave_stream[0], 8) != 0) return -1;
  } else return -1;
  if (AYInit(1, 4000000/2, SAMPLERATE, sizeof(fm1_buffer), buffer2) == 0) {
    if (init_wave_stream(&wave_stream[1], 8) !=0) return -1;
  } else return -1;
  init_wavestream_pointers();
  return 0;
}

static int init_ym3438()
{
  char *buffer[4];
  int i;

  buffer[0] = fm0_buffer;
  buffer[1] = fm1_buffer;
  buffer[2] = fm2_buffer;
  buffer[3] = fm3_buffer;
  
  if (YM2203Init(4, 2000000/*3582071/2*/, SAMPLERATE, 8, sizeof(fm0_buffer), buffer) == 0) {
    for (i = 0; i < 4; i++) {
//      OPNSetIrqHandler(i, NULL);
      if (init_wave_stream(&wave_stream[i], 8) != 0) return -1;
    }
  } else return -1;
  init_wavestream_pointers();
  return 0;
}

///////////////////////////////////////////////////////////////////
static struct {
  UINT32 pos0, pos1;
  int remainder_size;
  int cur_buf;
} wavestream_pointers;

int init_wavestream_pointers()
{
  wavestream_pointers.pos0 = 0;
  wavestream_pointers.remainder_size = 0;
  wavestream_pointers.cur_buf = -1;
  return 0;
}

// returns -1 for the 1st time, returns 0 afterwards.
int get_wavestream_pointers(int *cur_buf, int *cur_size)
{
  if (wavestream_pointers.cur_buf == -1) { // first-time
    wavestream_pointers.cur_buf = SAMPLERATE/20; // 1/20 sec delay for preventing sound sample over-playing
    *cur_buf = wavestream_pointers.cur_buf;
    *cur_size = 0;
    return -1;
  }
  // from 2nd time
  GetPlayPosition(Channels[FM0_CHANNEL], &wavestream_pointers.pos1); // pos1*=2;

  if (wavestream_pointers.pos1 >= wavestream_pointers.pos0) 
    *cur_size = wavestream_pointers.pos1 - wavestream_pointers.pos0;
  else 
    *cur_size = SIZEBUFFER - wavestream_pointers.pos0 + wavestream_pointers.pos1;

  *cur_size += wavestream_pointers.remainder_size;
  wavestream_pointers.remainder_size = 0;
  if (*cur_size > MAX_SLICE){
    wavestream_pointers.remainder_size += *cur_size - MAX_SLICE;
    *cur_size = MAX_SLICE;
  }
  *cur_buf = wavestream_pointers.cur_buf;
//printf("s=%d cb=%d p0=%d p1=%d\n", *cur_size, *cur_buf, wavestream_pointers.pos0, wavestream_pointers.pos1);
  wavestream_pointers.pos0 = wavestream_pointers.pos1;
  
  // advance sample buffer pointer
  wavestream_pointers.cur_buf += *cur_size; // calculates buffer offset for the next time
  if (wavestream_pointers.cur_buf >= SIZEBUFFER) wavestream_pointers.cur_buf -= SIZEBUFFER;
}

// re-sync buffer write/read pointers : must be called after a long time pause
void reset_wavestream_pointers()
{
  GetPlayPosition(Channels[FM0_CHANNEL], &wavestream_pointers.pos1); // pos1*=2;
  wavestream_pointers.pos0 = wavestream_pointers.pos1;
  wavestream_pointers.cur_buf = wavestream_pointers.pos1 + SAMPLERATE/20;
  if (wavestream_pointers.cur_buf >= SIZEBUFFER) wavestream_pointers.cur_buf -= SIZEBUFFER;
}
///////////////////////////////////////////////////////////////////

static void poll_ym2151()
{
  int cur_buf, cur_size;

  if (get_wavestream_pointers(&cur_buf, &cur_size) == -1) {
    PlayWaveStream(&Channels[FM0_CHANNEL], &wave_stream[0]);
  }

  UpdateOPM(0, cur_size);
  WriteToWaveBuffer(&wave_stream[0], cur_buf, fm0_buffer, cur_size);
}

static void poll_ym2203()
{
  int cur_buf, cur_size;

  if (get_wavestream_pointers(&cur_buf, &cur_size) == -1) {
    PlayWaveStream(&Channels[FM0_CHANNEL], &wave_stream[0]);
    PlayWaveStream(&Channels[FM1_CHANNEL], &wave_stream[1]);
    SetVol(Channels[FM1_CHANNEL], 56); // set PSG output volume a bit lower
  }

  YM2203UpdateOne(0, cur_size);
  AYUpdateOne(0, cur_size);
  WriteToWaveBuffer(&wave_stream[0], cur_buf, fm0_buffer, cur_size);
  WriteToWaveBuffer(&wave_stream[1], cur_buf, fm1_buffer, cur_size);
}

// two ym3438 chips
static void poll_ym3438()
{
  int cur_buf, cur_size;

  if (get_wavestream_pointers(&cur_buf, &cur_size) == -1) {
    PlayWaveStream(&Channels[FM0_CHANNEL], &wave_stream[0]);
    PlayWaveStream(&Channels[FM1_CHANNEL], &wave_stream[1]);
    PlayWaveStream(&Channels[FM2_CHANNEL], &wave_stream[2]);
    PlayWaveStream(&Channels[FM3_CHANNEL], &wave_stream[3]);
  }

  YM2203UpdateOne(0, cur_size);
  YM2203UpdateOne(1, cur_size);
  YM2203UpdateOne(2, cur_size);
  YM2203UpdateOne(3, cur_size);
  WriteToWaveBuffer(&wave_stream[0], cur_buf, fm0_buffer, cur_size);
  WriteToWaveBuffer(&wave_stream[1], cur_buf, fm1_buffer, cur_size);
  WriteToWaveBuffer(&wave_stream[2], cur_buf, fm2_buffer, cur_size);
  WriteToWaveBuffer(&wave_stream[3], cur_buf, fm3_buffer, cur_size);
}

static int expanded_rate; // 44 for 44100Hz or 22 for 22050Hz
static int fixed_expanded_sample_rate; // 44100 or 22050
#define expanded_sample_len(size)   (expanded_rate*(size)/8)

inline static INT8 translate_one_sample(INT8 sample, int version)
{
  if (version == 0) {
    // 00~7f -> 0~7f
    // ff~80 -> 80~ff
    if ((UINT8)sample == 0xff) sample = 0;
    else if (sample < 0) {
      sample = ((UINT8)sample&0x7f);
      sample = -sample;
    } else {

    }
  } else if (version == 1) {
    // 80~ff -> 0~7f
    // 00~7f -> 80~ff
    if (sample < 0) {
      sample = ((UINT8)sample&0x7f);
      if (sample < 5) sample = 0;
    } else {
      sample |= 0x80;
      if (sample > -5) sample = 0;
    }
  }
  return sample;
}

static void translate_6000Hz_to_22050Hz(INT8 *buf, INT8 *out, int size, int version)
{
  UINT8 cnt;
  double step;
  INT8 last_sample = 0;
  INT8 cur_sample;

  // 4 4 3 sequence for 6000->22050
  // using linear interpolation
  cnt = 0;
  while (size--) {
    cur_sample = translate_one_sample(*buf, version);

    if (cnt++ < 2) {
      step = (cur_sample - last_sample)/4.0;
      out[0] = rint(last_sample + step);
      out[1] = rint(last_sample + step*2);
      out[2] = rint(last_sample + step*3);
      out[3] = cur_sample;
      out += 4;
    } else {
      step = (cur_sample - last_sample)/3.0;
      out[0] = rint(last_sample + step);
      out[1] = rint(last_sample + step*2);
      out[2] = cur_sample;
      out += 3;
      cnt = 0;
    }
    last_sample = cur_sample;
    
    *buf++ = cur_sample;
  }
}


static void translate_8000Hz_to_22050Hz(INT8 *buf, INT8 *out, int size, int version)
{
  UINT8 cnt;
  double step;
  INT8 last_sample = 0;
  INT8 cur_sample;

  // 2 3 3 3 sequence for 8000->22050
  // using linear interpolation
  cnt = 0;
  while (size--) {
    cur_sample = translate_one_sample(*buf, version);

    if ((cnt++ & 3)) {
      step = (cur_sample - last_sample)/3.0;
      out[0] = rint(last_sample + step);
      out[1] = rint(last_sample + step*2);
      out[2] = cur_sample;
      out += 3;
    } else {
      step = (cur_sample - last_sample)/2.0;
      out[0] = rint(last_sample + step);
      out[1] = cur_sample;
      out += 2;
    }
    last_sample = cur_sample;
    
    *buf++ = cur_sample;
  }
}

static void translate_8000Hz_to_22050Hz_plain(INT8 *buf, INT8 *out, int size, int version)
{
  UINT8 cnt;
  double step;
  INT8 last_sample = 0;
  INT8 cur_sample;

  // 2 3 3 3 sequence for 8000->22050
  // using linear interpolation
  cnt = 0;
  while (size--) {
    cur_sample = translate_one_sample(*buf, version);

    if ((cnt++ & 3)) {
      step = (cur_sample - last_sample)/3.0;
      out[0] = last_sample +step;
      out[1] = last_sample +step*2;
      out[2] = cur_sample;
      out += 3;
    } else {
      step = (cur_sample - last_sample)/2.0;
      out[0] = last_sample+step;
      out[1] = cur_sample;
      out += 2;
    }
    last_sample = cur_sample;
    
    *buf++ = cur_sample;
  }
}

static void translate_8000Hz_to_22050Hz_16_plain(INT16 *buf, INT16 *out, int size, int version)
{
  UINT8 cnt;
  double step;
  INT16 last_sample = 0;
  INT16 cur_sample;

  // 2 3 3 3 sequence for 8000->22050
  // using linear interpolation
  cnt = 0;
  while (size--) {
    cur_sample = *buf;

    if ((cnt++ & 3)) {
      step = (cur_sample - last_sample)/3.0;
      out[0] = last_sample +step;
      out[1] = last_sample +step*2;
      out[2] = cur_sample;
      out += 3;
    } else {
      step = (cur_sample - last_sample)/2.0;
      out[0] = last_sample+step;
      out[1] = cur_sample;
      out += 2;
    }
    last_sample = cur_sample;
    
    *buf++ = cur_sample;
  }
}

static void translate_8000Hz_to_16000Hz(INT8 *buf, INT8 *out, int size, int version)
{
  UINT8 cnt;
  int step;
  INT8 last_sample = 0;
  INT8 cur_sample;

  // using linear interpolation
  while (size--) {
    cur_sample = translate_one_sample(*buf, version);

    step = (cur_sample - last_sample)/2;
    out[0] = last_sample + step;
    out[1] = cur_sample;
    out += 2;

    last_sample = cur_sample;
    *buf++ = cur_sample;
  }
}

static void translate_8000Hz_to_44100Hz(INT8 *buf, INT8 *out, int size, int version)
{
  UINT8 cnt;
  double step;
  INT8 last_sample = 0;
  INT cur_sample;

  // 5 6 sequence for 8000->44100
  // using linear interpolation
  cnt = 0;
  while (size--) {
    cur_sample = translate_one_sample(*buf, version);

    if ((cnt++ & 1)) {
      step = (cur_sample - last_sample)/6.0;
      out[0] = last_sample + step;
      out[1] = last_sample + step*2;
      out[2] = last_sample + step*3;
      out[3] = last_sample + step*4;
      out[4] = last_sample + step*5;
      out[5] = cur_sample;
      out += 6;
    } else {
      step = (cur_sample - last_sample)/5.0;
      out[0] = last_sample + step;
      out[1] = last_sample + step*2;
      out[2] = last_sample + step*3;
      out[3] = last_sample + step*4;
      out[4] = cur_sample;
      out += 5;
    }
    last_sample = cur_sample;
    *buf++ = cur_sample;
  }
}

/*
  SYSTEM18 PCM SOUND CHIP INFO

c000
addr  0  1  2  3  4  5  6  7   8
     ___    ____  _______ ___ ___
     vol    freq  envolop cmd key

7 cmd: cn  inform channel n
       8n  select sound ram access window to bank n (offset= n*4KB)
sound ram : 64KB splitted to 16 4KB blocks, each channel has 2.

8 key on/off: 0=key on, 1=key off
*/

struct sys18_pcm_chip {
  int cur_ch;
  int cur_ram_bank;
  UINT8 key_on_off;
  struct sys18_pcm_channel {
    UINT8 key_on;
    UINT8 raw_data[7];
    // raw sample address
    long start_offset;
    // cooked sound sample address
    UINT8 *begin_playing;
    UINT8 *cur_playing;
    UINT8 *end_playing;
    // internal buffer
    UINT32 pos0;
    int remainder_size;
    long cur_buf;
    int size_sound_sample_page;
    int load_samples;
    UINT8 had_recently_load_sound_ram;
    WaveBuffer wave;
  } ch[8];
} sys18_pcm_chip;
extern UINT8 *z80data;
extern UINT32 z80data_len;
static UINT8 *old_z80data;
#define SIZE_SOUND_SAMPLE_PAGE (chan->size_sound_sample_page)
#define PRESTORED_SOUND_SAMPLES (4*chan->size_sound_sample_page)
#define DEFAULT_SOUND_SAMPLES (8*chan->size_sound_sample_page)
#define ONE_BANK_OF_SOUND_SAMPLES (31*chan->size_sound_sample_page)

static void poll_sys18_pcm_chip()
{
  int ch;
  for (ch=0; ch<8; ch++) {
    UINT32 pos1;
    int cur_size;
    struct sys18_pcm_channel *chan = &sys18_pcm_chip.ch[ch];
    long over_limit;
  
    if (!chan->key_on) continue;

    GetPlayPosition(Channels[MIN_PCM_CHANNEL+ch], &pos1); // pos1*=2;

    if (pos1 >= chan->pos0) cur_size = pos1 - chan->pos0;
    else cur_size = SIZEBUFFER - chan->pos0 + pos1;

    cur_size += chan->remainder_size;
    chan->remainder_size = 0;
    
    if (cur_size > MAX_SLICE){
      chan->remainder_size += cur_size - MAX_SLICE;
      cur_size = MAX_SLICE;
    }

    // if whole bank(0000~1eff) is loaded and loopback is not set
    // it is highly possible that Z80 may load data into sound RAM later
    // need not to check over_limit for this case
    // also, if sound ram is loaded recently, do not check over_limit
    if ( ((chan->load_samples % ONE_BANK_OF_SOUND_SAMPLES) == 0 && chan->raw_data[4] == 0) ||
           chan->cur_playing > chan->end_playing ||
           chan->had_recently_load_sound_ram) {

      chan->had_recently_load_sound_ram = 0;
      over_limit = 0;

    } else {
      over_limit = chan->cur_playing ?(chan->cur_playing + cur_size) - chan->end_playing :0;

      if (over_limit > 0) { // over limit
//        printf("over limit=%d cur_size=%d p=%08x e=%08x pos0=%d pos1=%d load=%d\n", over_limit, cur_size, chan->cur_playing, chan->end_playing, chan->pos0, pos1, chan->had_recently_load_sound_ram);
        chan->remainder_size += over_limit;
        cur_size -= over_limit;
      }
    }
  
    chan->pos0 = pos1;
    
    // when chan->cur_playing is NULL
    // WriteToWaveBuffer will only fill wave buffer with 0
    WriteToWaveBuffer(&chan->wave, chan->cur_buf, chan->cur_playing, cur_size);
    
    if (over_limit > 0) {
      if (chan->raw_data[4] != 0) { // loop back
        chan->cur_playing = chan->begin_playing;
      } else {
        chan->cur_playing = NULL; // quit playing
      }
    } else {
      if (chan->cur_playing) chan->cur_playing += cur_size;
    }

    // advance sample buffer pointer
    chan->cur_buf += cur_size;
    if (chan->cur_buf >= SIZEBUFFER) chan->cur_buf -= SIZEBUFFER;
  }
}

static void sys18_pcm_sound_toggle_key(int ch)
{
  struct sys18_pcm_channel *chan = &sys18_pcm_chip.ch[ch];
  if (!chan->key_on) { // key on --> start playing
    chan->key_on = 1;
//    printf("ch %d : playing %08x vol=%02x freq=%04x\n", ch, chan->start_offset, chan->raw_data[0], *(UINT16*)&chan->raw_data[2]);

    // set output frequency
    {
      int freq = (fixed_expanded_sample_rate * (*(UINT16*)&chan->raw_data[2])) / 0x27b ; // 0x27b = 8KHz
      if (freq > 2*SAMPLERATE) { // use 8KHz low resolution sample data
        chan->size_sound_sample_page = 256;
        chan->begin_playing = old_z80data + chan->start_offset;
        // re-calculate output frequency
        freq = (8000 * (*(UINT16*)&chan->raw_data[2])) / 0x27b ; // 0x27b = 8KHz
      } else { // use pre-expanded 22KHz sample data
        chan->size_sound_sample_page = expanded_sample_len(256);
        chan->begin_playing = z80data + expanded_sample_len(chan->start_offset);
      }
      SetWaveBufferSampleRate(&chan->wave, freq);
    }

    // initialize output buffer
    WriteToWaveBuffer(&chan->wave, 0, chan->begin_playing, PRESTORED_SOUND_SAMPLES);
    chan->cur_playing = chan->begin_playing + PRESTORED_SOUND_SAMPLES;
    chan->end_playing = chan->begin_playing + DEFAULT_SOUND_SAMPLES;
    chan->cur_buf = PRESTORED_SOUND_SAMPLES;
    chan->load_samples = SIZE_SOUND_SAMPLE_PAGE;

    chan->had_recently_load_sound_ram = 1;
    chan->pos0 = chan->remainder_size = 0;

    PlayWaveStream(&Channels[MIN_PCM_CHANNEL+ch], &chan->wave);
    // set output vol
    {
      int vol = 72*chan->raw_data[0]/0x80;
      if (vol>64) vol = 64;
      SetVol(Channels[MIN_PCM_CHANNEL+ch], vol);
    }
    
  } else { // key off
//    printf("ch %d : off\n", ch);
/*
    {
      int i;
      for (i=0; i<7; i++) printf("%02x ",chan->raw_data[i]);
      printf("\n");
    }
*/
    chan->key_on = 0;
    chan->start_offset = -1;
    
    StopPlaying(Channels[MIN_PCM_CHANNEL+ch]);

  }
}

void write_sys18_pcm_chip(UINT8 addr, UINT8 data)
{
  extern long MemBank;
  UINT8 xor_key_state;
  struct sys18_pcm_channel *chan;

  if (addr < 7)
    sys18_pcm_chip.ch[sys18_pcm_chip.cur_ch].raw_data[addr] = data;

  switch (addr) {
  case 7:
    switch (data&0xf0) {
    case 0xc0 : // channel select
      sys18_pcm_chip.cur_ch = data&0x7;
      break;
    case 0x80 : // sound ram bank select
      sys18_pcm_chip.cur_ram_bank = data&0xf;
      break;
    }
    break;

  case 8:
    xor_key_state = data ^ sys18_pcm_chip.key_on_off;
    if (xor_key_state) {
      if (xor_key_state & 1) sys18_pcm_sound_toggle_key(0);
      if (xor_key_state & 2) sys18_pcm_sound_toggle_key(1);
      if (xor_key_state & 4) sys18_pcm_sound_toggle_key(2);
      if (xor_key_state & 8) sys18_pcm_sound_toggle_key(3);
      if (xor_key_state & 16) sys18_pcm_sound_toggle_key(4);
      if (xor_key_state & 32) sys18_pcm_sound_toggle_key(5);
      if (xor_key_state & 64) sys18_pcm_sound_toggle_key(6);
      if (xor_key_state & 128) sys18_pcm_sound_toggle_key(7);
    }
    sys18_pcm_chip.key_on_off = data;
    break; 
  
  case 0xff: // agent reports sound sample address
    // algorithm:
    // Z80 traps writes of d?00, which contains sound ram bank(a000~bfff) address signature(a0~bf)
    // This signature is sent here as a message for constructing the
    // starting address and ending address of sound samples in the ROM.
    if (data == 0xff) break; // not signature
    chan = &sys18_pcm_chip.ch[sys18_pcm_chip.cur_ram_bank>>1];
    chan->had_recently_load_sound_ram = 1;

    if (chan->start_offset == -1) { // starting
      chan->start_offset = MemBank + ((data-0xa0)<<8);
    } else { // already playing : increase limit
      chan->load_samples += SIZE_SOUND_SAMPLE_PAGE;
      if (chan->load_samples > DEFAULT_SOUND_SAMPLES) // extend limit
        chan->end_playing += SIZE_SOUND_SAMPLE_PAGE;
    }
    break;
  
  }

}

// return 0 if succ
int init_sys18_pcm_chip()
{
  int i;
  UINT8 *new_z80data;

  fixed_expanded_sample_rate = 22050;
  expanded_rate = 22;
  new_z80data = (UINT8 *)new_malloc(expanded_sample_len(z80data_len) + SIZEBUFFER);
  translate_8000Hz_to_22050Hz(z80data, new_z80data, z80data_len, 0);
  old_z80data = z80data;
  z80data = new_z80data;

  memset(&sys18_pcm_chip, 0, sizeof(sys18_pcm_chip));
  sys18_pcm_chip.key_on_off = 0xff;

  for (i=0; i<8; i++) {
    sys18_pcm_chip.ch[i].start_offset = -1;
    sys18_pcm_chip.ch[i].key_on = 0;
    init_wave_stream(&sys18_pcm_chip.ch[i].wave, 8);
  }

  return 0;
}


/*
  SEGA PCM SOUND CHIP INFO
  TOTAL 16 CHANNELS
  
  f000(e000): reg00~reg07 
  0      1     2     3      4     5      6     7
              vol   vol  addr_l addr_h addr_e freq
             left  right  ___begin___  _end_
  
  f080(e080): reg08~reg0f
  0      1     2     3      4     5      6     7
                         addr_l addr_h  key
                                       on/off
  
  reg0e:
  bit0 = key on(0) / key off(1)
  bit1 = not playing(0) / playing(1) : reset to 0 by hardware at the end
  bit2,3 = loop back? change freq during play? change vol during play?
  bit4~bit6 : addr_h2 (32KB sample bank selector?)
*/
struct sega_pcm_chip {
  struct sega_pcm_channel {
    UINT8 key_on;
    UINT8 wait;
    UINT8 raw_data[16];
    // cooked sound sample address
    UINT8 *begin_playing;
    UINT8 *cur_playing;
    UINT8 *end_playing;
    // internal buffer
    UINT32 pos0;
    UINT8 loop_back;
    int remainder_size;
    long cur_buf;
    long pre_load;
    int size_sound_sample_page;
    WaveBuffer wave;
  } ch[16];
} sega_pcm_chip;
#define PRESTORED_SEGA_PCM_SOUND_SAMPLES (4*SIZE_SOUND_SAMPLE_PAGE)
static int sega_pcm_freq;

/* change vol */
inline static void sega_pcm_sound_reload_vol(int ch)
{
  struct sega_pcm_channel *chan = &sega_pcm_chip.ch[ch];
  int vol = 86*(chan->raw_data[2]+chan->raw_data[3])/0x80;
  if (vol>64) vol = 64;
  else if (vol<0) vol = 0;
  SetVol(Channels[MIN_PCM_CHANNEL+ch], vol);
}

static void sega_pcm_sound_key_on(int ch, UINT8 data)
{
  struct sega_pcm_channel *chan = &sega_pcm_chip.ch[ch];
  long start_offset, end_offset, sample_len;
  int pre_load;
  int freq;
  
  if (chan->raw_data[2]+chan->raw_data[3] == 0) return;
  
  chan->key_on = 1;
  chan->loop_back = (data&0xf)==0; // engine roar

  start_offset = (*(UINT16 *)&chan->raw_data[4]) + ((data&0x70)<<11);
  end_offset = (chan->raw_data[6]<<8) + 0xff + ((data&0x70)<<11);
  if (end_offset < start_offset) end_offset += 0x10000; // for Space Harrier
  sample_len = end_offset - start_offset + 1; // raw sample length

  // set output frequency
  {
    freq = fixed_expanded_sample_rate * (chan->raw_data[7]) / sega_pcm_freq; // 0x21 = 8KHz for Space Harrier
    // if output frequency is more than 2 times of SAMPLERATE, use low resolution data
    // also if b0~b3 is all 0 for reg0e, use low resolution data since the frequency can be changed during play
    if (freq > 2*SAMPLERATE || ((data&0xf)==0)) { // use 8KHz low resolution sample data
      chan->size_sound_sample_page = 256;
      chan->begin_playing = old_z80data + start_offset;
      chan->end_playing = old_z80data + end_offset;
      // re-calculate output frequency
      freq = 8000 * (chan->raw_data[7]) / sega_pcm_freq; // 0x21 = 8KHz for Space Harrier
      pre_load = sample_len;
    } else { // use pre-expanded 22KHz sample data
      chan->size_sound_sample_page = expanded_sample_len(256);
      chan->begin_playing = z80data + expanded_sample_len(start_offset);
      chan->end_playing = z80data + expanded_sample_len(end_offset);

      pre_load = expanded_sample_len(sample_len);
    }
    SetWaveBufferSampleRate(&chan->wave, freq);
  }

  if (pre_load > PRESTORED_SOUND_SAMPLES)
    pre_load = PRESTORED_SOUND_SAMPLES;
    
  chan->pre_load = pre_load; // size of pre-loaded sound samples
  // initialize output buffer
  WriteToWaveBuffer(&chan->wave, 0, chan->begin_playing, pre_load);

  chan->cur_playing = chan->begin_playing + pre_load;
  chan->pos0 = chan->remainder_size = 0;
  chan->cur_buf = pre_load;

//  printf("ch %d : %02x playing %08x~%08x(%08x~%08x) vol=%02x %02x freq=%02x\n", ch, data, start_offset, end_offset, chan->begin_playing, chan->end_playing, chan->raw_data[2], chan->raw_data[3], chan->raw_data[7]);

  PlayWaveStream(&Channels[MIN_PCM_CHANNEL+ch], &chan->wave);
  // set volume
  sega_pcm_sound_reload_vol(ch);
}

static void sega_pcm_sound_key_off(int ch)
{
  struct sega_pcm_channel *chan = &sega_pcm_chip.ch[ch];
  
//  printf("ch %d : off\n", ch);
  
  chan->key_on = 0;
 // reset reg0e
  chan->raw_data[0xe] = 1;
  
  StopPlaying(Channels[MIN_PCM_CHANNEL+ch]);
}

static void poll_sega_pcm_chip()
{
  int ch;
  for (ch=0; ch<16; ch++) {
    UINT32 pos1;
    int cur_size;
    struct sega_pcm_channel *chan = &sega_pcm_chip.ch[ch];
    long over_limit;
  
    if (chan->wait) { // deferred command
      if (!--chan->wait) { // issue it now
        sega_pcm_sound_key_on(ch, chan->raw_data[0xe]);
      } else  { 
//        printf("skip %d\n", ch);
        continue;
      }
    }
    
    if (!chan->key_on) continue;

    GetPlayPosition(Channels[MIN_PCM_CHANNEL+ch], &pos1); // pos1*=2;

    if (pos1 >= chan->pos0) cur_size = pos1 - chan->pos0;
    else cur_size = SIZEBUFFER - chan->pos0 + pos1;

    cur_size += chan->remainder_size;
    chan->remainder_size = 0;
    
    if (cur_size > MAX_SLICE){
      chan->remainder_size += cur_size - MAX_SLICE;
      cur_size = MAX_SLICE;
    }

    over_limit = (chan->cur_playing + cur_size) - chan->end_playing;

    if (over_limit > 0 && chan->cur_playing<chan->end_playing) { // over limit
//      printf("over limit=%d cur_size=%d p=%08x e=%08x pos0=%d pos1=%d\n", over_limit, cur_size, chan->cur_playing, chan->end_playing, chan->pos0, pos1);
      chan->remainder_size += over_limit;
      cur_size -= over_limit;
    }
  
    chan->pos0 = pos1;
    
    if (chan->cur_playing < chan->end_playing) {
      WriteToWaveBuffer(&chan->wave, chan->cur_buf, chan->cur_playing, cur_size);
    } else { // fill wave buffer with 0
      WriteToWaveBuffer(&chan->wave, chan->cur_buf, NULL, cur_size);
    }
    
    chan->cur_playing += cur_size;
    
    if (over_limit > 0) {
      if (chan->loop_back) { // need to reset play pointer
        chan->cur_playing = chan->begin_playing;

        if ((chan->raw_data[0xe]&0xf) == 0) { // may change vol&freq during playing
          sega_pcm_sound_reload_vol(ch);
          SetChannelSampleRate(Channels[MIN_PCM_CHANNEL+ch], 8000 * (chan->raw_data[7]) / sega_pcm_freq);
        }

      } else if (over_limit >= chan->pre_load) { 
        // all samples had been played
        // turn off key here
        sega_pcm_sound_key_off(ch);
        return;
      }
    }

//      printf("chan->cur_playing = %08x\n", chan->cur_playing);

    // advance sample buffer pointer
    chan->cur_buf += cur_size;
    if (chan->cur_buf >= SIZEBUFFER) chan->cur_buf -= SIZEBUFFER;
  }
}


void write_sega_pcm_chip(UINT8 addr, UINT8 data)
{
  struct sega_pcm_channel *chan;
  int reg = (addr&7) + ((addr&0x80)>>4);
  int channel = (addr&0x78)>>3;
  extern UINT16 z80pc;
  extern UINT8 s_hangon;

  chan = &sega_pcm_chip.ch[channel];
  chan->raw_data[reg] = data;
//  printf("pcm ch%02d: %02x=%02x (%02x) pc=%04x\n", channel, reg ,data, addr, z80pc);

  switch (reg) {
  case 0xe:
    if (data&0x1) {
//      chan->wait = 0;
      sega_pcm_sound_key_off(channel);
    } else {
      if (s_hangon) sega_pcm_sound_key_on(channel, data);
      else {
        extern UINT8 game;
        if (game == 20) chan->wait = 2;
        else chan->wait = 1;
      }
    }
    break; 
  }

}

UINT8 read_sega_pcm_chip(UINT8 addr)
{
  struct sega_pcm_channel *chan;
  int reg = (addr&7) + ((addr&0x80)>>4);
  int channel = (addr&0x78)>>3;
  UINT8 data;
  extern UINT16 z80pc;
  
  chan = &sega_pcm_chip.ch[channel];
  data = chan->raw_data[reg];
//  printf("pcm read ch%02d: %02x=%02x (%02x) pc=%04x\n", channel, reg ,data, addr, z80pc);
  return data;
}

// return 0 if succ
int init_sega_pcm_chip()
{
  int i;
  UINT8 *new_z80data;

  fixed_expanded_sample_rate = 22050;
  expanded_rate = 22;
  new_z80data = (UINT8 *)new_malloc(expanded_sample_len(z80data_len) + SIZEBUFFER);
  translate_8000Hz_to_22050Hz(z80data, new_z80data, z80data_len, 1);
  old_z80data = z80data;
  z80data = new_z80data;

  memset(&sega_pcm_chip, 0, sizeof(sega_pcm_chip));

  for (i=0; i<16; i++) {
    sega_pcm_chip.ch[i].key_on = 0;
    init_wave_stream(&sega_pcm_chip.ch[i].wave, 8);
  }
  
  return 0;
}





//////uPD7759////////////////////////////////////////////////////////////////////

#define uPD7759_PCM_MAX_BLOCKS	256
#define uPD7759_PCM_CHANNELS	2
struct {
  int base_sampling_rate; // 6k or 8k
  int expanded_sampling_rate;
  int size_sample; // 0(8 bits) or 1(16bits): use left-shift for getting real size
  int num_blocks;
  struct {
    UINT32 offset; // physical ROM offset. presented for block seeking
    UINT8 *pcm_block;
    long pcm_num_samples; // number of samples
  } blocks[uPD7759_PCM_MAX_BLOCKS];
  struct uPD7759_pcm_channel {
    UINT8 key_on;
    // cooked sound sample address
    UINT8 *begin_playing;
    UINT8 *cur_playing;
    UINT8 *end_playing;
    // internal buffer
    UINT32 pos0;
    UINT8 loop_back;
    int remainder_size;
    long cur_buf;
    long pre_load;
    WaveBuffer wave;
  } ch[uPD7759_PCM_CHANNELS];
  int cur_ch;
} uPD7759_pcm_chip;

/////////////////////////////////////////////////////////////////////////////////
/////////MAME uPD7759 STUFF//////////////////////////////////////////////////
/* define the output rate */
#define CLOCK_DIVIDER 8000
#define SIGNAL_BITS 	13			/* signal range */
#define SIGNAL_SHIFT_L	0			/* adjustment for 16bit samples */
#define SIGNAL_SHIFT_R  (8-SIGNAL_SHIFT_L)           /* adjustment for 8bit samples */
#define SIGNAL_MAX      (0x7fff >> (15-(SIGNAL_SHIFT_L+SIGNAL_BITS)))
#define SIGNAL_MIN		-(SIGNAL_MAX+1)
#define SIGNAL_ZERO 	-2
#define STEP_MAX		48
#define STEP_MIN		0
#define STEP_ZERO		0
/* struct describing a single playing ADPCM voice */
typedef struct
{
	int sample;
	int mask;
	int count;              /* total samples to play */
	int signal;             /* current ADPCM signal */
	int step;               /* current ADPCM step */
} uPD7759_state;
/* step size index shift table */
//static int index_shift[8] = { -1, -1, -1, -1, 2, 4, 6, 8 };
static int index_shift[8] = {-1, 1, 1, 2, 2, 3, 5, 7 };
/* lookup table for the precomputed difference */
static int diff_lookup[49*16];
/*
 *   Compute the difference table
 */
static void ComputeTables (void)
{
	/* nibble to bit map */
	static int nbl2bit[16][4] = {
		{ 1, 0, 0, 0}, { 1, 0, 0, 1}, { 1, 0, 1, 0}, { 1, 0, 1, 1},
		{ 1, 1, 0, 0}, { 1, 1, 0, 1}, { 1, 1, 1, 0}, { 1, 1, 1, 1},
		{-1, 0, 0, 0}, {-1, 0, 0, 1}, {-1, 0, 1, 0}, {-1, 0, 1, 1},
		{-1, 1, 0, 0}, {-1, 1, 0, 1}, {-1, 1, 1, 0}, {-1, 1, 1, 1}
	};
	int step, nib;

	/* loop over all possible steps */
	for (step = 0; step <= STEP_MAX; step++) {

	/* compute the step value */
		int stepval = floor(16.0 * pow (11.0 / 10.0, (double)step));
		/* loop over all nibbles and compute the difference */
		for (nib = 0; nib < 16; nib++) {
			diff_lookup[step*16 + nib] = nbl2bit[nib][0] *
				(stepval   * nbl2bit[nib][1] +
				 stepval/2 * nbl2bit[nib][2] +
				 stepval/4 * nbl2bit[nib][3] +
				 stepval/8);
        }
    }
}

/* return number of processed adpcm samples (<=len_buffer) */
// size_sample : 0=8bit 1=16bit
static int decode_uPD7759(UINT8 *adpcm_data, int len_buffer, int size_sample, UINT8 *pcm_data)
{
  uPD7759_state uPD7759_chip;
  uPD7759_state *voice = &uPD7759_chip;
  UINT8 data;
  int i;
  
  voice->count = 0;
  voice->signal = SIGNAL_ZERO;
  voice->step = STEP_ZERO;
  voice->sample = 0;
  voice->mask = 0xffffffff;
  
  for (i=0; i<len_buffer; i++) {
    data = adpcm_data[i];

    /* detect end of a sample by inspection of the last 5 bytes */
    /* FF 00 00 00 00 is the start of the next sample */
    if (voice->count > 5 && voice->sample == 0xff000000 && data == 0x00) {

      break;
    }

    /* collect the data written in voice->sample */
    voice->sample = (voice->sample << 8) | (data);
    voice->count++;

    /* convert the upper nibble */
    voice->signal += diff_lookup[voice->step * 16 + ((voice->sample >> 4) & 15)];
    if (voice->signal > SIGNAL_MAX) voice->signal = SIGNAL_MAX;
    else if (voice->signal < SIGNAL_MIN) voice->signal = SIGNAL_MIN;
    voice->step += index_shift[(voice->sample >> 4) & 7];
    if (voice->step > STEP_MAX) voice->step = STEP_MAX;
    else if (voice->step < STEP_MIN) voice->step = STEP_MIN;
            
    {
      int sample = voice->signal * 4;
      if (sample > 0x7fff) sample = 0x7fff;
      if (sample <-0x8000) sample = 0x8000;
      if (size_sample == 0) { //8bits
        sample >>= 8;
        *(INT8 *)pcm_data = (INT8)sample;
        pcm_data += sizeof(INT8);
      } else {
        *(INT16 *)pcm_data = (INT16)sample;
        pcm_data += sizeof(INT16);
      }
    }

    /* convert the lower nibble */
    voice->signal += diff_lookup[voice->step * 16 + ((voice->sample) & 15)];
    if (voice->signal > SIGNAL_MAX) voice->signal = SIGNAL_MAX;
    else if (voice->signal < SIGNAL_MIN) voice->signal = SIGNAL_MIN;
    voice->step += index_shift[(voice->sample) & 7];
    if (voice->step > STEP_MAX) voice->step = STEP_MAX;
    else if (voice->step < STEP_MIN) voice->step = STEP_MIN;
            
    {
      int sample = voice->signal * 4;
      if (sample > 0x7fff) sample = 0x7fff;
      if (sample <-0x8000) sample = 0x8000;
      if (size_sample == 0) { // 8bits
        sample >>= 8;
        *(INT8 *)pcm_data = (INT8)sample;
        pcm_data += sizeof(INT8);
      } else {
        *(INT16 *)pcm_data = (INT16)sample;
        pcm_data += sizeof(INT16);
      }
    }
  }
  
  return voice->count;
}

/////////////////////////////////////////////////////////////////////////////////
// size_sample = 0:8bit 1:16bit
static void fadein_samples(int size_sample, UINT8 *data, int num_samples)
{
  int i;
  double dif = 1.0/num_samples;
  if (size_sample == 0) { // 8bits
    ((INT8 *)data)[0] = 0;
    for (i=1; i<num_samples; i++) {
      ((INT8 *)data)[i] = ((INT8 *)data)[i] * (dif*i);
    }
  } else { // 16bits
    ((INT16 *)data)[0] = 0;
    for (i=1; i<num_samples; i++) {
      ((INT16 *)data)[i] = ((INT16 *)data)[i] * (dif*i);
    }
  }
}
// size_sample = 0:8bit 1:16bit
static void fadeout_samples(int size_sample, UINT8 *data, int num_samples)
{
  int i;
  double dif = 1.0/(num_samples-16);
  if (size_sample == 0) { // 8bits
    for (i=0; i<num_samples-16; i++) {
      ((INT8 *)data)[i] = ((INT8 *)data)[i] * (dif*(num_samples-16-i));
    }
    for (i=num_samples-16; i<num_samples; i++) ((INT8 *)data)[i] = 0;
  } else { // 16 bits
    for (i=0; i<num_samples-16; i++) {
      ((INT16 *)data)[i] = ((INT16 *)data)[i] * (dif*(num_samples-16-i));
    }
    for (i=num_samples-16; i<num_samples; i++) ((INT16 *)data)[i] = 0;
  }
}

// return 0 if succ
static int  build_uPD7759_pcm_blocks(UINT8 *speech_data, long speech_data_len)
{
  UINT8 *output, *expanded;
  int output_data_len = speech_data_len*2<<uPD7759_pcm_chip.size_sample;
  int i;
  int expanded_len = ((int)(1024 + output_data_len * (double)uPD7759_pcm_chip.expanded_sampling_rate/8000))<<uPD7759_pcm_chip.size_sample;
  
  if ( (expanded = (UINT8 *)malloc(expanded_len)) == NULL) return -1;
  if ( (output = (UINT8 *)malloc(output_data_len)) == NULL) return -1;

  memset(output, 0 , output_data_len);
  memset(expanded, 0 , expanded_len);
  
  ComputeTables(); // init MAME uPD7759 decoding table

  uPD7759_pcm_chip.num_blocks = 0;

  i=0;
  while (i<speech_data_len) {
    int block_len;
    UINT8 *block_begin;
    int j;
    int len2;

    block_begin = &speech_data[i];
    block_len = speech_data_len-i;

    // search for one block
    // search for ff 00 00 00 00
    for (j=0; j<block_len; j++) {
      if (*(UINT32 *)&block_begin[j] == 0x000000ff && block_begin[j+4] == 0) {
        block_begin += j;
        block_len -= j;
        break;
      }
    }
    //printf("i=%08x j=%d  %d %08x\n", i, j, block_len, i+j);
    uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].offset = i+j;

    if (j < block_len) { // search for 00 ff
      for (j=0; j<block_len; j++) {
        if (*(UINT16 *)&block_begin[j] == 0xff00) {
          block_len = j;
          break;
        }
      }
    } else { // invalid block
      //printf("quit\n");
      break;
    }
    //printf("bl=%d\n", block_len);
    
    uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block = &output[i*2<<uPD7759_pcm_chip.size_sample];
    
    len2 = decode_uPD7759(block_begin, block_len, uPD7759_pcm_chip.size_sample, uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block);
    uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples = len2*2;
    
    #define FADE_SAMPLES 64
    fadein_samples(uPD7759_pcm_chip.size_sample, &uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block[0], FADE_SAMPLES);
    fadeout_samples(uPD7759_pcm_chip.size_sample, &uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block[(len2*2-FADE_SAMPLES)<<uPD7759_pcm_chip.size_sample], FADE_SAMPLES);
    
    // re-sampling pcm block to 22K
    {
      int expanded_len = ((int)(len2*2*(double)uPD7759_pcm_chip.expanded_sampling_rate/8000))<<uPD7759_pcm_chip.size_sample;

      if (uPD7759_pcm_chip.size_sample == 0) { // 8bit
        translate_8000Hz_to_22050Hz_plain(
          uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block, 
          expanded, 
          uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples,
          2);
      } else { //16bit
        translate_8000Hz_to_22050Hz_16_plain(
          (INT16 *)uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block, 
          (INT16 *)expanded, 
          uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples,
          2);
      }
      
      uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_block = expanded;
      uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples = uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples * (double)uPD7759_pcm_chip.expanded_sampling_rate/8000;
      expanded += uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].pcm_num_samples << uPD7759_pcm_chip.size_sample;
      expanded = (UINT8*)(((UINT32)expanded+2) & ~1);
    }

    i = uPD7759_pcm_chip.blocks[uPD7759_pcm_chip.num_blocks].offset + len2;
    //printf("i=%d len2=%d\n", i, len2);

    uPD7759_pcm_chip.num_blocks++;
  }

  printf("%d blocks\n", uPD7759_pcm_chip.num_blocks);
  for (i=0; i<uPD7759_pcm_chip.num_blocks; i++) {
    printf("%03d : offset=%08x len=%08x\n", i, uPD7759_pcm_chip.blocks[i].offset, uPD7759_pcm_chip.blocks[i].pcm_num_samples/2);
  }
 
  free(output);
  return 0;
}

// return 0 if succ
int init_uPD7759_pcm()
{
  extern UINT8 *speech_data;
  extern long speech_data_len;
  extern int speech_data_samplingrate;
  int i;
  int audio_device_bits = 8;
  
  memset(&uPD7759_pcm_chip, 0, sizeof(uPD7759_pcm_chip));
  uPD7759_pcm_chip.base_sampling_rate = speech_data_samplingrate;
  uPD7759_pcm_chip.size_sample = audio_device_bits/16;
  uPD7759_pcm_chip.expanded_sampling_rate = 22050;
  
  if (build_uPD7759_pcm_blocks(speech_data, speech_data_len) != 0) return -1;
  
  for (i=0; i<uPD7759_PCM_CHANNELS; i++) {
    uPD7759_pcm_chip.ch[i].key_on = 0;
    init_wave_stream(&uPD7759_pcm_chip.ch[i].wave, audio_device_bits);
  }
  return 0;
}

static void uPD7759_pcm_sound_key_on(int ch, int pcm_block_num, int base_sampling_rate)
{
  struct uPD7759_pcm_channel *chan = &uPD7759_pcm_chip.ch[ch];
  int pre_load;
  #define DEFAULT_PRELOAD_SIZE	1024
  int size_block = uPD7759_pcm_chip.blocks[pcm_block_num].pcm_num_samples << uPD7759_pcm_chip.size_sample;
  
  chan->key_on = 1;
  chan->loop_back = 0;

  chan->begin_playing = uPD7759_pcm_chip.blocks[pcm_block_num].pcm_block;
  chan->end_playing = chan->begin_playing + size_block;

  SetWaveBufferSampleRate(&chan->wave, uPD7759_pcm_chip.expanded_sampling_rate*base_sampling_rate/8000);

  {
  FILE *ff = NULL;
  ff = fopen("ooo","wb");
  fwrite(chan->begin_playing, 1, size_block, ff);
  fclose(ff);
  }

  pre_load = size_block >DEFAULT_PRELOAD_SIZE ?DEFAULT_PRELOAD_SIZE :size_block;
  chan->pre_load = pre_load;
  
  // initialize output buffer
  WriteToWaveBuffer(&chan->wave, 0, chan->begin_playing, pre_load);
  chan->cur_playing = chan->begin_playing + pre_load;
  chan->pos0 = chan->remainder_size = 0;
  chan->cur_buf = pre_load;

  //printf("ch %d : block#%03d playing (%08x~%08x)\n", ch, pcm_block_num, chan->begin_playing, chan->end_playing);

  PlayWaveStream(&Channels[MIN_PCM_CHANNEL+ch], &chan->wave);
}

void uPD7759_pcm_sound_key_off(int ch)
{
  struct uPD7759_pcm_channel *chan = &uPD7759_pcm_chip.ch[ch];

  //printf("ch %d : off\n", ch);
  
  chan->key_on = 0;
    
  StopPlaying(Channels[MIN_PCM_CHANNEL+ch]);
}

void poll_uPD7759_pcm()
{
  int ch;
  for (ch=0; ch<uPD7759_PCM_CHANNELS; ch++) {
    UINT32 pos1;
    int cur_size;
    struct uPD7759_pcm_channel *chan = &uPD7759_pcm_chip.ch[ch];
    long over_limit;
  
    if (!chan->key_on) continue;

    GetPlayPosition(Channels[MIN_PCM_CHANNEL+ch], &pos1); // measured in samples
    pos1 <<= uPD7759_pcm_chip.size_sample; // now in bytes

    if (pos1 >= chan->pos0) cur_size = pos1 - chan->pos0;
    else cur_size = SIZEBUFFER - chan->pos0 + pos1;

    cur_size += chan->remainder_size;
    chan->remainder_size = 0;
    
    if (cur_size > MAX_SLICE) {
      chan->remainder_size += cur_size - MAX_SLICE;
      cur_size = MAX_SLICE;
    }

    over_limit = (chan->cur_playing + cur_size) - chan->end_playing;

    if (over_limit > 0 && chan->cur_playing<chan->end_playing) { // over limit
//      printf("over limit=%d cur_size=%d p=%08x e=%08x pos0=%d pos1=%d\n", over_limit, cur_size, chan->cur_playing, chan->end_playing, chan->pos0, pos1);
      chan->remainder_size += over_limit;
      cur_size -= over_limit;
    }
  
    chan->pos0 = pos1;
    if (chan->cur_playing < chan->end_playing) {
      WriteToWaveBuffer(&chan->wave, chan->cur_buf, chan->cur_playing, cur_size);
    } else { // fill wave buffer with 0
      WriteToWaveBuffer(&chan->wave, chan->cur_buf, NULL, cur_size);
    }
    
    chan->cur_playing += cur_size;
    
    if (over_limit > 0) {
      if (chan->loop_back) { // need to reset play pointer
        chan->cur_playing = chan->begin_playing;

      } else if (over_limit >= chan->pre_load) { 
        // all samples had been played
        // turn off key here
        uPD7759_pcm_sound_key_off(ch);
        return;
      }
    }

//      printf("chan->cur_playing = %08x\n", chan->cur_playing);

    // advance sample buffer pointer
    chan->cur_buf += cur_size;
    if (chan->cur_buf >= SIZEBUFFER) chan->cur_buf -= SIZEBUFFER;
  }

}

void write_uPD7759_pcm_reg(int reg, UINT32 data)
{
  struct uPD7759_pcm_channel *chan;
  int channel = uPD7759_pcm_chip.cur_ch;
  UINT32 offset;
  int i;
  
  //printf("write to uPD7759 pcm %02x %08x\n", reg, data);
  chan = &uPD7759_pcm_chip.ch[channel];

  if (chan->key_on) uPD7759_pcm_sound_key_off(channel);
  
  // seek the pcm block for playing. if not found, drop it.
  for (i=0; i<uPD7759_pcm_chip.num_blocks; i++) {
    if (uPD7759_pcm_chip.blocks[i].offset >= data) { // found

      uPD7759_pcm_sound_key_on(channel, i, uPD7759_pcm_chip.base_sampling_rate);

      if (++uPD7759_pcm_chip.cur_ch >= uPD7759_PCM_CHANNELS) uPD7759_pcm_chip.cur_ch = 0;
      break;
    }
  }

}

/////////////////////////////////////////////////////////////////////////////////





static void reset_external_sample_playing()
{
  // initialize sfx cmd buffer contents to -1
  memset(voice_sample_to_play_queue, 0xff, sizeof(voice_sample_to_play_queue));
  memset(sound_effect_sample_to_play, 0xff, sizeof(sound_effect_sample_to_play));
  current_sfx_channel = MIN_SFX_CHANNEL;
  current_voice_queue_read_index = current_voice_queue_write_index = 0;
}

// return 0 if succ
int OpenAudio()
{
  int i,i2;
  char mymsg[256];
  extern void YMWriteReg(int,int,int);
  extern void OPMWriteReg(int,int,int);
  extern void YM2151UpdateOne(int,int);
  extern void OPMUpdateOne(int,int);

  if (ym2151_active) {
    if (Jarek2151) {
      WriteFMReg = YMWriteReg;
      UpdateOPM = YM2151UpdateOne;
    } else {
      WriteFMReg = OPMWriteReg;
      UpdateOPM = OPMUpdateOne;
    }
    poll_fm_emu = poll_ym2151;
  } else if (ym2203_active) {
    WriteFMReg = _YM2203WriteReg;
    poll_fm_emu = poll_ym2203;
  } else if (ym3438_active) {
    poll_fm_emu = poll_ym3438;
  } else poll_fm_emu = NULL;

  if (type_sound_subboard == 1) {
    poll_pcm_emu = poll_sys18_pcm_chip;
  } else if (type_sound_subboard == 2) {
    sega_pcm_freq = 0x21;
    poll_pcm_emu = poll_sega_pcm_chip;
  } else if (type_sound_subboard == 3) {
    sega_pcm_freq = 0x44;
    poll_pcm_emu = poll_sega_pcm_chip;
  } else if (uPD7759_active) {
    poll_pcm_emu = poll_uPD7759_pcm;
  } else if (uPD7751_active) {
    poll_pcm_emu = NULL;
  } else poll_pcm_emu = NULL;
  
  if (InitAudio() != 0) return -1;

  memset(sample_ok, 0, sizeof(sample_ok));
  imessage("Loading samples...\n", 47, 46, 0, 0);
  LoadSamplesWAV_V2(Samples);

  for (i=0; i<NUMBER_OF_CHANNELS; i++) {
    if (CreatAudioChannel(&Channels[i])) {
      sprintf(mymsg, "Audio voice create error (%d) !\n", i);
      ierror(mymsg);
      return -1;
    }
    SetVol(Channels[i], 64);
  }
  
  if (poll_fm_emu) {
    if (ym2151_active) init_ym2151();
    else if (ym2203_active) init_ym2203();
    else if (ym3438_active) init_ym3438();
  }
  if (poll_pcm_emu) {
    if (type_sound_subboard == 1) init_sys18_pcm_chip();
    else if (type_sound_subboard == 2 || type_sound_subboard == 3) init_sega_pcm_chip();
    else if (uPD7759_active) init_uPD7759_pcm();
  }
  
  reset_external_sample_playing();
  
  return 0;
}

void stop_voices(void);
void sound_request(UINT8 sound_cmd)
{
   char fn[256];
//printf("sound request = %02x\n", sound_cmd);   

   if (sound_cmd==0xff || sound_cmd==0x80) return;
   
   if (sound_cmd==0x00) stop_voices();

   if (bgm_flags[sound_cmd]) current_bgm = sound_cmd;
   
#ifndef RELEASE
//   if (LOG_ACTIVE) fprintf(LOG, "sound request = %02x\n", sound_cmd);
   sprintf(fn, "sound_cmd %02x %s", sound_cmd, bgm_flags[sound_cmd]?"(bgm)":"");
   //if (bgm_flags[sound_cmd]) 
   message_emul(0, 2, fn);
#endif

   if (sample_ok[sound_cmd])  {
     // for games like SHINOBI voice samples need to be queued,
     // they should not be played simutaneously.
     // other sound samples can be played simutaneously
     if (sound_cmd>=vce_start && sound_cmd<=vce_end) {
       // store it in voice cmd queue
       voice_sample_to_play_queue[current_voice_queue_write_index++] = sound_cmd;
       if (current_voice_queue_write_index >= SIZE_VOICE_QUEUE) current_voice_queue_write_index = 0;
     } else if (sound_cmd>=sfx_start && sound_cmd<=sfx_end) {
       sound_effect_sample_to_play[current_sfx_channel++ - MIN_SFX_CHANNEL] = sound_cmd;
       if (current_sfx_channel > MAX_SFX_CHANNEL) current_sfx_channel = MIN_SFX_CHANNEL;
     }
//goto SendToZ80;
   } else { /* send to Z80 */

     extern UINT8 ComPort;
     static UINT8 last_cmd=0;
SendToZ80:
     if (ym2203_active) { // stop PSG for Space Harrier
       AYWriteReg(0,8,0);
       AYWriteReg(0,9,0);
       AYWriteReg(0,0xa,0);
     }

     if (sound_cmd!=0xff) { /* FF: dummy marker don't send to Z80 */
       //printf("o:%x s:%x l:%x\n",ComPort,sound_cmd,last_cmd);
       last_cmd = ComPort = sound_cmd;
       inform_z80();
     }

   }

}

void samples_control(void)
{
   int i;
   int sound_cmd;
   int stopped;
   char message[256];
   extern UINT8 check_performance;
   extern unsigned performance_fm;

   if (check_performance) chk_tsc();
   UpdateAudio();
   if (poll_fm_emu) poll_fm_emu();
   if (poll_pcm_emu) poll_pcm_emu();
   if (check_performance) performance_fm = chk_tsc();
   
   // check voice sample playing state
   GetPlayStatus(Channels[VOICE_CHANNEL], &stopped);
   if (stopped) {
     if (current_voice_queue_read_index != current_voice_queue_write_index) {
       sound_cmd = voice_sample_to_play_queue[current_voice_queue_read_index];
       PlayStaticWave(&Channels[VOICE_CHANNEL], Samples[sound_cmd]);
#ifndef RELEASE
       sprintf(message, "Playing sample %02x on voice channel (%d)\n", sound_cmd, VOICE_CHANNEL);
       message_emul(0, 2, message);
       if (LOG_ACTIVE) {
         fprintf(LOG, message);
       }
#endif
       current_voice_queue_read_index++;
       if (current_voice_queue_read_index >= SIZE_VOICE_QUEUE) current_voice_queue_read_index = 0;
     }
   }
   
   // check sfx sample playing state
   for (i=MIN_SFX_CHANNEL; i<=MAX_SFX_CHANNEL; i++) {
     GetPlayStatus(Channels[i], &stopped);
     if (stopped) {
       if ((sound_cmd = sound_effect_sample_to_play[i-MIN_SFX_CHANNEL]) != -1) {
#ifndef RELEASE
         sprintf(message, "Playing sample %02x on channel %d\n", sound_cmd, i);
         message_emul(0, 2, message);
         if (LOG_ACTIVE) {
           fprintf(LOG, message);
         }
#endif
         PlayStaticWave(&Channels[i], Samples[sound_cmd]);
         sound_effect_sample_to_play[i-MIN_SFX_CHANNEL] = -1;
       }
     }
   }

}

void stop_voices(void)
{
   int i, i2;
   extern UINT8 ComPort;
   char message[256];

#ifndef RELEASE
   if (LOG_ACTIVE) {
      sprintf(message, "stop sounds\n");
      fprintf(LOG, message);
      message_emul(0, 2, message);
   }
#endif

//   printf("stop voice\n");
   // stop external sample playing
   reset_external_sample_playing();
   StopPlaying(Channels[VOICE_CHANNEL]);
   for (i=MIN_SFX_CHANNEL; i<=MAX_SFX_CHANNEL; i++) {
     StopPlaying(Channels[i]);
   }
}

void set_audio_volume(int volume)
{
   int i;
   for (i=0; i<NUMBER_OF_CHANNELS; i++) SetVol(Channels[i], volume);
   if (!volume) for (i=0; i<10000; i++) UpdateAudio();
}

void CloseAudio()
{
   int i;
   for (i=0; i<NUMBER_OF_CHANNELS; i++) {
     StopPlaying(Channels[i]);
//     ADestroyAudioVoice(Channels[i]);
   }
//   ACloseVoices();
//   for (i=0; i<256; i++) if (sample_ok[i]) AFreeWaveFile(Samples[i]);
   CloseAllChannels();
}

