#include <stdio.h>
#include "host.h"
#include "mz80.h"
#include "prototyp.h"

extern UINT8 pf_state;
extern UINT8 Z80_active;
extern UINT8 *z80data;
extern UINT32 z80data_len;
extern UINT32 z80data_win;
extern UINT32 z80data_win_size;
extern UINT8 z80com;
extern UINT8 *z80com_z80_addr;
extern UINT8 *z80com_68k_addr;
extern UINT8 z80com_len;
extern long speech_rom_size;
extern int speech_rom_numbers;
extern UINT8 uPD7759_active;

UINT8 Z80MEM[0x10000];
UINT8 ComPort;
UINT8 RegIndex;
long MemBank;
UINT8 *MemWin1, *MemWin2;
UINT8 YMStatus=0x00;
int YMStatus_read_times=3;
static UINT8 RegIndexes[4];

static CONTEXTMZ80 *Z80;                    
static int collect_dsp_data;
void OutZ80(UINT16 a, UINT8 v, struct z80PortWrite *self)
{
   extern void (*WriteFMReg)(int,int,int);
   
   switch(a) {
      case 0x00: 
        {
          RegIndex = v;
          return;
        }
      case 0x01: 
        {
//          printf("%02x:%02x\n", RegIndex, v);
          WriteFMReg(0, RegIndex, v);
          return;
        }
      case 0x40:
        {
//          printf("bank %02x\n", v);
          MemBank = v & ~(0xc0);
          return;
        }
      case 0x80:
        {
//          printf("write to dsp %02x\n", v);
          if (collect_dsp_data < 0) { // new DSP session
            collect_dsp_data = 0;
//            printf("cause NMI to Z80\n");
            mz80nmi();
          } else {
            UINT32 offset = v<<8;
            if (speech_rom_size <= 32*1024) {
              // MemBank:
              // b4,b5 = extended rom# (when b3,b2 = 11)
              // b3,b2 = rom# (10=A8 01=A9)
              int n = (MemBank&0x0c) >> 2;
              switch (n) {
              case 1: n=1; break;
              case 2: n=0; break;
              case 3: // use b4,b5
                n = ((MemBank&0x30) >> 4);
                switch (n) {
                case 1: n=3; break;
                case 2: n=2; break;
                default: n=0;
                }
                break;
              default: n=0;
              }
              offset +=  speech_rom_size * n + ( ((MemBank&0x0f)<<14) & (speech_rom_size-1));
            } else {
              // MemBank: 
              // b5,b4 = rom#
              // b3 b2 b1 b0 = 16KB bank#
              // seen as a single 256KB ROM
              offset += ((MemBank&0x0f)<<14);
            }
            
            collect_dsp_data = v;
//            printf("collect DSP value %02x : offset=%08x\n", v, offset);
            if (uPD7759_active) write_uPD7759_pcm_reg(0, offset);
            collect_dsp_data = -1;
          }
          return;
        }
   }
}

UINT16 InZ80(UINT16 a, struct z80PortRead *self)
{
   switch(a) {
      case 0x01: 
        { 
          UINT8 old_status = YMStatus;
          if (!YMStatus_read_times) YMStatus&=~(0x03); else YMStatus_read_times--;
//          printf("%02x\n",old_status);
          return old_status;
        }
        //YMReadReg(0); break;
//      case 0x40: // Out Run and Turbo Out Run
      
      case 0xC0: 
        {
//        printf("ComPort=%02x\n", ComPort);
          return ComPort; // Z80 read communication port
        }
   }
   return 0;
}

UINT16 Sys18_InZ80(UINT16 a, struct z80PortRead *self)
{
   switch(a) {
      case 0x80:
        { 
          UINT8 old_status = YMStatus;
          if (!YMStatus_read_times) YMStatus&=~(0x03); else YMStatus_read_times--;
//          printf("%02x\n",old_status);
          return old_status;
        }
      
      case 0xC0: 
        {
//        printf("ComPort=%02x\n", ComPort);
          return ComPort; // Z80 read communication port
        }
      default:;
//          printf("unknown read %02x\n", a);
        
   }

   return 0;
}

void Sys18_OutZ80(UINT16 a, UINT8 v, struct z80PortWrite *self)
{
   switch(a) {
      case 0x80:  // 1st YM3438
        {
          RegIndexes[0] = v;
          return;
        }
      case 0x81: 
        {
//          printf("0 %02x:%02x\n", RegIndexes[0], v);
  
          if (RegIndexes[0] == 0x28) {
            if (!(v&0x4)) { // 0 1 2
              YM2203Write(0, 0x28, v);
            } else { // 4 5 6
              v = (v&0xf0) | (v&0x3);
              YM2203Write(1, 0x28, v);
            }
          } else 
            YM2203Write(0, RegIndexes[0], v);
          return;
        }
      case 0x82: 
        {
          RegIndexes[1] = v;
          return;
        }
      case 0x83: 
        {
//          printf("1 %02x:%02x\n", RegIndexes[1], v);

          YM2203Write(1, RegIndexes[1], v);
          return;
        }
      case 0x90: // 2nd YM3438
        {
          RegIndexes[2] = v;
          return;
        }
      case 0x91: 
        {
//          printf("2 %02x:%02x\n", RegIndexes[2], v);

          if (RegIndexes[2] == 0x28) {
            if (!(v&0x4)) { // 0 1 2
              YM2203Write(2, 0x28, v);
            } else { // 4 5 6
              v = (v&0xf0) | (v&0x3);
              YM2203Write(3, 0x28, v);
            }
          } else 
            YM2203Write(2, RegIndexes[2], v);
          return;
        }
      case 0x92: 
        {
          RegIndexes[3] = v;
          return;
        }
      case 0x93: 
        {
//          printf("3 %02x:%02x\n", RegIndexes[3], v);

          YM2203Write(3, RegIndexes[3], v);
          return;
        }
      case 0xa0: // select access bank for a000~bfff
        {
//          printf("bank %02x\n", v);
          switch (v&0xc0) {
          case 0x00:
            MemBank = v<<13;
            break;
          case 0x40:
            MemBank = ((v&0x1f) + 128/8)<<13;
            break;
          case 0x80:
            MemBank = ((v&0x1f) + (256+128)/8)<<13;
            break;
          case 0xc0:
            MemBank = ((v&0x1f) + (512+128)/8)<<13;
            break;
          }
          
          //MemWin1 = z80data + MemBank - 0xa000; // 8KB bank
          return;
        }
      default:;
//          printf("unknown %02x:%02x\n", a, v);
     
   }
}

void WrZ80_BASE(UINT32 a, UINT8 v, struct MemoryWriteByte *self)
{
   Z80MEM[a] = v;
}

UINT8 RdZ80_BASE(UINT32 a, struct MemoryReadByte *self)
{
   return Z80MEM[a];
}

/* 4000~7FFF */
UINT8 RdZ80_WIN1(UINT32 a, struct MemoryReadByte *self)
{
   return Z80MEM[a];
}

/* 8000~BFFF */
UINT8 RdZ80_WIN2(UINT32 a, struct MemoryReadByte *self)
{
   return Z80MEM[a];
}

/* A000~BFFF */
UINT8 Sys18_RdZ80_WIN(UINT32 a, struct MemoryReadByte *self)
{
   return (UINT8)(a>>8); //MemWin1[a];
}

void Sys18_WrZ80_pcm_cmd(UINT32 a, UINT8 v, struct MemoryWriteByte *self)
{
   write_sys18_pcm_chip((UINT8)a&0xf, v);
//     printf("%04x:%02x\n", a, v);
}

void Sys18_WrZ80_pcm_sound_ram(UINT32 a, UINT8 v, struct MemoryWriteByte *self)
{
   if ((UINT8)a == 00) {
     write_sys18_pcm_chip(0xff, v);
//     printf("sndram %04x:%02x\n", a ,v);
   }
}

void Sys18_WrZ80_BASE(UINT32 a, UINT8 v, struct MemoryWriteByte *self)
{
   Z80MEM[a] = v;
}

void SEGA3D_WrZ80_pcm_cmd(UINT32 a, UINT8 v, struct MemoryWriteByte *self)
{
   write_sega_pcm_chip(a&0xff, v);

   Z80MEM[a] = v;
}

UINT8 SEGA3D_RdZ80_pcm_cmd(UINT32 a, struct MemoryReadByte *self)
{
   return read_sega_pcm_chip(a&0xff);
}

struct z80PortRead ReadPorts[] =
{
        {0x0,   0xff,   InZ80},

        {(UINT16)-1, (UINT16)-1, NULL}
};

struct z80PortWrite WritePorts[] =
{
        {0x0,   0xff,   OutZ80},

        {(UINT16)-1, (UINT16)-1, NULL}
};


struct z80PortRead Sys18_ReadPorts[] =
{
        {0x0,   0xff,   Sys18_InZ80},

        {(UINT16)-1, (UINT16)-1, NULL}
};

struct z80PortWrite Sys18_WritePorts[] =
{
        {0x0,   0xff,   Sys18_OutZ80},

        {(UINT16)-1, (UINT16)-1, NULL}
};

struct MemoryWriteByte Sys18_WriteMemory[] =
{
//        {0xe000, 0xffff, Sys18_WrZ80_BASE},
        {0xd000, 0xdfff, Sys18_WrZ80_pcm_sound_ram},
        {0xc000, 0xcfff, Sys18_WrZ80_pcm_cmd},
        {(UINT32)-1, (UINT32)-1, NULL}
};

struct MemoryReadByte Sys18_ReadMemory[] =
{
//        {0x0000, 0x9fff, RdZ80_BASE},
//        {0xc000, 0xffff, RdZ80_BASE},
        {0xa000, 0xbfff, Sys18_RdZ80_WIN},

        {(UINT32)-1, (UINT32)-1, NULL}
};

struct MemoryWriteByte SEGA3D_WriteMemory[] =
{
        {0xf000, 0xf0ff, SEGA3D_WrZ80_pcm_cmd},
        {(UINT32)-1, (UINT32)-1, NULL}
};

struct MemoryReadByte SEGA3D_ReadMemory[] =
{
        {0xf000, 0xf0ff, SEGA3D_RdZ80_pcm_cmd},
        {(UINT32)-1, (UINT32)-1, NULL}
};

struct MemoryWriteByte WriteMemory[] =
{
//        {0xc000, 0xffff, WrZ80_BASE},

        {(UINT32)-1, (UINT32)-1, NULL}
};

struct MemoryReadByte ReadMemory[] =
{
//        {0x0000, 0xffff, RdZ80_BASE},
//        {0x0000, 0x3fff, RdZ80_BASE},
//        {0x4000, 0x7fff, RdZ80_WIN1},
//        {0xc000, 0xffff, RdZ80_BASE},
//        {0x8000, 0xbfff, RdZ80_WIN2},

        {(UINT32)-1, (UINT32)-1, NULL}
};

UINT8 z80_readb(UINT32 addr)
{
  struct MemoryReadByte *mr = Z80->z80MemRead;
  while (mr->lowAddr != -1) {
    if (addr >= mr->lowAddr && addr <= mr->highAddr) {
      return mr->memoryCall(addr, mr);
    }
    mr++;
  }
  return Z80->z80Base[0xffff&addr];
}

void z80_writeb(UINT32 addr, UINT8 v)
{
  struct MemoryWriteByte *mr = Z80->z80MemWrite;
  while (mr->lowAddr != -1) {
    if (addr >= mr->lowAddr && addr <= mr->highAddr) {
      mr->memoryCall(addr, v, mr);
      return;
    }
    mr++;
  }
  Z80->z80Base[0xffff&addr] = v;
}

void newz80_init(int type)
{
   if (pf_state) printf("Z80: initialize for type %d subboard\n", type);

   Z80 = (CONTEXTMZ80 *)new_malloc(sizeof(CONTEXTMZ80));
   memset(Z80, 0, sizeof(CONTEXTMZ80));
   MemWin1 = MemWin2 = z80data;

   Z80->z80Base = Z80MEM;         /* memory space where my rom is loaded */
   
   switch (type) {
   case 0: // system16
     Z80->z80IoRead = ReadPorts;
     Z80->z80IoWrite = WritePorts;
     Z80->z80MemRead = ReadMemory;
     Z80->z80MemWrite = WriteMemory;
     collect_dsp_data = -1;
     {
       int i;
       for (i=0; i<0x4000; i++) Z80MEM[0x8000+i] = i>>8;
     }
     break;
   case 1: // system18
     Z80->z80IoRead = Sys18_ReadPorts;
     Z80->z80IoWrite = Sys18_WritePorts;
     Z80->z80MemRead = Sys18_ReadMemory;
     Z80->z80MemWrite = Sys18_WriteMemory;
     break;
   case 2: // Space Harrier
   case 3: // Outrun
     Z80->z80IoRead = ReadPorts;
     Z80->z80IoWrite = WritePorts;
     Z80->z80MemRead = SEGA3D_ReadMemory;
     Z80->z80MemWrite = SEGA3D_WriteMemory;
     break;
   }

   mz80SetContext(Z80);          /* copy my context to the z80 emu context */

   if (pf_state) printf("Z80 Reset!\n");
   mz80reset();
}

void inform_z80()
{
  if (Z80_active == 0) return;
  
  switch (z80com) {
    case 0: mz80int(0); break;
    case 1: *z80com_z80_addr = ComPort; break;
    case 2: *z80com_z80_addr = ComPort; mz80int(0); break;
    case 3: mz80nmi(); break;
    case 4: break; // handled in fastcpu.c
  }
}

