/*
 * SYSTEM 16 ARCADE EMULATOR SOURCE CODE
 *
 * Copyright 1996/97 Thierry Lescot & Li Jih Hwa
 */

#include <stdio.h>
#include <stdlib.h>
#include <allegro.h>
#include "host.h"
#include "main.h"
#include "shinobi.h"

#define DUP_LIMIT 100

/* source,dest,stuff_value */
UINT32 dup_addrs[DUP_LIMIT*3];
int dup_addrs_size=0;

extern FILE *LOG;
extern UINT8 LOG_ACTIVE;

extern UINT8 joy_player, joystick_support;
static int dummy_joy_b0=0;
int *joy_b[7]={&dummy_joy_b0,&joy_b1,&joy_b2,&joy_b3,&joy_b4,&dummy_joy_b0,&dummy_joy_b0};

extern UINT8 *s16_ptr[0x10000];
UINT8 *leds_status, old_leds_status = 0;
UINT8 leds_nflag, leds_cflag, leds_sflag, leds_mode;

extern UINT16 iopl1, iopl2, iopl3, iogen, iods1, iods2;
extern UINT8 ext_bank, vid_bank, txt_bank, ctr_bank, pal_bank;
extern UINT16 txt_offset;
extern BITMAP *vscreen, *v_vscreen;
extern UINT8 vertical_display,mouse_enabled;

int keydef[50] = {
   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
   KEY_F1, KEY_END, KEY_P, KEY_S, KEY_F4, KEY_F5, KEY_F7, KEY_F9, KEY_F11, 127,
};

UINT8 joydef[50] = {
0,0,0,0,0,0,0,0, 0,0,
0,0,0,0,0,0,0,0, 0,0,
0,0,0,0,0,0,0,0, 0,0,
0,0,0,0,0,0,0,0, 0,0,
0,0,0,0,0,0,0,0, 0,0
};

UINT8 dipswitch1=0, dipswitch2=0;

#ifdef SON_SIMULE
SAMPLE *Sons[0x100];
#endif

/* POINTEURS */

UINT8 *ptr_text_video;
extern UINT8 *base_scr_hor_fg, *base_scr_hor_bg, *base_scr_ver_fg, *base_scr_ver_bg;
extern UINT8 *base_scr_pag_fg, *base_scr_pag_bg;
extern UINT8 *base_scr_pag_bg2, *base_scr_hor_bg2, *base_scr_ver_bg2;
extern UINT8 *base_scr_pag_bg3, *base_scr_hor_bg3, *base_scr_ver_bg3;
extern UINT8 *base_scr_hor_bg_splittab0, *base_scr_hor_bg_splittab1;
extern UINT8 *base_spr_regist;
extern UINT8 *base_brq_page;
extern UINT8 *sprites_base_data;

extern UINT8 *mousex_ptr, *mousey_ptr;
extern UINT8 *mouseb_ptr;
extern UINT8 mouseb_msk;

UINT8 *bank_ff_base;
UINT8 *display_base, *vdisplay_base, *text_base, *tiles_base;
UINT8 *display_bottom;
UINT8 *vscreen_base;

extern int trace_play_mode;
extern UINT8 cur_trace_play_state[16];

void NewCheckInput(void) {
   UINT8 v_1p,v_2p,v_3p,v_panel,ds1,ds2;
   //    keydef 0: coin 1
   //     panel 1: coin 2
   //           2: test
   //           3: service
   //           4: 1P
   //           5: 2P
   //           6: 3P
   //           7~9: X
   //   player1 10: B1
   //           11: B2
   //           12: B3
   //           13: x
   //           14: DOWN
   //           15: UP
   //           16: RIGHT
   //           17: LEFT
   //           18~19: x
   //   player2 20~29
   //   player3 30~39
   //   general 40~49

   if (leds_status) {
      if (old_leds_status!=(*leds_status)) {
         old_leds_status = (leds_mode)? *leds_status:~(*leds_status);
         set_leds( ((old_leds_status&leds_nflag)? KB_NUMLOCK_FLAG:0)
                  |((old_leds_status&leds_cflag)? KB_CAPSLOCK_FLAG:0)
                  |((old_leds_status&leds_sflag)? KB_SCROLOCK_FLAG:0));
      }
   }

   v_1p = v_2p = v_3p = v_panel = 0;

   // 1P
   if (iopl1!=0xffff) {
      if (joystick_support && joy_player==1) {
        read_joystick();
        if (joydef[10] && joy[0].button[-1+joydef[10]].b) v_1p|=0x01;
        if (joydef[11] && joy[0].button[-1+joydef[11]].b) v_1p|=0x02;
/*        {
          static UINT8 rensha=0;
          if (v_1p&0x02) {
            if (++rensha & 1) {
              v_1p &= ~0x02;
            }
          }
        }*/
        if (joydef[12] && joy[0].button[-1+joydef[12]].b) v_1p|=0x04;
        if (joydef[13] && joy[0].button[-1+joydef[13]].b) v_1p|=0x08;
        if (joydef[14] && joy[0].button[-1+joydef[14]].b) v_1p|=0x10;
        if (joydef[15] && joy[0].button[-1+joydef[15]].b) v_1p|=0x20;
        if (joydef[16] && joy[0].button[-1+joydef[16]].b) v_1p|=0x40;
        if (joydef[17] && joy[0].button[-1+joydef[17]].b) v_1p|=0x80;
        if (joy[0].stick[0].axis[0].d1) v_1p|=0x80;  // left
        else if (joy[0].stick[0].axis[0].d2) v_1p|=0x40; // right
        if (joy[0].stick[0].axis[1].d2) v_1p|=0x10; // down
        else if (joy[0].stick[0].axis[1].d1) v_1p|=0x20; // up
      }
      if (key[keydef[10]]) v_1p|=0x01; // B1
      if (key[keydef[11]]) v_1p|=0x02; // B2
      if (key[keydef[12]]) v_1p|=0x04; // B3
      if (key[keydef[13]]) v_1p|=0x08; // PLAYER SERVICE IN ALIEN STORM
      if (key[keydef[15]]) v_1p|=0x20; // UP
      else if (key[keydef[14]]) v_1p|=0x10;// DOWN
      if (key[keydef[17]]) v_1p|=0x80;// LEFT
      else if (key[keydef[16]]) v_1p|=0x40;// RIGHT
      v_1p=~v_1p;
      s16_ptr[ctr_bank][iopl1]=v_1p;
   }

   // 2P
   if (iopl2!=0xffff) {
      if (joystick_support && joy_player==2) {
        read_joystick();
        if (joydef[10] && joy[0].button[-1+joydef[10]].b) v_2p|=0x01;
        if (joydef[11] && joy[0].button[-1+joydef[11]].b) v_2p|=0x02;
        if (joydef[12] && joy[0].button[-1+joydef[12]].b) v_2p|=0x04;
        if (joydef[13] && joy[0].button[-1+joydef[13]].b) v_2p|=0x08;
        if (joydef[14] && joy[0].button[-1+joydef[14]].b) v_2p|=0x10;
        if (joydef[15] && joy[0].button[-1+joydef[15]].b) v_2p|=0x20;
        if (joydef[16] && joy[0].button[-1+joydef[16]].b) v_2p|=0x40;
        if (joydef[17] && joy[0].button[-1+joydef[17]].b) v_2p|=0x80;
        if (joy[0].stick[0].axis[0].d1) v_2p|=0x80;  // left
        else if (joy[0].stick[0].axis[0].d2) v_2p|=0x40; // right
        if (joy[0].stick[0].axis[1].d2) v_2p|=0x10; // down
        else if (joy[0].stick[0].axis[1].d1) v_2p|=0x20; // up
      }
      if (key[keydef[20]]) v_2p|=0x01;
      if (key[keydef[21]]) v_2p|=0x02;
      if (key[keydef[22]]) v_2p|=0x04;
      if (key[keydef[23]]) v_2p|=0x08;
      if (key[keydef[25]]) v_2p|=0x20;
      else if (key[keydef[24]]) v_2p|=0x10;
      if (key[keydef[27]]) v_2p|=0x80;
      else if (key[keydef[26]]) v_2p|=0x40;
      v_2p=~v_2p;
      s16_ptr[ctr_bank][iopl2]=v_2p;
   }

   // 3P
   if (iopl3!=0xffff) {
      if (joystick_support && joy_player==3) {
        read_joystick();
        if (joydef[10] && joy[0].button[-1+joydef[10]].b) v_3p|=0x01;
        if (joydef[11] && joy[0].button[-1+joydef[11]].b) v_3p|=0x02;
        if (joydef[12] && joy[0].button[-1+joydef[12]].b) v_3p|=0x04;
        if (joydef[13] && joy[0].button[-1+joydef[13]].b) v_3p|=0x08;
        if (joydef[14] && joy[0].button[-1+joydef[14]].b) v_3p|=0x10;
        if (joydef[15] && joy[0].button[-1+joydef[15]].b) v_3p|=0x20;
        if (joydef[16] && joy[0].button[-1+joydef[16]].b) v_3p|=0x40;
        if (joydef[17] && joy[0].button[-1+joydef[17]].b) v_3p|=0x80;
        if (joy[0].stick[0].axis[0].d1) v_3p|=0x80;  // left
        else if (joy[0].stick[0].axis[0].d2) v_3p|=0x40; // right
        if (joy[0].stick[0].axis[1].d2) v_3p|=0x10; // down
        else if (joy[0].stick[0].axis[1].d1) v_3p|=0x20; // up
      }
      if (key[keydef[30]]) v_3p|=0x01;
      if (key[keydef[31]]) v_3p|=0x02;
      if (key[keydef[32]]) v_3p|=0x04;
      if (key[keydef[33]]) v_3p|=0x08;
      if (key[keydef[35]]) v_3p|=0x20;
      else if (key[keydef[34]]) v_3p|=0x10;
      if (key[keydef[37]]) v_3p|=0x80;
      else if (key[keydef[36]]) v_3p|=0x40;
      v_3p=~v_3p;
      s16_ptr[ctr_bank][iopl3]=v_3p;
   }

   // PANEL
   if (key[keydef[0]] || (joydef[0] && joy[0].button[-1+joydef[0]].b)) v_panel|=0x01; // COIN SWITCH 1
   if (key[keydef[1]] || (joydef[1] && joy[0].button[-1+joydef[1]].b)) v_panel|=0x02; // COIN SWITCH 2
   if (key[keydef[2]] || (joydef[2] && joy[0].button[-1+joydef[2]].b)) v_panel|=0x04; // TEST SWITCH
   if (key[keydef[3]] || (joydef[3] && joy[0].button[-1+joydef[3]].b)) v_panel|=0x08; // SERVICE SWITCH
   if (key[keydef[4]] || (joydef[4] && joy[0].button[-1+joydef[4]].b)) v_panel|=0x10; // 1P START
   if (key[keydef[5]] || (joydef[5] && joy[0].button[-1+joydef[5]].b)) v_panel|=0x20; // 2P START
   if (key[keydef[6]] || (joydef[6] && joy[0].button[-1+joydef[6]].b)) v_panel|=0x40; // 3P START
   if (key[keydef[7]] || (joydef[7] && joy[0].button[-1+joydef[7]].b)) v_panel|=0x80; // NOT USED
   v_panel=~v_panel;

   // Dip switch 1
   ds1=~dipswitch1;
   // Dip switch 2
   ds2=~dipswitch2;

   // analog stick
   {
     extern int analogstick_x,analogstick_y,analogstick_z;
     extern UINT8 *analogstick_x_addr, *analogstick_y_addr, *analogstick_z_addr;
     extern UINT8 analogstick_x_auto,analogstick_y_auto,analogstick_z_auto;
     extern int analogstick_x_center,analogstick_y_center,analogstick_z_center;
     extern int analogstick_x_speed,analogstick_y_speed,analogstick_z_speed;
     extern int analogstick_x_limit,analogstick_y_limit,analogstick_z_limit;
     extern UINT8 reverse_y, reverse_x;

      if ((UINT32)analogstick_x_addr!=0) {
         if (joystick_support==2) {
            /* joystick analogique */
            if (analogstick_x_speed>0) *analogstick_x_addr = ~(char)(0x80+(((joy[0].stick[0].axis[0].pos*1000*(0x80-analogstick_x_limit))/0x80)/1000));
            else *analogstick_x_addr = ~(char)(0x80-(((joy[0].stick[0].axis[0].pos*1000*(0x80-analogstick_x_limit))/0x80)/1000));

            *analogstick_y_addr = ~(char)(0x80+(((joy[0].stick[0].axis[1].pos*1000*(0x80-analogstick_y_limit))/0x80)/1000));

         } else { // keyboard and gamepad
            /* axe des X */
            if ((v_1p&0x80)==0) // left
              analogstick_x+=analogstick_x_speed;
            else if ((v_1p&0x40)==0) // right
              analogstick_x-=analogstick_x_speed;
            else if (analogstick_x_auto) { // auto-centering
              if (analogstick_x>analogstick_x_center) {
                analogstick_x -= analogstick_x_speed>0?analogstick_x_speed:-analogstick_x_speed;
                if (analogstick_x<analogstick_x_center) analogstick_x=analogstick_x_center;
              } else {
                analogstick_x += analogstick_x_speed>0?analogstick_x_speed:-analogstick_x_speed;
                if (analogstick_x>analogstick_x_center) analogstick_x=analogstick_x_center;
              }
              // reproduce wheel handle reading errors for OUTRUN
              {
                static unsigned error=0;
                if (analogstick_x == analogstick_x_center) {
                  analogstick_x -= 1&((error++)>>3);
                }
              }
            }
            if (analogstick_x<analogstick_x_limit) analogstick_x=analogstick_x_limit;
            else if (analogstick_x>0xff-analogstick_x_limit) analogstick_x=0xff-analogstick_x_limit;
            if (analogstick_x_addr) *analogstick_x_addr=analogstick_x;

            /* axe des Y */
            if ((v_1p&0x20)==0) //up
              analogstick_y+=analogstick_y_speed;
            else if ((v_1p&0x10)==0) //down
              analogstick_y-=analogstick_y_speed;
            else if (analogstick_y_auto) { // auto-centering
              if (analogstick_y>analogstick_y_center) {
                analogstick_y -= analogstick_y_speed>0?analogstick_y_speed:-analogstick_y_speed;
                if (analogstick_y<analogstick_y_center) analogstick_y=analogstick_y_center;
              } else {
                analogstick_y += analogstick_y_speed>0?analogstick_y_speed:-analogstick_y_speed;
                if (analogstick_y>analogstick_y_center) analogstick_y=analogstick_y_center;
              }
            }
            if (analogstick_y<analogstick_y_limit) analogstick_y=analogstick_y_limit;
            else if (analogstick_y>0xff-analogstick_y_limit) analogstick_y=0xff-analogstick_y_limit;
            if (analogstick_y_addr) *analogstick_y_addr=analogstick_y;
         }

         if (reverse_y) *analogstick_y_addr = ~(*analogstick_y_addr);
         if (reverse_x) *analogstick_x_addr = ~(*analogstick_x_addr);

         /* axe des Z */
         if ((v_1p&0x08)==0) //forward
           analogstick_z+=analogstick_z_speed;
         else if ((v_1p&0x04)==0) //backward
           analogstick_z-=analogstick_z_speed;
         else if (analogstick_z_auto) { // auto-centering
           if (analogstick_z>analogstick_z_center) {
             analogstick_z -= analogstick_z_speed>0?analogstick_z_speed:-analogstick_z_speed;
             if (analogstick_z<analogstick_z_center) analogstick_z=analogstick_z_center;
           } else {
             analogstick_z += analogstick_z_speed>0?analogstick_z_speed:-analogstick_z_speed;
             if (analogstick_z>analogstick_z_center) analogstick_z=analogstick_z_center;
           }
         }
         if (analogstick_z<analogstick_z_limit) analogstick_z=analogstick_z_limit;
         else if (analogstick_z>0xff-analogstick_z_limit) analogstick_z=0xff-analogstick_z_limit;
         if (analogstick_z_addr) *analogstick_z_addr=analogstick_z;

      }
   }

   s16_ptr[ctr_bank][iogen]=v_panel;
   s16_ptr[ctr_bank][iods1]=ds1;
   s16_ptr[ctr_bank][iods2]=ds2;

   if (mouse_enabled) {
     *mousex_ptr = 0xFF-mouse_x;
     *mousey_ptr = mouse_y;
     *mouseb_ptr = (*mouseb_ptr&(~mouseb_msk))|(((~mouse_b)&0x01) * mouseb_msk);
   }

   //////////////////////////////////////////////////////////////////////////////
   if (trace_play_mode != 0) {
     char ws[256];
     extern UINT8 *analogstick_x_addr, *analogstick_y_addr, *analogstick_z_addr;
     if (trace_play_mode == 1) { // record mode
       cur_trace_play_state[0] = v_1p;
       cur_trace_play_state[1] = v_2p;
       cur_trace_play_state[2] = v_panel;
       cur_trace_play_state[3] = ~0;
       cur_trace_play_state[4] = v_3p;
       if (analogstick_x_addr) cur_trace_play_state[5] = *analogstick_x_addr;
       if (analogstick_y_addr) cur_trace_play_state[6] = *analogstick_y_addr;
       if (analogstick_z_addr) cur_trace_play_state[7] = *analogstick_z_addr;
       if (mouse_enabled) {
         cur_trace_play_state[8] = *mousex_ptr;
         cur_trace_play_state[9] = *mousey_ptr;
         cur_trace_play_state[10] = *mouseb_ptr;
       }
       if (save_next_tpl_state(&cur_trace_play_state) < 0) {
         save_tpl_data();
         trace_play_mode = 0;
         message_emul(0, 2, "END RECORDING");
       }
     } else if (trace_play_mode == 2) { // play back mode
       if (get_next_tpl_state(&cur_trace_play_state) < 0) {
         trace_play_mode = 0;
         message_emul(0, 2, "END PLAYING");
       } else {
         s16_ptr[ctr_bank][iopl1]=cur_trace_play_state[0];
         s16_ptr[ctr_bank][iopl2]=cur_trace_play_state[1];
         s16_ptr[ctr_bank][iogen]=cur_trace_play_state[2];
         if (iopl3 != 0xffff) s16_ptr[ctr_bank][iopl3]=cur_trace_play_state[4];
         if (analogstick_x_addr) *analogstick_x_addr = cur_trace_play_state[5];
         if (analogstick_y_addr) *analogstick_y_addr = cur_trace_play_state[6];
         if (analogstick_z_addr) *analogstick_z_addr = cur_trace_play_state[7];
         if (mouse_enabled) {
           *mousex_ptr = cur_trace_play_state[8];
           *mousey_ptr = cur_trace_play_state[9];
           *mouseb_ptr = cur_trace_play_state[10];
         }
       }
     }
   }

   ///////////////////////////////////////////////////////////////////////////////
   if (dup_addrs_size) { // mirrors
      int i=dup_addrs_size; UINT32 *p=dup_addrs;
      while (i) {
        cpu_writeb(p[1], cpu_readb(p[0]));
        if (p[2]!=-1) cpu_writeb(p[0], (UINT8)p[2]);
        p+=3; i-=3;
      }
   }

}

void Setup_Asm() {
   vscreen_base=(UINT8 *)vscreen->line[0];
   bank_ff_base=s16_ptr[0xFF];
   display_base=&(vscreen->line[88][344]);
   if (vertical_display) vdisplay_base=&(v_vscreen->line[0][0]);
   display_bottom=vscreen->line[312];
   text_base=&s16_ptr[txt_bank][0x002E + txt_offset];
   ptr_text_video=s16_ptr[txt_bank];
}

static struct {
  UINT16 pag_fg,ver_fg,hor_fg;
  UINT16 pag_bg,ver_bg,hor_bg;
  UINT8 sprite[128*16];
} _pre16;

static void save_pre16(void)
{
  _pre16.pag_fg=*(UINT16 *)(base_scr_pag_fg-1);
  _pre16.ver_fg=*(UINT16 *)(base_scr_ver_fg-1);
  _pre16.hor_fg=*(UINT16 *)(base_scr_hor_fg);
  _pre16.pag_bg=*(UINT16 *)(base_scr_pag_bg-1);
  _pre16.ver_bg=*(UINT16 *)(base_scr_ver_bg-1);
  _pre16.hor_bg=*(UINT16 *)(base_scr_hor_bg);
  memcpy(&_pre16.sprite[0], (UINT8 *)base_spr_regist, sizeof(_pre16.sprite));
}

static void restore_pre16(void)
{
  *(UINT16 *)(base_scr_pag_fg-1)=_pre16.pag_fg;
  *(UINT16 *)(base_scr_ver_fg-1)=_pre16.ver_fg;
  *(UINT16 *)(base_scr_hor_fg)=_pre16.hor_fg;
  *(UINT16 *)(base_scr_pag_bg-1)=_pre16.pag_bg;
  *(UINT16 *)(base_scr_ver_bg-1)=_pre16.ver_bg;
  *(UINT16 *)(base_scr_hor_bg)=_pre16.hor_bg;
  memcpy((UINT16 *)base_spr_regist, &_pre16.sprite[0], sizeof(_pre16.sprite));
}

#define SWAP_NIBBLE(B) (((B)<<4)|((B)>>4))
#define SWAP_BYTE(W) (((W)<<8)|((W)>>8))
void translate_pre16(void)
{
  int i;
  extern UINT8 pre_16_flag;
  UINT8 *spr=(UINT8 *)base_spr_regist;
  UINT16 *sprW=(UINT16 *)base_spr_regist;
  UINT8 tmp;
  unsigned zoom,zoomx,zoomy;
  static UINT8 pattern_offsets[32768]={0xff};

  save_pre16();

  switch (pre_16_flag) {
    case 1: // QUATRE 2
      {
        *(UINT8 *)base_scr_pag_fg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_fg);
        *(UINT8 *)(base_scr_pag_fg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_fg-1));
        *(UINT8 *)base_scr_pag_bg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_bg);
        *(UINT8 *)(base_scr_pag_bg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_bg-1));

        *(UINT8 *)(base_scr_ver_fg-1) = 0;
        *(UINT8 *)(base_scr_ver_bg-1) &= 1;
        *(UINT8 *)(base_scr_hor_fg) &= 1;
        *(UINT8 *)(base_scr_hor_bg) &= 1;

        if ( (0xff&cpu_readb(0xc40005)) == 1) { // for title screen
          *(UINT16 *)(base_scr_hor_fg) = 0x0080;
          *(UINT16 *)(base_scr_hor_bg) = 0x0080;
          *(UINT16 *)(base_scr_ver_fg-1) = 0x0080;
          *(UINT16 *)(base_scr_ver_bg-1) = 0x0080;
        }

        for (i=0; i<64; i++,spr+=0x10) { // sprites
          if (spr[0]==0 || spr[0]==0xff || spr[1]>=spr[0]) continue;
          spr[0]+=1; spr[1]+=1; // down 1 lines

          tmp=spr[9];

          spr[9] = ((tmp&0x3)<<6) | (spr[8]&0x3f); // priority and palette
          spr[8] = tmp>>4; // bank

          spr[10] = spr[11] = 0;

          if ((spr[9]&0x3f)==0x3f) {
            spr[9]=(spr[9]&0xc0)|(spr[8]<<1);
          }

          if ((spr[6]&0x7f)==0x7f && (spr[7]&0x80)==0x80) {
            spr[8]=(spr[8]-1)&0xf; // to last bank
            spr[6]^=0x80;
          }

          spr[4] = 0;
          if (spr[6]&0x80) { // reverse
            spr[4] |= 1;
            spr[6] &= 0x7f;
          }
          spr[2]&=3;

        }

      }
      return;

    case 2: // HANG ON
      {
        *(UINT8 *)base_scr_pag_fg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_fg);
        *(UINT8 *)(base_scr_pag_fg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_fg-1));
        *(UINT8 *)base_scr_pag_bg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_bg);
        *(UINT8 *)(base_scr_pag_bg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_bg-1));
        *(UINT8 *)(base_scr_ver_fg-1) = 0;
        *(UINT8 *)(base_scr_ver_bg-1) &= 1;
        *(UINT8 *)(base_scr_hor_fg) &= 1;
        *(UINT8 *)(base_scr_hor_bg) &= 1;

        for (i=0; i<64; i++,spr+=0x10) { // sprites
          if (spr[0]==0 || spr[0]==0xff || spr[1]>=spr[0]) continue;

          tmp=spr[9];

          spr[9] = 0xc0 | /*((tmp&0x3)<<6) |*/ (spr[8]&0x3f); // priority and palette
          spr[8] = spr[2]>>4; // bank

          zoom = (tmp>>2)*(1024/64);
          zoomx = zoom;
          zoomy = (1060*zoom)/(2048-zoom);

          spr[10]=(zoomx&0xff00)>>8;
          spr[11]=(zoomx&0x00ff);
          spr[12]=(zoomy&0xff00)>>8;
          spr[13]=(zoomy&0x00ff);

          if ((spr[9]&0x3f)==0x3f) spr[9]=(spr[9]&0xc0)|(spr[8]<<1);

          if ((spr[6]&0x7f)==0x7f && (spr[7]&0x80)==0x80) {
            spr[8]=(spr[8]-1)&0xf; // to last bank
            spr[6]^=0x80;
          }

          if (spr[6]&0x80) { // reverse
            spr[4] |= 1;
            spr[6] &= 0x7f;
          }

          spr[2]&=3;
        }

      }
      return;


    case 3: // SPACE HARRIER
      {
        if (pattern_offsets[0]==0xff) { // initialize pattern_offsets array
          memset(pattern_offsets, 0xee, sizeof(pattern_offsets));
          /* pre-input pattern offset value */
          /* even addr: upper 4 bits  odd addr: lower 4 bits */
          pattern_offsets[0x2124>>1] = 0x0e; // small shadow
          pattern_offsets[0x2429>>1] = 0xe4; // ice berg of round 7
          pattern_offsets[0x211b>>1] = 0xe1; // small flying rock
          pattern_offsets[0x515b>>1] = 0xe0; // small flying ball
          pattern_offsets[0x611f>>1] = 0xe0; // small ceiling ball
          pattern_offsets[0x624a>>1] = 0x1e; // small ceiling ball
          pattern_offsets[0x5785>>1] = 0xe1; // 3 poses of the Harrier on the title screen
          pattern_offsets[0x5771>>1] = 0xe1; // these are the only patterns which do not need
          pattern_offsets[0x579a>>1] = 0x1e; // position compensations
          pattern_offsets[0x06f3>>1] = 0xe0; // missiles
          pattern_offsets[0x0735>>1] = 0xe0;
        }

        *(UINT8 *)base_scr_pag_fg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_fg);
        *(UINT8 *)base_scr_pag_bg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_bg);

        *(UINT8 *)(base_scr_ver_fg-1) &= 1;
        *(UINT8 *)(base_scr_ver_bg-1) &= 1;
        *(UINT8 *)(base_scr_hor_fg) &= 1;
        *(UINT8 *)(base_scr_hor_bg) &= 1;

        { // fake transparent shadow
          int i; UINT16 c=cpu_readw(0x110072);
          UINT16 c0=c&0x000f;
          UINT16 c1=c&0x00f0;
          UINT16 c2=c&0x0f00;
          UINT16 c3=c&0xf000;
          c = ( c0>4 ?c0-4 :0) | (c1>0x40 ?c1-0x40 :0) | (c2>0x400 ?c2-0x400 :0) | c3;
          for (i=0x110802; i<0x11081e; i+=2) {
            cpu_writew(i, c);
          }
        }

        for (i=0; i<64; i++,spr+=0x10) { // sprites

          if (spr[0]==0 || spr[0]==0xff || spr[1]>=spr[0]) continue;

          tmp=spr[9];
          spr[9] = spr[4]|0x80; // priority and palette
          spr[8] = spr[2]>>4; // bank

          // SYSTEM16 ZOOM RATE: X => (2048-w)/(2048) w:0~1023 linear
          //                     Y => (1024)/(1024+w) w:0~1023 non-linear
          // SPACE HARRIER ZOOM RATE : both X,Y are (128-w)/(128) w:0~63

          zoom = (tmp&0x3f)*(1024/64);
          zoomx = zoom;
          zoomy = (1024*zoom)/(2048-zoom);

          spr[10]=(zoomx&0xff00)>>8;
          spr[11]=(zoomx&0x00ff);
          spr[12]=(zoomy&0xff00)>>8;
          spr[13]=(zoomy&0x00ff);

          spr[4]=0;

          if ((spr[9]&0x3f)==0x3f) spr[9]=(spr[9]&0xc0)|(spr[8]<<1);

          if ((spr[6]&0x7f)==0x7f && (spr[7]&0x80)==0x80) {
            spr[8]=(spr[8]-1)&0xf; // to last bank
            spr[6]^=0x80;
          }

          if (spr[6]&0x80) { // reverse
            spr[4] |= 1;
            spr[6] &= 0x7f;
          }

          spr[2]&=3;

          if ( (!(spr[4]&1)) && (!(spr[5]&0x80)) &&spr[5]>0)
          // compute pattern offset
          {
            unsigned bank=spr[8];
            unsigned addr=(spr[6]<<8)|spr[7];
            unsigned offset_addr= (bank<<11) | (addr>>5);
            char rebuild = 0;

            if ((addr>>4)&1) {
              if ((pattern_offsets[offset_addr]&0x0f)!=0x0e) {
                spr[14] = pattern_offsets[offset_addr]&0xf;
              } else rebuild = 1;
            } else {
              if ((pattern_offsets[offset_addr]&0xf0)!=0xe0) {
                spr[14] = pattern_offsets[offset_addr] >> 4;
              } else rebuild = 1;
            }

            if (rebuild) { // build pattern offset
              unsigned width=spr[5];
              UINT8 *p,*p0;
              extern UINT8 spr_tab[];
              unsigned height=(spr[0]-spr[1])*(1024+zoomy)/1024;
              unsigned width_bytes;
              int len_pattern;
              int i,j;
              UINT8 offset;

              len_pattern = height*width;
              if (len_pattern < 16) {
                continue;
              }

              len_pattern >>= 5;

              // 32 bit sprite hardware ///////////////////////////////////////////
              width_bytes = width<<2;
              p0 = (UINT8 *)sprites_base_data + (spr_tab[bank]<<17) + (((addr+width)&0xffff)<<2);
              /////////////////////////////////////////////////////////////////////

              offset=0xe;
              for (i=0, p=p0; i<height; i++, p+=width_bytes) {
                for (j=0; j<width_bytes && j<offset; j++) {
                  if (p[j]!=0xf0 && p[j]!=0x00 && p[j]!=0x0f) break;
                }
                if (j<offset) offset = j;
              }

//              printf("rebuild %02x:%04x = %d\n", bank, addr, offset);

              p=&pattern_offsets[offset_addr];

              if ((addr>>4) & 1) { *p = (*p&0xf0) | offset; p++; }
              else { *p = (*p&0x0f) | (offset<<4); }

              for (i=1; i<len_pattern; i++,p++) *p = offset|(offset<<4);

              spr[14] = offset;

//              printf("v=%04x addr=%04x bank=%02x offset=%d height=%d zoomy=%04x pal=%02x len_pattern=%d width=%d offset_addr=%04x\n", spr-base_spr_regist,addr, bank, offset, height, zoomy, spr[9]&0x3f, len_pattern, width, offset_addr);
            }
          }

        }

      }
      return;

    case 4: // OUTRUN
      {
        UINT8 *sa=(UINT8 *)base_spr_regist;
        UINT8 *sb=(UINT8 *)&_pre16.sprite[0];
        UINT16 w,x;

        for (i=0; i<128; i++, sa+=16, sb+=16) {
          if (sb[0]==0xff && sb[1]==0xff) { // end mark
            sa[0]=sa[1]=0xff;
            break;
          }
          sa[0]=sb[1]+(sb[0xa])+1;
          sa[1]=sb[1];          // y
          sa[2]=sb[4]&0x1;      // x
          sa[2]++;
          sa[3]=sb[5];
          x=(((UINT16)sa[0x2])<<8)|sa[0x3];
          sa[4]=0;//(sb[2]&0x80)!=0; // rev bit
          sa[5]=(sb[0x4])>>1;   // width
          sa[6]=sb[2]&0x7f; // address
          sa[7]=sb[3];
          sa[8]=sb[0]&0x7;      //bank
          sa[9]=(0x7f&sb[0xb]);   //pal
          w=(((UINT16)(sb[0x8]&0xf))<<8)|(sb[0x9]);

          if (w==0x32c) w=0x3cf;

          if (w>=0x200) w=(w-0x200)*2; else w=(0x200-w)*2+1024;
          sa[0xa]=w>>8;            // zoom
          sa[0xb]=w&0xff;
          if (sb[2]&0x80) sa[8]=(sa[8]+1)&0x7;
          if (sb[0xb]==0 && sa[6]==0) sa[0]=0; // no shadows
          if (sb[0]&0x40) sa[0]=0;

          if (sa[0]) {
            if(!(sb[8]&0x20)) { // ?0
              if (!(sb[8]&0x40)) { // 00: RIGHTUP, normal

                sa[7]-=sa[5];
                if (sb[3]<sa[5]) sa[6]--;

                if (sa[7]<sa[5]-1) sa[6]--;
                sa[7]-=sa[5]-1;

                if (w<1024) { // zoom down
                  sa[4]=2;
                } else { // zoom up
                  sa[4]=2;
                }
                sa[2]=x>>8; sa[3]=x&0xff;

              } else { // 10: RIGHTUP, rev
                unsigned ofs;
                if (w<1024) { // zoom down
                  ofs=((unsigned)sa[5]*8)*1024/(1024+w);
                  x -= ofs;
                } else { // zoom up
                  ofs = ((unsigned)sa[5]*8)*1024/(1024-(w-1024));
                  x -= ofs;
                }

                // x position compensation for rocks in round2R and round4RRR
                if (sb[2]==0xc0 && sb[3]==0x27 && sb[0]==3) {
                  if (sb[4]==0x59) x += ofs/2;
                  else if (sb[4]>0x59) x += ofs;
                } else if (sb[2]==0xcf && sb[3]==0x73 && sb[0]==3) {
                  if (sb[4]==0x2d) x += ofs/2;
                  else if (sb[4]>0x2d) x += ofs;
                } else if (sb[2]==0xd3 && sb[3]==0x46 && sb[0]==3) {
                  if (sb[4]==0x19) x += ofs/2;
                  else if (sb[4]>0x19) x += ofs;
                } else if (sb[2]==0xd4 && sb[3]==0x4e && sb[0]==3) {
                  if (sb[4]==0x0d) x += ofs/2;
                  else if (sb[4]>0x0d) x += ofs;
                } else if (sb[2]==0xd4 && sb[3]==0x90 && sb[0]==3) {
                  if (sb[4]==0x09) x += ofs/2;
                  else if (sb[4]>0x09) x += ofs;
                }

                sa[2]=x>>8; sa[3]=x&0xff;

                if (sa[7]<1) sa[6]--;
                sa[7]-=1;

                sa[4]=1;
              }

            } else { // ?1
              if ((sb[8]&0x40)) { // 11
                sa[7]-=sa[5];
                if (sb[3]<sa[5]) sa[6]--;
                w=(((UINT16)(sa[0x2]&0x1))<<8)|(sa[0x3]);

                if (sb[8]&0x80) { // patch for car shadow position
                  if (sb[8]==0xe1 && sb[9]==0xa9 && sb[2]==0x58 && sb[3]==0x17) {
                    x-=4;
                    sa[2]=x>>8; sa[3]=x&0xff;
                  }
                }

              } else { // 01 : LEFTUP, rev
                sa[7]-=sa[5];
                if (sb[3]<sa[5]) sa[6]--;
                sa[4]=1;
              }
            }
            sa[2]&=0x3;
          }
        }

      }

      return;

    case 5: // PASSING SHOT
      {
        UINT16 *sa=(UINT16 *)base_spr_regist;
        UINT16 *sb=(UINT16 *)&_pre16.sprite[0];

        for (i=0; i<64; i++, sa+=8, sb+=8) {
          sa[0] = sb[1];
          if (sa[0]==0xffff) sa[0]=0xff00;
          sa[1] = sb[0];
          sa[2] = sb[3];
          sa[3] = sb[2];
          sa[4] = sb[5];
          sa[5] = sb[4];
          sa[6] = sb[7];
          sa[7] = sb[6];

          ((UINT8*)sa)[8] = ((UINT8*)sb)[11]>>4;
          ((UINT8*)sa)[4] = 0;

          if (((UINT8*)sa)[6]&0x80) {
            ((UINT8*)sa)[8]--; ((UINT8*)sa)[8]&=0xf; ((UINT8*)sa)[4] = 1;
          }

          if ( ((UINT8*)sa)[1] != 0x00 ) {
            ((UINT8*)sa)[0]+=2;
            ((UINT8*)sa)[1]+=2;
          }

          if ((sb[3]&0x8000) == 0) {
            UINT16 addr = (((UINT8*)sa)[6]<<8) | ((UINT8*)sa)[7];
            addr -= ((UINT8*)sa)[5]&0x7f;
            ((UINT8*)sa)[6] = addr>>8;
            ((UINT8*)sa)[7] = addr&0xff;
          } else {
            UINT16 addr = (((UINT8*)sa)[6]<<8) | ((UINT8*)sa)[7];
            addr -= (int)(((char*)sa)[5]);
            ((UINT8*)sa)[6] = addr>>8;
            ((UINT8*)sa)[7] = addr&0xff;
          }

          ((UINT8*)sa)[9] = ((UINT8*)sb)[10];

        }
      }
      return;

    case 6: // DYNAMAITE DUK
      {
        // BG page selectors
        {
          UINT8 lu = cpu_readb(0xc46021);
          UINT8 ru = cpu_readb(0xc46023);
          UINT8 ld = cpu_readb(0xc46025);
          UINT8 rd = cpu_readb(0xc46027);

          if (lu==4 && ld==4 && ru==5 && rd==5) { // fix a bug in chicago round
            UINT16 vs=cpu_readw(0xfff6ec);
            cpu_writew(0xc46000, vs&0xff);
            cpu_writew(0xc46010, vs&0xff);
            if (vs >= 0x100) {
              lu=0x26; ru=0x37;
              ld=0x04; rd=0x15;
            } else {
              ld=0x26; rd=0x37;
              lu=0x04; ru=0x15;
            }
          }
          cpu_writeb(0xc46126, (rd&0xf)|((ld&0xf)<<4));
          cpu_writeb(0xc46122, (rd>>4)|((ld>>4)<<4));
          cpu_writeb(0xc46127, (ru&0xf)|((lu&0xf)<<4));
          cpu_writeb(0xc46123, (ru>>4)|((lu>>4)<<4));
        }
        *(UINT8 *)(base_scr_hor_fg) &= 1;
        *(UINT8 *)(base_scr_hor_bg) &= 1;

        // sprites
        {
          UINT8 *sa=(UINT8 *)base_spr_regist;
          UINT8 *sb=(UINT8 *)&_pre16.sprite[0];

          // this game uses some unknown method to decide a zoomed sprite should get its x positon compensated or not, I just added this workaround.
          for (i=0; i<64; i++, sa+=16, sb+=16) {
            static const UINT16 check_addr[]={
              0xf1ad, 0xf2d9, 0xf0fe, // water springs
              0xea35,                 // hopping rabbits

            };
            UINT16 addr = (sb[6]<<8)|sb[7];
            int j;

//            if (sa[9]==0xff) sa[9]=0xbf; // shadow

            for (j=0; j<sizeof(check_addr)/sizeof(UINT16); j++) {
              if (addr == check_addr[j]) { // need x compensation
                UINT16 z = (sb[10]<<8) | sb[11];
                if (z != 0) {
                  UINT16 x = (sb[2]<<8) | sb[3];
                  int delta = 4*(sb[5]&0x7f)*z/2048/2;
                  x += delta;
                  sa[2] = x>>8;
                  sa[3] = x&0xff;
                }
                break;
              }
            }
          }
        }
      }
      return;

    case 7: // Aurail
      {
         for (i=0; i<64; i++, sprW+=8, spr+=16) {
            if (sprW[2]==0x0080) { // end of sprite list
               sprW[2] = 0xFFFF;
               return;
            }
            if (spr[9]==0x3f) spr[0]=spr[1]=0; // disable sprite mask
         }
         return;
      }
      return;

    case 8: // Fantasy Zone
      {
        extern UINT8 *sprites_base_data;
        sprites_base_data[0x4f*2+1] = 0x0f; // patch laser beam sprite data for a correct display

        *(UINT8 *)base_scr_pag_fg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_fg);
        *(UINT8 *)(base_scr_pag_fg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_fg-1));
        *(UINT8 *)base_scr_pag_bg = SWAP_NIBBLE(*(UINT8 *)base_scr_pag_bg);
        *(UINT8 *)(base_scr_pag_bg-1) = SWAP_NIBBLE(*(UINT8 *)(base_scr_pag_bg-1));

        *(UINT8 *)(base_scr_ver_fg-1) = 0;
        *(UINT8 *)(base_scr_ver_bg-1) &= 1;
        *(UINT8 *)(base_scr_hor_fg) &= 1;
        *(UINT8 *)(base_scr_hor_bg) &= 1;

        // solving Fantasy Zone scrolling bug
        {
          UINT8 t,td,t1,t2;
          if (*(UINT8 *)(base_scr_hor_fg)==0 && *(UINT8 *)(base_scr_hor_fg+1)<16) {
            t  = *(UINT8 *)base_scr_pag_fg;
            td = *(UINT8 *)(base_scr_pag_fg-1);
            t1 = t & 0x0f;
            t2 = t & 0xf0;
            if (t1<=0x03 && t2<=0x30 && t == td) {
              switch (t) {
              case 0x30: t = 0x10; break;
              case 0x23: t = 0x03; break;
              case 0x12: t = 0x32; break;
              case 0x01: t = 0x21; break;
              }
              *(UINT8 *)base_scr_pag_fg = *(UINT8 *)(base_scr_pag_fg-1) = t;
            }
          }
        }

        for (i=0; i<64; i++,spr+=0x10) { // sprites
          if (spr[0]==0 || spr[0]==0xff || spr[1]>=spr[0]) continue;
          spr[0]+=1; spr[1]+=1; // down 1 lines

          if (i==5 && spr[8]==0x18 && spr[9]==0x21) spr[9]=0x22; // tears fix for ending boss

          tmp=spr[9];

          spr[9] = ((tmp&0x3)<<6) | (spr[8]&0x3f); // priority and palette
          spr[8] = tmp>>4; // bank

          spr[10] = spr[11] = 0;

          if ((spr[9]&0x3f)==0x3f) { // shadow sprite
            spr[0]=spr[1]=0; // disable them
          }

          if ((spr[6]&0x7f)==0x7f && (spr[7]&0x80)==0x80) {
            spr[8]=(spr[8]-1)&0xf; // to last bank
            spr[6]^=0x80;
          }

          spr[4] = 0;
          if (spr[6]&0x80) { // reverse
            spr[4] |= 1;
            spr[6] &= 0x7f;
          }
          spr[2]&=3;

        }

      }
      return;

  }


}

void untranslate_pre16(void)
{

  restore_pre16();

}


short math_readw_func(UINT32 addr)
{
  // DIV
  if ( (addr&0xffff)==0x0008 )
   return ( (int)cpu_readl((addr&0xff0000)|0x4000) ) / ( (int)(short)(cpu_readw((addr&0xff0000)|0x4004)&0xffff) );
  else if ( (addr&0xffff)==0x800e )
   return ( (int)cpu_readl((addr&0xff0000)|0x8000) ) / ( (int)(short)(cpu_readw((addr&0xff0000)|0x8004)&0xffff) );
  else if ( (addr&0xffff)==0x8006 )
   return ( (int)cpu_readl((addr&0xff0000)|0x8000) ) % ( (int)(short)(cpu_readw((addr&0xff0000)|0x8004)&0xffff) );

  return 0;
}

int math_readl_func(UINT32 addr)
{
  // MUL
  return ( (int)(short)((UINT16)cpu_readw((addr&0xff0000)|0x0000)&0xffff) ) * ( (int)(short)((UINT16)cpu_readw((addr&0xff0000)|0x0002)&0xffff) );

}

