#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <allegro.h>
#include "host.h"
#include "main.h"
#include "starcpu.h"

#define MAX_TPL_DATA     0x40000

#define TPL_CMD_SKIP     0
#define TPL_CMD_PLAYER1  1
#define TPL_CMD_PLAYER2  2
#define TPL_CMD_PANEL1   3
#define TPL_CMD_PANEL2   4
#define TPL_CMD_PLAYER3  5
#define TPL_CMD_ANALOG_X  6
#define TPL_CMD_ANALOG_Y  7
#define TPL_CMD_ANALOG_Z  8
#define TPL_CMD_MOUSE_X 9
#define TPL_CMD_MOUSE_Y 10
#define TPL_CMD_MOUSE_B 11
#define TPL_CMD_END 15

static struct tpl_struct {
  UINT8 *data;
  UINT8 *cur_data;
  int length_data;
  int remaining;
  UINT8 input_state[16];
  int need_to_skip;
} tpl;

/* input_state:
[0] : player1
[1] : player2
[2] : panel1
[3] : panel2
[4] : player3
[5] : analog x
[6] : analog y
[7] : analog z
[8] : mouse x
[9] : mouse y
[10]: mouse z
*/

static PACKFILE *tpl_state_file;

/* return 0 if succ */
int init_tpl()
{
  memset(&tpl, 0, sizeof(tpl));
  if ((tpl.data = (UINT8 *)malloc(MAX_TPL_DATA)) == NULL) return -1;
  memset(tpl.data, 0, MAX_TPL_DATA);
  tpl_state_file = NULL;
  return 0;
}

int quit_tpl()
{
  if (tpl_state_file) pack_fclose(tpl_state_file);
  free(tpl.data);
  return 0;
}

inline static int length_tpl_cmd(UINT8 cmd)
{
  cmd &= 0xf0;
  if (cmd < 0x80) return (cmd>>4); // 0~7 = 0~7
  else return ( ((cmd>>4)-7) << 4); // 8~f = 16~128
}

inline static UINT8 combine_tpl_cmd(UINT8 cmd, int length)
{
  if (length >= 8) length = (length+15) >> 4;
  return cmd | (length<<4); // b7~b4 : length, b3~b0 : command
}

/* return NULL if failed */
inline static UINT8 *reserve_tpl_code_space(int length)
{
  UINT8 *old_data = tpl.cur_data;
  if (length > tpl.remaining) return NULL;
  tpl.cur_data += length;
  tpl.remaining -= length;
  tpl.length_data += length;
  return old_data;
}

/* return NULL if failed */
inline static UINT8 *fetch_tpl_code()
{
  if (tpl.remaining > 0) {
    UINT8 code = *tpl.cur_data;
    UINT8 *old_data = tpl.cur_data;
    int length = length_tpl_cmd(code);
//printf("len=%d code=%02x\n", length, code);

    tpl.cur_data += length;
    tpl.remaining -= length;

    return old_data;
  } else return NULL;
}

/* return -1 if failed */
/* should call exactly once after 1 int */
int get_next_tpl_state(UINT8 *next_state)
{
  if (tpl.need_to_skip > 0) {
    tpl.need_to_skip--;
    // no state changed
  } else {
    UINT8 *code;
    unsigned cur_code;
    while (1) {
      if ((code  = fetch_tpl_code()) == NULL) return -1; // end of tpl data stream
      cur_code = (*code)&0x0f;
      switch (cur_code) {
      case TPL_CMD_SKIP:
        tpl.need_to_skip = code[1] - 1;
        return 0;

      case TPL_CMD_END:
        // end of packet
        return 0;

      default:
        next_state[cur_code-1] = code[1];
//        printf("get %02x : %02x\n", code[0], code[1]);
      }
    }
  }
  return 0;
}

/* return -1 if failed */
/* should call exactly once after 1 int */
int save_next_tpl_state(UINT8 *next_state)
{
  int i;
  if (memcmp(tpl.input_state, next_state, sizeof(tpl.input_state)) == 0) { // no change

    if (++tpl.need_to_skip >= 256) { // skip counter overflow
      // output SKIP command
      UINT8 *code;
      if ((code=reserve_tpl_code_space(2)) == NULL) return -1;
      *code++ = combine_tpl_cmd(TPL_CMD_SKIP, 2);
      *code = 255;
      tpl.need_to_skip -= 255;
//      printf("save skip %02x %02x\n", code[-1], code[0]);
    }

  } else { // changes found: save the differences
    UINT8 *code;
    if (tpl.need_to_skip > 0) {
      // output SKIP command
      if ((code=reserve_tpl_code_space(2)) == NULL) return -1;
      *code++ = combine_tpl_cmd(TPL_CMD_SKIP, 2);
      *code = (UINT8)tpl.need_to_skip;
      tpl.need_to_skip = 0;
//      printf("save skip %02x %02x\n", code[-1], code[0]);
    }
    
    for (i=0; i<sizeof(tpl.input_state); i++) {
      if (tpl.input_state[i] != next_state[i]) {
        // output PLAYER command
        if ((code=reserve_tpl_code_space(2)) == NULL) return -1;
        *code++ = combine_tpl_cmd(i+1, 2);
        *code = (UINT8)next_state[i];
//        printf("save %02x %02x\n", code[-1], code[0]);
      }
    }
    
    // output END OF PACKET command
    if ((code=reserve_tpl_code_space(1)) == NULL) return -1;
    *code = combine_tpl_cmd(TPL_CMD_END, 1);
//    printf("save end of packet\n");

    memcpy(tpl.input_state, next_state, sizeof(tpl.input_state));
    
  }
  
  return 0;
}

extern char movie_path[];
extern long sound_driver_magic;
extern UINT8 current_bgm;
/* variables used by Save/Load game */
extern UINT8 *s16_ptr[0x10000];
extern UINT8 fakeio, vid_bank, pal_bank, txt_bank, spr_bank, reset_palette;

static int SaveGame(char *fn, char *desc)
{
   extern int num_cpus;
   char game_desc[61], full_path[256];

   if (tpl_state_file != NULL) {
     pack_fclose(tpl_state_file);
     tpl_state_file = NULL;
   }

   sprintf(game_desc, "%s", desc);
   sprintf(full_path, "%s%s", movie_path, fn);
   
   if (num_cpus<=1) { // normal system16/18 game
      UINT8 cpu_savebuf[256];
      PACKFILE *SG;
      extern int save_extra[];
      int i;

      if ((SG=pack_fopen(full_path,F_WRITE_PACKED)) != NULL) {
         pack_fwrite(game_desc, 60, SG);
         save_cpu_content(cpu_savebuf);
         pack_fwrite(cpu_savebuf, sizeof(cpu_savebuf), SG);
         pack_fwrite(s16_ptr[vid_bank], 65536, SG);
         pack_fwrite(s16_ptr[pal_bank], 65536, SG);
         pack_fwrite(s16_ptr[txt_bank], 4096, SG);
         pack_fwrite(s16_ptr[spr_bank], 4096, SG);
         pack_fwrite(s16_ptr[0xFF], 65536, SG);
         for (i=0; i<save_extra[0]; i++) {
            pack_fwrite(s16_ptr[(save_extra[1+i*2]&0xff0000)>>16] + (save_extra[1+i*2]&0xffff), save_extra[1+i*2+1], SG);
         }
         pack_fwrite(&sound_driver_magic, sizeof(sound_driver_magic), SG);
         pack_fwrite(&current_bgm, sizeof(current_bgm), SG);

         tpl_state_file = SG;
         // do not close file

      } else return -1;

   } else { // multi-cpu game
      PACKFILE *f;
      extern int save_extra[];
      extern UINT16 txt_offset;
      UINT8 cpu_savebuf[256];
      int old_cpu;
      int i;

      if ((f=pack_fopen(full_path,F_WRITE_PACKED)) != NULL) {
         pack_fwrite(game_desc, 60, f);
         old_cpu=context_switch(0);
         for (i=0; i<num_cpus; i++) {
            context_switch(i);
            save_cpu_content(cpu_savebuf);
            pack_fwrite(cpu_savebuf, sizeof(cpu_savebuf), f);
         }
         pack_fwrite(s16_ptr[vid_bank], 0x10000, f);
         pack_fwrite(s16_ptr[pal_bank], 0x4000, f);
         pack_fwrite(s16_ptr[txt_bank]+txt_offset, 0x1000, f);
         pack_fwrite(s16_ptr[spr_bank], 0x4000, f);
         pack_fwrite(s16_ptr[0xFF], 0x10000, f);
         context_switch(0);
         for (i=0; i<save_extra[0]; i++) {
            pack_fwrite(s16_ptr[(save_extra[1+i*2]&0xff0000)>>16] + (save_extra[1+i*2]&0xffff), save_extra[1+i*2+1], f);
         }
         pack_iputl(sound_driver_magic, f);
         pack_fwrite(&current_bgm, sizeof(current_bgm), f);

         tpl_state_file = f;
         // do not close file

         context_switch(old_cpu);
      } else return -1;
   }
   return 0;
}

void sound_request(UINT8 sound_cmd);
static int LoadGame(char *fn)
{
   extern int num_cpus;
   extern int save_extra[];
   char game_desc[61], full_path[256];

   if (tpl_state_file != NULL) {
     pack_fclose(tpl_state_file);
     tpl_state_file = NULL;
   }
 
   sprintf(full_path, "%s%s", movie_path, fn);

   if (num_cpus<=1) { // normal system16/18 game
      UINT8 cpu_savebuf[256];
      PACKFILE *SG;
      int i;

      if ((SG=pack_fopen(full_path, F_READ_PACKED)) != NULL) {
         pack_fread(game_desc, 60, SG);
         pack_fread(cpu_savebuf, sizeof(cpu_savebuf), SG);
         load_cpu_content(cpu_savebuf);
         pack_fread(s16_ptr[vid_bank], 65536, SG);
         pack_fread(s16_ptr[pal_bank], 65536, SG);
         pack_fread(s16_ptr[txt_bank], 4096, SG);
         pack_fread(s16_ptr[spr_bank], 4096, SG);
         pack_fread(s16_ptr[0xFF], 65536, SG);
         for (i=0; i<save_extra[0]; i++) {
            pack_fread(s16_ptr[(save_extra[1+i*2]&0xff0000)>>16] + (save_extra[1+i*2]&0xffff), save_extra[1+i*2+1], SG);
         }

         if (pack_igetl(SG) == sound_driver_magic) {
           UINT8 new_music_code;
           pack_fread(&new_music_code, sizeof(current_bgm), SG);
           if (new_music_code != current_bgm) {
             sound_request(new_music_code);
           }
           mz80exec(1000);
         }

         tpl_state_file = SG;
         // do not close file

      } else return -1;
   } else { // multi-cpu game
      PACKFILE *f;
      extern UINT16 txt_offset;
      UINT8 cpu_savebuf[256];
      int old_cpu;
      int i;
      if ((f=pack_fopen(full_path,F_READ_PACKED)) != NULL) {
         pack_fread(game_desc, 60, f);
         old_cpu=context_switch(0);
         for (i=0; i<num_cpus; i++) {
            context_switch(i);
            pack_fread(cpu_savebuf, sizeof(cpu_savebuf), f);
            load_cpu_content(cpu_savebuf);
         }
         pack_fread(s16_ptr[vid_bank], 0x10000, f);
         pack_fread(s16_ptr[pal_bank], 0x4000, f);
         pack_fread(s16_ptr[txt_bank]+txt_offset, 0x1000, f);
         pack_fread(s16_ptr[spr_bank], 0x4000, f);
         pack_fread(s16_ptr[0xFF], 0x10000, f);
         context_switch(0);
         for (i=0; i<save_extra[0]; i++) {
            pack_fread(s16_ptr[(save_extra[1+i*2]&0xff0000)>>16] + (save_extra[1+i*2]&0xffff), save_extra[1+i*2+1], f);
         }

         if (pack_igetl(f) == sound_driver_magic) {
           UINT8 new_music_code;
           pack_fread(&new_music_code, sizeof(current_bgm), f);
           if (new_music_code != current_bgm) {
             sound_request(new_music_code);
           }
         }

         tpl_state_file = f;
         // do not close file

         context_switch(old_cpu);
      } else return -1;
  }
  return 0;
}


#define MOVIE_EXT_NAME ".mov"
extern int current_tpl_slot;
extern char shortname[9];
/* return 0 if succ */
int load_tpl_data()
{
  char full_name[80];
  
  // load game data
  strcpy(full_name, shortname);
  strcat(full_name, MOVIE_EXT_NAME);
  if (current_tpl_slot >= 0) full_name[strlen(full_name) - 1] = '0' + current_tpl_slot;

  if (LoadGame(full_name) != 0) return -1;
  
  // load trace play data
  tpl.length_data = 0;
  pack_fread(&tpl.length_data, sizeof(tpl.length_data), tpl_state_file);
  pack_fread(tpl.data, tpl.length_data, tpl_state_file);
//printf("load len=%0d\n", tpl.length_data);
  pack_fclose(tpl_state_file);
  tpl_state_file = NULL;
}

/* return 0 if succ */
int save_tpl_data()
{
  // save game data
  // already done in start_record_tpl
  
  // save trace play data
  if (tpl_state_file) {
//printf("save len=%0d\n", tpl.length_data);
    pack_fwrite(&tpl.length_data, sizeof(tpl.length_data), tpl_state_file);
    pack_fwrite(tpl.data, tpl.length_data, tpl_state_file);
    pack_fclose(tpl_state_file);
    tpl_state_file = NULL;
  }
  return 0;
}

int start_record_tpl(UINT8 *state, char *desc)
{
  char full_name[80];
  static char default_desc[61]={0};

  if (desc) memcpy(default_desc, desc, sizeof(default_desc));
  tpl.cur_data = tpl.data;
  tpl.need_to_skip = 0;
  tpl.length_data = 0;
  tpl.remaining = MAX_TPL_DATA - 16;
  memset(tpl.input_state, 0xff, sizeof(tpl.input_state));
  memset(state, 0xff, sizeof(tpl.input_state));

  // save current game data
  strcpy(full_name, shortname);
  strcat(full_name, MOVIE_EXT_NAME);
  if (current_tpl_slot >= 0) full_name[strlen(full_name) - 1] = '0' + current_tpl_slot;

  if (SaveGame(full_name, default_desc) != 0) return -1;

  return 0;
}

int start_play_tpl(UINT8 *state)
{
  tpl.cur_data = tpl.data;
  tpl.need_to_skip = 0;
  tpl.remaining = tpl.length_data;
  memset(tpl.input_state, 0xff, sizeof(tpl.input_state));
  memset(state, 0xff, sizeof(tpl.input_state));
  return 0;
}

