#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ym2151.h"

/*
**  some globals ...
*/
FILE * errorlog=NULL;

/*
**  Shifts below are subject to change if sampling frequency changes...
*/ 
#define FREQ_SH 16  /* 16.16 fixed point for frequency calculations*/
#define ENV_SH 16   /* 16.16 fixed point for envelope calculations */
#define TIMER_SH 16 /* 16.16 fixed point for timers calculations   */

#define ENV_BITS 10
#define ENV_RES ((int)1<<ENV_BITS)
#define ENV_STEP (96.0/ENV_RES)
#define MAX_VOLUME_INDEX ((ENV_RES-1)<<ENV_SH)
#define MIN_VOLUME_INDEX (0)
#define VOLUME_OFF (ENV_RES<<ENV_SH)

#define SIN_BITS 10
#define SIN_LEN ((int)1<<SIN_BITS)
#define SIN_MASK (SIN_LEN-1)

int FINAL_SH=5;   /*this shift is applied to final output of all channels to get 8-bit sample*/

unsigned char FEED[8]={0,7,6,5,4,3,2,1}; /*shifts (divider) for output of op.0 which feeds into itself*/

#define TL_TAB_LEN (2*(ENV_RES + ENV_RES + ENV_RES + SIN_LEN))
signed int * TL_TAB = NULL;
/*
 *  Offset in this table is calculated as:
 *
 *  1st ENV_RES:
 *     TL- main offset (Total attenuation Level), range 0 to 1023 (0-96 dB)
 *  2nd ENV_RES:
 *     current envelope value of the operator, range 0 to 1023 (0-96 dB)
 *  3rd ENV_RES:
 *     Amplitude Modulation from LFO, range 0 to 1023 (0-96dB)
 *  4th SIN_LEN:
 *     Sin Wave Offset from sin_tab, range 0 to about 56 dB only, but lets
 *     simplify things and assume sin could be 96 dB, range 0 to 1023
 *
 *  Length of this table is doubled because we have two separate parts
 *  for positive and negative halves of sin wave.
 */

unsigned short sin_tab[SIN_LEN];  /* sin waveform table in decibel scale */






/*tables below are defined for usage by LFO */
signed int PMsaw     [SIN_LEN];   /*saw waveform table PM      */
signed int PMsquare  [SIN_LEN];   /*square waveform table PM   */
signed int PMtriangle[SIN_LEN];   /*triangle waveform table PM */
signed int PMnoise   [SIN_LEN];   /*noise waveform table PM    */

unsigned short AMsaw     [SIN_LEN];     /*saw waveform table AM      */
unsigned short AMsquare  [SIN_LEN];     /*square waveform table AM   */
unsigned short AMtriangle[SIN_LEN];     /*triangle waveform table AM */
unsigned short AMnoise   [SIN_LEN];     /*noise waveform table AM    */


int YM2151_CLOCK = 1;		/* this will be passed from ym2151in.c */
int YM2151_SAMPFREQ = 1;	/* this will be passed from ym2151in.c */ 


static int YMBufSize;		/* size of sound buffer, in samples */
static int YMNumChips;		/* total # of YM's emulated */

static int TimerA[1024];
static int TimerB[256];

static YM2151 * YMPSG = NULL;		/* array of YM's */

static void (*envelope_calc[5])(OscilRec *);
static void (*register_writes[256])(unsigned char , unsigned char, unsigned char);


//save output as raw 16-bit sample - just in case you would like to listen to it offline ;-)
//#define SAVE_SAMPLE  
//#define SAVE_SEPARATE_CHANNELS

#ifdef SAVE_SAMPLE
#ifdef SAVE_SEPARATE_CHANNELS
FILE *sample1;
FILE *sample2;
FILE *sample3;
FILE *sample4;
FILE *sample5;
FILE *sample6;
#endif
FILE *samplesum;
#endif


int PMTab[8]; /*8 channels */
/*  this table is used for PM setup of LFO */

int AMTab[8]; /*8 channels */
/*  this table is used for AM setup of LFO */


int decib45[16];        /*decibel table to convert D1L values(48dB) index in lookup table*/

int attack_curve[ENV_RES];

unsigned int divia[64]; //Attack dividers
unsigned int divid[64]; //Decay dividers
unsigned int A_DELTAS[64+31]; //Attack deltas (64 keycodes + 31 RKS's = 95)
unsigned int D_DELTAS[64+31]; //Decay  deltas (64 keycodes + 31 RKS's = 95)



float DT1Tab[32][4]={ /* 8 octaves * 4 key codes, 4 DT1 values */
/*
 *  Table defines offset in Hertz from base frequency depending on KC and DT1
 *  User's Manual page 21
*/
/*OCT NOTE DT1=0 DT1=1  DT1=2  DT1=3*/
/* 0   0*/{0,    0,     0.053, 0.107},
/* 0   1*/{0,    0,     0.053, 0.107},
/* 0   2*/{0,    0,     0.053, 0.107},
/* 0   3*/{0,    0,     0.053, 0.107},
/* 1   0*/{0,    0.053, 0.107, 0.107},
/* 1   1*/{0,    0.053, 0.107, 0.160},
/* 1   2*/{0,    0.053, 0.107, 0.160},
/* 1   3*/{0,    0.053, 0.107, 0.160},
/* 2   0*/{0,    0.053, 0.107, 0.213},
/* 2   1*/{0,    0.053, 0.160, 0.213},
/* 2   2*/{0,    0.053, 0.160, 0.213},
/* 2   3*/{0,    0.053, 0.160, 0.267},
/* 3   0*/{0,    0.107, 0.213, 0.267},
/* 3   1*/{0,    0.107, 0.213, 0.320},
/* 3   2*/{0,    0.107, 0.213, 0.320},
/* 3   3*/{0,    0.107, 0.267, 0.373},
/* 4   0*/{0,    0.107, 0.267, 0.427},
/* 4   1*/{0,    0.160, 0.320, 0.427},
/* 4   2*/{0,    0.160, 0.320, 0.480},
/* 4   3*/{0,    0.160, 0.373, 0.533},
/* 5   0*/{0,    0.213, 0.427, 0.587},
/* 5   1*/{0,    0.213, 0.427, 0.640},
/* 5   2*/{0,    0.213, 0.480, 0.693},
/* 5   3*/{0,    0.267, 0.533, 0.747},
/* 6   0*/{0,    0.267, 0.587, 0.853},
/* 6   1*/{0,    0.320, 0.640, 0.907},
/* 6   2*/{0,    0.320, 0.693, 1.013},
/* 6   3*/{0,    0.373, 0.747, 1.067},
/* 7   0*/{0,    0.427, 0.853, 1.173},
/* 7   1*/{0,    0.427, 0.907, 1.173},
/* 7   2*/{0,    0.480, 1.013, 1.173},
/* 7   3*/{0,    0.533, 1.067, 1.173}
};

int DT2Tab[4]={ /* 4 DT2 values */
/*
 *   DT2 defines offset in cents from base note
 *
 *   The table below defines offset in deltas table...
 *   User's Manual page 22
 *   Values below were calculated using formula:  value = orig.val * 1.5625
 *
 * DT2=0 DT2=1 DT2=2 DT2=3
 * 0     600   781   950
 */
	0,    384,  500,  608
};


unsigned short KC_TO_SEMITONE[16]={
/*translate note code KC into more usable number of semitone*/
0<<6, 1<<6, 2<<6, 0,
3<<6, 4<<6, 5<<6, 0,
6<<6, 7<<6, 8<<6, 0,
9<<6,10<<6,11<<6, 0};

unsigned char RKS_TAB[32*4]={
/*User's Manual page 24*/
0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3, 
0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,6,6,6,6,7,7,7,7,
0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,
0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31
};




int DT1deltas[32][8];

int deltas[8*13*64];/*8 octaves, 12 semitones, 64 'cents'*/
/*deltas in sintable (fixed point) to get the closest frequency possible */


void sin_init(void)
{
	int x,i;
	float m;

	for(i=0; i<SIN_LEN/2; i++)
	{
		if (i==0)
			m = ENV_RES-1;
		else
		{
			m = sin(2*PI*i/SIN_LEN);  /*count sin value*/
			m = 20*log10(1.0/m);      /* and how many decibels it would be? */
			m = m / ENV_STEP;
		}
		sin_tab[       i       ] = m;                /* positive part */
		sin_tab[ i + SIN_LEN/2 ] = m + TL_TAB_LEN/2; /* and negative one */
		//if (errorlog) fprintf(errorlog,"sin %i = %i\n",i,sin_tab[i] );
	}

	for( x=0; x<TL_TAB_LEN/2; x++ )
	{
		m = ((1<<12)-1) / pow(10, x * ENV_STEP /20);
		TL_TAB[         x        ] = m;
		TL_TAB[ x + TL_TAB_LEN/2 ] = -m;
		//if (errorlog) fprintf(errorlog,"tl %04i =%08x\n",x,TL_TAB[x]);
	}

	for (x=0; x<16; x++)
	{
		i = (x<15?x:16) * (3.0/ENV_STEP);  /*every 3 dB except for ALL '1' = 48dB*/ 
		i<<=ENV_SH;
		decib45[x]=i;
		//if (errorlog) fprintf(errorlog,"decib45[%04x]=%08x\n",x,i );
	}


#ifdef SAVE_SAMPLE
#ifdef SAVE_SEPARATE_CHANNELS
sample1=fopen("samp.raw","wb");
sample2=fopen("samp2.raw","wb");
sample3=fopen("samp3.raw","wb");
sample4=fopen("samp4.raw","wb");
sample5=fopen("samp5.raw","wb");
sample6=fopen("samp6.raw","wb");
#endif
samplesum=fopen("sampsum.raw","wb");
#endif
}


void hertz(void)
{
	int i,j;
	long int mult;
	float pom;
	float scaler;	/* formula below is true for chip clock=3579545 */
	/* so we need to scale its output accordingly to the chip clock */

/*this loop calculates Hertz values for notes from c#0 up to c-8*/
/*including 64 'cents' (100/64 which is 1.5625 of real cent) for each semitone*/

	scaler= (float)YM2151_CLOCK / 3579545.0;

	mult = (long int)1<<FREQ_SH;
	for (i=0; i<8*13*64; i++)
	{
		pom = scaler * 13.75 * pow (2, ((i+4*64)*1.5625/1200.0) ); /*13.75Hz is note A 12semitones below A-0, so C#0 is 4 semitones above then*/
		/*calculate phase increment for above precounted Hertz value*/
		deltas[i]=((pom*SIN_LEN)/(float)YM2151_SAMPFREQ)*mult; /*fixed point*/
		//if (errorlog) fprintf(errorlog,"deltas[%04i] = %08x\n",i,deltas[i]);
	}
	for (j=0; j<4; j++)
	{
		for (i=0; i<32; i++)
		{
			pom = scaler * DT1Tab[i][j];
			//calculate phase increment for above precounted Hertz value
			DT1deltas[i][j]=((pom*SIN_LEN)/(float)YM2151_SAMPFREQ)*mult; /*fixed point*/
			DT1deltas[i][j+4]= -DT1deltas[i][j];
		}
	}

/* create attack curve */
	for (i=0; i<ENV_RES; i++)
	{
		pom = (ENV_RES-1) / pow(10, i * (40.0/ENV_RES) /20);
		j = pom;
		attack_curve[ENV_RES-1-i]=j<<ENV_SH;
		if (errorlog) fprintf(errorlog,"attack %04x = %08x\n",ENV_RES-1-i,j);
	}


/* precalculate envelope times */
	for (i=0; i<64; i++)
	{
		pom=(16<<(i>>2))+(4<<(i>>2))*(i&0x03);
		if ((i>>2)==0)	pom=1; //infinity
		if ((i>>2)==15) pom=524288; //const
		divid[i]=pom;
	}

	for (i=0; i<64; i++)
	{
		pom=((128+64+32)<<(i>>2))+((32+16+8)<<(i>>2))*(i&0x03);
		if ((i>>2)==0)	pom=1;    //infinity
		if ((i>>2)==15) 
			if ((i&0x03)==3) 
				pom=153293300; //zero attack time
			else 
				pom=6422528; //const attack time
		divia[i]=pom;
	}

	mult = (long int)1<<ENV_SH;
	for (i=0; i<64; i++)
	{
		if (divid[i]==1)
			pom=0;  //infinity
		else
			pom=(ENV_RES * mult)/ ( (float)YM2151_SAMPFREQ*((float)YM2151_CLOCK/1000.0/(float)divid[i]));
		//if (errorlog) fprintf(errorlog,"i=%03i div=%i time=%f delta=%f\n",i,divid[i],
		//		(float)YM2151_CLOCK/1000.0/(float)divid[i], pom );
		D_DELTAS[i] = pom;
	}

	for (i=0; i<64; i++)
	{
		if (divia[i]==1)
			pom=0;  //infinity
		else
			pom=(ENV_RES * mult)/ ( (float)YM2151_SAMPFREQ*((float)YM2151_CLOCK/1000.0/(float)divia[i]));
		//if (errorlog) fprintf(errorlog,"i=%03i div=%i time=%f delta=%f\n",i,divia[i],
		//		(float)YM2151_CLOCK/1000.0/(float)divia[i], pom );
		A_DELTAS[i] = pom;
	}

	for (i=0; i<32; i++) /*this is only for speedup purposes -> to avoid testing if (keycode+RKS is over 63)*/
	{
		D_DELTAS[64+i]=D_DELTAS[63];
		A_DELTAS[64+i]=A_DELTAS[63];
	}


/* precalculate timers deltas */
/* User's Manual pages 15,16  */
	mult = (int)1<<TIMER_SH;
	for (i=0 ; i<1024; i++)
	{
		pom= ( 64.0  *  (1024.0-i) / YM2151_CLOCK ) * YM2151_SAMPFREQ ;
		TimerA[i] = pom * mult;  /*number of samples that timer period should last (fixed point) */
	}
	for (i=0 ; i<256; i++)
	{
		pom= ( 1024.0 * (256.0-i)  / YM2151_CLOCK ) * YM2151_SAMPFREQ ;
		TimerB[i] = pom * mult;  /*number of samples that timer period should last (fixed point) */
	}
}









void envelope_attack(OscilRec *op)
{
	if ( (op->attack_volume -= op->delta_AR) < MIN_VOLUME_INDEX )  //is volume index min already ?
	{
		op->volume = MIN_VOLUME_INDEX;
		op->state++;
	}
	else
		op->volume = attack_curve[op->attack_volume>>ENV_SH];
}

void envelope_decay(OscilRec *op)
{
	if ( (op->volume += op->delta_D1R) > op->D1L )
	{
		//op->volume = op->D1L;
		op->state++;
	}
}

void envelope_sustain(OscilRec *op)
{
	if ( (op->volume += op->delta_D2R) > MAX_VOLUME_INDEX )
	{
		op->state = 4;
		op->volume = VOLUME_OFF;
	}
}

void envelope_release(OscilRec *op)
{
	if ( (op->volume += op->delta_RR) > MAX_VOLUME_INDEX )
	{
		op->state = 4;
		op->volume = VOLUME_OFF;
	}
}


void envelope_nothing(OscilRec *op)
{
}

void envelope_state_count(OscilRec *op)
{
	if (op->KON == 0) /*KEYOFF*/
	{
		op->KON=2;
		op->state=3; /*release*/
	}
	if (op->KON == 1) /*KEYON*/
	{
		op->KON=2;
					 /*this is to remove the gap time if TL>0*/
		op->volume = VOLUME_OFF; //(ENV_RES - op->TL)<<ENV_SH; /***  <-  SURE ABOUT IT ?  No, but let's give it a try...*/
		op->attack_volume = op->volume;
		op->phase = 0;
		op->state = 0;    /*KEY ON = attack*/
		op->OscilFB = 0;  /*Clear feedback after key on */
	}
	envelope_calc[op->state](op);
}


void freq_calc(unsigned char chan, YM2151 *PSG)
{
signed int dt1;
unsigned short oct,j,dt2_offs;
unsigned char op,note,v,k;

	note = PSG->Regs[YM_KC_BASE + chan] & 0x0f;
	oct = (PSG->Regs[YM_KC_BASE + chan] >> 4) & 0x07;
	j =  (oct*768) + KC_TO_SEMITONE[note] + (PSG->Regs[YM_KF_BASE+chan] >> 2);
        //        768=12<<6

	for (k=0; k<32; k+=8)
	{
		op = chan + k;

		v = (PSG->Regs[YM_DT2_D2R_BASE+op]>>6) & 0x03;
	        dt2_offs = DT2Tab[v]; //DT2 value

		if ( PSG->Regs[YM_DT1_MUL_BASE+op]&0x0f )
			PSG->Oscils[ op ].freq = deltas[j+dt2_offs] * (PSG->Regs[YM_DT1_MUL_BASE+op]&0x0f);
		else
			PSG->Oscils[ op ].freq = deltas[j+dt2_offs] / 2;

		dt1 = DT1deltas[PSG->Oscils[op].KC][(PSG->Regs[YM_DT1_MUL_BASE+op]>>4)&0x07];
		PSG->Oscils[op].freq += dt1;  //negative or positive dt1
			
	}
}

void write_YM_NON_EMULATED(unsigned char n, unsigned char r, unsigned char v)
{
    if (errorlog)
	fprintf(errorlog,"Write to non emulated register %02x value %02x\n",r,v);
}

void write_YM_KON(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char chan;
    static YM2151 *PSG;
    PSG = &(YMPSG[n]);

        v &= 0x7f;
        PSG->Regs[r] = v;

	chan = v & 0x07;

        PSG->Oscils[chan   ].KON = (v >> 3) & 1;
        PSG->Oscils[chan+ 8].KON = (v >> 4) & 1;
        PSG->Oscils[chan+16].KON = (v >> 5) & 1;
        PSG->Oscils[chan+24].KON = (v >> 6) & 1;
}


void write_YM_CLOCKA1(unsigned char n, unsigned char r, unsigned char v)
{
    YMPSG[n].Regs[r]=v;
}

void write_YM_CLOCKA2(unsigned char n, unsigned char r, unsigned char v)
{
    YMPSG[n].Regs[r]=v & 0x03;
}

void write_YM_CLOCKB(unsigned char n, unsigned char r, unsigned char v)
{
    YMPSG[n].Regs[r]=v;
}

void write_YM_CLOCKSET(unsigned char n, unsigned char r, unsigned char v)
{
    static YM2151 *PSG;
    PSG = &(YMPSG[n]);

    v &= 0xbf;
    PSG->Regs[r]=v;
//	if (errorlog) fprintf(errorlog,"CSM= %01x FRESET=%02x, IRQEN=%02x, LOAD=%02x\n",v>>7,(v>>4)&0x03,(v>>2)&0x03,v&0x03 );
    if (v&0x80) //CSM
	{    }
    if (v&0x01) //LOAD A
    {
		PSG->TimA=1;
		PSG->TimAVal=TimerA[ (PSG->Regs[YM_CLOCKA1]<<2) | PSG->Regs[YM_CLOCKA2] ];
    }
    else
	PSG->TimA=0;
    if (v&0x02) //load B
    {
		PSG->TimB=1;
		PSG->TimBVal=TimerB[ PSG->Regs[YM_CLOCKB] ];
    }
    else
	PSG->TimB=0;
    if (v&0x04) //IRQEN A
	{	}
    if (v&0x08) //IRQEN B
	{       }
    if (v&0x10) //FRESET A
    {
		PSG->TimIRQ &= 0xfe;
    }
    if (v&0x20) //FRESET B
    {
		PSG->TimIRQ &= 0xfd;
    }	

}

void write_YM_CT1_CT2_W(unsigned char n, unsigned char r, unsigned char v)
{
	YMPSG[n].Regs[r]=v;
}


void write_YM_CONNECT_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char chan,i;
    static YM2151 *PSG;
    PSG = &(YMPSG[n]);

	PSG->Regs[r] = v;

	chan = r-YM_CONNECT_BASE;
        i = v & 7; /*connection number*/

	PSG->ConnectTab[chan]=i;
	
	i = (v>>3) & 7;
	PSG->FeedBack[chan] = FEED[i];
}


void write_YM_KC_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char chan,i,j;
    static OscilRec * osc;
    static YM2151 *PSG;

	PSG = &(YMPSG[n]);

	v &= 0x7f;
	PSG->Regs[r] = v;

	chan = r - YM_KC_BASE;

        i= v >> 2;

        PSG->Oscils[chan   ].KC = i;
        PSG->Oscils[chan+ 8].KC = i;
        PSG->Oscils[chan+16].KC = i;
        PSG->Oscils[chan+24].KC = i;

	j= PSG->Regs[YM_KS_AR_BASE + chan + 0];
	osc = &PSG->Oscils[ chan + 0 ];
        osc->RKS = RKS_TAB[((j>>6)<<5)+i];
	osc->delta_AR  = A_DELTAS[ osc->AR + osc->RKS];    /* 2*RR + RKS = max 95*/
	osc->delta_D1R = D_DELTAS[osc->D1R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_D2R = D_DELTAS[osc->D2R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_RR =  D_DELTAS[ osc->RR + osc->RKS];    /* 2*RR + RKS =max 95*/

	j= PSG->Regs[YM_KS_AR_BASE + chan + 8];
	osc = &PSG->Oscils[ chan + 8 ];
        osc->RKS = RKS_TAB[((j>>6)<<5)+i];
	osc->delta_AR  = A_DELTAS[ osc->AR + osc->RKS];    /* 2*RR + RKS = max 95*/
	osc->delta_D1R = D_DELTAS[osc->D1R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_D2R = D_DELTAS[osc->D2R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_RR =  D_DELTAS[ osc->RR + osc->RKS];    /* 2*RR + RKS =max 95*/

	j= PSG->Regs[YM_KS_AR_BASE + chan + 16];
	osc = &PSG->Oscils[ chan + 16 ];
        osc->RKS = RKS_TAB[((j>>6)<<5)+i];
	osc->delta_AR  = A_DELTAS[ osc->AR + osc->RKS];    /* 2*RR + RKS = max 95*/
	osc->delta_D1R = D_DELTAS[osc->D1R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_D2R = D_DELTAS[osc->D2R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_RR =  D_DELTAS[ osc->RR + osc->RKS];    /* 2*RR + RKS =max 95*/

	j= PSG->Regs[YM_KS_AR_BASE + chan + 24];
	osc = &PSG->Oscils[ chan + 24 ];
        osc->RKS = RKS_TAB[((j>>6)<<5)+i];
	osc->delta_AR  = A_DELTAS[ osc->AR + osc->RKS];    /* 2*RR + RKS = max 95*/
	osc->delta_D1R = D_DELTAS[osc->D1R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_D2R = D_DELTAS[osc->D2R + osc->RKS];    /* 2*RR + RKS =max 95*/
	osc->delta_RR =  D_DELTAS[ osc->RR + osc->RKS];    /* 2*RR + RKS =max 95*/

	//freq_calc(chan,PSG);

}

void write_YM_KF_BASE(unsigned char n, unsigned char r, unsigned char v)
{
	YMPSG[n].Regs[r] = v & 0xfc;
	//freq_calc(chan,PSG);
}


void write_YM_PMS_AMS_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char chan,i;
    static YM2151 *PSG;
    PSG = &(YMPSG[n]);

	PSG->Regs[r] = v;

	chan = r - YM_PMS_AMS_BASE;

	i = v>>4;  //PMS;
	PMTab[chan] = i;
	if (i && errorlog)
		fprintf(errorlog,"PMS CHN %02x =%02x\n",chan,i);

	i = v&0x03; //AMS;
	AMTab[chan] = i;
	if (i && errorlog)
		fprintf(errorlog,"AMS CHN %02x =%02x\n",chan,i);

}

void write_YM_DT1_MUL_BASE(unsigned char n, unsigned char r, unsigned char v)
{
	YMPSG[n].Regs[r] = v;
	//freq_calc(chan,PSG);
}

void write_YM_TL_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char op;
    static YM2151 *PSG;
    PSG = &(YMPSG[n]);

	v &= 0x7f;
	PSG->Regs[r] = v;

	op = r - YM_TL_BASE;
	PSG->Oscils[ op ].TL =  v<<(ENV_BITS-7); /*7bit TL*/

}


void write_YM_KS_AR_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char op;
    static OscilRec * osc;
    static YM2151 *PSG;

	op = r - YM_KS_AR_BASE;
	PSG = &(YMPSG[n]);
	osc = &PSG->Oscils[ op ];
	PSG->Regs[r] = v;

        osc->RKS = RKS_TAB[ ((v>>6)<<5) + osc->KC ];

	osc->AR = (v&0x1f) << 1;
	osc->delta_AR = A_DELTAS[ osc->AR + osc->RKS];    /* 2*RR + RKS = max 95*/

}


void write_YM_AMS_D1R_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char op;
    static OscilRec * osc;
    static YM2151 *PSG;

	op = r - YM_AMS_D1R_BASE;
	PSG = &(YMPSG[n]);
	osc = &PSG->Oscils[ op ];
	PSG->Regs[r] = v;


	if ((v & 0x80) && errorlog)
		fprintf(errorlog,"AMS ON oper%02x\n",op);

     //HERE something to do with AMS;

	osc->D1R = (v&0x1f) << 1;
	osc->delta_D1R = D_DELTAS[osc->D1R + osc->RKS];    /* 2*RR + RKS =max 95*/
}

void write_YM_DT2_D2R_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char op;
    static OscilRec * osc;
    static YM2151 *PSG;

	op = r - YM_DT2_D2R_BASE;
	PSG = &(YMPSG[n]);
	osc = &PSG->Oscils[ op ];
	PSG->Regs[r] = v;


	osc->D2R =  (v&0x1f) << 1;
	osc->delta_D2R = D_DELTAS[osc->D2R + osc->RKS];    /* 2*RR + RKS =max 95*/

	//freq_calc(chan,PSG);

}


void write_YM_D1L_RR_BASE(unsigned char n, unsigned char r, unsigned char v)
{
    static unsigned char j,op;
    static OscilRec * osc;
    static YM2151 *PSG;

	op = r - YM_D1L_RR_BASE;
	PSG = &(YMPSG[n]);
	osc = &PSG->Oscils[ op ];
	PSG->Regs[r] = v;


	j= (v>>4) & 0x0f;
	osc->D1L = decib45[ j ];

	osc->RR= (((v&0x0f)<<1)|0x01) << 1 ;
	osc->delta_RR = D_DELTAS[osc->RR + osc->RKS];    /* 2*RR + RKS =max 95*/
}






/*
** Initialize YM2151 emulator(s).
**
** 'num' is the number of virtual YM2151's to allocate
** 'clock' is the chip clock
** 'rate' is sampling rate and 'bufsiz' is the size of the
** buffer that should be updated at each interval
*/
int YMInit(int num, int rate, int bufsiz, SAMPLE **buffer )
{
    int i;

    if (YMPSG) return (-1);	/* duplicate init. */

    YMNumChips = num;
    YM2151_SAMPFREQ = rate;
    YM2151_CLOCK = 4050000/*3582071*/;
    YMBufSize = bufsiz;

    envelope_calc[0]=envelope_attack;
    envelope_calc[1]=envelope_decay;
    envelope_calc[2]=envelope_sustain;
    envelope_calc[3]=envelope_release;
    envelope_calc[4]=envelope_nothing;

    for (i=0; i<256; i++)
	    register_writes[i]=write_YM_NON_EMULATED;
	
    register_writes[YM_KON] 	= write_YM_KON;
    register_writes[YM_CLOCKA1]	= write_YM_CLOCKA1;
    register_writes[YM_CLOCKA2]	= write_YM_CLOCKA2;
    register_writes[YM_CLOCKB]	= write_YM_CLOCKB;
    register_writes[YM_CLOCKSET]= write_YM_CLOCKSET;
    register_writes[YM_CT1_CT2_W]= write_YM_CT1_CT2_W;

    for (i=YM_CONNECT_BASE; i<YM_CONNECT_BASE+8; i++)
	    register_writes[i]= write_YM_CONNECT_BASE;

    for (i=YM_KC_BASE; i<YM_KC_BASE+8; i++)
	    register_writes[i]= write_YM_KC_BASE;

    for (i=YM_KF_BASE; i<YM_KF_BASE+8; i++)
	    register_writes[i]= write_YM_KF_BASE;

    for (i=YM_PMS_AMS_BASE; i<YM_PMS_AMS_BASE+8; i++)
	    register_writes[i]= write_YM_PMS_AMS_BASE;

    for (i=YM_DT1_MUL_BASE; i<YM_DT1_MUL_BASE+32; i++)
	    register_writes[i]= write_YM_DT1_MUL_BASE;

    for (i=YM_TL_BASE; i<YM_TL_BASE+32; i++)
	    register_writes[i]= write_YM_TL_BASE;

    for (i=YM_KS_AR_BASE; i<YM_KS_AR_BASE+32; i++)
	    register_writes[i]= write_YM_KS_AR_BASE;

    for (i=YM_AMS_D1R_BASE; i<YM_AMS_D1R_BASE+32; i++)
	    register_writes[i]= write_YM_AMS_D1R_BASE;

    for (i=YM_DT2_D2R_BASE; i<YM_DT2_D2R_BASE+32; i++)
	    register_writes[i]= write_YM_DT2_D2R_BASE;

    for (i=YM_D1L_RR_BASE; i<YM_D1L_RR_BASE+32; i++)
	    register_writes[i]= write_YM_D1L_RR_BASE;


    YMPSG = (YM2151 *)malloc(sizeof(YM2151) * YMNumChips);
    if (YMPSG == NULL) return (1);

    TL_TAB = (signed int *)malloc(sizeof(signed int) * TL_TAB_LEN );
    if (TL_TAB == NULL) return (1);

    hertz();
    sin_init();

    for ( i = 0 ; i < YMNumChips; i++ )
    {
	YMPSG[i].Buf = buffer[i];
	YMPSG[i].bufp=0;
        YMResetChip(i);
    }
    return(0);
}

void YMShutdown()
{
    if (!YMPSG) return;

    free(YMPSG);
    YMPSG = NULL;

    if (TL_TAB)
    {
	free(TL_TAB);
	TL_TAB=NULL;
    }
    
    YM2151_SAMPFREQ = YMBufSize = 0;

#ifdef SAVE_SAMPLE
#ifdef SAVE_SEPARATE_CHANNELS
fclose(sample1);
fclose(sample2);
fclose(sample3);
fclose(sample4);
fclose(sample5);
fclose(sample6);
#endif
fclose(samplesum);
#endif
}

/* write a register on YM2151 chip number 'n' */
void YMWriteReg(int n, int r, int v)
{
	register_writes[ (unsigned char)r ] ( (unsigned char)n, (unsigned char)r, (unsigned char)v);
}

unsigned char YMReadReg( unsigned char n )
{
    return ( YMPSG[n].TimIRQ );
}


/*
** reset all chip registers.
*/
void YMResetChip(int num)
{
    int i;
    YM2151 *PSG = &(YMPSG[num]);

    memset(PSG->Buf,'\0',YMBufSize);

    /* initialize hardware registers */
    for( i=0; i<256; i++ )
	PSG->Regs[i]=0;

    for( i=0; i<32; i++) 
    {
	memset(&PSG->Oscils[i],'\0',sizeof(OscilRec));
	PSG->Oscils[i].volume=VOLUME_OFF;
	PSG->Oscils[i].state=4;  //envelope off
	//PSG->Oscils[i].phase=0;
	//PSG->Oscils[i].OscilFB=0;
	//PSG->Oscils[i].KON=0;
    }

    for ( i=0; i<8; i++)
    {
	PSG->ConnectTab[i]=0;
	PSG->FeedBack[i]=0;
    }

	PSG->TimA=0;
	PSG->TimB=0;
	PSG->TimAVal=0;
	PSG->TimBVal=0;
	PSG->TimIRQ=0;

}




static inline int op_calc(OscilRec *OP, signed int pm)
{
unsigned short tabindex;

	envelope_state_count(OP);

	tabindex = ( ((OP->phase+=OP->freq) >> FREQ_SH) + pm ) & SIN_MASK;
	tabindex = sin_tab[ tabindex ] + OP->TL +  (OP->volume>>ENV_SH);

	return ( TL_TAB[ tabindex ] );
}



void YM2151UpdateOne(int num, int endp)
{
    YM2151 * PSG = &(YMPSG[num]);
    static OscilRec *OP0, *OP1, *OP2, *OP3;
    static int ChannOuts[8];
    static int *ch0, *ch1, *ch2, *ch3, *ch4, *ch5, *ch6, *ch7;
    static int i, k, wy;
    static unsigned char chan;
    SAMPLE * PSGBUF;

	for (chan=0; chan<8; chan++)
		freq_calc(chan,PSG);

	//calculate timers...
	if (PSG->TimA)
	{
		PSG->TimAVal -= ((endp - PSG->bufp)<<TIMER_SH);
		if (PSG->TimAVal<=0)
		{
			PSG->TimA=0;
			PSG->TimIRQ|=1;
			if (PSG->handler !=0) PSG->handler();
		}
	}
	if (PSG->TimB)
	{
		PSG->TimBVal -= ((endp - PSG->bufp)<<TIMER_SH);
		if (PSG->TimBVal<=0)
		{
			PSG->TimB=0;
			PSG->TimIRQ|=2;
			if (PSG->handler !=0) PSG->handler();
		}
	}

ch0 = &ChannOuts[0];
ch1 = &ChannOuts[1];
ch2 = &ChannOuts[2];
ch3 = &ChannOuts[3];
ch4 = &ChannOuts[4];
ch5 = &ChannOuts[5];
ch6 = &ChannOuts[6];
ch7 = &ChannOuts[7];


  PSGBUF = &PSG->Buf[PSG->bufp];

  for( i=PSG->bufp; i<endp; ++i )
  {
    for (chan=0; chan<8; chan++)
    {
	OP0 = &PSG->Oscils[chan     ];
	OP1 = &PSG->Oscils[chan +  8];
	OP2 = &PSG->Oscils[chan + 16];
	OP3 = &PSG->Oscils[chan + 24];

	wy = op_calc(OP0,OP0->OscilFB);

	if (PSG->FeedBack[chan])
	    OP0->OscilFB = wy >> PSG->FeedBack[chan];
	else
	    OP0->OscilFB = 0;

	switch(PSG->ConnectTab[chan])
	{
	case 0:
		wy = op_calc(OP3, op_calc(OP1, op_calc(OP2,wy) ) );
		break;
	case 1:
		wy = op_calc(OP3, op_calc(OP1, op_calc(OP2,0)+wy ) );
		break;
	case 2:
		wy = op_calc(OP3, op_calc(OP1, op_calc(OP2,0)) +wy );
		break;
	case 3:
		wy = op_calc(OP3, op_calc(OP2, wy)+op_calc(OP1,0) );
		break;
	case 4:
		wy = op_calc(OP2,wy) + op_calc(OP3, op_calc(OP1,0)); 
		break;
	case 5:
		wy = op_calc(OP2,wy) + op_calc(OP1, wy) + op_calc(OP3,wy);
		break;
	case 6:
		wy = op_calc(OP2,wy) + op_calc(OP1,0) + op_calc(OP3,0);
		break;
	case 7:
		wy = wy + op_calc(OP2,0) + op_calc(OP1,0) + op_calc(OP3,0);
		break;
	}
	ChannOuts[chan]=wy;	
      }

      k = *ch0 +*ch1 +*ch2 +*ch3 +*ch4 +*ch5 +*ch6 +*ch7 ;

#ifdef SAVE_SAMPLE
#ifdef SAVE_SEPARATE_CHANNELS
fputc((unsigned short)(*ch0)&0xff,sample1);
fputc(((unsigned short)(*ch0)>>8)&0xff,sample1);

fputc((unsigned short)(*ch1)&0xff,sample2);
fputc(((unsigned short)(*ch1)>>8)&0xff,sample2);

fputc((unsigned short)(*ch2)&0xff,sample3);
fputc(((unsigned short)(*ch2)>>8)&0xff,sample3);

fputc((unsigned short)(*ch3)&0xff,sample4);
fputc(((unsigned short)(*ch3)>>8)&0xff,sample4);

fputc((unsigned short)(*ch4)&0xff,sample5);
fputc(((unsigned short)(*ch4)>>8)&0xff,sample5);

fputc((unsigned short)(*ch5)&0xff,sample6);
fputc(((unsigned short)(*ch5)>>8)&0xff,sample6);
#endif
fputc((unsigned short)(-k)&0xff,samplesum);
fputc(((unsigned short)(-k)>>8)&0xff,samplesum);
#endif

       k >>= FINAL_SH;  //AUDIO_CONV
        if (k>127)
	 k=127;
	else
	  if (k<-128)
	    k=-128;

	*(PSGBUF++) = (SAMPLE)k;

    }
    //PSG->bufp = endp;

}


/*
** called to update all chips
*/
void YM2151Update(void)
{
    int i;
    for ( i = 0 ; i < YMNumChips; i++ )
    {
	if( YMPSG[i].bufp <  YMBufSize )
		YM2151UpdateOne(i , YMBufSize );
	YMPSG[i].bufp = 0;
    }
}


/*
** return the buffer into which YM2151Update() has just written it's sample
** data
*/
SAMPLE *YMBuffer(int n)
{
    return YMPSG[n].Buf;
}

void YMSetBuffer(int n, SAMPLE *buf)
{
    YMPSG[n].Buf = buf;
}


void YMSetIrqHandler(int n, void(*handler)())
{
    YMPSG[n].handler = handler;
}
