#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <allegro.h>
#include "host.h"
#include "main.h"
#include "prototyp.h"
#include "shinobi.h"

/* extern */
extern UINT8 *mousex_ptr, *mousey_ptr;
extern UINT8 *mouseb_ptr;
extern int mouse_minX, mouse_minY, mouse_maxX, mouse_maxY, mouse_defX, mouse_defY;
extern UINT8 mouseb_msk, mouse_enabled;

extern UINT8 *leds_status;
extern UINT8 leds_nflag, leds_cflag, leds_sflag, leds_mode;

extern INT8 sound_card;
extern UINT8 joystick_type;
extern UINT8 menu_videomode;

extern UINT16 drx, dry, vrx, vry;
extern UINT16 rx, ry;

extern UINT8 game, rdtsc;

extern UINT8 bgm_flags[256];
extern UINT8 sfx_start, sfx_end, bgm_start, bgm_end, vce_start, vce_end;
extern char samples_path[80], shortname[9], game_gcs[60];

extern double initial_gamma;
/* pointers */

UINT8 *s16_ptr[0x10000];
extern void *s16_rpb[0x10000];
extern void *s16_wpb[0x10000];
extern void *s16_rpw[0x10000];
extern void *s16_wpw[0x10000];
extern void *s16_rpl[0x10000];
extern void *s16_wpl[0x10000];

UINT8 *code, *sprites_base_data, *tiles_base, *tiles_transparent, *tiles_hi;
int code_size, obj_size;

/* dipswitches variables & arrays */

#define DS_MAX 10
#define DS_MAX_VALUES 8
UINT8 total_ds=0;
UINT8 ds_mask[DS_MAX], ds_shift[DS_MAX];
char ds_desc[DS_MAX][30], ds_values[DS_MAX * DS_MAX_VALUES][20];

/* arrays */
char labeljoy[8][20],labelgen[8][20];

#define MAX_PARAMETER 32
#define MAX_PARAMETER_LENGTH 64
static char prm[MAX_PARAMETER][MAX_PARAMETER_LENGTH];

/* stored in 3-tuples : source,dest,stuff_value */
extern UINT32 dup_addrs[];
extern int dup_addrs_size;

/* global vars */

char zip_path[80] = "./", rom_path[80], spr_tab[16];
char pcx_path[80] = "./";
char sav_path[80] = "./";
char ram_path[80] = "./";
char zsmp_path[80] = "./";
char movie_path[80] = "./";

UINT8 vertical_display=0;
UINT8 spr_bank, vid_bank, ctr_bank, txt_bank, pal_bank, ext_bank;
UINT16 txt_offset=0;
UINT16 tile_palette_offset=0,sprite_palette_offset=0x800;
UINT8 s16_dac_type=0;
UINT8 pre_16_flag=0;
UINT8 reverse_bgpage_flag=0;
UINT8 obj_is_32bit=0;
int analogstick_x,analogstick_y,analogstick_z;
UINT8 analogstick_x_auto,analogstick_y_auto,analogstick_z_auto;
int analogstick_x_center,analogstick_y_center,analogstick_z_center;
int analogstick_x_speed,analogstick_y_speed,analogstick_z_speed;
int analogstick_x_limit,analogstick_y_limit,analogstick_z_limit;
UINT8 *analogstick_x_addr=0,*analogstick_y_addr=0,*analogstick_z_addr=0;
UINT8 *gr_control[4];
UINT8 *gr_palette,*gr_palette_default;
UINT8 *gr_bitmap;
int gr_bitmap_width;
UINT32 gr_color_flipindex[2];

UINT8 *second_road_on;
UINT8 s_hangon = 0;
UINT8 inversed_tile = 0;
UINT8 z80_speed = 2;
UINT8 pf_state =
#ifdef RELEASE
                 0;
#else
                 1;
#endif
UINT8 sound_type = 0, sampled_sound_support = 0, sound_disabled = 0;
UINT8 sample_loop[256];
UINT8 type_sound_subboard = 0;

// BG access
UINT8 bg_layer=0x8f;
UINT8 bg_colormode=0;
UINT8 bg_text_mode=0;
UINT8 fg_priority=0x20;
UINT8 bg_priority=0x18;

UINT16 iopl1 = 0xffff, iopl2 = 0xffff, iopl3 = 0xffff, iogen = 0x1001;
UINT16 iods1 = 0x2003, iods2 = 0x2001;

UINT8 *foreground_base, *background_base;
UINT8 *base_spr_regist, *base_brq_page;
UINT8 *base_scr_hor_fg, *base_scr_hor_bg, *base_scr_ver_fg, *base_scr_ver_bg;
UINT8 *base_scr_pag_fg, *base_scr_pag_bg;
UINT8 *base_scr_pag_bg2, *base_scr_hor_bg2, *base_scr_ver_bg2;
UINT8 *base_scr_pag_bg3, *base_scr_hor_bg3, *base_scr_ver_bg3;
UINT8 *base_scr_hor_bg_splittab0, *base_scr_hor_bg_splittab1;
UINT8 *base_scr_ver_bg_splittab0, *base_scr_ver_bg_splittab1;

UINT32 reg_screen_active=0;
UINT32 reg_screen_active18=0;
UINT32 reg_screen_active_sega3d=0;
UINT32 bg_xoffset=0,spr_xoffset=0;

#define MAX_SAVE_EXTRA 10
UINT32 save_extra[1+(MAX_SAVE_EXTRA)*2]={0,0};

static FILE *INI;
static UINT8 *progress, *progress_tt, *progress_hi;
static UINT8 END_OF_GCS = 0;

/* functions */

UINT8 *vptr2ptr(UINT32 vptr)
{
   UINT16 ptr = vptr & 0xFFFF;
   UINT8 bank = (vptr >> 16) & 0xFF;
   if (!vptr) return 0;
   return &s16_ptr[bank][ptr];
}

unsigned GetInteger(char *s)
{
   int v=0, m=1, p;
   p = strlen(s);
   if (p>2) {
      if ((s[0]=='0')&&(s[1]=='x')) v = -1;
   }
   if (v==-1) {
      v = 0;
      do {
	 p--;
	 if ((s[p]>='A')&&(s[p]<='Z'))
	   v+=(m*(10+(s[p]-'A')));
	 else if ((s[p]>='a')&&(s[p]<='z'))
      	v+=(m*(10+(s[p]-'a')));
	 else
	   v+=(m*(s[p]-'0'));
	 m*=16;
      } while (p!=2);
   } else {
      sscanf(s, "%d", &v);
   }
   return(v);
}

void LowerCase(char *s)
{
   int p = 0;
   while (s[p]) {
      if ((s[p]>='A')&&(s[p]<='Z')) s[p]+=32;
      p++;
   }
}

void UpperCase(char *s)
{
   int p = 0;
   while (s[p]) {
     if ((s[p]>='a')&&(s[p]<='z')) s[p]-=32;
     if (s[p]=='_') s[p]=' ';
     p++;
   }
}

int GetParams(int np)
{
   int i;
   char linebuf[256], *tok;

   if (fgets(linebuf, sizeof(linebuf)-1, INI)==NULL) return 0;

   for (i=0; i<MAX_PARAMETER; i++) prm[i][0]=0;

   tok=strtok(linebuf, " \t\n\r");
   for (i=0;i<MAX_PARAMETER;i++) {
     if (tok) strcpy(prm[i], tok);
     tok=strtok(NULL, " \t\n\r");
     if (tok==NULL) break;
   }
   return(i+1);
}

int FileSize(FILE *fp)
{
   int save_pos, size_of_file;
   
   save_pos = ftell(fp);
   fseek(fp, 0L, SEEK_END);
   size_of_file = ftell(fp);
   fseek(fp, save_pos, SEEK_SET);
   return(size_of_file);
}

void ierror(char *msg)
{
   imessage(msg,55,48,1,2);
}

void *new_malloc(int size)
{
   void *p;
   p = (void *)malloc(size);
   if (!p) {
      ierror("Fatal error, not enough memory !\n");
      exit(-1);
   }
   return p;
}

// TL078 - new function to support ZIP files
// return 0 if succ
int loadfile(char *filename, char *zipfn, UINT8 **buf, int *size)
{
   FILE *f;
   int fsize;
   UINT8 *buffer;
   char mymsg[256];

   sprintf(mymsg, "Loading '%s' ...", zipfn);
   imessage(mymsg, 47, 46, 0, 0);

   if ((f = fopen(filename,"rb")) != NULL) {
      fsize = FileSize(f);
      buffer = (UINT8 *)new_malloc(fsize);
      fread(buffer,1,fsize,f);
      fclose(f);
   } else {
      // file not found, try with zip method
      if (pf_state) printf("File '%s' not found, trying to load from zip file...\n", filename);
      if (load_zipped_file(zip_path,zipfn,&buffer,&fsize)) {
         sprintf(mymsg, "Error, can't load %s from %s\n", zipfn, zip_path);
         ierror(mymsg);
         return 1;
      }
   }
   *buf = buffer;
   *size = fsize;
   return 0;
}

// TL078 - modified to support ZIP files
int LoadTiles(char *filename, UINT8 *tile_pattern, UINT8 *transparent, UINT8 shift)
{
   UINT8 *tmp;
   char fn[256];
   int s, i, a, j, b;

   sprintf(fn, "%s%s", rom_path, filename);

   if (loadfile(fn, filename, &tmp, &s)) {
      if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
      exit(0);
   }

   a=0;

   for (i=0; i<s; i+=8) {
     /* plane combining */
     for (j=0; j<8; j++) {
       for (b=0; b<8; b+=2,a++) {
         if (!inversed_tile) {
           tile_pattern[a] |= (   ((tmp[i+j] & (1<<(7-b))) >> (7-b))       << shift);
           tile_pattern[a] |= (   ((tmp[i+j] & (1<<(7-b-1))) >> (7-b-1))   << (shift+4));
         } else {
           tile_pattern[a] |= ( (!((tmp[i+j] & (1<<(7-b))) >> (7-b)))      << shift);
           tile_pattern[a] |= ( (!((tmp[i+j] & (1<<(7-b-1))) >> (7-b-1)))  << (shift+4));
         }
       }
     }

     /* generate transparent-flag */
     for (j=0; j<32; j++) {
       if (tile_pattern[a-j-1] != 0) { /* non-transparent point found */
         transparent[i/8] = 1;
         break;
       }
     }

   }

   free(tmp);
   return(a);
}

int LoadFile2(char *filename, UINT8 *ptr)
{
   UINT8 *tmp;
   char fn[256];
   int s, i;

   sprintf(fn, "%s%s", rom_path, filename);

   if (loadfile(fn,filename,&tmp,&s)) {
      if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
      exit(0);
   }

   for (i=0; i<s; i++) {
      ptr[0] = tmp[i];
      ptr += 2;
   }

   free(tmp);
   return(s);
}

int LoadFile4(char *filename, char *ptr)
{
   UINT8 *tmp;
   char fn[256];
   int s, i;

   sprintf(fn, "%s%s", rom_path, filename);

   if (loadfile(fn,filename,&tmp,&s)) {
      if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
      exit(0);
   }

   for (i=0; i<s; i++) {
      ptr[0] = tmp[i];
      ptr += 4;
   }

   free(tmp);
   return(s);
}

void D_GAME(void)
{
   int i = 0;
   while (prm[0][i]) {
      if (prm[0][i] == '_') prm[0][i] = ' ';
      i++;
   }
   if (pf_state) printf("Game name : '%s'\n", prm[0]);
}

void D_ROMSIZE(void)
{
   int i;
   extern int num_cpus;

   code_size = GetInteger(prm[0]);
   if (pf_state) printf("Code size = 0x%x\n", code_size);
   code = (UINT8 *)new_malloc(code_size * 0x10000);
   for (i=0; i<code_size; i++) {
      s16_ptr[i] = &code[i * 0x10000];
      s16_rpb[i] = rom_readb;
      s16_rpw[i] = rom_readw;
      s16_rpl[i] = rom_readl;
      s16_wpb[i] = s16_wpw[i] = s16_wpl[i] = rom_write_illegal;
/*
      if (num_cpus>0) {
        s16_wpb[i] = rom_writeb;
        s16_wpw[i] = rom_writew;
        s16_wpl[i] = rom_writel;
      }
*/      
   }
   if (pf_state) printf("Code memory allocated. %08x\n", code);
   progress = code;
   add_cpu(code,code_size);
}

void D_ROMLOAD(void)
{
   UINT32 i;
   i = LoadFile2(prm[0], (UINT8 *)(progress + 1));
   i+= LoadFile2(prm[1], (UINT8 *)progress);
   progress += i;
}

void DefRamAccess(UINT8 i)
{
   s16_rpb[i] = mem_readb;
   s16_rpw[i] = mem_readw;
   s16_rpl[i] = mem_readl;
   s16_wpb[i] = mem_writeb;
   s16_wpw[i] = mem_writew;
   s16_wpl[i] = mem_writel;
}

void D_DEFRAM(void)
{
   UINT8 b1 = GetInteger(prm[0]);
   int s = GetInteger(prm[1]), d;
   int cpu;

   if (prm[2][0]) cpu=GetInteger(prm[2]);
   else cpu = -1;

   s16_ptr[b1] = (UINT8 *)new_malloc(s * 1024L);
   memset(s16_ptr[b1], 0x00, s * 1024L);
   DefRamAccess(b1);
   if (pf_state) printf("%ld bytes allocated from 0x%06lx. (%08x)\n", s * 1024L, (b1 * 0x10000L)&0xFFFFFF, s16_ptr[b1]);
   if (cpu>=0) {
     add_dedicated_bank(cpu, b1);
     if (pf_state) printf("bank 0x%02x dedicated to cpu%d\n", b1, cpu);
   }
   s-= 64;
   d = 0;
   while (s>0) {
      d++;
      s16_ptr[b1+d] = (UINT8 *)&(s16_ptr[b1][d * 65536L]);
      DefRamAccess(b1+d);
      if (pf_state) printf("bank 0x%02x remapped to 0x%06lx + 0x%06lx\n", b1 + d, b1 * 0x10000L, d * 0x10000L);
      s-= 64;
      if (cpu>=0) {
        add_dedicated_bank(cpu, b1+d);
        if (pf_state) printf("bank 0x%02x dedicated to cpu%d\n", b1+d, cpu);
      }
   }
}

void D_ZIP(void)
{
   char oldpath[128];
   strcpy(oldpath, zip_path);
   sprintf(zip_path, "%s%s", oldpath, prm[0]);
   strcpy(oldpath, zsmp_path);
   sprintf(zsmp_path, "%s\\%s", oldpath, prm[0]);
   if (pf_state) printf("Game/Sample ZIP file set to '%s'\n", zip_path);
}

void D_SCRSIZE(void)
{
   int s = GetInteger(prm[0]) * 256 * 32 * 2; /* base + hi */
   int st = s/32; /* 8x8 tile = 32 bytes */
   
   if (pf_state) printf("Tiles rom size = 0x%0x bytes.\n", s);
   tiles_base = (UINT8 *)new_malloc(s+st);
   tiles_transparent = tiles_base + s/2;
   tiles_hi = tiles_transparent + st;
   memset(tiles_base, 0, s+st);
   progress = tiles_base;
   progress_tt = tiles_transparent;
   progress_hi = tiles_hi;
   if (pf_state) printf("Tiles memory allocated.\n");
}

void D_SCRLOAD(void)
{
   int i;
   if (pf_state) printf("Loading tiles ROM : %s %s %s\n", prm[0], prm[1], prm[2]);
   LoadTiles(prm[0], progress, progress_tt, 0);
   LoadTiles(prm[1], progress, progress_tt, 1);
   i = LoadTiles(prm[2], progress, progress_tt, 2);

   memcpy(progress_hi, progress, i);
   progress += i;
   progress_tt += i/32;
   progress_hi += i;
}

void D_OBJSIZE(void)
{
   obj_size = GetInteger(prm[0])*1024;
   if (pf_state) printf("Objects rom size = 0x%0x bytes.\n", obj_size);
   sprites_base_data = (UINT8 *)new_malloc(obj_size);
   progress = sprites_base_data;
   if (pf_state) printf("Objects memory allocated.\n");
}

void D_OBJLOAD(void)
{
   int i;
   i = LoadFile2(prm[0], (UINT8 *)(progress + 1));
   i+= LoadFile2(prm[1], (UINT8 *)progress);
   progress+= i;
}

void D_OBJLOAD2(void)
{
   int i;
   i = LoadFile4(prm[0], (UINT8 *)(progress + 3));
   i+= LoadFile4(prm[1], (UINT8 *)(progress + 2));
   i+= LoadFile4(prm[2], (UINT8 *)(progress + 1));
   i+= LoadFile4(prm[3], (UINT8 *)(progress + 0));
   progress+= i;
}

void D_OBJOFFS(void)
{
   int i;
   for (i=0; i<16; i++) spr_tab[i] = GetInteger(prm[i]);
   if (pf_state) printf("Object offsets : ");
   for (i=0; i<16; i++) if (pf_state) printf("%02x ", spr_tab[i]);
   if (pf_state) printf("\n");
}

void D_MAPRAM(void)
{
   UINT8 b1 = GetInteger(prm[0]);
   UINT8 b2 = GetInteger(prm[1]);
   memcpy(s16_ptr[b1],s16_ptr[b2],0x10000);
   s16_ptr[b1] = s16_ptr[b2];
   if (pf_state) printf("bank 0x%02x remapped to bank 0x%02x\n", b1, b2);
}

void D_MAPPEDRAM(void)
{
   int b,bank_dest = GetInteger(prm[0]);
   int k,kilobytes = GetInteger(prm[1]);
   UINT32 s,src_addr = GetInteger(prm[2]);
   int cpu;
   if (prm[3][0]) cpu=GetInteger(prm[3]);
   else cpu = -1;

   s=src_addr; k=kilobytes; b=bank_dest;
   while (k>0) {
     s16_ptr[b] = (UINT8 *)vptr2ptr(s);
     k-=64;
     DefRamAccess(b);
     if (cpu>=0) {
       add_dedicated_bank(cpu, b);
       if (pf_state) printf("bank 0x%02x dedicated to cpu%d\n", b, cpu);
     }
     b++;
     s+=65536;
   }

   if (pf_state) printf("bank 0x%02x mirrors from 0x%06x for %dK bytes.\n", bank_dest, src_addr, kilobytes);
}

void D_DEFINE(void)
{
   #define NDP 55
   static char *defpar[NDP] = {
      "txt", "vid", "pal", "spr", "io",
      "iopl1", "iopl2", "iogen", "iods1", "iods2", 
      "reghsfg", "regvsfg", "regbs", "reghsbg", "regvsbg", 
      "regpsfg", "regpsbg", "regscract", "fakeio", "soundbank",
      "ext",
      "reghsbg2", "regvsbg2", "regpsbg2",
      "reghsbg3", "regvsbg3", "regpsbg3",
      "regsplit0", "regsplit1",
      "bgtextmode", "bgcolormode","bgpriority","fgpriority",
      "dactype", "bgxoffset", "sprxoffset",
      "regsplit2", "regsplit3", "bglayer", "regscract18",
      "txtoffset", "regscractsega3d", "loopedsample",
      "xres", "yres", "vertical_xres", "vertical_yres", "iopl3", "soundcard",
      "secondroadon", "z80speed", "brightness", "sprpal", "tilepal",
      "soundsubboard"
   };
   int i, p, v;
   char mymsg[160];

   for (i=0,p=-1; i<NDP; i++) {
     if (strcmp(prm[0], defpar[i]) == 0) {
       p = i;
       v = GetInteger(prm[1]);
       if (pf_state) printf("Define '%s' to 0x%02x\n", prm[0], v);

       switch (p) {
         case 0: txt_bank = v; break;
         case 1: 
           vid_bank = v;
           foreground_base = background_base = s16_ptr[vid_bank];
           break;
         case 2: pal_bank = v; break;
         case 3: 
           spr_bank = v;
           base_spr_regist = s16_ptr[spr_bank];
           break;
         case 4: 
           ctr_bank = v & 0xFF;
           s16_rpb[v] = mem_readb; //io_readb;
           s16_rpw[v] = mem_readw;
           s16_rpl[v] = mem_readl;
           s16_wpb[v] = mem_writeb;
           s16_wpw[v] = mem_writew;
           s16_wpl[v] = io_noeffect;
           memset(s16_ptr[ctr_bank], 0xff, 0x10000);
           break;
         case 5: iopl1 = v; break;
         case 6: iopl2 = v; break;
         case 7: iogen = v; break;
         case 8: iods1 = v; break;
         case 9: iods2 = v; break;
         case 10: base_scr_hor_fg = vptr2ptr(v); break;
         case 11: base_scr_ver_fg = vptr2ptr(v); break;
         case 12: base_brq_page = vptr2ptr(v); break;
         case 13: base_scr_hor_bg = vptr2ptr(v); break;
         case 14: base_scr_ver_bg = vptr2ptr(v); break;
         case 15: base_scr_pag_fg = vptr2ptr(v); break;
         case 16: base_scr_pag_bg = vptr2ptr(v); break;
         case 17: reg_screen_active = v; break;
         case 18: break; //s16_rpb[v & 0xFF] = mem_readb_goldenaxe; break;
         case 19: s16_wpb[v & 0xFF] = mem_writeb_soundrequest; break;
         case 20: ext_bank=v; break;
         case 21: base_scr_hor_bg2 = vptr2ptr(v); break;
         case 22: base_scr_ver_bg2 = vptr2ptr(v); break;
         case 23: base_scr_pag_bg2 = vptr2ptr(v); break;
         case 24: base_scr_hor_bg3 = vptr2ptr(v); break;
         case 25: base_scr_ver_bg3 = vptr2ptr(v); break;
         case 26: base_scr_pag_bg3 = vptr2ptr(v); break;
         case 27: base_scr_hor_bg_splittab0 = vptr2ptr(v); break;
         case 28: base_scr_hor_bg_splittab1 = vptr2ptr(v); break;
         case 29: bg_text_mode = v; break;
         case 30: bg_colormode = v; break;
         case 31: bg_priority = v; break;
         case 32: fg_priority = v; break;
         case 33: s16_dac_type = v; break;
         case 34: bg_xoffset = v; break;
         case 35: spr_xoffset = v; break;
         case 36: base_scr_ver_bg_splittab0 = vptr2ptr(v); break;
         case 37: base_scr_ver_bg_splittab1 = vptr2ptr(v); break;
         case 38: bg_layer = v; break;
         case 39: reg_screen_active18 = v; break;
         case 40: txt_offset= v; break;
         case 41: reg_screen_active_sega3d = v; break;
         case 42: sample_loop[v] = 1; break;
         case 43: drx = v; break;
         case 44: dry = v; break;
         case 45: vrx = v; break;
         case 46: vry = v; break;
         case 47: iopl3 = v; break;
         case 48: sound_card = v; break;
         case 49: second_road_on = vptr2ptr(v); break;
         case 50: z80_speed = v; break;
         case 51: { extern int color_addition; color_addition = v; } break;
         case 52: sprite_palette_offset = v; break;
         case 53: tile_palette_offset = v; break;
         case 54: type_sound_subboard = v; break;
         default:
           sprintf(mymsg, "Error, DEFINE: '%s' doesn't exists.\n", prm[0]);
           ierror(mymsg);
           exit(-1);
           break;
       }
     }
   }

}

void D_PATCHCODE(void)
{
   UINT32 a = GetInteger(prm[0]);
   UINT8 v = GetInteger(prm[1]);
   if (pf_state) printf("Patch %06x to %02x\n", a, v);
   code[a^1] = v;
}

void D_SOUND(void)
{
   char *types[] = {"No sound", "Emulation", "Rom samples", "External samples"};
   sound_type = GetInteger(prm[0]);
   if (pf_state) printf("Sound type = %s\n", types[sound_type]);
}

void D_GAMENUMBER(void)
{
   game = GetInteger(prm[0]);
   if (pf_state) printf("Game number = %d\n", game);
}

void D_DUP(void)
{
   UINT32 src=GetInteger(prm[0]);
   UINT32 dest=GetInteger(prm[1]);
   dup_addrs[dup_addrs_size++]=src;
   dup_addrs[dup_addrs_size++]=dest;
   dup_addrs[dup_addrs_size++]=GetInteger(prm[2]);
   if (pf_state) printf("Duplicate byte %06x->%06x\n", src, dest);
}

void D_SAVEROM(void)
{
   long s = (GetInteger(prm[1]) * 1024L);
   UINT8 b = (GetInteger(prm[0]));
   UINT16 *buf;
   int i;
   FILE *f;
   
   f = fopen(prm[2], "wb+");
   if (!f) {
      if (pf_state) printf("Can't open save rom file '%s'", prm[2]);
      exit(-1);
   }
   buf = (UINT16 *)new_malloc(s);
   memcpy(buf, s16_ptr[b], s);
   for (i=0;i<(s/2);i++) {
      buf[i] = (((UINT16)(buf[i]&0xFF))<<8)+((UINT16)(buf[i]>>8)&0xFF);
   }
   fwrite(buf, 1, s, f);
   fclose(f);
   free(buf);
   if (pf_state) printf("%ld bytes saved into %s\n", s, prm[2]);
}

void D_LOADDATA(void)
{
   UINT8 *tmp, *ptr;
   UINT16 offs = GetInteger(prm[0])&0xFFFF;
   UINT8 bank = (GetInteger(prm[0])>>16)&0xFF;
   UINT8 mode = GetInteger(prm[1]);
   char fn[256];
   int s, i;

   sprintf(fn, "%s%s", rom_path, prm[2]);

   if (loadfile(fn, prm[2], &tmp, &s)) {
      if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
      exit(0);
   }

   if (mode) {
      ptr = (UINT8 *)&s16_ptr[bank][offs];
      for (i=0;i<s;i++) {
         ptr[0] = tmp[i];
         ptr+= mode;
      }
   } else {
      memcpy(&s16_ptr[bank][offs], tmp, s);
   }
   free(tmp);
}

void D_VERTICAL(void)
{
   if (pf_state) printf("Vertical display.\n");
   vertical_display = 1;
   if (rx==320) rx = 640;
   if (ry==240) ry = 480;
}

void D_SHORTNAME()
{
   char mymsg[160];
   if (strlen(prm[0]) > 8) {
      sprintf(mymsg, "Error, '%s' shortname is too long (max 8 chars)\n", prm[0]);
      ierror(mymsg);
      exit(0);
   }
   strcpy(shortname, prm[0]);
   if (pf_state) printf("Game short name = %s\n", shortname);
}

void D_SFXRANGE()
{
   sfx_start = GetInteger(prm[0]);
   sfx_end = GetInteger(prm[1]);
   if (pf_state) printf("SFX samples range = %02x to %02x\n", sfx_start, sfx_end);
}

void D_BGMRANGE()
{
   bgm_start = GetInteger(prm[0]);
   bgm_end = GetInteger(prm[1]);
   if (pf_state) printf("BGM samples range = %02x to %02x\n", bgm_start, bgm_end);
   {
     int i;
     memset(bgm_flags, 0, sizeof(bgm_flags));
     for (i=bgm_start; i<=bgm_end; i++) bgm_flags[i] = 1;
   }
   bgm_flags[0] = 1;
}

void D_BGM()
{
  int i;
  for (i=0; i<MAX_PARAMETER && prm[i][0]; i++) {
    bgm_flags[GetInteger(prm[i])] = 1;
    if (pf_state) printf("BGM = %02x\n", GetInteger(prm[i]));
  }
}

void D_VOICERANGE()
{
   vce_start = GetInteger(prm[0]);
   vce_end = GetInteger(prm[1]);
   if (pf_state) printf("VOICES samples range = %02x to %02x\n", vce_start, vce_end);
}

static void *special_funcs[10*6]={
  mem_readb,mem_readw,mem_readl,mem_writeb,mem_writew,mem_writel,
  mem_readb,math_readw,math_readl,mem_writeb,math_writew,math_writel,
};

void D_SPECIALFUNC()
{
   int func = GetInteger(prm[0]);
   UINT32 addr = GetInteger(prm[1]);
   UINT8 b=addr>>16;
   if (pf_state) printf("Special function%d activates for %06x\n", func, addr);

   s16_rpb[b] = special_funcs[func*6+0];
   s16_rpw[b] = special_funcs[func*6+1];
   s16_rpl[b] = special_funcs[func*6+2];
   s16_wpb[b] = special_funcs[func*6+3];
   s16_wpw[b] = special_funcs[func*6+4];
   s16_wpl[b] = special_funcs[func*6+5];
}

void D_PRE_16()
{
   pre_16_flag=GetInteger(prm[0]);
   if (pf_state) printf("PRE-16 hardware version=%d\n", pre_16_flag);
}

void D_ANALOGSTICK()
{
   analogstick_x_addr = vptr2ptr(GetInteger(prm[0]));
   analogstick_x_auto = GetInteger(prm[1]);
   analogstick_x = analogstick_x_center = GetInteger(prm[2]);
   analogstick_x_speed = GetInteger(prm[3]);
   analogstick_x_limit = GetInteger(prm[4]);
   analogstick_y_addr = vptr2ptr(GetInteger(prm[5]));
   analogstick_y_auto = GetInteger(prm[6]);
   analogstick_y = analogstick_y_center = GetInteger(prm[7]);
   analogstick_y_speed = GetInteger(prm[8]);
   analogstick_y_limit = GetInteger(prm[9]);
   analogstick_z_addr = vptr2ptr(GetInteger(prm[10]));
   analogstick_z_auto = GetInteger(prm[11]);
   analogstick_z = analogstick_z_center = GetInteger(prm[12]);
   analogstick_z_speed = GetInteger(prm[13]);
   analogstick_z_limit = GetInteger(prm[14]);
}

void D_GRSCRLOAD()
{
  int s;
  char fn[256];
  UINT8 *old_buf, *buf;

  sprintf(fn, "%s%s", rom_path, prm[0]);
  if (loadfile(fn,prm[0],&buf,&s)) {
     if (pf_state) printf("Bad news, I can't find/load the file '%s' !!\n", fn);
     exit(0);
  }

  {
    UINT8 *gr,*grr=NULL;
    int i,j,k,w,skip;
    int start_color,end_color;
    int center_offset=0;

    old_buf = buf; // for use in free mem

    w=GetInteger(prm[1]); // file width
    gr_bitmap_width=GetInteger(prm[2]); // bitmap width
    skip=GetInteger(prm[3]);
    start_color=GetInteger(prm[4]); // start color
    end_color=GetInteger(prm[5]); // end color
    gr_bitmap = (UINT8 *)new_malloc(256*gr_bitmap_width);
    gr = gr_bitmap;
    memset(gr,0,256*gr_bitmap_width);

    if (pf_state) printf("GR width:%d %d %d GR bitmap:%x\n", w, gr_bitmap_width, skip, gr);
    if (w!=gr_bitmap_width) {
      if (skip!=0) // needs mirrored RHS
        grr=gr;
      else {
        center_offset=(gr_bitmap_width-w);
        gr+=center_offset/2;
      }
    }

    // build gr_bitmap
    for (i=0; i<256; i++) {
      UINT8 last_bit;
      UINT8 color_data[4];

      color_data[0]=start_color; color_data[1]=start_color+1;
      color_data[2]=start_color+2; color_data[3]=start_color+3;
      last_bit=((buf[0]&0x80)==0)|(((buf[0x4000]&0x80)==0)<<1);
      for (j=0; j<w/8; j++) {
        for (k=0; k<8; k++) {
          UINT8 bit=((buf[0]&0x80)==0)|(((buf[0x4000]&0x80)==0)<<1);
          if (bit!=last_bit && bit==0 && i>1) { // color flipped to 0,advance color[0]
            if (color_data[0]+end_color <= end_color) {
              color_data[0]+=end_color;
            } else {
              color_data[0]-=end_color;
            } 
          }
          *gr = color_data[bit];
          last_bit=bit;
          buf[0] <<= 1; buf[0x4000] <<= 1; gr++;
        }
        buf++;
      }

      if (grr!=NULL) { // need mirrored RHS
        UINT8 *_gr=gr-1;
        _gr -= skip;
        for (j=0; j<w-skip; j++) {
          *gr++ = *_gr--;
        }
        for (j=0; j<skip; j++) *gr++ = 0;
      } else if (center_offset!=0) {
        gr+=center_offset;
      }
    }
    free(old_buf);
  }

  {
    int i=1;
    while ( (1<<i) < gr_bitmap_width ) i++;
    gr_bitmap_width=i; // power of 2
    if (pf_state) printf("%d\n",i);
  }

  gr_control[0] = vptr2ptr(GetInteger(prm[6])); // VER
  gr_control[1] = gr_control[0]+GetInteger(prm[7]); // HOR
  gr_control[2] = gr_control[0]+GetInteger(prm[8]); // PAL
  gr_control[3] = gr_control[0]+GetInteger(prm[9]); // FLIP
 
  gr_palette = vptr2ptr((((UINT32)pal_bank)<<16)|GetInteger(prm[10]));
  gr_palette_default = vptr2ptr((((UINT32)pal_bank)<<16)|GetInteger(prm[11]));
  gr_color_flipindex[0]=GetInteger(prm[12]);
  gr_color_flipindex[1]=GetInteger(prm[13]);
}

void D_DIPSWITCH()
{
  int i;
  ds_mask[total_ds] = GetInteger(prm[0]);
  ds_shift[total_ds] = GetInteger(prm[1]);
  for (i=2; i<(2+DS_MAX_VALUES); i++) UpperCase(prm[i]);
  sprintf(ds_desc[total_ds], "%s", prm[2]);
  for (i=0; i<DS_MAX_VALUES; i++)
    sprintf(ds_values[(total_ds * DS_MAX_VALUES) + i], "%s", prm[3 + i]);
  total_ds++;
}

void D_COMMONAREA()
{
  UINT32 addr=GetInteger(prm[0]);
  int len=GetInteger(prm[1]);
  set_common_area(addr, len);
  if (pf_state) printf("set common area: %06x~%06x\n", addr, addr+len-1);
}

void D_LABELJOY()
{
   int i;
   for (i=0; i<8; i++) {
      if (prm[i][0]!='_') strcpy(labeljoy[i], prm[i]);
      else labeljoy[i][0]=0;
   }
   if (pf_state) printf("Joystick labels defined.\n");
}
	 
void D_LABELGEN()
{
   int i;
   for (i=0; i<8; i++) {
      if (prm[i][0]!='_') strcpy(labelgen[i], prm[i]);
      else labelgen[i][0]=0;
   }
   if (pf_state) printf("Control panel labels defined.\n");
}

void D_DEFAULTGCS()
{
   strcpy(game_gcs, prm[0]);
   if (pf_state) printf("Default GCS = %s\n", game_gcs);
}

void D_NORDTSC()
{
   rdtsc = 0;
   if (pf_state) printf("RDTSC disabled.\n");
}

void D_FRAME()
{
  extern int frame;
  frame=GetInteger(prm[0]);
  if (pf_state) printf("Skip frame=%d\n",frame);
}

void D_SAVEEXTRA()
{
  if (save_extra[0] < MAX_SAVE_EXTRA) {
    int i=1+(save_extra[0]*2);
    save_extra[i]=GetInteger(prm[0]);
    save_extra[i+1]=GetInteger(prm[1]);
    save_extra[0]++;
    if (pf_state) printf("SAVE EXTRA INFO #%d $%x=%d\n", save_extra[0], save_extra[i], save_extra[i+1]);
  }
}

UINT8 Z80_active = 0;
extern UINT8 Z80MEM[65536];
// loading fixed data
void D_Z80LOAD()
{
   int size;
   char fn[256];
   UINT8 *tmp;
   UINT32 addr;
   int len;
   int i;
   
   for (i=addr=0; prm[i][0] && addr<0x10000 && i<10; i++,addr+=len) {
     sprintf(fn, "%s%s", rom_path, prm[i]);
     if (loadfile(fn, prm[i], &tmp, &size)) {
       if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
       exit(0);
     } else {
       len = size<=0x10000 ?size :0x10000;
       if (pf_state) printf("Z80 ROM addr=%04x size=%04x\n", addr, len);
       memcpy(Z80MEM+addr, tmp, len);
       free(tmp);
     }
   }
   
   Z80_active = 1;
}

UINT8 *z80data;
UINT32 z80data_len;
UINT32 z80data_win=0x4000;
UINT32 z80data_win_size=0x4000;
// loading banked data
void D_Z80DATA()
{
   int size;
   char fn[256];
   UINT8 *tmp;
   UINT32 addr;
   int len;
   int i;

   z80data_len = GetInteger(prm[0]);
   z80data = (UINT8 *)new_malloc(z80data_len);
   
   for (i=1, addr=0; prm[i][0] && addr<z80data_len && i<10; i++,addr+=len) {
     sprintf(fn, "%s%s", rom_path, prm[i]);
     if (loadfile(fn, prm[i], &tmp, &size)) {
       if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
       exit(0);
     } else {
       len = size;
       if (pf_state) printf("Z80 sound data addr=%06x size=%06x\n", addr, len);
       memcpy(z80data+addr, tmp, len);
       free(tmp);
     }
   }

}

UINT8 z80com=0;
UINT8 *z80com_z80_addr;
UINT8 *z80com_68k_addr;
UINT8 z80com_len;
UINT8 z80com_port_delay=1;
void D_Z80COM()
{
  z80com = GetInteger(prm[0]);
  if (z80com&0x80) {
    z80com_port_delay=3;
    z80com&=0x7f;
  }

  if (pf_state) printf("Using method %d in Z80/68000 communication\n", z80com);
  switch (z80com) {
    case 0:  // int only
      break;
    case 1:  // memory port. no int/nmi.
    case 2:  // memory port. int only
      z80com_z80_addr = Z80MEM+GetInteger(prm[1]); 
      break;
    case 3:  // nmi only 
      break;
    case 4:  // memory port. copy from 68k memory directy. no int/nmi.
      z80com_z80_addr = Z80MEM+GetInteger(prm[1]);
      z80com_68k_addr = vptr2ptr(GetInteger(prm[2]));
      z80com_len = GetInteger(prm[3]);
      break;
  }
}

void D_Z80PATCH()
{
  UINT32 a = GetInteger(prm[0]) & 0xffff;
  UINT8 v = GetInteger(prm[1]);
  if (Z80_active) {
    if (pf_state) printf("Patch Z80 ROM %04x to %02x\n", a, v);
    Z80MEM[a] = v;
  }
}

UINT8 ym2151_active = 0;
void D_YM2151()
{
  extern int FINAL_SH;
  if (prm[0][0]) {
    FINAL_SH = GetInteger(prm[0]);
    if (pf_state) printf("set ym2151 level=%d\n",FINAL_SH);
  }
  ym2151_active = 1;
}

UINT8 ym2203_active = 0;
void D_YM2203()
{
  ym2203_active = 1;
}

UINT8 ym3438_active = 0;
void D_YM3438()
{
  ym3438_active = 1;
}

void D_SPECIALFORSHANGON() 
{
  s_hangon = 1;
}

void D_MOUSE() 
{
   UINT32 mx = GetInteger(prm[0]);
   UINT32 my = GetInteger(prm[1]);
   UINT32 mb = GetInteger(prm[2]);
   mousex_ptr = vptr2ptr(mx);
   mousey_ptr = vptr2ptr(my);
   mouseb_ptr = vptr2ptr(mb);
   mouse_minX = GetInteger(prm[3]);
   mouse_defX = GetInteger(prm[4]);
   mouse_maxX = GetInteger(prm[5]);
   mouse_minY = GetInteger(prm[6]);
   mouse_defY = GetInteger(prm[7]);
   mouse_maxY = GetInteger(prm[8]);
   mouseb_msk = GetInteger(prm[9]);
   mouse_enabled = 1;
}

void D_END(void)
{
   END_OF_GCS = 1;
}

void D_INVERSEDTILES(void)
{
   inversed_tile = 1;
}

void D_REVERSEDBGPAGE(void)
{
   reverse_bgpage_flag = 1;
}

void D_OBJ32BITS(void)
{
   obj_is_32bit = 1;
}

void fix_path(char *p)
{
   int len=strlen(p);
   if (p[len-1]!='\\') {
     p[len] = '\\';
     p[len+1] = 0;
   }
}

void D_LEDS(void)
{
   UINT32 p = GetInteger(prm[0]);
   leds_status = vptr2ptr(p);
   leds_nflag = GetInteger(prm[1]); // NUM LOCK
   leds_cflag = GetInteger(prm[2]); // CAPS LOCK
   leds_sflag = GetInteger(prm[3]); // SCROLL LOCK
   leds_mode = GetInteger(prm[4]); // 1 = normal , 0 = reversed
   if (pf_state) printf("LEDS INFO: ADRS = 0x%08x\n"
                        "           NUM FLAG = 0x%02x , CAPS FLAG = 0x%02x , SCROLL FLAG = 0x%02x\n",
                        p, leds_nflag, leds_cflag, leds_sflag);
}

void D_SETPATH(int p)
{
   if (p!=2) if (prm[0][strlen(prm[0])-1]!='\\') sprintf(prm[0],"%s\\",prm[0]);
   switch(p) {
      case 1: sprintf(rom_path, "%s", prm[0]); break;
      case 2: sprintf(samples_path, "%s", prm[0]); break;
      case 3: sprintf(zip_path, "%s", prm[0]); break;
      case 5: sprintf(sav_path, "%s", prm[0]); break;
      case 6: sprintf(pcx_path, "%s", prm[0]); break;
      case 7: sprintf(ram_path, "%s", prm[0]); break;
   }
}

void D_TILEHIMASK(void)
{
   unsigned pattern_num = GetInteger(prm[0]);
   UINT8 masks[8];
   int i;
   UINT8 *tile;
   for (i=0; i<8; i++) masks[i] = GetInteger(prm[1+i]);
   if (pf_state) printf("mask tile#%04x to %02x%02x%02x%02x%02x%02x%02x%2x\n", pattern_num, masks[0],masks[1],masks[2],masks[3],masks[4],masks[5],masks[6],masks[7]);
   // clear all masked bits to 0(transparent)
   for (i=0,tile = &tiles_hi[32*pattern_num]; i<8; i++,tile+=4) {
     if (!(masks[i] & 0x80)) tile[0] &= 0xf0;
     if (!(masks[i] & 0x40)) tile[0] &= 0x0f;
     if (!(masks[i] & 0x20)) tile[1] &= 0xf0;
     if (!(masks[i] & 0x10)) tile[1] &= 0x0f;
     if (!(masks[i] & 0x08)) tile[2] &= 0xf0;
     if (!(masks[i] & 0x04)) tile[2] &= 0x0f;
     if (!(masks[i] & 0x02)) tile[3] &= 0xf0;
     if (!(masks[i] & 0x01)) tile[3] &= 0x0f;
   }
}

UINT8 *speech_data;
long speech_data_len;
long speech_rom_size;
int speech_rom_numbers;
UINT8 uPD7759_active;
UINT8 uPD7751_active;
int speech_data_samplingrate;
void D_SPEECH()
{
   int size;
   char fn[256];
   UINT8 *tmp;
   UINT32 addr;
   int len;
   int i;
   #define SPEECH_DATA_SIZE (1024*1024)
   
   speech_data = (UINT8 *)new_malloc(SPEECH_DATA_SIZE);
   memset(speech_data, 0xff, SPEECH_DATA_SIZE);
   uPD7759_active = 0;
   uPD7751_active = 0;
   
   if (strcasecmp(prm[0], "uPD7759") == 0) uPD7759_active = 1;
   if (strcasecmp(prm[0], "uPD7751") == 0) uPD7751_active = 1;
   if (strcasecmp(prm[1], "6k") == 0) speech_data_samplingrate = 6000;
   else speech_data_samplingrate = 8000;
   
   for (i=2, addr=0, len=0; prm[i][0] && addr<SPEECH_DATA_SIZE && i<10; i++,addr+=len) {
     sprintf(fn, "%s%s", rom_path, prm[i]);
     if (loadfile(fn, prm[i], &tmp, &size)) {
       if (pf_state) printf("Bad news, I can't find the file '%s' !!\n", fn);
       exit(0);
     } else {
       speech_rom_size = len = size;
       if (pf_state) printf("Load speech data addr=%06x size=%06x\n", addr, len);
       memcpy(speech_data+addr, tmp, len);
       free(tmp);
     }
   }
   speech_data_len = addr;
   speech_rom_numbers = i-2;
   if (pf_state) printf("Speech data len=%06x ROM size = %06x ROM numbers=%d\n", speech_data_len, speech_rom_size, speech_rom_numbers);

}

typedef struct {
  char *command;
  char nparams;
  void *ptr;
  int p;
} t_cmdlist;

static t_cmdlist cmdlist[] =
   {{"game"          , 1   , D_GAME       , 0},
    {"romsize"       , 1   , D_ROMSIZE    , 0},
    {"romload"       , 2   , D_ROMLOAD    , 0},
    {"defram"        , 2   , D_DEFRAM     , 0},
//    {"io"            , 1   , D_GAME       , 0},
    {"end"           , 0   , D_END        , 0},
    {"scrsize"       , 1   , D_SCRSIZE    , 0},
    {"scrload"       , 3   , D_SCRLOAD    , 0},
    {"objsize"       , 1   , D_OBJSIZE    , 0},
    {"objload"       , 2   , D_OBJLOAD    , 0},
    {"objoffs"       , 16  , D_OBJOFFS    , 0},
    {"mapram"        , 2   , D_MAPRAM     , 0},
    {"define"        , 2   , D_DEFINE     , 0},
    {"patchcode"     , 2   , D_PATCHCODE  , 0},
    {"sound"         , 1   , D_SOUND      , 0},
    {"gamenumber"    , 1   , D_GAMENUMBER , 0},
    {"vertical"      , 0   , D_VERTICAL   , 0},
    {"dup"           , 3   , D_DUP        , 0},
    {"saverom"       , 3   , D_SAVEROM    , 0},
    {"pre16"         , 1   , D_PRE_16     , 0},
    {"reversebgpage" , 0   , D_REVERSEDBGPAGE , 0},
    {"loaddata"      , 3   , D_LOADDATA   , 0},
    {"sfxrange"      , 2   , D_SFXRANGE   , 0},
    {"bgmrange"      , 2   , D_BGMRANGE   , 0},
    {"bgm"      , 1   , D_BGM   , 0},
    {"voicerange"    , 2   , D_VOICERANGE , 0},
    {"shortname"     , 1   , D_SHORTNAME  , 0},
    {"objload2"      , 4   , D_OBJLOAD2   , 0},
    {"32bitobj"      , 0   , D_OBJ32BITS  , 0},
    {"defmappedram"  , 3   , D_MAPPEDRAM  , 0},
    {"analogstick"   , 15  , D_ANALOGSTICK, 0},
    {"grscrload"     , 14  , D_GRSCRLOAD  , 0},
    {"specialfunc"   , 2   , D_SPECIALFUNC, 0},
    {"dipswitch"     , 11  , D_DIPSWITCH  , 0},
    {"commonarea"    , 2   , D_COMMONAREA , 0},
    {"labelgen"      , 8   , D_LABELGEN   , 0},
    {"labeljoy"      , 8   , D_LABELJOY   , 0},
    {"defaultgcs"    , 1   , D_DEFAULTGCS , 0},
    {"nordtsc"       , 0   , D_NORDTSC    , 0},
    {"frame"         , 1   , D_FRAME      , 0},
    {"saveextra"     , 2   , D_SAVEEXTRA  , 0},
    {"z80load"       , 1   , D_Z80LOAD    , 0}, // .z80load file1 [file2]...
    {"z80data"       , 2   , D_Z80DATA    , 0}, // .z80data #size file1 [file2]...
    {"z80com"        , 1   , D_Z80COM     , 0}, // .z80com #method [addr_z80 addr_68000 len]
    {"z80patch"      , 2   , D_Z80PATCH   , 0},
    {"ym2151"        , 0   , D_YM2151     , 0},
    {"ym2203"        , 0   , D_YM2203     , 0},
    {"ym3438"        , 0   , D_YM3438     , 0},
    {"specialforshangon", 0   , D_SPECIALFORSHANGON,0},
    {"mouse"         , 10  , D_MOUSE      , 0},
    {"leds"          , 5   , D_LEDS       , 0},
    {"inversedtile"  , 0   , D_INVERSEDTILES, 0},
    {"zip"           , 1   , D_ZIP        , 0},
    {"rompath"       , 1   , D_SETPATH    , 1},
    {"samplespath"   , 1   , D_SETPATH    , 2},
    {"zippath"       , 1   , D_SETPATH    , 3},
    {"savpath"       , 1   , D_SETPATH    , 5},
    {"pcxpath"       , 1   , D_SETPATH    , 6},
    {"rampath"       , 1   , D_SETPATH    , 7},
    {"tilehimask"    , 9   , D_TILEHIMASK , 0}, // .tilehimask #tilenum mask0 ... mask7
    {"speech"	,3 , D_SPEECH, 0}, // .speech uPD7759|uPD7751 8k|6k file1 [fil2]...
    {0               , 0   , 0            , 0}
   };

int ExecuteGCS(char *sn)
{
   char L[256];
   int dn;
   void (*gcs_routine)();
   void (*gcs_routine2)(int p);
   char mymsg[160];

   CheckGCSName(sn);
   
   strcpy(rom_path, ".\\");
   
   INI = fopen(sn, "r");
   if (!INI) {
      sprintf(mymsg, "Can't open game configuration script %s, sorry.\n", sn);
      ierror(mymsg);
      exit(0);
   }
   
   /* some variables init */
   memset(sample_loop, 0, sizeof(sample_loop));
   
   /* GCS execution */
   while (!feof(INI)) {
      fscanf(INI, "%s", L);
      if (L[0]) LowerCase(L);
      if (L[0] == '.') {
         dn = 0;
         while ((strcmp(cmdlist[dn].command,&L[1]))&&(cmdlist[dn].command)) dn++;
         if (cmdlist[dn].command) {
            if (GetParams(cmdlist[dn].nparams)<cmdlist[dn].nparams) {
               sprintf(mymsg, "%d parameters are needed for %s !\n", cmdlist[dn].nparams, &L[1]);
               ierror(mymsg);
               fclose(INI);
               exit(0);
            }
            if (cmdlist[dn].p) {
               gcs_routine2 = cmdlist[dn].ptr;
               (*gcs_routine2)(cmdlist[dn].p);
            } else {
               gcs_routine = cmdlist[dn].ptr;
               (*gcs_routine)();
            }
         } else {
            sprintf(mymsg, "GCS Error : Unknow command '%s' !\n", L);
            ierror(mymsg);
            fclose(INI);
            exit(0);
         }
      }
   }

   fclose(INI);
   return(1);
}

/* new routines to support emulator configuration load/save ****************/

static UINT8 game_uses;
extern int keydef[];
extern byte sync, Jarek2151, SCANLINES, EAGLE, reminder, DOUBLELINES;
extern int SAMPLERATE, frame;
extern UINT8 vertical_display, dipswitch1, dipswitch2, joydef[];
extern char shortname[];

void load_configuration()
{
   set_config_file("SYSTEM16.INI");
   sprintf(game_gcs,"%s",get_config_string("general","defaultgcs","select"));
   rdtsc = get_config_int("general", "rdtsc", 0);
   joystick_type = get_config_int("general", "joytype", 3);
   reminder = get_config_int("general", "uses", 0);
   sprintf(zip_path,"%s",get_config_string("paths","zip",".\\ziproms"));
   fix_path(zip_path);
   sprintf(ram_path,"%s",get_config_string("paths","ram",".\\datas"));
   fix_path(ram_path);
   sprintf(sav_path,"%s",get_config_string("paths","sav",".\\saves"));
   fix_path(sav_path);
   sprintf(pcx_path,"%s",get_config_string("paths","pcx",".\\pcx"));
   fix_path(pcx_path);
   sprintf(zsmp_path,"%s",get_config_string("paths","zsmp",".\\zipsampl"));
   fix_path(zip_path);
   sprintf(movie_path,"%s",get_config_string("paths","movie",".\\movie"));
   fix_path(movie_path);

   drx = get_config_int("screen", "xres", 320);
   dry = get_config_int("screen", "yres", 240);
   vrx = get_config_int("screen", "xres_vert", 400);
   vry = get_config_int("screen", "yres_vert", 300);

   EAGLE = get_config_int("screen", "eagle", 0);
   SCANLINES = get_config_int("screen", "scanlines", 0);
   initial_gamma = get_config_float("screen", "gamma", 1.4);
   DOUBLELINES = get_config_int("screen", "doublelines", 0);
   menu_videomode = get_config_int("screen", "menuvideomode", 5);

   sound_card = get_config_int("sound", "soundcard", -1);
   Jarek2151 = get_config_int("sound", "old2151", 0);
   SAMPLERATE = get_config_int("sound", "samplerate", 22050);

   // Emulator Keys
   keydef[40] = get_config_int("keys", "interface", KEY_F1);
   keydef[41] = get_config_int("keys", "exit", KEY_END);
   keydef[42] = get_config_int("keys", "pause", KEY_P);
   keydef[43] = get_config_int("keys", "screenshot", KEY_S);
   keydef[44] = get_config_int("keys", "background", KEY_F4);
   keydef[45] = get_config_int("keys", "recordmovie", KEY_F9);
   keydef[46] = get_config_int("keys", "playmovie", KEY_F11);
   keydef[47] = get_config_int("keys", "quicksave", KEY_F5);
   keydef[48] = get_config_int("keys", "quickload", KEY_F7);
}

void load_game_configuration(void)
{
   char game_section[80], item[80];
   int i, ip;

   set_config_file("SYSTEM16.INI");

   // Game Keys & Switches
   sprintf(game_section, "%s_config", shortname);
   game_uses = get_config_int(game_section, "uses", 0) + 1;
   dipswitch1 = get_config_int(game_section, "dipswitch1", 0);
   dipswitch2 = get_config_int(game_section, "dipswitch2", 0);
   // load control panel keys
   for (i=0; i<8; i++) {
      sprintf(item, "cp_%d", i);
      keydef[i] = get_config_int(game_section, item, 127);
   }
   // load players keys
   for (ip=1; ip<=3; ip++) {
     for (i=0; i<8; i++) {
        sprintf(item, "player%d_%d", ip, i);
        keydef[i+10*ip] = get_config_int(game_section, item, 127);
     }
   }
   // load players joystick
   for (i=0; i<50; i++) {
      sprintf(item, "joyinfo_%d", i);
      joydef[i] = get_config_int(game_section, item, 0);
   }
   if (sync == 0xff) sync = get_config_int(game_section, "vsync", 0);
}

void remove_file_from_path(char *p)
{
   int i;
   for (i = strlen(p)-1; i>0; i--) {
     if (p[i] == '\\' && p[i-1] != '\\') {
       p[i+1] = 0;
       break;
     }
   }
}

void save_configuration(void)
{
   char game_section[80], item[80];
   int i, ip;

   set_config_file("SYSTEM16.INI");
   set_config_int("general", "rdtsc", rdtsc);
   set_config_int("general", "joytype", joystick_type);
   reminder++;
   set_config_int("general", "uses", reminder);
   remove_file_from_path(zip_path);
   set_config_string("paths", "zip", zip_path);
   set_config_string("paths", "ram", ram_path);
   set_config_string("paths", "sav", sav_path);
   set_config_string("paths", "pcx", pcx_path);
   if ((!EAGLE) && (!SCANLINES)) {
      if (vertical_display) {
         set_config_int("screen", "xres_vert", rx);
         set_config_int("screen", "yres_vert", ry);
      } else {
         set_config_int("screen", "xres", rx);
         set_config_int("screen", "yres", ry);
      }
   }

//   set_config_int("screen", "eagle", EAGLE);
   set_config_int("screen", "scanlines", SCANLINES);
   set_config_float("screen", "gamma", initial_gamma);
   set_config_int("screen", "doublelines", DOUBLELINES);
   set_config_int("screen", "menuvideomode", menu_videomode);

   set_config_int("sound", "soundcard", sound_card);
//   set_config_int("sound", "old2151", Jarek2151);
//   set_config_int("sound", "samplerate", SAMPLERATE);

   // Emulator Keys
   set_config_int("keys", "interface", keydef[40]);
   set_config_int("keys", "exit", keydef[41]);
   set_config_int("keys", "pause", keydef[42]);
   set_config_int("keys", "screenshot", keydef[43]);
   set_config_int("keys", "background", keydef[44]);
   set_config_int("keys", "recordmovie", keydef[45]);
   set_config_int("keys", "playmovie", keydef[46]);
   set_config_int("keys", "quicksave", keydef[47]);
   set_config_int("keys", "quickload", keydef[48]);
   
   // Game Keys & Switches
   sprintf(game_section, "%s_config", shortname);
   set_config_int(game_section, "uses", game_uses);
   set_config_int(game_section, "dipswitch1", dipswitch1);
   set_config_int(game_section, "dipswitch2", dipswitch2);
   // save control panel keys
   for (i=0; i<8; i++) if (keydef[i]!=127) {
      sprintf(item, "cp_%d", i);
      set_config_int(game_section, item, keydef[i]);
   }
   // save players keys
   for (ip=1; ip<=3; ip++) {
     for (i=0; i<8; i++) if (keydef[ip*10+i]!=127) {
        sprintf(item, "player%d_%d", ip, i);
        set_config_int(game_section, item, keydef[i+10*ip]);
     }
   }
   // save players joystick
   for (i=0; i<50; i++) if (joydef[i]) {
      sprintf(item, "joyinfo_%d", i);
      set_config_int(game_section, item, joydef[i]);
   }
   set_config_int(game_section, "vsync", sync);
}
