/*
 * SYSTEM 16 ARCADE EMULATOR SOURCE CODE
 *
 * Copyright 1996/97 Thierry Lescot & NAO
 */

#include <stdio.h>
#include <string.h>
#include <allegro.h>
#include "host.h"
#include "main.h"
#include "shinobi.h"
#include "starcpu.h"
#include "prototyp.h"
#include "mz80.h"

char version_string[] = "The System 16 Arcade Emulator v0.82a";
#define Z80_CPU_FREQ  5000000
#define MAX_SKIP_COUNT 8

extern UINT8 pf_state;
extern void *s16_rpb[0x10000];
extern FILE *LOG;
extern UINT8 LOG_ACTIVE;
extern UINT8 dipswitch1, dipswitch2;
extern INT8 sound_card;
extern UINT8 Z80_active,ym2151_active,ym2203_active;
extern UINT8 sound_type, sampled_sound_support, sound_disabled;
extern UINT8 ComPort;
extern UINT8 z80_speed;
extern UINT8 z80com_port_delay;
extern int never_update_leds;

/* DEBUG VARIABLES */

#ifndef RELEASE
#define SPR_MAX 32
extern UINT8 spr_tab[16];
UINT8 spr_index=0;
#endif

/* SCREEN VARIABLES */

UINT8 SCREEN_IS_LINEAR = 0;
UINT8 SCREEN_IS_PLANAR = 0;
UINT8 SCANLINES;
UINT8 DOUBLELINES;
UINT8 EAGLE;
UINT8 sync = 0xff;
double initial_gamma = 1.4;

/* FIXED VARIABLES */

UINT8 *VIDEOPTR, *SOURCEPTR;
UINT32 LINEOFFSET;
UINT16 VIDEOSEG;
UINT16 drx = 320, dry = 240, vrx = 640, vry = 480;
UINT16 rx=320, ry=240;
int total_x, total_y;
int sx=0, sy=0;
UINT8 joystick_support = 1;
UINT8 rdtsc=1,fin_emulateur=0;
char reminder = 0, escexit = 0;
#ifdef RELEASE
UINT8 notitle = 0;
#else
UINT8 notitle = 1;
#endif
UINT8 menu_videomode = 5;

char game_gcs[80], game_name[20], shortname[9]="noname";
UINT8 game=1;
int opcodeb,opcodeb2;
UINT8 pentium_pro_cpu = 1;
UINT8 joystick_type  = 3;

/* MOUSE VARIABLES */
UINT8 *mousex_ptr, *mousey_ptr;
UINT8 *mouseb_ptr;
int mouse_minX, mouse_minY, mouse_maxX, mouse_maxY, mouse_defX, mouse_defY;
UINT8 mouseb_msk, mouse_enabled = 0;

UINT8 fp_rx=0,fp_ry=0;

/* SOUND VARIABLES */
UINT8 Jarek2151 = 0;
UINT8 midi = 0;
extern UINT8 type_sound_subboard;

/* EXTERNAL FIXED VARIABLES */

extern int keydef[50];

/* TO TEST */

UINT8 joy_player=0;
UINT32 global_timer = 0;

extern UINT8 reset_palette;
extern UINT8 vertical_display;
UINT8 check_performance=0;
unsigned performance_68k,performance_z80,performance_graphics,performance_fm,performance_blit;

UINT8 sfx_enabled = 1;
UINT8 bgm_enabled = 1;
UINT8 voice_enabled = 0;
UINT8 last_bgm = 0;

extern UINT32 show_datavalue;
extern UINT8 *s16_ptr[0x10000];

extern UINT8 shinobi_ds1[10], shinobi_ds2[10];

UINT8 reverse_y = 0, reverse_x = 0;

long total_render=0;

int int_request=0;

UINT8 frame=1;
UINT8 frame_number, skip_count;

UINT8 background_active=1;
int pcx_number=-1;
int current_tpl_slot = -1;
int trace_play_mode = 0;
UINT8 cur_trace_play_state[16];

long render_countdown=1;
int int_speed;

static int pause=0;

/* screen variables */
BITMAP *vscreen, *vscreen_sub, *v_vscreen, *linear_screen;

void pause_game()
{
  if (pause == 0) {
    pause = 1;
    if (sampled_sound_support) set_audio_volume(0);
  } else pause++;
}

void resume_game()
{
  if (pause > 0) {
    if (--pause == 0) {
      if (sync) reset_video_sync();
      global_timer = read_timer(); // reset global timer to current value
      if (sampled_sound_support) {
        set_audio_volume(64);
        reset_wavestream_pointers();
      }
    }
  }
}

void Video(int f) {
   static PALETTE pal_save;
   int vx = 320, vy =224;
   int double_dots = SCANLINES + 1;
   
   if (EAGLE) double_dots = 2;

   total_x = (vertical_display) ?vy*double_dots :vx*double_dots;
   total_y = (vertical_display) ?vx*double_dots :vy*double_dots;

   if (f != 0) {
      if (EAGLE || !SCANLINES) {
        if (set_gfx_mode(GFX_VESA2L, rx, ry, 0, 0)<0) {
           if (set_gfx_mode(GFX_AUTODETECT, rx, ry, 0, 0)<0) {
              allegro_exit();
              printf("Error, cannot set %dx%d video mode !\n", rx, ry);
              exit(-1);
           }
        }
        set_palette(pal_save);

      } else {
         UINT8 had_set_once = 0;

reset_video_mode:
         if (rx == 0 || had_set_once) { // use default res.
           if (!vertical_display) {
             switch (double_dots) {
             case 2: rx=640; ry=480; break;
             case 3: rx=1024; ry=768; break;
             case 4: rx=1280; ry=1024; break;
             }
           } else {
             switch (double_dots) {
             case 2: rx=1024; ry=768; break;
             case 3: 
             case 4: rx=1280; ry=1024; break;
             }
           }
         }

         if (set_gfx_mode(GFX_VESA2L, rx, ry, 0, 0)<0) {
            if (had_set_once) {
               allegro_exit();
               ierror("Error, cannot set %dx%d linear frame buffer video mode (VESA 2.0 or above) !\n", rx, ry);
               exit(1);
            } else {
               had_set_once = 1;
               goto reset_video_mode;
            }
         }

         VIDEOPTR = (UINT8 *)screen->line[8];
         VIDEOPTR = (UINT8 *)screen->line[(ry-(vertical_display ?vx:vy)*double_dots)/2]+(rx-(vertical_display ?vy:vx)*(ry>rx?1:double_dots))/2*8/*color_depth*//8;
         LINEOFFSET = ( ((long)screen->line[9]) - ((long)screen->line[8]) );
         VIDEOSEG = screen->seg;
      }

      SCREEN_IS_LINEAR = is_linear_bitmap(screen);
      SCREEN_IS_PLANAR = is_planar_bitmap(screen);

      sx = (rx-total_x)/2;
      sy = (ry-total_y)/2;

      if (mouse_enabled) {
         set_mouse_range(mouse_minX,mouse_minY,mouse_maxX,mouse_maxY);
         position_mouse(mouse_defX,mouse_defY);
      }

   } else {
      get_palette(pal_save);
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
   }
}

void calibrate_joystick_text() {
   if (exists("SYSTEM16.JOY")) {
     joystick_support = (load_joystick_data("SYSTEM16.JOY"))? 0:2;

   } else {
     imessage2("Joystick calibration procedure starting...(press ESC to skip)\n",
               "Put joystick in center position, press button A or B when ready...");
     do {
       if (key[KEY_ESC]) {
         joystick_support = 1;
         clear_imessage2();
         return;

       }
       poll_joystick();
     } while ((joy_b1+joy_b2)==0);

     initialise_joystick();
     imessage2("Put joystick to top/left position, press button A or B when ready...", 0);
     do { poll_joystick(); } while ((joy_b1+joy_b2)!=0);
     do { poll_joystick(); } while ((joy_b1+joy_b2)==0);
     clear_imessage2();
     calibrate_joystick_tl();
     imessage2("Put joystick to bottom/right, press button A or B when ready...", 0);
     do { poll_joystick(); } while ((joy_b1+joy_b2)!=0);
     do { poll_joystick(); } while ((joy_b1+joy_b2)==0);
     clear_imessage2();
     calibrate_joystick_br();
     save_joystick_data("SYSTEM16.JOY");
     joystick_support = 2;
     imessage2("Joystick calibration terminated !", "Delete SYSTEM16.JOY to calibrate again. (press any key)");
     clear_keybuf(); while (!keypressed()); clear_keybuf();
     clear_imessage2();
  }
}

void check_for_pentium_pro_cpu()
{
  int r;
  if (rdtsc) {
    chk_tsc();
    test_for_pentium_pro();
    r =  chk_tsc();
    if (r >= 4000000) {
      pentium_pro_cpu = 1;
      if (pf_state) printf("%d Optimized for pentium_pro cpu\n",r);
      return;
    }
  }

  if (pf_state) printf("%d Optimized for pentium cpu\n",r);
  pentium_pro_cpu = 0;
}

void Initialisation() 
{
   /* Joystick detection */
   if (joystick_support) {
     if (init_joystick(joystick_type) == 0) {
       if (pf_state) printf("Joystick found !\n");
       joy_player=1;
       if (joystick_support==2) calibrate_joystick_text();
     } else joystick_support = 0;
   } else init_joystick(-1);
   
   check_for_pentium_pro_cpu();
}

static void inline make_sound()
{
   if (Z80_active) {
     int i;
     extern UINT8 YMStatus;
     extern int YMStatus_read_times;

     if (check_performance) chk_tsc();
     for (i=0; i<z80_speed; i++) {
//     printf("------z80\n");
       mz80exec(Z80_CPU_FREQ/60/z80_speed);
       YMStatus |= 0x3;
       YMStatus_read_times=z80com_port_delay;
     }
     if (check_performance) performance_z80 = chk_tsc();
   }
}

void OutrunEmule()
{
   extern UINT8 mask_int_cnt;
   if (sync || int_request) {
      if (check_performance) chk_tsc();

      context_switch(0);
      cpu_interrupt(2);
      cpu_dispatch(opcodeb);

      context_switch(1);
      cpu_interrupt(4);
      cpu_dispatch(25000);

      context_switch(0);
      NewCheckInput();

      if (!mask_int_cnt) {
        cpu_interrupt(4);
        {
           extern UINT32 cpu_checker;
           cpu_checker&=0xfffffffe;
           cpu_dispatch(50);
           while (!(cpu_checker&1)) // till RTE
           cpu_dispatch(1);
        }
      } else {
         if (LOG_ACTIVE) fprintf(LOG,"masked int %d\n",mask_int_cnt);
         mask_int_cnt--;
      }

      context_switch(1);
      cpu_interrupt(4);
      cpu_dispatch(25000);
      if (check_performance) performance_68k = chk_tsc();

      int_request--;
      make_sound();
      if (sampled_sound_support) samples_control();
   }

   /* Output a frame if skip counter is large enough */
   if (sync || (skip_count>=frame-1 && (!int_request || skip_count>MAX_SKIP_COUNT))) {
     RenderScreen();
     skip_count = 0;
   } else skip_count++;

}

void inline NewEmule() {
   extern int num_cpus; 
   int i;

   if (sync || int_request) {
    
      if (check_performance) chk_tsc();

      for (i=0; i<num_cpus; i++) {
        context_switch(i);

        if (opcodeb2<opcodeb) {
          cpu_dispatch(opcodeb-opcodeb2); opcodeb2=0;
        } else opcodeb2-=opcodeb;

      }
      opcodeb2=0;

      if (check_performance) performance_68k = chk_tsc();

      NewCheckInput();
      for (i=0; i<num_cpus; i++) {
        context_switch(i);
        cpu_interrupt(4);

        if (num_cpus>1) {
          extern UINT32 cpu_checker;
          cpu_checker&=0xfffffffe;
          cpu_dispatch(50);
          while (!(cpu_checker&1)) { // till RTE
            cpu_dispatch(1);
         }
        }
      }

      make_sound();
      int_request--;
      if (sampled_sound_support) samples_control();
   }

   /* Output a frame if skip counter is large enough */
   if (sync || (skip_count>=frame-1 && (!int_request || skip_count>MAX_SKIP_COUNT))) {
     RenderScreen();
     skip_count = 0;
   } else skip_count++;

}

/* SavePCX */
void SauvegardePCX() {
   extern char pcx_path[80];
   char fn[20];
   char full_path[256];
   char ws[256];
   int l;
   PALLETE pal;
   
   get_palette(pal);
   do {
      pcx_number++;
      sprintf(fn, "%s", shortname);
      l = strlen(shortname);
      if (l>6) sprintf(&fn[l-2], "%02d.PCX", pcx_number);
      else sprintf(fn, "%s%02d.PCX", shortname, pcx_number);
      sprintf(full_path, "%s%s", pcx_path, fn);
   } while (file_exists(full_path, 0, NULL));
   if (vertical_display) {
      rotate_screen();
      save_pcx(full_path, v_vscreen, pal);
   } else save_pcx(full_path, vscreen_sub, pal);
   sprintf(ws, "%s SAVED", fn);
   message_emul(1, 2, ws);
}

int spr_limit = 0x800;

/* keyboard handler */
void Clavier() 
{
   int code, cont=1;
   static UINT32 old_show_datavalue=0;
   char ws[256];
   #define MIN_WAIT_TIME 10
   
   do {
      // when the game is paused, timer must be reload constantly and do check the message line
      if (pause) {
        global_timer = read_timer(); //reload global timer
        update_message(); //update message line
      }
   
      {
#ifndef RELEASE
         if (key[KEY_PADADD]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              if (key[KEY_LSHIFT] || key[KEY_RSHIFT]) {
                spr_limit = spr_limit + 0x10;
                if (spr_limit > 0x800) spr_limit = 0x800;
                sprintf(ws, "SPR LIMIT = %x", spr_limit);
                message_emul(0, 2, ws);
              } else {
                spr_index = (spr_index==15)? 0:spr_index+1;
                sprintf(ws, "SPR INDEX = %d", spr_index);
                message_emul(0, 2, ws);
              }
            }
         } else if (key[KEY_PADSUB]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              if (key[KEY_LSHIFT] || key[KEY_RSHIFT]) {
                spr_limit = spr_limit - 0x10;
                if (spr_limit <= 0) spr_limit = 0x10;
                sprintf(ws, "SPR LIMIT = %x", spr_limit);
                message_emul(0, 2, ws);
              } else {
                spr_index = (spr_index)? spr_index-1:15;
                sprintf(ws, "SPR INDEX = %d", spr_index);
                message_emul(0, 2, ws);
              }
            }
         } else if (key[KEY_PGUP]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
#ifdef THIERRY
              if (!(key[KEY_LSHIFT] || key[KEY_RSHIFT] || key[KEY_ALT])) {
#else
              if (key[KEY_LSHIFT] || key[KEY_RSHIFT]) {
#endif
                spr_tab[spr_index] = (spr_tab[spr_index]==SPR_MAX)? 0:spr_tab[spr_index]+1;
                sprintf(ws, "SPR INDEX = %d SET TO %d", spr_index, spr_tab[spr_index]);
                message_emul(0, 2, ws);
              } else {
                if (show_datavalue==0) show_datavalue=old_show_datavalue;
                show_datavalue-=key[KEY_ALT]?4096:64;
                show_datavalue&=0xffffff;
              }
            }
         } else if (key[KEY_PGDN]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
#ifdef THIERRY
              if (!(key[KEY_LSHIFT] || key[KEY_RSHIFT] || key[KEY_ALT])) {
#else
              if (key[KEY_LSHIFT] || key[KEY_RSHIFT]) {
#endif
                spr_tab[spr_index] = (spr_tab[spr_index])? spr_tab[spr_index]-1:SPR_MAX;
                sprintf(ws, "SPR INDEX = %d SET TO %d", spr_index, spr_tab[spr_index]);
                message_emul(0, 2, ws);
              } else {
                if (show_datavalue==0) show_datavalue=old_show_datavalue;
                show_datavalue+=key[KEY_ALT]?4096:64;
                show_datavalue&=0xffffff;
              }
            }
         } else if (key[KEY_HOME]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              if (show_datavalue) {
                old_show_datavalue=show_datavalue;
                show_datavalue=0;
              } else show_datavalue=old_show_datavalue;
            }
         } else if (key[KEY_D]) {
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              NewDebugger();
              Reset_Palette();
              resume_game();
            }
         } else /* continue with standard commands */
#endif
         if (key[keydef[40]]) { /* INTERFACE */
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              if (Interface(((key[KEY_LSHIFT])||(key[KEY_RSHIFT]))? 1:0)) {
                 pause = 0;
                 fin_emulateur=1;
              }
              resume_game();
            }
         } else if (key[keydef[44]]) { /* BG ON/OFF */
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              background_active=~(background_active&0x01);
            }
         } else if (key[keydef[43]]) { /* SAVE PCX */
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              SauvegardePCX();
              resume_game();
            }
         } else if (key[keydef[41]]) { /* EXIT */
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause=0;
            }
         } else if (key[keydef[42]]) { /* PAUSE */
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              if (!pause) {
                message_emul(0, 1, "Game paused !");
                pause_game();
              } else {
                message_emul(0, 1, "Game resumed !");
                resume_game();
              }
            }
         } else if (key[keydef[47]]) { /* QUICKSAVE */
            char fn[128]; int ret;
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              sprintf(fn, "%s.saq", shortname);
              ret = SaveGame(fn, "Quicksave");
              resume_game();
              message_emul(1, 2, ret==0?"Quicksave complete.":"Quicksave failed.");
            }
         } else if (key[keydef[48]]) { /* QUICKLOAD */
            char fn[128]; int ret;
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              sprintf(fn, "%s.saq", shortname);
              ret = LoadGame(fn);
              resume_game();
              message_emul(1, 2, ret==0?"Quickload complete.":"Quickload failed.");
            }
         } else if (key[keydef[45]]) { // RECORD MOVIE
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              if (trace_play_mode == 1) {
                save_tpl_data();
                trace_play_mode = 0;
                message_emul(0, 2, "END RECORDING");
              } else if (trace_play_mode == 0) {
                if (start_record_tpl(cur_trace_play_state, NULL) == 0) {;
                  trace_play_mode = 1;
                  message_emul(0, 2, "START RECORDING");
                }
              }
              resume_game();
            }
         } else if (key[keydef[46]]) { // PLAY MOVIE
            static int time=0;
            if (global_timer > time+MIN_WAIT_TIME) {
              time = global_timer;
              pause_game();
              if (trace_play_mode == 2) {
                trace_play_mode = 0;
                message_emul(0, 2, "END PLAYING");
              } else if (trace_play_mode == 0) {
                if (load_tpl_data() == 0) {
                  trace_play_mode = 2;
                  start_play_tpl(cur_trace_play_state);
                  message_emul(0, 2, "START PLAYING");
                }
              }
              resume_game();
            }
         }

         if (pause) RenderScreen();
      }
   } while (pause);
}

void CommandLineHelp() 
{
   printf("Syntax : SYSTEM16 [gcs file name] (options...)\n\n"
          "options:\n"
          "-vsync       : enable video synchronization\n"
          "-novsync     : disable video synchronization\n"
          "-rx        # : change horizontal resolution [320]\n"
          "-ry        # : change vertical resolution [240]\n"
          "-frame     # : skip frame [1]\n"
          "-list        : list all GCS files\n"
          "-rdtsc       : use RDTSC (make sure you do have PENTIUM or K6)\n"
          "-nordtsc     : don't use RDTSC (if you don't have PENTIUM or K6)\n"
          "-r         # : change sample rate [22050]\n"
          "-soundcard # : select an audio device [1]\n"
          "                0 = Disable sounds\n"
          "                1 = Sound Blaster8/16/32/64\n"
          "                3 = Pro Audio Spectrum\n"
          "                7 = Ensoniq Soundscape\n"
          "-new2151     : select new YM2151 (Satoh)\n"
          "-old2151     : select alternate YM2151 (Jarek)\n"
          "-eagle       : enable EAGLE post-processing\n"
          "-noeagle     : disable EAGLE post-processing\n"
          "-scanlines   : enable scanline display\n"
          "-scanlines2  : enable thick scanline display\n"
          "-noscanlines : disable scanline display\n"
          "-noleds      : disable keyboard led control\n"
          "-gamma     # : set video output gamma value [1.4]\n"
          "-joy         : enable analog joystick support\n"
          "-nojoy       : disable joystick/gamepad support\n"
          "-reversey    : reverse direction Y of analogstick\n"
          "-reversex    : reverse direction X of analogstick\n"
          "-joytype   # : set joystick device type [3]\n"
          "                3 = Normal Joystick\n"
          "                8 = MS SideWinder Gamepad\n"
          "                9 = Gravis GamePad Pro\n"
          "                10,11,12 = SNESPAD at LPT1,2,3\n"
          "-notitle     : disable title\n"
          "-?/-help     : show this message\n"
          );
   exit(0);
}

int CheckArguments(int argc, char **argv) {
   char *CLineOptions[] = {
      "-regspeed", "-int", "-vsync", "-joy", "-nomidi",
      "-show", "-rx", "-ry", "-list", "-help", "-?",
      "-nordtsc", "-soundcard", "-frame", "-nojoy", "-r",
      "-old2151", "-scanlines", "-reversey", "-eagle",
      "-novsync", "-noeagle", "-new2151", "-rdtsc", "-noscanlines", "-escexit",
      "-notitle", "-perf", "-noleds", "-scanlines2", "-scanlines3", "-gamma",
      "-doublelines", "-reversex", "-joytype", "-menuvideomode",
      NULL
   };
   int NumParams[] = {
      1, 500, 0x7fffffff,  1, 1, 500, 0, 0, 0,  0, 0, 0,  0, 0, 0,
      1601, 0, 0xffffff,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,
      0, 0, 0, 1, 0, 20, 1, 1, 10, 0, 0, 0, 1, 11025, 44100, 0, 0, 0,
      0, 0, 0,  0, 0, 0,  0 ,0 ,0,
      0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,
      0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,
      0, 0, 0,  0, 0, 0,  1, 0, 12,  1,0,12,
   };
   int np = 0, ok = 0, pc, i, ok2 = 0;
   char mymsg[160];

   /* get the name of the GCS file */
   for (i=0; i<argc; i++) LowerCase(argv[i]);
   if ((argc>1) && (argv[1][0]!='-')) strcpy(game_gcs, argv[1]);
   
   /* default values */
   int_speed = 60;
   opcodeb = 15000;
   
   /* check parameters */
   for (i=1; i<argc; i++) {
      int value = 0;
      if (argv[i][0]=='-') {
         np = ok = 0;
         while (CLineOptions[np]) {
            if (!strcmp(CLineOptions[np], argv[i])) { ok = np + 1; break; }
            np++;
         }
         if (!ok) {
            sprintf(mymsg, "Unknow parameter : '%s'\n", argv[i]);
            ierror(mymsg);
            ok2 = 1;
         } else {
            char hex=0;
            int num_args=NumParams[--ok * 3];
            if (num_args>1600) { hex=1; num_args-=1600; }
            for (pc=(i+1);pc!=(i+1+num_args);pc++) {
               if ((argv[pc][0]==0)||(argv[pc][0]=='-')||(pc==argc)) {
                  sprintf(mymsg, "Error, %s needs %d arguments.", argv[i], NumParams[ok * 3]);
                  ierror(mymsg);
                  return(1);
               } else {
                  value=strtol(argv[pc],NULL,hex?16:10);
                  if (NumParams[ok*3+1]!=-1)
                    if (NumParams[ok*3+1]>value) {
                       sprintf(mymsg, "Bad value for %s, only from %d to %d.", argv[i], NumParams[ok*3+1], NumParams[ok*3+2]);
                       ierror(mymsg);
                       return(1);
                    }
                  if (NumParams[ok*3+2]!=-1)
                    if (NumParams[ok*3+2]<value) {
                       sprintf(mymsg, "Bad value for %s, only from %d to %d.", argv[i], NumParams[ok*3+1], NumParams[ok*3+2]);
                       ierror(mymsg);
                       return(1);
                    }
               }
            }
            switch (ok) {
             case 0: opcodeb = atoi(argv[i+1]); break;
             case 1: int_speed = atoi(argv[i+1]); break;
             case 2: sync = 1; break;
             case 3: joystick_support = 2; break;
             case 4: midi=0; break;
             case 5: show_datavalue=value; break;
             case 6: rx = atoi(argv[i+1]); fp_rx = 1; break;
             case 7: ry = atoi(argv[i+1]); fp_ry = 1; break;
             case 8: ListGCS(); break;
             case 9:
             case 10: CommandLineHelp(); break;
             case 11: rdtsc=0; break;
             case 12: sound_card = atoi(argv[i+1]); break;
             case 13: frame = atoi(argv[i+1]); break;
             case 14: joystick_support = 0; break;
             case 15: { extern int SAMPLERATE; SAMPLERATE=atoi(argv[i+1]); } break;
             case 16: Jarek2151 = 1; break;
             case 17: SCANLINES = 1; break;
             case 18: reverse_y = 1; break;
             case 19: EAGLE = 1; break;
             case 20: sync = 0; break;
             case 21: EAGLE = 0; break;
             case 22: Jarek2151 = 0; break;
             case 23: rdtsc = 1; break;
             case 24: SCANLINES = 0; break;
             case 25: escexit = 1; break;
             case 26: notitle = 1; break;
             case 27: check_performance = 1; break;
             case 28: never_update_leds = 1; break;
             case 29: SCANLINES = 2; break;
             case 30: SCANLINES = 3; break;
             case 31: sscanf(argv[i+1], "%lf", &initial_gamma); break;
             case 32: DOUBLELINES = 1; break;
             case 33: reverse_x = 1; break;
             case 34: joystick_type = atoi(argv[i+1]); break;
             case 35: menu_videomode = atoi(argv[i+1]); break;
            }
         }
      }
   }
   return(ok2);
}

int main(int argc, char **argv) 
{
   load_configuration();

   if (CheckArguments(argc, argv)) {
      ierror("Command line error !\n");
      CommandLineHelp();
      exit(-1);
   }
#ifndef RELEASE
   if (!strcmp(game_gcs,"select")) {
      CommandLineHelp();
      exit(-1);
   }
#endif
   allegro_init();
   install_keyboard();
   init_timer(int_speed);

   if (!notitle) {
     if (first_screen()) {
        allegro_exit();
        exit(-1);
     }
     if (!(reminder%10))
     imessage("PLEASE, DON'T FORGET TO READ THE DOC (SYSTEM16.DOC)",47,46,1,1);
   }

   if (!strcmp(game_gcs,"select")) {
      if (!boot_interface(game_gcs)) {
         allegro_exit();
         printf("Back to dos !\n\n");
         exit(0);
      }
   }

   Initialisation();
   fastcpu_init();
   if (Z80_active) {
     newz80_init(type_sound_subboard);
     ComPort=0;
   }

   Load_GameSRAM();
   load_game_configuration();
   fastcpu_init2();

   vscreen=create_bitmap(1024, 400);
   vscreen_sub=create_sub_bitmap(vscreen, (1024-320)/2, (400-224)/2, 320, 224);
   SOURCEPTR = (UINT8 *)vscreen_sub->line[0];

   if (vertical_display) v_vscreen = create_bitmap(224, 320);

   clear_keybuf();
   remove_keyboard();
   install_keyboard();

   if (sound_type) {
     if (OpenAudio() != 0) { // open audio error, disable Z80&sound now
       Z80_active = 0;
       sampled_sound_support = 0;
     }
   }

   if (mouse_enabled) mouse_enabled = (install_mouse()==-1) ? 0:1;

   if (!fp_rx) rx = (vertical_display)? vrx:drx;
   if (!fp_ry) ry = (vertical_display)? vry:dry;

   if (EAGLE) {
      rx = (vertical_display)? 1024:640;
      ry = (vertical_display)? 768:480;
   } 
   else if (SCANLINES) {
     if (!vertical_display && SCANLINES<=1) {
       rx = 320; ry=480;
     } else {
       rx = ry = 0; // need to be set by Video()
     }
   }

   bootintf_off();

   if (escexit) keydef[41] = KEY_ESC;
#ifdef RELEASE
   if (game==20) sync = 0; // OUTRUN is not vsync-compatible
#endif
   Video(1);

   init_blit_routine();
   init_tpl();
   
   Setup_Asm();
   Create_Palette();

   clear(vscreen_sub);
   if (rdtsc) init_rdtsc();
  

#ifdef RELEASE
   message_emul(0, 5, version_string);
#endif

   NewCheckInput();

#ifdef DEBUGGER
   NewDebugger();
#endif

   int_request = 0;
   global_timer = 0;
   start_timer();

   do {
      UINT32 new_timer = read_timer();
      int_request  = new_timer - global_timer;
      /* check temperary message line */
      if (sync||int_request) {
        global_timer++;
        check_message();
      }

      if (game==20) OutrunEmule(); else NewEmule();
      Clavier();

   } while ((!key[keydef[41]])&&(!fin_emulateur));

   if (sampled_sound_support) CloseAudio();

   remove_keyboard();
   quit_timer();

   allegro_exit();

   Save_GameSRAM();
   save_configuration();
   quit_tpl();

   printf("Video mode used : ");
   if (SCREEN_IS_PLANAR) printf("Planar mode. (X-Mode)\n");
   else if (SCREEN_IS_LINEAR) printf("Linear VGA/VESA mode.\n");
   else printf("Banked VGA/VESA mode.\n");
   printf("Resolution : %dx%d\n", rx,ry);
   if (global_timer) printf("Average frame rate = %6.2lf FPS\n", total_render/(global_timer/(double)int_speed));

#ifdef RELEASE
   printf("\n%s\n\n"
          "Presented by Thierry Lescot and Li Jih Hwa.\n\n"
          "See you later...\n", version_string);
#endif
   return 0;
}
